# This module provides basic dialogs of Alicq
#
# Author: Igor Vergeichik

package require Tk
package require BWidget

namespace eval meta {
	set author "Ihat Viarheickyk <iverg@mail.ru>"
	set description "Basic implementation of chat dialogs."
	array set popup {
		type boolean default 0 property Global:UI:Messages|Popup
		description "Popup dialog window on new messages" save exit
		menu {Interface "Popup Dialogs"}
	}
	array set tabbing {
		type boolean default 0 save exit menu {Interface Tabbing}
		description "Use tabbed panels within single wnidow"
	}
	array set add:contact {type action menu {Add Contact}}
	array set add:group {type action menu {Add Group}}
}

option add *timeFormat {%X %x} widgetDefault
option add *Text.underlineTag alias widgetDefault
option add *Text.headerBackground gray85

option add *EditBar.send.text [mc Send] widgetDefault
option add *EditBar.close.text x widgetDefault
option add *EditBar.close.relief flat widgetDefault
option add *EditBar.type.direction above widgetDefault
option add *EditBar.type.anchor w widgetDefault
option add *EditBar.AuxBar.cite.text [mc Cite] widgetDefault
option add *Tablist.Radiobutton.selectColor Green widgetDefault
option add *Tablist.Radiobutton.foreground black widgetDefault
option add *Tablist.Radiobutton.highlight blue widgetDefault

foreach {item value} {incoming	darkMagenta	outgoing	red
	client	darkBlue	server	SteelBlue	sent	darkGray
	authrequest brown action blue} {
	option add *${item}Color $value widgetDefault
}


event add <<Send>> <Control-Return>
event add <<Cite>> <Control-q>
event add <<NextTab>> <Control-Right>
event add <<PrevTab>> <Control-Left>
foreach x {1 2 3 4 5 6 7 8 9} { event add <<ToTab>> <Control-Key-$x> }

proc WinByUid {prefix uid} { return .${prefix}[string map {: -} $uid] }

proc members {uid} {
	if {[string match Contact:* $uid]} {
		return [list]
	} else {
		set gr [lindex [split $uid :] end]
		return [select Contact "\[lsearch \$Groups $gr\]!=-1"]
	}
}

# Raise windows, but not take focus if another dialog window has it
proc Raise {win {flag 0}} {
	set focus [focus]
	if {$focus!=""} { set focus [winfo toplevel $focus] }
	if {$focus=="" || $focus=="." || $flag} {
		raise $win
		focus $win
	} elseif {$focus!=$win} { lower $win $focus }
}

# Main dialog for text messages used in single window mode
proc SingleDialog {uid {flag 0}} {
	set top [WinByUid msg $uid]
	if {![winfo exists $top]} {
		toplevel $top -class AlicqUserWindow
		wm title $top "[get $uid Alias] ([lrange [split $uid :] 1 2])"
		DialogFrame $top $uid
	}
	Raise $top $flag
}

# Main dialog for text messages used in tabbed window mode
proc TabbedDialog {uid {flag 0}} {
	set top .all
	set var [namespace current]::frameSelected
	if {![winfo exists $top]} {
		toplevel $top -class AlicqUserWindow
		wm title $top "Alicq chat"
		grid [frame $top.btn -class Tablist] -sticky new -row 0
		grid columnconfigure $top 0 -weight 1
		grid rowconfigure $top 1 -weight 1
		trace variable $var w [nc SelectTab $top]
		bind $top <FocusIn> [namespace code { 
		     if {[winfo toplevel %W]!={%W}} continue
		     event generate %W[WinByUid msg $frameSelected] <FocusIn>
		}]
		set flag 1
	}
	set win [WinByUid msg $uid]
	set btn "$top.btn$win"
	if {![winfo exists $btn]} {
		radiobutton $btn -indicatoron 0\
			-text [get $uid Alias] -value $uid -variable $var
		CompoundSend $btn $uid
		pack $btn -side left -ipadx 2 -ipady 1 
		DialogFrame [frame $top$win -class Tab] $uid
	}
	if {$flag} { $btn invoke }
}

# When tab is deleted, delete assotiated button as well
bind Tab <Destroy> [nc DeleteTab %W]
bind Tab <Map> [namespace code { highlight $frameSelected } ]
proc DeleteTab {win} {
	set parent [winfo parent $win].btn
	set btn $parent.[winfo name $win]
	if {![winfo exists $btn]} return
	set list [pack slaves $parent]
	if {[set pos [lsearch -exact $list $btn]]==-1} return
	set list [lreplace $list $pos $pos]
	if {![llength $list]} { 
		destroy [winfo toplevel $win]
	} else {
		destroy $btn
		if {![winfo exists [lindex $list $pos]]} { set pos end }
		[lindex $list $pos] invoke 
	}
}

# Display tab selected with assotiated button
proc SelectTab {top name args} {
	upvar $name uid
	if {![winfo exists [set win $top[WinByUid msg $uid]]]} return
	foreach x [grid slaves $top] { 
		if {$x==$win} return
		if {[winfo class $x]=="Tab"} {
			grid forget $x
			grid propagate [winfo parent $x] 0
		}
	}
	grid $win -sticky news -row 1
	focus $win
}

# Select tab with keyboard
proc ToTab {win key} {
	set slaves [winfo children [winfo toplevel $win].btn]
	set next [lindex $slaves [expr $key-1]]
	if {$next!=""} { $next invoke }
}

proc NextTab {win step} {
	variable frameSelected
	set slaves [winfo children [winfo toplevel $win].btn]
	set curr [lsearch $slaves *[WinByUid msg $frameSelected]]
	set next [lindex $slaves [expr $curr + $step]]
	if {$next!=""} { $next invoke }
}

handler *|TaggedText highlightTab {uid args} {
	if {[winfo exists ".all.btn[WinByUid msg $uid]"]} { highlight $uid }
}

# Common dialog frame used ni both single and tabbed modes
proc DialogFrame {top uid} {
	foreach {chat in out} [ChatBar $top.pw] break
	grid $chat -sticky news
	grid [EditBar $top.edit $uid $in $out] -sticky we
	foreach x {row column} {grid ${x}configure $top 0 -weight 1}
	set ids [list [hook $uid|HistoryItem [nc FillHistory $in]]]
	foreach x [concat [members $uid] $uid] {
		lappend ids [hook $x|Acknowledgement [nc AckMessage $in]]\
			    [hook $x|TaggedText [nc DisplayMessage $in] 0.9]
	}
	bind $top <FocusIn> [list focus $out]
	bind $in <Destroy> [nc unhook $ids]
	foreach x [list $top $out] { bind $x <<Close>> [list destroy $top] }
}

# Editbar contains buttons to control message edition and sending
proc EditBar {name uid in out} {
	set multi [string match Group:* $uid]
	frame $name -class EditBar
	button $name.send -default active -command [nc SendMessage $uid $out]
	bind $out <<Send>> "[list $name.send] invoke; break"
	grid $name.send -row 0 -column 0 -sticky wns
	# Check if uid supports several message types
	set types [Event MessageTypes $uid]
	if {[llength $types]} {
		TypeButton $name.type $uid $types $name.send
		grid $name.type	-row 0 -column 1 -sticky w
	}
	if {[winfo class [winfo parent $name]]!="Tab"} { 
		CompoundSend $name.send $uid 
	}
	grid [button $name.close] -row 0 -column 10 -sticky e
	$name.close configure -command [list event generate [winfo parent $name] <<Close>>]
	frame $name.aux -class AuxBar
	# If multicast mode, add contact aliases to citations
	button $name.aux.cite -command [nc CiteLast $in $out $multi]
	bind $out <<Cite>> [list $name.aux.cite invoke]
	grid $name.aux.cite -row 0 -column 1 -sticky wns
	# History button available only for unicast messages
	if {!$multi} {
		button $name.aux.history
		HistoryButton $name.aux.history $uid [Event HistoryChunks $uid]
		grid $name.aux.history -row 0 -column 2 -sticky wns
	}
	if {[llength [info command tooltip]]} { 
		tooltip $name.aux.cite "Cite last message" <<Cite>>
		tooltip $name.close "Close dialog" <<Close>>
		tooltip $name.send "Send message" <<Send>>
	}
	grid $name.aux -sticky wns -row 0 -column 2 -ipadx 1 -padx 5
	foreach x {1 2 3} { grid columnconfigure $name $x -weight 1}
	set name
}

# Determine minimal set of common types for group messages
handler MessageTypes grouptypes {uid} {
	if {![string match Group:* $uid]} {return -code continue}
	set members [members $uid]
	set types [Event MessageTypes [lindex $members 0]]
	foreach x [lrange $members 1 end] {
		set next [Event MessageTypes $x]
		foreach t $types {
			if {[set pos [lsearch $next $t]]==-1} {
				set types [lreplace $types $pos $pos]
			}
		}
	}
	set types
}

# Create menubutton for selection of message type from a list
# Selected type is stored in variable with UID-based name
proc TypeButton {name uid types send} {
	set var [namespace current]::${uid}-type
	menubutton $name -menu $name.vals
	menu $name.vals -tearoff no
	set max 0
	foreach x $types {
		set t [mc $x]
		$name.vals add radiobutton -label $t -value $x -variable $var\
			-command [list $name configure -text $t]
		set len	[string length $t]
		if {$len>$max} {set max $len}	
	}
	$name configure -width $max
	bind $name.vals <Destroy> [list unset $var] 
	$name.vals invoke 0
	$send configure -command [join [list [$send cget -command]\
		[list $name.vals invoke 0]] "\n"]
	set name	
}

# Chatbar is panned window with text widgets for incoming and outgoing messages
# It uses either BWidget or native Tk panedwindow widget (depends on Tk version)
if {[package vsatisfies [package present Tk] 8.4]} {
	option add *Panedwindow*in.txt.state disabled
	option add *Framein.height 200 widgetDefault
	option add *Frameout.height 100 widgetDefault
	option add *EditBar.AuxBar.cite.image cite widgetDefault

	image create bitmap cite -foreground brown -data {
		#define cite_width 16
		#define cite_height 16
		static unsigned char cite_bits[] = {
		   0,0,0x40,0,0xe0,0,0xf0,1,0xe0,3,0xc0,7,0x80,0x0f,0,0x1f,0,
		   0x1f,0x80,0x0f,0xc0,0x07,0xe0,3,0xf0,1,0xe0,0,0x40,0,0,0};
	}

	proc ChatBar {n} {
		panedwindow $n -orient vertical -showhandle 0
		foreach x {in out} {
			$n add [set $x [MFrame [frame $n.$x -class Frame$x] $x]]
		}
		list $n $in.txt $out.txt
	}
	proc CompoundSend {send uid} {
		if {[string match Contact:* $uid]} {
			set ref [ref $uid](Status)
			set cmd [nc StatusButton $send]
			set image img:[get $uid Status offline]
			trace variable $ref w $cmd
			bind $send <Destroy> [list trace vdelete $ref w $cmd]
		} else { set image img:group }
		$send configure -image $image
	}

	proc highlight {uid {auto 0} {stage 1}} {
		variable flash
		set w [WinByUid msg $uid]
		set vis [winfo viewable .all$w]
		if {$vis} { 
			if {[info exists flash($uid)]} {
				set stage 0
				after cancel flash($uid)
				unset flash($uid)
			} else return
		} else {
			if {!$auto && [info exists flash($uid)]} return
			set flash($uid) [after 300 [nc highlight $uid 1 [expr $stage^1]]]
		}
		set image [expr {$stage?"message":[get $uid Status offline]}]
		.all.btn$w configure -image img:$image
	}
	
} else {
	option add *PanedWindow.f0*txt.state disabled
	option add *Frame.height 200 widgetDefault
	proc ChatBar {n} {
		PanedWindow $n -side right -activator line
		foreach x {in out} {set $x [MFrame [$n add -weight 1] $x]}
		list $n $in.txt $out.txt
	}
	proc CompoundSend {args} {}

	proc highlight {uid} {
		set w [WinByUid msg $uid]
		set btn .all.btn$w
		set col [expr [winfo viewable .all$w]?"foreground":"highlight"]
		set color [option get $btn $col Foreground]
		if {$color!=""} { $btn configure -foreground $color }
	}
}

proc MFrame {fr x} {
	bindtags $fr [concat MsgFrame Frame$x [bindtags $fr]]
	[ScrolledWindow $fr.sw] setwidget [text $fr.txt -wrap word -bd 1]
	bindtags $fr.txt [concat MsgText Text$x [bindtags $fr.txt]]
	grid $fr.sw -sticky news
	grid propagate $fr 0
	foreach y {row column} {grid ${y}configure $fr 0 -weight 1}
	set fr
}

proc StatusButton {btn arr key args} { 
	upvar 1 ${arr}($key) new
	$btn configure -image img:$new 
}

# Tag message as acknowledged (different types) if such message exists
proc AckMessage {txt type uid msgid} {
	foreach {i1 i2} [$txt tag ranges $msgid] { $txt tag add $type $i1 $i2 }
}

proc CiteLast {in out multi} {
	foreach {i1 i2} [$in tag prevrange incoming end] {
		set msg [$in get $i1 $i2]
		set prefix ""
		if {$multi} {
			set tags [$in tag names $i1]
			if {[set pos [lsearch $tags Contact:*]]!=-1} {
				set prefix [get [lindex $tags $pos] Alias]
			} 
		}
		foreach ln [lrange [split $msg "\n"] 1 end] {
			$out insert end "${prefix}> $ln\n" citation
		}
	}	
}

# Retrieve message from text widget, format according to it's type and
# send to recipient
proc SendMessage {uid txt} {
	set msg [$txt get 1.0 "end -1 chars"]
	if {$msg=={}} return
	set var [namespace current]::${uid}-type
	if {[info exists $var]} { set type [set $var] } else { set type text }
	set m $msg
	if {[llength [info commands Form$type]]} {set m [Form$type $msg]}
	set tags [concat outgoing $type [Event Send $type $uid $m]]
	Event $uid|TaggedText Me [clock seconds] [list $msg] $tags
	$txt delete 1.0 end
}

# Send message to a group to all group memebers and return list of 
# all message IDs as result
handler Send multicaster {type uid msg} {
	if {![string match Group:* $uid]} { return -code continue }
	set res [list]
	foreach x [members $uid] { lappend res [Event Send $type $x $msg] }
	set res
}

proc FormURL {msg} {
	set lines [split $msg "\n"]
	list [join [lrange $lines 1 end] "\n"] [lindex $lines 0] 
}

proc DisplayMessage {txt uid time message {tags {}} {pos end} {see 1}} {
	$txt configure -state normal
	$txt tag delete last
	set fmt [option get $txt timeFormat TimeFormat]
	set stime "\[[clock format $time -format $fmt]\] "
	set alias [get $uid Alias]
	lappend tags $uid
	set message [concat [list $stime {header time} $alias {header alias}\
		":\n" header] $message]
	set msg [list]
	foreach {part t} $message {lappend msg $part [concat $t $tags last]}
	eval [list $txt insert $pos] $msg {\n\n}
	event generate $txt <<Message>>
	if {$see} {$txt see end}
	$txt configure -state disabled
	delayed [nc Raise [winfo toplevel $txt]]
	set txt
}

proc NewObjectDialog {} {
	set top .new-obj
	if {[winfo exists $top]} {
		raise $top
		return
	}
	toplevel $top -class NewObject
	grid [frame $top.content] -sticky news -padx 2 -pady 2
	grid [frame $top.btn] -sticky we -padx 2
	foreach x {row column} { grid ${x}configure $top 0 -weight 1 }
	grid columnconfigure $top.content 1 -weight 1
	button $top.btn.ok -text [mc Create] -default active
	button $top.btn.cancel -text [mc Cancel] -command [list destroy $top]
	grid $top.btn.ok $top.btn.cancel -sticky w -padx 10 -pady 2
	New:Contact $top.content $top.btn.ok
}
proc New:Contact {top ok} {
	wm title [winfo toplevel $top] "New ICQ contact"
	foreach x {"ICQ UIN" "Alias"} {
		label $top.lb$x -text [mc $x] -anchor e
		entry $top.en$x
		grid $top.lb$x $top.en$x -sticky we -padx 4 -pady 4
	}
}

# New objects dialogs
hook add:group [namespace code {
	NewObjDialog group ::AddObject {Enter group name:}
	}]
hook add:contact [namespace code {
	NewObjDialog contact ::AddObject {Enter UIN:}
	}]

proc NewObjDialog {type action caption args} {
	#NewObjectDialog
	set top ".new-$type"
	if {[catch {toplevel $top}]} return
	wm title $top "New $type"
	label $top.lb -text $caption
	entry $top.en
	pack $top.lb $top.en -fill x
	append action " $type \[$top.en get\]; destroy $top"
	pack [ButtonBar $top.buttons [list\
		[list Create $action <<Accept>>]\
		[list Close "destroy $top" <<Close>>]]] -side bottom
}

proc AddIncomingContacts {list} {
	foreach x [$list selection get] {
		set uid Contact:ICQ:$x
		if {![info exists [ref $uid]]} {
			new $uid [list Groups other\
				Alias [$list itemcget $x -data]]
		} 
		$list delete $x
	}
	update idletasks
	if {![llength [$list items]]} { destroy [winfo toplevel $list]}
}

handler {*|Send:*} popup {id} {
	foreach {uid action} [split $id |] break
	set var [ref $uid](Pending)
	if {![info exists $var]} return
	foreach item [set $var] { eval Event PendingIncoming $item }
	unset $var
	return -code break
} 0.10

handler {*|Send:text} dialog {id} { 
	set uid [lindex [split $id |] 0]
	Event $uid|show:text $uid 1
}

handler {Contact:ICQ:*|Send:contacts} SendContacts {id} {
	set uid [lindex [split $id |] 0]
	set top [WinByUid contacts $uid]
	if {![catch {raise $top}]} return
	toplevel $top -class AlicqContactsWindow
	wm title $top "Contacts for $uid"
	set lbox [ListBox $top.list -deltay 20 -dropenabled yes\
		-droptypes {TREE_NODE} -dropcmd [nc Drop]]
	$lbox insert end zz -fill blue -text "Drad contacts to send here"
	grid $lbox -sticky news
	grid [ButtonBar $top.btn [list\
		[list Send "Event Send contacts $uid \[[nc SendContactlist $lbox]\];destroy $top" <<Send>>]\
		[list Cancel "destroy $top"]\
	]]
	grid rowconfigure $top 0 -weight 1
	grid columnconfigure $top 0 -weight 1
}

handler {*|Send:authorization} authorization {id} { 
	Event Send authorization [lindex [split $id |] 0] grant
}

proc SendContactlist {name} {
	set res [list]
	foreach uid [$name items 1 end] {
		lappend res [lindex [split $uid :] 2] [get $uid Alias]
	}
	set res
}

proc Drop {widget src dest op datatype data} {
	set uid [[namespace parent]::tree::UidByNode $data]
	if {[string match Group:* $uid]} return
	if {[$widget exists $uid]} return
	$widget insert end $uid\
		-text [get $uid Alias]\
		-image img:[get $uid Status offline]

}

proc chunks {uid} {
	variable chunks
	if {![info exists chunks($uid)]} {
		set chunks($uid) [Event HistoryChunks $uid]
	}	
	set chunks($uid)
}

proc HistoryButton {button uid chunks} {
	if {[llength $chunks]} {
		$button configure -command [namespace code [subst {
			Event HistoryRequest $uid [lindex $chunks 0]
			HistoryButton $button $uid [list [lrange $chunks 1 end]]
		}]] -text "[mc History] (+[llength $chunks])"
	} else { $button configure -state disabled -text [mc "No history"] } 
}

proc FillHistory {txt type uid time msg tags chunk} {
	set mark history:$chunk
	set mid [lindex $tags 0]
	if {[string match *:* $mid]&&[llength [$txt tag ranges $mid]]} return
	if {[lsearch [$txt mark names] $mark]==-1} {
		$txt mark set $mark 1.0
	} else { $txt mark gravity $mark right }
	DisplayMessage $txt $uid $time $msg $tags $mark 0
	$txt mark gravity $mark left
	update idletasks
}

# Set message blcoker if Popup is false

handler ConfigLoaded onConfig {args} {
	trace variable [namespace current]::popup w [nc PopupChanged]
	trace variable [namespace current]::tabbing w [nc TabbingChanged]
	foreach x [select Contact] { MonitorAlias $x }
	hook New:Contact:* [nc MonitorAlias]
	PopupChanged
	TabbingChanged
}

proc MonitorAlias {uid args} {
	trace variable [ref $uid](Alias) w [nc AliasChanged $uid]
}

proc TabbingChanged {args} {
	variable tabbing
	set ns [namespace current]
	set mode [expr {[string is true $tabbing]?"Tabbed":"Single"}]
	interp alias {} ${ns}::UserDialog {} ${ns}::${mode}Dialog
	if {[string is true $tabbing]} {
		bind Textout <<ToTab>> [nc ToTab %W %K]
		bind Textout <<NextTab>> [nc NextTab %W 1]
		bind Textout <<PrevTab>> [nc NextTab %W -1]
	} else {
		foreach x {SelectTab NextTab PrevTab} { bind Textout <<$x>> {}}
	}
}

proc PopupChanged {args} {
	variable popup
	if {[string is true $popup]} {
		unhook Incoming [namespace current]::blocker
	} else {
		handler Incoming blocker {args} {
			if {[lindex $args 0]=="included"} return
			lappend [ref [lindex $args 1]](Pending) $args
			return -code break
		} 0.92
	}
}

# If alias changed, change message window caption as well
proc AliasChanged {uid ref field args} {
	upvar 1 ${ref}($field) alias
	variable tabbing
	set win [WinByUid msg $uid]
	if {[string is true $tabbing]} {
		set win .all.btn$win
		if {[winfo exists $win]} { $win configure -text $alias }
	} elseif {[winfo exists $win]} {
		wm title $win "$alias \([lrange [split $uid :] 1 end]\)"
	}
}

# Transform Incoming message to typed incoming:type messages
handler {Incoming PendingIncoming} transformer {type uid time message msgid} {
	set res [Event $uid|incoming:$type $uid $time $message $msgid]
	if {[winfo exists $res]} { return -code break }
} 0.90

hook Incoming [nc retransform 0] 0.94
hook PendingIncoming [nc retransform 1] 0.94

proc retransform {flag type uid time message msgid} {
	Event $uid|show:$type $uid $flag
	update idletasks
	Event $uid|incoming:$type $uid $time $message $msgid
}


handler *|incoming:text tagtext {uid time message id} { 
	Event $uid|TaggedText $uid $time [list $message {}] [list incoming $id]
} 0.90

handler *|incoming:URL tagurl {uid time msg msgid} {
	Event $uid|TaggedText $uid $time [list "[lindex $msg 0]\n" {}\
		[lindex $msg 1] url] [list incoming $msgid]
} 0.90

handler *|incoming:authrequest tagauthrequest {uid time msg msgid} {
	foreach {uin nick fname lname email xxx reason} $msg break
	Event $uid|TaggedText $uid $time [list [lindex $msg end] {} "\n" {}\
		grant {action action:grant} "\t" {}\
		deny {action action:deny}] [list incoming $msgid]
} 0.90

handler *|incoming:authorization tagautorization {uid time msg msgid} {
	Event $uid|TaggedText $uid $time [list "Autorization $msg"\
		authorization] [list incoming $msgid]
}

hook {*|show:text *|show:URL *|show:authrequest *|show:authorization} [namespace current]::UserDialog

# Display incoming contacts dialog
handler *|show:contacts ContactsDialog {uid args} {
	set UIN [lindex [split $uid :] 2]
	set top ".contacts-$UIN"
	if {![catch {raise $top}]} return

	toplevel $top -class AlicqContactsWindow
	wm title $top "Contacts from [get $uid Alias] ($UIN)"

	grid [ListBox $top.list -dragenabled 1] -sticky news
	$top.list bindText <1> [list $top.list selection add]
	$top.list bindText <3> [list $top.list selection remove]
	grid [ButtonBar $top.btn [list\
		[list Add [nc AddIncomingContacts $top.list]]\
		[list Close "destroy $top" <<Close>>]\
	]]
	grid rowconfigure $top 0 -weight 1
	grid columnconfigure $top 0 -weight 1
	set id [hook $uid|incoming:contacts [nc DisplayContacts $top.list] 0.9]
	bind $top.list <Destroy> [list unhook $id]
}

# Show incoming contacts in dialog
proc DisplayContacts {list uid time ContactList args} {
	foreach {uin alias} $ContactList {
		if {[$list exists $uin]} continue
		$list insert end $uin\
			-text "$alias (ICQ $uin)"\
			-image img:offline -data $alias
	}
	set list
}

proc ContactProperties {txt} {
	set tags [$txt tag names current]
	if {[set pos [lsearch $tags Contact:*]]!=-1} {
		set uid	[lindex $tags $pos]
		Event $uid|info:view $uid
	} elseif {[set pos [lsearch $tags Me]]!=-1} { Event info:update }
}

proc AuthorizeFromDialog {txt} {
	set tags [$txt tag names current]
	if {[set pos [lsearch $tags Contact:*]]!=-1} {
		set uid	[lindex $tags $pos]
	} else return
	if {[set pos [lsearch $tags action:*]]!=-1} {
		set action [lindex [split [lindex $tags $pos] :] end]
	} else return
	Event Send authorization $uid $action
}

proc CloseButton {name} {
	button $name
	$name configure -command [list destroy [winfo parent $name]]
	if {[$name cget -label]==""} { $name configure -label x}
}

# Event bindings
bind MsgFrame <Destroy> {+
	option add *[string range %W 1 end].height [winfo height %W]
}

bind Textin <Map> [namespace code {
	# Setup colors
	foreach x {incoming outgoing sent client server action authrequest} {
		set val [option get %W ${x}Color [string totitle $x]Color]
		if {$val!=""} { %W tag configure $x -foreground $val }
	}
	# Underline given alias if needed
	set underline [option get %W underlineTag UnderlineTag]
	if {$underline!=""} { %W tag configure $underline -underline yes}

	# Configure header options
	foreach x {background relief font} {
		set opt header[string totitle $x]
		set val [option get %W $opt [string totitle $opt]]
		if {$val!=""} { %W tag configure header -$x $val}
	}
	foreach {x y} {alias ContactProperties action AuthorizeFromDialog} {
		%W tag bind $x <Enter> { %W configure -cursor hand2 }
		%W tag bind $x <Leave> { %W configure -cursor {} }
		%W tag bind $x <1> [namespace code [list $y %W]]
	}
	%W tag configure action -underline 1
	%W tag bind incoming <3> {.incoming post %%X %%Y}
}]

bind Textout <Map> {
	%W tag configure citation -foreground brown
}

menu .incoming -tearoff no
.incoming add command -label "Cite" 

