/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef __CAPTURE_H
#define __CAPTURE_H

#ifndef WIN32
#include <stdint.h>
#include <pcap.h>
#else
#include <windows.h>
#endif //WIN32

#include "PacketSource.h"
#include "Packet.h"

//data that is the same for all types of packets
typedef struct PacketInfo_t {
  int channel;           //receive channel for the packet
  char *name;            //ssid name if present
  const unsigned char *raw;    //pointer to the start of the raw packet data
  time_t rxTime;
  int wep;               //was this packet encrypted?
  Packet *pack;          //bundled packet data for 'interesting' packets
} PacketInfo;

//data that varies based on what type of packet it is
typedef struct CaptureRec_t {
  PacketInfo *pInfo;     //pointer to common data
  const unsigned char *bssid;  //pointer to the AP bssid
  const unsigned char *sta;    //pointer to the sta 
  const unsigned char *iv;     //pointer to the frame body
} CaptureRec;

typedef struct CaptureArg_t {
   PacketSource *src;
   int offset;
} CaptureArg;

#define PRISM 0
#define ORINOCO 1
#define OTHER 2

#define P80211_IOCTL_MAGIC (0x4a2d464dUL)

#ifndef WIN32
#define WLAN_DEVNAMELEN_MAX 16
#else
#define WLAN_DEVNAMELEN_MAX 128
#endif

extern int useOrinoco;
extern char dev[WLAN_DEVNAMELEN_MAX];
extern char errbuf[PCAP_ERRBUF_SIZE];

#ifndef WIN32
void *capture(void *arg);
#else
DWORD WINAPI capture(void *arg);
#endif

int setChannel( unsigned char channel );
void loadOpts(void);
void saveOpts(void);

#define P80211ENUM_truth_false			0
#define P80211ENUM_truth_true			1
#define P80211ENUM_msgitem_status_no_value		1

/*===============================================================*/
/* Types */

/*----------------------------------------------------------------*/
/* A ptr to the following structure type is passed as the third */
/*  argument to the ioctl system call when issuing a request to */
/*  the p80211 module. */

typedef struct p80211ioctl_req
{
	char 	name[WLAN_DEVNAMELEN_MAX];
	void	*data;
	unsigned int magic;
	unsigned short len;
	unsigned int result;
} p80211ioctl_req_t;

typedef struct p80211msg
{
	unsigned int msgcode;
	unsigned int msglen;
	unsigned char devname[WLAN_DEVNAMELEN_MAX];
} p80211msg_t;

/* message data item for INT, BOUNDEDINT, ENUMINT */
typedef struct p80211item_uint32
{
	unsigned int		did;
	unsigned short		status;
	unsigned short		len;
	unsigned int		data;
}  p80211item_uint32_t;

//THESE constants reflect the wlan-ng 0.1.13 values
#define DIDmsg_lnxreq_wlansniff 0x0083
#define DIDmsg_lnxreq_wlansniff_enable 0x1083
#define DIDmsg_lnxreq_wlansniff_channel 0x2083
#define DIDmsg_lnxreq_wlansniff_prismheader 0x3083
#define DIDmsg_lnxreq_wlansniff_keepwepflags 0x4083
#define DIDmsg_lnxreq_wlansniff_resultcode 0x5083

typedef struct p80211msg_lnxreq_wlansniff
{
	unsigned int		msgcode; //		__attribute__ ((packed));
	unsigned int		msglen; //		__attribute__ ((packed));
	unsigned char		    devname[WLAN_DEVNAMELEN_MAX]; //	__attribute__ ((packed));
	p80211item_uint32_t	enable; //	       __attribute__ ((packed));
	p80211item_uint32_t	channel; //	       __attribute__ ((packed));
	p80211item_uint32_t	prismheader; //	   __attribute__ ((packed));
	p80211item_uint32_t	keepwepflags; //   __attribute__ ((packed));
	p80211item_uint32_t	resultcode; //     __attribute__ ((packed));
} /* __attribute__ ((packed)) */ p80211msg_lnxreq_wlansniff_t;

#define DIDmsg_lnxind_wlansniffrm 0x0041
#define DIDmsg_lnxind_wlansniffrm_hosttime 0x1041
#define DIDmsg_lnxind_wlansniffrm_mactime 0x2041
#define DIDmsg_lnxind_wlansniffrm_channel 0x3041
#define DIDmsg_lnxind_wlansniffrm_rssi 0x4041
#define DIDmsg_lnxind_wlansniffrm_sq 0x5041
#define DIDmsg_lnxind_wlansniffrm_signal 0x6041
#define DIDmsg_lnxind_wlansniffrm_noise 0x7041
#define DIDmsg_lnxind_wlansniffrm_rate 0x8041
#define DIDmsg_lnxind_wlansniffrm_istx 0x9041
#define DIDmsg_lnxind_wlansniffrm_frmlen 0xA041

typedef struct p80211msg_lnxind_wlansniffrm
{
	unsigned int		msgcode;
	unsigned int		msglen;
	unsigned char		    devname[WLAN_DEVNAMELEN_MAX];
	p80211item_uint32_t	hosttime;
	p80211item_uint32_t	mactime;
	p80211item_uint32_t	channel;
	p80211item_uint32_t	rssi;
	p80211item_uint32_t	sq;
	p80211item_uint32_t	signal;
	p80211item_uint32_t	noise;
	p80211item_uint32_t	rate;
	p80211item_uint32_t	istx;
	p80211item_uint32_t	frmlen;
}  p80211msg_lnxind_wlansniffrm_t;


#endif
