/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Copyright (c) 2003, Snax
*/

#include <fcntl.h>
#include "PacketSource.h"

#ifdef WIN32
#include <sys/timeb.h>

void *packetCallback(WlanPacket *ptr, unsigned int len, int val2, __int64 timeStamp, 
                     int val3, void *userPtr){
   //return NULL to pause the thread
   struct timeb timev;

   PacketSource *src = (PacketSource*) userPtr;
   PacketNode *temp = (PacketNode *) malloc(sizeof(PacketNode));

   len = len - 4;
   temp->pkt = ptr;
   temp->nxt = NULL;
   temp->hdr.caplen = ((unsigned int) len) < src->snaplen ? len : src->snaplen;
   temp->hdr.len = len;
   ftime(&timev);
   temp->hdr.ts.tv_sec = timev.time;
   temp->hdr.ts.tv_usec = timev.millitm * 1000;

   WaitForSingleObject(src->queueSem, INFINITE);
   if (src->pktHead) {
      src->pktTail->nxt = temp;
   }
   else {
      src->pktHead = temp;
   }
   src->pktTail = temp;
   ReleaseSemaphore(src->queueSem, 1, NULL);
   ReleaseSemaphore(src->qtySem, 1, NULL);
   return (void*) 1;
}

#endif


PacketSource *openPacketSource(char *dev, int snaplen, int promisc, int to_ms, char *errbuf) {
   PacketSource *src = (PacketSource*) calloc(sizeof(PacketSource), 1);
#ifndef WIN32
   src->pcap = pcap_open_live(dev, snaplen, promisc, to_ms, errbuf);
   src->dlType = pcap_datalink(src->pcap);
#else
   HANDLE hAdapter = 0;
   LONG result;
//char adapterName[] = "\\Device\\{6CB9D388-3838-4282-9B9D-54A90338FC8A}";

   //init PacketSource structure including packet queue
   src->snaplen = snaplen;
   src->pbuf = (unsigned char*) malloc(snaplen);
   src->dlType = DLT_IEEE802_11;
   src->pktHead = src->pktTail = NULL;
   src->queueSem = CreateSemaphore(NULL, 1, 1, NULL);
   src->qtySem = CreateSemaphore(NULL, 0, 0x7FFFFFFF, NULL);
   hAdapter = WincapOpenAdapter(dev);
   src->ctx = WincapCreateContext(hAdapter, packetCallback, 0xFA000, SCAN, (void*)src);
   result = WincapStart(src->ctx);
#endif
   return src;
}

void closePacketSource(PacketSource *src) {
   if (!src) return;
#ifndef WIN32
   pcap_close(src->pcap);
#else
   if (src->readfd > 0) {
      close(src->readfd);
   }
   else {
      WincapClose(src->ctx->hFileAdapter);
      WincapDestroyContext(src->ctx);
      CloseHandle(src->queueSem);
      CloseHandle(src->qtySem);
   }
   free(src->pbuf);
#endif
   free(src);
}

PacketSource *openOfflinePacketSource(const char *name, char *errbuf) {
   PacketSource *src = (PacketSource*) calloc(sizeof(PacketSource), 1);
#ifndef WIN32
   src->pcap = pcap_open_offline(name, errbuf);
   src->dlType = pcap_datalink(src->pcap);
#else
   struct pcap_file_header hdr;
   src->readfd = _open(name, O_RDONLY);
   read(src->readfd, &hdr, sizeof(hdr));
   //check magic
   src->dlType = hdr.linktype;
   src->pbuf = (unsigned char*) malloc(hdr.snaplen);
#endif
   return src;
}

void openPacketDumpFile(PacketSource *src, char *name) {
#ifndef WIN32
   src->dumpFile = pcap_dump_open(src->pcap, name);
#else
   struct pcap_file_header hdr;
   src->dumpfd = open(name, O_WRONLY | O_CREAT, 0644);
   hdr.magic = 0xA1B2C3D4;
   hdr.version_major = 4;
   hdr.version_minor = 2;
   hdr.thiszone = 0;
   hdr.sigfigs = 0;
   hdr.snaplen = src->snaplen;
   hdr.linktype = src->dlType;
   write(src->dumpfd, &hdr, sizeof(hdr));
#endif
   src->dump = 1;
}

void closePacketDumpFile(PacketSource *src) {
#ifndef WIN32
   pcap_dump_close(src->dumpFile);
#else
   close(src->dumpfd);
#endif
   src->dump = 0;
}

const unsigned char *nextPacket(PacketSource *src, struct pcap_pkthdr *hdr) {
#ifndef WIN32
   return pcap_next(src->pcap, hdr);
#else
   if (src->readfd) {
      read(src->readfd, hdr, sizeof(struct pcap_pkthdr));
      read(src->readfd, src->pbuf, hdr->caplen);
   }
   else {
      PacketNode *temp;
      WaitForSingleObject(src->qtySem, INFINITE);
      WaitForSingleObject(src->queueSem, INFINITE);
      //pull a packet off the queue
      temp = src->pktHead;
      src->pktHead = src->pktHead->nxt;
      memcpy(src->pbuf, temp->pkt->data, temp->hdr.caplen);
      *hdr = temp->hdr;
      free(temp);
      ReleaseSemaphore(src->queueSem, 1, NULL);
   }
   return src->pbuf;
#endif
}

void dumpPacket(PacketSource *src, struct pcap_pkthdr *hdr, const unsigned char *pkt) {
#ifndef WIN32
   pcap_dump((unsigned char*)src->dumpFile, hdr, pkt);
#else
   write(src->dumpfd, hdr, sizeof(struct pcap_pkthdr));
   write(src->dumpfd, pkt, hdr->caplen);
#endif
}
