/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <memory>
#include "gcs-debug.h"
#include "core/gcs-color.h"
#include "gcs-cellrendererswatch.h"

namespace gcs
{
    namespace Widgets
    {

        CellRendererSwatch::CellRendererSwatch(void) :
            Glib::ObjectBase(typeid(CellRendererSwatch)),
            Gtk::CellRenderer(),
            m_property_size(*this, "size", 18),
            m_property_color(*this, "color", Color::create("#FFFFFF")),
            m_property_border_width(*this, "border-width", 1),
            m_property_border_color(*this, "border-color", Color::create("#000000"))
        {
            property_xpad() = 3;
            property_ypad() = 3;
            property_yalign() = 0.5;
        }


        void CellRendererSwatch::get_size_vfunc(Gtk::Widget& widget,
                const Gdk::Rectangle* cell_area,
                int* x_offset, int* y_offset,
                int* width, int* height) const
        {
            const unsigned int size = property_size();
            //debug("Calling get_size_vfunc");
            if (width)
            {
                *width = 2 * property_xpad() + size;
            }

            if (height)
            {
                *height = 2 * property_ypad() + size;
            }

            if (cell_area)
            {
                if (x_offset)
                {
                    // property_xalign() can be between 0 and 1, inclusive
                    *x_offset = static_cast<int>(property_xalign() *
                            (cell_area->get_width() - *width));
                    *x_offset = std::max(0, *x_offset);
                }

                if (y_offset)
                {
                    *y_offset = static_cast<int>(property_yalign() *
                            (cell_area->get_height() - *height));
                    *y_offset = std::max(0, *y_offset);
                }
            }
        }


        void CellRendererSwatch::render_vfunc(
                const Glib::RefPtr<Gdk::Drawable>& window,
                Gtk::Widget& widget,
                const Gdk::Rectangle& background_area,
                const Gdk::Rectangle& cell_area,
                const Gdk::Rectangle& expose_area,
                Gtk::CellRendererState flags)
        {
            //debug("Calling render_vfunc");
            const unsigned int cell_xpad = property_xpad();
            const unsigned int cell_ypad = property_ypad();

            int x_offset = 0, y_offset = 0, width = 0, height = 0;
            this->get_size(widget, cell_area, x_offset, y_offset, width, height);

            width -= cell_xpad * 2;
            height -= cell_ypad * 2;

            if (width <= 0 || height <= 0)
                return;

            Gtk::StateType state = Gtk::STATE_INSENSITIVE;

            if (flags & Gtk::CELL_RENDERER_SELECTED != 0)
            {
                state = (widget.has_focus())
                    ? Gtk::STATE_SELECTED
                    : Gtk::STATE_ACTIVE;
            }

            //Cast the drawable to a Window.
            Glib::RefPtr<Gdk::Window> window_casted =
                Glib::RefPtr<Gdk::Window>::cast_dynamic<>(window);
            if(window)
            {
                Glib::RefPtr<Gdk::Colormap> refCmap = widget.get_default_colormap();
                refCmap->alloc_color(m_property_color.get_value()->gdk());
                Glib::RefPtr<Gdk::GC> refGC = Gdk::GC::create(window);
                Glib::RefPtr<Gdk::GC> refBgGC = Gdk::GC::create(window);
                refBgGC->set_line_attributes(property_border_width(),
                        Gdk::LINE_SOLID, Gdk::CAP_BUTT, Gdk::JOIN_MITER);
                refGC->set_foreground(m_property_color.get_value()->gdk());
                window->draw_rectangle(refGC,
                        true, cell_area.get_x() + x_offset + cell_xpad,
                        cell_area.get_y() + y_offset + cell_ypad,
                        width, height);
                window->draw_rectangle(refBgGC,
                        false, cell_area.get_x() + x_offset + cell_xpad,
                        cell_area.get_y() + y_offset + cell_ypad,
                        width, height);
            }
        }

        CellRendererSwatch::~CellRendererSwatch(void)
        {}

        Glib::PropertyProxy<gint>
            CellRendererSwatch::property_size(void)
            {
                return m_property_size.get_proxy();
            }

        Glib::PropertyProxy_ReadOnly<gint>
            CellRendererSwatch::property_size(void) const
            {
                return Glib::PropertyProxy_ReadOnly<gint>(this, "size");
            }

        Glib::PropertyProxy<ColorPtr>
            CellRendererSwatch::property_color(void)
            {
                return m_property_color.get_proxy();
            }

        Glib::PropertyProxy_ReadOnly<ColorPtr>
            CellRendererSwatch::property_color(void) const
            {
                return Glib::PropertyProxy_ReadOnly<ColorPtr>(this, "color");
            }

        Glib::PropertyProxy<gint>
            CellRendererSwatch::property_border_width(void)
            {
                return m_property_border_width.get_proxy();
            }
        Glib::PropertyProxy_ReadOnly<gint>
            CellRendererSwatch::property_border_width(void) const
            {
                return Glib::PropertyProxy_ReadOnly<gint>(this,
                        "border-width");
            }

        Glib::PropertyProxy<ColorPtr>
            CellRendererSwatch::property_border_color(void)
            {
                return m_property_border_color.get_proxy();
            }
        Glib::PropertyProxy_ReadOnly<ColorPtr>
            CellRendererSwatch::property_border_color(void) const
            {
                return Glib::PropertyProxy_ReadOnly<ColorPtr>(this,
                        "border-color");
            }



    } // namespace Widgets
} // namespace gcs

