/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <stdlib.h>
#include <string.h>
#include <X11/Xatom.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include "aewm.h"
#include "atom.h"

static int send_xmessage(Window w, Atom a, long x);

client_t *find_client(Window w, int mode)
{
    client_t *c;

    if (mode == MATCH_FRAME) {
        for (c = head; c; c = c->next)
            if (c->frame == w)
                return c;
    } else /* mode == MATCH_WINDOW */ {
        for (c = head; c; c = c->next)
            if (c->win == w)
                return c;
    }

    return NULL;
}

/* For a regular window, c->trans is None (false), and we include enough space
 * to draw the title. For a transient window we just make a small strip (based
 * on the font height). */

int theight(client_t *c)
{
    int font_height;

    if (c && c->titled) {
#ifdef XFT
        font_height = xftfont->ascent + xftfont->descent;
#else
        font_height = font->ascent + font->descent;
#endif
        return c->trans ? font_height / 2 : font_height + 2*opt_pad + BW(c);
    } else {
        return 0;
    }
}

int set_wm_state(client_t *c, unsigned long state)
{
    return set_atoms(c->win, wm_state, wm_state, &state, 1);
}

/* This can be called before we map, so check c->frame. If c->frame is still
 * None, we merely set the variables, and then later (in reparent) adjust
 * window and/or frame sizes based on them. */

void handle_net_wm_state_item(client_t *c, Atom state)
{
    if (!c->shaded && state == net_wm_state_shaded) {
        if (c->frame)
            shade_win(c);
        else
            c->shaded = 1;
    } else if (!c->zoomed && (state == net_wm_state_mh ||
                state == net_wm_state_mv)) {
        if (c->frame)
            zoom_win(c);
        else
            c->zoomed = 1;
    }
}

/* If we frob the geom for some reason, we need to inform the client. (In an
 * OO language you could force this, perhaps.) */

void send_config(client_t *c)
{
    XConfigureEvent ce;

    ce.type = ConfigureNotify;
    ce.event = c->win;
    ce.window = c->win;
    ce.x = c->geom.x;
    ce.y = c->geom.y;
    ce.width = c->geom.w;
    ce.height = c->geom.h;
    ce.border_width = 0;
    ce.above = None;
    ce.override_redirect = 0;

    XSendEvent(dpy, c->win, False, StructureNotifyMask, (XEvent *)&ce);
}

/* The name of this function is a little misleading: if the client doesn't
 * listen to WM_DELETE then we just terminate it with extreme prejudice. */

void send_wm_delete(client_t *c)
{
    int i, n, found = 0;
    Atom *protocols;

    if (XGetWMProtocols(dpy, c->win, &protocols, &n)) {
        for (i=0; i<n; i++) if (protocols[i] == wm_delete) found++;
        XFree(protocols);
    }
    if (found)
        send_xmessage(c->win, wm_protos, wm_delete);
    else
        XKillClient(dpy, c->win);
}

static int send_xmessage(Window w, Atom a, long x)
{
    XClientMessageEvent e;

    e.type = ClientMessage;
    e.window = w;
    e.message_type = a;
    e.format = 32;
    e.data.l[0] = x;
    e.data.l[1] = CurrentTime;

    return XSendEvent(dpy, w, False, NoEventMask, (XEvent *)&e);
}

/* I've decided to carefully ignore any errors raised by this function, rather
 * that attempt to determine asychronously if a window is ``valid''. Xlib
 * calls should only fail here if that a window has removed itself completely
 * before the Unmap and Destroy events get through the queue to us. It's not
 * pretty.
 *
 * The 'withdrawing' argument specifes if the client is actually destroying
 * itself or being destroyed by us, or if we are merely cleaning up its data
 * structures when we exit mid-session. */

void remove_client(client_t *c, int mode)
{
    client_t *p;

    XGrabServer(dpy);
    XSetErrorHandler(ignore_xerror);

#ifdef DEBUG
    dump_title(c, "removing", 'r');
    dump_removal(c, mode);
#endif

    if (mode == REMOVE_WITHDRAW) {
        set_wm_state(c, WithdrawnState);
    } else /* mode == REMOVE_REMAP */ {
        if (c->zoomed) {
            c->geom.x = c->save.x;
            c->geom.y = c->save.y;
            c->geom.w = c->save.w;
            c->geom.h = c->save.h;
            XResizeWindow(dpy, c->win, c->geom.w, c->geom.h);
        }
        XMapWindow(dpy, c->win);
    }

    remove_atom(root, net_client_list, XA_WINDOW, c->win);
    remove_atom(root, net_client_stack, XA_WINDOW, c->win);

    if (c->bordered)
        XSetWindowBorderWidth(dpy, c->win, 1);
#ifdef XFT
    if (c->xftdraw)
        XftDrawDestroy(c->xftdraw);
#endif

    XReparentWindow(dpy, c->win, root, c->geom.x, c->geom.y);
    XRemoveFromSaveSet(dpy, c->win);
    XDestroyWindow(dpy, c->frame);

    if (head == c)
        head = c->next;
    else for (p = head; p && p->next; p = p->next)
        if (p->next == c)
            p->next = c->next;

    if (c->name) XFree(c->name);
    if (c->size) XFree(c->size);
    free(c);

    XSync(dpy, False);
    XSetErrorHandler(handle_xerror);
    XUngrabServer(dpy);
}

/* I've changed this to just clear the window every time. The amount of
 * ``flicker'' is basically imperceptable. Also, we might be drawing an
 * anti-aliased font with Xft, in which case we always have to clear to draw
 * the text properly. This allows us to simplify handle_property_change as
 * well. */

void redraw(client_t *c)
{
    if (c && c->titled) {
        XClearWindow(dpy, c->frame);
        if (!c->shaded) XDrawLine(dpy, c->frame, border_gc,
            0, theight(c) - BW(c) + BW(c)/2,
            c->geom.w, theight(c) - BW(c) + BW(c)/2);
        XDrawLine(dpy, c->frame, border_gc,
            c->geom.w - theight(c) + BW(c)/2, 0,
            c->geom.w - theight(c) + BW(c)/2, theight(c));

        if (!c->trans && c->name) {
#ifdef XFT
#ifdef X_HAVE_UTF8_STRING
            XftDrawStringUtf8(c->xftdraw, &xft_fg,
                xftfont, opt_pad, opt_pad + xftfont->ascent,
                (unsigned char *)c->name, strlen(c->name));
#else
            XftDrawString8(c->xftdraw, &xft_fg,
                xftfont, opt_pad, opt_pad + xftfont->ascent,
                (unsigned char *)c->name, strlen(c->name));
#endif
#else
#ifdef X_HAVE_UTF8_STRING
            Xutf8DrawString(dpy, c->frame, font_set, string_gc,
                opt_pad, opt_pad + font->ascent,
                c->name, strlen(c->name));
#else
            XDrawString(dpy, c->frame, string_gc,
                opt_pad, opt_pad + font->ascent,
                c->name, strlen(c->name));
#endif
#endif
        }
    }
}

/* The frame is bigger than the client window. Which direction it extends
 * outside of the theoretical client geom is determined by the window gravity.
 * The default is NorthWest, which means that the top left corner of the frame
 * stays where the top left corner of the client window would have been, and
 * the client window moves down. For SouthEast, etc, the frame moves up. For
 * Static the client window must not move (same result as South), and for
 * Center the center point of the frame goes where the center point of the
 * unmanaged client window was. */

geom_t frame_geom(client_t *c)
{
    geom_t f = c->geom;
    int gravity = (c->size->flags & PWinGravity) ?
        c->size->win_gravity : NorthWestGravity;

    if (c->shaded)
        f.h = theight(c) - BW(c);
    else
        f.h += theight(c);

    switch (gravity) {
        case CenterGravity:
            f.y -= theight(c)/2;
            break;
        case SouthWestGravity:
        case SouthEastGravity:
        case SouthGravity:
        case StaticGravity:
            f.y -= theight(c); break;
    }

    return f;
}

/* Well, the man pages for the shape extension say nothing, but I was able to
 * find a shape.PS.Z on the x.org FTP site. What we want to do here is make
 * the window shape be a boolean OR (or union) of the client's shape and our
 * titlebar. The titlebar requires both a bound and a clip because it has a
 * border; the server will paint the border in the region between the two. */

#ifdef SHAPE
void set_shape(client_t *c)
{
    int n, order;
    XRectangle temp, *rects;

    rects = XShapeGetRectangles(dpy, c->win, ShapeBounding, &n, &order);

    if (n > 1) {
        XShapeCombineShape(dpy, c->frame, ShapeBounding,
            0, theight(c), c->win, ShapeBounding, ShapeSet);
        temp.x = -BW(c);
        temp.y = -BW(c);
        temp.width = c->geom.w + 2*BW(c);
        temp.height = theight(c) + BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeBounding,
            0, 0, &temp, 1, ShapeUnion, YXBanded);
        temp.x = 0;
        temp.y = 0;
        temp.width = c->geom.w;
        temp.height = theight(c) - BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeClip,
            0, theight(c), &temp, 1, ShapeUnion, YXBanded);
        c->shaped = 1;
    } else if (c->shaped) {
        /* I can't find a ``remove all shaping'' function... */
        temp.x = -BW(c);
        temp.y = -BW(c);
        temp.width = c->geom.w + 2*BW(c);
        temp.height = c->geom.h + theight(c) + 2*BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeBounding,
            0, 0, &temp, 1, ShapeSet, YXBanded);
    }

    XFree(rects);
}
#endif
