#
# $Id: s.UserAdd.pm 1.38 03/11/09 23:04:30+01:00 robert@zaron.debian.pl $
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License Version 2 as
#  published by the Free Software Foundation.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
#  USA.
#
# This file is part of adduser-ng program.
#
# Copyright (C) 2003 Bartosz Oler <liar@bzimage.us>
# Copyright (C) 2003 Robert Olejnik <robert@zaron.debian.pl>
#

package AddUser::plugins::UserAdd;

use strict;
use AddUser::StdLib;
use File::Path qw{mkpath rmtree};

my $_PLUGIN = 'UserAdd';
my $_APIVERSION = 0x1;


# new
# {{{

sub new {
	my ($c, %args) = @_;
	my $class = ref($c) || $c;

	$args{opts} = {
		'comment' => '',
		'home_dir' => '',
		'home_dir_mode' => '0711',
		'expire_date' => '',
		'inactive_days' => -1,
		'initial_group' => 1,
		'extra_groups' => '',
		'create_home_dir' => 'no',
		'skeleton_dir' => '/etc/skel',
		'allow_duplicate_uid' => 'no',
		'password' => '!',
		'shell' => '',
		'uid' => ''};

        $args{APIVERSION} = $_APIVERSION;
        $args{NAME} = $_PLUGIN;

	bless \%args, $class;
}

# }}}




# configure
# =========
#
# {{{

sub configure {
	my $self     = shift;
        my $conf     = $self->{plugins_options}{GroupConfig};
        my $ui       = $self->{plugins_options}{UI};
        my $verbose  = $self->{plugins_options}{verbose};
        my $keywords = $self->{plugins_options}{keywords};
	my $docs     = &get_documentation($self->{plugins_options}{documentation_dir}, $_PLUGIN);

        my $value;

	$self->{plugins_options}{docs} = $docs->{options}{option};
	$ui->display_error("$_PLUGIN", $docs->{description}."\n") if $verbose;

	wrapper_getlong($self, $_PLUGIN, 'comment');
	wrapper_getlong($self, $_PLUGIN, 'expire_date');
	wrapper_getlong($self, $_PLUGIN, 'inactive_days');
	wrapper_getlong($self, $_PLUGIN, 'initial_group');
	wrapper_getlong($self, $_PLUGIN, 'extra_groups');
	if (wrapper_getlong($self, $_PLUGIN, 'home_dir')
	    and wrapper_getsingle($self, $_PLUGIN, 'create_home_dir', ('no', 'yes')) eq 'yes') {
		wrapper_getlong($self, $_PLUGIN, 'skeleton_dir');
		while (1) {
			last if wrapper_getlong($self, $_PLUGIN, 'home_dir_mode') =~ /^\d+$/;
		}
	}
	wrapper_getsingle($self, $_PLUGIN, 'allow_duplicate_uid', ('no', 'yes'));
	wrapper_getpassword($self, $_PLUGIN, 'password');
	wrapper_getlong($self, $_PLUGIN, 'shell');
	wrapper_getlong($self, $_PLUGIN, 'uid');

	open(FH, '<', '/etc/passwd') || return $ERRNO{'ERROR'};
	if (scalar(grep {/^$self->{plugins_options}{login}:/} <FH>)) {
		close FH;
		return $ERRNO{'DONE'};
	}
	close FH;

	return $ERRNO{'OK'};
}

# }}}




# execute
# =======
#
# {{{

sub execute {
	my $self    = shift;
        my $conf    = $self->{plugins_options}{GroupConfig};
        my $ui      = $self->{plugins_options}{UI};
        my $verbose = $self->{plugins_options}{verbose};

        my @cmd;


	if ($self->{opts}{comment}) {
		push(@cmd, ('-c', $self->{opts}{comment}));
	}
	if ($self->{opts}{home_dir}) {
		push(@cmd, ('-d', $self->{opts}{home_dir}));
	}
	if ($self->{opts}{expire_date}) {
		push(@cmd, ('-e', $self->{opts}{expire_date}));
	}
	if ($self->{opts}{inactive_days}) {
		push(@cmd, ('-f', $self->{opts}{inactive_days}));
	}
	if ($self->{opts}{initial_group}) {
		push(@cmd, ('-g', $self->{opts}{initial_group}));
	}
	if ($self->{opts}{extra_groups}) {
		push(@cmd, ('-G', $self->{opts}{extra_groups}));
	}
	if ($self->{opts}{create_home_dir} eq 'yes') {
		push(@cmd, ('-m'));
	}
	if ($self->{opts}{home_dir} and $self->{opts}{create_home_dir} eq 'yes'
	    and $self->{opts}{skeleton_dir}) {
		push(@cmd, ('-k', $self->{opts}{skeleton_dir}));
	}
	if ($self->{opts}{allow_duplicate_uid} eq 'yes') {
		push(@cmd, '-o');
	}
	if ($self->{opts}{password}) {
		push(@cmd, ('-p', $self->{opts}{password}));
	}
	if ($self->{opts}{shell}) {
		push(@cmd, ('-s', $self->{opts}{shell}));
	}
	if ($self->{opts}{uid}) {
		push(@cmd, ('-u', $self->{opts}{uid}));
	}

        push(@cmd, $self->{plugins_options}{login});

	$ui->display("Command to run", "useradd ".join " ", @cmd) if $verbose;

	if ($self->{opts}{home_dir} and $self->{opts}{create_home_dir} eq 'yes') {
		my $home_dir = $self->{opts}{home_dir};
		$home_dir =~ s#/\w+$#/#g;
		eval { mkpath($home_dir, 0, oct($self->{opts}{home_dir_mode})) };
		if ($@) {
			$ui->display_error($_PLUGIN, $@) if $verbose;
			return $ERRNO{'ERROR'};
		}

	}

	if (system("useradd", @cmd) == 0) {
		if (chmod oct($self->{opts}{home_dir_mode}), $self->{opts}{home_dir}) {
		    return $ERRNO{'OK'};
		} else {
		    $ui->display_error("UserAdd ERROR", "Can't change permitions to newly created users directory");
		    return $ERRNO{'ERROR'};
		}   
	} else {
		return $ERRNO{'ERROR'};
	}
}

# }}}


# rollback
# ========
#
# {{{

sub rollback {
	my $self    = shift;
        my $conf    = $self->{plugins_options}{GroupConfig};
        my $ui      = $self->{plugins_options}{UI};
        my $verbose = $self->{plugins_options}{verbose};

	if ($self->{opts}{home_dir} and $self->{opts}{create_home_dir} eq 'yes') {
		eval { rmtree($self->{opts}{home_dir}, 0, 0) };
		if ($@) {
			$ui->display_error($_PLUGIN, $@) if $verbose;
		}
	}

        my @cmd = ('userdel', $self->{plugins_options}{login});

        $ui->display("Rollback...", "Doing a rollback.");
        $ui->display("Command to run", "@cmd") if $verbose;

        system(@cmd);

	#
	# We don't care about result of the rollback.
	#
	return $ERRNO{'OK'};
}

# }}}

1;
