/*
	Actiona
	Copyright (C) 2005 Jonathan Mercier-Ganady

	Actiona is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Actiona is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program. If not, see <http://www.gnu.org/licenses/>.

	Contact : jmgr@jmgr.info
*/
/* keysym_map.h - Maps the unicode number of charactes to X11 keysym codes.
 *                The first element in the array is unicode number 0x0100.
 *                Latin-1 characters (keysym = unicode_number <= 0xff) and newer
 *                characters (keysym = unicode_number + 0x01000000) are not included.
 *
 * Marco Steinacher <marco@websource.ch> - 12.04.2009
 *
 * The contents below have been created from /usr/include/X11/keysymdef.h with the following
 * perl script:
---------------
#!/usr/bin/perl -w
$max=0;
while ($l = <>) {
	if ($l =~ m/^#define\s+\S+\s+0x([0-9a-f]{4})\s*\/\*[ (]*U\+([0-9a-f]{4})/i) {
		$shifted = hex($2) - hex("0x0100");
		if (($shifted >= 0) && !defined($map[$shifted])) {
			$map[$shifted] = $1;
			$max=$shifted if ($shifted>$max);
		}
	}
}
print "#define WCHAR_TO_KEYSYM_MAP_SIZE ".($max+1)."\n";
print "uint16_t wchar_to_keysym_map[] = { ";
for($u=0;$u<=$max;$u++) {
	print "," if ($u>0);
	print "\n" if ($u%16==0);
	if (defined($map[$u])) {
		print "0x".$map[$u];
	} else {
		print "0";
	}
}
print "\n};\n";
---------------
 */
/* multikey_map.h - Multi_key combinations for some characters:
 *                  <Character> = Multi_key + <Fist> + <Second>
 *                  The unicode numbers of <Character>, <Fist>,
 *                  and <Second> are in the arrays below.
 *
 * Marco Steinacher <marco@websource.ch> - 12.04.2009
 *
 * Combinations that don't start with Multi_key or require more
 * than two characters are currently not supported.
 *
 * The contents below have been created from /usr/share/X11/locale/en_US.UTF-8/Compose
 * and /usr/include/X11/keysymdef.h with the following (very slow) perl script:
---------------
#!/usr/bin/perl -w
$compose_file = "/usr/share/X11/locale/en_US.UTF-8/Compose";
$keysym_file  = "/usr/include/X11/keysymdef.h";
open(FH,"<:encoding(UTF-8)",$compose_file);
while ($l = <FH>) {
	if ($l =~ m/^\s*<Multi_key>\s+<(\S+)>\s+<(\S+)>\s+:\s+"(.)"/) {
		$utf8 = sprintf("%x",ord($3));
		next if (length($utf8)>4);
		$first  = get_utf8($1);
		$second = get_utf8($2);
		if ($first && $second) {
			$map_first{$utf8} = $first;
			$map_second{$utf8} = $second;
		}
	}
}
close(FH);
$char  = "uint16_t multikey_map_char[] = { ";
$first = "uint16_t multikey_map_first[] = { ";
$second = "uint16_t multikey_map_second[] = { ";
$count=0;
foreach $c (sort { hex($a) <=> hex($b)} keys %map_first) {
	if ($count != 0) {
		$char.=",";
		$first.=",";
		$second.=",";
	}
	if ($count%16==0) {
		$char.="\n";
		$first.="\n";
		$second.="\n";
	}
	$char.="0x".$c;
	$first.="0x".$map_first{$c};
	$second.="0x".$map_second{$c};
	$count++;
}
$char.="\n};\n";
$first.="\n};\n";
$second.="\n};\n";
print "#define MULTIKEY_MAP_SIZE ".$count."\n";
print $char.$first.$second;

sub get_utf8 {
	$sym = shift;
	return $1 if ($sym =~ m/^U([0-9a-f]{4})$/i);
	$cmd = "grep \"#define XK_".$sym." \" ".$keysym_file;
	$re = "^#define\\s+XK_".$sym."\\s+0x[0-9a-f]+\\s*\\/\\*[ (]*U\\+([0-9a-f]{4})";
	return $1 if (`$cmd` =~ m/$re/i);
	return 0;
}
---------------
 */
#include <QtGlobal>

#ifdef Q_OS_LINUX
#include <QX11Info>
#endif

#include "keysymhelper.h"

#ifdef Q_OS_LINUX
namespace ActionTools
{
	void KeySymHelper::loadKeyCodes()
	{
		int minKeyCode;
		int maxKeyCode;
		int keysymsPerKeycode;
		int numModifiers;
		KeySym *keysyms;

		XDisplayKeycodes(QX11Info::display(), &minKeyCode, &maxKeyCode);
		keysyms = XGetKeyboardMapping(QX11Info::display(),
									  minKeyCode,
									  maxKeyCode + 1 - minKeyCode,
									  &keysymsPerKeycode);

		if(keysymsPerKeycode < NUM_KEY_MODIFIERS * 2)
			numModifiers = keysymsPerKeycode;
		else
			numModifiers = NUM_KEY_MODIFIERS * 2;

		for(unsigned int i = 0; i < KeySymHelper::MAX_KEYSYM; ++i)
		{
			mKeySymToModifier[i] = -1;
			mKeySymToKeyCode[i] = 0;
		}

		for(int keycodeIndex = 0; keycodeIndex < (maxKeyCode + 1 - minKeyCode); ++keycodeIndex)
		{
			int keycode = keycodeIndex + minKeyCode;
			for(int wrapKeyIndex = 0; wrapKeyIndex < numModifiers; ++wrapKeyIndex)
			{
				if(char *str = XKeysymToString(keysyms[keycodeIndex * keysymsPerKeycode + wrapKeyIndex]))
				{
					KeySym keysym = XStringToKeysym(str);

					if(keysym < MAX_KEYSYM && mKeySymToModifier[keysym] == -1 )
					{
						mKeySymToModifier[keysym] = wrapKeyIndex;
						mKeySymToKeyCode[keysym] = keycode;
					}
				}
			}
		}
		XFree(keysyms);
	}

	KeySym KeySymHelper::wcharToKeySym(wchar_t c)
	{
		KeySym keysym = 0;

		if(c <= 0xff)//keysym = wchar value for Latin-1 (ISO-8859-1) characters
			keysym = c;
		else
		{
			if(c - 0x0100 < MAP_SIZE)
				keysym = mWCharToKeySym[c];

			if(!keysym)//Not found -> assume that it's a newer unicode character
				keysym = c + 0x01000000;
		}

		if(keysym >= MAX_KEYSYM)
			keysym = 0;

		return keysym;
	}

	int KeySymHelper::keySymToModifier(KeySym keySym)
	{
		return mKeySymToModifier[keySym];
	}

	KeyCode KeySymHelper::keySymToKeyCode(KeySym keySym)
	{
		return mKeySymToKeyCode[keySym];
	}

	const char *KeySymHelper::keyModifiers[] =
	{
		NULL, "Mode_switch", "ISO_Level3_Shift"
	};

	const quint16 KeySymHelper::mWCharToKeySym[] =
	{
		0x03c0,0x03e0,0x01c3,0x01e3,0x01a1,0x01b1,0x01c6,0x01e6,0x02c6,0x02e6,0x02c5,0x02e5,0x01c8,0x01e8,0x01cf,0x01ef,
		0x01d0,0x01f0,0x03aa,0x03ba,0,0,0x03cc,0x03ec,0x01ca,0x01ea,0x01cc,0x01ec,0x02d8,0x02f8,0x02ab,0x02bb,
		0x02d5,0x02f5,0x03ab,0x03bb,0x02a6,0x02b6,0x02a1,0x02b1,0x03a5,0x03b5,0x03cf,0x03ef,0,0,0x03c7,0x03e7,
		0x02a9,0x02b9,0,0,0x02ac,0x02bc,0x03d3,0x03f3,0x03a2,0x01c5,0x01e5,0x03a6,0x03b6,0x01a5,0x01b5,0,
		0,0x01a3,0x01b3,0x01d1,0x01f1,0x03d1,0x03f1,0x01d2,0x01f2,0,0x03bd,0x03bf,0x03d2,0x03f2,0,0,
		0x01d5,0x01f5,0x13bc,0x13bd,0x01c0,0x01e0,0x03a3,0x03b3,0x01d8,0x01f8,0x01a6,0x01b6,0x02de,0x02fe,0x01aa,0x01ba,
		0x01a9,0x01b9,0x01de,0x01fe,0x01ab,0x01bb,0x03ac,0x03bc,0x03dd,0x03fd,0x03de,0x03fe,0x02dd,0x02fd,0x01d9,0x01f9,
		0x01db,0x01fb,0x03d9,0x03f9,0,0,0,0,0x13be,0x01ac,0x01bc,0x01af,0x01bf,0x01ae,0x01be,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x08f6,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0x01b7,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0x01a2,0x01ff,0,0x01b2,0,0x01bd,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0x07ae,0x07a1,0,0x07a2,0x07a3,0x07a4,0,0x07a7,0,0x07a8,0x07ab,
		0x07b6,0x07c1,0x07c2,0x07c3,0x07c4,0x07c5,0x07c6,0x07c7,0x07c8,0x07c9,0x07ca,0x07cb,0x07cc,0x07cd,0x07ce,0x07cf,
		0x07d0,0x07d1,0,0x07d2,0x07d4,0x07d5,0x07d6,0x07d7,0x07d8,0x07d9,0x07a5,0x07a9,0x07b1,0x07b2,0x07b3,0x07b4,
		0x07ba,0x07e1,0x07e2,0x07e3,0x07e4,0x07e5,0x07e6,0x07e7,0x07e8,0x07e9,0x07ea,0x07eb,0x07ec,0x07ed,0x07ee,0x07ef,
		0x07f0,0x07f1,0x07f3,0x07f2,0x07f4,0x07f5,0x07f6,0x07f7,0x07f8,0x07f9,0x07b5,0x07b9,0x07b7,0x07b8,0x07bb,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0x06b3,0x06b1,0x06b2,0x06b4,0x06b5,0x06b6,0x06b7,0x06b8,0x06b9,0x06ba,0x06bb,0x06bc,0,0x06be,0x06bf,
		0x06e1,0x06e2,0x06f7,0x06e7,0x06e4,0x06e5,0x06f6,0x06fa,0x06e9,0x06ea,0x06eb,0x06ec,0x06ed,0x06ee,0x06ef,0x06f0,
		0x06f2,0x06f3,0x06f4,0x06f5,0x06e6,0x06e8,0x06e3,0x06fe,0x06fb,0x06fd,0x06ff,0x06f9,0x06f8,0x06fc,0x06e0,0x06f1,
		0x06c1,0x06c2,0x06d7,0x06c7,0x06c4,0x06c5,0x06d6,0x06da,0x06c9,0x06ca,0x06cb,0x06cc,0x06cd,0x06ce,0x06cf,0x06d0,
		0x06d2,0x06d3,0x06d4,0x06d5,0x06c6,0x06c8,0x06c3,0x06de,0x06db,0x06dd,0x06df,0x06d9,0x06d8,0x06dc,0x06c0,0x06d1,
		0,0x06a3,0x06a1,0x06a2,0x06a4,0x06a5,0x06a6,0x06a7,0x06a8,0x06a9,0x06aa,0x06ab,0x06ac,0,0x06ae,0x06af,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x06bd,0x06ad,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x0ce0,0x0ce1,0x0ce2,0x0ce3,0x0ce4,0x0ce5,0x0ce6,0x0ce7,0x0ce8,0x0ce9,0x0cea,0x0ceb,0x0cec,0x0ced,0x0cee,0x0cef,
		0x0cf0,0x0cf1,0x0cf2,0x0cf3,0x0cf4,0x0cf5,0x0cf6,0x0cf7,0x0cf8,0x0cf9,0x0cfa,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0x05ac,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0x05bb,0,0,0,0x05bf,
		0,0x05c1,0x05c2,0x05c3,0x05c4,0x05c5,0x05c6,0x05c7,0x05c8,0x05c9,0x05ca,0x05cb,0x05cc,0x05cd,0x05ce,0x05cf,
		0x05d0,0x05d1,0x05d2,0x05d3,0x05d4,0x05d5,0x05d6,0x05d7,0x05d8,0x05d9,0x05da,0,0,0,0,0,
		0x05e0,0x05e1,0x05e2,0x05e3,0x05e4,0x05e5,0x05e6,0x05e7,0x05e8,0x05e9,0x05ea,0x05eb,0x05ec,0x05ed,0x05ee,0x05ef,
		0x05f0,0x05f1,0x05f2,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0x0da1,0x0da2,0x0da3,0x0da4,0x0da5,0x0da6,0x0da7,0x0da8,0x0da9,0x0daa,0x0dab,0x0dac,0x0dad,0x0dae,0x0daf,
		0x0db0,0x0db1,0x0db2,0x0db3,0x0db4,0x0db5,0x0db6,0x0db7,0x0db8,0x0db9,0x0dba,0x0dbb,0x0dbc,0x0dbd,0x0dbe,0x0dbf,
		0x0dc0,0x0dc1,0x0dc2,0x0dc3,0x0dc4,0x0dc5,0x0dc6,0x0dc7,0x0dc8,0x0dc9,0x0dca,0x0dcb,0x0dcc,0x0dcd,0x0dce,0x0dcf,
		0x0dd0,0x0dd1,0x0dd2,0x0dd3,0x0dd4,0x0dd5,0x0dd6,0x0dd7,0x0dd8,0x0dd9,0x0dda,0,0,0,0,0x0ddf,
		0x0de0,0x0de1,0x0de2,0x0de3,0x0de4,0x0de5,0x0de6,0x0de7,0x0de8,0x0de9,0x0dea,0x0deb,0x0dec,0x0ded,0,0,
		0x0df0,0x0df1,0x0df2,0x0df3,0x0df4,0x0df5,0x0df6,0x0df7,0x0df8,0x0df9,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x0aa2,0x0aa1,0x0aa3,0x0aa4,0,0x0aa5,0x0aa6,0x0aa7,0x0aa8,0,0,0,0,0,
		0,0,0x0abb,0x0aaa,0x0aa9,0x07af,0,0x0cdf,0x0ad0,0x0ad1,0x0afd,0,0x0ad2,0x0ad3,0x0afe,0,
		0x0af1,0x0af2,0x0ae6,0,0,0x0aaf,0x0aae,0,0,0,0,0,0,0,0,0,
		0,0,0x0ad6,0x0ad7,0,0,0,0,0x0afc,0,0,0,0,0,0x047e,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0x0eff,0,0,0x20ac,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0x0ab8,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0x06b0,0x0afb,0,0,0,0,0,0,0x0ad4,0,
		0,0,0x0ac9,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0x0ab0,0x0ab1,0x0ab2,0x0ab3,0x0ab4,0x0ab5,0x0ab6,0x0ab7,0x0ac3,0x0ac4,0x0ac5,0x0ac6,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x08fb,0x08fc,0x08fd,0x08fe,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x08ce,0,0x08cd,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x08ef,0,0,0,0,0x08c5,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0x0bca,0,0x08d6,0,0,0x08c1,0x08c2,0,
		0,0,0,0,0,0,0,0x08de,0x08df,0x08dc,0x08dd,0x08bf,0,0,0,0,
		0,0,0,0,0x08c0,0,0,0,0,0,0,0,0x08c8,0,0,0,
		0,0,0,0x08c9,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x08bd,0x08cf,0,0,0x08bc,0x08be,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x08da,0x08db,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x0bdc,0x0bfc,0x0bce,0x0bc2,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0x0bd3,0,0x0bc4,0,0,0,0,0,
		0,0,0,0,0,0x0afa,0,0,0,0,0,0,0,0,0,0,
		0x08a4,0x08a5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0x0bcc,0,0,0,0,0,0x08ab,0,0x08ac,0x08ad,0,
		0x08ae,0x08a7,0,0x08a8,0x08a9,0,0x08aa,0,0x08af,0,0,0,0x08b0,0,0,0,
		0,0,0,0,0,0,0,0x08a1,0,0,0x09ef,0x09f0,0x09f2,0x09f3,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0x09e2,0x09e5,0x09e9,0x09e3,0x09e4,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0x0aac,0x09e8,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x08a3,0,0x08a6,0,0,0,0,0,0,0,0,0,0x08a2,0,0,0,
		0x09eb,0,0,0,0x09ed,0,0,0,0x09ea,0,0,0,0x09f4,0,0,0,
		0,0,0,0,0x09f5,0,0,0,0,0,0,0,0x09f7,0,0,0,
		0,0,0,0,0x09f6,0,0,0,0,0,0,0,0x09ee,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0x09e1,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0x0ae7,0x0ae1,0x0adb,0x0ae2,0x0adf,0x0acf,
		0,0,0x0ae8,0x0ae3,0,0,0x0add,0x0acd,0,0,0,0,0x0ae9,0x0ae4,0,0,
		0x0adc,0x0acc,0,0,0,0,0x09e0,0,0,0,0,0x0ace,0,0,0,0x0ade,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0x0ae0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0x0ae5,0,0,0,0,0,0,0,0x0af9,0,
		0,0,0,0x0aca,0,0,0,0,0,0,0,0,0x0aea,0,0x0aeb,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0x0af8,0,0x0af7,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0x0aec,0,0x0aee,0x0aed,0,0,0,0,0,0,0x0af6,0,0x0af5,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0x0af3,0,0,0,0x0af4,0,0,0,0,0,0x0ad9,0,0,
		0x0af0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0x0abc,0x0abe,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0x04a4,0x04a1,0,0,0,0,0,0,0,0,0,0x04a2,0x04a3,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0x04de,0x04df,0,0,0,
		0,0x04a7,0x04b1,0x04a8,0x04b2,0x04a9,0x04b3,0x04aa,0x04b4,0x04ab,0x04b5,0x04b6,0,0x04b7,0,0x04b8,
		0,0x04b9,0,0x04ba,0,0x04bb,0,0x04bc,0,0x04bd,0,0x04be,0,0x04bf,0,0x04c0,
		0,0x04c1,0,0x04af,0x04c2,0,0x04c3,0,0x04c4,0,0x04c5,0x04c6,0x04c7,0x04c8,0x04c9,0x04ca,
		0,0,0x04cb,0,0,0x04cc,0,0,0x04cd,0,0,0x04ce,0,0,0x04cf,0x04d0,
		0x04d1,0x04d2,0x04d3,0x04ac,0x04d4,0x04ad,0x04d5,0x04ae,0x04d6,0x04d7,0x04d8,0x04d9,0x04da,0x04db,0,0x04dc,
		0,0,0x04a6,0x04dd,0,0,0,0,0,0,0,0x04a5,0x04b0
	};

	int KeySymHelper::mKeySymToModifier[KeySymHelper::MAX_KEYSYM] = {-1};
	KeyCode KeySymHelper::mKeySymToKeyCode[KeySymHelper::MAX_KEYSYM] = {0};

	const quint16 KeySymHelper::multikeyMapChar[] =
	{
		0x23,0x27,0x40,0x5b,0x5d,0x5e,0x60,0x7b,0x7c,0x7d,0x7e,0xa0,0xa1,0xa2,0xa3,0xa4,
		0xa5,0xa6,0xa7,0xa9,0xab,0xac,0xae,0xb0,0xb1,0xb2,0xb3,0xb5,0xb6,0xb7,0xb8,0xb9,
		0xbb,0xbc,0xbd,0xbe,0xbf,0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,
		0xcb,0xcc,0xcd,0xce,0xcf,0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,
		0xdb,0xdc,0xdd,0xde,0xdf,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,
		0xeb,0xec,0xed,0xee,0xef,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,
		0xfb,0xfc,0xfd,0xfe,0xff,0x100,0x101,0x102,0x103,0x104,0x105,0x106,0x107,0x108,0x109,0x10a,
		0x10b,0x10c,0x10d,0x10e,0x10f,0x110,0x111,0x112,0x113,0x114,0x115,0x116,0x117,0x118,0x119,0x11a,
		0x11b,0x11c,0x11d,0x11e,0x11f,0x120,0x121,0x122,0x123,0x124,0x125,0x126,0x127,0x128,0x129,0x12a,
		0x12b,0x12c,0x12d,0x12e,0x12f,0x130,0x131,0x134,0x135,0x136,0x137,0x138,0x139,0x13a,0x13b,0x13c,
		0x13d,0x13e,0x141,0x142,0x143,0x144,0x145,0x146,0x147,0x148,0x14a,0x14b,0x14c,0x14d,0x14e,0x14f,
		0x150,0x151,0x152,0x153,0x154,0x155,0x156,0x157,0x158,0x159,0x15a,0x15b,0x15c,0x15d,0x15e,0x15f,
		0x160,0x161,0x162,0x163,0x164,0x165,0x166,0x167,0x168,0x169,0x16a,0x16b,0x16c,0x16d,0x16e,0x16f,
		0x170,0x171,0x172,0x173,0x174,0x175,0x176,0x177,0x178,0x179,0x17a,0x17b,0x17c,0x17d,0x17e,0x17f,
		0x180,0x197,0x1a0,0x1a1,0x1af,0x1b0,0x1b5,0x1b6,0x1cd,0x1ce,0x1cf,0x1d0,0x1d1,0x1d2,0x1d3,0x1d4,
		0x1d5,0x1d6,0x1d7,0x1d8,0x1d9,0x1da,0x1db,0x1dc,0x1de,0x1df,0x1e0,0x1e1,0x1e2,0x1e3,0x1e4,0x1e5,
		0x1e6,0x1e7,0x1e8,0x1e9,0x1ea,0x1eb,0x1ec,0x1ed,0x1ee,0x1ef,0x1f0,0x1f4,0x1f5,0x1f8,0x1f9,0x1fa,
		0x1fb,0x1fc,0x1fd,0x1fe,0x1ff,0x21e,0x21f,0x226,0x227,0x228,0x229,0x22a,0x22b,0x22c,0x22d,0x22e,
		0x22f,0x230,0x231,0x232,0x233,0x259,0x268,0x2a1,0x344,0x385,0x386,0x388,0x389,0x38a,0x38c,0x38e,
		0x38f,0x390,0x3aa,0x3ab,0x3ac,0x3ad,0x3ae,0x3af,0x3b0,0x3ca,0x3cb,0x3cc,0x3cd,0x3ce,0x3d4,0x400,
		0x401,0x403,0x407,0x40c,0x40d,0x40e,0x419,0x439,0x450,0x451,0x453,0x457,0x45c,0x45d,0x45e,0x492,
		0x493,0x49e,0x49f,0x4b0,0x4b1,0x4c1,0x4c2,0x4d0,0x4d1,0x4d2,0x4d3,0x4d6,0x4d7,0x4da,0x4db,0x4dc,
		0x4dd,0x4de,0x4df,0x4e2,0x4e3,0x4e4,0x4e5,0x4e6,0x4e7,0x4ea,0x4eb,0x4ec,0x4ed,0x4ee,0x4ef,0x4f0,
		0x4f1,0x4f2,0x4f3,0x4f4,0x4f5,0x4f8,0x4f9,0x622,0x623,0x624,0x625,0x626,0x6c0,0x6c2,0x6d3,0x929,
		0x931,0x934,0x958,0x959,0x95a,0x95b,0x95c,0x95d,0x95e,0x95f,0x9cb,0x9cc,0x9dc,0x9dd,0x9df,0xa33,
		0xa36,0xa59,0xa5a,0xa5b,0xa5e,0xb48,0xb4b,0xb4c,0xb5c,0xb5d,0xb94,0xbca,0xbcb,0xbcc,0xc48,0xcc0,
		0xcc7,0xcc8,0xcca,0xccb,0xd4a,0xd4b,0xd4c,0xdda,0xddc,0xddd,0xdde,0xf43,0xf4d,0xf52,0xf57,0xf5c,
		0xf69,0xf73,0xf75,0xf76,0xf78,0xf81,0xf93,0xf9d,0xfa2,0xfa7,0xfac,0xfb9,0x1026,0x1101,0x1104,0x1108,
		0x110a,0x110d,0x1113,0x1114,0x1115,0x1116,0x1117,0x1118,0x1119,0x111a,0x111b,0x111c,0x111d,0x111e,0x111f,0x1120,
		0x1121,0x1122,0x1123,0x1124,0x1125,0x1126,0x1127,0x1128,0x1129,0x112a,0x112b,0x112c,0x112d,0x112e,0x112f,0x1130,
		0x1131,0x1132,0x1133,0x1134,0x1135,0x1136,0x1137,0x1138,0x1139,0x113a,0x113b,0x113d,0x113f,0x1141,0x1142,0x1143,
		0x1144,0x1145,0x1146,0x1147,0x1148,0x1149,0x114a,0x114b,0x114d,0x114f,0x1151,0x1152,0x1153,0x1156,0x1157,0x1158,
		0x1162,0x1164,0x1166,0x1168,0x116a,0x116b,0x116c,0x116f,0x1170,0x1171,0x1174,0x1176,0x1177,0x1178,0x1179,0x117a,
		0x117b,0x117c,0x117d,0x117e,0x117f,0x1180,0x1181,0x1182,0x1183,0x1184,0x1185,0x1186,0x1187,0x1188,0x1189,0x118a,
		0x118b,0x118c,0x118d,0x118e,0x118f,0x1190,0x1191,0x1192,0x1193,0x1194,0x1195,0x1196,0x1197,0x1198,0x1199,0x119a,
		0x119b,0x119c,0x119d,0x119f,0x11a0,0x11a1,0x11a2,0x11a9,0x11aa,0x11ac,0x11ad,0x11b0,0x11b1,0x11b2,0x11b3,0x11b4,
		0x11b5,0x11b6,0x11b9,0x11bb,0x11c3,0x11c4,0x11c5,0x11c6,0x11c7,0x11c8,0x11c9,0x11ca,0x11cb,0x11cc,0x11cd,0x11ce,
		0x11cf,0x11d0,0x11d1,0x11d2,0x11d3,0x11d4,0x11d5,0x11d6,0x11d7,0x11d8,0x11d9,0x11da,0x11db,0x11dc,0x11dd,0x11de,
		0x11df,0x11e0,0x11e1,0x11e2,0x11e3,0x11e4,0x11e5,0x11e6,0x11e7,0x11e8,0x11e9,0x11ea,0x11ec,0x11ed,0x11ee,0x11ef,
		0x11f1,0x11f2,0x11f3,0x11f4,0x11f5,0x11f6,0x11f7,0x11f8,0x1e02,0x1e03,0x1e04,0x1e05,0x1e08,0x1e09,0x1e0a,0x1e0b,
		0x1e0c,0x1e0d,0x1e10,0x1e11,0x1e14,0x1e15,0x1e16,0x1e17,0x1e1c,0x1e1d,0x1e1e,0x1e1f,0x1e20,0x1e21,0x1e22,0x1e23,
		0x1e24,0x1e25,0x1e26,0x1e27,0x1e28,0x1e29,0x1e2e,0x1e2f,0x1e30,0x1e31,0x1e32,0x1e33,0x1e36,0x1e37,0x1e38,0x1e39,
		0x1e3e,0x1e3f,0x1e40,0x1e41,0x1e42,0x1e43,0x1e44,0x1e45,0x1e46,0x1e47,0x1e4c,0x1e4d,0x1e4e,0x1e4f,0x1e50,0x1e51,
		0x1e52,0x1e53,0x1e54,0x1e55,0x1e56,0x1e57,0x1e58,0x1e59,0x1e5a,0x1e5b,0x1e5c,0x1e5d,0x1e60,0x1e61,0x1e62,0x1e63,
		0x1e64,0x1e65,0x1e66,0x1e67,0x1e68,0x1e69,0x1e6a,0x1e6b,0x1e6c,0x1e6d,0x1e78,0x1e79,0x1e7a,0x1e7b,0x1e7c,0x1e7d,
		0x1e7e,0x1e7f,0x1e80,0x1e81,0x1e82,0x1e83,0x1e84,0x1e85,0x1e86,0x1e87,0x1e88,0x1e89,0x1e8a,0x1e8b,0x1e8c,0x1e8d,
		0x1e8e,0x1e8f,0x1e90,0x1e91,0x1e92,0x1e93,0x1e97,0x1e98,0x1e99,0x1e9b,0x1ea0,0x1ea1,0x1ea2,0x1ea3,0x1ea4,0x1ea5,
		0x1ea6,0x1ea7,0x1ea8,0x1ea9,0x1eaa,0x1eab,0x1eac,0x1ead,0x1eae,0x1eaf,0x1eb0,0x1eb1,0x1eb2,0x1eb3,0x1eb4,0x1eb5,
		0x1eb6,0x1eb7,0x1eb8,0x1eb9,0x1eba,0x1ebb,0x1ebc,0x1ebd,0x1ebe,0x1ebf,0x1ec0,0x1ec1,0x1ec2,0x1ec3,0x1ec4,0x1ec5,
		0x1ec6,0x1ec7,0x1ec8,0x1ec9,0x1eca,0x1ecb,0x1ecc,0x1ecd,0x1ece,0x1ecf,0x1ed0,0x1ed1,0x1ed2,0x1ed3,0x1ed4,0x1ed5,
		0x1ed6,0x1ed7,0x1ed8,0x1ed9,0x1eda,0x1edb,0x1edc,0x1edd,0x1ede,0x1edf,0x1ee0,0x1ee1,0x1ee2,0x1ee3,0x1ee4,0x1ee5,
		0x1ee6,0x1ee7,0x1ee8,0x1ee9,0x1eea,0x1eeb,0x1eec,0x1eed,0x1eee,0x1eef,0x1ef0,0x1ef1,0x1ef2,0x1ef3,0x1ef4,0x1ef5,
		0x1ef6,0x1ef7,0x1ef8,0x1ef9,0x1f00,0x1f01,0x1f02,0x1f03,0x1f04,0x1f05,0x1f06,0x1f07,0x1f08,0x1f09,0x1f0a,0x1f0b,
		0x1f0c,0x1f0d,0x1f0e,0x1f0f,0x1f10,0x1f11,0x1f12,0x1f13,0x1f14,0x1f15,0x1f18,0x1f19,0x1f1a,0x1f1b,0x1f1c,0x1f1d,
		0x1f20,0x1f21,0x1f22,0x1f23,0x1f24,0x1f25,0x1f26,0x1f27,0x1f28,0x1f29,0x1f2a,0x1f2b,0x1f2c,0x1f2d,0x1f2e,0x1f2f,
		0x1f30,0x1f31,0x1f32,0x1f33,0x1f34,0x1f35,0x1f36,0x1f37,0x1f38,0x1f39,0x1f3a,0x1f3b,0x1f3c,0x1f3d,0x1f3e,0x1f3f,
		0x1f40,0x1f41,0x1f42,0x1f43,0x1f44,0x1f45,0x1f48,0x1f49,0x1f4a,0x1f4b,0x1f4c,0x1f4d,0x1f50,0x1f51,0x1f52,0x1f53,
		0x1f54,0x1f55,0x1f56,0x1f57,0x1f59,0x1f5b,0x1f5d,0x1f5f,0x1f60,0x1f61,0x1f62,0x1f63,0x1f64,0x1f65,0x1f66,0x1f67,
		0x1f68,0x1f69,0x1f6a,0x1f6b,0x1f6c,0x1f6d,0x1f6e,0x1f6f,0x1f70,0x1f72,0x1f74,0x1f76,0x1f78,0x1f7a,0x1f7c,0x1f80,
		0x1f81,0x1f82,0x1f83,0x1f84,0x1f85,0x1f86,0x1f87,0x1f88,0x1f89,0x1f8a,0x1f8b,0x1f8c,0x1f8d,0x1f8e,0x1f8f,0x1f90,
		0x1f91,0x1f92,0x1f93,0x1f94,0x1f95,0x1f96,0x1f97,0x1f98,0x1f99,0x1f9a,0x1f9b,0x1f9c,0x1f9d,0x1f9e,0x1f9f,0x1fa0,
		0x1fa1,0x1fa2,0x1fa3,0x1fa4,0x1fa5,0x1fa6,0x1fa7,0x1fa8,0x1fa9,0x1faa,0x1fab,0x1fac,0x1fad,0x1fae,0x1faf,0x1fb0,
		0x1fb1,0x1fb2,0x1fb3,0x1fb4,0x1fb6,0x1fb7,0x1fb8,0x1fb9,0x1fba,0x1fbc,0x1fc1,0x1fc2,0x1fc3,0x1fc4,0x1fc6,0x1fc7,
		0x1fc8,0x1fca,0x1fcc,0x1fcd,0x1fce,0x1fcf,0x1fd0,0x1fd1,0x1fd2,0x1fd6,0x1fd7,0x1fd8,0x1fd9,0x1fda,0x1fdd,0x1fde,
		0x1fdf,0x1fe0,0x1fe1,0x1fe2,0x1fe4,0x1fe5,0x1fe6,0x1fe7,0x1fe8,0x1fe9,0x1fea,0x1fec,0x1fed,0x1ff2,0x1ff3,0x1ff4,
		0x1ff6,0x1ff7,0x1ff8,0x1ffa,0x1ffc,0x2008,0x2018,0x2019,0x201a,0x201c,0x201d,0x201e,0x2022,0x2026,0x2030,0x2039,
		0x203a,0x203d,0x2070,0x2074,0x2075,0x2076,0x2077,0x2078,0x2079,0x207a,0x207b,0x207c,0x207d,0x207e,0x2080,0x2081,
		0x2082,0x2083,0x2084,0x2085,0x2086,0x2087,0x2088,0x2089,0x208a,0x208b,0x208c,0x208d,0x208e,0x20a0,0x20a1,0x20a2,
		0x20a3,0x20a4,0x20a5,0x20a6,0x20a7,0x20a8,0x20a9,0x20ab,0x20ac,0x2116,0x2120,0x2122,0x219a,0x219b,0x21ae,0x2204,
		0x2209,0x220c,0x2224,0x2226,0x2241,0x2244,0x2247,0x2249,0x2260,0x2262,0x226d,0x226e,0x226f,0x2270,0x2271,0x2274,
		0x2275,0x2278,0x2279,0x2280,0x2281,0x2284,0x2285,0x2288,0x2289,0x22ac,0x22ad,0x22ae,0x22af,0x22e0,0x22e1,0x22e2,
		0x22e3,0x22ea,0x22eb,0x22ec,0x22ed,0x266d,0x266e,0x266f,0x2adc,0x301d,0x301e,0x3192,0x3193,0x3194,0x3195,0x3196,
		0x3197,0x3198,0x3199,0x319a,0x319b,0x319c,0x319d,0x319e,0x319f,0xfb1d,0xfb1f,0xfb2a,0xfb2b,0xfb2c,0xfb2d,0xfb2e,
		0xfb2f,0xfb30,0xfb31,0xfb32,0xfb33,0xfb34,0xfb35,0xfb36,0xfb38,0xfb39,0xfb3a,0xfb3b,0xfb3c,0xfb3e,0xfb40,0xfb41,
		0xfb43,0xfb44,0xfb46,0xfb47,0xfb48,0xfb49,0xfb4a,0xfb4b,0xfb4c,0xfb4d,0xfb4e
	};

	const quint16 KeySymHelper::multikeyMapFirst[] =
	{
		0x002B,0x0020,0x0041,0x0028,0x0029,0x0020,0x0020,0x002D,0x006C,0x002D,0x0020,0x0020,0x0021,0x002F,0x002D,0x0078,
		0x003D,0x0021,0x043F,0x004F,0x003C,0x002D,0x004F,0x006F,0x002B,0x005E,0x005E,0x006D,0x0050,0x002E,0x0020,0x005E,
		0x003E,0x0031,0x0031,0x0033,0x003F,0x0060,0x0027,0x005E,0x007E,0x0022,0x006F,0x0041,0x00B8,0x0060,0x0027,0x005E,
		0x0022,0x0060,0x0027,0x005E,0x0022,0x0044,0x007E,0x0060,0x0027,0x005E,0x007E,0x0022,0x0078,0x002F,0x0060,0x0027,
		0x005E,0x0022,0x0027,0x0054,0x0073,0x0060,0x0027,0x005E,0x007E,0x0022,0x006F,0x0061,0x00B8,0x0060,0x0027,0x005E,
		0x0022,0x0060,0x0027,0x005E,0x0022,0x0064,0x007E,0x0060,0x0027,0x005E,0x007E,0x0022,0x002D,0x002F,0x0060,0x0027,
		0x005E,0x0022,0x0027,0x0074,0x0022,0x005F,0x005F,0x0062,0x0062,0x003B,0x003B,0x0027,0x0027,0x005E,0x005E,0x002E,
		0x002E,0x0063,0x0063,0x0063,0x0063,0x002F,0x002F,0x005F,0x005F,0x0062,0x0062,0x002E,0x002E,0x003B,0x003B,0x0063,
		0x0063,0x005E,0x005E,0x0062,0x0062,0x002E,0x002E,0x00B8,0x00B8,0x005E,0x005E,0x002F,0x002F,0x007E,0x007E,0x005F,
		0x005F,0x0062,0x0062,0x003B,0x003B,0x002E,0x0069,0x005E,0x005E,0x00B8,0x00B8,0x006B,0x0027,0x0027,0x00B8,0x00B8,
		0x0063,0x0063,0x002F,0x002F,0x0027,0x0027,0x00B8,0x00B8,0x0063,0x0063,0x004E,0x006E,0x005F,0x005F,0x0062,0x0062,
		0x003D,0x003D,0x004F,0x006F,0x0027,0x0027,0x00B8,0x00B8,0x0063,0x0063,0x0027,0x0027,0x005E,0x005E,0x00B8,0x00B8,
		0x0063,0x0063,0x00B8,0x00B8,0x0063,0x0063,0x002F,0x002F,0x007E,0x007E,0x005F,0x005F,0x0062,0x0062,0x006F,0x006F,
		0x003D,0x003D,0x003B,0x003B,0x005E,0x005E,0x005E,0x005E,0x0022,0x0027,0x0027,0x002E,0x002E,0x0063,0x0063,0x0066,
		0x002F,0x002F,0x002B,0x002B,0x002B,0x002B,0x002F,0x002F,0x0063,0x0063,0x0063,0x0063,0x0063,0x0063,0x0063,0x0063,
		0x005F,0x005F,0x0027,0x0027,0x0063,0x0063,0x0060,0x0060,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x002F,0x002F,
		0x0063,0x0063,0x0063,0x0063,0x003B,0x003B,0x005F,0x005F,0x0063,0x0063,0x0063,0x0027,0x0027,0x0060,0x0060,0x0027,
		0x0027,0x0027,0x0027,0x0027,0x0027,0x0063,0x0063,0x002E,0x002E,0x00B8,0x00B8,0x005F,0x005F,0x005F,0x005F,0x002E,
		0x002E,0x005F,0x005F,0x005F,0x005F,0x0065,0x002F,0x002F,0x0022,0x00A8,0x0027,0x0027,0x0027,0x0027,0x0027,0x0027,
		0x0027,0x0027,0x0022,0x0022,0x0027,0x0027,0x0027,0x0027,0x0027,0x0022,0x0022,0x0027,0x0027,0x0027,0x0022,0x0060,
		0x0022,0x0027,0x0022,0x0027,0x0060,0x0062,0x0062,0x0062,0x0060,0x0022,0x0027,0x0022,0x0027,0x0060,0x0062,0x002F,
		0x002F,0x002F,0x002F,0x002F,0x002F,0x0062,0x0062,0x0062,0x0062,0x0022,0x0022,0x0062,0x0062,0x0022,0x0022,0x0022,
		0x0022,0x0022,0x0022,0x005F,0x005F,0x0022,0x0022,0x0022,0x0022,0x0022,0x0022,0x0022,0x0022,0x005F,0x005F,0x0022,
		0x0022,0x003D,0x003D,0x0022,0x0022,0x0022,0x0022,0x0653,0x0654,0x0654,0x0655,0x0654,0x0654,0x0654,0x0654,0x093C,
		0x093C,0x093C,0x093C,0x093C,0x093C,0x093C,0x093C,0x093C,0x093C,0x093C,0x09C7,0x09C7,0x09BC,0x09BC,0x09BC,0x0A3C,
		0x0A3C,0x0A3C,0x0A3C,0x0A3C,0x0A3C,0x0B47,0x0B47,0x0B47,0x0B3C,0x0B3C,0x0BD7,0x0BC6,0x0BC7,0x0BC6,0x0C46,0x0CBF,
		0x0CC6,0x0CC6,0x0CC6,0x0CCA,0x0D46,0x0D47,0x0D46,0x0DD9,0x0DD9,0x0DDC,0x0DD9,0x0FB7,0x0FB7,0x0FB7,0x0FB7,0x0FB7,
		0x0FB5,0x0F71,0x0F71,0x0FB2,0x0FB3,0x0F71,0x0F92,0x0F9C,0x0FA1,0x0FA6,0x0FAB,0x0F90,0x102E,0x1100,0x1103,0x1107,
		0x1109,0x110C,0x1102,0x1102,0x1102,0x1102,0x1103,0x1105,0x1105,0x1105,0x1105,0x1106,0x1106,0x1107,0x1107,0x1107,
		0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1107,0x1109,0x1109,0x1109,0x1109,
		0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x1109,0x113C,0x113E,0x110B,0x110B,0x110B,
		0x110B,0x110B,0x110B,0x110B,0x110B,0x110B,0x110B,0x110B,0x110C,0x114E,0x1150,0x110E,0x110E,0x1111,0x1111,0x1112,
		0x1161,0x1163,0x1165,0x1167,0x1169,0x1169,0x1169,0x116E,0x116E,0x116E,0x1173,0x1161,0x1161,0x1163,0x1163,0x1165,
		0x1165,0x1165,0x1167,0x1167,0x1169,0x1169,0x1169,0x1169,0x1169,0x116D,0x116D,0x116D,0x116D,0x116D,0x116E,0x116E,
		0x116E,0x116E,0x116E,0x1172,0x1172,0x1172,0x1172,0x1172,0x1172,0x1172,0x1173,0x1173,0x1174,0x1175,0x1175,0x1175,
		0x1175,0x1175,0x1175,0x119E,0x119E,0x119E,0x119E,0x11A8,0x11A8,0x11AB,0x11AB,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,
		0x11AF,0x11AF,0x11B8,0x11BA,0x11A8,0x11A8,0x11AB,0x11AB,0x11AB,0x11AB,0x11AB,0x11AE,0x11AE,0x11AF,0x11AF,0x11AF,
		0x11CE,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11AF,0x11B7,0x11B7,0x11B7,0x11B7,0x11B7,
		0x11B7,0x11B7,0x11B7,0x11B7,0x11B8,0x11B8,0x11B8,0x11B8,0x11BA,0x11BA,0x11BA,0x11BA,0x11BC,0x11BC,0x11BC,0x11BC,
		0x11F0,0x11F0,0x11C1,0x11C1,0x11C2,0x11C2,0x11C2,0x11C2,0x002E,0x002E,0x0021,0x0021,0x0027,0x0027,0x002E,0x002E,
		0x0021,0x0021,0x00B8,0x00B8,0x0060,0x0060,0x0027,0x0027,0x0062,0x0062,0x002E,0x002E,0x005F,0x005F,0x002E,0x002E,
		0x0021,0x0021,0x0022,0x0022,0x00B8,0x00B8,0x0027,0x0027,0x0027,0x0027,0x0021,0x0021,0x0021,0x0021,0x005F,0x005F,
		0x0027,0x0027,0x002E,0x002E,0x0021,0x0021,0x002E,0x002E,0x0021,0x0021,0x0027,0x0027,0x0022,0x0022,0x0060,0x0060,
		0x0027,0x0027,0x0027,0x0027,0x002E,0x002E,0x002E,0x002E,0x0021,0x0021,0x005F,0x005F,0x002E,0x002E,0x0021,0x0021,
		0x002E,0x002E,0x002E,0x002E,0x002E,0x002E,0x002E,0x002E,0x0021,0x0021,0x0027,0x0027,0x0022,0x0022,0x007E,0x007E,
		0x0021,0x0021,0x0060,0x0060,0x0027,0x0027,0x0022,0x0022,0x002E,0x002E,0x0021,0x0021,0x002E,0x002E,0x0022,0x0022,
		0x002E,0x002E,0x005E,0x005E,0x0021,0x0021,0x0022,0x006F,0x006F,0x002E,0x0021,0x0021,0x003F,0x003F,0x0027,0x0027,
		0x0060,0x0060,0x003F,0x003F,0x007E,0x007E,0x005E,0x005E,0x0027,0x0027,0x0060,0x0060,0x003F,0x003F,0x007E,0x007E,
		0x0062,0x0062,0x0021,0x0021,0x003F,0x003F,0x007E,0x007E,0x0027,0x0027,0x0060,0x0060,0x003F,0x003F,0x007E,0x007E,
		0x005E,0x005E,0x003F,0x003F,0x0021,0x0021,0x0021,0x0021,0x003F,0x003F,0x0027,0x0027,0x0060,0x0060,0x003F,0x003F,
		0x007E,0x007E,0x005E,0x005E,0x0027,0x0027,0x0060,0x0060,0x003F,0x003F,0x007E,0x007E,0x0021,0x0021,0x0021,0x0021,
		0x003F,0x003F,0x0027,0x0027,0x0060,0x0060,0x003F,0x003F,0x007E,0x007E,0x0021,0x0021,0x0060,0x0060,0x0021,0x0021,
		0x003F,0x003F,0x007E,0x007E,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,0x0029,0x0028,0x0060,0x0060,
		0x0027,0x0027,0x007E,0x007E,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,
		0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,
		0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,
		0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x0029,0x0028,0x0060,0x0060,
		0x0027,0x0027,0x007E,0x007E,0x0028,0x0060,0x0027,0x007E,0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,
		0x0029,0x0028,0x0060,0x0060,0x0027,0x0027,0x007E,0x007E,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x03B9,
		0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,
		0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,
		0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x03B9,0x0062,
		0x005F,0x03B9,0x03B9,0x03B9,0x007E,0x03B9,0x0062,0x005F,0x0060,0x03B9,0x00A8,0x03B9,0x03B9,0x03B9,0x007E,0x03B9,
		0x0060,0x0060,0x03B9,0x1FBF,0x1FBF,0x1FBF,0x0062,0x005F,0x0060,0x007E,0x007E,0x0062,0x005F,0x0060,0x1FFE,0x1FFE,
		0x1FFE,0x0062,0x005F,0x0060,0x0029,0x0028,0x007E,0x007E,0x0062,0x005F,0x0060,0x0028,0x00A8,0x03B9,0x03B9,0x03B9,
		0x007E,0x03B9,0x0060,0x0060,0x03B9,0x0020,0x0027,0x0027,0x0027,0x0022,0x0022,0x0022,0x002E,0x002E,0x0025,0x002E,
		0x002E,0x0021,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005F,0x005F,
		0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x005F,0x0043,0x002F,0x0043,
		0x0046,0x003D,0x002F,0x003D,0x0050,0x0052,0x003D,0x0064,0x003D,0x041D,0x0073,0x0074,0x002F,0x002F,0x002F,0x2203,
		0x2208,0x220B,0x2223,0x2225,0x223C,0x2243,0x223C,0x2248,0x003D,0x2261,0x224D,0x003C,0x003E,0x2264,0x2265,0x2272,
		0x2273,0x2276,0x2277,0x227A,0x227B,0x2282,0x2283,0x2286,0x2287,0x22A3,0x22A8,0x22A9,0x22AB,0x227C,0x227D,0x2291,
		0x2292,0x22B2,0x22B3,0x22B4,0x22B5,0x0023,0x0023,0x0023,0x2ADD,0x0022,0x0022,0x005E,0x005E,0x005E,0x005E,0x005E,
		0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x005E,0x05B4,0x05B7,0x05C1,0x05C2,0x05C1,0x05C2,0x05B7,
		0x05B8,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,
		0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05BC,0x05B9,0x05BF,0x05BF,0x05BF
	};

	const quint16 KeySymHelper::multikeyMapSecond[] =
	{
		0x002B,0x0027,0x0054,0x0028,0x0029,0x003E,0x0060,0x0028,0x0076,0x0029,0x002D,0x0020,0x0021,0x0063,0x004C,0x006F,
		0x0059,0x005E,0x0430,0x0043,0x003C,0x002C,0x0052,0x006F,0x002D,0x0032,0x0033,0x0075,0x0050,0x002D,0x002C,0x0031,
		0x003E,0x0034,0x0032,0x0034,0x003F,0x0041,0x0041,0x0041,0x0041,0x0041,0x0041,0x0045,0x0043,0x0045,0x0045,0x0045,
		0x0045,0x0049,0x0049,0x0049,0x0049,0x0048,0x004E,0x004F,0x004F,0x004F,0x004F,0x004F,0x0078,0x004F,0x0055,0x0055,
		0x0055,0x0055,0x0059,0x0048,0x0073,0x0061,0x0061,0x0061,0x0061,0x0061,0x0061,0x0065,0x0063,0x0065,0x0065,0x0065,
		0x0065,0x0069,0x0069,0x0069,0x0069,0x0068,0x006E,0x006F,0x006F,0x006F,0x006F,0x006F,0x003A,0x006F,0x0075,0x0075,
		0x0075,0x0075,0x0079,0x0068,0x0079,0x0041,0x0061,0x0041,0x0061,0x0041,0x0061,0x0043,0x0063,0x0043,0x0063,0x0043,
		0x0063,0x0043,0x0063,0x0044,0x0064,0x0044,0x0064,0x0045,0x0065,0x0045,0x0065,0x0045,0x0065,0x0045,0x0065,0x0045,
		0x0065,0x0047,0x0067,0x0047,0x0067,0x0047,0x0067,0x0047,0x0067,0x0048,0x0068,0x0048,0x0068,0x0049,0x0069,0x0049,
		0x0069,0x0049,0x0069,0x0049,0x0069,0x0049,0x002E,0x004A,0x006A,0x004B,0x006B,0x006B,0x004C,0x006C,0x004C,0x006C,
		0x004C,0x006C,0x004C,0x006C,0x004E,0x006E,0x004E,0x006E,0x004E,0x006E,0x0047,0x0067,0x004F,0x006F,0x004F,0x006F,
		0x004F,0x006F,0x0045,0x0065,0x0052,0x0072,0x0052,0x0072,0x0052,0x0072,0x0053,0x0073,0x0053,0x0073,0x0053,0x0073,
		0x0053,0x0073,0x0054,0x0074,0x0054,0x0074,0x0054,0x0074,0x0055,0x0075,0x0055,0x0075,0x0055,0x0075,0x0055,0x0075,
		0x0055,0x0075,0x0055,0x0075,0x0057,0x0077,0x0059,0x0079,0x0059,0x005A,0x007A,0x005A,0x007A,0x005A,0x007A,0x0053,
		0x0062,0x0049,0x004F,0x006F,0x0055,0x0075,0x005A,0x007A,0x0041,0x0061,0x0049,0x0069,0x004F,0x006F,0x0055,0x0075,
		0x00DC,0x00FC,0x00DC,0x00FC,0x00DC,0x00FC,0x00DC,0x00FC,0x00C4,0x00E4,0x0226,0x0227,0x00C6,0x00E6,0x0047,0x0067,
		0x0047,0x0067,0x004B,0x006B,0x004F,0x006F,0x01EA,0x01EB,0x01B7,0x0292,0x006A,0x0047,0x0067,0x004E,0x006E,0x00C5,
		0x00E5,0x00C6,0x00E6,0x00D8,0x00F8,0x0048,0x0068,0x0041,0x0061,0x0045,0x0065,0x00D6,0x00F6,0x00D5,0x00F5,0x004F,
		0x006F,0x022E,0x022F,0x0059,0x0079,0x0065,0x0069,0x0294,0x0027,0x0027,0x0391,0x0395,0x0397,0x0399,0x039F,0x03A5,
		0x03A9,0x03CA,0x0399,0x03A5,0x03B1,0x03B5,0x03B7,0x03B9,0x03CB,0x03B9,0x03C5,0x03BF,0x03C5,0x03C9,0x03D2,0x0415,
		0x0415,0x0413,0x0406,0x041A,0x0418,0x0423,0x0418,0x0438,0x0435,0x0435,0x0433,0x0456,0x043A,0x0438,0x0443,0x0413,
		0x0433,0x041A,0x043A,0x04AE,0x04AF,0x0416,0x0436,0x0410,0x0430,0x0410,0x0430,0x0415,0x0435,0x04D8,0x04D9,0x0416,
		0x0436,0x0417,0x0437,0x0418,0x0438,0x0418,0x0438,0x041E,0x043E,0x04E8,0x04E9,0x042D,0x044D,0x0423,0x0443,0x0423,
		0x0443,0x0423,0x0443,0x0427,0x0447,0x042B,0x044B,0x0627,0x0627,0x0648,0x0627,0x064A,0x06D5,0x06C1,0x06D2,0x0928,
		0x0930,0x0933,0x0915,0x0916,0x0917,0x091C,0x0921,0x0922,0x092B,0x092F,0x09BE,0x09D7,0x09A1,0x09A2,0x09AF,0x0A32,
		0x0A38,0x0A16,0x0A17,0x0A1C,0x0A2B,0x0B56,0x0B3E,0x0B57,0x0B21,0x0B22,0x0B92,0x0BBE,0x0BBE,0x0BD7,0x0C56,0x0CD5,
		0x0CD5,0x0CD6,0x0CC2,0x0CD5,0x0D3E,0x0D3E,0x0D57,0x0DCA,0x0DCF,0x0DCA,0x0DDF,0x0F42,0x0F4C,0x0F51,0x0F56,0x0F5B,
		0x0F40,0x0F72,0x0F74,0x0F80,0x0F80,0x0F80,0x0FB7,0x0FB7,0x0FB7,0x0FB7,0x0FB7,0x0FB5,0x1025,0x1100,0x1103,0x1107,
		0x1109,0x110C,0x1100,0x1102,0x1103,0x1107,0x1100,0x1102,0x1105,0x1112,0x110B,0x1107,0x110B,0x1100,0x1102,0x1103,
		0x1109,0x112D,0x112F,0x1132,0x110A,0x1136,0x110C,0x110E,0x1110,0x1111,0x110B,0x112B,0x1100,0x1102,0x1103,0x1105,
		0x1106,0x1107,0x111E,0x110A,0x110B,0x110C,0x110E,0x110F,0x1110,0x1111,0x1112,0x113C,0x113E,0x1100,0x1103,0x1106,
		0x1107,0x1109,0x1140,0x110B,0x110C,0x110E,0x1110,0x1111,0x110B,0x114E,0x1150,0x110F,0x1112,0x1107,0x110B,0x1112,
		0x1175,0x1175,0x1175,0x1175,0x1161,0x1162,0x1175,0x1165,0x1166,0x1175,0x1175,0x1169,0x116E,0x1169,0x116D,0x1169,
		0x116E,0x1173,0x1169,0x116E,0x1165,0x1166,0x1168,0x1169,0x116E,0x1163,0x1164,0x1167,0x1169,0x1175,0x1161,0x1162,
		0x117C,0x1168,0x116E,0x1161,0x1165,0x1166,0x1167,0x1168,0x116E,0x1175,0x116E,0x1173,0x116E,0x1161,0x1163,0x1169,
		0x116E,0x1173,0x119E,0x1165,0x116E,0x1175,0x119E,0x11A8,0x11BA,0x11BD,0x11C2,0x11A8,0x11B7,0x11B8,0x11BA,0x11C0,
		0x11C1,0x11C2,0x11BA,0x11BA,0x11AF,0x11E7,0x11A8,0x11AE,0x11BA,0x11EB,0x11C0,0x11A8,0x11AF,0x11AA,0x11AB,0x11AE,
		0x11C2,0x11AF,0x11DA,0x11DD,0x11B9,0x11E5,0x11E6,0x11BB,0x11EB,0x11BF,0x11F9,0x11A8,0x11AF,0x11B8,0x11BA,0x11BB,
		0x11EB,0x11BE,0x11C2,0x11BC,0x11AF,0x11C1,0x11C2,0x11BC,0x11A8,0x11AE,0x11AF,0x11B8,0x11A8,0x11A9,0x11BC,0x11BF,
		0x11BA,0x11EB,0x11B8,0x11BC,0x11AB,0x11AF,0x11B7,0x11B8,0x0042,0x0062,0x0042,0x0062,0x00C7,0x00E7,0x0044,0x0064,
		0x0044,0x0064,0x0044,0x0064,0x0112,0x0113,0x0112,0x0113,0x0228,0x0229,0x0046,0x0066,0x0047,0x0067,0x0048,0x0068,
		0x0048,0x0068,0x0048,0x0068,0x0048,0x0068,0x00CF,0x00EF,0x004B,0x006B,0x004B,0x006B,0x004C,0x006C,0x1E36,0x1E37,
		0x004D,0x006D,0x004D,0x006D,0x004D,0x006D,0x004E,0x006E,0x004E,0x006E,0x00D5,0x00F5,0x00D5,0x00F5,0x014C,0x014D,
		0x014C,0x014D,0x0050,0x0070,0x0050,0x0070,0x0052,0x0072,0x0052,0x0072,0x1E5A,0x1E5B,0x0053,0x0073,0x0053,0x0073,
		0x015A,0x015B,0x0160,0x0161,0x1E62,0x1E63,0x0054,0x0074,0x0054,0x0074,0x0168,0x0169,0x016A,0x016B,0x0056,0x0076,
		0x0056,0x0076,0x0057,0x0077,0x0057,0x0077,0x0057,0x0077,0x0057,0x0077,0x0057,0x0077,0x0058,0x0078,0x0058,0x0078,
		0x0059,0x0079,0x005A,0x007A,0x005A,0x007A,0x0074,0x0077,0x0079,0x017F,0x0041,0x0061,0x0041,0x0061,0x00C2,0x00E2,
		0x00C2,0x00E2,0x00C2,0x00E2,0x00C2,0x00E2,0x1EA0,0x1EA1,0x0102,0x0103,0x0102,0x0103,0x0102,0x0103,0x0102,0x0103,
		0x1EA0,0x1EA1,0x0045,0x0065,0x0045,0x0065,0x0045,0x0065,0x00CA,0x00EA,0x00CA,0x00EA,0x00CA,0x00EA,0x00CA,0x00EA,
		0x1EB8,0x1EB9,0x0049,0x0069,0x0049,0x0069,0x004F,0x006F,0x004F,0x006F,0x00D4,0x00F4,0x00D4,0x00F4,0x00D4,0x00F4,
		0x00D4,0x00F4,0x1ECC,0x1ECD,0x01A0,0x01A1,0x01A0,0x01A1,0x01A0,0x01A1,0x01A0,0x01A1,0x01A0,0x01A1,0x0055,0x0075,
		0x0055,0x0075,0x01AF,0x01B0,0x01AF,0x01B0,0x01AF,0x01B0,0x01AF,0x01B0,0x01AF,0x01B0,0x0059,0x0079,0x0059,0x0079,
		0x0059,0x0079,0x0059,0x0079,0x03B1,0x03B1,0x1F00,0x1F01,0x1F00,0x1F01,0x1F00,0x1F01,0x0391,0x0391,0x1F08,0x1F09,
		0x1F08,0x1F09,0x1F08,0x1F09,0x03B5,0x03B5,0x1F10,0x1F11,0x1F10,0x1F11,0x0395,0x0395,0x1F18,0x1F19,0x1F18,0x1F19,
		0x03B7,0x03B7,0x1F20,0x1F21,0x1F20,0x1F21,0x1F20,0x1F21,0x0397,0x0397,0x1F28,0x1F29,0x1F28,0x1F29,0x1F28,0x1F29,
		0x03B9,0x03B9,0x1F30,0x1F31,0x1F30,0x1F31,0x1F30,0x1F31,0x0399,0x0399,0x1F38,0x1F39,0x1F38,0x1F39,0x1F38,0x1F39,
		0x03BF,0x03BF,0x1F40,0x1F41,0x1F40,0x1F41,0x039F,0x039F,0x1F48,0x1F49,0x1F48,0x1F49,0x03C5,0x03C5,0x1F50,0x1F51,
		0x1F50,0x1F51,0x1F50,0x1F51,0x03A5,0x1F59,0x1F59,0x1F59,0x03C9,0x03C9,0x1F60,0x1F61,0x1F60,0x1F61,0x1F60,0x1F61,
		0x03A9,0x03A9,0x1F68,0x1F69,0x1F68,0x1F69,0x1F68,0x1F69,0x03B1,0x03B5,0x03B7,0x03B9,0x03BF,0x03C5,0x03C9,0x1F00,
		0x1F01,0x1F02,0x1F03,0x1F04,0x1F05,0x1F06,0x1F07,0x1F08,0x1F09,0x1F0A,0x1F0B,0x1F0C,0x1F0D,0x1F0E,0x1F0F,0x1F20,
		0x1F21,0x1F22,0x1F23,0x1F24,0x1F25,0x1F26,0x1F27,0x1F28,0x1F29,0x1F2A,0x1F2B,0x1F2C,0x1F2D,0x1F2E,0x1F2F,0x1F60,
		0x1F61,0x1F62,0x1F63,0x1F64,0x1F65,0x1F66,0x1F67,0x1F68,0x1F69,0x1F6A,0x1F6B,0x1F6C,0x1F6D,0x1F6E,0x1F6F,0x03B1,
		0x03B1,0x1F70,0x03B1,0x03AC,0x03B1,0x1FB6,0x0391,0x0391,0x0391,0x0391,0x007E,0x1F74,0x03B7,0x03AE,0x03B7,0x1FC6,
		0x0395,0x0397,0x0397,0x0060,0x0027,0x007E,0x03B9,0x03B9,0x03CA,0x03B9,0x03CA,0x0399,0x0399,0x0399,0x0060,0x0027,
		0x007E,0x03C5,0x03C5,0x03CB,0x03C1,0x03C1,0x03C5,0x03CB,0x03A5,0x03A5,0x03A5,0x03A1,0x0060,0x1F7C,0x03C9,0x03CE,
		0x03C9,0x1FF6,0x039F,0x03A9,0x03A9,0x002E,0x003C,0x003E,0x002C,0x003C,0x003E,0x002C,0x003D,0x002E,0x006F,0x003C,
		0x003E,0x003F,0x0030,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x002B,0x2212,0x003D,0x0028,0x0029,0x0030,0x0031,
		0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x002B,0x2212,0x003D,0x0028,0x0029,0x0045,0x0043,0x0072,
		0x0072,0x004C,0x006D,0x004E,0x0074,0x0073,0x0057,0x002D,0x0415,0x041E,0x006D,0x006D,0x2190,0x2192,0x2194,0x0338,
		0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,
		0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,0x0338,
		0x0338,0x0338,0x0338,0x0338,0x0338,0x0062,0x0066,0x0023,0x0338,0x005C,0x002F,0x4E00,0x4E8C,0x4E09,0x56DB,0x4E0A,
		0x4E2D,0x4E0B,0x7532,0x4E59,0x4E19,0x4E01,0x5929,0x5730,0x4EBA,0x05D9,0x05F2,0x05E9,0x05E9,0xFB49,0xFB49,0x05D0,
		0x05D0,0x05D0,0x05D1,0x05D2,0x05D3,0x05D4,0x05D5,0x05D6,0x05D8,0x05D9,0x05DA,0x05DB,0x05DC,0x05DE,0x05E0,0x05E1,
		0x05E3,0x05E4,0x05E6,0x05E7,0x05E8,0x05E9,0x05EA,0x05D5,0x05D1,0x05DB,0x05E4
	};
}
#endif
