; VL Verilog Toolkit
; Copyright (C) 2008-2011 Centaur Technology
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; This program is free software; you can redistribute it and/or modify it under
; the terms of the GNU General Public License as published by the Free Software
; Foundation; either version 2 of the License, or (at your option) any later
; version.  This program is distributed in the hope that it will be useful but
; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
; more details.  You should have received a copy of the GNU General Public
; License along with this program; if not, write to the Free Software
; Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
;
; Original author: Jared Davis <jared@centtech.com>

(in-package "VL")
(include-book "../mlib/find-module")
(local (include-book "../util/arithmetic"))

(defxdoc drop-blankports
  :parents (transforms)
  :short "Eliminate \"blank ports\" from modules and delete all corresponding
arguments from module instances."

  :long "<p>For background on blank ports and argument see @(see vl-port-p) and
@(see vl-plainarg-p).</p>

<p>Blank ports are completely useless.  This transform removes blank ports from
all modules and simultaneously removes all arguments given to blank ports from
all instances of all modules.</p>

<p>Prerequisites.  We expect to only see plain argument lists, i.e., @(see
argresolve) should have already been run.  There are no other requirements.
However, note that we do not respect @(see vl-module->hands-offp) since it does
not seem legitimate to do so: the correctness of this transformation requires
that the instances throughout every module be updated.</p>

<p>We may add fatal warnings if we find instances of undefined modules or arity
mismatches.  But we do <i>not</i> add any warnings about connections to blank
ports because we expect @(see argresolve) to have done that.</p>")

(define vl-plainarglist-drop-blankports
  ((args  "arguments to some module instance"      vl-plainarglist-p)
   (ports "corresponding ports from the submodule" vl-portlist-p))
  :guard (same-lengthp args ports)
  :returns (new-args "copy of @('args') with blank ports removed"
                     vl-plainarglist-p :hyp :fguard)
  :parents (drop-blankports)
  :short "Drop arguments to blank ports from an plain argument list."

  (cond ((atom args)
         nil)
        ((vl-port->expr (car ports))
         (cons (car args)
               (vl-plainarglist-drop-blankports (cdr args) (cdr ports))))
        (t
         (vl-plainarglist-drop-blankports (cdr args) (cdr ports)))))

(define vl-modinst-drop-blankports
  ((x        "module instance to rewrite" vl-modinst-p)
   (mods     "list of all modules" vl-modulelist-p)
   (modalist (equal modalist (vl-modalist mods)))
   (warnings vl-warninglist-p))
  :returns (mv (warnings vl-warninglist-p :hyp :fguard)
               (new-x    vl-modinst-p     :hyp :fguard))
  :parents (drop-blankports)
  :short "Drop arguments to blank ports from a module instance."

  (b* (((vl-modinst x) x)

       (target-mod (vl-fast-find-module x.modname mods modalist))
       ((unless target-mod)
        (b* ((w (make-vl-warning
                 :type :vl-bad-instance
                 :msg "~a0 refers to undefined module ~m1."
                 :args (list x x.modname)
                 :fatalp t
                 :fn 'vl-modinst-drop-blankports)))
          (mv (cons w warnings) x)))

       ((when (vl-arguments->namedp x.portargs))
        (b* ((w (make-vl-warning
                 :type :vl-programming-error
                 :msg "~a0: expected all modules to be using plain arguments, ~
                       but found named arguments.  Did you forget to run the ~
                       argresolve transform first?"
                 :args (list x)
                 :fatalp t
                 :fn 'vl-modinst-drop-blankports)))
          (mv (cons w warnings) x)))

       (ports         (vl-module->ports target-mod))
       (plainargs     (vl-arguments->args x.portargs))

       ((unless (same-lengthp plainargs ports))
        (b* ((w (make-vl-warning
                 :type :vl-bad-instance
                 :msg "~a0: bad arity.  Expected ~x1 arguments but found ~x2 ~
                       arguments."
                 :args (list x (len ports) (len plainargs))
                 :fatalp t
                 :fn 'vl-modinst-drop-blankports)))
          (mv (cons w warnings) x)))

       (new-plainargs (vl-plainarglist-drop-blankports plainargs ports))
       (new-arguments (vl-arguments nil new-plainargs))
       (new-x         (change-vl-modinst x :portargs new-arguments)))

    (mv warnings new-x)))

(define vl-modinstlist-drop-blankports
  ((x        "modinsts to rewrite" vl-modinstlist-p)
   (mods     "list of all modules" vl-modulelist-p)
   (modalist (equal modalist (vl-modalist mods)))
   (warnings vl-warninglist-p))
  :returns (mv (warnings vl-warninglist-p :hyp :fguard)
               (new-x    vl-modinstlist-p :hyp :fguard))
  :parents (drop-blankports)
  :short "Drop arguments to blank ports from module instances."

  (b* (((when (atom x))
        (mv warnings nil))
       ((mv warnings car)
        (vl-modinst-drop-blankports (car x) mods modalist warnings))
       ((mv warnings cdr)
        (vl-modinstlist-drop-blankports (cdr x) mods modalist warnings)))
    (mv warnings (cons car cdr))))

(define vl-portlist-drop-blankports ((x vl-portlist-p))
  :returns (new-x vl-portlist-p :hyp :fguard)
  :parents (drop-blankports)
  :short "Drop any blank ports from a list of ports."

  (cond ((atom x)
         nil)
        ((vl-port->expr (car x))
         (cons (car x) (vl-portlist-drop-blankports (cdr x))))
        (t
         (vl-portlist-drop-blankports (cdr x)))))

(define vl-module-drop-blankports
  ((x    "module to rewrite"   vl-module-p)
   (mods "list of all modules" vl-modulelist-p)
   (modalist (equal modalist (vl-modalist mods))))
  :returns (new-x vl-module-p :hyp :fguard)
  :parents (drop-blankports)
  :short "Drop any blank ports from a module, and simultaneously remove all
arguments to blank ports from all module instances within the module."

  (b* (((vl-module x) x)
       (ports (vl-portlist-drop-blankports x.ports))
       ((mv warnings modinsts)
        (vl-modinstlist-drop-blankports x.modinsts mods modalist x.warnings)))
    (change-vl-module x
                      :ports ports
                      :modinsts modinsts
                      :warnings warnings))
  ///
  (defthm vl-module->name-of-vl-module-drop-blankports
    (let ((ret (vl-module-drop-blankports x mods modalist)))
      (equal (vl-module->name ret)
             (vl-module->name x)))))

(defprojection vl-modulelist-drop-blankports-aux (x mods modalist)
  (vl-module-drop-blankports x mods modalist)
  :guard (and (vl-modulelist-p x)
              (vl-modulelist-p mods)
              (equal modalist (vl-modalist mods)))
  :result-type vl-modulelist-p
  :parents (drop-blankports)
  :rest
  ((defthm vl-modulelist->names-of-vl-modulelist-drop-blankports-aux
     (let ((ret (vl-modulelist-drop-blankports-aux x mods modalist)))
       (equal (vl-modulelist->names ret)
              (vl-modulelist->names x))))))

(define vl-modulelist-drop-blankports ((x vl-modulelist-p))
  :returns (new-x vl-modulelist-p :hyp :fguard)
  :parents (drop-blankports)
  :short "Top-level @(see drop-blankports) transformation."

  (b* ((modalist (vl-modalist x))
       (new-x    (vl-modulelist-drop-blankports-aux x x modalist)))
    (fast-alist-free modalist)
    new-x)
  ///
  (defthm vl-modulelist->names-of-vl-modulelist-drop-blankports
    (equal (vl-modulelist->names (vl-modulelist-drop-blankports x))
           (vl-modulelist->names x))))

