; ACL2 Version 3.4 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2008 University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic
; NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Regarding authorship of ACL2 in general:

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; The original version of this file was contributed by Bob Boyer and
; Warren A. Hunt, Jr.  The design of this system of Hash CONS,
; function memoization, and fast association lists (applicative hash
; tables) was initially implemented by Boyer and Hunt.  The code has
; been augmented by Matt Kaufmann, Sol Swords, and others.

(in-package "ACL2")

;; FEATURES

(eval-when (:execute :compile-toplevel :load-toplevel)

; #+Clozure (pushnew :parallel *features*)

; #+Clozure (pushnew :Sol *features*)

  #+Clozure
  (if (fboundp 'ccl::rdtsc) (pushnew :RDTSC *features*))

)

;; COMPILATION OPTIONS

; (setq *compile-verbose* t)

; (setq *load-print* t)

; (setq *load-verbose* t)


;;;;;;;; UTILITIES ;;;;;;;; UTILITIES ;;;;;;;; UTILITIES
;;;;;;;; UTILITIES ;;;;;;;; UTILITIES ;;;;;;;; UTILITIES


; DEFGLOBAL

(defmacro defg (&rest r)

  "DEFG is a short name for DEFPARAMETER.  However, in CCL, its use
  includes three promises: (1) never to locally bind the variable,
  e.g., with LET or LAMBDA, (2) never to reference the variable when
  it is not set, which would be an error anyway, of course, and (3)
  never to test whether the variable is BOUNDP.  CCL uses about ten
  fewer machine instructions to reference such a variable."

  #-Clozure
  `(defparameter ,@r)
  #+Clozure
  `(ccl::defstatic ,@r))

; PRINL

(defmacro prinl (&rest r)

  "PRINL is for debugging.  PRINL PRIN1s the members of r and their
  values.  For example, (prinl a b (+ a b)) might print:
    A 
    => 1
    B
     => 2
    (+ A B)
     => 3
  PRINL returns the principal value of the last member of r.  PRINL
  does not evaluate the members of r that are neither symbols nor
  conses, but it does PRINC those members.  PRINL leaves (ofd ...)
  forms in place."

  (let ((tem (make-symbol "TEM")))
    `(our-syntax-nice
      (let ((,tem nil))
        ,@(loop
           for x in r collect
           (cond
            ((or (and (consp x)
                      (not (eq (car x) 'ofd)))
                 (symbolp x))
             `(progn (ofd "~&~:s ~10t " ',x)
                     (let ((val (ofn "~:d" (setq ,tem ,x))))
                       (cond ((< (length val) 40)
                              (ofd "=> ~a" ,tem))
                             (t (ofd "~%   => ~:d" ,tem))))))
            ((and (consp x) (eq (car x) 'ofd))
             x)
            (t `(ofd "~&~:d" (setq ,tem ',x)))))
        ,tem))))

(defg *number-of-arguments-and-values-ht*
  (let ((ht (make-hash-table)))
    (loop for pair in
      '((apropos . (nil . 0))
        (aref . (nil . 1))
        (array-displacement . (1 . 2))
        (decode-float . (1 . 3))
        (find-symbol . (nil . 2))
        (function . (nil . 1))
        (get-properties . (2 . 3))
        (gethash . (nil . 2))
        (integer-decode-float (1 . 3))
        (intern . (nil . 2))
        (lambda . (nil . 1))
        (list . (nil . 1))
        (list* . (nil . 1))
        (macroexpand . (nil . 2))
        (macroexpand-1 . (nil . 2))
        (pprint-dispatch  . (nil . 2))
        (prog1 . (nil . 1))
        (prog2 . (nil . 1))
        (quote . (1 . 1))) do
      (setf (gethash (car pair) ht)
            (cdr pair)))
    (loop for sym in
          '(car cdr caar cadr cdar cddr caaar cadar cdaar
            cddar caadr caddr cdadr cdddr caaaar cadaar cdaaar cddaar
            caadar caddar cdadar cdddar caaadr cadadr cdaadr cddadr
            caaddr cadddr cdaddr cddddr) do
            (setf (gethash sym ht) '(1 . 1)))
    ht)

  "The hash table *NUMBER-OF-ARGUMENTS-AND-VALUES-HT* maps a symbol fn
  to a cons pair (a . d), where a is the number of inputs and d is the
  number of outputs of fn.  NIL for a or d indicates 'don't know'.")

(defun tlist (n) (make-list n :initial-element t))

(defmacro defn1 (f a &rest r)
  `(progn
     (setf (gethash ',f *number-of-arguments-and-values-ht*)
           (cons ,(length a) 1))
     (declaim (ftype (function ,(tlist (len a)) (values t)) ,f))
     (defun ,f ,a (declare (xargs :guard t)) ,@r)))

(defmacro defn1-one-output (f a &rest r)
  `(progn
     (setf (gethash ',f *number-of-arguments-and-values-ht*)
           (cons ,(length a) 1))
     (declaim (ftype (function ,(tlist (len a)) (values t)) ,f))
     (defun-one-output ,f ,a (declare (xargs :guard t)) ,@r)))

(defmacro defn2 (f a &rest r)
  `(progn
     (setf (gethash ',f *number-of-arguments-and-values-ht*)
           (cons ,(length a) 2))
     (declaim (ftype (function ,(tlist (len a)) (values t t)) ,f))
     (defun ,f ,a (declare (xargs :guard t)) ,@r)))


; TIMING UTILITIES

; *float-ticks/second* is set correctly by hons-init.
(defg *float-ticks/second* 1.0)
(defg *float-internal-time-units-per-second*
  (float internal-time-units-per-second))

(declaim (float *float-ticks/second*
                *float-internal-time-units-per-second*))

(defn1 internal-real-time ()
  (the fixnum
    (let ((n #+RDTSC (ccl::rdtsc)
             #-RDTSC (get-internal-real-time)))
      #+RDTSC
      n
      #-RDTSC
      (if (typep n 'fixnum)
          n
        (error "~&; Error: ** get-internal-real-time returned a ~
                 nonfixnum.")))))

(defn1 float-ticks/second-init ()
  (setq *float-ticks/second*
        #+RDTSC
        (let ((i1 (ccl::rdtsc64))
              (i2 (progn (sleep .01) (ccl::rdtsc))))
          (if (>= i2 i1)
              (* 100 (float (- i2 i1)))
            (ofe "(float-ticks/second-init).")))
        #-RDTSC
        *float-internal-time-units-per-second*)
  (check-type *float-ticks/second*
              (and float (satisfies plusp))))


; VERY-UNSAFE-INCF

(defmacro very-unsafe-incf (x inc &rest r)

; returns NIL !

  (declare (ignore r))
  (unless (symbolp x)
    (error "Do not use very-unsafe-incf on conses: ~a" x))
  `(locally (declare (fixnum ,x))
            (setq ,x
                  (the fixnum (+ ,x
                                 (the fixnum ,inc))))
            nil))

; SAFE-INCF

; *WATCH-FILE* is bound by WATCH-DO, so we use DEFPARAMETER rather
; than DEFG.
(defparameter *watch-file* nil
  "If *WATCH-FILE* is not NIL, it is the name of the file to which
  watch-do will print.")

(defg *watch-string*
  (make-array 0 :element-type 'character :adjustable t
              :fill-pointer t)
  "After (WATCH) has been invoked, *WATCH-STRING* is a header for the
  *watch-file*.")
(declaim (type (array character (*)) *watch-string*))

(defg *memoize-safe-incf-counter* most-positive-fixnum
  "Whenever a counter is incremented by SAFE-INCF, the
  value of *MEMOIZE-SAFE-INCF-COUNTER* is checked.  If it is 0, then
  *MEMOIZE-SAFE-INCF-COUNTER* is reset and WATCH-DO1 is called.
  Otherwise, it is decremented by 1.")

(defg *memoize-safe-incf-delta* 1000000)

(declaim (fixnum *memoize-safe-incf-delta*
                 *memoize-safe-incf-counter*))

(defmacro safe-incf (x inc &optional where)

  "SAFE-INCF is a raw Lisp macro that behaves the same as INCF when
  both X and INCF are nonnegative fixnums and their sum is a
  nonnegative FIXNUM.  In a call of (SAFE-INCF x inc), X must be a
  place that holds a FIXNUM.  INC must evaluate to a FIXNUM.  Both X
  and INC must evaluate without side effects, so that it is impossible
  to tell which was executed first or whether only one or both were
  executed.  If INC is not positive, no update takes place at all.
  Otherwise, if the sum of the values of X and INC is not a FIXNUM,
  which is tested without causing an error, a run-time error will be
  caused.  Else, if the sum is a FIXNUM then, as with INCF, the place
  X will be set to hold the sum of the old value of that place and the
  value of INC.  The value returned by SAFE-INCF is NIL.  Caution: INC
  may be evaluated first, which is why side effects are prohibited.

  An optional third parameter is merely to help with error location
  identification.

  In (SAFE-INCF (AREF A (FOO)) INC), (FOO) is only evaluted once.
  Same for SVREF."

  (cond ((integerp inc)
         (cond ((<= inc 0) nil)
               (t `(safe-incf-aux ,x ,inc ,where))))
        ((symbolp inc)
         `(cond ((>= 0 (the fixnum ,inc)) nil)
                (t (safe-incf-aux ,x ,inc ,where))))
        (t (let ((incv (make-symbol "INCV")))
             `(let ((,incv (the fixnum ,inc)))
                (declare (fixnum ,incv))
                (cond ((>= 0 ,incv) nil)
                      (t (safe-incf-aux ,x ,incv ,where))))))))

(defn1 safe-incf-aux-error (x inc where)
  (error "~%; SAFE-INCF-AUX: ** Error: ~a."
         (list :x x :inc inc :where where)))

(defmacro safe-incf-aux (x inc where)
  (cond
   ((not (or (symbolp inc)
             (and (typep inc 'fixnum)
                  (> inc 0))))
    (safe-incf-aux-error x inc where))
   ((and (true-listp x)
         (equal (len x) 3)
         (member (car x) '(aref svref))
         (symbolp (nth 1 x))
         (consp (nth 2 x)))
    (let ((idx (make-symbol "IDX")))
      `(let ((,idx (the fixnum ,(nth 2 x))))
         (declare (fixnum ,idx))
         (safe-incf (,(nth 0 x)
                     ,(nth 1 x)
                     ,idx)
                    ,inc
                    ',where))))
   (t (let ((v (make-symbol "V")))
        `(let ((,v (the fixnum ,x)))
           (declare (fixnum ,v))
           (cond ((<= ,v (the fixnum (- most-positive-fixnum
                                        (the fixnum ,inc))))
                  
                  (setf (the fixnum ,x)
                        (the fixnum (+ ,v (the fixnum ,inc))))
                  nil)
                 (t (safe-incf-aux-error ',x ',inc ',where))))))))

(defmacro memoize-incf ()
  (let ((w (make-symbol "W")))
    `(let ((,w (the fixnum *memoize-safe-incf-counter*)))
       (declare (fixnum ,w))
       (cond ((eql ,w 0) (watch-do))
             (t (setf (the fixnum *memoize-safe-incf-counter*)
                      (the fixnum (1- ,w))))))))

; PARALLEL

; (pushnew :parallel *features*) causes a lot of locking that is of no
; value in a sequentially executing system.

; We have attempted to make honsing, memoizing, and Emod-compilation
; 'thread safe', whatever in hell that means, but we have no idea what
; we are really doing and are simply coding based upon what we feel is
; intuitive common sense.  Very subtle stuff.

(defmacro unwind-mch-lock (&rest forms)

; Returns NIL.

  #+parallel
  (let ((v (make-symbol "V")))
    `(let ((,v nil))
       (unwind-protect
         (progn
           (ccl::lock-hash-table *hons-str-ht*)
           (ccl::lock-hash-table *compiled-module-ht*)
           (ccl::lock-hash-table *memoize-info-ht*)
           ,@forms
           (setq ,v t)
           nil)
         (ccl::unlock-hash-table *memoize-info-ht*)
         (ccl::unlock-hash-table *compiled-module-ht*)
         (ccl::unlock-hash-table *hons-str-ht*)
         (unless ,v (ofe "unwind-mch-lock failure.")))))
  #-parallel
  `(progn ,@forms nil))

#+parallel
(unless (member :Clozure *features*)
  (error "We use CCL primitives for parallelism."))

; We limit our efforts at thread-safedness to locking/unlocking some
; hash tables.

(declaim (special *hons-cdr-ht* *compiled-module-ht*
                  *memoize-info-ht*))

; Lock order.  To avoid deadlock, we always lock HONS before COMPILED
; and COMPILED before MEMOIZE; we unlock in the exact reverse order.

; If there is a pons table for a memoized function, it may be locked
; and unlocked during the execution of that memoized function, to
; ponsing together new arguments to a multiple argument function.  It
; would be DEADLY PONS could result in honsing, memoizing, or
; Emod-compiling.

(defmacro our-lock-unlock-ht1 (ht &rest r)
  (declare (ignorable ht))
  #+parallel
  `(progn (ccl::lock-hash-table ,ht)
          (prog1 ,@r
                 (ccl::unlock-hash-table ,ht)))
  #-parallel `(prog1 ,@ r))

(defmacro our-lock-unlock-hons1 (&rest r)
  `(our-lock-unlock-ht1 *hons-str-ht* ,@r))

(defmacro our-lock-unlock-compile1 (&rest r)
  `(our-lock-unlock-hons1
    (our-lock-unlock-ht1 *compiled-module-ht* ,@r)))

(defmacro our-lock-unlock-memoize1 (&rest r)
  `(our-lock-unlock-compile1
    (our-lock-unlock-ht1 *memoize-info-ht* ,@r)))

(defmacro our-lock-unlock-htmv1 (ht &rest r)
  (declare (ignorable ht))
  #+parallel
  `(progn (ccl::lock-hash-table ,ht)
          (multiple-value-prog1 ,@r
                 (ccl::unlock-hash-table ,ht)))
  #-parallel `(multiple-value-prog1 ,@r))

(defmacro our-lock-unlock-honsmv1 (&rest r)
  `(our-lock-unlock-htmv1 *hons-str-ht* ,@r))

(defmacro our-lock-unlock-compilemv1 (&rest r)
  `(our-lock-unlock-honsmv1
    (our-lock-unlock-htmv1 *compiled-module-ht* ,@r)))

(defmacro our-lock-unlock-memoizemv1 (&rest r)
  `(our-lock-unlock-compilemv1
    (our-lock-unlock-htmv1 *memoize-info-ht* ,@r)))


;  OUR-SYNTAX

(defg *print-pprint-dispatch-orig* *print-pprint-dispatch*)

(defmacro our-syntax (&rest args)

  "OUR-SYNTAX is derived from Common Lisp's WITH-STANDARD-IO-SYNTAX;
  we note below with an asterisk lines that differ from
  WITH-STANDARD-IO-SYNTAX.

  These settings are oriented towards reliable, standard, vanilla,
  mechanical reading and printing, but not towards debugging or human
  interaction.  Please, before changing the following, consider
  existing uses of this macro insofar as the changes might impact
  reliable, standard, vanilla, mechanical printing.  Especially
  consider COMPACT-PRINT-FILE.  For mere convenience, where
  abbreviated output might be appropriate, consider using
  OUR-SYNTAX-NICE rather than OUR-SYNTAX."

  `(let ((*package*                    *acl2-package*) ; *
         (*print-array*                t)
         (*print-base*                 10)
         (*print-case*                 :upcase)
         (*print-circle*               nil)
         (*print-escape*               t)
         (*print-gensym*               t)
         (*print-length*               nil)
         (*print-level*                nil)
         (*print-lines*                nil)
         (*print-pretty*               nil)
         (*print-radix*                nil)
         (*print-readably*             t)
         (*print-right-margin*         nil)
         (*print-pprint-dispatch*      *print-pprint-dispatch-orig*)
         (*read-base*                  10)
         (*read-default-float-format*  'single-float)
         (*print-miser-width*          nil)
         (*read-eval*                  nil)            ; *
         (*read-suppress*              nil)
         (*readtable*                  *acl2-readtable*))  ; *
     ,@args))

(defmacro our-syntax-nice (&rest args)

; for more pleasant human interaction

  `(let ((*package*                    *acl2-package*)
         (*print-array*                t)
         (*print-base*                 10)
         (*print-case*                 :downcase)
         (*print-circle*               nil)
         (*print-escape*               t)
         (*print-gensym*               t)
         (*print-length*               nil)
         (*print-level*                nil)
         (*print-lines*                nil)
         (*print-pretty*               t)
         (*print-radix*                nil)
         (*print-readably*             nil)
         (*print-right-margin*         70)
         (*print-pprint-dispatch*      *print-pprint-dispatch-orig*)
         (*read-base*                  10)
         (*read-default-float-format*  'single-float)
         (*print-miser-width*          100)
         (*read-eval*                  nil)
         (*read-suppress*              nil)
         (*readtable*                  *acl2-readtable*))
     ,@args))

(defmacro ofd (&rest r) ; For writing to *debug-io*.
  `(format *debug-io* ,@r))

(defg *hons-verbose* t)
(defg *ofv-note-printed* nil)
(defg *ofv-msg-list* nil)
(defg *ofe-msg-list* nil)
(defg *ofb-msg-list* nil)

(defn1 ofv (&rest r) ; For verbose but helpful info.
  (our-syntax-nice
   (when *hons-verbose*
     (format *debug-io* "~%; Aside:  ")
     (let ((*print-level* 3)
           (*print-length* 5))
       (let ((ab (loop for x in r collect (abbrev x))))
         (apply #'format *debug-io* ab)
         (when (loop for x in ab as y in r thereis (not (eq x y)))
           (push (cons 'ofv r) *ofv-msg-list*)
           (format *debug-io* "~%; See *ofv-msg-list*."))
         (unless *ofv-note-printed*
           (format *debug-io*
                   "~%; Aside:  Do (setq acl2::*hons-verbose* nil) ~
                    to suppress asides.")
           (setq *ofv-note-printed* t))))
     (force-output *debug-io*))))

(defn1 ofvv (&rest r) ; For very verbose but helpful info.
  (our-syntax-nice
   (when (and (integerp *hons-verbose*) (> *hons-verbose* 1))
     (format *debug-io* "~%; Aside:  ")
     (let ((*print-level* 3) (*print-length* 5))
       (let ((ab (loop for x in r collect (abbrev x))))
         (apply #'format *debug-io* ab)
         (when (loop for x in ab as y in r thereis (not (eq x y)))
           (push (cons 'ofv r) *ofv-msg-list*)
           (format *debug-io* "~%; See *ofv-msg-list*."))
         (unless *ofv-note-printed*
           (format *debug-io*
                   "~%; Aside:  Do (setq acl2::*hons-verbose* nil) ~
                    to suppress asides.")
           (setq *ofv-note-printed* t))))
     (force-output *debug-io*))))

(defmacro ofg (&rest r) ; For verbose gc info.
    `(when *hons-verbose*
       (format *debug-io* ,@r)
       (force-output *debug-io*)))

(defn1 ofe (&rest r)  ; For writing to *error-output*; calls (error).
  (our-syntax-nice
   (format *error-output* "~%; ** Error:  ")
   (let ((*print-level* 3) (*print-length* 5))
     (let ((ab (loop for x in r collect (abbrev x))))
       (apply #'format *error-output* ab)
       (when (loop for x in ab as y in r thereis (not (eq x y)))
         (push (cons 'ofe r) *ofe-msg-list*)
         (format *error-output* "~%; See *ofe-msg-list*."))
       (force-output *error-output*)
       (error "")))))

(defn1 ofw (&rest r) ; For writing to *debug-io*, with a warning.
  (our-syntax-nice
   (format *debug-io* "~%; ** Warning:  ")
   (apply #'format *debug-io* r)
   (force-output *debug-io*)))

(defn1 ofb (&rest r) ; For writing to *debug-io* and breaking.
  (our-syntax-nice
   (format *debug-io* "~%; ** Warning and break:  ")
   (let ((*print-level* 3) (*print-length* 5))
     (let ((ab (loop for x in r collect (abbrev x))))
       (apply #'format *debug-io* ab)
       (when (loop for x in ab as y in r thereis (not (eq x y)))
         (push (cons 'ofe r) *ofb-msg-list*)
         (format *error-output* "~%; See *ofb-msg-list*."))
       (force-output *debug-io*)
       (error "")))
   (break "ofb")))

(defmacro ofn (&rest r) ; For forming strings.
  `(format nil ,@r))

(defn1 ofnum (n) ; For forming numbers.
  (check-type n number)
  (if (= n 0) (setq n 0))
  (cond ((typep n '(integer -99 999))
         (format nil "~d" n))
        ((or (< -1000 n -1/100)
             (< 1/100 n 1000))
         (format nil "~,2f" n))
        (t (format nil "~,1e" n))))

(defmacro ofni (&rest r) ; For forming symbols in package ACL2.
  `(our-syntax (intern (format nil ,@r) *acl2-package*)))

(defmacro ofnm (&rest r) ; For forming uninterned symbols.
  `(our-syntax (make-symbol (format nil ,@r))))

(defmacro oft (&rest r) ; For writing to *standard-output*.
  `(progn (format t ,@r) (force-output *standard-output*)))

(defmacro oftr (&rest r) ; For writing to *trace-output*.
  `(progn (format *trace-output* ,@r) (force-output *trace-output*)))

(defn1 suffix (str sym)
  (check-type str string)
  (check-type sym symbol)
  (let ((spkn (package-name (symbol-package sym)))
        (sn (symbol-name sym)))
    (ofn "~s,~s,~s" str spkn sn)))


;  PHYSICAL MEMORY

#-Clozure
(defn1 physical-memory ()
  (cond ((probe-file "/proc/meminfo")
         (let* (n kb
                  (key "MemTotal:")
                  (info (with-open-file (si "/proc/meminfo")
                          (with-output-to-string
                            (so)
                            (let (c)
                              (loop while (setq c (read-char
                                                   si nil nil))
                                    do (write-char c so))))))
                  (loc (search key info)))
           (our-syntax
            (with-input-from-string
             (s info :start (+ loc (length key)))
             (setq n (read s))
             (setq kb (read s))
             (cond ((and (integerp n) (equal kb 'kb))
                    (* n 1024)))))))))

#+Clozure
(with-standard-io-syntax
 (let ((*package* (find-package "CCL")))
   (eval (read-from-string "

   ;;; Work of Gary Byers.

   ;;; The #_ and #$ reader macros in the code below are part of
   ;;; CCL's ffi; you'd basically need to hide this code in
   ;;; a file that's isolated from other implementations.
   (defun acl2::physical-memory ()
      #+darwin-target
      (rlet ((count :mach_msg_type_number_t #$HOST_BASIC_INFO_COUNT)
             (info :host_basic_info))
        (if (eql 0 (#_host_info (#_mach_host_self)
                                #$HOST_BASIC_INFO
                                info
                                count))
          (pref info :host_basic_info.max_mem)))
      #+freebsd-target
       (rlet ((ret :unsigned-doubleword 0)
              (mib (:array :uint 2))
              (oldsize :uint (ccl::record-length
                               :unsigned-doubleword)))
         (setf (paref mib (:* :uint) 0) #$CTL_HW
               (paref mib (:* :uint) 1) #$HW_PHYSMEM)
         (if (eql 0 (#_sysctl mib 2 ret oldsize (%null-ptr) 0))
           (pref ret unsigned-doubleword)))
      #+linux-target
      (rlet ((info :sysinfo))
        (if (eql 0 (#_sysinfo info))
          (pref info :sysinfo.totalram))))"))))


; NUMBER OF ARGS AND RETURN VALUES

(defn io-error-fn (fn)
  (ofe "Could not determine a signature for ~a.  ~%; To assert ~
        that ~a takes, say, 2 inputs and returns 1 output, do ~% ~
        (setf (gethash '~a ~ ~
        acl2::*number-of-arguments-and-values-ht*) (cons 2 1))."
       fn fn fn))

(defn1 number-of-arguments (fn)

; A NIL value returned by NUMBER-OF-ARGUMENTS means 'don't know'.

  (let* ((state *the-live-state*)
         (w (w state))
         (pair (gethash fn *number-of-arguments-and-values-ht*)))
    (cond
     ((not (symbolp fn)) nil)
     ((and (consp pair) (integerp (car pair))) (car pair))
     ((let ((formals (getprop fn 'formals t 'current-acl2-world w)))
        (and (not (eq t formals))
             (length formals))))
     ((not (fboundp fn)) nil)
     ((macro-function fn) nil)
     ((special-operator-p fn) nil)
     #+Clozure
     ((multiple-value-bind (req opt restp keys)
          (ccl::function-args (symbol-function fn))
        (and (null keys)
             (null restp)
             (and (integerp req)
                  (eql opt 0))
             req)))
     (t nil))))

(defg lets-and-ifs '(let mv-let let* if g-if flet cond g-cond progn))

(defg fns-and-macros-that-return-last

; The ACL2 functions in FNS-AND-MACROS-THAT-RETURN-LAST, like the
; function IF, do not return a fixed number of values, which should
; scare the reader half to death.  The current value of
; FNS-AND-MACROS-THAT-RETURN-LAST was calculated with this raw Lisp
; code:
 
#||
  
 (loop for x in 
  (functions-defined-in-file "stp:acl2;axioms.lisp")
  when (and (macro-function x)
            (not (eq t (fgetprop x 'formals t 
                                    (w *the-live-state*)))))
  collect x)

||#

; The members of FNS-AND-MACROS-THAT-RETURN-LAST all have a
; 'predefined property of T, and thus (for example) are illegal for
; FLET.

; The logical meanings of the members of
; FNS-AND-MACROS-THAT-RETURN-LAST are given by this theorem:

#||
  (thm
   (and 
    (equal (must-be-equal x y) x)
    (equal (prog2$ x y) y)
    (equal (time$ x) x)
    (equal (with-prover-time-limit x y) y)))
||#

; Furthermore, guard checking insures that when MUST-BE-EQUAL is
; called within guard-checked code, MUST-BE-EQUAL's two arguments will
; be EQUAL, so the value of either argument may be returned.

; GTRANS compiles a call to a member of
; FNS-AND-MACROS-THAT-RETURN-LAST as though it were simply a call of
; the last argument.  However, the overly scrupulous may detect that
; some supposedly undetectable behavior takes place, or vice versa;
; are we talking Alice in Wonderland?  For example, TIME$'s message
; will not get printed, and a computation requested via a call of
; WITH-PROVER-TIME-LIMIT will NOT be stopped, as requested, from
; running over the given time-limit.  This behavior can be thought of
; as somewhat like the treatment of:

;        (thm (equal (time$ 1) 1))

; in which TIME$ does not print because ACL2 is proving a theorem
; rather than executing code, whatever that undetectable difference
; may mean.  That we find our selves talking this way may be of some
; relief to those who have been deemed insane for asserting that
; anathema are those who do not believe that for certain x, y, and p,
; x=y, p(x), and -p(y).  Cf. the Athanasian Creed.

; NUMBER-OF-RETURN-VALUES-FORM depends upon
; FNS-AND-MACROS-THAT-RETURN-LAST.

  '(must-be-equal prog2$ time$ with-prover-time-limit))

(defn1 number-of-return-values (fn)

; A NIL value returned by NUMBER-OF-RETURN-VALUES means 'don't know'.

  (let*
    ((pair (gethash fn *number-of-arguments-and-values-ht*))
     (state *the-live-state*)
     (w (w state)))
    (cond
     ((not (symbolp fn)) nil)
     ((and (consp pair) (integerp (cdr pair))) (cdr pair))
     ((or (eq fn 'if)
          (member fn lets-and-ifs)
          (member fn fns-and-macros-that-return-last))
      (ofe "number-of-return-values: It is curious to ask about ~
            'the' number of return values of ~a because the answer ~
            is that it depends."
           fn))
     ((not (eq t (getprop fn 'formals t 'current-acl2-world w)))
      (len (stobjs-out fn w))))))

(defn1 event-number (fn)
  (check-type fn symbol)
  (fgetprop fn 'absolute-event-number t (w *the-live-state*)))


;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS
;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS


; HONS VARIABLES, MACROS, AND DATA STRUCTURES

; Gary Byers recalls Lisp folklore that alists are faster than hash
; tables up to length 18.

(defg *start-car-ht-size*            18)

(defg *hons-acons-ht-threshold*      18)

(defg *hons-cdr-ht-size*             (expt 2 20))

(defg *nil-ht-size*                  (expt 2 17))

(defg *mht-default-size*                    60)

(defg *mht-default-rehash-size*             1.5)

(defg *mht-default-rehash-threshold*        0.7)

(defg *mht-default-weak*                    nil)

(defg *mht-default-shared* #+Parallel t
                             #-Parallel nil)

(defg *weak-ok-in-mht* nil)

(declaim (fixnum *start-car-ht-size* *hons-acons-ht-threshold*
                 *mht-default-size* *nil-ht-size* *hons-cdr-ht-size*)
         (float *mht-default-rehash-size*
                *mht-default-rehash-threshold*))

(defmacro mht (&key (test         (quote (function eql)))
                    (size         *mht-default-size*)
                    (shared       *mht-default-shared*)
                    (rehash-size  *mht-default-rehash-size*)
                    (rehash-threshold *mht-default-rehash-threshold*)
                    (weak         *mht-default-weak*))
  (declare (ignorable shared weak))
  `(make-hash-table :test             ,test
                    :size             ,size
                    :rehash-size      ,rehash-size
                    :rehash-threshold ,rehash-threshold
                    #+Clozure :weak
                    #+Clozure (and *weak-ok-in-mht* ,weak)
                    #+Clozure :shared #+Clozure ,shared
                    ))

(defg *count-hons-calls*                   t
  "If *COUNT-HONS-CALLS*, then each call of HONS increments
  *HONS-CALL-COUNTER* by 1, and each call of HONS that does not find
  the desired HONS to already exist increments *HONS-MISSES-COUNTER*
  by 1.")

(defg *count-pons-calls*                   t
  "If *COUNT-PONS-CALLS*, then each call of PONS increments
  *PONS-CALL-COUNTER* by 1, and each call of PONS that does not find
  the desired PONS to already exist increments *PONS-MISSES-COUNTER*
  by 1.

  Warning:  because the hons and/or pons call and hit counters may not
  be protected by locks, hons and/or pons call/hit info is remotely
  possibly somewhat low.")

(defg *break-honsp*                        nil)

(defg *hons-report-discipline-failure*     'break)

; The choice of :weak below deserves careful explanation. !!

(defg *hons-cdr-ht*        (mht :test #'eq :weak :key))

(defg *hons-cdr-ht-eql*    (mht))

(defg *nil-ht*             (mht :weak :value))

(defg *hons-acons-ht*      (mht :test #'eq :weak :key))

; We use *HONS-STR-HT* for a crucial lock, and so it can never ever
; change.

(defg *hons-str-ht*        (make-hash-table :test #'equal
                                            #+Clozure :weak
                                            #+Clozure :value))

(defg *hons-copy-aux-ht*   (mht :test #'eq))

(defg *init-hash-tables-done* nil)

(declaim (hash-table
          *hons-cdr-ht*
          *hons-cdr-ht-eql*
          *nil-ht* *hons-acons-ht*
          *hons-str-ht*
          *compact-print-file-ht*
          *compact-read-file-ht*))

(declaim (fixnum *hons-call-counter* *hons-misses-counter*
                 *pons-call-counter* *pons-misses-counter*))

(defg *hons-call-counter* 0)

(defg *hons-misses-counter* 0)

(defmacro hons-let (form)
  (let ((old-cdr-ht     (gensym "OLD-CDR-HT"))
        (old-cdr-ht-eql (gensym "OLD-CDR-HT-EQL"))
        (old-nil-ht     (gensym "OLD-NIL-HT")))
    `(our-lock-unlock-honsmv1

; The above lock is crippling to the parallel use of hons-let, but I
; don't yet see any alternative.  So far, use of CLEAR-HASH-TABLES
; seem inconsistent with parallel hons.

      (let ((,old-cdr-ht *hons-cdr-ht*)
            (,old-cdr-ht-eql *hons-cdr-ht-eql*)
            (,old-nil-ht *nil-ht*))
        (unwind-protect
            (progn (clear-hash-tables) ,form)
          (setq *hons-cdr-ht* ,old-cdr-ht)
          (setq *hons-cdr-ht-eql* ,old-cdr-ht-eql)
          (setq *nil-ht* ,old-nil-ht))))))

; Definition. ***** means 'Do not call this function unless you are
; sure that a superior caller in this thread has the lock on
; *HONS-STR-HT*.'

(defn1 maybe-str-hash (x)

;  *****

  (cond ((typep x '(and array string))
         (cond ((gethash x *hons-str-ht*))
               (t (setf (gethash x *hons-str-ht*) x))))
        (t x)))

(defmacro maybe-break-honsp ()
  (cond (*break-honsp*
         `(break "~&; HONSP returned nil.")
         )))

(defmacro honsp (x)

; HONSP checks whether the cons X is in our hons-related tables at
; this moment.  WASH-HONSES may make the answer returned by
; HONSP temporarily change to NIL.

  `(cond ((let* ((x ,x)
                 (ax (car x))
                 (dx (cdr x))
                 (v (cond ((null dx) *nil-ht*)
                          ((consp dx) (gethash dx *hons-cdr-ht*))
                          (t (gethash dx *hons-cdr-ht-eql*)))))
            (if (listp v)
                (let ((av (car v)))
                  (if (typep ax '(or cons symbol (and array string)))
                      (loop (if (eq ax (car av)) (return (eq x av)))
                            (setq v (cdr v))
                            (if (null v) (return nil))
                            (setq av (car v)))
                    (loop (if (eql ax (car av)) (return (eq x av)))
                          (setq v (cdr v))
                          (if (null v) (return nil))
                          (setq av (car v)))))
              (eq x (gethash (car x) v)))))
         (t (maybe-break-honsp) nil))
  ; `(ccl::%staticp ,x)
  )

(defmacro maybe-count-hons-calls ()
  (and *count-hons-calls*
       '(safe-incf *hons-call-counter* 1 maybe-count-hons-calls)))

(defmacro maybe-count-hons-misses ()
  (and *count-hons-calls*
       '(safe-incf *hons-misses-counter* 1 maybe-count-hons-misses)))

(defun hons-report-discipline-failure-message ()
  (ofd
   "
          You may ignore this 'discipline' warning.

   There is nothing wrong except that some calls to HONS-GET may be
   executing slower than you would like or might assume.  You may do

      (setq *hons-report-discipline-failure* nil)

   in raw Lisp to avoid seeing such messages.  Why the warning?  When
   a concrete alist, say X, has been formed with calls to HONS-ACONS,
   and is currently secretly backed by a hash table, and then some new
   concrete object, say Y, is formed by HONS-ACONSing something
   concrete on to X, the hash table of X is 'stolen' to support Y.
   HONS-GET on X will then be possibly much slower, calling
   ASSOC-EQUAL instead of a GETHASH.

   "))

(defmacro maybe-report-discipline-failure (fn args)
  (cond
   (*hons-report-discipline-failure*
    `(cond
      ((eq *hons-report-discipline-failure* 't)
       (ofd "~&; Warning: ~s discipline failure on args:~% ~s~%"
            ',fn ,args)
       (hons-report-discipline-failure-message))
      ((eq *hons-report-discipline-failure* 'break)
       (hons-report-discipline-failure-message)
       (break "~&; Break: ~s discipline failure on args:~% ~s~%"
              ',fn ,args)
       nil)))))

; HONS INVARIANTS

; If A and B are consp+honsp, then (eq A B) iff (equal A B).  The car
; of a consp+honsp is an atom or a consp+honsp.  The cdr of a
; consp+honsp is an atom or a consp+honsp.  No consp+honsp is
; circular.  If a string occurs in any consp+honsp, then no other
; EQUAL string occurs in any consp+honsp.

; Here are some basic data structures for honsing and memoizing.  Some
; of these are significantly expanded in size later by hons-init, but
; there is no reason to clog up saved images with large empty versions
; of them.



; HONS FUNCTIONS

(defn1 assoc-no-error-at-end (x l)

; We assume that every element of L is CONSP.

  (if (typep x '(or cons symbol (and array string)))
      (loop (if (consp l)
                (let ((al (car l)))
                  (if (eq x (car al))
                      (return al)
                    (setq l (cdr l))))
              (return nil)))
    (loop (if (consp l)
              (let ((al (car l)))
                (if (eql x (car al))
                    (return al)
                  (setq l (cdr l))))
            (return nil)))))

(defn1 too-long (x n)
  (declare (fixnum n))

; (TOO-LONG x n) == (> (LENGTH x) n) provided x a noncircular list and
; n is a nonnegative fixnum.  TOO-LONG is perhaps faster than LENGTH
; because (a) LENGTH has to worry about its argument being a circular
; list, (b) LENGTH may worry about the answer exceeding
; MOST-POSITIVE-FIXNUM, and (c) LENGTH must consider the possibility
; that its argument is a vector.
  
  (loop (cond ((atom x) (return nil))
              ((eql n 0) (return t))
              (t (setq x (cdr x))
                 (setq n (the fixnum (1- n)))))))

; Definition of NORMED.  An ACL2 Common Lisp object x is normed iff
; both (a) if x is a string, then x is the value of (GETHASH x
; *HONS-STR-HT*) and (b) if x is a CONSP, then it is also a HONSP.
; HONS-COPY and HONS produce normed objects.

; A non-HONSP cons, say c, whose car and cdr are normed is made HONSP
; by putting it in the right place in the hons-related hash tables,
; but only after we check that c can be legitimately placed there,
; i.e., no other cons is already doing the job.

(defmacro hcons (x y)
  ; `(ccl::static-cons ,x ,y)
  `(cons ,x ,y))

(defn1 hons-normed (x y)

;  *****

  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (let ((nhons (hcons x y)))
               (cond
                (yp
                 (cond ((too-long yval *start-car-ht-size*)
                        (let ((tab (mht :weak :value)))
                          (loop for pair in yval do
                                (setf (gethash (car pair) tab)
                                      pair))
                          (setf (gethash (car nhons) tab) nhons)
                          (setf (gethash y yt) tab)
                          nhons))
                       (t (setf (gethash y yt) (hcons nhons yval))
                          nhons)))
                (t (setf (gethash x yval) nhons))))))))

(defn1 hons-normed-top (x y)
  (our-lock-unlock-hons1 (hons-normed x y)))
   
; HONS-COPY

; In general, HONS-COPY has no justification for turning a cons into a
; HONSP, much less smashing is argument with RPLACA or RPLACD.
; However, during CLEAR-HASH-TABLES and WASH-HONSES, in the re-hons
; phase, such reuse is precisely what is needed and permitted.  If
; some day we learn that no cons in, say, ACL2's large object for the
; world, (w state), will ever be RPLACAed or RPLACDed and that
; REPLACAing any cons in it with an EQUAL CAR value is ok, and same
; for RPLACD/CDR, then we may legitimately absorb the conses in (w
; state) as honses via HONS-COPY1-CONSUME.

; HONS-COPY is partially and temporarily self-memoizing.
       
(defn1 hons-normed-with-suggestion (x y nhons)

; *****

  ; (unless (ccl::%staticp nhons)
  ; (defg nhons? nhons)
  ; (ofe "hons-normed-with: ~a" nhons))
  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (cond (yp
                    (cond ((too-long yval *start-car-ht-size*)
                           (let ((tab (mht :weak :value
                                       )))
                             (loop for pair in yval do
                                   (setf (gethash (car pair) tab)
                                         pair))
                             (setf (gethash (car nhons) tab) nhons)
                             (setf (gethash y yt) tab)
                             nhons))
                          (t (setf (gethash y yt) (hcons nhons yval))
                             nhons)))
                   (t (setf (gethash x yval) nhons)))))))

(defn1 hons-copy2-consume (x)

; *****

  (let ((a (hons-copy3-consume (car x)))
        (d (hons-copy3-consume (cdr x))))
    (or (eql a (car x)) (rplaca x a))
    (or (eql d (cdr x)) (rplacd x d))
    (hons-normed-with-suggestion a d x)))

(defn1 hons-copy3-consume (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (let ((ans (hons-copy2-consume x)))
             (cond ((not (eq x ans))
                    (setf (gethash x *hons-copy-aux-ht*) ans)))
             ans))))

(defn1 hons-copy1-consume (x)

; *****

    (cond ((atom x) (maybe-str-hash x))
          ((honsp x) x)
          (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               (clrhash *hons-copy-aux-ht*))
             (let ((ans (hons-copy2-consume x)))
               (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
                 (cond ((> (hash-table-size *hons-copy-aux-ht*)
                           10000)
                        (setq *hons-copy-aux-ht* (mht :test #'eq)))
                       (t (clrhash *hons-copy-aux-ht*))))
               ans))))

(defn hons-copy1-consume-top (x)

; This function should only be called when we are sure that no other
; threads are running, or when we are sure that no other threads have
; ever had access to the conses we are consuming.

  (our-lock-unlock-hons1 (hons-copy1-consume x)))
    
(defn1 hons-copy2 (x)

; *****

  (let ((a (hons-copy3 (car x)))
        (d (hons-copy3 (cdr x))))
    (hons-normed a d)))

(defn hons-copy3 (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (setf (gethash x *hons-copy-aux-ht*)
                 (hons-copy2 x)))))

(defn hons-copy1 (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
             (clrhash *hons-copy-aux-ht*))
           (let ((ans (hons-copy2 x)))
             (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               (cond ((> (hash-table-size *hons-copy-aux-ht*)
                         10000)
                      (setq *hons-copy-aux-ht* (mht :test #'eq)))
                     (t (clrhash *hons-copy-aux-ht*))))
             ans))))

(defn hons-copy1-top (x)
  (our-lock-unlock-hons1 (hons-copy1 x)))
  
(defn1 hons-copy (x)
  (hons-copy1-top x))

(defn1 hons-copy-restore (x)

;  *****

  (cond ((atom x) x)
        ((honsp x) x)
        (t (hons-normed-with-suggestion
            (hons-copy-restore (car x))
            (hons-copy-restore (cdr x))
            x))))

(defn1 hons-when-x-is-honsp (x y)

; *****

  (our-lock-unlock-hons1 (hons-normed x (hons-copy1 y))))

(defn1 hons-when-y-is-honsp (x y)

; *****

  (our-lock-unlock-hons1 (hons-normed (hons-copy1 x) y)))

(defn1 hons-equal-rec (x y)

; Cf. hons.lisp.
  
  (cond ((eq x y))
        ((atom x)
         (cond ((consp y) nil)
               ((symbolp y) nil)
               ((typep y 'fixnum) (eql x y))
               (t (equal x y))))
        ((atom y) nil)
        ((honsp x)
         (cond ((honsp y) nil)
               (t (and (hons-equal-h1-rec (car x) (car y))
                       (hons-equal-h1-rec (cdr x) (cdr y))))))
        ((honsp y)
         (and (hons-equal-h1-rec (car y) (car x))
              (hons-equal-h1-rec (cdr y) (cdr x))))
        (t (and (hons-equal-rec (car y) (car x))
                (hons-equal-rec (cdr y) (cdr x))))))

(defn hons-equal (x y)
  (hons-equal-rec x y))

; HONS-EQUAL-H1 is like HONS-EQUAL, but with the assumption that X has
; been normed.

(defn1 hons-equal-h1-rec (x y)
  (cond ((eq x y))
        ((atom x)
         (cond ((consp y) nil)
               ((symbolp y) nil)
               ((typep y 'fixnum) (eql x y))
               (t (equal x y))))
        ((atom y) nil)
        ((honsp y) nil)
        (t (and (hons-equal-h1-rec (car x) (car y))
                (hons-equal-h1-rec (cdr x) (cdr y))))))

(defn1 hons-equal-h1 (x y)
  (hons-equal-h1-rec x y))


; HONS

(defn1 hons (x y)

;  See also hons.lisp.
    (our-lock-unlock-hons1
     (hons-normed (hons-copy1 x) (hons-copy1 y))))


; HONS-GET, HONS-ACONS and HONS-ACONS!

; HONS-ACONS and HONS-GET provide fast lookup in alists, with
; ASSOC-EQUAL semantics but with the speed of hash tables in some
; cases.  These operations permit one to reasonably efficiently work
; with extremely long alists in some cases.  Informally speaking, each
; HONS-ACONS operation steals the hash table associated with the alist
; that is being extended.  The key is always honsed before the hashing
; operation in HONS-ACONS, and HONS-ACONS!.  In order to take
; advantage of the hash table lookup speed, when coding one must be
; very conscious of which object is the most recent extension of an
; alist and use that extension exclusively.  This may require careful
; passing of the alist up and down through function calls, as with any
; single threaded object in an applicative setting.  There is no
; syntactic enforcement to force one to only use the most recent
; extension of an alist, as there is for single threaded objects.  The
; only penalty for a failure to keep track of the most recent
; extension is a loss of execution speed, not of correctness.  And
; perhaps the annoyance of some warning messages about 'discipline'.

; If we limit ourselves to alists that are recognized by ALISTP, a
; possible gotcha when using HONS-ACONS! is that it might "steal" a
; hash table without one's expecting it.  For example, if you start
; two alists with (HONS-ACONS! '1 '2 nil) and (HONS-ACONS! '1 '2 nil),
; then adding something to the first with HONS-ACONS! will "steal" the
; table associated with what you thought was the second, with the
; result that adding things to the second will result in slow access.
; One can get around this annoyance to some extent by putting
; something 'unique' at the end of the alist, e.g., the final cdr.
; Our (partial) fix for this is to permit the final NIL of an
; HONS-GET/HONS-ACONS! association list to be any symbol, effectively
; naming each association list and maybe preventing the "gotcha" just
; mentioned.

(defn1-one-output hons-get-fn-do-not-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-not-hopy nil))
  (let (h)
    (when (or (and (consp key) (not (honsp key)))
              (null (setq h (gethash l *hons-acons-ht*))))
      (return-from hons-get-fn-do-not-hopy (hons-assoc-equal key l)))
    (let ((key (hons-copy1-top key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

(defn1-one-output hons-get-fn-do-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-hopy nil))
  (let ((h (gethash l *hons-acons-ht*)))
    (when (null h)
      (maybe-report-discipline-failure
       'hons-get-fn-do-hopy (list key l)))
    (let ((key (hons-copy1-top key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

; Why do we want both HONS-ACONS and HONS-ACONS!, which is the HONSing
; version of HONS-ACONS?  On the one hand, since it is quite possible
; that one will not want to look further into the alist that is
; formed, given fast hashing lookup, one may not wish the overhead of
; HONSing it.  On the other hand, if the alist is going to be an
; argument to a function that is to be memoized, then the memoization
; process may hons it -- possibly over and over and over, which can be
; very time consuming if it is very long.

(defn1 hons-acons (key value l)
  (setq key (hons-copy1-top key))
  (our-lock-unlock-hons1
   (let ((ans (cons (cons key value) l)))
     (cond ((atom l)
            (setf (gethash ans *hons-acons-ht*) 0))
           (t (let ((tab (gethash l *hons-acons-ht*)))
                (remhash l *hons-acons-ht*)
                (cond
                 ((typep tab 'fixnum)
                  (cond ((< (the fixnum tab)
                            *hons-acons-ht-threshold*)
                         (setf (gethash ans *hons-acons-ht*)
                               (the fixnum (+ 1 tab))))
                        (t (let ((tab (mht :test #'eq  ;
                                           :weak :key
                                           )))
; if you change this, see also fast-alist-len.
                             (loop for tail on ans
                                   unless (gethash (caar tail) tab)
                                   do (setf (gethash (caar tail) tab)
                                            (car tail)))
                             (setf (gethash ans *hons-acons-ht*)
                                   tab)))))
                 (tab
                  (setf (gethash key tab) (car ans))
                  (setf (gethash ans *hons-acons-ht*) tab))
                 (t (maybe-report-discipline-failure
                     'hons-acons (list key value l)))))))
     ans)))

(defn1 hons-acons! (key value l)
  (setq key (hons-copy1-top key))
  (our-lock-unlock-hons1
   (let ((ans (hons-when-x-is-honsp
               (hons-when-x-is-honsp key value)
               l)))
     (cond
      ((atom l)
       (setf (gethash ans *hons-acons-ht*) 0))
      (t (let ((tab (gethash l *hons-acons-ht*)))
           (remhash l *hons-acons-ht*)
           (cond
            ((typep tab 'fixnum)
             (cond ((< (the fixnum tab) *hons-acons-ht-threshold*)
                    (setf (gethash ans *hons-acons-ht*)
                          (the fixnum (+ 1 tab))))
                   (t (let ((tab (mht :test #'eq :weak :key
                                      )))
                        (loop for tail on ans
                              unless (gethash (caar tail) tab)
                              do (setf (gethash (caar tail) tab)
                                       (car tail)))
                        (setf (gethash ans *hons-acons-ht*)
                              tab)))))
            (tab
             (setf (gethash key tab) (car ans))
             (setf (gethash ans *hons-acons-ht*) tab))
            (t (maybe-report-discipline-failure
                'hons-acons! (list key value l)))))))
     ans)))

(defn fast-alist-len (al)
  (our-lock-unlock-hons1 
   (cond ((atom al) 0)
            (t (let ((h (gethash al *hons-acons-ht*)))
                 (cond ((typep h 'fixnum)
                        ;; worry:  hons-acons vs. hons-acons!
                        (let ((tab (mht :test #'eq :weak :key
                                        )))
                          (loop for tail on al
                                unless (gethash (caar tail) tab)
                                do (setf (gethash (caar tail) tab)
                                         (car tail)))
                          (setf (gethash al *hons-acons-ht*) tab)
                          (hash-table-count tab)))
                       (h (hash-table-count h))
                       (t (maybe-report-discipline-failure
                           'fast-alist-len al)
                          (fast-alist-len-acc al nil))))))))

(defn number-subtrees (x)
  (let ((x (hons-copy x))
        (number-subtrees-ht (mht :test 'eq :shared nil :weak nil)))
    (labels ((number-subtrees1 (x)
              (cond ((atom x) nil)
                    (t (cond ((not (gethash x number-subtrees-ht))
                              (setf (gethash x number-subtrees-ht) t)
                              (number-subtrees1 (car x))
                              (number-subtrees1 (cdr x))))))))
      (number-subtrees1 x)
      (hash-table-count number-subtrees-ht))))
    
(defn1 hons-shrink-alist-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist-orig
                       (cdr alcdr)
                       (hons-acons (car (car alcdr))
                                   (cdr (car alcdr))
                                   ans))))))))

(defn1 hons-shrink-alist-help (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair)) ; We know (car pair) is HONS-NORMEDP
           (val (gethash key tab))
           (ans (if val ans (cons (cons key (cdr pair)) ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist-help (cdr alcdr) ans tab))))

(defn1 hons-shrink-alist!-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist!-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist!-orig
                       (cdr alcdr)
                       (hons-acons! (car (car alcdr))
                                    (cdr (car alcdr))
                                    ans))))))))

(defn1 hons-shrink-alist!-help (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed
                             (hons-when-x-is-honsp key (cdr pair))
                             ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help (cdr alcdr) ans tab))))

(defn1 hons-shrink-alist!-help-honsp-alcdr (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed pair ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help-honsp-alcdr (cdr alcdr) ans tab))))

(defn1 hons-shrink-alist (alcdr ans)

; fixed by Sol Swords

  (our-lock-unlock-hons1
      (if (atom alcdr)
          ans

; Question:  Why do we call (maybe-str-hash ans) in
; HONS-SHRINK-ALIST! but not in HONS-SHRINK-ALIST?

        (let* ((tab (gethash alcdr *hons-acons-ht*))
               (ans-size
                (if (and tab (not (integerp tab)))
                    #-Clozure (1+ (ceiling
                                   (hash-table-count tab)
                                   .7))
                    #+Clozure (1+ (hash-table-count tab))
                    nil)))
          (if (or (not ans-size) (consp ans))
              (hons-shrink-alist-orig alcdr ans)
            (let ((ans-tab (mht :test #'eq :size ans-size
                                :weak :key
                                )))
              (hons-shrink-alist-help alcdr ans ans-tab)))))))

(defn1 hons-shrink-alist! (alcdr ans)
  (our-lock-unlock-hons1
   (if (atom alcdr)
       ans
     (let* ((ans (maybe-str-hash ans))
            (tab (gethash alcdr *hons-acons-ht*))
            (ans-size
             (if (and tab (not (integerp tab)))
                 #-Clozure (1+ (ceiling
                                (hash-table-count tab)
                                .7))
                 #+Clozure (1+ (hash-table-count tab))
                 nil)))
       (if (or (not ans-size) (consp ans))
           (hons-shrink-alist!-orig alcdr ans)
         (let ((ans-tab (mht :test #'eq :size ans-size :weak :key
                             )))
           (if (honsp alcdr)
               (hons-shrink-alist!-help-honsp-alcdr alcdr ans ans-tab)
             (hons-shrink-alist!-help alcdr ans ans-tab))))))))

; GC HACKS FOR HONS

(defn1 our-gc ()
  #+Clozure
  (let ((current-gcs (ccl::full-gccount)))
    (gc$)
    (loop (when (> (ccl::full-gccount) current-gcs) (return))
          (sleep 1)
          (ofvv "Sleeping while waiting for a GC to finish.")))
  #+Closure
  (gc$))

(defn1 maybe-shrink-some-hash-tables ()
  (our-lock-unlock-hons1
   (labels
       ((needs-shrinking
         (h)
         (let ((size (hash-table-size h)))
           (declare (fixnum size))
           (and (> size (cond ((eq h *hons-cdr-ht*)
                               *hons-cdr-ht-size*)
                              ((eq h *nil-ht*) *nil-ht-size*)
                              (t *mht-default-size*)))
                (< (* 3 (hash-table-count h)) size))))
        (maybe-shrink-sub
         (h)
         (cond ((consp h)
                (let ((nh (mht :size (length h) :weak :value
                               )))
                  (loop for x in h
                        do (setf (gethash (car x) nh) x))
                  nh))
               ((needs-shrinking h)
                (let ((nh (mht :size (* 3 (hash-table-count h))
                               :weak :value
                               )))
                  (maphash (lambda (k v) (setf (gethash k nh) v))
                           h)
                  nh))
               (t h)))

; We sometimes replace lists with subsidiary hash tables even though
; short lists are faster to search.  By converting to hash tables, we
; permit the garbage collection of honses that are referenced by no
; one else, thanks to idea of 'weak' in CCL.  We sometimes convert
; subsidiary hash tables back to lists when their counts are low
; enough because short lists are faster to search.

; Possible improvement: In maybe-shrink-..., it might be a better idea
; to move all the honses on lists to a single weak value hash table,
; instead of to many small hash tables, then, do the gc, and then
; rehash all those values (back to lists).  One downside to putting
; all those honses into a hash table is that they would all have to be
; rehashed to be placed back into the correct sub lists; if they are
; kept in separate small hash-tables, one merely needs to maphash and
; create a list.

        (maybe-shrink-main
         (h)
         (maphash (lambda (k v)
                    (cond ((and (not (listp v))
                                (eql 0 (hash-table-count v)))
                           (remhash k h))
                          (t (let ((nv (maybe-shrink-sub v)))
                               (or (eq nv v)
                                   (setf (gethash k h) nv))))))
                  h)
         (cond ((needs-shrinking h)
                (let ((nh (mht :test (hash-table-test h)
                               :size (* 3 (hash-table-count h))
                               #+Clozure :weak
                               #+Clozure (ccl::hash-table-weak-p h)
                               )))
                  (maphash (lambda (k v) (setf (gethash k nh) v)) h)
                  nh))
               (t h)))
        (ht-list
         (h)
         (maphash
          (lambda (k v)
            (if (not (listp v))
                (let ((c (hash-table-count v)))
                  (declare (fixnum c))
                  (cond ((eql 0 c) (remhash k h))
                        ((< c *start-car-ht-size*)
                         (setf (gethash k h)
                               (let (l)
                                 (maphash (lambda (k v)
                                            (declare (ignore k))
                                            (push v l))
                                          v)
                                 l)))))))
          h)))
     (setq *hons-cdr-ht*     (maybe-shrink-main *hons-cdr-ht*))
     (setq *hons-cdr-ht-eql* (maybe-shrink-main *hons-cdr-ht-eql*))
     (setq *nil-ht*     (maybe-shrink-sub  *nil-ht*))
     (our-gc)
     (ht-list *hons-cdr-ht*)
     (ht-list *hons-cdr-ht-eql*)
     (our-gc))))

; HONSP-CHECK

(defn1 honsp-check (x)
  (cond
   ((consp x)
    #+hons
    (unless (honsp x)
      (er hard 'honsp-check
          "The consp ~X01 is not, contrary to expectation, a honsp."
          x
          (list nil 3 4 nil)))
    t)
   ((typep x '(and array string))
    #+hons
    (unless (eq x (gethash x *hons-str-ht*))
      (er hard 'honsp-check
          "The string ~X01, contrary to expectation, is not a string ~
           hashed for the HONS implementation."
          x (list nil 3 4 nil)))
    t)
   (t nil)))




;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;;
;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;;

;  MEMOIZE VARIABLES, MACROS, AND DATA STRUCTURES

(defg *never-profile-hash-table*
  (let ((h (make-hash-table :test 'eq)))
    (loop for x in
          '(#+rdtsc ccl::rdtsc
            * 
            + 
            - 
            < 
            <= 
            = 
            > 
            >= 
            abort 
            adjoin 
            adjust-array 
            allocate-instance 
            append 
            apply 
            apropos 
            apropos-list 
            aref 
            arrayp 
            assoc 
            assoc-if 
            assoc-if-not 
            atan 
            atom 
            bit 
            bit-and 
            bit-andc1 
            bit-andc2 
            bit-eqv 
            bit-ior 
            bit-nand 
            bit-nor 
            bit-not 
            bit-orc1 
            bit-orc2 
            bit-xor 
            break 
            butlast 
            car 
            cdr 
            ceiling 
            cerror 
            change-class 
            char-equal 
            char-greaterp 
            char-lessp 
            char-not-equal 
            char-not-greaterp 
            char-not-lessp 
            char/= 
            char< 
            char<= 
            char= 
            char> 
            char>= 
            clear-hash-tables 
            clear-input 
            clear-memoize-tables 
            clear-output 
            compile 
            compile-file 
            compile-file-pathname 
            compiler-macro-function 
            complex 
            compute-restarts 
            concatenate 
            continue 
            copy-pprint-dispatch 
            copy-readtable 
            copy-symbol 
            count 
            count-if 
            count-if-not 
            decode-universal-time 
            delete 
            delete-duplicates 
            delete-if 
            delete-if-not 
            describe 
            digit-char 
            digit-char-p 
            directory 
            dribble 
            ed 
            encode-universal-time 
            enough-namestring 
            ensure-directories-exist 
            ensure-generic-function 
            eq 
            eql 
            error 
            eval 
            every 
            export 
            fboundp 
            fceiling 
            ffloor 
            file-position 
            fill 
            find 
            find-class 
            find-if 
            find-if-not 
            find-method 
            find-restart 
            find-symbol 
            finish-output 
            fixnum-to-symbol 
            float 
            float-sign 
            floor 
            force-output 
            format 
            fresh-line 
            fround 
            ftruncate 
            funcall 
            gensym 
            gentemp 
            get 
            get-dispatch-macro-character 
            get-internal-real-time 
            get-internal-run-time 
            get-macro-character 
            get-properties 
            get-setf-expansion 
            getf 
            gethash 
            if 
            import 
            initialize-instance 
            intern 
            internal-real-time 
            intersection 
            invalid-method-error 
            invoke-restart 
            last 
            ld-fn 
            len 
            len1 
            length 
            lisp-implementation-type 
            list 
            list* 
            listen 
            load 
            log 
            macro-function 
            macroexpand 
            macroexpand-1 
            make-array 
            make-broadcast-stream 
            make-concatenated-stream 
            make-condition 
            make-dispatch-macro-character 
            make-hash-table 
            make-instance 
            make-list 
            make-load-form 
            make-load-form-saving-slots 
            make-package 
            make-pathname 
            make-random-state 
            make-sequence 
            make-string 
            make-string-input-stream 
            make-string-output-stream 
            map 
            map-into 
            mapc 
            mapcan 
            mapcar 
            mapcon 
            mapl 
            maplist 
            max 
            member 
            member-if 
            member-if-not 
            memoize-call-array-grow 
            memoize-eval-compile 
            memoize-fn 
            merge 
            merge-pathnames 
            method-combination-error 
            mf-1st-warnings 
            mf-2nd-warnings 
            mf-warnings 
            mismatch 
            muffle-warning 
            nbutlast 
            nconc 
            nintersection 
            no-applicable-method 
            no-next-method 
            not 
            notany 
            notevery 
            nset-difference 
            nset-exclusive-or 
            nstring-capitalize 
            nstring-downcase 
            nstring-upcase 
            nsublis 
            nsubst 
            nsubst-if 
            nsubst-if-not 
            nsubstitute 
            nsubstitute-if 
            nsubstitute-if-not 
            null 
            nunion 
            open 
            pairlis 
            parse-integer 
            parse-namestring 
            pathname-device 
            pathname-directory 
            pathname-host 
            pathname-name 
            pathname-type 
            peek-char 
            position 
            position-if 
            position-if-not 
            pprint 
            pprint-dispatch 
            pprint-fill 
            pprint-indent 
            pprint-linear 
            pprint-newline 
            pprint-tab 
            pprint-tabular 
            prin1 
            princ 
            princ-to-string 
            print 
            print-object 
            profile 
            profile-acl2 
            profile-all 
            random 
            rassoc 
            rassoc-if 
            rassoc-if-not 
            read 
            read-byte 
            read-char 
            read-char-no-hang 
            read-delimited-list 
            read-from-string 
            read-line 
            read-preserving-whitespace 
            read-sequence 
            reduce 
            reinitialize-instance 
            remove 
            remove-duplicates 
            remove-if 
            remove-if-not 
            rename-file 
            rename-package 
            replace 
            require 
            room 
            round 
            sbit 
            search 
            set-difference 
            set-dispatch-macro-character 
            set-exclusive-or 
            set-macro-character 
            set-pprint-dispatch 
            set-syntax-from-char 
            shadow 
            shadowing-import 
            shared-initialize 
            signal 
            signum 
            slot-missing 
            some 
            sort 
            stable-sort 
            store-value 
            string-capitalize 
            string-downcase 
            string-equal 
            string-greaterp 
            string-lessp 
            string-not-equal 
            string-not-greaterp 
            string-not-lessp 
            string-upcase 
            string/= 
            string< 
            string<= 
            string= 
            string> 
            string>= 
            stringp 
            sublis 
            subseq 
            subsetp 
            subst 
            subst-if 
            subst-if-not 
            substitute 
            substitute-if 
            substitute-if-not 
            subtypep 
            svref 
            symbol-to-fixnum 
            symbol-to-fixnum-create 
            symbolp 
            sync-memoize-call-array 
            sync-watch-array 
            terpri 
            translate-logical-pathname 
            translate-pathname 
            tree-equal 
            true-listp 
            truncate 
            typep 
            unexport 
            unintern 
            union 
            unread-char 
            unuse-package 
            update-instance-for-different-class 
            update-instance-for-redefined-class 
            upgraded-array-element-type 
            upgraded-complex-part-type 
            use-package 
            use-value 
            user-homedir-pathname 
            values 
            vector-push-extend 
            w 
            warn 
            watch-array-grow 
            wild-pathname-p 
            write 
            write-byte 
            write-char 
            write-line 
            write-sequence 
            write-string 
            write-to-string 
            y-or-n-p 
            yes-or-no-p)
          do (setf (gethash x h) t))
    h))

;  recording vars

; To minimize metering overhead costs, one may set these "*RECORD-"
; variables to NIL before memoizing.

; *RECORD-BYTES* and other *RECORD-...* variables are bound in
; REMEMOIZE-ALL, so we use DEFPARAMETER rather than DEFG.

(defparameter *record-bytes*
  #+Clozure
  (> most-positive-fixnum (expt 2 32))
  #-Clozure
  nil
  "If *RECORD-BYTES* when a function is memoized, we keep track of
  heap bytes allocated during calls of that function.")

(defparameter *record-calls* t
  "If *RECORD-CALLS* when a function is memoized,
  we count all calls of the function.")

(defparameter *record-hits* t
  "If *RECORD-HITS* when a function is memoized, we count
  the number of times that a previously computed answer
  is used again.")

(defparameter *record-hons-calls* t
  "If *RECORD-HONS-CALLS* when a function is memoized,
  hons calls are counted.")

(defparameter *record-mht-calls* t
  "If *REPORT-HONS-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that a memo hash-table for the function was created.  This may
  be of interest to those who memoize functions that deal in changing
  stobjs; the memoization machinery sometimes 'forgets' an entire
  memoization hash table out of an abundance of caution, and then may
  later need to create it afresh.")

(defparameter *record-pons-calls* t
  "If *RECORD-PONS-CALLS* when a function is memoized,
  pons calls are counted.")

(defparameter *record-time* t
  "If *RECORD-TIME* when a function is memoized, we
  record the elapsed time for each outermost call of the function.")


;  reporting vars

(defg *report-bytes* #+Clozure t #-Clozure nil
  "If *REPORT-BYTES*, then MEMOIZE-SUMMARY prints the
  number of bytes allocated on the heap.")

(defg *report-calls* t
  "If *REPORT-CALLS*, MEMOIZE-SUMMARY prints the number of calls.")

(defg *report-calls-from* t
  "If *REPORT-CALLS-FROM*, MEMOIZE-SUMMARY prints which functions
  called a function, how many times, and and how long the calls
  took.")

(defg *report-hits* t
  "If *REPORT-HITS*, MEMOIZE-SUMMARY prints the number of times
  that a previously computed answer was reused.")

(defg *report-hons-calls* t
  "If *REPORT-HONS-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that hons was called.")

(defg *report-mht-calls* t
  "If *REPORT-MHT-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that a memo hash-table for the function was created.  This may
  be of interest to those who memoize functions that deal in changing
  stobjs; the memoization machinery sometimes 'forgets' an entire
  memoization hash table out of an abundance of caution, and then may
  later need to create it afresh.")

(defg *report-pons-calls* t
  "If *REPORT-PONS-CALLS*, MEMOIZE-SUMMARY prints the number of calls
  of PONS.")

(defg *report-time* t
  "If *REPORT-TIME*, MEMOIZE-SUMMARY prints the total time used to
   compute the outermost calls.")

(defg *report-on-memo-tables* t
  "If *REPORT-ON-MEMO-TABLES*, MEMOIZE-SUMMARY prints
  information about memo tables.")

(defg *report-on-pons-tables* nil
  "If *REPORT-ON-PONS-TABLES*, MEMOIZE-SUMMARY prints
  information about pons tables.")

; counters

(defg *pons-call-counter* 0)

(defg *pons-misses-counter* 0)

(defmacro maybe-count-pons-calls ()
  (and *count-pons-calls*
       '(safe-incf *pons-call-counter* 1 maybe-count-pons-calls)))

(defmacro maybe-count-pons-misses ()
  (and *count-pons-calls*
       '(safe-incf *pons-misses-counter* 1 maybe-count-pons-misses)))

; array and hash-tables

(defg *memoize-info-ht* (mht))

(defg *memoize-call-array*
  (make-array 1 :element-type 'fixnum :initial-element 0)
  
  "*MEMOIZE-CALL-ARRAY*, 'ma' for short, is used for storage of the
  monitoring information for memoized functions.  ma has as its length
  4 times the square of the maximum number of memoized functions.

  ma is initialized in MEMOIZE-INIT.  Think of ma as a two dimensional
  array with dimensions (twice the max number of memoized functions) x
  (twice the max number of memoized functions).  Each 'column'
  corresponds to info about a memoized function, but the first five
  columns are 'special'.  We count rows and columns starting at 0.
  Column 0 is used as scratch space by COMPUTE-CALLS-AND-TIMES for
  sums across all functions.  Columns 1, 2, and 3 are not currently
  used at all.  Column 4 is for the anonymous 'outside caller'.
  Column 5 is for the first memoized function.  In columns 5 and
  greater, row 0 is used to count 'bytes', 1 'hits', 2 MHT calls, 3
  HONS calls, and 4 PONS calls.

  The elements of an ma column starting at row 10 are for counting and
  timing info.  Suppose column 7 corresponds to the memoized function
  FOO and column 12 corresponds to the memoized function BAR.
  Whenever FOO calls BAR, element 2*12 of column 7 will be incremented
  by 1, and the total elapsed time for the call will be added to
  element 2*12+1 of column 7.

  Though ma may 'grow', it may not grow while any memoized function is
  running, and here is why: every memoized function has a cached
  opinion about the size of ma.  To avoid an abort during a call of
  MEMOIZE one may call (MEMOIZE-HERE-COME n) to assure that ma has
  room for at least n more memoized functions.")

(defg *compute-array* (make-array 0)
  
  "*COMPUTE-ARRAY*, ca for short, is an array of proper lists.  At the
  end of a call of COMPUTE-CALLS-AND-TIMES, which is called by
  MEMOIZE-SUMMARY, (aref ca n) will contain the numbers of the
  functions that have called the function numbered n.")

(declaim (type (simple-array t (*)) *compute-array*))

#+cmu
(declaim (type (simple-array fixnum (*)) *memoize-call-array*))
#-cmu
(eval-when
 #-cltl2
 (load eval)
 #+cltl2
 (:load-toplevel :execute)
 (proclaim `(type (simple-array fixnum (*)) *memoize-call-array*)))

(defg *initial-max-memoize-fns* 500)

(defg *2max-memoize-fns* (* 2 *initial-max-memoize-fns*))

(defconstant *ma-bytes-index*       0)

(defconstant *ma-hits-index*        1)

(defconstant *ma-mht-index*         2)

(defconstant *ma-hons-index*        3)

(defconstant *ma-pons-index*        4)

(defconstant *ma-initial-max-symbol-to-fixnum* 4)

(defg *max-symbol-to-fixnum* *ma-initial-max-symbol-to-fixnum*)

(declaim (fixnum *max-symbol-to-fixnum*
                 *initial-2max-memoize-fns*
                 *ma-initial-max-symbol-to-fixnum*
                 *2max-memoize-fns*))

; for debugging

(defg *memoize-hack-condition* nil)

(defg *memoize-hack-inline* nil)

(defg *memoize-hack-trace* nil)


; for initialization

(defg *memoize-init-done* nil)


; locals used in memoize-on and memoize-off

(defg *mo-f* (make-symbol "F"))

(defg *mo-h* (make-symbol "H"))

(defg *mo-o* (make-symbol "O"))


; locals used in functions generated by memoize-fn

(defg *mf-old-caller* (make-symbol "OLD-CALLER"))

(defg *mf-start-hons* (make-symbol "START-HONS"))

(defg *mf-start-pons* (make-symbol "START-PONS"))

(defg *mf-start-bytes* (make-symbol "START-BYTES"))

(defg *mf-ans* (make-symbol "ANS"))

(defg *mf-ans-p* (make-symbol "ANS-P"))

(defg *mf-ma* (make-symbol "MA"))

(defg *mf-args* (make-symbol "ARGS"))

(defg *mf-2mmf* (make-symbol "MF-2MMF"))

(defg *mf-2mmf-fnn* (make-symbol "MF-2MMF-FNN"))

(defg *mf-count-loc* (make-symbol "MF-COUNT-LOC"))

(defg *mf-cl-error-msg*
  "~%; Redefining a function in the COMMON-LISP package ~
   is forbidden.")

(defg *caller* (* *ma-initial-max-symbol-to-fixnum*
                  *2max-memoize-fns*)
  "When memoized functions are executing in parallel, the value of
  *CALLER* and of statistics derived therefrom may be meaningless and
  random.")

(declaim (fixnum *caller*))

; The :CONDITION parameter of MEMOIZE-FN can either be T, or a
; function symbol defined by the user within the ACL2 loop, or a LISTP
; (CONSP or NIL).  In the last case we think of the condition as an
; expression in the formals of FN.  If the :INLINE parameter T, then
; the body of FN is placed inline in the memoized definition;
; otherwise, a funcall of the original function is placed there.

(defg *profile-reject-hash-table* (mht :test 'eq)

  "The user may freely add to the hash table
  *PROFILE-REJECT-HASH-TABLE*, which inhibits the collection of
  functions into lists of functions to be memoized and/or profiled.

  Here are some reasons for adding a function fn to
  *PROFILE-REJECT-HASH-TABLE*.

  1. A call of fn is normally so fast or fn is called so often that
  the extra instructions executed when a memoized version of fn is run
  will distort measurements excessively.  We tend not to profile any
  function that runs in under 6000 clock ticks or about 2
  microseconds.  The number of extra instructions seem to range
  between 20 and 100, depending upon what is being measured.
  Measuring only calls is relatively fast.  If one measures elapsed
  time, one might as well measure everything else too.  Or so it seems
  in 2007 on terlingua.csres.utexas.edu.

  2. fn is a subroutine of another function being profiled, and we
  wish to reduce the distortion that profiling fn will cause.

  3. fn is 'transparent', like EVAL.  Is EVAL fast or slow?  The
  answer, of course, is that it mostly depends upon what one is
  EVALing.

  4. fn's name ends in '1', meaning 'auxiliary' to some folks.

  5. fn is boring.")

(defn1 dubious-to-profile (fn)
  (cond ((not (symbolp fn)) "not a symbol.")
        ((not (fboundp fn)) "not fboundp.")
        ((eq (symbol-package fn) *main-lisp-package*)
         "in *main-lisp-package*.")
        ((member fn (eval '(trace))) "a member of (untrace).")
        ((member fn (eval '(old-trace)))
         "a member of (old-untrace).")
        ((gethash fn *never-profile-hash-table*)
         "a member of *never-profile-hash-table*.")
        ((gethash fn *profile-reject-hash-table*)
         (ofn "in *profile-reject-hash-table*.~%~
               ; Override with (remhash '~a ~
               *profile-reject-hash-table*).~%;" fn))
        ((macro-function fn) "a macro.")
        ((special-form-or-op-p fn) "a special operator.")
        ((memoizedp-raw fn) "currently memoized or profiled.")
        ((null (number-of-arguments fn))
         "of an indeterminate number of arguments.  Cf. profile.")
        ((null (number-of-return-values fn))
         "of an indeterminate number of return values.  Cf. profile."
         )))

; memoize-flush 'forgets' all that was remembered for certain
; functions that use certain stobjs.  We must keep memoize-flush very
; fast in execution so as not to slow down stobj update or resize
; operations in general.  We 'forget' the pons table later.

(defmacro memoize-flush (st)
  (let ((s (st-lst st)))
    `(loop for sym in ,s do
           (let ((old (symbol-value (the symbol sym))))
             (unless (or (null old) (empty-ht-p old))
               (setf (symbol-value (the symbol sym)) nil))))))

(declaim (hash-table *memoize-info-ht*))

(defmacro pist (table &rest x)
  (cond ((atom x) nil)
        (t (list 'pons (car x)
                 (cons 'pist (cdr x)) table))))

(defmacro pist* (table &rest x)
  (cond ((atom x) x)
        ((atom (cdr x)) (car x))
        (t (list 'pons (car x)
                 (cons 'pist* (cons table (cdr x))) table))))

;  THE MEMO-INFO-HT-ENTRY DATA STRUCTURE

; *MEMOIZE-INFO-HT* maps each currently memoized function symbol, fn,
; to a DEFREC record of type MEMO-INFO-HT-ENTRY with 22 fields.

; fn             a symbol, the name of the function being memoized
; tablename      a symbol whose value is the memoize table for fn
; ponstablename  a symbol whose value is the pons table for fn
; old-fn         the old value of (symbol-function fn), or nil.
; memoized-fn    the new value of (symbol-function fn)
; condition      T or NIL. :condition arg as passed to memoize-fn
; inline         T or NIL. :inline arg as passed to memoize-fn
; num            an integer, unique to fn
; sts            the stobj memotable lists for fn
; trace          T or NIL. :trace arg as passed to memoize-fn
; start-time     a symbol whose val is the start time of the current,
;                   outermost call of fn, or -1 if no call of fn
;                   is in progress.
; cl-defun       the function body actually used, in the inline=t
;                case, as supplied (or as computed, if not supplied)
; formals        as supplied (or as computed, if not supplied)
; specials       never to be used or explained
; stobjs-in      as supplied (or as computed, if not supplied)
; stobjs-out     as supplied (or as computed, if not supplied)
; record-bytes   value as bound at the time MEMOIZE-FN is called
; record-calls            ''
; record-hits             ''
; record-hons-calls       ''
; record-mht-calls        ''
; record-pons-calls       ''
; record-time             ''
; memo-table-init-size integer, default *mht-default-size*

; *memoize-info-ht* also maps num back to the corresponding symbol.

(defrec memoize-info-ht-entry
  (start-time  ; vaguely ordered by most frequently referenced first
   num
   tablename
   ponstablename
   condition
   inline
   memoized-fn
   old-fn
   fn
   sts
   trace
   cl-defun
   formals
   specials
   stobjs-in
   stobjs-out
   record-bytes
   record-calls
   record-hits
   record-hons-calls
   record-mht-calls
   record-pons-calls
   record-time
   memo-table-init-size
   )
  t)

; MEMOIZE FUNCTIONS

#+Clozure
(defmacro heap-bytes-allocated ()
  '(the fixnum (ccl::%heap-bytes-allocated)))

(defn sync-memoize-call-array ()

  ; To be called only by MEMOIZE-INIT or MEMOIZE-CALL-ARRAY-GROW.

  (let ((n1 (the fixnum (* *2max-memoize-fns* *2max-memoize-fns*)))
        (n2 (1+ *max-symbol-to-fixnum*)))
    (declare (fixnum n1 n2))
    (unless (eql n1 (length *memoize-call-array*))
      (unless (eql 1 (length *memoize-call-array*))
        (setq *memoize-call-array*
              (make-array 1 :element-type 'fixnum
                          :initial-element 0))
        (gc$))
      (setq *memoize-call-array*
            (make-array n1
                        :element-type 'fixnum
                        :initial-element 0)))
    (unless (eql n2 (length *compute-array*))
      (setq *compute-array*
            (make-array n2 :initial-element nil)))
    (setq *caller* (* *ma-initial-max-symbol-to-fixnum*
                      *2max-memoize-fns*))))

(defn1 memoize-call-array-grow
  (&optional (2nmax (* 2 (ceiling (* 3/2 (/ *2max-memoize-fns* 2))))))
  (unwind-mch-lock
   (unless (integerp 2nmax)
     (ofe "(memoize-call-array-grow ~s).  Arg must be an integer."
          2nmax))
   (unless (evenp 2nmax)
     (ofe "(memoize-call-array-grow ~s).  Arg must be even." 2nmax))
   (unless (> 2nmax 100)
     (ofe "(memoize-call-array-grow ~s).  Arg must be > 100." 2nmax))
   (when (<= 2nmax *2max-memoize-fns*)
     (ofv "memoize-call-array-grow: *memoize-call-array* already ~
           big enough.")
     (return-from memoize-call-array-grow))
   (unless (<= (* 2nmax 2nmax) most-positive-fixnum)
     (ofe "memoize-call-array-grow:  most-positive-fixnum~%~
            exceeded.  Too many memoized functions."))
   (unless (<= (* 2nmax 2nmax) array-total-size-limit)
     (ofe "memoize-call-array-grow: ARRAY-TOTAL-SIZE-LIMIT ~%~
            exceeded.  Too many memoized functions."))
   (unless (eql *caller* (* *ma-initial-max-symbol-to-fixnum*
                            *2max-memoize-fns*))
     (ofv "MEMOIZE-CALL-ARRAY-GROW was called while a ~
           memoized-function~% was executing, so call reports may ~
           be quite inaccurate."))

   (setq *memoize-call-array*
     (make-array 1 :element-type 'fixnum :initial-element 0))
   (setq *2max-memoize-fns* 2nmax)
   (sync-memoize-call-array)
   (rememoize-all)))

(defn1 symbol-to-fixnum-create (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (let (new)
        (loop for i from
              (if (eql *caller*
                       (* *ma-initial-max-symbol-to-fixnum*
                          *2max-memoize-fns*))
                  (1+ *ma-initial-max-symbol-to-fixnum*)
                (1+ *max-symbol-to-fixnum*))
              below (the fixnum (floor *2max-memoize-fns* 2))
              do (unless (gethash i *memoize-info-ht*)
                   (setq new i)
                   (return)))
        (cond (new
               (setq *max-symbol-to-fixnum*
                     (max *max-symbol-to-fixnum* new))
               new)
              (t (memoize-call-array-grow)
                 (safe-incf *max-symbol-to-fixnum*
                            1 symbol-to-fixnum-create)
                 *max-symbol-to-fixnum*))))))

(defn1 symbol-to-fixnum (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (ofe "(symbol-to-fixnum ~s).  Illegal symbol."
           s))))

(defn1 fixnum-to-symbol (n)
  (check-type n fixnum)
  (or (gethash n *memoize-info-ht*)
      (ofe "(fixnum-to-symbol ~s). Illegal number."
           n)))

(defn1 coerce-index (x)
  (if (and (typep x 'fixnum)
           (>= x 0)
           (< x (length *memoize-call-array*)))
      x
    (symbol-to-fixnum x)))

; This code has the 'feature' that if the condition causes an error,
; so will the memoized function.

; PONS differs from HONS in that it does not honsify its arguments and
; in that it takes a hash table as a third argument.  We use PONS in
; memoization.

; We use PONS instead of HONS in memoization because we could not
; afford to honsify (using hons-shrink-alist!) certain alists in
; certain biology tests.  About the same time, we (gratuitously)
; decided to stop hons'ifying the output of memoized functions.

(defn1 pons (x y ht)

; ***** pons *****

; If pons can create a hons, that will lead to a deadlock over locks!

  (let ((xval nil)
        (yval nil)
        (ans nil))
    (maybe-count-pons-calls)

; We have taken string normalization out of pons because there might
; be a chance of confusing a 'normal' string with a stobj.

; If x1, ..., xn is pointwise EQL to y1, ..., yn, then are we sure
; that (pist* x1 ... xn) is EQ to (pist* y1 ... yn)?

; If CONS exists, then return it.  Does CDR exist in hash table?

    (setq yval (gethash y ht))

; Does CAR exist in hash table?

    (cond (yval
           (cond ((not (consp yval))
                  (setq xval (gethash x yval))
                  (cond (xval (setq ans xval))))
                 ((setq ans (assoc-no-error-at-end x yval))))))
    (cond

; If PONS found, then return previous CONS from hash table.
     (ans)

; Otherwise, maybe create new CONS and install in hash table.

     (t
      (setq yval (gethash y ht))
      (cond
       ((null yval)
        (setq ans (cons x y))
        (setf (gethash y ht) (list ans))
        ans)
       ((consp yval)
        (let ((ans (assoc-no-error-at-end x yval)))
            (cond
             (ans)
             (t (let ((ans (cons (cons x y) yval)))
                  (maybe-count-pons-misses)
                  (cond
                   ((too-long ans *start-car-ht-size*)
                    (let ((tab (mht)))
                      (loop for pair in ans do
                            (setf (gethash (car pair) tab) pair))
                      (setf (gethash y ht) tab)
                      (car ans)))
                   (t (setf (gethash y ht) ans)
                      (car ans))))))))
       (t (setq xval (gethash x yval))
          (cond ((not xval)
                 (maybe-count-pons-misses)
                 (setf (gethash x yval)
                       (setq ans (cons x y))))
                (t (setq ans xval)))
          ans))))))

(defn1 memoize-eval-compile (def)
  (unless (and (consp def)
               (eq 'defun (car def))
               (consp (cdr def))
               (symbolp (cadr def)))
    (ofe "MEMOIZE-EVAL-COMPILE:  Bad input ~a." def))
  (compile (eval def))
  (cadr def))

(defn1 memoizedp-raw (fn)
  (our-lock-unlock-memoize1
   (and (symbolp fn)
        (values (gethash fn *memoize-info-ht*)))))

(defg *hons-gentemp-counter* 0)
(declaim (fixnum *hons-gentemp-counter*))
(defn1-one-output hons-gentemp (root)
  (check-type root string)
  (loop
   (safe-incf *hons-gentemp-counter* 1 hons-gentemp)
   (let ((name (ofn "HONS-G-~s,~s" root *hons-gentemp-counter*)))
     (multiple-value-bind (sym status)
         (intern name (find-package "ACL2_INVISIBLE"))
       (if (null status) (return sym))))))

(defn1 st-lst (st)

; ST-LST returns a symbol whose value is a list in which are saved the
; names of the memoize tables that will be set to nil whenever the
; stobj st is changed.

  (check-type st symbol)
  (multiple-value-bind (symbol status)
      (intern (ofn "HONS-S-~s,~s"
                   (package-name (symbol-package st))
                   (symbol-name st))
              (find-package "ACL2_INVISIBLE"))
    (or status (eval `(defg ,symbol nil)))
    symbol))

(defn1 dcls (l)
     (loop for dec in l nconc
           (let ((temp
                  (if (consp dec)
                      (loop for d in (cdr dec) nconc
                            (if (and (consp d) (eq (car d) 'ignore))
                                nil
                              (cons d nil))))))
             (if temp (cons (cons 'declare temp) nil)))))

(defn1 timer-error ()
  (ofe "timer-error."))

; PRINE  - princ eviscerated

(defg *assoc-eq-hack-ht* (mht :test 'eql))

(defg *hons-vars-alist* nil)

(defn assoc-eq-hack (x y)
  (cond ((atom y) nil)
        (t (let ((h (gethash y *assoc-eq-hack-ht*)))
             (cond (h (gethash x h))
                   (t (setq h (mht :test 'eq))
                      (setf (gethash y *assoc-eq-hack-ht*)
                            h)
                      (loop for pair in y do
                            (setf (gethash (car pair)
                                           h)
                                  pair))
                      (gethash x h)))))))

(defn abbrev (x &optional
                (level *print-level*) (length *print-length*))
  (cond ((atom x) x)
        ((eql level 0) '?)
        ((eql length 0) '?)
        (t (let ((pair (or (assoc-eq-hack x *hons-vars-alist*)
                           (assoc-eq-hack
                            x (table-alist 'evisc-table
                                           (w *the-live-state*))))))
             (cond (pair (cdr pair))
                   (t (let ((a (abbrev (car x)
                                       (and level (1- level))
                                       length))
                            (d (abbrev (cdr x)
                                       level
                                       (and length (1- length)))))
                        (cond ((and (eq a (car x))
                                    (eq d (cdr x)))
                               x)
                              ((and (eq a '?)
                                    (eq d '?))
                               '?)
                              (t (cons a d))))))))))
                                       
(defn1 prine (obj &optional stream)
  (let ((*print-pretty* nil))
    (princ (abbrev obj *print-level* *print-length*) stream)))

; MEMOIZE-FN

(defn1 mf-trace-exit (fn nrv ans)
  (oftr "~%< ~s " fn)
  (cond ((> nrv 1)
         (oftr "returned ~@r values:" nrv)
         (loop for i from 1 to nrv do
               (oft "~%~@r.~8t  " i)
               (prine (car ans) *trace-output*)))
        (t (prine ans *trace-output*)))
  (oftr ")~%"))

(defn1 maybe-reduce-memoize-tables () nil) ; user may redefine.

(defmacro defnmem (fn args body &key
                      (condition t)
                      (inline t)
                      (trace nil)
                      (specials nil))

  "DEFNMEM is a raw Lisp macro. (DEFNMEM fn args body) both DEFUNs and
   MEMOIZEs FN.  The CL-DEFUN, STOBJS-IN, FORMALS, and STOBJS-OUT
   options to MEMOIZE-FN are taken from ARGS, which may not use STATE
   or STOBJS.  FN is declared to return only one value."

  `(progn
     (setf (gethash ',fn *number-of-arguments-and-values-ht*)
           (cons ,(length args) 1))
     (declaim (ftype (function ,(tlist (length args))
                               (values t))
                     ,fn))
     (defun ,fn ,args ,body)
     (memoize-fn ',fn
                 :condition ',condition
                 :inline ',inline
                 :trace ',trace
                 :specials ',specials
                 :cl-defun '(defun ,fn ,args ,body)
                 :stobjs-in (make-list (length ',args))
                 :formals ',args
                 :stobjs-out '(nil))
     ',fn))

(defn fle (x)

  "(fle 'foo) returns the defining body of the function FOO."

  (cond ((functionp x)
         (function-lambda-expression x))
        ((symbolp x)
         (let ((fn (symbol-function x)))
           (cond ((functionp fn)
                  (function-lambda-expression fn))
                 #+Clozure
                 ((and (arrayp fn) (functionp (aref fn 1)))
                  (function-lambda-expression (aref fn 1)))
                 (t (ofe "Can't figure out ~a as a function." x)))))))

(defn1 memoize-fn (fn &key (condition t) (inline t) (trace nil)
                      (cl-defun :default)
                      (formals :default)
                      (stobjs-in :default)
                      (stobjs-out :default)
                      (specials nil)
                      (memo-table-init-size *mht-default-size*))

  "One might think our parity for the INLINE value is backwards, but
  we do not.  If :INLINE is T then when MEMOIZE-FN creates a new body
  for fn, we place the old body of fn within the new body, i.e., 'in
  line'.  However, if :INLINE in NIL, then we place code to call the
  old function for fn within the new body.  (By the way, in CCL,
  recursive calls are inlined in the Common Lisp sense by default.)"

  (unwind-mch-lock
   (maybe-untrace! fn)
   (with-warnings-suppressed
    (unless *memoize-init-done*
      (ofe "Memoize-fn:  *MEMOIZE-INIT-DONE* is still nil."))
    (unless (symbolp fn)
      (ofe "Memoize-fn: ~s is not a symbol.") fn)
    (unless (or (fboundp fn) (not (eq cl-defun :default)))
      (ofe "Memoize-fn: ~s is not fboundp." fn))
    (when (gethash fn *never-profile-hash-table*)
      (ofe "Memoize-fn: Cannot redefine ~s, which is in ~
            *never-profile-hash-table*."
           fn))
    (when (memoizedp-raw fn)
      (ofd "~%; Memoize-fn: ** Warning: ~s is currently memoized. ~
          ~%; So first we will unmemoize it, then rememoize it."
           fn)
      (unmemoize-fn fn))
    (when (member fn (eval '(trace)))
      (ofd "~%; Memoize-fn:  Untracing ~s before memoizing it." fn)
      (eval `(untrace ,fn)))
    (when (and (boundp 'old-trace)
               (member fn (eval '(old-trace))))
      (ofd "~%; Memoize-fn:  Old-untracing ~s before memoizing it."
           fn)
      (eval `(old-untrace ,fn)))
    
; TRACE, UNTRACE, OLD-TRACE, and OLD-UNTRACE are macros that get
; redefined sometimes.  So we use EVAL in calling them.

    (let*
      ((w (w *the-live-state*))
       (cl-defun (if (eq cl-defun :default)
                     (if inline
                         (cond ((cltl-def-from-name fn nil w))
                               ((and (fboundp fn)
                                     (function-lambda-expression
                                      (symbol-function fn))))
                               (t (ofe "Memoize-fn: cannot find ~
                                        the definition of ~a." fn)))
                       nil)
                   cl-defun))
       (formals
        (if (eq formals :default)
            (let ((fo (getprop fn 'formals t
                               'current-acl2-world w)))
              (if (eq fo t)
                  (if (consp cl-defun)
                      (cadr cl-defun)
                    (let ((n (number-of-arguments fn)))
                      (if n
                          (loop for i below n
                                collect (ofni "X~a" i))
                        (ofe "Memoize-fn: could not determine the ~
                              number of arguments of ~a.  ~%; ~
                              Please put a cons of two numbers in ~
                              the hash-table ~
                              *NUMBER-OF-ARGUMENTS-AND-VALUES-HT* ~
                              under ~a.  ~%; For example, do (setf ~
                              (gethash '~a ~
                              *NUMBER-OF-ARGUMENTS-AND-VALUES-HT*) ~
                              '(1 . 1))"
                             fn fn fn))))
                fo))
          formals))
       (stobjs-in (if (eq stobjs-in :default)
                      (let ((s (getprop fn 'stobjs-in t
                                        'current-acl2-world w)))
                        (if (eq s t) (make-list (len formals)) s))
                    stobjs-in))
       (stobjs-out
        (if (eq stobjs-out :default)
            (let ((s (getprop fn 'stobjs-out t
                              'current-acl2-world w)))
              (if (eq s t)
                  (let ((n (number-of-return-values fn)))
                    (if n (make-list n)
                      (ofe "Memoize-fn: cannot determine number of ~
                            return values for ~a.  ~%; Please put ~
                            a cons of two numbers in the ~
                            hash-table ~
                            *NUMBER-OF-ARGUMENTS-AND-VALUES-HT* ~
                            under ~a. ~%; For example, do (setf ~
                            (gethash '~a ~
                            *NUMBER-OF-ARGUMENTS-AND-VALUES-HT*) ~
                            '(1 . 1))"
                           fn fn fn)))
                s))
          stobjs-out)))
      (unless (and (symbol-listp formals)
                   (no-duplicatesp formals)
                   (loop for x in formals never (constantp x)))
        (ofe "Memoize-fn: Formals, ~a, must be a true list of ~
              distinct, nonconstant symbols."
             formals))
      (when (intersection lambda-list-keywords formals)
        (ofe "Memoize-fn: Formals, ~a, may not intersect ~
              lambda-list-keywords."))
      (when (and condition (or (member 'state stobjs-in)
                               (member 'state stobjs-out)))
        (ofe "Memoize-fn:  ~s uses STATE." fn))
      (let*
        ((fnn (symbol-to-fixnum-create fn))
         (2mfnn (* *2max-memoize-fns* fnn))
         (*acl2-unwind-protect-stack*
          (cons nil *acl2-unwind-protect-stack*))
         (old-fn (if (fboundp fn) (symbol-function fn)))
         (body (if (or inline (null old-fn))
                   (car (last cl-defun))
                 `(funcall (the ,(if (compiled-function-p
                                       old-fn)
                                      'compiled-function
                                    'function)
                             ,old-fn)
                           ,@formals)))
         (condition-body
          (cond ((booleanp condition) condition)
                ((symbolp condition)
                 (car (last (cltl-def-from-name condition nil w))))
                (t condition)))
         (dcls (dcls (cdddr (butlast cl-defun))))
         (start-time (let ((v (hons-gentemp
                               (suffix "START-TIME-" fn))))
                       (eval `(prog1 (defg ,v -1)
                                (declaim (fixnum ,v))))))
         (tablename
          (eval `(defg
                   ,(hons-gentemp (suffix "MEMOIZE-HT-FOR-" fn))
                   nil)))
         (localtablename (make-symbol "TABLENAME"))
         (ponstablename
          (eval `(defg
                   ,(hons-gentemp (suffix "PONS-HT-FOR-" fn))
                   nil)))
         (localponstablename (make-symbol "PONSTABLENAME"))
         (sts (loop for x in (union stobjs-in stobjs-out)
                    when x collect (st-lst x)))
         (nra (+ (len formals) (len specials)))
         def
         success)
        (let*
          ((mf-trace-exit
            (and trace `((mf-trace-exit ',fn ,(length stobjs-out)
                                        ,*mf-ans*))))
           (mf-record-mht
            (and *record-mht-calls*
                 `((safe-incf (aref ,*mf-ma* ,2mfnn) 1 ,fn))))
           (mf-record-hit
            (and *record-hits* condition-body
                 `((safe-incf (aref ,*mf-ma*
                                    ,(+ 2mfnn *ma-hits-index*))
                              1 ,fn))))
           (body3
            `(let (,*mf-ans* ,*mf-args* ,*mf-ans-p*)
               (declare (ignorable ,*mf-ans* ,*mf-args* ,*mf-ans-p*))
               (cond
                ((not ,condition-body)
                 ,@mf-record-hit ; sort of a hit
                 ,(if (not trace)
                      body
                    (if (cdr stobjs-out)
                        `(progn
                           (setq ,*mf-ans*
                             (multiple-value-list ,body))
                           ,@mf-trace-exit
                           (values-list ,*mf-ans*))
                      `(progn (setq ,*mf-ans* ,body)
                         ,@mf-trace-exit
                         ,*mf-ans*))))
                ,@(if condition-body
                      `((t (when (null ,tablename)
                             ,@mf-record-mht
                             (setq ,tablename
                               (mht :size ,memo-table-init-size))
                             ,@(if (> nra 1)
                                   `((setq ,ponstablename
                                       (mht :size
                                            (* (1- ,nra)
                                               ,memo-table-init-size
                                               ))))))
                           ;; To avoid a remotely possible
                           ;; parallelism gethash error.
                           ,@(if (> nra 1)
                                 `((setq ,localponstablename
                                     (or ,ponstablename (mht)))))
                           #+parallel
                           ,@(if (> nra 1)
                                 `((ccl::lock-hash-table
                                    ,localponstablename)))
                           (setq ,*mf-args* (pist* ,localponstablename
                                                   ,@formals
                                                   ,@specials))
                           #+parallel
                           ,@(if (> nra 1)
                                 `((ccl::unlock-hash-table
                                    ,localponstablename)))
                           (setq ,localtablename
                             (or ,tablename (mht)))
                           ,@(if (> nra 1)
                                 '((maybe-reduce-memoize-tables)))
                           (multiple-value-setq
                               (,*mf-ans* ,*mf-ans-p*)
                             (gethash ,*mf-args* ,localtablename))
                           (cond
                            (,*mf-ans-p*
                             ,@(if trace `((oftr "~% ~s remembered."
                                                 ',fn)))
                             ,@mf-record-hit
                             ,@(cond
                                ((null (cdr stobjs-out))
                                 `(,@mf-trace-exit ,*mf-ans*))
                                (t
                                 `(,@ (and trace
                                           `((let*
                                               ((,*mf-ans*
                                                 (append
                                                  (take
                                                   ,(1- (length
                                                         stobjs-out))
                                                   ,*mf-ans*)
                                                  (list
                                                   (nthcdr
                                                    ,(1-
                                                      (length
                                                       stobjs-out))
                                                    ,*mf-ans*)))))
                                               ,@mf-trace-exit)))
                                      ,(cons
                                        'mv
                                        (nconc
                                         (loop for i below
                                               (1- (length
                                                    stobjs-out))
                                               collect
                                               `(pop ,*mf-ans*))
                                         (list *mf-ans*)))))))
                            (t ,(cond
                                 ((cdr stobjs-out)
                                  (let
                                    ((vars
                                      (loop for i below
                                            (length stobjs-out)
                                            collect (ofni "O~a" i))))
                                    `(mv-let ,vars ,body
                                       (setf (gethash ,*mf-args*
                                                      ,localtablename)
                                             (setq ,*mf-ans*
                                               (list* ,@vars)))
                                       ,@mf-trace-exit
                                       (mv ,@vars))))
                                 (t `(progn
                                       (setf (gethash ,*mf-args*
                                                      ,localtablename)
                                             (setq ,*mf-ans* ,body))
                                       ,@mf-trace-exit
                                       ,*mf-ans*)))))))))))
           (body2
            `(let ((,*mf-old-caller* *caller*)
               #+Clozure
               ,@(and *record-bytes*
                      `((,*mf-start-bytes*
                         (heap-bytes-allocated))))
               ,@(and *record-hons-calls*
                      `((,*mf-start-hons* *hons-call-counter*)))
               ,@(and *record-pons-calls*
                      `((,*mf-start-pons* *pons-call-counter*))))
            (declare
             (ignorable
              #+Clozure
              ,@(and *record-bytes* `(,*mf-start-bytes*))
              ,@(and *record-hons-calls* `(,*mf-start-hons*))
              ,@(and *record-pons-calls* `(,*mf-start-pons*)))
             (fixnum
              ,*mf-old-caller*
              ,@(and *record-hons-calls* `(,*mf-start-hons*))
              ,@(and *record-pons-calls* `(,*mf-start-pons*))
              #+Clozure
              ,@(and *record-bytes* `(,*mf-start-bytes*))))
            (unwind-protect
              (progn
                (setq ,start-time ,(if *record-time*
                                       '(internal-real-time)
                                     '0))
                (setq *caller* ,2mfnn)
                (,(if (cdr stobjs-out) 'multiple-value-prog1 'prog1)
                 ,body3
                 ,@(and *record-hons-calls*
                        `((safe-incf
                           (aref
                            ,*mf-ma*
                            ,(+ *ma-hons-index* 2mfnn))
                           (the fixnum (- *hons-call-counter*
                                          ,*mf-start-hons*))
                           ,fn)))
                 ,@(and *record-pons-calls*
                        `((safe-incf
                           (aref ,*mf-ma* ,(+ *ma-pons-index* 2mfnn))
                           (the fixnum (- *pons-call-counter*
                                          ,*mf-start-pons*))
                           ,fn)))
                 #+Clozure
                 ,@(and *record-bytes*
                        `((safe-incf
                           (aref ,*mf-ma* ,(+ *ma-bytes-index* 2mfnn))
                           (the fixnum
                             (- (heap-bytes-allocated)
                                ,*mf-start-bytes*))
                           ,fn)))
                 ,@(and *record-time*
                        `((safe-incf
                           (aref ,*mf-ma*
                                 (the fixnum (1+ ,*mf-count-loc*)))
                           (the fixnum (- (internal-real-time)
                                          ,start-time))
                           ,fn)))))
              (setq ,start-time -1)
              (setq *caller* ,*mf-old-caller*)))))
          (setq def
            `(defun ,fn ,formals ,@dcls
               ,@(if trace
                     (if (member trace '(notinline inline))
                         `((declare (,trace ,fn)))
                       `((declare (notinline ,fn)))))
               (declare (ignorable ,@formals ,@specials))
               ,@(and trace
                      `((oftr "~%(> ~s (" ',fn)
                        ,@(loop for v in (append formals specials)
                                append
                                `((oftr "~& ~s = " ',v)
                                  (prine ,v *trace-output*)))
                        (oftr "~& )")))
               (let* ((,*mf-count-loc*
                       (the fixnum (+ *caller* (* 2 ,fnn))))
                      (,*mf-ma* *memoize-call-array*)
                      ,localtablename ,localponstablename)
                 (declare (fixnum ,*mf-count-loc*)
                          (ignorable ,*mf-count-loc* ,*mf-ma*
                                     ,localponstablename
                                     ,localtablename)
                          (type (simple-array fixnum (*))
                                ,*mf-ma*))
                 ,@(and *record-calls*
                        `((memoize-incf)
                          (safe-incf (aref ,*mf-ma*
                                           ,*mf-count-loc*)
                                     1
                                     ,fn)))
                 (if (eql -1 ,start-time) ,body2 ,body3)))))
        (setf (gethash fn *number-of-arguments-and-values-ht*)
              (cons (length stobjs-in) (length stobjs-out)))
        (unwind-protect
          (progn
            (let ((ma *memoize-call-array*))
              (declare (type (simple-array fixnum (*)) ma))
              (loop for i fixnum from 2mfnn
                    below (the fixnum (+ 2mfnn *2max-memoize-fns*))
                    unless (eql (aref ma i) 0)
                    do (setf (aref ma i) 0)))
            (memoize-eval-compile def)
            (setf (gethash fn *memoize-info-ht*)
                  (make memoize-info-ht-entry
                        :fn fn
                        :tablename tablename
                        :ponstablename ponstablename
                        :old-fn old-fn
                        :memoized-fn old-fn
                        :condition condition
                        :inline inline
                        :num fnn
                        :sts sts
                        :trace trace
                        :start-time start-time
                        :cl-defun cl-defun
                        :formals formals
                        :specials specials
                        :stobjs-in stobjs-in
                        :stobjs-out stobjs-out
                        :record-bytes      *record-bytes*
                        :record-calls      *record-calls*
                        :record-hits       *record-hits*
                        :record-hons-calls *record-hons-calls*
                        :record-mht-calls  *record-mht-calls*
                        :record-pons-calls *record-pons-calls*
                        :record-time       *record-time*
                        :memo-table-init-size memo-table-init-size))
            (setf (gethash fnn *memoize-info-ht*) fn)
            (and condition (loop for s in sts do
                                 (push tablename
                                       (symbol-value s))))
            (setq success t))
          (unless success
            (setf (symbol-function fn) old-fn)
            (remhash fn *memoize-info-ht*)
            (remhash fnn *memoize-info-ht*)
            (and condition
                 (loop for s in sts
                       when (eq tablename
                                (car (symbol-value
                                      (the symbol s))))
                       do (pop (symbol-value (the symbol s)))))
            (ofd "~&; Memoize-fn:  Failed to memoize ~s." fn)
            (setq fn nil)))))))
  fn)

(defn1 unmemoize-fn (fn)
  (unwind-mch-lock
   (maybe-untrace! fn)
   (let* ((ma *memoize-call-array*)
          (l (memoizedp-raw fn)))
     (declare (type (simple-array fixnum (*)) ma))
     (unless l (ofe "Unmemoize-fn: ~s is not memoized." fn))
     (let* ((num (the fixnum (access memoize-info-ht-entry l
                                     :num)))
            (tablename (and l (access memoize-info-ht-entry l
                                      :tablename)))
            (n2 (* num *2max-memoize-fns*)))
       (declare (fixnum num n2))

; Note: condition is a first-class ACL2 function, not to be messed
; with here.

       (let (#+Clozure (ccl:*warn-if-redefine-kernel* nil))
         (let ((old-fn (access memoize-info-ht-entry
                               l :old-fn)))
           (if old-fn
               (setf (symbol-function (the symbol fn)) old-fn)
             (fmakunbound fn))))
       (loop for i fixnum from n2
             below (the fixnum (+ n2 *2max-memoize-fns*))
             unless (eql (aref ma i) 0)
             do (setf (aref ma i) 0))
       (remhash fn *memoize-info-ht*)
       (remhash num *memoize-info-ht*)
       (setf (symbol-value (the symbol tablename)) nil)
       (setf (symbol-value
              (the symbol (access memoize-info-ht-entry
                                  l :ponstablename)))
             nil)
       (loop for s in (access memoize-info-ht-entry l :sts) do
             (setf (symbol-value (the symbol s))
                   (remove tablename (symbol-value
                                      (the symbol s))))))))
   fn)

(defn1 maybe-unmemoize (fn)
  (when (memoizedp-raw fn) (unmemoize-fn fn)))

(defn1 memoized-functions ()
  (our-lock-unlock-memoize1
  (let (l)
    (maphash (lambda (fn v) (declare (ignore v))
               (when (symbolp fn) (push fn l))) *memoize-info-ht*)
             l)))

(defn1 length-memoized-functions ()
  (floor (1- (hash-table-count *memoize-info-ht*))
         2))

(defn1 unmemoize-all ()

  "(UNMEMOIZE-ALL) unmemoizes all currently memoized functions,
  including all profiled functions."

; A warning to would-be code improvers.  It would be a bad idea to
; redefine UNMEMOIZE-ALL to maphash over *memoize-info-ht* because of
; the ANSI rules concerning which hash table entries may be modified
; during a maphash.

   (loop for x in (memoized-functions) do (unmemoize-fn x))
   (memoize-init))

(defn1 rememoize-all ()
  (our-lock-unlock-memoize1
      (let (l)
        (maphash
         (lambda (k v)
           (when (symbolp k)
             (push
              (list (list (access memoize-info-ht-entry v :fn)
                          :condition
                          (access memoize-info-ht-entry v :condition)
                          :inline
                          (access memoize-info-ht-entry v :inline)
                          :trace
                          (access memoize-info-ht-entry v :trace)
                          :cl-defun
                          (access memoize-info-ht-entry v :cl-defun)
                          :formals
                          (access memoize-info-ht-entry v :formals)
                          :stobjs-in
                          (access memoize-info-ht-entry v :stobjs-in)
                          :specials
                          (access memoize-info-ht-entry v :specials)
                          :stobjs-out
                          (access memoize-info-ht-entry v
                                  :stobjs-out)
                          :memo-table-init-size
                          (access memoize-info-ht-entry v
                                  :memo-table-init-size))
                    (list
                     (access memoize-info-ht-entry v :record-bytes)
                     (access memoize-info-ht-entry v :record-calls)
                     (access memoize-info-ht-entry v :record-hits)
                     (access memoize-info-ht-entry v
                             :record-hons-calls)
                     (access memoize-info-ht-entry v
                             :record-mht-calls)
                     (access memoize-info-ht-entry v
                             :record-pons-calls)
                     (access memoize-info-ht-entry v :record-time)
                     (access memoize-info-ht-entry v
                             :memo-table-init-size)))
              l)))
         *memoize-info-ht*)
        (loop for x in l do (unmemoize-fn (caar x)))
        (gc$)
        (setq *max-symbol-to-fixnum*
              *ma-initial-max-symbol-to-fixnum*)
        (loop for x in l do
              (progv '(*record-bytes*
                       *record-calls*
                       *record-hits*
                       *record-hons-calls*
                       *record-mht-calls*
                       *record-pons-calls*
                       *record-time*)
                  (cadr x)
                (apply 'memoize-fn (car x)))))))

(defn1 compliant-and-ideal ()
  (let* ((logic-fns
          (eval '(let ((world (w *the-live-state*)))
                   (strip-cadrs (set-difference-theories
                                 (function-theory :here)
                                 (universal-theory 'ground-zero))))))
         (ideal-fns (collect-non-common-lisp-compliants
                     logic-fns (w *the-live-state*))))
    (mv (set-difference-eq logic-fns ideal-fns) ideal-fns)))

(defn1 uses-state (fn)
  (let* ((w (w *the-live-state*))
         (stobjs-in (getprop fn 'stobjs-in t 'current-acl2-world w))
         (stobjs-out (getprop fn 'stobjs-out t
                              'current-acl2-world w)))
    (or (and (consp stobjs-in) (member 'state stobjs-in))
        (and (consp stobjs-out) (member 'state stobjs-out)))))

(defn memoize-here-come (n)
  (let ((m (ceiling
            (+ 100 (* 1.1 (- n (- (/ *2max-memoize-fns* 2)
                                  (floor
                                   (/ (hash-table-count
                                       *memoize-info-ht*)
                                      2)))))))))
    (when (posp m) (memoize-call-array-grow (* 2 m)))))

(defn1 profile (fn &key trace)

  "PROFILE is a raw Lisp function.  PROFILE is much too wicked to ever
  be a proper part of ACL2, but can be useful in Common Lisp debugging
  and performance analysis, including examining the behavior of ACL2
  functions.  PROFILE returns FN.

  (PROFILE fn) redefines fn so that, whenever called, FN will record
  information about the call.  The information recorded may be
  displayed, for example, by invoking (MEMOIZE-SUMMARY).

  Call PROFILE from the very top-level of raw Common Lisp, not inside
  the ACL2 loop.  (To exit the ACL2 loop, type :q and later reenter
  the ACL2 loop with (lp).)

  (PROFILE fn) calls MEMOIZE-FN to memoize the function FN.  PROFILE
  may lie to MEMOIZE-FN by asserting that FN does not receive or
  return ACL2's STATE.  Profile takes the keyword parameter
  :TRACE, which defaults to NIL.

  It is possible to call MEMOIZE-FN in such a way as to force the
  memoization of an ACL2 function that has STATE as an explicit
  parameter using fraudulent FORMALS, STOBJS-IN, SPECIALS and
  STOBJS-OUT parameters; this could be useful for gathering
  information, but could be unsound, too."

  (unless (and (symbolp fn)
               (fboundp fn)
               (not (macro-function fn))
               (not (special-operator-p fn)))
    (ofe "Profile: The argument to profile must be an fboundp ~
          symbol."))
  (memoize-fn fn
              :inline nil
              :condition nil
              :trace trace))

(defn1 unmemoize-profiled ()

  "UNMEMOIZE-PROFILED is a raw Lisp function.  (UNMEMOIZE-PROFILED)
  unmemoizes and unprofiles all functions currently memoized with
  :CONDITION=NIL and :INLINE=NIL."

  (let (l)
    (maphash
     (lambda (k v)
       (if (and (symbolp k)
                (null (access memoize-info-ht-entry
                              v :condition))
                (null (access memoize-info-ht-entry
                              v :inline)))
           (push k l)))
     *memoize-info-ht*)
    (loop for x in l do (unmemoize-fn x))))

(defn1 profile-acl2 (&key (start 0) (trace nil) (lots-of-info t))

  "PROFILE-ACL2 is a raw Lisp function.  (PROFILE-ACL2 :start 'foo)
  profiles all the Common Lisp compliant ACL2 functions that have been
  accepted by ACL2, starting with the function foo."

  (let ((*record-bytes* #+Clozure lots-of-info #-Clozure nil)
        (*record-calls* lots-of-info)
        (*record-hits* lots-of-info)
        (*record-hons-calls* lots-of-info)
        (*record-mht-calls* lots-of-info)
        (*record-pons-calls* lots-of-info)
        (*record-time* lots-of-info))
    (when (symbolp start) (setq start (event-number start)))
    (check-type start integer)
    (let* ((packs (cons "ACL2"
                        (f-get-global 'packages-created-by-defpkg
                                      *the-live-state*)))
           (fns-ht (make-hash-table :test 'eq)))
      (loop for p in packs do
            (do-symbols (fn p)
              (cond ((gethash fn fns-ht) nil)
                    ((or (not (fboundp fn))
                         (macro-function fn)
                         (not (integerp (event-number fn)))
                         (< (event-number fn) start))
                     (setf (gethash fn fns-ht) 'no))
                    ((dubious-to-profile fn)
                     (setf (gethash fn fns-ht) 'no)
                     (ofvv "Not profiling ~a because it's ~a"
                          fn (dubious-to-profile fn)))
                    (t (setf (gethash fn fns-ht) 'yes)))))
      (maphash (lambda (k v)
                 (if (eq v 'no) (remhash k fns-ht)))
               fns-ht)
      (memoize-here-come (hash-table-count fns-ht))
      (maphash
       (lambda (k v)
         (declare (ignore v))
         (profile k :trace trace))
       fns-ht)
      (clear-memoize-call-array)
      (ofn "~a fns profiled." (hash-table-count fns-ht)))))

(defn1 profile-all (&key (trace nil) (lots-of-info t))

  "PROFILE-ALL is a raw Lisp function.  (PROFILE-ALL) profiles all the
  functions in all the ACL2 packages for which the number of arguments
  and number of return values can be guessed."

  (let ((*record-bytes* #+Clozure lots-of-info #-Clozure nil)
        (*record-calls* lots-of-info)
        (*record-hits* lots-of-info)
        (*record-hons-calls* lots-of-info)
        (*record-mht-calls* lots-of-info)
        (*record-pons-calls* lots-of-info)
        (*record-time* lots-of-info))
    (let* ((packs (cons "ACL2"
                        (f-get-global 'packages-created-by-defpkg
                                      *the-live-state*)))
           (fns-ht (make-hash-table :test 'eq)))
      (loop for p in packs do
            (do-symbols (fn p)
              (cond ((gethash fn fns-ht) nil)
                    ((or (not (fboundp fn))
                         (macro-function fn)
                         (special-operator-p fn))
                     (setf (gethash fn fns-ht) 'no))
                    ((dubious-to-profile fn)
                     (setf (gethash fn fns-ht) 'no)
                     (ofvv "Not profiling ~a because it's ~a"
                           fn (dubious-to-profile fn)))
                    (t (setf (gethash fn fns-ht) 'yes)))))
      (maphash (lambda (k v)
                 (if (eq v 'no) (remhash k fns-ht)))
               fns-ht)
      (memoize-here-come (hash-table-count fns-ht))
      (maphash
       (lambda (k v) (declare (ignore v))
         (profile k :trace trace))
       fns-ht)
      (clear-memoize-call-array)
      (ofn "~a fns profiled." (hash-table-count fns-ht)))))

(defn functions-defined-in-form (form)
  (cond ((consp form)
         (cond ((member (car form) '(defn defn1 defn2 defun
                                      defn1-one-output defmacro
                                      defun-one-output))
                (list (cadr form)))
               ((eq (car form) 'progn)
                (loop for z in (cdr form) nconc
                      (functions-defined-in-form z)))
               (t (mv-let (form p)
                    (macroexpand-1 form)
                    (and p (functions-defined-in-form
                            form))))))))

(defn functions-defined-in-file (file)
  (when (null (probe-file file))
    (ofw "functions-defined-in-file could not find file ~a." file)
    (return-from functions-defined-in-file nil))
  (let ((x nil)
        (avrc (cons nil nil)))
    (our-syntax ; protects against changes to *package*, etc.
     (with-open-file (stream file)
       (loop while (not (eq (setq x (read stream nil avrc))
                            avrc))
             nconc
             (progn
               (when (and (consp x)
                          (eq (car x) 'in-package))
                 (ignore-errors (eval x)))
               (functions-defined-in-form x)))))))

(defn1 profile-file (file)
   
  "PROFILE-FILE is a raw Lisp function.  (PROFILE-FILE file) calls
  PROFILE on all the functions defined in FILE.  If packages are
  changed in FILE in a sneaky way, or if macros are defined and then
  used at the top of FILE, who knows which functions will be profile.
  Functions that do not pass the test DUBIOUS-TO-PROFILE are not
  profiled.  A list of the names of the functions profiled is
  returned."

  (loop for fn in (functions-defined-in-file file)
        unless (dubious-to-profile fn)
        collect (profile fn)))

;  MEMOIZE-LET

; It might be a good enhancement to HT-LET to permit the carrying
; forward, with HOPY-CONS-CONSUME, of other honses.

(defn1 not-memoized-error (f)
  (ofe "NOT-MEMOIZED-ERROR:  ~s is not memoized." f))

(defmacro memoize-let (fn form)
  (let ((fn-name (gensym "FN-NAME"))
        (tablevalue (gensym "TABLEVALUE"))
        (ponstablevalue (gensym "PONSTABLEVALUE"))
        (h (gensym "H"))
        (ht1 (gensym "HT1")))
    `(let* ((,fn-name ,fn)
            (,h (memoizedp-raw ,fn-name)))
       (unless ,h (not-memoized-error ,fn-name))
       (let* ((,tablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :tablename)))
              (,ponstablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :ponstablename)))
              (,ht1 (mht)))
         (unwind-protect
             (progn (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :tablename))
                          ,ht1)
                    (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :ponstablename))
                           (mht))
                    ,form)
           ;; During the evaluation of form, a change to a stobj may
           ;; result in tablename getting a new value, in which case
           ;; we may not restore its old value.  And a change in the
           ;; memoization status of fn would make a restoration
           ;; pointless.
           (let ((test (and (eq (symbol-value
                                 (access memoize-info-ht-entry
                                         ,h :tablename))
                                ,ht1)
                            (eq ,h (memoizedp-raw ,fn-name)))))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :tablename))
                   (and test ,tablevalue))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :ponstablename))
                   (and test ,ponstablevalue))))))))


; MEMOIZE-ON AND MEMOIZE-OFF

(defmacro memoize-on (fn x)
  `(let* ((,*mo-f* ,fn) (,*mo-h* (memoizedp-raw ,*mo-f*)))
     (unless ,*mo-h* (not-memoized-error ,*mo-f*))
     (let ((,*mo-o* (symbol-function (the symbol ,*mo-f*))))
       (unwind-protect
           (progn (setf (symbol-function (the symbol ,*mo-f*))
                        (access memoize-info-ht-entry ,*mo-h*
                                :memoized-fn))
                  ,x)
         (setf (symbol-function (the symbol ,*mo-f*)) ,*mo-o*)))))

(defmacro memoize-off (fn x)
  `(let* ((,*mo-f* ,fn) (,*mo-h* (memoizedp-raw ,*mo-f*)))
       (unless ,*mo-h* (not-memoized-error ,*mo-f*))
       (let ((,*mo-o* (symbol-function (the symbol ,*mo-f*))))
         (unwind-protect
             (progn (setf (symbol-function (the symbol ,*mo-f*))
                          (access memoize-info-ht-entry ,*mo-h*
                                  :old-fn))
                    ,x)
           (setf (symbol-function (the symbol ,*mo-f*)) ,*mo-o*)))))

(defn global-suppress-condition-nil-memoization ()
  (maphash
   (lambda (k l)
     (when (symbolp k)
       (when (null (access memoize-info-ht-entry l :condition))
         (setf (symbol-function k)
               (access memoize-info-ht-entry l :old-fn)))))
   *memoize-info-ht*))

(defn global-restore-memoize ()
  (maphash (lambda (k l)
             (when (symbolp k)
               (setf (symbol-function k)
                     (access memoize-info-ht-entry l :memoized-fn))))
           *memoize-info-ht*))


; STATISTICS GATHERING AND PRINTING ROUTINES

(defg *memoize-summary-order-list*
  '(total-time number-of-calls)

  "*MEMOIZE-SUMMARY-ORDER-LIST* is a raw Lisp variable.  It is a list
  of order functions that MEMOIZE-SUMMARY uses to sort all functions
  that are currently memoized in preparation for displaying
  information about them.  The order is lexicographical with the first
  order having the most weight.  Each order function must take one
  argument, a symbol, and return a rational.

  The default is '(total-time number-of-calls).

  Options for the functions include:

     bytes-allocated
     bytes-allocated/call
     event-number
     execution-order
     hits/calls
     hons-calls
     pons-calls
     number-of-calls
     number-of-hits
     number-of-memoized-entries
     number-of-mht-calls
     symbol-name-order
     time-for-non-hits/call
     time/call
     total-time.
  ")

(defg *memoize-summary-limit* 20

  "*MEMOIZE-SUMMARY-LIMIT* is a raw Lisp variable whose value is the
  maximum number of functions upon which MEMOIZE-SUMMARY reports.  A
  nil value means report on all.")

(defg *shorten-ht* (mht :test 'eq))

(defn shorten (x n)
    (cond ((gethash x *shorten-ht*))
          (t (let ((*print-pretty* nil)
                   (str (with-output-to-string
                          (s)
                          (cond ((stringp x) (princ x s))
                                (t (prin1 x s))))))
               (cond ((< (length str) n) str)
                     ((setf (gethash x *shorten-ht*)
                            (concatenate 'string
                                         (subseq str 0 (max 0 n))
                                         "..."))))))))

(defg *memoize-summary-order-reversed* nil

  "*MEMOIZE-SUMMARY-ORDER-REVERSED* is a raw Lisp variable.  When it
  is not NIL, then MEMOIZE-SUMMARY reports on functions in order from
  least to greatest.")

(defn1 print-alist (alist separation)
  (check-type separation (integer 0))
  (setq alist
        (loop for x in alist collect
              (progn
                (check-type x
                            (cons (or string symbol)
                                  (cons (or string (integer 0))
                                        null)))
                (list (shorten (car x) 35)
                      (if (integerp (cadr x))
                          (ofnum (cadr x))
                        (cadr x))))))
  (let* ((max1 (loop for x in alist maximize (length (car x))))
         (max2 (loop for x in alist maximize (length (cadr x))))
         (width (max (or *print-right-margin* 70)
                     (+ separation max1 max2))))
    (loop for x in alist do
          (fresh-line)
          (princ (car x))
          (loop for i fixnum
                below (the fixnum
                        (- width (the fixnum (+ (length (car x))
                                                (length (cadr x))))))
                do (write-char #\Space))
          (princ (cadr x))))
  nil)

(defn1 hons-statistics ()

  "HONS-STATISTICS is a raw Lisp function (HONS-STATISTICS) prints
  detailed info about the hons system."

  (our-syntax-nice
   (oft "~&; Examining *hons-cdr-ht*:")
   (maphash (lambda (key value)
              (oft "~&; To ~s has been honsed:~%" key)
              (cond ((not (listp value))
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht*)
   (oft "~&; End of *hons-cdr-ht* examination. ~
         ~%~%; Examining *hons-cdr-ht-eql*:")
   (maphash (lambda (key value)
              (oft "~%; To ~s has been honsed:~%" key)
              (cond ((not (listp value))
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht-eql*)
   (oft "~%; End of *hons-cdr-ht-eql* examination. ~
         ~%~%; Examining *nil-ht*:~%")
   (oft "; To NIL has been honsed:~%")
   (maphash (lambda (key v2) (declare (ignore v2))
              (oft "~s, " key))
            *nil-ht*)
   (oft "~% End of *nil-ht* examination.")))

(defn1 hons-count ()
  (let ((n 0))
    (declare (fixnum n))
    (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
          (maphash (lambda (k v) (declare (ignore k))
                     (cond ((not (listp v))
                            (very-unsafe-incf n (hash-table-count v)
                                              hons-count))
                           (t (very-unsafe-incf n (length v)
                                                hons-count))))
                   (symbol-value tab)))
    (+ n (hash-table-count *nil-ht*))))

(defn1 hons-summary ()

  "HONS-SUMMARY prints information about all HONS calls and tables."

  (our-syntax-nice
   (oft "(defun hons-summary~%")
   (let ((sssub 0) (nhonses 0) (nsubs 0))
     (declare (fixnum sssub nhonses nsubs))
     (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
           (maphash
            (lambda (k v) (declare (ignore k))
              (cond
               ((not (listp v))
                (very-unsafe-incf sssub (hash-table-size v)
                                  hons-summary)
                (very-unsafe-incf nhonses (hash-table-count v)
                                  hons-summary)
                (very-unsafe-incf nsubs 1 hons-summary))
               (t (very-unsafe-incf nhonses (length v)
                                    hons-summary))))
            (symbol-value tab)))
     (very-unsafe-incf nhonses (hash-table-count *nil-ht*)
                       hons-summary)
     (print-alist
      `(,@(if (> *hons-call-counter* 0)
              `((" Hons hits/calls"
                 ,(let* ((c *hons-call-counter*)
                         (d (- c *hons-misses-counter*)))
                    (ofn "~,1e / ~,1e = ~,2f" d c (float (/ d c)))))))
          ,@(loop for tab in '(*hons-cdr-ht*
                               *hons-cdr-ht-eql*
                               *nil-ht*
                               *hons-str-ht*)
                  unless (eql 0 (hash-table-size (symbol-value tab)))
                  collect
                  (let* ((tabv (symbol-value tab))
                         (c (hash-table-count tabv))
                         (s (hash-table-size tabv)))
                    (list (ofn " ~a count/size"
                               (symbol-name tab))
                          (ofn "~,1e / ~,1e = ~,2f"
                               (ofnum c) (ofnum s) (float (/ c s))))))
          (" Number of sub tables" ,nsubs)
          (" Sum of sub table sizes" ,sssub)
          (" Number of honses" ,nhonses))
      5)
     (oft ")")
     nhonses)))

(defn1 pons-summary ()
  (our-syntax-nice
   (let ((sssub 0) (nponses 0) (nsubs 0) (nponstab 0))
     (declare (fixnum sssub nponses nsubs))
   (oft "(defun pons-summary~%")
   (maphash
    (lambda (k v)
      (cond ((symbolp k)
             (let ((tab (symbol-value (access memoize-info-ht-entry v
                                              :ponstablename))))
               (when tab
                 (very-unsafe-incf nponstab 1 pons-summary)
                 (maphash
                  (lambda (k v) (declare (ignore k))
                    (cond
                     ((not (listp v))
                      (very-unsafe-incf sssub (hash-table-size v)
                                        pons-summary)
                      (very-unsafe-incf nponses (hash-table-count v)
                                        pons-summary)
                      (very-unsafe-incf nsubs 1 pons-summary))
                     (t (very-unsafe-incf nponses (length v)
                                          pons-summary))))
                  tab))))))
    *memoize-info-ht*)
   (print-alist
    `((" Pons hits/calls"
       ,(let* ((c *pons-call-counter*)
               (d *pons-misses-counter*))
          (ofn "~,1e / ~,1e = ~,2f" d c (/ (- c d) (+ .0000001 c)))))
      (" Number of pons tables" ,(ofnum nponstab))
      (" Number of pons sub tables" ,(ofnum nsubs))
      (" Sum of pons sub table sizes" ,(ofnum sssub))
      (" Number of ponses" ,(ofnum nponses)))
    5)
   (oft ")")
   nil)))

(defn1 memoize-statistics (&optional (fn (memoized-functions)))

  "(MEMOIZE-STATISTICS fn) prints all the memoized values for fn."

  (cond ((listp fn) (mapc #'memoize-statistics fn))
        ((not (memoizedp-raw fn))
         (oft "~%; Memoize-statistics:  ~s is not memoized." fn))
        (t (let ((tb (symbol-value
                      (access memoize-info-ht-entry
                              (gethash fn *memoize-info-ht*)
                              :tablename))))
             (cond ((and tb (not (eql 0 (hash-table-count tb))))
                    (oft "~%; Memoized values for ~s." fn)
                    (maphash (lambda (key v)
                               (format t "~%~s~%=>~%~s" key v))
                             tb))))))
  nil)

(defn print-call-stack ()

  "(PRINT-CALL-STACK) prints the stack of memoized function calls
  currently running and the time they have been running."

  (let (l
        (time (internal-real-time))
        (*print-case* :downcase))
    (declare (fixnum time))
    (maphash (lambda (k v)
               (cond ((symbolp k)
                      (let ((x (symbol-value
                                (the symbol
                                  (access memoize-info-ht-entry
                                          v :start-time)))))
                        (declare (fixnum x))
                        (when (> x 0)
                          (push (cons k x) l))))))
             *memoize-info-ht*)
    (setq l (sort l #'< :key #'cdr))
    (setq l (loop for pair in l collect
                  (list (car pair)
                        (ofnum (/ (- time (cdr pair))
                                  *float-ticks/second*)))))
    (when l
      (terpri)
      (print-alist
       (cons '("Stack of memoized function calls."
               "Time since outermost call.")
             l)
       5))))

(defn1 hons-calls (x)

  "For a memoized function fn, (HONS-CALLS fn) is the number of times
  fn has called hons."

  (setq x (coerce-index x))
  (aref *memoize-call-array*
        (the fixnum (+ *ma-hons-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defn1 pons-calls (x)

  "For a memoized function fn, (PONS-CALLS fn) is the number of times
   fn has called pons."

  (setq x (coerce-index x))
  (aref *memoize-call-array*
        (the fixnum (+ *ma-pons-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

#+Clozure
(defn1 bytes-allocated (x)

  "For a memoized function fn, (BYTES-ALLOCATED fn) is the number of
  bytes fn has caused to be allocated on the heap."

  (setq x (coerce-index x))
  (aref *memoize-call-array*
        (the fixnum (+ *ma-bytes-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defn1 number-of-hits (x)

  "For a memoized function fn, (NUMBER-OF-HITS fn) is the number of
  times that a call of fn returned a remembered answer."

  (setq x (coerce-index x))
  (aref *memoize-call-array*
        (the fixnum (+ *ma-hits-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defn number-of-memoized-entries (x)

  "For a memoized function FN, (NUMBER-OF-MEMOIZED-ENTRIES FN) is the
  number of entries currently stored for FN."
  
  (hash-table-count (gethash x *memoize-info-ht*)))

(defn1 number-of-mht-calls (x)

  "For a memoized function FN, (NUMBER-OF-MHT-CALLS fn) is the number
  of times that the memoize hash-table for fn was created."

  (setq x (coerce-index x))
  (aref *memoize-call-array*
        (the fixnum (+ *ma-mht-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defn1 time-for-non-hits/call (x)
  (setq x (coerce-index x))
  (let ((n (- (number-of-calls x) (number-of-hits x))))
    (if (zerop n) 0 (/ (total-time x) n))))

(defn1 time/call (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (zerop n) 0 (/ (total-time x) n))))

(defn1 hits/calls (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (zerop n) 0 (/ (number-of-hits x) (float n)))))

(defn1 hons-acons-summary ()

  "(HONS-ACONS-SUMMARY) prints information about the existing fast
  hons alists."

  (let ((count 0) (size 0) (number 0) last-key alist)
    (declare (fixnum count size number))
    (maphash
     (lambda (key v)
       (cond ((typep v 'fixnum)
              (push (list
                     (if (setq last-key (cdr (last key)))
                         last-key number)
                     (len key)
                     (len key))
                    alist)
              (very-unsafe-incf number 1 hons-acons-summary)
              (very-unsafe-incf size (len key) hons-acons-summary)
              (very-unsafe-incf count (len key) hons-acons-summary))
             (t (push (list
                       (if (setq last-key (cdr (last key)))
                           last-key number)
                       (hash-table-size v)
                       (hash-table-count v))
                      alist)
                (very-unsafe-incf number 1 hons-acons-summary)
                (very-unsafe-incf size (hash-table-size v)
                           hons-acons-summary)
                (very-unsafe-incf count (hash-table-count v)
                           hons-acons-summary))))
     *hons-acons-ht*)
    (cond (alist
           (oft "~&~%Hons-acons statistics")
           (print-alist
            (list (list "Count/size"
                        (ofn "~,1e/~,1e = ~,2f"
                             (ofnum (hash-table-count
                                   *hons-acons-ht*))
                             (ofnum (hash-table-size
                                   *hons-acons-ht*))
                             (/ (hash-table-count *hons-acons-ht*)
                                (hash-table-size
                                 *hons-acons-ht*))))
                  (list "Total of counts" count)
                  (list "Total of sizes" size))
            5)
           (oft "~&~%For each HONS-ACONS entry~%(name size count)")
           (loop for x in alist do (print x))))))

#+Clozure
(defn1 bytes-allocated/call (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (eql n 0)
        0
      (/ (bytes-allocated x) n))))

(defn char-list-fraction (l)
  (if (atom l) 0
    (+ (char-code (car l))
       (/ (char-list-fraction (cdr l))
          256))))

(defn symbol-name-order (s)

  "SYMBOL-NAME-ORDER maps symbols to rationals preserving
  lexicographic order."

  (unless (symbolp s) (setq s (fixnum-to-symbol s)))
  (- (char-list-fraction (coerce (symbol-name s) 'list))))

(defn1 execution-order (s)
  (unless (symbolp s) (setq s (fixnum-to-symbol s)))
  (the fixnum (symbol-value
               (the symbol
                 (access memoize-info-ht-entry
                         (gethash s *memoize-info-ht*)
                         :start-time)))))

(defn compute-calls-and-times ()
  (let ((ma *memoize-call-array*)
        (2m *2max-memoize-fns*)
        (ca *compute-array*)
        (n (the fixnum (1+ *max-symbol-to-fixnum*))))
    (declare (type (simple-array fixnum (*)) ma)
             (type (simple-array t (*)) ca)
             (fixnum 2m n))
    (cond ((eql (length ca) n)
           (loop for i fixnum below n
                 do (setf (aref ca i) nil)))
          (t (setq *compute-array*
                   (make-array n :initial-element nil))
             (setq ca *compute-array*)))
    (loop for i fixnum below (the fixnum (* 2 n))
          do (setf (aref ma i) 0))
    (loop for i fixnum
          from *ma-initial-max-symbol-to-fixnum*
          to *max-symbol-to-fixnum* do
          (let ((2im (the fixnum (* i 2m))))
            (declare (fixnum 2im))
            (loop for j fixnum
                  from *ma-initial-max-symbol-to-fixnum*
                  to *max-symbol-to-fixnum* do
                  (let* ((2j (the fixnum (* 2 j)))
                         (index (the fixnum (+ 2j 2im))))
                    (declare (fixnum 2j index))
                    (let ((calls (the fixnum (aref ma index))))
                      (declare (fixnum calls))
                      (when (> calls 0)
                        (let ((time (aref ma (the fixnum
                                               (1+ index)))))
                          (declare (fixnum time))
                          (setf (aref ma 2j)
                                (the fixnum (+ (aref ma 2j) calls)))
                          (setf (aref ma (the fixnum (1+ 2j)))
                                (the fixnum (+ (aref
                                                ma
                                                (the fixnum (1+ 2j)))
                                               time)))
                          (push i (aref ca j)))))))))))

(defn1 number-of-calls (x)
  (setq x (coerce-index x))

; One must call COMPUTE-CALLS-AND-TIMES before invoking
; NUMBER-OF-CALLS to get sensible results.

  (aref *memoize-call-array*
        (the fixnum (* 2 (the fixnum x)))))

(defn1 total-time (x)

  (setq x (coerce-index x))

; One must call COMPUTE-CALLS-AND-TIMES before invoking
; TOTAL-TIME to get sensible results.

  (/ (aref *memoize-call-array*
           (the fixnum (1+ (the fixnum (* 2 x)))))
     *float-ticks/second*))

(defn1 list-fast-fns (ticks)
  (let ((ma *memoize-call-array*))
    (declare (type (simple-array fixnum (*)) ma))
    (compute-calls-and-times)
    (sort
     (loop for i fixnum from (1+ *ma-initial-max-symbol-to-fixnum*)
           to *max-symbol-to-fixnum*
           when (and
                 (let* ((2i (* 2 i))
                        (calls (aref ma 2i))
                        (time (aref ma (the fixnum (1+ 2i)))))
                   (declare (fixnum 2i calls time))
                   (and (>= calls 1000)
                        (< time (* calls ticks))))
                 (let ((l (gethash (gethash i *memoize-info-ht*)
                                   *memoize-info-ht*)))
                   (and l (null (access memoize-info-ht-entry l
                                        :condition)))))
           collect (fixnum-to-symbol i))
     #'>
     :key #'symbol-name-order)))

(defn lex-> (l1 l2)
  (cond ((or (atom l1)
             (atom l2))
         nil)
        ((> (car l1) (car l2)) t)
        ((< (car l1) (car l2)) nil)
        (t (lex-> (cdr l1) (cdr l2)))))

(defn1 memoize-summary-sort ()
  (let (pairs)
    (maphash
     (lambda (fn v)
       (when (symbolp fn)
       (let ((num (access memoize-info-ht-entry v :num)))
         (declare (fixnum num))
         (when (< 0 (number-of-calls num))
           (push (cons fn (loop for order
                                in *memoize-summary-order-list*
                                collect (funcall order fn)))
                 pairs)))))
     *memoize-info-ht*)
    (sort pairs
          (if *memoize-summary-order-reversed*
              (lambda (x y) (lex-> y x))
            #'lex->)
          :key #'cdr)))

(defn1 memoize-summary ()

  "(MEMOIZE-SUMMARY) reports data stored during the execution of the
  functions in (MEMOIZED-FUNCTIONS).

  Typically each call of a memoized function, fn, is counted.
  The elapsed time until an outermost function call of fn ends, the
  number of heap bytes allocated in that period (CCL only), and other
  'charges' are 'billed' to fn.  That is, quantities such as elapsed
  time and heap bytes allocated are not charged to subsidiary
  recursive calls of fn while an outermost call of fn is running.
  Recursive calls of fn, and memoized 'hits', are counted, unless fn
  was memoized with NIL as the value of the :INLINE parameter of
  MEMOIZE.

  The settings of the following determine, at the time a function is
  given to MEMOIZE, the information that is collected for calls of
  the function:

         Variable              type

         *RECORD-BYTES*       boolean    (available in CCL only)
         *RECORD-CALLS*       boolean 
         *RECORD-HITS*        boolean 
         *RECORD-HONS-CALLS*  boolean 
         *RECORD-MHT-CALLS*   boolean 
         *RECORD-PONS-CALLS*  boolean 
         *RECORD-TIME*        boolean 

  The settings of the following determine, at the time that
  MEMOIZE-SUMMARY is called, what information is printed:

         *REPORT-BYTES*       boolean   (available in CCL only)
         *REPORT-CALLS*       boolean 
         *REPORT-CALLS-FROM*  boolean 
         *REPORT-HITS*        boolean 
         *REPORT-HONS-CALLS*  boolean 
         *REPORT-MHT-CALLS*   boolean 
         *REPORT-PONS-CALLS*  boolean 
         *REPORT-TIME*        boolean 

         *REPORT-ON-MEMO-TABLES*   boolean 
         *REPORT-ON-PONS-TABLES*   boolean 
         *MEMOIZE-SUMMARY-LIMIT*            (or integerp null)
         *MEMOIZE-SUMMARY-ORDER-LIST*       (symbol symbol ... symbol)
         *MEMOIZE-SUMMARY-ORDER-REVERSED*   boolean

  Functions are sorted lexicographically according to the ordering
  induced by the function names in *MEMOIZE-SUMMARY-ORDER-LIST*, each
  member of which must be a unary function that returns a rational.

  The times reported by MEMOIZE-SUMMARY are always elapsed, i.e.,
  wall-clock times in seconds, unless 'run-time' is explicitly
  mentioned in the output that WATCH prints.

  (CLEAR-MEMOIZE-TABLES) forgets the remembered values of all memoized
  function calls.  It does not alter a function's status as being a
  memoized function, nor does not it the monitoring data accumulated.

  (UNMEMOIZE-ALL) undoes the memoization status of all memoized
  functions.

  (CLEAR-MEMOIZE-CALL-ARRAY) zeroes out the monitoring information for
  all functions.  It does not alter any function's status as a
  memoized function nor does it change the values remembered for any
  memoized function.

  Here is an example of hacking with *MEMOIZE-SUMMARY-ORDER-LIST* that
  instructs MEMOIZE-SUMMARY to print information about, say,
  1ST-MOD-ERR first:

    (PUSH (LAMBDA (FN)
            (IF (EQ FN '1ST-MOD-ERR) 1 0))
          *MEMOIZE-SUMMARY-ORDER-LIST*)."

  (compute-calls-and-times)
  (memoize-summary-after-compute-calls-and-times))

(defn short-symbol-name (sym)
  (let ((str (symbol-name sym)))
    (cond ((> (length str) 20)
           (intern (ofn "~a..." (subseq str 0 20))
                   (symbol-package sym)))
          (t sym))))

(defn1 memoize-summary-after-compute-calls-and-times ()
  
;  If COMPUTE-CALLS-AND-TIMES is not called shortly before this
;  function, MEMOIZE-SUMMARY-AFTER-COMPUTE-CALLS-AND-TIMES, is called,
;  the information reported may be quite untimely.

;  See the end of trace-emod.lisp for documentation.

 (let* ((fn-pairs (memoize-summary-sort))
        (ma *memoize-call-array*))
   (declare (type (simple-array fixnum (*)) ma))
   (when (and *memoize-summary-limit*
              (> (len fn-pairs) *memoize-summary-limit*))
     (setq fn-pairs
           (loop for i from 1 to *memoize-summary-limit* as
                 x in fn-pairs collect x)))
   (loop for pair in fn-pairs do
         (let* ((fn (car pair))
                (l (gethash fn *memoize-info-ht*))
                (tablename
                 (symbol-value (access memoize-info-ht-entry l
                                       :tablename)))
                (ponstablename (symbol-value
                                (access memoize-info-ht-entry
                                        l :ponstablename)))
                (start-time
                 (the fixnum
                   (symbol-value
                    (the symbol
                      (access memoize-info-ht-entry
                              l :start-time)))))
                (num (the fixnum
                       (access memoize-info-ht-entry l :num)))
                (nhits (the fixnum (number-of-hits num)))
                (nmht (the fixnum (number-of-mht-calls num)))
                (ncalls (the fixnum
                          (max (the fixnum (number-of-calls num))
                               1)))
                (hons-calls (the fixnum (hons-calls num)))
                (pons-calls (the fixnum (pons-calls num)))
                #+Clozure
                (heap-bytes-allocated (bytes-allocated num))
                (tt (total-time num))
                (t/c (time/call num))
                (tnh (time-for-non-hits/call num))
                (in-progress-str
                 (if (eql start-time -1) " " ", running, ")))
           (declare (fixnum start-time num nhits nmht ncalls
              hons-calls pons-calls
              #+Clozure heap-bytes-allocated))
           (print-alist
            `((,(ofn "(defun ~s~a~a"
                     (short-symbol-name fn)
                     (if (or (eql 0 nhits)
                             (not *report-hits*))
                         (ofn " call~a"
                              (if (eql nhits 1) "" "s"))
                       " hits/calls")
                     in-progress-str)
               ,(if (or *report-calls* *report-hits*)
                    (if (or (eql 0 nhits)
                            (not *report-hits*))
                        (ofn "~a" (ofnum ncalls))
                      (ofn "~a / ~a = ~a"
                           (ofnum nhits)
                           (ofnum ncalls)
                           (ofnum (/ nhits (float ncalls)))))
                  ""))
              ,@(if (and *report-mht-calls* (>= nmht 2))
                    `((" Number of calls to mht" ,(ofnum nmht))))
              ,@(if (and *report-time* (> tt 0))
                      `((" Time of all outermost calls"
                         ,(ofnum tt))
                        (" Time per call"
                         ,(ofnum t/c))))
              #+Clozure
              ,@(if (and (> heap-bytes-allocated 0) *report-bytes*)
                `((" Heap bytes allocated"
                    ,(ofnum heap-bytes-allocated))
                   (" Heap bytes allocated per call"
                    ,(ofnum (/ heap-bytes-allocated ncalls)))))
              ,@(if (and (> hons-calls 0) *report-hons-calls*)
               `((" Hons calls"
                       ,(ofnum hons-calls))))
              ,@(if (and (> pons-calls 0) *report-pons-calls*)
                `((" Pons calls"
                       ,(ofnum pons-calls))))
              ,@(if (and *report-hits* *report-time*
                         (not (eql 0 nhits)))
                    `((" Time per missed call"
                       ,(ofnum tnh))))
              ,@(if *report-calls-from*
                    (let (l (2num (the fixnum (* 2 num))))
                      (declare (fixnum 2num))
                      (loop for callern fixnum
                            in (aref *compute-array* num) do
                            (let* ((call-loc
                                    (the fixnum
                                      (+ 2num
                                         (the fixnum
                                           (* callern
                                              *2max-memoize-fns*)))))
                                   (calls (aref ma call-loc))
                                   (time 0))
                             (declare (fixnum call-loc calls time))
                             (push
                              `((,(ofn " From ~a"
                                   (if (<= callern
                                    *ma-initial-max-symbol-to-fixnum*)
                                           "outside"
                                         (fixnum-to-symbol callern)))
                                 ,(ofn "~a call~a~a"
                                       (ofnum calls)
                                       (if (= calls 1) "" "s")
                                       (let ((time-loc
                                              (the fixnum
                                                (1+ call-loc))))
                                         (declare (fixnum time-loc))
                                         (if (> (setq time
                                                      (aref ma
                                                            time-loc))
                                                0)
                                             (ofn " took ~a"
                                                  (ofnum
                                                   (/ time
                                            *float-ticks/second*)))
                                           ""))))
                                . ,calls)
                              l)))
                      (setq l (sort l #'> :key #'cdr))
                      (strip-cars l)))
              .
              ,(if (and (not *report-on-memo-tables*)
                        (not *report-on-pons-tables*))
                   nil
                 (let ((spsub 0) (nponses 0) (npsubs 0))
                   (declare (fixnum spsub nponses npsubs))
                   (and
                    (and ponstablename *report-on-pons-tables*)
                    (maphash
                     (lambda (key value)
                       (declare (ignore key))
                       (cond
                        ((not (listp value))
                         (very-unsafe-incf spsub (hash-table-size
                                                  value)
                                    memoize-summary)
                         (very-unsafe-incf nponses
                                    (hash-table-count value)
                                    memoize-summary)
                         (very-unsafe-incf npsubs 1 memoize-summary))
                        (t (very-unsafe-incf nponses
                                      (length value)
                                      memoize-summary))))
                     ponstablename))
                   `(,@(and
                        (and tablename *report-on-memo-tables*)
                        `((,(ofn " Memoize table count/size")
                           ,(ofn "~a / ~a = ~a"
                                 (ofnum (hash-table-count tablename))
                                 (ofnum (hash-table-size tablename))
                                 (ofnum
                                  (/ (hash-table-count tablename)
                                     (+ .001 (hash-table-size
                                              tablename))))))))
                       ,@(and
                          (and ponstablename *report-on-pons-tables*)
                          `((" (Pons table count/size"
                             ,(ofn "~a / ~a = ~a)"
                                   (ofnum (hash-table-count
                                           ponstablename))
                                   (ofnum (hash-table-size
                                           ponstablename))
                                   (ofnum (/ (hash-table-count
                                              ponstablename)
                                             (+ .001
                                                (hash-table-size
                                                 ponstablename))))))
                            (" (Number of pons sub tables"
                             ,(ofn "~a)" (ofnum npsubs)))
                            (" (Sum of pons sub table sizes"
                             ,(ofn "~a)" (ofnum spsub)))
                            (" (Number of ponses"
                             ,(ofn "~a)" (ofnum nponses)))))))))
            5))
         (oft ")"))))

;  CLEARING HONS AND MEMOIZE TABLES

;  Redefined later.
(defn clear-all-emod-compiled-functions () nil)

(defn1 clear-hash-tables ()
  (ofvv "Running (clear-hash-tables).")
  (our-lock-unlock-memoize1
   (clear-memoize-tables)
   (let (l)
     (maphash (lambda (k v)
                (declare (ignore v))
                (if (honsp k) (push k l)))
              *hons-acons-ht*)
     (cond ((eql (hash-table-size *hons-cdr-ht*)
                   *hons-cdr-ht-size*)
            (clrhash *hons-cdr-ht*))
           (t 
            (setq *hons-cdr-ht* (mht :test #'eq :weak :key))
            (setq *hons-cdr-ht* (mht
                                 :size *hons-cdr-ht-size*
                                 :test #'eq
                                 :weak :key
                                 ))))
     (setq *hons-cdr-ht-eql* (mht))
     (cond ((eql (hash-table-size *nil-ht*) *nil-ht-size*)
            (clrhash *nil-ht*))
           (t 
            (setq *nil-ht* (mht :weak :value))
            (setq *nil-ht* (mht :size *nil-ht-size*
                                :weak :value))))
     (loop for x in (table-alist 'persistent-hons-table
                                 (w *the-live-state*))
           when (car x)
           do (hons-copy-restore (car x)))
     (loop for x in *hons-vars-alist* do (hons-copy-restore (car x)))
     (clear-all-emod-compiled-functions)
     (mapc #'hons-copy-restore l)
     (maphash (lambda (k v)
                (cond ((and (consp k) (honsp k))
                       ;; all parts of k are already honsed.
                       nil)
                      ((integerp v)
                       (loop for tail on k while (consp tail) do
                             (hons-copy-restore (caar tail))))
                      (t (maphash (lambda (k v)
                                    (declare (ignore v))
                                    (hons-copy-restore k))
                                  v))))
              *hons-acons-ht*)))
  (ofvv "Finished (clear-hash-tables)."))
     
(defn1 empty-ht-p (x)
  (and (hash-table-p x)
       (eql 0 (hash-table-count x))))

(defn clear-one-memo-and-pons-hash (l)
  (setf (symbol-value
         (the symbol (access memoize-info-ht-entry l :tablename)))
        nil)
  (setf (symbol-value
         (the symbol (access memoize-info-ht-entry l :ponstablename)))
        nil))

(defn1 clear-memoize-table (k)

; See also hons.lisp.

  (when (symbolp k)
    (let ((l (gethash k *memoize-info-ht*)))
      (when l (clear-one-memo-and-pons-hash l)))))


(defn1 clear-memoize-tables ()

; See hons.lisp.

  (ofvv "Running (clear-memoize-tables).")
  (let (success)
    (unwind-protect
        (progn
          (maphash (lambda (k l)
                     (when (symbolp k)
                       (clear-one-memo-and-pons-hash l)))
                   *memoize-info-ht*)
          (setq success t))
      (or success (ofe "clear-memoize-tables failed."))))
  nil)

(defn clear-memoize-call-array ()
  (ofvv "Running (clear-memoize-call-array).")
  (let ((m *memoize-call-array*))
    (declare (type (simple-array fixnum (*)) m))
    (loop for i fixnum below (length m)
          unless (eql (aref m i) 0)
          do (setf (aref m i) 0))))

(defn1 flush-hons-get-hash-table-link (x)

  ;  See hons.lisp.

  (unless (atom x) (remhash x *hons-acons-ht*))
  x)


; HONS READ

; Hash consing when reading is implemented via a change to the
; readtable for the characters open parenthesis, close parenthesis,
; and period, the consing dot.

; *** NOTE:  The following collection of functions is just that: a
;            collection.  Unless you understand everything about the
;            various read table macros, then don't touch this code!

; See matching comment below.

; Note: our implementation of the #=/## reader, which we built because
; some Lisps would not us get past #500 or so, does not comply with
; ANSI at least in this regard: it does not allow the entry of looping
; structures as in '(#0= (3 #0#)), which is no problem for ACL2 users.

; WARNING: Any call of READ using *hons-readtable* as *readtable*
; needs to worry about the possible top-level return of
; *CLOSE-PAREN-OBJ* and *DOT-OBJ*, which are simply part of the
; fiction whereby we read while honsing.  Those two objects should
; absolutely not be returned as the value of an ACL2 function.  See,
; for example, the definition of HONS-READ.

(defg *close-paren-obj* '(#\)))

(defg *dot-obj*         '(#\.))

(defg *hons-readtable* (copy-readtable *acl2-readtable*))
(declaim (readtable *acl2-readtable* *hons-readtable*))

(defg *compact-print-file-ht* (mht))

(defg *compact-read-file-ht* (mht))

(defg *compact-print-file-n* 0)
(declaim (fixnum *compact-print-file-n*))

(defg *space-owed* nil)
(declaim (type boolean *space-owed*))

(defg *compact-read-file-readtable*
  (copy-readtable *hons-readtable*))
(declaim (readtable *compact-read-file-readtable*))

(defg *compact-read-init-done* nil)

(defg *hons-readtable-init-done* nil)

(defn1 nonsense (x)
  (or (eq x *close-paren-obj*) (eq x *dot-obj*)))

(defn1 check-nonsense (x)
  (cond ((nonsense x)
         (hread-error "~&;  Illegal object: ~s." (car x)))))

(defn1 hread-error (string &rest r)
  (our-syntax-nice
   (let* ((stream *standard-input*)
          (*standard-output* *error-output*)
          (*standard-input* *debug-io*))
     (apply #'format *error-output* string r)
     (cond ((and (streamp stream) (file-position stream))
            (format *error-output*
                    "~&; near file-position ~s in stream ~s."
                    (file-position stream) stream)))
     (ofe "hread."))))

(defn1 illegal-error1 (x)
  (hread-error "~&; ** Error:  Illegal:  ~s." x))

(defn1 illegal-error2 (stream char)
  (declare (ignore stream))
  (illegal-error1 char))

(defn1 close-paren-read-macro (stream char)
  (declare (ignore char stream))
  (if *read-suppress* (illegal-error1 #\)))
  *close-paren-obj*)

(defn1 dot-read-macro (stream char)
  (declare (ignore char))
  (if *read-suppress* (illegal-error1 #\.))
  (let ((ch (peek-char nil stream t nil t)))
    (cond ((or (member ch '(#\( #\) #\' #\` #\, #\" #\;
                            #\Tab #\Space #\Newline))
               (eql 13 (char-code ch))
               (multiple-value-bind (fn nonterminating)
                   (get-macro-character ch)
                 (and fn (not nonterminating))))
           *dot-obj*)
          (t (let ((*readtable* *acl2-readtable*))
               (unread-char #\. stream)
               (read nil t nil t))))))

(defn1 hons-read-list ()

  ; ***** hons

  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     ((eq o *dot-obj*)
      (let ((lo (read nil t nil t))
            (lp (read nil t nil t)))
        (check-nonsense lo)
        (cond
         ((eq lp *close-paren-obj*) lo)
         (t (illegal-error1 #\.)))))
     (t (hons-normed (maybe-str-hash o) (hons-read-list))))))

(defn1 hons-read-list-top ()

  ; ***** hons

  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     (t (check-nonsense o)
        (hons-normed (maybe-str-hash o)
                     (hons-read-list))))))

(defn1 hons-read-reader (stream char)

  ; ***** hons

  (declare (ignore char))
  (cond (*read-suppress*
         (unread-char #\( stream)
         (let ((*readtable* *acl2-readtable*))
           (read nil t nil t)))
        (t (hons-read-list-top))))

(defn1 hons-read (&optional i (eep t) eofv rp)

  ; ***** hons

  (let* ((*readtable* *hons-readtable*)
         (*standard-input* (or i *standard-input*)))
    (let ((x (read nil eep eofv rp)))
      (check-nonsense x)
      x)))

(defn1 hons-read-object (channel state-state)
  (our-lock-unlock-honsmv1
   (let ((*acl2-readtable* *hons-readtable*))
     (mv-let
      (eofp obj state)
      (read-object channel state-state)
      (check-nonsense obj)
      (mv eofp obj state)))))

(defn1 hons-read-file (file-name)
  (our-lock-unlock-hons1
   (with-open-file (file file-name)
     (let (temp ans (eof (cons nil nil)))
       (declare (dynamic-extent eof))
       (loop (setq temp (hons-read file nil eof nil))
             (if (eq eof temp)
                 (return (hons-copy1-consume-top (nreverse ans))))
             (setq ans (cons temp ans)))))))


;  COMPACT PRINT AND READ

(defmacro space-if-necessary ()

; do not call

  '(when *space-owed* (write-char #\Space) (setq *space-owed* nil)))

(defn1 compact-print-file-scan (x)

; do not call

  (unless (or (and (symbolp x)
                   (let ((p (symbol-package x)))
                     (or (eq p *main-lisp-package*)
                         (eq p *package*)))
                   (<= (length (symbol-name x)) 4))
              (and (stringp x) (<= (length x) 2))
              (and (integerp x) (< -100 x 1000))
              (characterp x))
    (let ((g (gethash x *compact-print-file-ht*)))
      (unless (or (atom x) g)
        (compact-print-file-scan (car x))
        (compact-print-file-scan (cdr x)))
      (unless (eq g 'give-it-a-name)
        (setf (gethash x *compact-print-file-ht*)
              (if g 'give-it-a-name 'found-at-least-once))))))

(defn1 compact-print-file-help (x hash)

; do not call

  (cond ((typep hash 'fixnum)
         (space-if-necessary)
         (write-char #\#)
         (princ hash)
         (write-char #\#))
        (t (cond ((eq hash 'give-it-a-name)
                  (let ((n *compact-print-file-n*))
                    (declare (fixnum n))
                    (when (eql n most-positive-fixnum)
                      (ofe "*compact-print-file-n* overflow."))
                    (setq n (the fixnum (+ 1 n)))
                    (setq *compact-print-file-n* n)
                    (setf (gethash x *compact-print-file-ht*) n)
                    (space-if-necessary)
                    (write-char #\#)
                    (princ n)
                    (write-char #\=))))
           (cond
            ((atom x)
             (space-if-necessary)
             (prin1 x)
             (setq *space-owed* t))
            (t (write-char #\()
               (setq *space-owed* nil)
               (loop (compact-print-file-help
                      (car x)
                      (gethash (car x) *compact-print-file-ht*))
                     (cond
                      ((null (cdr x))
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      ((or (progn
                             (setq hash
                                   (gethash (cdr x)
                                            *compact-print-file-ht*))
                             (or (eq hash 'give-it-a-name)
                                 (typep hash 'fixnum)))
                           (atom (cdr x)))
                       (space-if-necessary)
                       (write-char #\.)
                       (setq *space-owed* t)
                       (compact-print-file-help (cdr x) hash)
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      (t (pop x)))))))))

(defn1 compact-print-file (data file-name)

  "(COMPACT-PRINT-FILE x str) PRIN1s x to a new file named
   str so a Common Lisp can READ it and get back something
   EQUAL, assuming the package structures are the same on
   print and read.  COMPACT-PRINT-FILE prints as though
   *PRINT-CIRCLE* were T to minimize printing by a kind of
   compression."

  (unwind-protect
    (our-lock-unlock-hons1 
     (progn
       (setq *compact-print-file-ht* (mht))
       (setq *compact-print-file-n* 0)
       (setq *space-owed* nil)
       (with-open-file (*standard-output* file-name
                                          :direction :output
                                          :if-exists :supersede)
         (our-syntax
          (compact-print-file-scan data)
          (compact-print-file-help
           data (gethash data *compact-print-file-ht*))
          (namestring (truename *standard-output*))))))
    (setq *compact-print-file-ht* (mht))))
  
(defn1 ns-=-reader (stream subchar arg)

; do not call

  (declare (ignore stream subchar))
  (when (gethash arg *compact-read-file-ht*)
    (hread-error
     "~&; ns-=-reader:  ** Error:  #~s= is already defined to be ~s."
     arg (gethash arg *compact-read-file-ht*)))
  (setf (gethash arg *compact-read-file-ht*) (read nil t nil t)))

(defn1 ns-ns-reader (stream subchar arg)

; do not call

  (declare (ignore stream subchar))
  (or (gethash arg *compact-read-file-ht*)
      (hread-error "~&; ns-ns-reader:  ** Error:  meaningless #~s#."
                   arg)))

(defn1 compact-read-file (fn)

   "(COMPACT-READ-FILE str) READs a file named str that
   should have just one Lisp object in it.  Uses HONS
   instead of CONS while reading.  Respects ##.  The
   restriction about 'one thing' is related to the
   subtlety of the ## stuff."

  (unwind-protect 
    (our-lock-unlock-hons1
     (progn
       (setq *compact-read-file-ht* (mht))
       (with-open-file (*standard-input* fn)
         (our-syntax
          (let* ((*readtable* *compact-read-file-readtable*)
                 (eof (cons nil nil))
                 (p (read)))
            (check-nonsense p)
            (unless (eq (read nil nil eof) eof)
              (ofe "compact-read-file: ~s has too many ~
                forms." fn))
            (hons-copy1-consume-top p))))))
    (setq *compact-read-file-ht* (mht))))


;  HONS READTABLE INIT

(defn1 hons-readtable-init ()
  (setq *hons-readtable* (copy-readtable *acl2-readtable*))
  (let ((*readtable* *hons-readtable*))
    (set-macro-character #\( #'hons-read-reader)
    (set-macro-character #\) #'close-paren-read-macro)
    (set-macro-character #\. #'dot-read-macro t))
  (setq *hons-readtable-init-done* t))

; COMPACT READ INIT

(defn1 compact-read-init ()
  (setq *compact-read-file-readtable*
        (copy-readtable *hons-readtable*))
  (let ((*readtable* *compact-read-file-readtable*))
    (set-dispatch-macro-character #\# #\# #'ns-ns-reader)
    (set-dispatch-macro-character #\# #\= #'ns-=-reader))
  (setq *compact-read-init-done* t))


; MEMOIZE INIT

(defn1 memoize-init ()

; Should only be repeated by unmemoize-fn.
  (unwind-mch-lock
   (unless (eql *caller* (the fixnum
                          (* *ma-initial-max-symbol-to-fixnum*
                             *2max-memoize-fns*)))
     (ofe "memoize-init:  A memoized function is running."))
   (let (success)
     (unwind-protect
       (progn
         (setq *pons-call-counter* 0)
         (setq *pons-misses-counter* 0)
         (setq *memoize-info-ht* (mht))
         (setf (gethash *ma-initial-max-symbol-to-fixnum*
                        *memoize-info-ht*)
               "outside-caller")
         (setq *max-symbol-to-fixnum*
           *ma-initial-max-symbol-to-fixnum*)
         (setq *2max-memoize-fns*
           (* 2 *initial-max-memoize-fns*))
         (sync-memoize-call-array)
         (setq success t))
       (if success (setq *memoize-init-done* t)
         (ofd "~%; memoize init: Error **"))))))

;  HONS INIT

; The ACL2 persistent-hons-table, which is updated by DEFHONST, should
; be an alist whose keys are honses to be preserved as honses through
; any cleaning of hash tables, but not through an init-hash-tables.

(defn1 init-hons-acons-table ()

; See also hons.lisp.

  (setq *hons-acons-ht* (mht :test #'eq :weak :key
                             )))


(defn1 init-hash-tables ()

; See also hons.lisp.

; Only to be called as ACL2h is built.

  (unwind-protect
    (progn
      (setq *init-hash-tables-done* nil)
      (setq *hons-call-counter* 0)
      (setq *hons-misses-counter* 0)
      (setq *hons-cdr-ht* (mht))
      (setq *hons-cdr-ht* (mht :size *hons-cdr-ht-size*
                               :weak :key
                               :test #'eq))
      (setq *hons-cdr-ht-eql*    (mht))
      (setq *nil-ht* (mht))
      (setq *nil-ht* (mht :size *nil-ht-size* :weak :value))
      (init-hons-acons-table)
      (setq *hons-copy-aux-ht* (mht :test #'eq))
      (setq *init-hash-tables-done* t))
    (unless *init-hash-tables-done*
      (ofe "init-hash-tables failed."))))
  
(defg *max-mem-usage*
  (let ((phys (physical-memory)))
    (max (floor (* 7/8 phys))
         (- phys (expt 2 30)))))

(defg *gc-min-threshold* (expt 2 30))

#+Clozure
(defn1 set-and-reset-gc-thresholds ()
  (let ((n (max (- *max-mem-usage* (ccl::%usedbytes))
                *gc-min-threshold*)))
    (cond ((not (eql n (ccl::lisp-heap-gc-threshold)))
           (ccl::set-lisp-heap-gc-threshold n)
           (ofg "~&; set-and-reset-gc-thresholds: Setting ~
                       (lisp-heap-gc-threshold) to ~:d bytes.~%"
                n))))
  (ccl::use-lisp-heap-gc-threshold)
  (ofg "~&; set-and-reset-gc-thresholds: Calling ~
              ~%(use-lisp-heap-gc-threshold).")
  (cond ((not (eql *gc-min-threshold*
                   (ccl::lisp-heap-gc-threshold)))
         (ccl::set-lisp-heap-gc-threshold *gc-min-threshold*)
         (ofg "~&; set-and-reset-gc-thresholds: Setting ~
                       (lisp-heap-gc-threshold) to ~:d bytes.~%"
              *gc-min-threshold*))))

#+Clozure
(defn1 start-sol-gc ()
  
; Trigger GC after we've used all but (1/8, but not more than 1 GB) of
; the physical memory.

  (ccl::add-gc-hook
   #'(lambda ()
       (ccl::process-interrupt
        (slot-value ccl:*application* 'ccl::initial-listener-process)
        #'set-and-reset-gc-thresholds))
   :post-gc)

  (set-and-reset-gc-thresholds))

(defg *hons-init-hook*
  '(progn
     #+Clozure

; Some things that a CCL user might want to put into
; ~/ccl-init.lisp but may not know to do so:

     (progn

       (defn1 set-gc-threshold (bound)
         (when (< (ccl::lisp-heap-gc-threshold) bound)
           (ofg "~&; *hons-init-hook*:  Setting ~
                 (ccl::lisp-heap-gc-threshold) to ~:d bytes."
                bound)
           (ccl::set-lisp-heap-gc-threshold bound)
           (ccl::use-lisp-heap-gc-threshold))
         nil)

       (defn1 maybe-set-gc-threshold (&optional (fraction 1/32))
         (let (n)
           (ignore-errors (setq n (physical-memory)))
           (cond ((and (integerp n) (> n (* 2 (expt 10 9))))
                  (setq n (floor (* n fraction)))
                  (set-gc-threshold n)))))

; Try to determine whether *terminal-io* is a file.

       (when (fboundp 'live-terminal-p)
         (eval '(live-terminal-p)))

; It is important to watch what the gc is doing when using hons and
; memoize.

       (unless (equal '(t t)
                      (multiple-value-list (ccl::gc-verbose-p)))
         (ofvv "*hons-init-hook*:  Setting CCL's gc verbose.")
         (ccl::gc-verbose t t))

; CCL's ephemeral gc doesn't work well with honsing and memoizing,
; it seems.

       (when (ccl::egc-active-p)
         (ofvv "*hons-init-hook*:  Turning off CCL's ~
                ephemeral gc.")
         (ccl::egc nil))

; Allocate heap space perhaps a little more generously than CCL's
; default.

       (maybe-set-gc-threshold)

; Allow control-d to exit.

       (when (and (boundp 'ccl::*quit-on-eof*)
                  (not (eq t (symbol-value 'ccl::*quit-on-eof*))))
         (ofvv "*hons-init-hook*:  Control-d now exits CCL.")
         (setf (symbol-value 'ccl::*quit-on-eof*) t))

       nil)

; Sol Sword's scheme to control GC in CCL.  See long comment
; below.

     #+Sol (start-sol-gc)))


#||

         Sol Sword's scheme to control GC in CCL

The goal is to get CCL to perform a GC whenever we're using almost
all the physical memory, but not otherwise.

The usual way of controlling GC on CCL is via
LISP-HEAP-GC-THRESHOLD.  This value is approximately amount of memory
that will be allocated immediately after GC.  This means that the next
GC will occur after LISP-HEAP-GC-THRESHOLD more bytes are used (by
consing or array allocation or whatever.)  But this means the total
memory used by the time the next GC comes around is the threshold plus
the amount that remained in use at the end of the previous GC.  This
is a problem because of the following scenario:

 - We set the LISP-HEAP-GC-THRESHOLD to 3GB since we'd like to be able
   to use most of the 4GB physical memory available.

 - A GC runs or we say USE-LISP-HEAP-GC-THRESHOLD to ensure that 3GB
   is available to us.

 - We run a computation until we've exhausted this 3GB, at which point
   a GC occurs.  

 - The GC reclaims 1.2 GB out of the 3GB used, so there is 1.8 GB
   still in use.

 - After GC, 3GB more is automatically allocated -- but this means we
   won't GC again until we have 4.8 GB in use, meaning we've gone to
   swap.

What we really want is, instead of allocating a constant additional
amount after each GC, to allocate up to a fixed total amount including
what's already in use.  To emulate that behavior, we use the hack
below.  This operates as follows (assuming the same 4GB total physical
memory as in the above example:)

1. We set the LISP-HEAP-GC-THRESHOLD to (3.5G - used bytes) and call
USE-LISP-HEAP-GC-THRESHOLD so that our next GC will occur when we've
used a total of 3.5G.

2. We set the threshold back to 1GB without calling
USE-LISP-HEAP-GC-THRESHOLD.

3. Run a computation until we use up the 3.5G and the GC is called.
Say the GC reclaims 1.2GB so there's 2.3GB in use.  1GB more (the
current LISP-HEAP-GC-THRESHOLD) is allocated so the ceiling is 3.3GB.)

4. A post-GC hook runs which again sets the threshold to (3.5G -
used bytes), calls USE-LISP-HEAP-GC-THRESHOLD to raise the ceiling to
3.5G, then sets the threshold back to 1GB, and the process repeats.

A subtlety about this scheme is that post-GC hooks runs in a separate
thread from the main execution.  A possible bug is that in step 4,
between checking the amount of memory in use and calling
USE-LISP-HEAP-GC-THRESHOLD, more memory might be used up by the main
execution, which would set the ceiling higher than we intended.  To
prevent this, we interrupt the main thread to run step 4.

||#

(defn1 hons-init ()

; HONS-INIT may be called more than once, so choose things to repeat
; carefully.
  (unwind-mch-lock
   (eval *hons-init-hook*)
   (unless *init-hash-tables-done* (init-hash-tables))
   (unless *hons-readtable-init-done* (hons-readtable-init))
   (unless *memoize-init-done* (memoize-init))
   (unless *compact-read-init-done* (compact-read-init))
   (float-ticks/second-init)

; Good idea to repeat float-ticks/second-init; speed varies from cpu
; to cpu.

   nil))

(defn1 all-module-names ()
  (loop for x in
        (sort (strip-cars (table-alist 'defm-table
                                       (w *the-live-state*)))
              (lambda (x y) (< (event-number x) (event-number y))))
        collect x))

(defn1 all-modules ()
  (loop for x in (all-module-names) collect (eval x)))

;;; SHORTER, OLDER NAMES

(defn1 memsum (&rest r)
  (apply #'memoize-summary r))

(defn1 hsum (&rest r)
  (apply #'hons-summary r))

(defn1 memstat (&rest r)
  (apply #'memoize-statistics r))

(defn1 hstat (&rest r)
  (apply #'hons-statistics r))

(defmacro memo-on (&rest r)
  `(memoize-on ,@r))

(defmacro memo-off (&rest r)
  `(memoize-off ,@r))

(defn1 clear-memo-tables (&rest r)
  (apply #'clear-memoize-tables r))

(defg *compiled-module-ht* (mht :test 'eq)
  
  "The hash table *COMPILED-MODULE-HT* maps a module name n to a cons
  of (1) the module named n with (2) the compiled version of module n,
  which is a compiled function, for primitive modules, and an array,
  for nonprimitive modules.  The car is EQ to the module that was the
  symbol value of n when the compilation was done.")


;;;                                WATCH

(defg *watch-forms*
  '("A string or a quoted form in *WATCH-FORMS* is ignored."
    (watch-current-form)
    (print-call-stack)
    (memoize-summary)
    (time-since-watch-start)
    (time-of-last-watch-update)
    '(print-items *watch-items*)
    '(hons-calls/sec-run-time)
    '(hons-hits/calls)
    '(hons-acons-summary)
    '(pons-calls/sec-run-time)
    '(pons-hits/calls)
    '(pons-summary)
    '(hons-summary)
    #+Clozure '(watch-shell-command "pwd")
    '(number-of-profiled-fast-fns)
    '(physical-memory-on-this-machine)
    #+Clozure '(number-of-cpus-on-this-machine)
    #+Clozure '(gc-count)
    )

  "The forms in *WATCH-FORMS* are evaluated periodically and the
  output is written to *WATCH-FILE*.  Change *WATCH-FORMS*
  to produce whatever watch output you want.")

(defg *watch-items*
  '((length-memoized-functions)
    *memoize-summary-limit*
    *memoize-summary-order-list*
    *memoize-summary-order-reversed*
    #+Clozure (ccl::lisp-heap-gc-threshold)
    #+Clozure (ccl::%freebytes)
    *max-mem-usage*

    *watch-forms*
    *watch-file*
    *watch-items*

    *count-hons-calls*
    *count-pons-calls*

    *memoize-summary-order-list*
    *memoize-summary-order-reversed*
    *memoize-summary-limit*

    *record-bytes*
    *record-calls*
    *record-hits*
    *record-hons-calls*
    *record-mht-calls*
    *record-pons-calls*
    *record-time*

    *report-bytes*
    *report-calls*
    *report-calls-from*
    *report-hits*
    *report-hons-calls*
    *report-mht-calls*
    *report-pons-calls*
    *report-time*
    *report-on-memo-tables*
    *report-on-pons-tables*


    )
  "*WATCH-ITEMS* is a list forms to be printed by WATCH-DO, 
  each followed by its value and documentation."
  )

(defg *watch-last-run-time* 0)

(defg *watch-last-real-time* 0)

(defg *watch-start-real-time* 0)

(defg *watch-start-run-time* 0)

(defg *watch-start-gc-count*  0)

(defg *watch-start-gc-time* 0)

(defg *watch-count* 0)

(defg *watch-start-heap-bytes-allocated* 0)

(defg *watch-string-start* 0)

(defn watch-do ()

  "(WATCH-DO) evaluates the members of *WATCH-FORMS* and prints the
  resulting output to *WATCH-FILE*, which is returned."

  (setq *memoize-safe-incf-counter* *memoize-safe-incf-delta*)
  (cond ((and *watch-file*
              (or (> (floor (- (get-internal-run-time)
                               *watch-last-run-time*)
                            internal-time-units-per-second)
                     5)
                  (> (floor (- (get-internal-real-time)
                               *watch-last-real-time*)
                            internal-time-units-per-second)
                     5)))
         (when (equal *watch-string* "")
           (ofe "Invoke (watch) before calling (watch-do) the ~
                 first time."))
         (let* ((file *watch-file*)
                (*watch-file* nil)) ;; Prevents a recursive call.
           (incf *watch-count*)
           (setf (fill-pointer *watch-string*) *watch-string-start*)
           (our-syntax-nice
            (with-output-to-string
              (*standard-output* *watch-string*)
              (setq *print-miser-width* 100)
              (prog ((l *watch-forms*) form)
                next
                (when (atom l) (return))
                (setq form (pop l))
                (handler-case (eval form)
                              (error ()
                                     (oft "~&~s~50t? error in eval ?"
                                          form)
                                     (go on)))
                on
                (fresh-line)
                (go next))))
           (with-open-file (stream file
                                   :direction :output
                                   :if-does-not-exist :create
                                   :if-exists #+Clozure :overwrite
                                   #-Clozure :supersede)
             #+Clozure (ccl::stream-length stream 0)
             (write-string *watch-string* stream)
             #+Clozure (ccl::stream-length stream
                                           (length *watch-string*))
             (ofvv "~a written." *watch-file*)
             nil)
           (setq *watch-last-run-time* (get-internal-run-time))
           (setq *watch-last-real-time* (get-internal-real-time)))))
  *watch-file*)

(defn1 watch ()

  "WATCH is a raw Lisp function.  Invoking (WATCH) initializes the
  variable *WATCH-FILE* and starts the periodic evaluation of
  (WATCH-DO), which evaluates the members of *WATCH-FORMS* and writes
  the output to *WATCH-FILE*.

  *WATCH-FILE* is returned by (WATCH) and by (WATCH-DO).

  If *WATCH-FILE* is viewed in Emacs, the display of the file will be
  updated when the file changes periodically because the file has
  'mode: auto-revert' in its first line.

  (SETQ *WATCH-FILE* NIL) stops the periodic printing to the watch
  file.

  You are most welcome to change the members of *WATCH-FORMS* to get
  your desired output in the watch file.

  Usually, the form (MEMOIZE-SUMMARY) is a member of *WATCH-FORMS*.
  (MEMOIZE-SUMMARY), q.v., prints information about calls of memoized
  and/or profiled functions.

  A suggested approach for getting formation about calls to Common
  Lisp functions:

    0. Invoke (WATCH).
  
    1. Profile some functions that have been defined.

       For example, call (PROFILE 'foo1), ...

       Or, for example, invoke call PROFILE-FILE on the name of a file
       that contains the definitions of functions that have been
       defined.

       Or, as a perhaps extreme example, invoke
       (PROFILE-ACL2), which will profile many of the functions that
       have been introduced to ACL2, but may take a minute.

       Or, as a very extreme example, invoke
       (PROFILE-ALL), which will profile many functions, but may take
       a minute.

    2. Run a Lisp computation of interest to you that causes some of
       the functions you have profiled to be executed.
 
    3. Invoke (WATCH-DO).
  
    4. Examine, perhaps in Emacs, the file whose name was returned by
       (WATCH-DO), which is always the value of the variable
       *WATCH-FILE*.  *WATCH-FILE* contains information about calls to
       the ACL2 functions you have profiled during the computation of
       interest.

  Consider MEMOIZING some of your functions that are called frequently
  on the same arguments.

  In Emacs, the command 'M-X AUTO-REVERT-MODE' toggles auto-revert
  mode, i.e., causes a buffer to exit auto-revert mode if it is in
  auto-revert mode, or to enter auto-revert mode if it is not.  In
  other words, to stop a buffer from being auto-reverted, simply
  toggle auto-revert mode; toggle it again later if you want more
  updating.  'M-X AUTO-REVERT-MODE' may be thought of as a way of
  telling Emacs, 'keep the watch buffer still'.

  What is the relationship between memoization and watch?  And when
  does WATCH-DO get called?  We have tried a variety of approaches,
  even involving parallel processes and timers.  However, our current
  approach is this: Profiled and/or memoized code increments various
  counters.  After each one million such increments, WATCH-DO runs
  provided further that five seconds have elapsed since the last
  WATCH-DO report.

  Invoke (WATCH-HELP) outside of ACL2 for further details."

  (setq *watch-count* 0)
  (setq *memoize-safe-incf-counter* 0)
  (setq *watch-file*
    (format nil "watch-output-temp-~D.lsp" (getpid$)))
  (setq *watch-string*
        (let ((ws (make-array 10
                              :element-type 'character
                              :adjustable t :fill-pointer t)))
          (setf (fill-pointer ws) 0)
          (with-output-to-string
            (*standard-output* ws)
            (format t "| -*- mode: auto-revert; mode: ~
               font-lock; -*- |~%"))
          (setq *watch-string-start* (fill-pointer ws))
          ws))

; Test that the file can indeed be written.

  (with-open-file (*standard-output* *watch-file*
                                     :direction :output
                                     :if-exists :rename)
    (princ "(WATCH) has been invoked, but (WATCH-DO) has not, yet."))
  (setq *watch-start-real-time* (get-internal-real-time))
  (setq *watch-start-run-time* (get-internal-run-time))
  #+Clozure
  (progn
    (setq *watch-start-gc-count* (ccl::full-gccount))
    (setq *watch-start-gc-time* (ccl::gctime))
    (setq *watch-start-heap-bytes-allocated* (heap-bytes-allocated)))
  (ofvv "(WATCH) called.  Printing to *WATCH-FILE*.")
  (ofvv "(SETQ *WATCH-FILE* NIL) stops printing to *WATCH-FILE*.")
  (ofvv "(WATCH-HELP) prints some information.")
  *watch-file*)

;  USER RESETTABLE WATCH FUNCTIONS AND VARIABLES

(defn1 first-string (l)
  (loop for x in l when (stringp x) do (return x)))

(defn print-items (vars)

  "PRINT-ITEMS prints the names, values, and documentation if any, of
  of the forms in vars."

  (loop for v in vars do
        (oft "~s => " v)
        (let* ((x (eval v))
               (y (if (and (symbolp x)
                           (not (typep x 'boolean)))
                      (list 'quote x)
                    x)))
          (let ((str (ofn "~:d" y)))
            (when (or (consp x) (> (length str) 40))
              (oft "~& "))
            (oft "~a" str))
          (let (doc)
            (ignore-errors
              (setq doc (documentation (if (atom v)
                                           v
                                         (car v))
                                       (if (atom v)
                                           'variable
                                         'function))))
            (when doc (oft "~&  ~a~%" doc))
            (oft "~%")))))

(defmacro defw (fn &rest r)
  `(defn ,fn ()
     (let ((fn (string-capitalize (symbol-name ',fn))))
       ,@r)))

(defmacro oft-wrm (str &rest r)
  `(oft ,str (or *print-right-margin* 70) ,@r))

(defw time-of-last-watch-update
  (multiple-value-bind (sec mi h d mo y)
      (decode-universal-time (get-universal-time))
    (let (m)
      (cond ((> h 12)
             (setq m " p.m.")
             (setq h (- h 12)))
            (t (setq m " a.m.")))
      (let* ((ans (ofn "~2,d:~2,'0d:~2,'0d~a ~4d/~d/~d"
                       h mi sec m y mo d)))
        (terpri)
        (oft-wrm "~v<~a~;~a~>" fn ans)))))

(defun watch-time ()
  (/ (- (get-internal-real-time) *watch-start-real-time*)
     *float-internal-time-units-per-second*))

(defun watch-run-time ()
  (/ (- (get-internal-run-time) *watch-start-run-time*)
     *float-internal-time-units-per-second*))

(defw hons-calls/sec-run-time
  (let* ((c *hons-call-counter*)
         (ans
          (cond ((eql c 0) "No hons calls yet.")
                (t (ofn "~,1e" (round (/ c (+ .000001
                                              (watch-run-time)))))))))
    (oft-wrm "~v<~a~;~a~>" fn ans)))

(defw pons-calls/sec-run-time
  (let* ((c *pons-call-counter*)
         (ans
          (cond ((eql c 0) "No pons calls yet.")
                (t (ofn "~,1e" (round (/ c (+ .000001
                                              (watch-run-time)))))))))
    (oft-wrm "~v<~a~;~a~>" fn ans)))

(defw pons-hits/calls
  (let* ((c *pons-call-counter*)
         (h (- c *pons-misses-counter*))
         (ans
          (cond ((eql c 0) "No pons calls yet.")
                (t (ofn "~,1e / ~,1e = ~,2f" h c (/ h c))))))
    (oft-wrm "~v<~a~;~a~>" fn ans)))

(defw hons-hits/calls
  (let* ((c *hons-call-counter*)
         (h (- c *hons-misses-counter*))
         (ans
          (cond ((eql c 0) "No hons calls yet.")
                (t (ofn "~,1e / ~,1e = ~,2f" h c (/ h c))))))
    (oft-wrm "~v<~a~;~a~>" fn ans)
    #+Clozure
    (oft-wrm "~%~v<Heap bytes allocated since watch start~;~,1e~>"
             (- (heap-bytes-allocated)
                *watch-start-heap-bytes-allocated*))))

#+Clozure
(defw gc-count ()
  (if (boundp '*watch-start-gc-time*)
      (let ((h 0)
            (mi 0)
            (sec (floor (- (ccl::gctime) *watch-start-gc-time*)
                        internal-time-units-per-second)))
        (multiple-value-setq (mi sec) (floor sec 60))
        (multiple-value-setq (h mi) (floor mi 60))
        (oft-wrm "~v,20<~a~;~a.~;~a~;~2d:~2,'0d:~2,'0d.~>"
             fn
             (- (ccl::full-gccount) *watch-start-gc-count*)
             "Time in gc since watch start"
             h mi sec))))

(defw number-of-profiled-fast-fns
  (let ((ma *memoize-call-array*)
        (ticks 6000)
        (ans))
    (declare (type (simple-array fixnum (*)) ma))
    (setq ans
          (loop for i fixnum
                from (1+ *ma-initial-max-symbol-to-fixnum*)
                to *max-symbol-to-fixnum*
                when (and
                      (let* ((2i (* 2 i))
                             (calls (aref ma 2i))
                             (time (aref ma (the fixnum (1+ 2i)))))
                        (declare (fixnum 2i calls time))
                        (and (>= calls 1000)
                             (< time (* calls ticks))))
                      (let ((l (gethash (gethash i *memoize-info-ht*)
                                        *memoize-info-ht*)))
                        (and l (null (access memoize-info-ht-entry l
                                             :condition)))))
                count 1))
    (if (> ans 2)
        (oft-wrm "~v<; ~a~;~a.~>" fn ans))))

#+Clozure
(defw number-of-cpus-on-this-machine
  (let* ((ans (ofn "~:d" (ccl::cpu-count))))
    (oft-wrm "~v<~a~;~a~>" fn ans)))

(defw physical-memory-on-this-machine
  (let* ((ans (ofn "~:d" (physical-memory))))
    (oft-wrm "~v<~a~;~a bytes.~>" fn ans)))

#+Clozure
(defun watch-shell-command (command-string)
  (let ((file (ofn "/tmp/watch-~a.XXXX" (getpid$))))
    (ccl::os-command (ofn "~a >& ~a" command-string file))
    (let ((output
           (with-output-to-string
             (*standard-output*)
             (with-open-file (stream file)
               (let (c)
                 (loop while (setq c (read-char stream nil nil))
                       do (write-char c)))))))
      (oft-wrm "~v<~a~;~a~>" command-string output))))

(defun watch-current-form ()
  (when (consp -)
    (let ((*print-pretty* nil))
      (princ "- ")
      (princ (shorten - 65)))))
                          
(defw time-since-watch-start ()
  (if (boundp '*watch-start-real-time*)
      (multiple-value-bind (mi1 sec1) (floor (round (watch-time)) 60)
        (multiple-value-bind (h1 mi1) (floor mi1 60)
          (multiple-value-bind (mi2 sec2)
              (floor (round (watch-run-time)) 60)
            (multiple-value-bind (h2 mi2) (floor mi2 60)
              (terpri)
              (oft-wrm "~v<Watch update ~a. ~
                ~;~a~;~2d:~2,'0d:~2,'0d.~;~a~;~2d:~2,'0d:~2,'0d.~>"
                   *watch-count* fn h1 mi1 sec1 "Run-time"
                   h2 mi2 sec2)))))))

(defw timer-function-name
  (let ((ans #+RDTSC64 "CCL::RDTSC64."
             #-RDTSC64 "GET-INTERNAL-REAL-TIME."))
    (oft-wrm "~v<~a~;~a~>" fn ans)))

#+Clozure
(defun make-watchdog (duration)

;   Thanks to Gary Byers for this!

   (let* ((done (ccl:make-semaphore))
          (current ccl:*current-process*))
      (ccl::process-run-function "watchdog"
        (lambda ()
          (or (ccl:timed-wait-on-semaphore done duration)
              (ccl:process-interrupt
               current #'error "Time exceeded"))))
      done))

#+Clozure
(defun call-with-timeout (function duration)

  "(CALL-WITH-TIMEOUT function duration) calls FUNCTION on no
  arguments and returns its values unless more than DURATION seconds
  elapses before completion, in which case the FUNCTION computation is
  interrupted and calls ERROR.

  Thanks to Gary Byers for this beaut."

   (let* ((semaphore (make-watchdog duration)))
      (unwind-protect
          (funcall function)
        (ccl:signal-semaphore semaphore)))) 
