; ACL2 Version 3.2 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2007  University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic
; NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; The original version of this file was contributed by Bob Boyer and
; Warren A. Hunt, Jr.  The design of this system of Hash CONS,
; function memoization, and fast association lists (applicative hash
; tables) was initially implemented by Boyer and Hunt.

(in-package "ACL2")

; The next few functions require special "under the hood"
; implementations so that we always use a previously allocated CONS
; object if a pair EQUAL to (CONS x y) is requested and currently
; exists.

#+(or acl2-loop-only (not hons))
(defun hons (x y)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.

  #+hons
  ":Doc-Section Hons

  hash cons and function memoization~/

  This documentation topic relates to an experimental extension of
  ACL2 under development by Bob Boyer and Warren Hunt, which is far
  from fully incorporated into the ACL2 build.  Stay tuned.....~/~/"

  (cons x y))

#+(or acl2-loop-only (not hons))
(defun hons-equal (x y)
  (declare (xargs :guard t))
  #+hons
  ":Doc-Section Hons

  test equality of hons structures~/

  To be written.~/~/"
  ;; Has an "under the hood" implementation.
  (equal x y))

#+(or acl2-loop-only (not hons))
(defun hons-copy (x)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.
  x)

(defun hons-assoc-equal (x y)
  (declare (xargs :guard t))
  (cond ((atom y) nil)
        ((and (consp (car y))
              (hons-equal x (car (car y))))
         (car y))
        (t (hons-assoc-equal x (cdr y)))))

; Hash-Cons'ed association lists used with HONS-GET,
; HONS-ACONS, and HONS-ACONS!, may have any symbol at the
; end instead of just NIL as is with ALISTP, allowing each
; association list to be "named".

#+(or acl2-loop-only (not hons))
(defun hons-get-fn-do-hopy (x l)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.
  (hons-assoc-equal x l))

#+(or acl2-loop-only (not hons))
(defun hons-get-fn-do-not-hopy (x l)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.
  (hons-assoc-equal x l))

(defmacro hons-get (x l)
  (list 'hons-get-fn-do-hopy x l))

(add-macro-alias hons-get hons-get-fn-do-hopy)

#+(or acl2-loop-only (not hons))
(defun hons-acons (key value l)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.
  ;; Note:  under the hood, the key will be hopied, but the alist
  ;; and its top-level pairs are built with CONS, not HONS.
  (cons (cons (hons-copy key) value) l))

#+(or acl2-loop-only (not hons))
(defun hons-acons! (key value l)
  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.  The (HONS KEY VALUE)
  ;; below will cause VALUE to have a unique representation, which,
  ;; for large structures, may require a substantial amount of work.
  (hons (hons (hons-copy key) value) l))

; Invoking set-state-ok permits us to define a function with
; state as a parameter, in particular HONS-READ-OBJECT.

#+(or acl2-loop-only (not hons))
(defun hons-read-object (channel state)
  (declare (xargs :stobjs state
                  :guard
                  (and (state-p state)
                       (symbolp channel)
                       (open-input-channel-p channel
                                             :object state))))
  ;; Has an "under the hood" implementation.
  (read-object channel state))

; Someday it would be good to call COMPACT-READ and
; COMPACT-PRINT from within ACL2 code.  This is currently
; obscure because in ACL2, READ-OBJECT does not yet have a
; serious specification, i.e., the user is left to browse
; his copy of the ANSI specification for Common Lisp to
; figure out what the Lisp reader does.  A correct
; definition of something like COMPACT-READ would have to be
; based upon a not yet existent specification of
; READ-OBJECT.

; Here we provide ACL2 functions that are designed to effect
; the underlying implementation of our HONS procedure, our
; fast association list mechanism, and our function
; memoization feature.

#+(or acl2-loop-only (not hons))
(defun init-hash-tables ()
  (declare (xargs :guard t))
  ;; A logical no-op.  Reduces to zero the number of honses.  Calls
  ;; INIT-HONS-ACONS-TABLE.
  nil)

#+(or acl2-loop-only (not hons))
(defun clear-hash-tables ()
  (declare (xargs :guard t))
  ;; A logical no-op.  Reduces to a minimum the number of honses,
  ;; preserving only honses established by defhonst or in use via
  ;; hons-acons tables.
  nil)

#+(or acl2-loop-only (not hons))
(defun init-hons-acons-table ()
  (declare (xargs :guard t))
  ;; A logical no-op.  Clears the table that is used to identify an
  ;; association list with a hash table for hash-based access.
  nil)

#+(or acl2-loop-only (not hons))
(defun flush-hons-get-hash-table-link (x)
  (declare (xargs :guard t))
  ;; A logical no-op.  Breaks the link between x and a hash table in
  ;; the *hons-acons-ht*, if such a link exists, thus permitting the
  ;; garbage collection of that table.
   x)

#+(or acl2-loop-only (not hons))
(defun clear-memoize-tables ()
  (declare (xargs :guard t))
  ;; A logical no-op.  For each memoized function, clear its table of
  ;; remembered values.
  nil)

; The macros hons-let, memoize-let, memoize-on, and memoize-off have
; the utterly bizarre property that if evaluated at the top level of
; the ACL2 loop, they really, really do nothing but evaluate form, as
; their semantics suggest.  However, if they are used within an ACL2
; function and either (a) that function is executed in program mode or
; (b) that function is known to be Common Lisp compliant and is
; executed in any mode, then there may be "under the hood" effects
; that, though not changing the semantics of what ACL2 returns, may
; affect the speed and/or space utilization of the computation.

#+(or acl2-loop-only (not hons))
(defmacro hons-let (form)

; HONS-LET evaluates form.  At the beginning of that evaluation the
; only honses are those created by CLEAR-HASH-TABLES.  At the end of
; that evaluation, the previous hons structure is restored.

form)

#+(or acl2-loop-only (not hons))
(defmacro memoize-let (fn form)
  (declare (ignore fn))

; MEMOIZE-LET evaluates form.  At the beginning of that evaluation, no
; old values are remembered of calls of the symbol fn.  Afterwards,
; those old values will be restored if no stobjs have been altered,
; but all newer values are forgotten.  The symbol fn must be memoized
; before MEMOIZE-LET is called.

  form)

; The functions memoize and unmemoize have rather innocent looking
; semantics.  But under the hood, they enable and disable memoization.
; The function memoize might cause errors due to compilation problems.

(defconst *hons-primitive-fns*
  '(hons hons-equal hons-copy hons-get-fn-do-hopy
         hons-get-fn-do-not-hopy
         hons-acons hons-acons!
         clear-hash-tables init-hash-tables init-hons-acons-table
         flush-hons-get-hash-table-link clear-memoize-tables
         hons-read-object hons-shrink-alist! hons-shrink-alist plev0))

(defconst *hons-primitives* ; hons-related macros and primitive fns
  (append '(hons-let memoize-let
            hons-get
            #+hons memoize #+hons unmemoize
            memoize-on memoize-off)
          *hons-primitive-fns*))

#+(and acl2-loop-only hons)
(defmacro memoize (fn &key
                      (condition 't condition-p)
                      condition-fn hints otf-flg
                      (inline 't))

; If condition and condition-fn are both non-nil, then the intent is
; that we do exactly what we would normally do for condition except
; that we use the name condition-fn.

  ":Doc-Section Events

  turn on memoization for specified functions~/
  ~bv[]
  Example:
  (memoize 'foo)                     ; remember the values of calls of foo
  (memoize 'foo :condition t)        ; same as above
  (memoize 'foo :condition '(test x) ; memoize for args satisfying condition
  (memoize 'foo :condition 'test-fn) ; memoize for args satisfying test-fn
  (memoize 'foo :inline nil)         ; do not inline the definition of foo~/

  General Form:
  (memoize fn                         ; memoizes fn and returns fn
           :condition    condition    ; optional (default t)
           :condition-fn condition-fn ; optional
           :hints        hints        ; optional
           :otf-flg      otf-flg      ; optional
           :inline       inline       ; optional (default t)
           )
  ~ev[]
  
  where ~c[fn] evaluates to a user-defined function symbol with no
  ~il[stobj] parameters; ~c[condition] is either ~c[t] (the default)
  or ~c['t] or else evaluates to an expression whose free variables
  are among the formal parameters of ~c[fn]; ~c[condition-fn] is
  either ~c[nil] (the default) or else evaluates to a legal function
  symbol.  Further restrictions are discussed below.

  Generally ~c[fn] must evaluate to a defined function symbol whose
  ~ilc[guard]s have been verified.  However, this value can be the name of a
  macro that is associated with such a function symbol;
  ~pl[macro-aliases-table].  That associated function symbol is the one called
  ``memoized'' in the discussion below, but we make no more mention of this
  subtlety.

  It is illegal to memoize a function (by calling this macro) that is already
  memoized.  To turn off memoization, ~pl[unmemoize].

  In the most common case, ~c[memoize] takes a single argument, which evaluates
  to a function symbol.  We call this function symbol the ``memoized function''
  because ``memos'' are saved and re-used, in the following sense.  When a call
  of the memoized function is evaluated, the result is ``memoized'' by
  associating the call's arguments with that result, in a suitable table.  But
  first an attempt is made to avoid such evaluation, by doing a lookup in that
  table on the given arguments for the result, as stored for a previous call on
  those arguments.  If such a result is found, then it is returned without
  further computation.  This paragraph also applies if ~c[:condition] is
  supplied but is ~c[t] or ~c['t].

  If in addition ~c[:condition-fn] is supplied, but ~c[:condition] is not, then
  the result of evaluating ~c[:condition-fn] must be a defined function symbol
  whose ~il[guard]s have been verified and whose formal parameter list is the
  same as for the function being memoized.  Such a ``condition function'' will
  be run whenever the memoized function is called, on the same parameters, and
  the lookup or table store described above are only performed if the result
  from the condition function call is non-~c[nil].

  If however ~c[:condition] is supplied, then an attempt will be made to define
  a condition function whose ~il[guard] and formal parameters list are the same
  as those of the memoized function, and whose body is the result, ~c[r], of
  evaluating the given ~c[condition].  The name of that condition function is
  the result of evaluating ~c[:condition-fn] if supplied, else is the result of
  concatenating the string ~c[\"-MEMOIZE-CONDITION\"] to the end of the name of
  the memoized function.  The condition function will be defined with
  ~il[guard] verification turned off, but that definition will be followed
  immediately by a ~ilc[verify-guards] event; and this is where the optional
  ~c[:hints] and ~c[:otf-flg] are attached.  At evaluation time the condition
  function is used as described in the preceding paragraph; so in effect, the
  condition (~c[r], above) is evaluated, with its variables bound to the
  corresponding actuals of the memoized function call, and the memoized
  function attempts a lookup or table store if and only if the result of that
  evaluation is non-~c[nil].

  Calls of this macro generate events of the form
  ~c[(table memoize-table fn (list condition-fn inline))].  When
  successful, the returned value is of the form
  ~c[(mv nil function-symbol state)].

  When ~c[:inline] has value ~c[nil], then ~c[memoize] does not use
  the definitional body of ~c[fn] in the body of the new, memoized
  definition of ~c[fn].  Instead, ~c[memoize] lays down a call to the
  ~c[symbol-function] for ~c[fn] that was in effect prior to
  memoization.  Use value ~c[t] for ~c[:inline] to avoid memoizing
  recursive calls to ~c[fn] directly from within ~c[fn]."

  (declare (xargs :guard t))
  (cond ((and condition-fn (null condition-p))
         `(progn (table memoize-table
                        (deref-macro-name ,fn (macro-aliases world))
                        (list ,condition-fn ,inline))
                 (value-triple (deref-macro-name
                                ,fn
                                (macro-aliases (w state))))))
        ((and condition-p
              (not (eq condition t))
              (not (equal condition ''t)))
         `(make-event
           (let* ((wrld (w state))
                  (fn (deref-macro-name ,fn (macro-aliases wrld)))
                  (condition ,condition)
                  (formals
                   (and (symbolp fn) ; guard for getprop
                        (getprop fn 'formals t
                                 'current-acl2-world wrld)))
                  (condition-fn (or ,condition-fn
                                    (intern-in-package-of-symbol
                                     (concatenate
                                      'string
                                      (symbol-name fn)
                                      "-MEMOIZE-CONDITION")
                                     fn)))
                  (hints ,hints)
                  (otf-flg ,otf-flg)
                  (inline ,inline))
             (cond ((not (and
                          (symbolp fn)
                          (not (eq t formals))
                          (not (eq t (getprop
                                      fn 'stobjs-in t
                                      'current-acl2-world wrld)))
                          (not (eq t (getprop
                                      fn 'stobjs-out t
                                      'current-acl2-world wrld)))
                          (cltl-def-from-name fn nil wrld)))
                    (er hard 'memoize
                        "The symbol ~x0 is not a known function ~
                         symbol, and thus it cannot be memoized."
                        fn))
                   ((not (eq :common-lisp-compliant
                             (symbol-class fn wrld)))
                    (er hard 'memoize
                        "~x0 is not Common Lisp compliant, so is ~
                         best memoized and called from raw Lisp (but ~
                         raw Lisp should be avoiding unless you are ~
                         hacking)."
                        fn))
                   ((cdr (assoc-eq fn (table-alist 'memoize-table
                                                   wrld)))
                    (er hard 'memoize "~x0 is already memoized." fn))
                   ((not (member-eq inline '(t nil)))
                    (er hard 'memoize
                        "The value ~x0 for inline is illegal (must ~
                         be ~x1 or ~x2)."
                        inline t nil))
                   (t
                    `(progn
                       (defun ,condition-fn ,formals
                         (declare
                          (ignorable ,@formals)
                          (xargs :guard
                                 ,(getprop fn 'guard *t*
                                           'current-acl2-world wrld)
                                 :verify-guards nil))
                         ,condition)
                       (verify-guards ,condition-fn
                                      ,@(and hints `(:hints ,hints))
                                      ,@(and otf-flg
                                             `(:otf-flg ,otf-flg)))
                       (table memoize-table
                              ',fn
                              (list ',condition-fn ',inline))
                       (value-triple ',fn)))))))
        (t `(progn (table memoize-table
                          (deref-macro-name ,fn (macro-aliases world))
                          (list t ,inline))
                   (value-triple (deref-macro-name
                                  ,fn
                                  (macro-aliases (w state))))))))

#+(and (not acl2-loop-only) hons)
(defmacro memoize (fn &key
                      condition condition-fn hints otf-flg inline)

; Warnings are ignored under include-book, so this is unlikely to be
; too noisy.

  (declare (ignore condition condition-fn hints otf-flg inline))
  `(progn (when (eql *ld-level* 0)

; We are not inside the ACL2 loop (hence not in certify-book, for example).

            (let ((state *the-live-state*))
              (warning$ 'memoize nil
                        "No change for function ~x0: Memoization ~
                         requests are ignored in raw Lisp.  In raw ~
                         Lisp, use memoize-fn."
                        ',fn)))
          (value-triple nil)))

#+(and acl2-loop-only hons)
(defmacro unmemoize (fn)

  ":Doc-Section Events

  turn off memoization for specified functions~/
  ~bv[]
  Example:
  (unmemoize 'foo) ; turn off memoization of foo~/

  General Form:
  (unmemoize fn)
  ~ev[]
  where ~c[fn] evaluates to a function symbol that is currently memoized;
  ~pl[memoize].  An exception is that as with ~ilc[memoize], ~c[fn] may
  evaluate to the name of a macro that is associated with such a function
  symbol; ~pl[macro-aliases-table].

  Calls of this macro generate events of the form
  ~c[(table memoize-table fn nil)].  When successful, the returned value
  is of the form ~c[(mv nil function-symbol state)]."

  `(progn (table memoize-table
                 (deref-macro-name ,fn (macro-aliases world)) nil)
          (value-triple
           (deref-macro-name ,fn (macro-aliases (w state))))))

#+(and (not acl2-loop-only) hons)
(defmacro unmemoize (fn)

; Warnings are ignored under include-book, so this is unlikely to be
; too noisy.

  `(progn (warning$ 'unmemoize nil
                    "No change for function ~x0: Unmemoization ~
                     requests are ignored in raw Lisp.  In raw Lisp, ~
                     use unmemoize-fn."
                    ',fn)
          nil))

#+(or acl2-loop-only (not hons))
(defun memoize-on (fn x)

; MEMOIZE-ON evaluates x.  During the evaluation the symbol fn has as
; its symbol-function what it had immediately AFTER the memoization of
; fn.  Hence, the values of calls of fn may be remembered during the
; evaluation and later.  Warning: to use MEMOIZE-ON, fn must already
; be memoized.

  (declare (ignore fn)
           (xargs :guard t))
  x)

#+(or acl2-loop-only (not hons))
(defun memoize-off (fn x)

; MEMOIZE-OFF evaluates x.  During the evaluation the symbol fn has as
; its symbol-function what it had immediately BEFORE the memoization
; of fn.  Hence the values of calls of fn may not be remembered during
; the evaluation.  Warning: to use MEMOIZE-OFF, fn must already be
; memoized."

  (declare (ignore fn)
           (xargs :guard t))
  x)

#+hons
(defmacro memoizedp (fn)
  (let ((form `(cdr (assoc-eq ,fn (table-alist 'memoize-table
                                               (w state))))))
    #-hons
    `(prog2$
      (er hard 'memoizedp
          "Memoizedp cannot be called in this ACL2 image, as it ~
           requires a hons-aware ACL2.")
      ,form)
    #+hons form))

;;; hons-shrink-alist

; HONS-SHRINK-ALIST, when called with an atomic second
; argument, produces an alist that is alist-equivalent
; to the first argument, but with all irrelevant entries in
; the first argument deleted.  Informal remark: the alist
; returned is a hons when the initial ANS is not an atom.

#+(or acl2-loop-only (not hons))
(defun hons-shrink-alist! (alst ans)
  (declare (xargs :guard t))
  (cond
   ((atom alst) ans)
   ((not (consp (car alst))) nil)
   (t (let ((p (hons-get (car (car alst)) ans)))
        (cond (p (hons-shrink-alist! (cdr alst) ans))
              (t (hons-shrink-alist! (cdr alst)
                                     (hons-acons! (car (car alst))
                                                  (cdr (car alst))
                                                  ans))))))))

#+(or acl2-loop-only (not hons))
(defun hons-shrink-alist (alst ans)
  (declare (xargs :guard t))
  (cond ((atom alst) ans)
        ((not (consp (car alst))) nil)
        (t (let ((p (hons-get (car (car alst)) ans)))
             (cond (p (hons-shrink-alist (cdr alst) ans))
                   (t (hons-shrink-alist (cdr alst)
                                         (hons-acons (car (car alst))
                                                     (cdr (car alst))
                                                     ans))))))))

#+(or acl2-loop-only (not hons))
(defun plev0 (length level tuple)
  (declare (xargs :guard t) (ignore length level tuple))
  nil)

#+(or acl2-loop-only (not hons))
(defun honsp-check (x)

; Logically, this function is (or (consp x) (stringp x)).  However, it
; causes an error if x a cons but not a hons, or if x is a string that
; isn't hashed.

  (declare (xargs :guard t))
  ;; Has an "under the hood" implementation.
  (or (consp x) (stringp x)))

(table persistent-hons-table nil nil
       :guard

; For technical reasons, namely easing support for defhonst, we allow
; nil as a key.  Clear-hash-tables thus needs to ignore the case that
; the key is a hons.

       (or (null key)
           (honsp-check key)))

; Should we also put above "the cut" basic things like HIST* that
; should perhaps be mentioned in the documentation but do not require
; under the hood implementations?

; *** NOTE:  End of functions requiring an "under the hood"
;            implementation!  Which functions are defined past this
;            point is somewhat a matter of style and taste since they
;            could all be defined in ordinary ACL2.  Some of the
;            following functions and macros, such as HIST and HPPEND,
;            are natural hons'ing equivalents of their ACL2/Common
;            Lisp versions.  On the other hand NUMBER-SUBTREES
;            illustrates the use of HONS-GET in the somewhat efficient
;            calculation of the number of subtrees of a Lisp tree.

; See hons.lisp.

; For some additional helper functions and lemmas, see the file
; once named "hons-help.lisp".

; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; HONS Extension, End.                 Edited by Hunt

; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

(deftheory definition-minimal-theory
  (definition-runes
    *definition-minimal-theory*
    nil
    world))

(deftheory executable-counterpart-minimal-theory
  (definition-runes
    *built-in-executable-counterparts*
    t
    world))

(deftheory minimal-theory

; Warning: The resulting value must be a runic-theoryp.  See
; theory-fn-callp.

; Keep this definition in sync with translate-in-theory-hint.

  (union-theories (theory 'definition-minimal-theory)
                  (union-theories

; Without the :executable-counterpart of force, the use of (theory
; 'minimal-theory) will produce the warning "Forcing has transitioned
; from enabled to disabled", at least if forcing is enabled (as is the
; default).

                   '((:executable-counterpart force))
                   (theory 'executable-counterpart-minimal-theory)))
  :doc
  ":Doc-Section Theories

  a minimal theory to enable~/~/

  This ~ilc[theory] (~pl[theories]) enables only a few built-in functions and
  executable counterparts.  It can be useful when you want to formulate lemmas
  that rather immediately imply the theorem to be proved, by way of a ~c[:use]
  hint (~pl[hints]), for example as follows.
  ~bv[]
  :use (lemma-1 lemma-2 lemma-3)
  :in-theory (union-theories '(f1 f2) (theory 'minimal-theory))
  ~ev[]
  In this example, we expect the current goal to follow from lemmas
  ~c[lemma-1], ~c[lemma-2], and ~c[lemma-3] together with rules ~c[f1] and
  ~c[f2] and some obvious facts about built-in functions (such as the
  ~il[definition] of ~ilc[implies] and the ~c[:]~ilc[executable-counterpart] of
  ~ilc[car]).  The ~c[:]~ilc[in-theory] hint above is intended to speed up the
  proof by turning off all inessential rules.~/

  :cited-by theory-functions")

(deftheory ground-zero (current-theory :here)

; WARNING: Keep this near the end of *acl2-pass-2-files* in order for the
; ground-zero theory to be as expected.

  :doc
  ":Doc-Section Theories

  ~il[enable]d rules in the ~il[startup] theory~/

  ACL2 concludes its initialization ~c[(boot-strapping)] procedure by
  defining the theory ~c[ground-zero]; ~pl[theories].  In fact, this
  theory is just the theory defined by ~c[(current-theory :here)] at the
  conclusion of initialization; ~pl[current-theory].~/

  Note that by evaluating the event
  ~bv[]
  (in-theory (current-theory 'ground-zero))
  ~ev[]
  you can restore the current theory to its value at the time you started up
  ACL2.~/

  :cited-by theory-functions")

(deflabel

; WARNING: Put this at the end of the last file in *acl2-pass-2-files*.

  end-of-pass-2)
