; Contributed by Bob Boyer
; watch.el
(require 'autorevert)
(require 'cl)
(defvar *watch-buffers* nil)
(defvar *watch-root1* "~/%s.watch.tmp.text")
(defvar *watch-root2* "~/%s.watch.tmp.lisp")
(defvar *watch-locals*
  '(watch-line-number watch-column-number watch-eval-file-name))
(loop for v in *watch-locals* do (make-variable-buffer-local v))

(defun watch-check-colors ()
  (loop for pair in color-alist do
        (let ((d (cdr pair)))
          (unless (facep d)
            (copy-face 'default d)
            (set-face-foreground d (symbol-name d))))))

(defun watch-check-buffers ()
  (setq *watch-buffers*
        (loop for buf in *watch-buffers*
              when (and (bufferp buf) (buffer-name buf))
              collect buf)))

(defun watch (n)
  (interactive "p")
  (watch-check-colors)
  (watch-check-buffers)
  (save-excursion
    (find-file (format *watch-root1* n))
    (setq watch-line-number 0)
    (setq watch-column-number 0)
    (setq watch-eval-file-name (format *watch-root2* n))
    (pushnew (current-buffer) *watch-buffers*)
    (auto-revert-mode 1)))

(defadvice auto-revert-buffers
  (around revert-buffer-around)
  (let ((b (current-buffer)))
    (loop for buf in *watch-buffers* do
          (cond ((and (bufferp buf) (buffer-name buf))
                 (set-buffer buf)
                 (let ((p (point)))
                   (beginning-of-line 1)
                   (setq watch-line-number
                         (count-lines (point-min) (point)))
                   (setq watch-column-number (- p (point)))
                   (goto-char p)))))
    (set-buffer b))
  ad-do-it
  (let ((b (current-buffer)))
    (loop for buf in *watch-buffers* do
          (cond ((and (bufferp buf) (buffer-name buf))
                 (set-buffer buf)
                 (color-buffer)
                 (goto-line (1+ watch-line-number))
                 (forward-char watch-column-number))))
    (set-buffer b)))

(ad-activate 'auto-revert-buffers)

; Unless one is running multiple OpenMCLs, it might not matter much to
; which listener a form is sent for evaluation.  However, to send a
; form to a specific listener, rather than to the first one on
; *watch-buffers*, one may set the Emacs variable watch-eval-file-name
; in the current buffer, i.e., the buffer where the form is now, to
; the value it has in the watch buffer of that listener.
; watch-eval-file-name is local to each buffer.

(defun watch-eval-input-file-name ()
  (watch-check-buffers)
  (cond
   (watch-eval-file-name)
   ((null *watch-buffers*)
    (error "No watch buffers.  Try M-X watch first."))
   (t (let ((b (car *watch-buffers*))
            (x (current-buffer)))
        (set-buffer b)
        (prog1 watch-eval-file-name
          (set-buffer x))))))

(defun my-lisp-eval-defun ()
  (interactive)
  (save-excursion
    (end-of-line)
    (beginning-of-defun)
    (let ((beg (point)))
      (forward-sexp)
      (let ((str (buffer-substring beg (point))))
        (save-excursion
          (find-file (watch-eval-input-file-name))
          (widen)
          (delete-region (point-min) (point-max))
          (insert str)
          (save-buffer 0)
          (kill-buffer (current-buffer)))))))

(defun my-acl2-eval-defun ()
  (interactive)
  (save-excursion
    (end-of-line)
    (beginning-of-defun)
    (let ((beg (point)))
      (forward-sexp)
      (let ((str (buffer-substring beg (point))))
        (save-excursion
          (find-file (watch-eval-input-file-name))
          (widen)
          (delete-region (point-min) (point-max))
          (insert "(ld '(")
          (insert str)
          (insert "))")
          (save-buffer 0)
          (kill-buffer (current-buffer)))))))

(defun color-buffer ()
  (interactive)
  (font-lock-mode 0)
  (loop for pair in color-alist do
        (goto-char (point-min))
        (loop while (search-forward (car pair) nil t)
              do
              (beginning-of-line 1)
              (let ((p (point)))
                (end-of-line 1)
                (put-text-property p (point) 'face (cdr pair)))))
  (set-buffer-modified-p nil))

(defvar color-alist '(
    ("\n(" . blue)
    ("hons" . blue)
    ("count" . blue)
    ("size"  . blue)
    ("-ht" . blue)
    ("table" . blue)
    ("mht" .  dark\ green)
    ("calls from" . dark\ green)
    ("pons" . dark\ green)
    ("memoize" . dark\ green)
    ("time" . red)
    ("seconds" . red)
    ("\n " . default)
    ))

(cond ((not (boundp 'ctl-t-keymap))
       (setq ctl-t-keymap (make-sparse-keymap))
       (define-key (current-global-map) "\C-T" ctl-t-keymap)))

(define-key ctl-t-keymap "2"    'my-acl2-eval-defun)
(define-key ctl-t-keymap "3"    'my-lisp-eval-defun)
(define-key ctl-t-keymap "\C-w" 'watch)
