;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; collect.lisp
;;;
;;; This book contains the rules used to collect like terms,
;;; after things have been prepared by the bind-free rules.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(local (include-book "../pass1/top"))

(local (include-book "basic-helper"))

(local
 (defthm hack516
     (equal (EXPT X (- M))
            (/ (EXPT X M)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; These next two sections of rules do the actual work of combining
;;; "like" terms for the rules in normalize.

(defthm |(+ c (+ d x))|
  (implies (and (syntaxp (quotep c))
		(syntaxp (quotep d)))
	   (equal (+ c (+ d x))
		  (+ (+ c d) x))))

(defun collect-+ (x y)
  (declare (xargs :guard (and (acl2-numberp x)
                              (acl2-numberp y))))
  (+ x y))

(defthm collect-+-problem-finder
    (implies (equal x x)
             (equal (collect-+ x y)
                    (+ x y))))

(in-theory (disable collect-+-problem-finder))

(theory-invariant (not (member-equal '(:rewrite collect-+-problem-finder)
                                     theory))
                  :error nil)

(defthm |(+ x x)|
    (equal (collect-+ x x)
           (* 2 x)))

(defthm |(+ x (- x))|
    (equal (collect-+ x (- x))
           0))

(defthm |(+ x (* c x))|
    (implies (syntaxp (quotep c))
             (equal (collect-+ x (* c x))
                    (* (+ c 1) x))))


(defthm |(+ (- x) (* c x))|
    (implies (syntaxp (quotep c))
             (equal (collect-+ (- x) (* c x))
                    (* (- c 1) x))))

(defthm |(+ (* c x) (* d x))|
    (implies (and (syntaxp (quotep c))
                  (syntaxp (quotep d)))
             (equal (collect-+ (* c x) (* d x))
                    (* (+ c d) x))))

(defthm |(collect-+ y x)|
    (equal (collect-+ y x)
           (collect-+ x y)))

(theory-invariant (or (not (member-equal '(:rewrite collect-+)
                                         theory))
                      (and (member-equal '(:rewrite |(collect-+ y x)|)
                                         theory)
                           (member-equal '(:rewrite |(+ x x)|)
                                         theory)
                           (member-equal '(:rewrite |(+ x (- x))|)
                                         theory)
                           (member-equal '(:rewrite |(+ x (* c x))|)
                                         theory)
                           (member-equal '(:rewrite |(+ (- x) (* c x))|)
                                         theory)
                           (member-equal '(:rewrite |(+ (* c x) (* d x))|)
                                         theory)))
                  :error nil)

(in-theory (disable collect-+))

(theory-invariant (not (member-equal '(:definition collect-+)
                                     theory))
                  :error nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defthm |(* c (* d x))|
  (implies (and (syntaxp (quotep c))
		(syntaxp (quotep d)))
	   (equal (* c (* d x))
		  (* (* c d) x))))

(defun collect-* (x y)
  (declare (xargs :guard (and (acl2-numberp x)
                              (acl2-numberp y))))
  (* x y))

(defthm collect-*-problem-finder
    (implies (equal x x)
             (equal (collect-* x y)
                    (* x y))))

(in-theory (disable collect-*-problem-finder))

(theory-invariant (not (member-equal '(:rewrite collect-*-problem-finder)
                                     theory))
                  :error nil)

(defthm |(* (expt x n) (expt y n))|
    (implies (integerp n)
             (equal (collect-* (expt x n) (expt y n))
                    (expt (* x y) n))))

(defthm |(* x x)|
    (equal (collect-* x x)
           (expt x 2)))

(defthm |(* x (/ x))|
    (equal (collect-* x (/ x))
           (if (equal (fix x) 0)
               0
             1)))

(defthm |(* x (expt x n))|
    (implies (integerp n)
             (equal (collect-* x (expt x n))
                    (if (equal (fix x) 0)
                        0
                      (expt x (+ n 1))))))

;; RBK: !!! Missing (* (- x) (expt x n))

(defthm |(* x (expt (- x) n))|
    (implies (integerp n)
             (equal (collect-* x (expt (- x) n))
                    (cond ((equal (fix x) 0)
                           0)
                          ((evenp n)
                           (expt x (+ n 1)))
                          (t
                           (- (expt x (+ n 1))))))))

(defthm |(* x (/ (expt x n)))|
    (implies (integerp n)
             (equal (collect-* x (/ (expt x n)))
                    (if (equal (fix x) 0)
                        0
                      (/ (expt x (- n 1)))))))

(defthm |(* x (/ (expt (- x) n)))|
    (implies (integerp n)
             (equal (collect-* x (/ (expt (- x) n)))
                    (cond ((equal (fix x) 0)
                           0)
                          ((evenp n)
                           (/ (expt x (- n 1))))
                          (t
                           (- (/ (expt x (- n 1)))))))))

(defthm |(* (/ x) (expt x n))|
    (implies (integerp n)
             (equal (collect-* (/ x) (expt x n))
                    (if (equal (fix x) 0)
                        0
                      (expt x (- n 1))))))

(defthm |(* (/ x) (expt (- x) n))|
    (implies (integerp n)
             (equal (collect-* (/ x) (expt (- x) n))
                    (cond ((equal (fix x) 0)
                           0)
                          ((evenp n)
                           (expt x (- n 1)))
                          (t
                           (- (expt x (- n 1))))))))

(defthm |(* (expt x m) (expt x n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt x m) (expt x n))
                    (if (and (equal (fix x) 0)
                             (not (equal m 0))
                             (not (equal n 0)))
                        0
                      (expt x (+ m n))))))

(defthm |(* (expt (- x) m) (expt x n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt (- x) m) (expt x n))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp m)
                           (expt x (+ m n)))
                          (t
                           (- (expt x (+ m n))))))))

(defthm |(* (expt x m) (expt (- x) n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt x m) (expt (- x) n))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp n)
                           (expt x (+ m n)))
                          (t
                           (- (expt x (+ m n))))))))

(defthm |(* (/ (expt x m)) (expt x n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (/ (expt x m)) (expt x n))
                    (if (and (equal (fix x) 0)
                             (not (equal m 0))
                             (not (equal n 0)))
                        0
                      (expt x (- n m))))))

(defthm |(* (/ (expt (- x) m)) (expt x n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (/ (expt (- x) m)) (expt x n))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp m)
                           (expt x (- n m)))
                          (t
                           (- (expt x (- n m))))))))

(defthm |(* (/ (expt x m)) (expt (- x) n))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (/ (expt x m)) (expt (- x) n))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp n)
                           (expt x (- n m)))
                          (t
                           (- (expt x (- n m))))))))

(defthm |(* (expt x m) (/ (expt x n)))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt x m) (/ (expt x n)))
                    (if (and (equal (fix x) 0)
                             (not (equal m 0))
                             (not (equal n 0)))
                        0
                      (expt x (- m n))))))

(defthm |(* (expt (- x) m) (/ (expt x n)))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt (- x) m) (/ (expt x n)))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp m)
                           (expt x (- m n)))
                          (t
                           (- (expt x (- m n))))))))

(defthm |(* (expt x m) (/ (expt (- x) n)))|
    (implies (and (integerp m)
                  (integerp n))
             (equal (collect-* (expt x m) (/ (expt (- x) n)))
                    (cond ((and (equal (fix x) 0)
                                (not (equal m 0))
                                (not (equal n 0)))
                           0)
                          ((evenp n)
                           (expt x (- m n)))
                          (t
                           (- (expt x (- m n))))))))

(defthm |(* (expt c n) (expt d n))|
    (implies (and (integerp n)
                  (syntaxp (quotep c))
                  (syntaxp (quotep d)))
             (equal (collect-* (expt c n) (expt d n))
                    (expt (* c d) n))))

(defthm |(collect-* y x)|
    (equal (collect-* y x)
           (collect-* x y)))

(theory-invariant (or (not (member-equal '(:rewrite collect-*)
                                         theory))
                      (and (member-equal '(:rewrite |(collect-* y x)|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt x n) (expt y n))|)
                                         theory)
                           (member-equal '(:rewrite |(* x x)|)
                                         theory)
                           (member-equal '(:rewrite |(* x (/ x))|)
                                         theory)
                           (member-equal '(:rewrite |(* x (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* x (expt (- x) n))|)
                                         theory)
                           (member-equal '(:rewrite |(* x (/ (expt x n)))|)
                                         theory)
                           (member-equal '(:rewrite |(* x (/ (expt (- x) n)))|)
                                         theory)
                           (member-equal '(:rewrite |(* (/ x) (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (/ x) (expt (- x) n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt x m) (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt (- x) m) (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt x m) (expt (- x) n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (/ (expt x m)) (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (/ (expt (- x) m)) (expt x n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (/ (expt x m)) (expt (- x) n))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt x m) (/ (expt x n)))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt (- x) m) (/ (expt x n)))|)
                                         theory)
                           (member-equal '(:rewrite |(* (expt x m) (/ (expt (- x) n)))|)
                                         theory)))
                  :error nil)

(in-theory (disable collect-*))

(theory-invariant (not (member-equal '(:definition collect-*)
                                     theory))
                  :error nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This set of commutative rules puts terms into a proper form
;;; for the above two sets of rules to fire.  They are needed by
;;; the rules in normaize.

(local
 (in-theory (enable collect-+ collect-*)))

(defun bubble-down (x match)
  (declare (xargs :guard t))
  (declare (ignore match))
  x)

(defthm bubble-down-+-problem-finder
    (implies (equal x x)
             (equal (+ (bubble-down x match) y)
                    (+ x y))))

(in-theory (disable bubble-down-+-problem-finder))

(theory-invariant (not (member-equal '(:rewrite bubble-down-+-problem-finder)
                                     theory))
                  :error nil)

(defthm bubble-down-+-bubble-down
    (equal (+ (bubble-down x match) y z)
           (+ y (bubble-down x match) z)))

(defthm bubble-down-+-match-1
    (implies (syntaxp (equal match y))
             (equal (+ (bubble-down x match) y)
                    (collect-+ x y))))

(defthm bubble-down-+-match-2
    (implies (syntaxp (equal match y))
             (equal (+ y (bubble-down x match))
                    (collect-+ x y))))

(defthm bubble-down-+-match-3
    (implies (syntaxp (equal match y))
             (equal (+ (bubble-down x match) y z)
                    (+ (collect-+ x y) z))))

(theory-invariant (or (not (member-equal '(:rewrite bubble-down)
                                         theory))
                      (and (member-equal '(:rewrite bubble-down-+-bubble-down)
                                         theory)
                           (member-equal '(:rewrite bubble-down-+-match-1)
                                         theory)
                           (member-equal '(:rewrite bubble-down-+-match-2)
                                         theory)
                           (member-equal '(:rewrite bubble-down-+-match-3)
                                         theory)))
                  :error nil)

(defthm bubble-down-*-problem-finder
    (implies (equal x x)
             (equal (* (bubble-down x match) y)
                    (* x y))))

(in-theory (disable bubble-down-*-problem-finder))

(theory-invariant (not (member-equal '(:rewrite bubble-down-*-problem-finder)
                                     theory))
                  :error nil)

(defthm bubble-down-*-bubble-down
    (equal (* (bubble-down x match) y z)
           (* y (bubble-down x match) z)))

(defthm bubble-down-*-match-1
    (implies (syntaxp (equal match y))
             (equal (* (bubble-down x match) y)
                    (collect-* x y))))

(defthm bubble-down-*-match-2
    (implies (syntaxp (equal match y))
             (equal (* y (bubble-down x match))
                    (collect-* x y))))

(defthm bubble-down-*-match-3
    (implies (syntaxp (equal match y))
             (equal (* (bubble-down x match) y z)
                    (* (collect-* x y) z))))

(theory-invariant (or (not (member-equal '(:rewrite bubble-down)
                                         theory))
                      (and (member-equal '(:rewrite bubble-down-*-bubble-down)
                                         theory)
                           (member-equal '(:rewrite bubble-down-*-match-1)
                                         theory)
                           (member-equal '(:rewrite bubble-down-*-match-2)
                                         theory)
                           (member-equal '(:rewrite bubble-down-*-match-3)
                                         theory)))
                  :error nil)

(in-theory (disable bubble-down (:executable-counterpart bubble-down)))

(theory-invariant 
 (and (not (member-equal '(:rewrite bubble-down)
                         theory))
      (not (member-equal '(:executable-counterpart bubble-down)
                         theory)))
 :error nil)
