!{\src2tex{textfont=tt}}
!!****f* ABINIT/defs_pawps
!! NAME
!! defs_pawps
!!
!! FUNCTION
!! This module contains the following structured data types:
!!   pawps_type   : Specific paw variables
!!   pshead_type  : Abinit psp file header variables
!!   pawdata_type : Temporary paw data
!!   usdata_type  : Various data inherited from uspp code
!!   pawarray_type: Various arrays used in paw atomic data generation
!!
!! COPYRIGHT
!! Copyright (C) 2005 ABINIT group (FJ,MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! NOTES
!!
!! The paw wavefunctions tphi and phi, and the projectors tprojector radial
!! parts are defined as: f(r_vector)=f(r)/r* Ylm(r_vector)
!!
!! SOURCE

 module defs_pawps

 use defs_basis

 implicit none


!Structures
!**********************************************************************

 type pawps_type

! ===== Real (double precision) arrays

  real(dp), pointer :: phi(:,:)
   ! phi(wav_meshsz,basis_size)
   ! Gives, on the radial grid, the atomic partial waves

  real(dp), pointer :: tphi(:,:)
   ! tphi(wav_meshsz,basis_size)
   ! Gives, on the radial grid, the atomic pseudo partial waves

  real(dp), pointer :: tproj(:,:)
   ! tproj(prj_msz_max,basis_size)
   ! Gives, on the radial grid, the projectors on partial waves

  real(dp), pointer :: coredens(:)
   ! coredens(core_meshsz)
   ! Gives the core density of the atom

  real(dp), pointer :: tcoredens(:)
   ! tcoredens(core_meshsz)
   ! Gives the pseudo core density of the atom

  real(dp), pointer :: dij0(:)
   ! dij0(lmn2_size)
   ! Part of the Dij term (non-local operator) completely
   !  calculated in the pseudopotential part

  real(dp), pointer :: rhoij0(:)
   ! rhoij0(lmn2_size)
   ! Atomic initialization of rhoij

  real(dp), pointer :: vhtnzc(:)
   ! vhtnzc(rho_meshsz)
   ! Gives the Hartree potential of the pseudo density
   !  of the nucleus + core electrons of the atom

 end type pawps_type

! =====================================================================

 type pshead_type

! ===== Integer scalars

  integer :: basis_size
   ! Number of elements for the paw nl basis

  integer :: core_meshsz
   ! Dimension of radial mesh for core and tcore densities

  integer :: creatorid
   ! ID of psp generator (here creatorID=1 !)

  integer :: l_size
   ! Maximum value of l+1 leading to a non zero Gaunt coefficient

  integer :: lambda
   ! Lambda in gaussian type g(r)

  integer :: lmax
   ! Maximum value of l 

  integer :: lmn_size
   ! Number of elements for the paw basis

  integer :: lmn2_size
   ! lmn2_size=lmn_size*(lmn_size+1)/2

  integer :: mesh_type
   ! Flag: 1 if all meshes are regular grids 
   !       2 if all meshes are log. grids

  integer :: prj_meshsz
   ! Dimension of radial mesh for tproj

  integer :: prj_msz_max
   ! Maximal dimension of radial mesh for tproj

  integer :: pspcod
   ! Psp code number for Abinit (here PAW->pspcod=7 !)

  integer :: pspxc_abinit
   ! Abinit s code number for the exchange-correlation functionnal

  integer :: rho_meshsz
   ! Mesh size for densities inherited from uspp

  integer :: shape_type
   ! Shape function type
   ! shape_type=-1; g(r)=tabulated
   ! shape_type=1 ; g(r)=sin
   ! shape_type=2 ; g(r)=exp

  integer :: sph_meshsz
   ! Dimension of radial mesh for spheres

  integer :: vloc_meshsz
   ! Dimension of radial mesh for vloc=vhnzc
   ! For r>r(vloc_meshsz), we assume vloz=-Z/r

  integer :: wav_meshsz
   ! Mesh size for partial waves and projectors inherited from uspp

! ===== Integer arrays

   integer, pointer :: orbitals(:)
   ! orbitals(basis_size)
   ! Gives the quantum number l for each basis function

! ===== Real (double precision) scalars

  real(dp) :: atomic_charge
   ! Total atomic charge

  real(dp) :: log_step
   ! Logarithimic step corresponding to radial mesh

  real(dp) :: rad_step
   ! Radial step corresponding to radial mesh

  real(dp) :: rc_sph
   ! Default PAW sphere radius

  real(dp) :: rcomp
   ! Radius for compensation charge

  real(dp) :: sigma
   ! Sigma for gaussian type g(r)

  real(dp) :: valence_charge
   ! Valence charge

! ===== Strings

  character*(fnlen) :: title
   ! Title for pseudopotential

 end type pshead_type

! =====================================================================

 type pawdata_type

! ===== Integer scalars

  integer :: big_meshsz
   ! Max. dimension of radial mesh

! ===== Integer arrays

  integer, pointer :: indlmn(:,:)
   ! indlmn(6,lmn_size)
   ! Gives l,m,n,ln,lm,s for i=lmn

! ===== Real (double precision)

  real(dp) :: factzero
   ! Factor to compute value of functions at r=0

! ===== Real (double precision) arrays

  real(dp), pointer :: rad(:)
   ! rad(big_meshsz)
   ! Coordinates of all the points of the radial grid

  real(dp), pointer :: rad_(:)
   ! rad_(big_meshsz)
   ! rad_(:)= rad(:) + AA where rad(i)=AA*(exp[BB*(i-1)]-1)


 end type pawdata_type

! =====================================================================

 type usdata_type

! ===== Integer scalars

  integer :: nval
   ! Number of valence states

! ===== Real (double precision) scalars

  real(dp) :: exfact
   ! USpp s code number for the exchange-correlation functionnal

! ===== Integer arrays

   integer, pointer :: lval(:)
   ! lval(nval)
   ! l angular momentum for each valence state

   integer, pointer :: refkeys(:)
   ! refkeys(basis_size)
   ! Key for identifying each reference energie (vs eigenstates)

! ===== Real (double precision) arrays

  real(dp), pointer :: occ(:)
   ! occ(nval)
   ! Valence states occupancies inherited from USPP

  real(dp), pointer :: snl(:,:)
   ! snl(wav_meshsz,nval)
   ! Radial part of atomic eigenfunction for each valence state

  real(dp), pointer :: sss(:,:)
   ! sss(basis_size,basis_size)
   ! sss(i,j)=<chi(i)|chi(j)> inherited from uspp code

 end type usdata_type

! =====================================================================

 type pawarray_type

! ===== Real (double precision) arrays

  real(dp), pointer :: coreden4pr2(:)
   ! coreden4pr2(rho_meshsz)
   ! Gives the core density multiplied by 4Pi.r2 (read from uspp)

  real(dp), pointer :: gnorm(:)
   ! gnorm(l_size)
   ! Gives the normalisation of qijl for each l

  real(dp), pointer :: hatden4pr2(:)
   ! hatden4pr2(sph_meshsz)
   ! Compensation charge density multiplied by 4Pi.r2

  real(dp), pointer :: kij(:)
   ! kij(lmn2_size)
   ! Kinetic overlap operator

  real(dp), pointer :: occ_(:,:)
   ! occ_(sph_meshsz,basis_size)
   ! Partial waves occupancies deduced from densities considerations

  real(dp), pointer :: qij0(:)
   ! qij0(lmn2_size)
   ! q_ij0 terms

  real(dp), pointer :: q00hat(:)
   ! q00hat(sph_meshsz)
   ! Qij_hat terms with i,j=(0,0)

  real(dp), pointer :: rvloc(:)
   ! rvloc(rho_meshsz)
   ! Gives the effective all electron atomic potential*r

  real(dp), pointer :: shapefunc(:)
   ! shapefunc(sph_meshsz)
   ! Gives the normalized shape function of the atom used for
   ! the compensation density

  real(dp), pointer :: tcoreden4pr2(:)
   ! tcoreden4pr2(rho_meshsz)
   ! Gives the pseudo core density multiplied by 4Pi.r2 (read from uspp)

  real(dp), pointer :: tvaleden4pr2(:)
   ! valeden4pr2(rho_meshsz)
   ! Gives the pseudo valence density multiplied by 4Pi.r2 (read from uspp)

  real(dp), pointer :: valeden4pr2(:)
   ! valeden4pr2(rho_meshsz)
   ! Gives the valence density multiplied by 4Pi.r2 (read from uspp)

  real(dp), pointer :: vhnzcij(:)
   ! vhnzcij(lmn2_size)
   ! (i,j) channel of the hartree term inside
   ! the spheres for the density nZc

  real(dp), pointer :: vhnzc(:)
   ! vhnzc(rho_meshsz)
   ! Gives the Hartree potential of the density
   !  of the nucleus + core electrons of the atom 

  real(dp), pointer :: vhtnzcij(:)
   ! vhtnzcij(lmn2_size)
   ! Gives the matrix element of the Hartree potential
   ! of the pseudodensity of the nucleus+core electrons of the atom

 end type pawarray_type

! =====================================================================

 type pawrso_type

! ===== Logical

  logical :: userso
   ! TRUE if Real Space Optimization is required

! ===== Real (double precision)

  real(dp) :: ecut
   ! Real Space Optimization parameter: plane wave cutoff = 1/2 Gmax**2

  real(dp) :: gfact
   ! Real Space Optimization parameter: Gamma/Gmax ratio

  real(dp) :: werror
   ! Real Space Optimization parameter: max. error W_l allowed

 end type pawrso_type

! =====================================================================

 type pawopt_type

! ===== Integer scalars

  integer :: iortho
   !  Option governing the re-orthogonalization of Phi, tPhi and tProj
   !  iortho= 0: Phi, tPhi and tProj from USPP are used
   !          1: Phi, tPhi and tProj are re-orthogonalized
   !          2: tProj are re-computed using a N. Holzwarth scheme
   !             Phi, tPhi and tProj are re-orthogonalized

  integer :: ishapenum
   !  Option governing the writing of shape functions
   !  ishapenum= 0: shape functions are written as analytic
   !             1: shape functions are written as numeric


  integer :: ivhtnzc
   !  Option governing the computation of vH(tnZC)
   !  ivhtnzc= 1: compute tnZc and Vh(tnZc) directly
   !           2: use pseudized veff from USPP to deduce vH(tnzC)

! ===== Real (double precision)

  real(dp) :: rcompfact
   ! Define radius for compensation charge (-1if rcomp=rcompfact*rcut)
   ! shape_function(r>=rcomp)=0

 end type pawopt_type

!**********************************************************************

 end module defs_pawps
!!***
