/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net;

import java.util.StringTokenizer;

public class ServerVersion {

    //--- Constant(s) ---

    public static final ServerVersion OPENNAP044 
	= new ServerVersion("opennap 0.44.0", null);

    public static final ServerVersion SLAVANAP1
	= new ServerVersion("slavanap 1.0.0", null);

    public static final ServerVersion SLAVANAP2 
	= new ServerVersion("slavanap 2.0.0", SLAVANAP1);

    public static final ServerVersion UNKNOWN = new ServerVersion("", null);

    public static final ServerVersion[] TOP_VERSIONS = {
	OPENNAP044, SLAVANAP2
    };

    //--- Data field(s) ---

    public String software;
    public int[] versions;
    public ServerVersion nextVersion;

    //--- Constructor(s) ---

    public ServerVersion(String version, ServerVersion nextVersion)
    {
	parse(version);

	this.nextVersion = nextVersion;
    }

    public ServerVersion(String version)
    {
	parse(version);

	this.nextVersion = getClosestVersion();
	if (this.nextVersion == null) {
	    // fall back to opennap 0.44
	    this.nextVersion = OPENNAP044;
	}
    }

    //--- Method(s) ---

    /**
     * Find a known version that is smaller than ours. That should be 
     * compatible for sure. 
    */
    public ServerVersion getClosestVersion()
    {
	for (int i = 0; i < TOP_VERSIONS.length; i++) {
	    ServerVersion next = TOP_VERSIONS[i];
	    if (software.equalsIgnoreCase(next.software)) {
		while (next != null) {
		    if (this.isCompatibleTo(next)) {
			return next;
		    }
		    next = next.getNextVersion();
		}
	    }
	}

	return null;
    }

    public ServerVersion getNextVersion()
    {
	return nextVersion;
    }

    public boolean isOpenNap()
    {
	return software.equalsIgnoreCase("opennap");
    }

    public boolean isSlavaNap()
    {
	return software.equalsIgnoreCase("slavanap");
    }

    public String toString() 
    {
	StringBuffer sb = new StringBuffer();
	sb.append(software);
	if (versions.length > 0) {
	    sb.append(" ");
	}
	for (int i = 0; i < versions.length; i++) {
	    sb.append(versions[i]);
	    if (i < versions.length - 1) {
		sb.append(".");
	    }
	}
	return sb.toString();
    }

    /**
     * Returns true, if this version is greater or equal than 
     * <code>v</code>.
     */
    public boolean isCompatibleTo(ServerVersion v)
    {
	for (int i = 0; i < versions.length && i < v.versions.length; i++) {
	    if (this.versions[i] > v.versions[i]) {
		return true;
	    }
	    else if (this.versions[i] < v.versions[i]) {
		return false;
	    }
	}

	return this.versions.length >= v.versions.length;
    }

    protected void parse(String version)
    {
	// separate by " " first
	StringTokenizer t = new StringTokenizer(version, " .");
	if (t.hasMoreTokens()) {
	    software = t.nextToken();
	}
	else {
	    software = "";
	}
	
	// now parse version
	versions = new int[t.countTokens()];
	if (t.hasMoreTokens()) {
	    for (int i = 0; i < versions.length; i++) {
		try {
		    versions[i] = Integer.parseInt(t.nextToken());
		}
		catch (NumberFormatException e) {
		    versions[i] = 0;
		}
	    }
	} 
    }
	
}
