/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.gui;

import xnap.gui.prefs.*;
import xnap.plugin.*;
import xnap.util.*;
import xnap.util.event.*;
import xnap.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;

public class PreferencesDialog extends DefaultDialog 
    implements PropertyChangeListener, StateListener {
    
    //--- Data field(s) ---

    private static PreferencesDialog me = null;

    private Preferences prefs = Preferences.getInstance();

    private ToggableIconPane pane;
    private JTabbedPane jtpPlugins;
    private LinkedList panels = new LinkedList();
    /**
     * Maps plugin to plugin preference panel.
     */
    private Hashtable pluginPanels = new Hashtable();

    private boolean needToRestartXNap = false;

    //--- Constructor(s) ---

    private PreferencesDialog() 
    {
	super(BUTTON_OKAY | BUTTON_APPLY | BUTTON_CANCEL);
	
	// icon panel
	pane = new ToggableIconPane(false, 0);
	initializePanels();

	// content
        setTitle(XNap.tr("Preferences"));
	getMainPanel().setLayout(new BorderLayout());
	getMainPanel().add(pane, BorderLayout.CENTER);
	pack();

	setSize(prefs.getPrefsWindowWidth(), prefs.getPrefsWindowHeight());
    }

    // --- Methods ---

    public void initializePanels()
    {
	JTabbedPane jtpGeneral 
	    = addPanel(XNap.tr("General"), XNapFrame.getSmallIcon("gohome.png"));
	addSubPanel(jtpGeneral, new PersonalPrefsPanel());
	addSubPanel(jtpGeneral, new DialogPrefsPanel());
  	addSubPanel(jtpGeneral, new PluginPrefsPanel());

	JTabbedPane jtpApperance = addPanel
	    (XNap.tr("Appearance"), XNapFrame.getSmallIcon("list.png"));
	addSubPanel(jtpApperance, new ApplicationPrefsPanel());
	addSubPanel(jtpApperance, new LookAndFeelPrefsPanel());
	addSubPanel(jtpApperance, new ChatAppearancePrefsPanel());
	addSubPanel(jtpApperance, new ConsolePrefsPanel());

	JTabbedPane jtpFiles
	    = addPanel(XNap.tr("Files"), XNapFrame.getSmallIcon("folder.png"));
	addSubPanel(jtpFiles, new FilesPrefsPanel());
	addSubPanel(jtpFiles, new ProgramsPrefsPanel());

	IPlugin[] plugins = GUIPluginManager.getInstance().getPlugins();
	for (int i = 0; i < plugins.length; i++) {
	    if (plugins[i] instanceof IGUIPlugin && plugins[i].isEnabled()) {
		addPanels((IGUIPlugin)plugins[i]);
	    }
	}
	
	JTabbedPane jtpNetwork = addPanel
	    (XNap.tr("Network"), XNapFrame.getSmallIcon("world.png"));
	addSubPanel(jtpNetwork, new NetworkPrefsPanel());
	addSubPanel(jtpNetwork, new ProxyPrefsPanel());

	JTabbedPane jtpSearch
	    = addPanel(XNap.tr("Search"), XNapFrame.getSmallIcon("find.png"));
	addSubPanel(jtpSearch, new SearchPrefsPanel());

	JTabbedPane jtpTransfer 
	    = addPanel(XNap.tr("Transfer"), 
		       XNapFrame.getSmallIcon("connect_established.png"));
	addSubPanel(jtpTransfer, new TransferPrefsPanel());
	addSubPanel(jtpTransfer, new AdvancedTransferPrefsPanel());

	JTabbedPane jtpChat 
	    = addPanel(XNap.tr("Chat"), 
		       XNapFrame.getSmallIcon("mail_generic.png"));
	addSubPanel(jtpChat, new ChatPrefsPanel());

	GUIPluginManager.getInstance().addStateListener(this);
    }

    public void addSubPanel(JTabbedPane parent, AbstractPreferencesPanel panel)
    {
	panels.add(panel);
	panel.setBorder(new EmptyBorder(10, 10, 10, 10));
	JScrollPane jsp = new JScrollPane(panel);
	parent.addTab(panel.getTitle(), jsp);
    }

    public JTabbedPane addPanel(String title, Icon icon)
    {
	JTabbedPane jtp = new JTabbedPane();
	pane.addTab(title, icon, jtp);
	return jtp;
    }

    public void addPanels(IGUIPlugin pl)
    {
	AbstractPreferencesPanel[] p = pl.getPrefsPanels();
	if (p != null && p.length > 0) {
	    JTabbedPane jtp = addPanel(pl.getName(), pl.getPreferencesIcon());
	    pluginPanels.put(pl, jtp);
	    for (int j = 0; j < p.length; j++) {
		addSubPanel(jtp, p[j]);
	    }
	}
    }

    public void apply()
    {
	// register for restart notification
	prefs.addPropertyChangeListener(this);

	// panels
	for (Iterator i = panels.iterator(); i.hasNext();) {
	    ((AbstractSaveablePanel)i.next()).apply();
	}

	prefs.removePropertyChangeListener(this);
    }

    public void close()
    {
	GUIPluginManager.getInstance().removeStateListener(this);	

	// save dialog prefs
	prefs.setPrefsWindowHeight(getBounds().getSize().height);
	prefs.setPrefsWindowWidth(getBounds().getSize().width);

	// close panels
	for (Iterator i = panels.iterator(); i.hasNext();) {
	    ((AbstractSaveablePanel)i.next()).close();
	}

	// save the preferences to file
	if (isOkay() && !prefs.write()) {
	    String msg = XNap.tr("Could not write") + prefs.getFilename() + ".";
	    JOptionPane.showMessageDialog
		(this, msg, XNap.tr("Preferences"), 
		 JOptionPane.ERROR_MESSAGE);
	    return;
	}

	if (needToRestartXNap) {
	    Dialogs.showRestartNotification(me);
	}

	dispose();
	me = null;		    
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();

	if (p.equals("lookAndFeel") || p.equals("showTextIcons")
	    || p.equals("theme") || p.equals("language")) {
	    needToRestartXNap = true;    
	}
    }

    public static void showDialog(Component c)
    {
        if (me == null) {
            me = new PreferencesDialog();
	    if (c != null) {
		me.setLocationRelativeTo(c);
	    }
        }
	me.show();
    }

    /**
     * Takes care of gui if plugin is enabled.
     */
    public void stateEnabled(StateEvent e)
    {
	if (e.getSource() instanceof IGUIPlugin) {
	    addPanels((IGUIPlugin)e.getSource());
	}
    }

    /**
     * Takes care of gui if plugin is diabled.
     */
    public void stateDisabled(StateEvent e)
    {
	if (e.getSource() instanceof IGUIPlugin) {
	    IGUIPlugin pl = (IGUIPlugin)e.getSource();

	    JTabbedPane jtp = (JTabbedPane)pluginPanels.get(pl);
	    if (jtp != null) {
		Component[] c = jtp.getComponents();
		for (int i = 0; i < c.length; i++) {
		    panels.remove(c[i]);
		}
		pane.remove(jtp);
	    }
	}
    }

}
