/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
==========================================================================

- SPLITMODELS -

==========================================================================
*/
// - Adding the View Weapon in split pieces


#include "cg_local.h"

viewweaponinfo_t	vweap;


//======================================================================
//						Predict WeaponChange
//======================================================================

static qboolean CG_UseWeapon( int newweapon, qboolean feedback )
{
#ifdef PREDICTSHOOTING
	int sent;
#endif
	gitem_t *item;

	if( cgs.demoPlaying )
		return qfalse;

	if( newweapon < WEAP_GUNBLADE || newweapon >= WEAP_TOTAL )
		return qfalse;

	if( newweapon == cg.latched_weapon ||
		(newweapon == cg.frame.playerState.stats[STAT_WEAPON_ITEM] && cg.latched_weapon == WEAP_NONE) )
		return qfalse;

	item = GS_FindItemByTag( newweapon );
	if( !item )
		return qfalse;

	//check if we have the weapon
	if( !cg.frame.playerState.weaponlist[newweapon-1][0] ) {
		if( feedback ) {
			//CG_Printf( "Out of item: %s\n", item->pickup_name );
			trap_S_StartGlobalSound( CG_MediaSfx(cgs.media.sfxWeaponUpNoAmmo), CHAN_AUTO, cg_volume_effects->value );
		}
		return qfalse;
	}

	//check we have ammo for it
	if( !cg.frame.playerState.weaponlist[newweapon-1][1] && !cg.frame.playerState.weaponlist[newweapon-1][2] &&
			newweapon != WEAP_GUNBLADE ) {
		if( feedback ) {
			//CG_Printf( "No ammo for %s\n", item->pickup_name );
			trap_S_StartGlobalSound( CG_MediaSfx( cgs.media.sfxWeaponUpNoAmmo ), CHAN_AUTO, cg_volume_effects->value );
		}
		return qfalse;
	}

	cg.last_weapon =
		(cg.latched_weapon != WEAP_NONE ? cg.latched_weapon : cg.frame.playerState.stats[STAT_WEAPON_ITEM]);
	cg.latched_weapon = newweapon;
	trap_Cmd_ExecuteText( EXEC_NOW, va("svuse %s", item->pickup_name ) );

#ifdef PREDICTSHOOTING
	// save info about the change, so we can use it for prediction
	trap_NET_GetCurrentState( NULL, NULL, &sent );
	// if we already have one with same time, we will overwrite it
	if( cg.weaponChanges[cg.weaponChangesPosition][0] != sent )
		cg.weaponChangesPosition = (cg.weaponChangesPosition + 1) % CMD_BACKUP;
	cg.weaponChanges[cg.weaponChangesPosition][0] = sent;
	cg.weaponChanges[cg.weaponChangesPosition][1] = newweapon;
#endif

	return qtrue;
}

//=================
//CG_Cmd_Use_f
//=================
void CG_Cmd_Use_f( void )
{
	gitem_t		*item;

	if( cgs.demoPlaying )
		return;

	if( cg.frame.playerState.pmove.pm_type == PM_CHASECAM ||
		cg.frame.playerState.pmove.pm_type == PM_DEAD ||
		cg.frame.playerState.pmove.pm_type == PM_SPECTATOR )
		return;

	if( trap_Cmd_Argc() < 2 )
		return;

	item = GS_FindItemByName( trap_Cmd_Args() );
	if( !item ) {
		CG_Printf( "unknown item: %s\n", trap_Cmd_Args() );
		return;
	}

	if( !(item->flags & ITFLAG_USABLE) ) {
		CG_Printf( "%s is not usable.\n", item->pickup_name );
		return;
	}

	if( item->type & IT_WEAPON ) {
		CG_UseWeapon( item->tag, qtrue );
		return;
	}

	trap_Cmd_ExecuteText( EXEC_NOW, va("svuse %s", item->pickup_name ) );
}

/*
=================
CG_WeapLast_f
=================
*/
void CG_WeapLast_f( void )
{
	if( cg.last_weapon != WEAP_NONE )
		CG_UseWeapon( cg.last_weapon, qtrue );
}

/*
=================
CG_WeapPrev_f
=================
*/
void CG_WeapPrev_f( void )
{
	int			tag;
	int			selected_weapon;

	if( cg.frame.playerState.pmove.pm_type == PM_CHASECAM ) {
		CG_ChasePrev();
		return;
	}

	if( cg.frame.playerState.pmove.pm_type == PM_DEAD )
		return;

	if( cgs.demoPlaying )
		return;

	if( cg.latched_weapon == WEAP_NONE ) {
		selected_weapon = cg.frame.playerState.stats[STAT_WEAPON_ITEM];
	} else {
		selected_weapon = cg.latched_weapon;
	}
	if( selected_weapon < WEAP_GUNBLADE || selected_weapon >= WEAP_TOTAL )
		selected_weapon = WEAP_GUNBLADE; // don't get stuck to a loop with invalid weapon data

	tag = selected_weapon;
	tag--;
	if( tag < WEAP_GUNBLADE ) tag = WEAP_TOTAL - 1;
	while( tag != selected_weapon )
	{
		if( CG_UseWeapon( tag, qfalse ) )
			return; // successful

		tag--;
		if( tag < WEAP_GUNBLADE ) 
			tag = WEAP_TOTAL - 1;
	}
}

/*
=================
CG_WeapNext_f
=================
*/
void CG_WeapNext_f( void )
{
	int			tag;
	int			selected_weapon;

	if( cg.frame.playerState.pmove.pm_type == PM_CHASECAM ) {
		CG_ChaseNext();
		return;
	}

	if( cg.frame.playerState.pmove.pm_type == PM_DEAD )
		return;

	if( cgs.demoPlaying )
		return;

	if( cg.latched_weapon == WEAP_NONE ) {
		selected_weapon = cg.frame.playerState.stats[STAT_WEAPON_ITEM];
	} else {
		selected_weapon = cg.latched_weapon;
	}
	if( selected_weapon < WEAP_GUNBLADE || selected_weapon >= WEAP_TOTAL )
		selected_weapon = WEAP_GUNBLADE; // don't get stuck to a loop with invalid weapon data

	tag = selected_weapon;
	tag++;
	if( tag >= WEAP_TOTAL ) tag = WEAP_GUNBLADE;
	while( tag != selected_weapon )
	{
		if( CG_UseWeapon( tag, qfalse ) )
			return; // successful

		tag++;
		if( tag >= WEAP_TOTAL )
			tag = WEAP_GUNBLADE;
	}
}

void CG_WeaponAutoswitch( int weapon )
{
	int i;

	assert( weapon > WEAP_NONE && weapon < WEAP_TOTAL );

	// we don't want an automatic switch?
	if( !cg_weaponAutoswitch->integer )
		return;

	// weapon autoswitch 2 only switches away from the gunblade
	if( cg_weaponAutoswitch->integer == 2 && cg.frame.playerState.stats[STAT_WEAPON_ITEM] != WEAP_GUNBLADE )
		return;

	// already had it?
	if( cg.oldFrame.playerState.weaponlist[weapon-1][0] )
		return;

	// already had a better weapon?
	// if setting was 2, we only change if we only had the gunblade
	for( i = WEAP_TOTAL - 1; i > (cg_weaponAutoswitch->integer == 2 ? WEAP_GUNBLADE : weapon); i-- ) {
		if( cg.oldFrame.playerState.weaponlist[i-1][0] )
			return;
	}

	CG_UseWeapon( weapon, qfalse );
}

//=================
//CG_NoAmmoWeaponChange
//
// Called only from the ViewWeapon events filter
//=================
void CG_NoAmmoWeaponChange( void )
{
	int				weaptag;
	int				newweapon = WEAP_GUNBLADE;

	if( cg.frame.playerState.pmove.pm_type == PM_DEAD )
		return;

	// check if we're already changing weapon
	if( cgs.demoPlaying || cg.frame.playerState.pmove.pm_type == PM_CHASECAM ) {
		if( cg.changing_weapon )
			return;
	} else {
		if( cg.latched_weapon != WEAP_NONE )
			return;
	}

	trap_S_StartGlobalSound( CG_MediaSfx(cgs.media.sfxWeaponUpNoAmmo), CHAN_AUTO, cg_volume_effects->value );

	if( cgs.demoPlaying || cg.frame.playerState.pmove.pm_type == PM_CHASECAM ) {
		cg.changing_weapon = qtrue;
		return;
	}

	// in playerstate, GUNBLADE is 0
	// we could ignore the gunblade here, but I prefer the 'if' being visible
	for( weaptag = WEAP_TOTAL-1; weaptag > WEAP_NONE; weaptag-- )
	{
		if( !cg.frame.playerState.weaponlist[weaptag-1][0] )	// has the weapon?
			continue;

		if( !cg.frame.playerState.weaponlist[weaptag-1][1] )	// has strong ammo?
			continue;

		if( weaptag != WEAP_GUNBLADE )	{		// gunblade is ignored in strong list
			newweapon = weaptag;
			goto found;
		}
	}

	for( weaptag = WEAP_TOTAL-1; weaptag > WEAP_NONE; weaptag-- )
	{
		if( !cg.frame.playerState.weaponlist[weaptag-1][0] )	// has the weapon?
			continue;

		if( weaptag != WEAP_GUNBLADE )
		{
			if( !cg.frame.playerState.weaponlist[weaptag-1][2] )	// has weak ammo?
				continue;
		}

		newweapon = weaptag;
		goto found;
	}
found:
	{
		CG_UseWeapon( newweapon, qfalse );
	}
}

void CG_CheckWeaponState( void )
{
	if( cg.frame.playerState.pmove.pm_type == PM_CHASECAM || cg.frame.playerState.pmove.pm_type == PM_DEAD )
	{
		static int last_weapon = WEAP_NONE;

		if( cg.changing_weapon && last_weapon != cg.frame.playerState.stats[STAT_WEAPON_ITEM] )
			cg.changing_weapon = qfalse;

		last_weapon = cg.frame.playerState.stats[STAT_WEAPON_ITEM];
		cg.latched_weapon = WEAP_NONE;
		cg.last_weapon = WEAP_NONE;

		return;
	}

	// check if the weapon change is complete
	if( cg.latched_weapon == cg.frame.playerState.stats[STAT_WEAPON_ITEM] )
		cg.latched_weapon = WEAP_NONE;

	// check if we don't have the weapon/ammo we're changing to anymore
	if( cg.latched_weapon != WEAP_NONE ) {
		if( !cg.frame.playerState.weaponlist[cg.latched_weapon-1][0] ||
				(!cg.frame.playerState.weaponlist[cg.latched_weapon-1][1] &&
				!cg.frame.playerState.weaponlist[cg.latched_weapon-1][2] && cg.latched_weapon != WEAP_GUNBLADE) )
			cg.latched_weapon = WEAP_NONE;
	}
}


#ifdef PREDICTSHOOTING

static void CG_Predict_Laserbeam( int commandNum, firedef_t *firedef )
{
	entity_state_t *state;
	centity_t *cent;
	unsigned int type;
	int i;

	if( firedef->fire_mode == FIRE_MODE_STRONG ) {
		type = ET_LASERBEAM;
	} else {
		type = ET_CURVELASERBEAM;
	}

	for( i = 0; i < cg.frame.numEntities; i++ )
	{
		state = &cg.frame.parsedEntities[i & (MAX_PARSE_ENTITIES-1)];
		cent = &cg_entities[state->number];
		if( cent->type == type && cent->current.ownerNum == cg.chasedNum + 1 ) {
			for( i = 0, cent = &cg_predicted_entities[0]; i < MAX_PREDICTED_EDICTS; i++, cent++ ) {
				if( cent->type == type && cent->current.ownerNum == cg.chasedNum + 1 )
					CG_Printf( "Warning: Both server and client side laser beam entities active\n" );
			}
			return; // there is already server side entity, use that
		}
	}

	for( i = 0, cent = &cg_predicted_entities[0]; i < MAX_PREDICTED_EDICTS; i++, cent++ ) {
		if( cent->type == type && cent->current.ownerNum == cg.chasedNum + 1 )
			return; // there is already predicted entity, we don't have to do anything
	}

	// no entity yet, we will create one
	cent = CG_NewPredictedEntity( commandNum );
	cent->type = cent->current.type = type;
	cent->current.ownerNum = cg.chasedNum + 1;
	if( firedef->fire_mode == FIRE_MODE_STRONG ) {
		cent->current.sound = CG_SoundIndex( S_WEAPON_LASERGUN_S_HUM );
	} else {
		//cent->current.modelindex = CG_ModelIndex( PATH_LASERBEAM_WEAK_MODEL );
		cent->current.sound = CG_SoundIndex( S_WEAPON_LASERGUN_W_HUM );
	}
	cent->current.range = firedef->timeout;
	cent->prev = cent->current;

	// muzzleflash is fired only once for LG
	CG_PlayerMuzzleFlash( cg.chasedNum+1, (firedef->fire_mode == FIRE_MODE_STRONG) );
}

void CG_Predict_Weaponstate( int *weapon, int *latched_weapon, weapon_state_t *weaponstate, int commandNum,
	const usercmd_t *ucmd, qboolean act )
{
	int actions;
	firedef_t *firedef = NULL;

	assert( *weapon >= WEAP_NONE && *weapon < WEAP_TOTAL );

	if( *weapon == WEAP_NONE )
		return;

	if( cg.frame.playerState.pmove.pm_type != PM_NORMAL )
		return;

	if( cg.frame.playerState.weaponlist[*weapon-1][1] >= gs_weaponInfos[*weapon].firedef->usage_count ) {
		firedef = gs_weaponInfos[*weapon].firedef;
	} else {
		firedef = gs_weaponInfos[*weapon].firedef_weak;
	}

	actions = GS_Weaponstate_Run( weaponstate, ucmd->msec, firedef, (ucmd->buttons & BUTTON_ATTACK) );

	if( actions & WEAPON_ACTION_WEAPON_CHANGE )
	{
		*weapon = *latched_weapon;
		*latched_weapon = -1;

		if( act ) { // simulate EV_WEAPONUP event
			int pnum;
			entity_state_t *ent;

			for( pnum = 0; pnum < cg.frame.numEntities; pnum++ ) {
				ent = &cg.frame.parsedEntities[pnum&(MAX_PARSE_ENTITIES-1)];
				if( ent->number == cg.chasedNum+1 )
					break;
			}
			if( pnum == cg.frame.numEntities ) {
				CG_Printf( "Warning: CG_Predict_Weaponstate: Player entity not found\n" );
			} else {
				CG_WeaponSwitchSound( ent, 1 );
				CG_AddPModelAnimation( cg.chasedNum+1, 0, TORSO_WEAP_UP, 0, EVENT_CHANNEL);
				cg.predictedWeaponEvent = EV_WEAPONUP;
			}
		}
	}

	if( actions & WEAPON_ACTION_FIRE )
	{
		if( cg.frame.match.state == MATCH_STATE_COUNTDOWN ) // FIXME: does this code keep consistency?
			return;
		if( cg.frame.playerState.stats[STAT_GAMETYPE] == GAMETYPE_CA && cg.frame.match.state != MATCH_STATE_WARMUP && cg.frame.match.roundstate != MATCH_STATE_PLAYTIME ) // FIXME: does this code keep consistency?
			return;

		if( firedef->ammo_id && (!firedef->usage_count ||
			cg.frame.playerState.weaponlist[*weapon-1][2-firedef->fire_mode] >= gs_weaponInfos[*weapon].firedef->usage_count) )
		{
			if( act )
			{
				if( *weapon == WEAP_LASERGUN ) {
					CG_Predict_Laserbeam( commandNum, firedef );
				} else if( CG_IsPredictableWeapon(*weapon) ) {
					CG_PlayerMuzzleFlash( cg.chasedNum+1, (firedef->fire_mode == FIRE_MODE_STRONG) );
				}
			}
		}
		else
		{
			//if( act )
			//	CG_NoAmmoWeaponChange();
		}
	}
}

qboolean CG_IsPredictableWeapon( int weapon )
{
	switch( weapon ) {
		case WEAP_RIOTGUN:
		case WEAP_GRENADELAUNCHER:
		case WEAP_ROCKETLAUNCHER:
		case WEAP_PLASMAGUN:
		case WEAP_ELECTROBOLT:
		case WEAP_LASERGUN:
			return qtrue;

		case WEAP_NONE:
		case WEAP_GUNBLADE:
		case WEAP_SHOCKWAVE:
		default:
			return qfalse;
	}
}

#endif


//======================================================================
//						ViewWeapon
//======================================================================

/*
==============
CG_CalcGunOffset
==============
*/
static void CG_CalcGunOffset( vec3_t angles )
{
	int		i;
	float	delta;

	// gun angles from bobbing
	if ( cg_gunbob->integer == 1 )
	{
		if( cg.bobCycle & 1 ) {
			angles[ROLL] -= cg.xyspeed * cg.bobFracSin * 0.012;
			angles[YAW] -= cg.xyspeed * cg.bobFracSin * 0.006;
		} else {
			angles[ROLL] += cg.xyspeed * cg.bobFracSin * 0.012;
			angles[YAW] += cg.xyspeed * cg.bobFracSin * 0.006;
		}
		angles[PITCH] += cg.xyspeed * cg.bobFracSin * 0.012;
	}

	// gun angles from delta movement
	for( i = 0; i < 3; i++ ) {
		delta = ( cg.player.oldps->viewangles[i] - cg.player.curps->viewangles[i] ) * cg.lerpfrac;
		if( delta > 180 )
			delta -= 360;
		if( delta < -180 )
			delta += 360;
		clamp( delta, -45, 45 );

		
		if( i == YAW )
			angles[ROLL] += 0.001 * delta;
		angles[i] += 0.002 * delta;
	}

	// gun angles from kicks
	if( !cg_damage_kick->integer )
		CG_AddKickAngles( angles );
}

/*
==============
CG_vWeapStartFallKickEff
==============
*/
static void CG_vWeapStartFallKickEff( int parms )
{
	int	bouncetime;

	bouncetime = ((parms + 1)*50)+150;
	vweap.fallEff_Time = cg.time + 2*bouncetime;
	vweap.fallEff_rebTime = cg.time + bouncetime;
}

/*
===============
CG_vWeapSetPosition 

  Custom gun position
===============
*/
static void CG_vWeapGetPosition( vec3_t origin, vec3_t axis[3] )
{
	vec3_t			gun_angles;
	vec3_t			forward, right, up;
	float			gunx, guny, gunz;

	// set up gun position
	VectorCopy( cg.refdef.vieworg, origin );
	VectorCopy( cg.refdef.viewangles, gun_angles );

	//offset by client cvars
	gunx = cg_gunx->value;
	guny = cg_guny->value;
	gunz = cg_gunz->value;

	//move hand to the left/center
	if( cgs.demoPlaying && !cg_demo_truePOV->integer ) {
		if( hand->integer == 0 )
			gunx += cg_handOffset->value;
		else if( hand->integer == 1 )
			gunx -= cg_handOffset->value;
	} else {
		if( cgs.clientInfo[cg.chasedNum].hand == 0 )
			gunx += cg_handOffset->value;
		else if( cgs.clientInfo[cg.chasedNum].hand == 1 )
			gunx -= cg_handOffset->value;
	}

	//Add fallkick effect
	if( vweap.fallEff_Time > cg.time )
		vweap.fallKick += (vweap.fallEff_rebTime*0.001f - cg.time*0.001f);
	else
		vweap.fallKick = 0;

	guny -= vweap.fallKick;

	//Move the gun
	AngleVectors( gun_angles, forward, right, up );
	VectorMA( origin, gunx, right, origin );
	VectorMA( origin, gunz, forward, origin );
	VectorMA( origin, guny, up, origin );

	//add bobbing
	CG_CalcGunOffset( gun_angles );
	AnglesToAxis( gun_angles, axis );

	if( cg_gun_fov->integer && !(cg.frame.playerState.stats[STAT_LAYOUTS] & STAT_LAYOUT_ZOOM) ) {
		VectorScale( axis[0], cg.view_fracWeapFOV, axis[0] ); 
	}
}

/*
===============
CG_vWeapUpdateProjectionSource
===============
*/
static void CG_vWeapUpdateProjectionSource( vec3_t hand_origin, vec3_t hand_axis[3], vec3_t weap_origin, vec3_t weap_axis[3] )
{
	orientation_t *tag_result = &vweap.projectionSource;
	orientation_t	tag_weapon;

	VectorCopy( vec3_origin, tag_weapon.origin );
	Matrix_Copy( axis_identity, tag_weapon.axis );

	// move to tag_weapon
	CG_MoveToTag( tag_weapon.origin, tag_weapon.axis,
		hand_origin, hand_axis,
		weap_origin, weap_axis );
	
	// move to projectionSource tag
	if( vweap.pweapon.weaponInfo ) {
		VectorCopy( vec3_origin, tag_result->origin );
		Matrix_Copy( axis_identity, tag_result->axis );
		CG_MoveToTag( tag_result->origin, tag_result->axis,
			tag_weapon.origin, tag_weapon.axis,
			vweap.pweapon.weaponInfo->tag_projectionsource.origin, vweap.pweapon.weaponInfo->tag_projectionsource.axis );
		return;
	}

	// fallback: copy gun origin and move it front by 16 units and 8 up
	VectorCopy( tag_weapon.origin, tag_result->origin );
	Matrix_Copy( tag_weapon.axis, tag_result->axis );
	VectorMA( tag_result->origin, 16, tag_result->axis[0], tag_result->origin );
	VectorMA( tag_result->origin, 8, tag_result->axis[2], tag_result->origin );
}

/*
===============
CG_vWeapSetFrame
===============
*/
static void CG_vWeapSetFrame( void )
{
	if( cg_gunbob->integer == 0 && vweap.currentAnim == VWEAP_STANDBY ) {
		vweap.oldframe = 0;
		vweap.frame = 0;
	}
	else {
		vweap.oldframe = vweap.frame;
		vweap.frame++;
	}

	//looping
	if (vweap.frame > vweap.pweapon.weaponInfo->lastframe[vweap.currentAnim])
	{
		if (vweap.pweapon.weaponInfo->loopingframes[vweap.currentAnim]) {
			vweap.frame = (vweap.pweapon.weaponInfo->lastframe[vweap.currentAnim] - (vweap.pweapon.weaponInfo->loopingframes[vweap.currentAnim] - 1));
		}
		else if (!vweap.newAnim)
			vweap.newAnim = VWEAP_STANDBY;
	}

	//new animation
	if ( vweap.newAnim )
	{
		if ( vweap.newAnim == VWEAP_WEAPONUP ) //weapon change
		{
			vweap.pweapon.weaponInfo = vweap.newWeaponInfo;
			vweap.oldframe = vweap.pweapon.weaponInfo->firstframe[vweap.newAnim];//don't lerp
		}
		vweap.currentAnim = vweap.newAnim;
		vweap.frame = vweap.pweapon.weaponInfo->firstframe[vweap.newAnim];
		vweap.newAnim = 0;
	}
}


/*
===============
CG_vWeapUpdateAnimation
===============
*/
static void CG_vWeapUpdateAnimation( void )
{
	if (cg.time > vweap.nextframetime)
	{
		vweap.backlerp = 1.0f;

		CG_vWeapSetFrame();//the model can change at this point

		vweap.prevframetime = cg.time;
		vweap.nextframetime = cg.time + vweap.pweapon.weaponInfo->frametime[vweap.currentAnim];
			
	} else {
		
		vweap.backlerp = 1.0f - ((cg.time - vweap.prevframetime)/(vweap.nextframetime - vweap.prevframetime));
		if(vweap.backlerp > 1)
			vweap.backlerp = 1.0f;
		else if(vweap.backlerp < 0)
			vweap.backlerp = 0;
	}
}

void CG_vWeap_StartShootEffect( int fireMode ) {

	//WEAK
	if( fireMode == FIRE_MODE_WEAK && vweap.newAnim < VWEAP_ATTACK_WEAK )
	{
		vweap.newAnim = VWEAP_ATTACK_WEAK;
		//activate flash
		if( vweap.state->weapon != WEAP_GUNBLADE ) { // gunblade knife doesn't flash
			if( cg_weaponFlashes->integer == 2 && vweap.pweapon.weaponInfo )
				vweap.pweapon.flashtime = cg.time + (int)vweap.pweapon.weaponInfo->frametime[VWEAP_ATTACK_WEAK];
		}
		vweap.pweapon.barreltime = cg.time + (int)vweap.pweapon.weaponInfo->frametime[VWEAP_ATTACK_STRONG];
	}
	//STRONG
	else if( fireMode == FIRE_MODE_STRONG && vweap.newAnim < VWEAP_ATTACK_STRONG )
	{
		vweap.newAnim = VWEAP_ATTACK_STRONG;
		//activate flash
		if (cg_weaponFlashes->integer == 2 && vweap.pweapon.weaponInfo)
			vweap.pweapon.flashtime = cg.time + (int)vweap.pweapon.weaponInfo->frametime[VWEAP_ATTACK_STRONG];

		// gunblade is special
		if( vweap.state->weapon != WEAP_GUNBLADE )
			vweap.pweapon.barreltime = cg.time + (int)vweap.pweapon.weaponInfo->frametime[VWEAP_ATTACK_STRONG];
	}

	//eject brass-debris
	if (cg_ejectBrass->integer && cg_ejectBrass->integer < 3 && vweap.pweapon.weaponInfo && vweap.active) {
		vec3_t origin;
		vec3_t	forward, right, up;

		VectorCopy( cg.refdef.vieworg, origin );
		AngleVectors( cg.refdef.viewangles, forward, right, up );
		//move it a bit fordward and up
		VectorMA( origin, 16, forward, origin );
		VectorMA( origin, 4, up, origin );
		if( cgs.clientInfo[cg.chasedNum].hand == 0)
			VectorMA( origin, 8, right, origin );
		else if( cgs.clientInfo[cg.chasedNum].hand == 1)
			VectorMA( origin, -4, right, origin );				
	}
}

/*
==================
CG_vWeapUpdateState

  Called each new serverframe
==================
*/
void CG_vWeapUpdateState( void )
{
	centity_t	*cent;
	int			torsoNewAnim, i, weapon;

	cent = &cg_entities[cg.chasedNum+1]; // player in POV
	if( cent->serverFrame != cg.frame.serverFrame ) 
	{
		weapon = 0;
	}
	else {
#ifdef PREDICTSHOOTING
		if( CG_WeaponPredictionActive() )
			weapon = cg.predictedWeapon;
		else
#endif
			weapon = cent->current.weapon;
	}

	//update newweapon info
	vweap.newWeaponInfo = CG_GetWeaponFromPModelIndex( &cg_entPModels[cg.chasedNum+1], weapon );

	if( !vweap.state || cent->current.number != vweap.state->number ) {
		// force new animation
		if( chaseCam.mode || cg.frame.multipov ) // if in chasecam standby (for POV switches)
			vweap.newAnim = VWEAP_STANDBY;
		else 
			vweap.newAnim = VWEAP_WEAPONUP;

		// animation based on Torso
		torsoNewAnim = (cent->current.frame>>6 &0x3F);
	} 
	else {
		// delta update animations based on Torso
		torsoNewAnim = (cent->current.frame>>6 &0x3F) * ((cent->current.frame>>6 &0x3F) != (cent->prev.frame>>6 &0x3F));
	}

	if( torsoNewAnim == TORSO_WEAP_DOWN && vweap.newAnim < VWEAP_WEAPDOWN )
		vweap.newAnim = VWEAP_WEAPDOWN;
	

	vweap.state = &cent->current;

	//Update based on Events
	for( i = 0; i < 2; i++ ) {

		switch( cent->current.events[i] )
		{
			case EV_FALL:
				if( cg_gunbob->integer == 1 )
                    CG_vWeapStartFallKickEff( cent->current.eventParms[i] );
				break;

			case EV_PAIN:
				break;

			case EV_JUMP:
				break;

			case EV_JUMP_PAD:
				break;

			case EV_MUZZLEFLASH:
				break;

			case EV_DROP:
				break;

			case EV_WEAPONUP:
#ifdef PREDICTSHOOTING
				if( !CG_WeaponPredictionActive() )
#endif
					vweap.newAnim = VWEAP_WEAPONUP;//is top priority
				break;
		}
	}

#ifdef PREDICTSHOOTING
	// update based on predicted event
	if( CG_WeaponPredictionActive() && cg.predictedWeaponEvent )
	{
		switch( cg.predictedWeaponEvent )
		{
			case EV_WEAPONUP:
				vweap.newAnim = VWEAP_WEAPONUP;//is top priority
				break;
		}
	}
#endif

	//update
	if( vweap.pweapon.weaponInfo != vweap.newWeaponInfo ) {
		if( vweap.newAnim )
			vweap.currentAnim = vweap.newAnim;

		vweap.nextframetime = cg.time;
		vweap.pweapon.weaponInfo = vweap.newWeaponInfo;
		vweap.frame = vweap.pweapon.weaponInfo->firstframe[vweap.currentAnim];
		vweap.oldframe = vweap.frame;
	}
}

static entity_t	gun;	// hand model

/*
==============
CG_CalcViewWeapon
==============
*/
void CG_CalcViewWeapon( void )
{
	orientation_t	tag, *handposition;

	//setup
	CG_vWeapUpdateAnimation();
	CG_vWeapGetPosition( vweap.origin, vweap.axis );

	// offset the viewweapon to the tag_position in the model *_handposition
	handposition = &vweap.pweapon.weaponInfo->tag_handposition;
	VectorCopy( vec3_origin, tag.origin );
	Matrix_Copy( axis_identity, tag.axis );

	// move the empty tag to the handposition tag in vweap space
	CG_MoveToTag( tag.origin, tag.axis,
		vweap.origin, vweap.axis,
		handposition->origin, handposition->axis );

	VectorCopy( tag.origin, vweap.origin );
	Matrix_Copy( tag.axis, vweap.axis );
	
	//hand entity
	memset( &gun, 0, sizeof(gun) );
	gun.model = vweap.pweapon.weaponInfo->model[HAND];

	//if the player doesn't want to view the weapon we still have to build the projection source
	if( CG_GrabTag( &tag, &gun, "tag_weapon" ) )
		CG_vWeapUpdateProjectionSource( vweap.origin, vweap.axis, tag.origin, tag.axis );
	else
		CG_vWeapUpdateProjectionSource( vweap.origin, vweap.axis, vec3_origin, axis_identity );
}
/*
==============
CG_AddViewWeapon
==============
*/
void CG_AddViewWeapon( void )
{
	orientation_t	tag;
	
	//hand entity
	gun.model = vweap.pweapon.weaponInfo->model[HAND];

	//update the position
	VectorCopy( vweap.origin, gun.origin );
	VectorCopy( vweap.origin, gun.origin2 );
	VectorCopy( vweap.origin, gun.lightingOrigin );
	Matrix_Copy( vweap.axis, gun.axis );

	gun.renderfx = RF_MINLIGHT|RF_WEAPONMODEL;
	gun.scale = 1.0f;
	gun.frame = vweap.frame;
	gun.oldframe = vweap.oldframe;
	gun.backlerp = vweap.backlerp;

	CG_AddEntityToScene( &gun );

	// if POV has changed, wait until it's updated to draw anything
	if( !vweap.state || (vweap.state->number != cg.chasedNum + 1) )
		return;

	// weapon disabled
	if( !vweap.active || !cg_gun->integer )
		return;

	CG_AddColoredOutLineEffect( &gun, cg.effects, 0, 0, 0, 255 );
	CG_AddShellEffects( &gun, cg.effects );

	// add attached weapon
	if( CG_GrabTag( &tag, &gun, "tag_weapon" ) )
		CG_AddWeaponOnTag( &gun, &tag, &vweap.pweapon, cg.effects|EF_OUTLINE, NULL );
}


