#!/bin/sh
libs="$1"
shift
binary="$1"
shift
confs="$1"
shift

libpath=

# Apppend a directory to LD_LIBRARY_PATH
vm_append_lib() {
   local lib="$1" # IN: The library to append
   local path="$libs"'/'"$lib"

   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      path="$path"'/'"$lib"
   fi
    
   if [ "$libpath" = '' ]; then
      libpath="$path"
   else
      libpath="$libpath":"$path"
   fi
}

# Append the GTK+ 2.4 libaries to LD_LIBRARY_PATH
vm_fix_gtk24() {
   # GLib
   vm_append_lib 'libglib-2.0.so.0'
   vm_append_lib 'libgmodule-2.0.so.0'
   vm_append_lib 'libgobject-2.0.so.0'
   vm_append_lib 'libgthread-2.0.so.0'

   # ATK
   vm_append_lib 'libatk-1.0.so.0'

   # Pango
   vm_append_lib 'libpango-1.0.so.0'
   vm_append_lib 'libpangoft2-1.0.so.0'
   vm_append_lib 'libpangoxft-1.0.so.0'
   vm_append_lib 'libpangox-1.0.so.0'

   # GTK+
   vm_append_lib 'libgdk-x11-2.0.so.0'
   vm_append_lib 'libgdk_pixbuf-2.0.so.0'
   vm_append_lib 'libgtk-x11-2.0.so.0'

   # librsvg
   vm_append_lib 'librsvg-2.so.2'

   # libsexy
   vm_append_lib 'libsexy.so.1'

   # GnomeCanvas
   # We use gnomecanvas 2.6. Apparently 2.4 is symbol compatible but not
   # compatible enough to avoid crashing at times, so consider gnomecanvas to
   # be part of the base Gtk set.
   vm_append_lib 'libgnomecanvas-2.so.0'

   # These are not a direct dependency of GTK+ 2.4, but they keep breaking
   # binary compatibility in both directions, so use ours.
   vm_append_lib 'libfreetype.so.6'
   vm_append_lib 'libXft.so.2'
   vm_append_lib 'libXrender.so.1'

   # FontConfig is actually fairly well behaved but the fonts.conf that we
   # force when VMWARE_USE_SHIPPED_GTK=yes generates seemingly harmless
   # warnings for the older version of fontconfig shipped with RH9.
   # If we're going to force the config file, we should force a lib
   # that parses it properly
   vm_append_lib 'libfontconfig.so.1'

   echo "$libpath"
}

# Append the remaining, usually safe, libraries
# This is used in the 'force' case to ensure all our libs are used
vm_force_other_libs() {
   # Expat
   vm_append_lib 'libexpat.so.0'

   # libart
   vm_append_lib 'libart_lgpl_2.so.2'

   # libxml2
   vm_append_lib 'libxml2.so.2'

   # libglade
   vm_append_lib 'libglade-2.0.so.0'

   # ssl
   vm_append_lib 'libcrypto.so.0.9.7'
   vm_append_lib 'libssl.so.0.9.7'

   echo "$libpath"
}

# Parse ldd's output and append all missing libraries to LD_LIBRARY_PATH
vm_append_missing() {
   local lib
   local dummy
   local status

   while read -r lib dummy status; do
      if [ "$status" = 'not found' ]; then
         vm_append_lib "$lib"
      fi
   done

   echo "$libpath"
}

# Run "$binary" while watching its progress on its stderr.
vm_run() {
   local exitCode;

   # Append any libraries that are still missing.
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export LD_PRELOAD="$LD_PRELOAD":"`LANGUAGE=C LANG=C ldd "$binary" | vm_append_missing`"
   else
      export LD_LIBRARY_PATH="$LD_LIBRARY_PATH":"`LANGUAGE=C LANG=C ldd "$binary" | vm_append_missing`"
   fi

   #
   # "$binary"'s stdout goes unmodified to the shell's stdout.
   # "$binary"'s stderr goes almost unmodified to the shell's stderr
   # (lines starting with "UI: Wrapper: " are swallowed.)
   #

   exec 3>&1
   "$binary" "$@" 2>&1 1>&3 | (
      local exitCode=1 # Failure
      local line

      while IFS='' read -r line; do
         # When "$binary" enters the event loop, pretty much all the GTK stuff
         # should have been initialized.
         case "$line" in
         'UI: Wrapper: '*)
            exitCode=0 # Success
            ;;
         *)
            echo "$line"
            ;;
         esac
      done

      exit "$exitCode"
   ) 1>&2
   exitCode="$?"
   exec 3>&-

   return "$exitCode"
}

#
# This function tests if an environment variable is actually set or not.
# As we had to dig quite a bit to work out what to do, we've put it in
# a separate function.
#
vm_is_var_set() {
   local var="$1"
   
   eval '[ "${'"$var"'+yes}" = 'yes' ]'
}
      
#
# We must save all environment variables that we manipulate because we will
# be fork/exec'ing help/web browsers. We must also explicitly indicate which
# vars were originally unset. There is a third state where we don't try and
# save a var at all and this is indicated by not setting VMWARE_<foo> at all.
#
vm_save_var() {
   local var="$1"

   if vm_is_var_set 'VMWARE_'"$var"; then
      echo 'VMWARE_'"$var"' is already set, but it will be overriden.'
   fi

   if vm_is_var_set "$var"; then
      eval 'export VMWARE_'"$var"'="1$'"$var"'"'
   else
      eval 'export VMWARE_'"$var"'="0"'
   fi
}

#
# Function to strip qtengine out of the set of GTK2_RC_FILES. SuSE evilness
# necessitates this. It has to be a function because we wipe the 'command line'
# argument space when we do the stripping.
#
vm_strip_out_qt_engine() {
   local gtk2_rc_files="$1"
   local tmp_rc_files=""

   #
   # gtk-qt-engine is evil and SuSE use an evil mechanism to force it.
   # So, we have to strip it out like this:
   #
   set -- `IFS=':'; echo $gtk2_rc_files`
   while [ "$#" -gt 0 ]; do
      case "$1" in
         *'qtengine'*)
         ;;
         *'/Qt/'*)
         ;;
         *)
         tmp_rc_files="$tmp_rc_files""$1"':'
         ;;
      esac
      shift
   done
   echo "$tmp_rc_files"
}

#
# This list must be kept in sync with the corresponding UI code that restores
# these variables.
#
vm_save_var 'LD_LIBRARY_PATH'
vm_save_var 'LD_PRELOAD'
vm_save_var 'PANGO_RC_FILE'
vm_save_var 'GDK_PIXBUF_MODULE_FILE'
vm_save_var 'GTK_IM_MODULE_FILE'
vm_save_var 'FONTCONFIG_PATH'
vm_save_var 'GTK2_RC_FILES'
vm_save_var 'GTK_PATH'

#
# Dot releases of g++ keep breaking binary compatibility. Unconditionally use
# the C++ libraries we compiled with the same compiler.
#
vm_append_lib 'libgcc_s.so.1'
vm_append_lib 'libstdc++.so.5'
vm_append_lib 'libsigc-2.0.so.0'
vm_append_lib 'libglibmm-2.4.so.1'
vm_append_lib 'libglibmm_generate_extra_defs-2.4.so.1'
vm_append_lib 'libatkmm-1.6.so.1'
vm_append_lib 'libpangomm-1.4.so.1'
vm_append_lib 'libgdkmm-2.4.so.1'
vm_append_lib 'libgtkmm-2.4.so.1'
vm_append_lib 'libgnomecanvasmm-2.6.so.1'
vm_append_lib 'libview.so.2'
vm_append_lib 'libsexymm.so.1'

#
# Redhat-7.3 doesn't have libpng12.so.0.  the executable doesn't link to
# it directly, so we explicitly include it here
#
vm_append_lib 'libpng12.so.0'

#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "force", we
# forcefully use the 32-bit GTK+ 2.4 runtime environment (libraries + their
# configuration files) that we ship.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "yes", we
# thoughtfully use the 32-bit GTK+ 2.4 runtime environment (libraries + their
# configuration files) that we ship.  This works for all but the most broken
# environments, so if we're guessing we try this.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "no", we use 
# the system's 32-bit GTK+ 2.4 runtime environment.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is not set (the default), we
# try to make an educated guess.
#

if [ "$VMWARE_USE_SHIPPED_GTK" = '' ]; then
   if LANGUAGE=C LANG=C ldd -r "$binary" 2>&1 >/dev/null \
         | grep -q 'gtk_file_filter_'; then
      # No 32-bit GTK+ 2.4 runtime environment on the system.
      export VMWARE_USE_SHIPPED_GTK=yes
   fi
fi

if [ "$VMWARE_USE_SHIPPED_GTK" = '' ]; then
   #
   # A 32-bit GTK+ 2.4 runtime environment is present on the system.
   # Unfortunately, the environment is broken on many 64-bit distributions
   # including Fedora Core 3, excluding Debian unstable.
   #

   vm_run "$@"
   exitCode="$?"
   if [ "$exitCode" -eq 0 ]; then
      # Assume that the system's environment works. Then we are done.
      exit "$exitCode"
   fi
   if [ "$exitCode" -gt 128 ]; then
      # The watcher process received a signal, maybe a SIGINT if the user
      # pressed Ctrl-C.
      exit "$exitCode"
   fi

   # Assume that the system's environment is broken.
   export VMWARE_USE_SHIPPED_GTK=yes
fi

if [ "$VMWARE_USE_SHIPPED_GTK" = 'yes' -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
   #
   # Libraries
   #
   
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'yes' ]; then
      export LD_LIBRARY_PATH="`vm_fix_gtk24`"
   else
      export LD_PRELOAD="`vm_fix_gtk24`":"`vm_force_other_libs`"
   fi

   #
   # Library configuration files
   #

   if [ -z "$PANGO_RC_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export PANGO_RC_FILE="$confs"'/etc/pango/pangorc'
   fi

   if [ -z "$GDK_PIXBUF_MODULE_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GDK_PIXBUF_MODULE_FILE="$confs"'/etc/gtk-2.0/gdk-pixbuf.loaders'
   fi

   if [ -z "$GTK_IM_MODULE_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK_IM_MODULE_FILE="$confs"'/etc/gtk-2.0/gtk.immodules'
   fi

   # We always want to append our gtkrc, unless we're forcing it
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK2_RC_FILES="$confs"'/etc/gtk-2.0/gtkrc'
   else
      if [ -z "$GTK2_RC_FILES" ]; then
         GTK2_RC_FILES='/etc/gtk-2.0/gtkrc:'"$HOME"'/.gtkrc-2.0:'
      else
         GTK2_RC_FILES="`vm_strip_out_qt_engine $GTK2_RC_FILES`"
      fi
      export GTK2_RC_FILES="$GTK2_RC_FILES""$confs"'/etc/gtk-2.0/gtkrc'
   fi

   if [ -z "$FONTCONFIG_PATH" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export FONTCONFIG_PATH="$confs"'/etc/fonts'
   fi

   # always append our libdir to GTK_PATH
   vm_gtk_path="$confs"'/lib/gtk-2.0/2.4.0/'
   if [ -z "$GTK_PATH" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK_PATH="$vm_gtk_path"
   else
      export GTK_PATH="$GTK_PATH"':'"$vm_gtk_path"
   fi
fi

vm_run "$@"
exit 0
