package org.herac.tuxguitar.io.pt;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.MeasureHeader;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.TimeSignature;

public class PTB4InputStream extends PTBInputStream{
	private static final long MIN_START = 1000;
	
	private static final int nullTag = 0;
	private static final int NEW_TAG_CLASS = 0xffff;
	private static final int tagClass = 0x8000;
	private static final int BigTagClass = 0x80000000;
	private static final int objectTag = 0x7fff;
	private static final int MaxMapCount = 0x3ffffffe;	
	
	
    private static final String PTB_VERSION = "ptab-4";        
    
    public PTB4InputStream(FileInputStream file) throws FileNotFoundException {
        super(file);
    }

    public PTB4InputStream(String fileName) throws FileNotFoundException {        
    	 super(fileName);            
    }          
    
    public boolean isSupportedVersion(String version){
    	return (version.equals(PTB_VERSION));
    }

    public boolean isSupportedVersion(){
    	try{
    		readVersion();
    		return isSupportedVersion(version);	  
    	}catch(Exception e){
    		return false;
    	}catch(Error e){
    		return false;
    	}   		
    }	
	
    private void readVersion(){
    	if(this.version == null){
    		this.version = readString(4) + "-" + readUInt();
    	}
    }	
	
	public Song readSong() throws IOException{		
		readVersion();		
        if (!isSupportedVersion(version)) {
            throw new IOException("Unsuported Version");
        }
		
        SongInfo info = readSongInfo();
		// Guitar		
		readDataInstruments(TRACK_TYPE_GUITAR);
		// Bass
		readDataInstruments(TRACK_TYPE_BASS);
						
		return checkSong(new Song(info.getName(),info.getInterpret(),info.getAlbum(),info.getAuthor(),tracks,headers,Song.MAX_VOLUME));
	}

    private SongInfo readSongInfo(){
    	SongInfo info = new SongInfo();
    	
    	int classification = readUCType();    	     	
		if(classification == 0) {
    		int content = readUCType();    		
    		String title = readString();
    		String artist = readString();    		
    		int releaseType = readUCType();   
    		String albumTitle = new String();
    		if (releaseType == 0){
    			int albumType = readUCType();    			    			    			    			
    			albumTitle = readString();    			    			
    			int year = readUInt();    			    
    			boolean liveRecording = readUChar();    	    			
    		}else if(releaseType == 1){
    			albumTitle = readString();
    			boolean liveRecording = readUChar();
    		}else if(releaseType == 2){
    			albumTitle = readString();    			    	
    			int day = readUInt();
    			int month = readUInt();
    			int year = readUInt();   
    		}    		
    		boolean hasAuthor = (readUCType() == 0);
    		String composer = new String();
    		String lyricist = new String();
    		if (hasAuthor) {
    			composer = readString();
    			lyricist = readString();
    		}    		
    		String arrenger = readString();    		
    		String guitarTranscriber = readString();    
    		String bassTranscriber = readString();
    		String copyright = readString();	
    		String lyrics = readString();
    		String guitarInstructions = readString();
    		String bassInstructions = readString();    	
    		
    		info.setName(title);
    		info.setAlbum(albumTitle);
    		info.setInterpret(artist);
    		info.setAuthor(composer);
		}
		//Lesson Classification
		else if(classification == 1){
    		String title = readString();    		
    		String subtitle = readString();    		    	    		
    		int style = readUInt();    	    
    		int level = readUCType();    		
    		String author = readString();    		
    		String instructions = readString();    		
    		String copyright = readString();
    		
    		info.setName(title);
    		info.setAlbum(subtitle);
    		info.setAuthor(author);    		
    	}		
		return info;
    }
    
    private void readDataInstruments(int trackType){
    	int trackCount = tracks.size();
    	List tempoHelpers = new ArrayList();
    	List guitarInHelpers = new ArrayList();    	
    	
       	// CGuitar section
    	int itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readTrackInfo(trackType);
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	// CChordDiagram section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readChord();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}    		
    	}    	

    	// CFloatingText section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readFloattingText();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	    	
    	// CGuitarIn section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		guitarInHelpers.add(readGuitarIn());
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	// CTempoMarker
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		TempoHelper tempo = readTempoMarker();
    		if(tempo != null){
    			tempoHelpers.add(tempo);
    		}
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	// CDynamic section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readDynamic();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}

    	// CSectionSymbol section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readSectionSymbol();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	// CSection section
    	int number = trackCount + 1;
    	long start = MIN_START;
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		//start += readSection(number,trackType,start,tempoHelpers);
    		start = readSection(number,trackType,start,tempoHelpers);
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	makeTrackChannels(trackCount,trackType,guitarInHelpers);
    }
    

    private int readHeaderItems(){    	    	
    	int nbItems = readUInt();    	
    	if (nbItems == 0){
    		return 0;
    	}
    	
    	int header = readUInt();    	
    	if (header == NEW_TAG_CLASS) {    		    		
    		if (readUInt() != 1) {
    			return -1;
    		}    		
    		String name = readString(readUInt());    		
    	}    	
    	return nbItems;
    }
    
    
    
    
    
    
    
    
    
    
    private void readTrackInfo(int trackType){
    	TrackInfo info = new TrackInfo();
    	info.setType(trackType);
    	info.setChannel(new SongChannel((short)0,(short)0,(short)0));
    	info.setStrings(new ArrayList());    	
    	
    	// Track info
    	info.setTrackInfoNumber(readByte());
    	
    	info.setName(readString());

    	// Channel
    	info.getChannel().setInstrument((short)readByte());

    	info.getChannel().setVolume((short)readByte());

    	info.getChannel().setBalance((short)readByte());
    	
    	info.getChannel().setReverb((short)readByte());
    	
    	info.getChannel().setChorus((short)readByte());
    	
    	info.getChannel().setTremolo((short)readByte());
    	
    	info.getChannel().setPhaser((short)readByte());
    	
    	int capo = readByte();

    	// Tuning
    	String tunningName = readString();

    	//bit 7 = Music notation offset sign, bits 6 to 1 = Music notation offset value, bit 0 = display sharps or flats;
    	int offset = readByte();    	

    	int stringCount = readByte();    	
    	for (int i = 1; i <= stringCount; i++) {
    		info.getStrings().add(new InstrumentString(i,readByte()));
    	}
    	trackInfos.add(info);
    }
    
    
    
    private void readChord(){    	    	
    	int chordKey = readUInt();    	
    	
    	readByte();    	
    	int chordModification = readUInt();    	
    	readByte();    	
    	readByte();
    	
    	int stringCount = readByte();
    	
    	for (int j = 0; j < stringCount; j++) {
    		int fret = readByte();    		
    	}

    }    
    
    
    
    private void readFloattingText(){    	
    	// Floating text
    	String text = readString();

    	// Read mfc rect
    	long left   = readULong();
    	long top    = readULong();
    	long right  = readULong();
    	long bottom = readULong();
    	
    	readByte();

    	readFontSetting();
    	
    	
    }
	
    private void readFontSetting(){    	
    	String fontName = readString();    	
    	int pointSize = readULong();    	
    	int weight = readULong();    	
    	boolean italic = readUChar();
    	boolean underline = readUChar();
    	boolean strikeout = readUChar();
    	int color = readULong();
    }    
    
    
    private GuitarInHelper readGuitarIn(){    	    	
    	int system = readUInt();   
    	int staff = readByte();
    	int position = readByte();    	
    	int data = readUInt();
    	return new GuitarInHelper(staff + 1,(data >> 8) - 1);
    }
    
    
    
    private TempoHelper readTempoMarker(){    	
    	int system = readUInt();    	
    	int position = readByte();    	
    	int tempo = readUInt();
    	int data = readUInt();    	
    	String description = readString();
    	    	
    	int tripletFeel = MeasureHeader.TRIPLET_FEEL_NONE;
    	if((data & 0x01) != 0){
    		tripletFeel = MeasureHeader.TRIPLET_FEEL_EIGHTH;
    	}else if((data & 0x02) != 0){
    		tripletFeel = MeasureHeader.TRIPLET_FEEL_SIXTEENTH;
    	}
    	if(tempo > 0 && system == 0){
    		return new TempoHelper(position,tempo,tripletFeel);
    	}
    	return null;
    }

    private void readDynamic(){    	
    	readUInt();    	
    	readByte();    	
    	readByte();    	
    	readUInt();
    }
    
    
    
    
    private void readSectionSymbol(){
    	readUInt();    	
    	readByte();    	
    	readULong();
    }
    
    

    private long readSection(int trackNumber,int trackType,long start,List tempoHelpers){    	
    	List trackHelpers = new ArrayList();
    	List barHelpers = new ArrayList();
    	
    	int left = readULong();
    	int top = readULong();
    	int right = readULong();
    	int bottom = readULong();

    	readByte();    
    	readByte();    	
    	readByte();    	
    	readByte();    	
    	readByte();
    	
    	// BarLine
    	barHelpers.add(readBarLine());
    	/*
    	//readByte();    	
    	int type = readByte();    	
    	boolean repeatStart = ((type >>> 5) == 3);
    	
    	// KeySignature
    	readByte();
    	
    	// TimeSignature
    	int data = readULong();
    	
    	//solo permito el cambio en el primer track
    	if(trackNumber == 1){
    		MeasureHeader header = getHeader(start);
    		header.getTimeSignature().setNumerator((((data >> 24) - ((data >> 24) % 8)) / 8) + 1);
    		header.getTimeSignature().setDenominator(new Duration((int)Math.pow(2,(data >> 24) % 8)));
    	}    	
    	int measurePulses = readByte();
    	
    	readRehearsalSign();
    	*/
    	
    	// CDirection section
    	int itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readDirection();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	
    	// CChordText section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readChordText();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}    		
    	}
    	
    	
    	// CRhythmSlash section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		readRhythmSlash();
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}

    	
    	// Staff
    	//long maxLength = 0;    	
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		TrackHelper helper = new TrackHelper(trackNumber + j,trackType);
    		trackHelpers.add(helper);
    		
    		/*long length = */readStaff(helper,start);
    		//maxLength = (length > maxLength)?length:maxLength;
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	
    	// CMusicBar section
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		barHelpers.add(readBarLine());
    		
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	return makeSectionNotes(start,trackHelpers,barHelpers,tempoHelpers);
    	
    	//return maxLength;
    }
    
    
    private void readRehearsalSign(){
    	readByte();    	
    	readString();
    }
    
    private void readDirection(){
    	readByte();    	
    	int symboleCount = readByte();

    	for (int i = 0; i < symboleCount; i++) {
    		int word = readUInt();
    	}
    }

    private void readChordText(){
    	readByte();
    	
    	int key = readUInt();    	
    	readByte();
    	
    	int formulaModification = readUInt();
    	
    	readByte();
    }
    
    private void readRhythmSlash(){
    	readByte();    	
    	readByte();
    	
    	int rhythmData = readULong();
    }
    
    private void readStaff(TrackHelper helper,long minStart){
    	//long length1 = 0;
    	//long length2 = 0;
    	
    	// Staff   
    	 	
    	readByte();
    	readByte();
    	readByte();
    	readByte();
    	readByte();    	

    	
    	//long start = minStart;
    	int itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		/*length1 += */readPosition(helper.getVoice1());
    		//start = (minStart + length1);
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	
    	// CPosition section 
    	//start = minStart;
    	itemCount = readHeaderItems();
    	for (int j = 0; j < itemCount; j++) {
    		/*length2 +=*/ readPosition(helper.getVoice2());
    		//start = (minStart + length2);
    		if (j < itemCount - 1){
    			readUInt();
    		}    		
    	}    

    	//return (length1 > length2)?length1:length2;
    }
    
    
    private /*long*/void readPosition(List beatHelpers){
    	BeatHelper helper = new BeatHelper();
    	    	
    	int position = readByte();    	
    	int beaming = readByte();
    	beaming = ((beaming - 128 < 0)?beaming:beaming - 128);
    	
    	readByte();    	
    	
    	int data1 = readByte();
    	int data2 = readByte();
    	int data3 = readByte();
    	int durationValue = readByte();    	
    	
    	int multiBarRest = 1;
    	int complexCount = readByte();    	
    	for (int i = 0; i < complexCount; i++) {
    		//int symbolArray = readULong();
    		int count = readUInt();  
    		readByte();
    		
    		int type = readByte();
    		if((type & 0x08) != 0){
    			multiBarRest = count;
    		}
    	}    	
    	
    	int itemCount = readHeaderItems();    	
    	for (int j = 0; j < itemCount; j++) {
    		readNote(helper);
    		if (j < itemCount - 1){
    			readUInt();
    		}
    	}
    	    	
    	helper.setPosition(position);
    	helper.setMultiBarRest((itemCount == 0)?multiBarRest:1);
    	helper.setAcciaccatura((data3 & 0x01) != 0);
    	helper.setVibrato(((data1 & 0x08) != 0) || ((data1 & 0x10) != 0));
    	
    	// Set the duration
    	helper.setDuration(new Duration(durationValue));
    	helper.getDuration().setDotted((data1 & 0x01) != 0);
    	helper.getDuration().setDoubleDotted((data1 & 0x02) != 0);
    	helper.getDuration().getTupleto().setEnters(((beaming - (beaming % 8)) / 8) + 1);
    	helper.getDuration().getTupleto().setTimes((beaming % 8) + 1);
    	
    	beatHelpers.add(helper);
    	
    	//return (helper.isAcciaccatura())?0:helper.getDuration().getTime();
    }
        
    
    private BarHelper readBarLine(){
    	BarHelper helper = new BarHelper(readByte());

    	int type = readByte();
    	
    	//repeat start
    	helper.setRepeatStart(((type >>> 5) == 3));

    	//repeat end
    	helper.setNumberOfRepetitions((((type >>> 5) == 4)?(type - 128):0));
    	
    	/*
    	if((type >>> 5) == 4){
    		int numberOfRepetitions = (type - 128);
    		debug(numberOfRepetitions);
    	}    	
    	*/
    	readKeySignature();
    	helper.setTimeSignature(readTimeSignature());
    	readRehearsalSign();
    	
    	return helper;
    }
    
    private void readKeySignature(){
    	readByte();
    }

    private TimeSignature readTimeSignature(){
    	int data = readULong();    	    		
    	int measurePulses = readByte();    	
    	return new TimeSignature(((((data >> 24) - ((data >> 24) % 8)) / 8) + 1),new Duration((int)Math.pow(2,(data >> 24) % 8)));
    }

    
    private void readNote(BeatHelper helper){
    	NoteHelper note = new NoteHelper();    	
    	int position = readByte();       	
    	int simpleData = readUInt();    	
    	int symbolCount = readByte();

    	for (int i = 0; i < symbolCount; i++) {
    		int data1 = readByte();
    		int data2 = readByte();
    		int data3 = readByte();
    		int data4 = readByte();    		    		
    		note.setBend((data4 == 101)?((data3 / 16) + 1):0);
    		note.setSlide((data4 == 100));
    	}
    	
    	note.setValue(position & 0x1f);
    	note.setString(((position & 0xe0) >> 5) + 1);
    	note.setTied((simpleData & 0x01) != 0);
    	note.setDead((simpleData & 0x02) != 0);    	
    	
    	boolean ghost = (((simpleData >> 4) & 0x08) != 0);
    	boolean hammer = (((simpleData >> 4) & 0x02) != 0);
    	    	
    	helper.addNote(note);    	    
    }
    
    
    
    
    
    
    
    
    
    
    
    
    

    
    
    
    
    
    
    
    
    
    
    
    
    
    

    public static void main(String[] s){
    	String file = "/home/julian/test.ptb";
    	try {
			new PTB4InputStream(file).readSong();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
    }
}
