package tijmp.ui;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.management.MemoryMXBean;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import tijmp.HeapWalkEntry;
import tijmp.OwnerInfoHeader;
import tijmp.ProfilerHandler;
import tijmp.UIHandler;
import tijmp.actions.DefaultFilter;
import tijmp.actions.MemoryInfo;
import tijmp.actions.ThreadInfo;
import tijmp.filter.AcceptAllFilter;

/** An ui handler that runs in the same jvm as the program 
 *  being profiled and uses swing to present information.
 */
public class InVMUI implements UIHandler {
    private ProfilerHandler ph;
    private JFrame frame;
    private JLabel statusLabel;
    private JButton gc;
    private JButton heapWalker;
    private long heapWalkStarted;
    private FilterConfig fc;

    public void init (ProfilerHandler ph) {
	this.ph = ph;
	EventQueue.invokeLater (new Runnable () {
		public void run () {
		    fc = new FilterConfig ();
		    buildGui ();
		}
	    });
    }

    public FilterConfig getFilterConfig () {
	return fc;
    }

    private void runGCOffEDT () {
	ph.submitTask (new Runnable () {
		public void run () {
		    ph.runGC ();
		    gcFinished ();
		}
	    });
    }

    private void walkHeapOffEDT () {
	ph.submitTask (new Runnable () {
		public void run () {
		    heapWalkStarted = System.currentTimeMillis ();
		    ph.walkHeap ();
		}
	    });
    }
    
    private void buildGui () {
	frame = new JFrame ("TIJMPController");
	JPanel buttons = new JPanel ();
	gc = new JButton ("GC");
	gc.addActionListener (new ActionListener () {
		public void actionPerformed (ActionEvent e) {
		    gc.setEnabled (false);
		    runGCOffEDT ();
		}
	    });
	buttons.add (gc);
	heapWalker = new JButton ("Walk heap");
	heapWalker.addActionListener (new ActionListener () {
		public void actionPerformed (ActionEvent e) {
		    heapWalker.setEnabled (false);
		    walkHeapOffEDT ();
		}
	    });
	buttons.add (heapWalker);
	JButton dfilter = new JButton (new DefaultFilter (fc));
	buttons.add (dfilter);
	JButton cfilter = new JButton ("Clear filter");
	cfilter.addActionListener (new ActionListener () {
		public void actionPerformed (ActionEvent e) {
		    fc.setFilter (new AcceptAllFilter ());
		}
	    });
	buttons.add (cfilter);
	MemoryMXBean mbean = ph.getMemoryMXBean ();
	JButton memory = new JButton (new MemoryInfo (ph));
	buttons.add (memory);
	
	JButton threads = new JButton (new ThreadInfo (ph));
	buttons.add (threads);

	statusLabel = new JLabel ("Starting up");
	statusLabel.setBorder (BorderFactory.createLoweredBevelBorder ());

	frame.add (buttons, BorderLayout.NORTH);
	MemoryGraph mg = new MemoryGraph (mbean);
	mg.start ();
	frame.add (mg, BorderLayout.CENTER);
	frame.add (statusLabel, BorderLayout.SOUTH);
	frame.setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE);
	frame.pack ();
	frame.setVisible (true);
    }    

    public void showHeapWalkResult (final List<HeapWalkEntry> ls) {
	long heapWalkEnded = System.currentTimeMillis ();
	long diff = heapWalkStarted - heapWalkEnded;
	showStatus (String.format ("heap walking took %d millis", diff));
	if (EventQueue.isDispatchThread ()) {
	    showHeapWalkTable (ls);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			showHeapWalkTable (ls);
		    }
		}); 
	}
    }

    private void showHeapWalkTable (List<HeapWalkEntry> ls) {
	heapWalker.setEnabled (true);
	HeapWalkTable tableComponent = new HeapWalkTable (ls, ph, fc);
	tableComponent.showFrame ();	
    }


    public void gcFinished () {
	EventQueue.invokeLater (new Runnable () {
		public void run () {
		    gc.setEnabled (true);
		}
	    });
    }

    private void showInstances (final Class<?> clz, final Object[] objects,
				final long[] sizes, final int[] lengths) {
	InstanceTable it = new InstanceTable (ph, clz, objects, sizes, lengths);
	it.showFrame ();	
    }
    
    public void instances (final Class<?> clz, final Object[] objects,
			   final long[] sizes, final int[] lengths) {
	if (EventQueue.isDispatchThread ()) {
	    showInstances (clz, objects, sizes, lengths);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			showInstances (clz, objects, sizes, lengths);
		    }
		}); 
	}
    }

    private void showStrings (final Object[] objects) {
	StringTree st = new StringTree (objects);
	showFrame ("Strings", st);
    }

    public void strings (final Object[] objects) {
	if (EventQueue.isDispatchThread ()) {
	    showStrings (objects);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			showStrings (objects);
		    }
		}); 
	}
    }

    private void showChildObjects (Object[] objects) {
	ChildObjectTable ct = new ChildObjectTable (objects);
	showFrame ("Child objects summary", ct);
    }

    public void childObjects (final Object[] objects) {
	if (EventQueue.isDispatchThread ()) {
	    showChildObjects (objects);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			showChildObjects (objects);
		    }
		}); 
	}
    }

    private void showFrame (String title, JComponent comp) {
	new ShowSimpleFrame ().showFrame (title, comp);
    }

    public void owners (final Map<Long, OwnerInfoHeader> owners, 
			final long[] startObjects) {
	if (EventQueue.isDispatchThread ()) {
	    OwnerInfoTree oit = new OwnerInfoTree (ph, owners, startObjects);
	    showFrame ("Object owner information", oit);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			OwnerInfoTree oit = 
			    new OwnerInfoTree (ph, owners, startObjects);
			showFrame ("Object owner information", oit);
		    }
		}); 
	}
    }

    public void showStatus (final String status) {
	if (EventQueue.isDispatchThread ()) {
	    statusLabel.setText (status);
	} else {
	    EventQueue.invokeLater (new Runnable () {
		    public void run () {
			statusLabel.setText (status);			
		    }
		}); 
	}
    }
}
