package tijmp.ui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import tijmp.HeapWalkEntry;
import tijmp.ProfilerHandler;
import tijmp.actions.DefaultFilter;
import tijmp.actions.FilterOutClass;
import tijmp.actions.FilterOutPackage;
import tijmp.actions.InspectClass;
import tijmp.actions.OnlyPackage;
import tijmp.actions.ShowAllInstances;
import tijmp.actions.ShowAllStrings;
import tijmp.actions.ShowOwners;
import tijmp.actions.StrictPackage;
import tijmp.filter.Filter;
import tijmp.filter.RegexpFilter;

/** A class that show a table with heap walk statistics.
 */
class HeapWalkTable implements FilterListener {
    private ATM m;
    private List<HeapWalkEntry> ls;
    private ProfilerHandler ph;
    private FilterConfig fc;

    private static int windowId = 1;

    public HeapWalkTable (List<HeapWalkEntry> ls, ProfilerHandler ph, 
			  FilterConfig fc) {
	this.ls = ls;
	this.ph = ph;
	this.fc = fc;
    }
    
    public void showFrame () {
	m = new ATM (ls, fc.getFilter ());
	JTable table = new JTable (m);
	table.setAutoCreateRowSorter (true);
	JScrollPane scrollPane = new JScrollPane (table);
	table.setAutoResizeMode (JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	table.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
	TableColumn column = null;
	ReadableSizeRenderer r = new ReadableSizeRenderer ();
	for (int i = 0; i < m.getColumnCount (); i++) {
	    column = table.getColumnModel ().getColumn (i);
	    if (i == ATM.COL_NAME)
		column.setPreferredWidth (350);
	    else
		column.setPreferredWidth (75);
	    if (i == ATM.COL_SIZE || i == ATM.COL_SIZE_DIFF) 
		column.setCellRenderer (r);
	}
	table.setDefaultRenderer (Class.class, new ClassRenderer ());
	table.removeColumn (table.getColumnModel ().getColumn (ATM.COL_HWE));
        table.setPreferredScrollableViewportSize (new Dimension (750, 200));
	table.addMouseListener (new MouseHandler (table, ph, fc));
	table.getRowSorter ().toggleSortOrder (ATM.COL_SIZE);
	table.getRowSorter ().toggleSortOrder (ATM.COL_SIZE);

	JFrame f = new JFrame ("Heap Walk Result " + windowId++);
	JPanel p = new JPanel (new FlowLayout (FlowLayout.LEFT, 0, 0));
	JLabel l = new JLabel ("Set filter (regexp matching)");
	p.add (l);
	JTextField tf = new JTextField (30);
	p.add (tf);
	tf.addActionListener (new SetRegexpFilter (tf, fc));
	
	f.add (p, BorderLayout.NORTH);
	f.add (scrollPane, BorderLayout.CENTER);
	f.setDefaultCloseOperation (JFrame.DISPOSE_ON_CLOSE);
	fc.addFilterListener (this);
	f.addWindowListener (new WindowAdapter () {
		@Override public void windowClosed (WindowEvent e) {
		    fc.removeFilterListener (HeapWalkTable.this);
		}
	    });
	f.pack ();
	f.setVisible (true);	
    }

    public void filterChanged (Filter filter) {
	m.setNewFilter (filter);
    }
}

class SetRegexpFilter implements ActionListener {
    private JTextField f;
    private FilterConfig fc;
    
    public SetRegexpFilter (JTextField f, FilterConfig fc) {
	this.f = f;
	this.fc = fc;
    }
    
    public void actionPerformed (ActionEvent e) {
	fc.setFilter (new RegexpFilter (f.getText ()));	
    }
}

class MouseHandler extends MouseAdapter {
    private JTable table;
    private ProfilerHandler ph;
    private FilterConfig fc;

    public MouseHandler (JTable table, ProfilerHandler ph, FilterConfig fc) {
	this.table = table;
	this.ph = ph;
	this.fc = fc;
    }
    
    @Override public void mousePressed (MouseEvent e) {
	if (!SwingUtilities.isRightMouseButton (e))
	    return;
	int viewRow = table.rowAtPoint (e.getPoint ());
	int modelRow = table.convertRowIndexToModel (viewRow);
	Class<?> c = (Class<?>)table.getModel ().getValueAt (modelRow, ATM.COL_NAME);
	String name = Translator.translate (c);
	table.getSelectionModel ().setSelectionInterval (viewRow, viewRow);
	String[] packages = name.split ("\\.");
	JPopupMenu m = new JPopupMenu ();
	for (int i = 0; i < packages.length - 1; i++) {
	    String packageName = getName (packages, 0, i);
	    JMenu menu = new JMenu (packageName);
	    JMenuItem mi = new JMenuItem (new FilterOutPackage (fc, packageName));
	    menu.add (mi);
	    mi = new JMenuItem (new OnlyPackage (fc, packageName));
	    menu.add (mi);
	    mi = new JMenuItem (new StrictPackage (fc, packageName));
	    menu.add (mi);
	    m.add (menu);
	}
	HeapWalkEntry hwe = 
	    (HeapWalkEntry)table.getModel ().getValueAt (modelRow, ATM.COL_HWE);
	Class<?> clz = hwe.getEntryClass ();
	m.add (new JMenuItem (new FilterOutClass (fc, clz)));
	m.add (new JMenuItem (new DefaultFilter (fc)));
	m.addSeparator ();
	m.add (new JMenuItem (new ShowAllInstances (ph, clz)));
	if (c == String.class || c == char[].class)
	    m.add (new JMenuItem (new ShowAllStrings (ph)));
	m.add (new JMenuItem (new ShowOwners (ph, clz)));
	m.addSeparator ();
	m.add (new JMenuItem (new InspectClass (clz)));
	m.show (e.getComponent (), e.getX (), e.getY ());	
    }
    
    private String getName (String[] p, int start, int end) {
	if (start == end)
	    return p[start];
	StringBuilder sb = new StringBuilder ();
	for (int i = start; i <= end; i++) {
	    if (i > start)
		sb.append ('.');
	    sb.append (p[i]);
	}
	return sb.toString ();
    }
}

class ATM extends AbstractTableModel {
    private List<HeapWalkEntry> orig;
    private List<HeapWalkEntry> ls;
    
    private String[] columnNames = {"HWE", "Class", 
				    "Count", "Count diff", 
				    "Size", "Size diff"};
    
    public static final int COL_HWE = 0;
    public static final int COL_NAME = COL_HWE + 1;
    public static final int COL_COUNT = COL_NAME + 1;
    public static final int COL_COUNT_DIFF = COL_COUNT + 1;
    public static final int COL_SIZE = COL_COUNT_DIFF + 1;
    public static final int COL_SIZE_DIFF = COL_SIZE + 1;

    public ATM (List<HeapWalkEntry> ls, Filter filter) {
	orig = ls;
	setNewFilter (filter);
    }

    public void setNewFilter (Filter filter) {
	ls = new ArrayList<HeapWalkEntry> (orig.size ());
	for (HeapWalkEntry e : orig)
	    if (filter.accept (e.getEntryClass ()))
		ls.add (e);
	fireTableDataChanged ();
    }

    @Override public String getColumnName (int col) {
        return columnNames[col];
    }

    public int getRowCount () { 
	return ls.size (); 
    }
    
    public int getColumnCount () { 
	return columnNames.length; 
    }

    public Object getValueAt (int row, int col) {
	HeapWalkEntry hwe = ls.get (row);
	switch (col) {
	case COL_HWE:
	    return hwe;
	case COL_NAME:
	    return hwe.getEntryClass ();
	case COL_COUNT:
	    return hwe.getInstanceCount ();
	case COL_COUNT_DIFF: 
	    return hwe.getInstanceChange ();
	case COL_SIZE:
	    return hwe.getTotalSize ();
	case COL_SIZE_DIFF:
	    return hwe.getSizeChange ();
	default:
	    throw new IllegalArgumentException ("do not know how to handle col: " + 
						col);
	}
    }

    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_HWE:
	    return HeapWalkEntry.class;
	case COL_NAME:
	    return Class.class;
	case COL_COUNT:
	    return Long.class;
	case COL_COUNT_DIFF:
	    return Long.class;
	case COL_SIZE:
	    return Long.class;
	case COL_SIZE_DIFF:
	    return Long.class;
	default:
	    throw new IllegalArgumentException ("do not know how to handle col: " + 
						col);	    
	}
    }
    
    @Override public boolean isCellEditable (int row, int col) { 
	return false; 
    }
    
    @Override public void setValueAt (Object value, int row, int col) {
	throw new IllegalStateException ("non editable table");
    }
}
