/*
 * seti_control.c
 * 
 * Description: Source file that provides functions to control
 * the setiathome process.
 *
 * Author: Matt Herbert <mherbert@bit-net.com>
 *
 * Date Feb 20, 2000
 *
 * Changelog:
 * $Log: seti_control.c,v $
 * Revision 1.9.4.8  2003/03/05 05:57:55  r_kinder
 * GUI fixes, bug fixes.
 *
 * Revision 1.9.4.7  2003/01/28 08:11:23  r_kinder
 * Minor bug fixes, change the default skin look so it looks good on small
 * panels (when fixed size with only the radar is selected).
 *
 * Revision 1.9.4.6  2003/01/23 23:37:28  r_kinder
 * Do a 'make indent' on the files. Remove legacy code that is no longer
 * pertinant.
 *
 * Revision 1.9.4.5  2002/04/09 11:26:13  r_kinder
 * Fixes for crash on removal from panel, misc warning fixes.
 *
 * Revision 1.9.4.4  2002/04/07 23:44:36  r_kinder
 * Functionally almost complete version - still need to work on the graphics, but
 * the splitting of UI from the main code is almost complete (still debating
 * whether to put the properties interface in a new module). Works.
 *
 * Revision 1.9.4.3  2002/03/25 11:10:55  r_kinder
 * Properties now save and restore correctly. Intermediate checkin.
 *
 * Revision 1.9.4.2  2002/02/28 21:02:12  r_kinder
 * Clean up major cruft.
 *
 * Revision 1.9.4.1  2002/02/25 22:58:54  r_kinder
 * First version that builds (NOTE: WILL NOT RUN!!!)
 *
 * Revision 1.9  2001/06/12 07:27:20  r_kinder
 * Fix all of the known bugs from 0.3.4-2, add the new user detection code and
 * gui to set up seti@home accounts. Clean up some of the code.
 *
 */

//#include <gnome.h>
//#include <applet-widget.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <fcntl.h>

#include "seti_control.h"
#include "seti_applet.h"

#include <glibtop.h>
#include <glibtop/procstate.h>

//Defines internal to seti_control

//Filenames for pid and lock files
#define PID_SAH "pid.sah"
#define PID_TXT "pid.txt"
#define LOCK_SAH "lock.sah"
#define LOCK_TXT "lock.txt"

//Prototypes
static gboolean read_pipe_timeout(GIOChannel * source,
                                  GIOCondition condition,
                                  setiapplet * sa);
static gchar **get_extra_args(setiapplet * sa);

/*******************************************************************************
 *
 * int is_seti_process( pid_t pid )
 *
 * This function will check a process id to see if it is indeed a setiathome
 * process.
 *
 * returns: 0 if it's not a seti process, 1 if it is
 *
 ******************************************************************************/
static int
is_seti_process(pid_t pid)
{
    glibtop_proc_state *proc_buf = g_malloc0(sizeof(glibtop_proc_state));
    gchar      *return_str;

    glibtop_get_proc_state(proc_buf, pid);

    /* look for setiathome in the cmdline of the process */
    return_str = strstr(proc_buf->cmd, "setiathome");

    /* clean up */
    g_free(proc_buf);

    /* null pointer returned, that means setiathome wasn't found in the
     * command line of the process, and it's not a setiathome process!
     */
    if(!return_str)
        return (0);

    /* process exists and is a setiathome process */
    return (1);
}

/*******************************************************************************
 * is_seti_running, check if the seti process we have found is running.
 *
 * This method is present because even if we find a seti process on the machine,
 * it could be a zombie, meaning it isn't really running (just waiting for the
 * parent process to exit).
 *
 * Returns: FALSE if the process is a zombie, TRUE otherwise
 ******************************************************************************/
static int
is_seti_running(pid_t pid)
{
    glibtop_proc_state *proc_buf = g_malloc0(sizeof(glibtop_proc_state));
    gchar       status;

    glibtop_get_proc_state(proc_buf, pid);

    /* look for setiathome in the cmdline of the process */
    status = proc_buf->state;

    /* clean up */
    g_free(proc_buf);

    if(status == 'Z')
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
 *
 * get_seti_pid( setiapplet *sa )
 *
 * This function will get the process id of a setiathome process from the
 * pid.sah file that setiathome creates.
 *
 * returns: pid_t of the setiathome process, or 0 if it's not found
 *
 ******************************************************************************/
static      pid_t
get_seti_pid(setiapplet * sa)
{
    gchar      *file_contents;
    gchar      *path_to_pid_file;
    pid_t       pid;

    /* Try the new seti pid.sah file name */
    path_to_pid_file = g_strdup_printf("%s/%s", sa->seti_dir, PID_SAH);
    file_contents = re_read_file(path_to_pid_file);

    /* Try the old seti pid.txt file name */
    if(!file_contents)
    {
        g_free(path_to_pid_file);
        path_to_pid_file = g_strdup_printf("%s/%s", sa->seti_dir, PID_TXT);
        file_contents = re_read_file(path_to_pid_file);

        /* couldn't open any process ID file, bail out */
        if(!file_contents)
        {
            g_free(path_to_pid_file);
            return (0);
        }
    }

    /* convert the pid from the file into an integer */
    pid = atoi(file_contents);

    /* free any memory we allocated */
    g_free(path_to_pid_file);
    g_free(file_contents);

    return (pid);
}

/*******************************************************************************
 *
 * unlink_seti_lock_file( setiapplet *sa )
 *
 * This function will try to unlink the setiathome lock file.  It will
 * try both the seti2 and seti1 lock file names.
 * 
 * TODO: This function could probably use some error checking...
 *
 ******************************************************************************/
static void
unlink_seti_lock_file(setiapplet * sa)
{
    gchar      *lock_file_path;

    /*Try the new seti lock.sah file name */
    lock_file_path = g_strdup_printf("%s/%s", sa->seti_dir, LOCK_SAH);
    unlink(lock_file_path);

    g_free(lock_file_path);

    /* try the old seti lock.txt file name */
    lock_file_path = g_strdup_printf("%s/%s", sa->seti_dir, LOCK_TXT);
    unlink(lock_file_path);

    /* clean up */
    g_free(lock_file_path);
}


/*******************************************************************************
 *
 * seti_status( setiapplet *sa )
 *
 * This function will check to see if setiathome is running.
 *
 * returns: SETI_RUNNING if seti is running and SETI_STOPPED if it's not running
 *
 ******************************************************************************/
int
seti_status(setiapplet * sa)
{
    int         fd;
    int         seti_pid;
    int         stat_rv;
    struct stat *stat_buf;
    gchar      *lock_file_path;

    //If the setiathome path is null, return
    if(sa->seti_dir == NULL)
    {
        return SETI_NONEXISTANT;
    }

    stat_buf = (struct stat *)g_malloc0(sizeof(struct stat));
    if(!stat_buf)
    {
        return SETI_NONEXISTANT;
    }

    //Try the path given - see if it exists - if not, return non-existant
    stat_rv = stat(sa->seti_dir, stat_buf);
    if(stat_rv == -1)
    {
        //Clean up and return non-existant
        g_free(stat_buf);
        return SETI_NONEXISTANT;
    }

    /*Try the new seti lock.sah file name */
    lock_file_path = g_strdup_printf("%s/%s", sa->seti_dir, LOCK_SAH);

    /* Check if the lock file exists */
    stat_rv = stat(lock_file_path, stat_buf);
    if(stat_rv == -1)
    {
        /* Also look for the old seti lock.txt file name */
        g_free(lock_file_path);
        lock_file_path = g_strdup_printf("%s/%s", sa->seti_dir, LOCK_TXT);
        stat_rv = stat(lock_file_path, stat_buf);
        if(stat_rv == -1)
        {
            /* Nope there is no lock file! that means we are ok to
             * start the seti client */
            g_free(lock_file_path);
            g_free(stat_buf);
            return SETI_STOPPED;
        }
    }

    /* clean up */
    g_free(lock_file_path);
    g_free(stat_buf);

    /* ok, there is a lock file, lets check the process and see if it's
     * actually a setiathome process...
     */
    seti_pid = get_seti_pid(sa);
    if(!is_seti_process(seti_pid))
    {
        /* that isn't a setiathome process! it must have died, lets
         * remove the lock
         */
        unlink_seti_lock_file(sa);
        return SETI_STOPPED;
    }
    /*RK 29/2/00 - check if the process is alive, if zombie, return 
     * SETI_STOPPED else return SETI_RUNNING */
    if(is_seti_running(seti_pid))
    {
        /* seti is running */
        return SETI_RUNNING;
    }
    else
    {
        return SETI_STOPPED;
    }

}


/*******************************************************************************
 *
 * stop_seti( setiapplet *sa )
 *
 * This function will stop the currently running seti process
 *
 ******************************************************************************/
static void
stop_seti(setiapplet * sa)
{
    pid_t       pid;
    gchar      *lock_file_path;
    int         rc;

    //console_clear(sa);

    /* get the process id from the pid.sah file */
    pid = get_seti_pid(sa);

    /* check to make sure this is actually a setiathome process */
    if(!is_seti_process(pid))
    {
        /* that pid doesn't belong to a setiathome process, that means
         * seti probably died, and left a mess lying around.  We
         * should probably clean up after it by removing the lock file.
         */
        unlink_seti_lock_file(sa);
        return;
    }


    /* kill the setiathome process, trying the "nice" signal first */
    kill(pid, 2);

    console_log_message(sa,
                        _
                        ("Stopping the seti@home client with HUP signal..."));

    /* give the setiathome process time to die */
    sleep(2);

    /* check to see if it actually died... */
    if(is_seti_process(pid))
    {
        int         status;
        /* the process could not be interupted, try SIGKILL (-9) */
        console_log_message(sa,
                            _("Killing the seti@home client with 'waitpid'"));
        waitpid(pid, &status, WNOHANG);
        if(!is_seti_process(pid))
        {
            /* succesfully killed setiathome with a mean
             * kill signal.  we need to remove the lock file.
             */
            unlink_seti_lock_file(sa);
        }
        else
        {
            /* maybe popup a dialog box here?? */
            console_log_message(sa, _("Unable to kill setiathome process!"));
        }

    }
}

static void
read_pipe_callback(setiapplet * sa)
{
    gint        MAX_READ = 256;
    gint        bytes_read;
    gchar       temp_buf[MAX_READ];

    gchar      *full_msg = 0;

    memset(&temp_buf, 0, MAX_READ);
    if(sa->seti_output > -1)
    {
        while((bytes_read = read(sa->seti_output, temp_buf, MAX_READ - 1)) > 0)
        {
            if(!full_msg)
            {
                full_msg = g_strdup(temp_buf);
            }
            else
            {
                full_msg = g_strconcat(full_msg, temp_buf, NULL);
            }
            memset(&temp_buf, 0, MAX_READ);
        }
        if(full_msg)
        {
            //Log the message to the console
            console_log_message(sa, full_msg);
            //Check if the client output should be parsed/responded
            //to in some way.
            if(!sa->registered)
            {
                check_handle_output(full_msg, sa);
            }
            g_free(full_msg);
        }
    }
}

static gboolean
read_pipe_timeout(GIOChannel * source,
                  GIOCondition condition,
                  setiapplet * sa)
{
    read_pipe_callback(sa);
    return TRUE;
}

static void
close_channel(GIOChannel * source,
              setiapplet * sa)
{
    close(sa->seti_output);
    g_io_channel_close(source);
    g_io_channel_unref(source);
    g_source_remove(sa->watch_in);
    g_source_remove(sa->watch_hup);
}

static gboolean
close_pipe_callback(GIOChannel * source,
                    GIOCondition condition,
                    setiapplet * sa)
{
    close_channel(source, sa);
    return TRUE;
}

/*******************************************************************************
 *
 * start_seti( setiapplet *sa )
 *
 * This function will start the setiathome process
 *
 ******************************************************************************/
static void
start_seti(setiapplet * sa)
{
    int         pid, rc;
    gchar     **argsr;
    gchar      *seti_loc;
    gint        pipe_fd[2];
    gint        pipe_fd2[2];
    GIOChannel *io_channel;

    console_log_message(sa, _("Trying to start the seti@home client..."));
    /*Open a pipe which can be attached to the stdout of the seti client */
    pipe(pipe_fd);
    pipe(pipe_fd2);
    sa->seti_output = pipe_fd[0];
    sa->seti_input = pipe_fd2[1];
    fcntl(sa->seti_output, F_SETFL, O_NONBLOCK);

    pid = fork();
    if(pid == 0)
    {
        close(pipe_fd[0]);
        close(pipe_fd2[1]);
        dup2(pipe_fd2[0], STDIN_FILENO);
        dup2(pipe_fd[1], STDOUT_FILENO);

        argsr = get_extra_args(sa);
        /*
         * NOTE:we do not care about memory cleaning as this method is
         * called only after a fork and before an exec to start the seti
         * process.
         */
        chdir(sa->seti_dir);
        if(sa->separate_exe_dir)
        {
            seti_loc = g_strdup_printf("%s/setiathome", sa->seti_exe_dir);
        }
        else
        {
            seti_loc = g_strdup_printf("%s/setiathome", sa->seti_dir);
        }
        rc = execv(seti_loc, argsr);
        close(pipe_fd[1]);
        printf("could not start seti@home: errno=%d", rc);
        _exit(127);
    }
    else
    {
        close(pipe_fd[1]);
        //Set the new user thing to true - allow the parsing of the
        //initial client output
        sa->registered = FALSE;
    }

    /* because it takes a moment for setiathome to startup, we
     * need to sleep for a second and let it get going.  This will
     * allow setiathome to create it's lock file
     */
    sleep(1);
    /*We should really check to see if the seti client started... */
    /* Hook the IO callback into the main loop */
    io_channel = g_io_channel_unix_new(sa->seti_output);
    g_io_channel_init(io_channel);
    sa->watch_in =
        g_io_add_watch(io_channel, G_IO_IN, (GIOFunc) read_pipe_timeout, sa);
    sa->watch_hup =
        g_io_add_watch(io_channel, G_IO_HUP, (GIOFunc) close_pipe_callback, sa);
}

/*******************************************************************************
 * get_extra_args( setiapplet *sa )
 *
 * This method parses the extraparams member of the setiapplet structure to
 * retrieve any extra command line parameters for starting the seti client
 *
 *
 * Returns: An array of gchar* containing each of the extra parameters
 ******************************************************************************/
static gchar **
get_extra_args(setiapplet * sa)
{
    gchar      *nice_concat;

    if(sa->extra_params)
    {
        nice_concat =
            g_strdup_printf("setiathome -nice %d %s", sa->nice_value,
                            sa->extra_params);
    }
    else
    {
        nice_concat = g_strdup_printf("setiathome -nice %d", sa->nice_value);
    }

    if(sa->launch_xseti)
    {
        gchar *tmp = nice_concat;
        nice_concat = g_strdup_printf("%s %s", nice_concat, "-graphics");
        g_free(tmp);
    }
    return g_strsplit(nice_concat, " ", 19);
}

void start_xseti(gpointer user_data)
{
    int         pid, rc;
    setiapplet *sa = (setiapplet *)user_data;
    gchar *seti_loc;
    //Check the xsetiathome client is available
    const char *argv[] = {"xsetiathome", NULL};

    console_log_message(sa, _("Trying to start the xseti@home display..."));

    pid = fork();
    if(pid == 0)
    {
        chdir(sa->seti_dir);
        if(sa->separate_exe_dir)
        {
            seti_loc = g_strdup_printf("%s/xsetiathome", sa->seti_exe_dir);
        }
        else
        {
            seti_loc = g_strdup_printf("%s/xsetiathome", sa->seti_dir);
        }
        rc = execv(seti_loc,argv);
        printf("could not start xseti@home: errno=%d", rc);
        _exit(127);
    }
}

/*******************************************************************************
 *
 * start_stop_seti_cb( AppletWidget *widget, gpointer sa )
 *
 * This is the callback for the stop/start seti menu pick of the applet
 *
 ******************************************************************************/
void
start_stop_seti_cb(PanelApplet * widget,
                   gpointer sa)
{
    if(seti_status(sa) == SETI_RUNNING)
    {
        stop_seti(sa);
    }
    else
    {
        start_seti(sa);
    }

    /* Change the menu entry */
    //update_applet_menu();
}
