/*
 * Seti@Home console
 * (C) 2002 Jean-Michel Ardantz
 *
 * Author: Richard Kinder <r_kinder@yahoo.com>
 * Author: Jean-Michel Ardantz <jmardantz@ifrance.com>
 *
 * Changelog:
 * $Log: seti_console.c,v $
 * Revision 1.1.2.5  2003/03/05 05:57:55  r_kinder
 * GUI fixes, bug fixes.
 *
 * Revision 1.1.2.4  2003/01/30 22:32:15  r_kinder
 * Fixes for vertically aligned panels - shrink it to be only a radar.
 *
 * Revision 1.1.2.3  2003/01/23 23:37:28  r_kinder
 * Do a 'make indent' on the files. Remove legacy code that is no longer
 * pertinant.
 *
 * Revision 1.1.2.2  2003/01/23 23:09:11  r_kinder
 * Intercept the delete_event signal to prevent a crash on console close
 * (dangling pointer being accessed).
 *
 * Revision 1.1.2.1  2002/11/05 21:31:30  jmardantz
 * Gnome 2 port
 *
 *
 */

#include "config.h"
#include "seti_applet.h"
#include "seti_console.h"

#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>
#include <glade/glade.h>

// Defines
#define GLADE_FILE   GLADEDIR "/seti-applet.glade.glade"

// Prototypes of statics
static void on_SetiAppletConsole_ButtonClose_clicked(GtkButton *,
                                                     gpointer);
static void on_SetiAppletConsole_ButtonClear_clicked(GtkButton *,
                                                     gpointer);
static void on_SetiAppletConsole_ButtonCopy_clicked(GtkButton *,
                                                    gpointer);
static gboolean replace_existing_file(GtkWindow * parent,
                                      const gchar * file_name);
static void console_save(GtkWidget * window,
                         gpointer user_data);
static void on_SetiAppletConsole_ButtonSave_clicked(GtkButton *,
                                                    gpointer);


/*
 * console_log_message
 * Add a message in the Console texte area
 *
 * Param: sa the setiapplet structure for the applet in use
 * Param: message the message to display in the dialog
*/
void
console_log_message(setiapplet * sa,
                    gchar * message)
{
    gint        msg_len = strlen(message);
    GtkTextIter iter;
    gchar      *msg = (gchar *)0;

    //If no buffer exists, create one and copy the first string in.
    if(!sa->seti_console_buffer)
    {
        if(message[msg_len - 1] == '\n')
        {
            sa->seti_console_buffer = g_strdup_printf("%s", message);
            msg = g_strdup(message);
        }
        else
        {
            sa->seti_console_buffer = g_strdup_printf("%s\n", message);
            msg = g_strdup_printf("%s\n",message);
        }
    }
    //String exists, concatenate with this message.
    else
    {
        if(message[msg_len - 1] == '\n')
        {
            gchar      *tmp = sa->seti_console_buffer;
            msg = g_strdup(message);
            sa->seti_console_buffer =
                g_strconcat(sa->seti_console_buffer, message, NULL);
            g_free(tmp);
        }
        else
        {
            gchar      *tmp = sa->seti_console_buffer;
            msg = g_strdup_printf("%s\n",message);
            sa->seti_console_buffer =
                g_strconcat(sa->seti_console_buffer, message, "\n\0", NULL);
            g_free(tmp);
        }
    }
    //The console is open - guaranteed
    if(sa->applet_gui->seti_console_text)
    {
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &iter);
        gtk_text_buffer_insert(GTK_TEXT_BUFFER
                               (sa->applet_gui->seti_console_text), &iter, msg,
                               strlen(msg));

    }
    g_free(msg);
}


/*
 * console_clear
 * Clear the Console text buffer area
 *
 * Param: sa the setiapplet structure for the applet in use
*/
void
console_clear(setiapplet * sa)
{
    GtkTextIter start_iter;
    GtkTextIter end_iter;

    if(sa->seti_console_buffer)
    {
        g_free(sa->seti_console_buffer);
        sa->seti_console_buffer = NULL;
    }
    if(sa->applet_gui->seti_console_text)
    {
        gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER
                                       (sa->applet_gui->seti_console_text),
                                       &start_iter);
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &end_iter);
        gtk_text_buffer_delete(GTK_TEXT_BUFFER
                               (sa->applet_gui->seti_console_text), &start_iter,
                               &end_iter);
    }
}


/*
 * replace_existing_file
 * Ask for replacing an existing file
 *
 * Param: parent window
 * Param: file_name
 */
static      gboolean
replace_existing_file(GtkWindow * parent,
                      const gchar * file_name)
{
    GtkWidget  *msgbox;
    gint        ret;

    msgbox = gtk_message_dialog_new(parent,
                                    GTK_DIALOG_MODAL |
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
                                    _("A file named ''%s'' already exists.\n"
                                      "Do you want to replace it with the "
                                      "one you are saving?"), file_name);
    ret = gtk_dialog_run(GTK_DIALOG(msgbox));
    gtk_widget_destroy(msgbox);

    return (ret == GTK_RESPONSE_YES);
}


/*
 * console_save
 * Save the text buffer area to disk
 *
 * Param: the window
 * Param: sa the setiapplet structure for the applet in use
 */
static void
console_save(GtkWidget * window,
             gpointer user_data)
{
    const gchar *file_name;
    gchar      *message;
    gchar      *data_text;
    GtkTextIter start_iter;
    GtkTextIter end_iter;
    gsize       nb_written;
    GError     *gerr = NULL;
    GIOChannel *ioc;
    setiapplet *sa = (setiapplet *) user_data;

    // Getting file name
    file_name =
        gtk_file_selection_get_filename(GTK_FILE_SELECTION
                                        (sa->applet_gui->
                                         seti_console_file_sel));

    if(g_file_test(file_name, G_FILE_TEST_EXISTS))
        if(!replace_existing_file(GTK_WINDOW(window), file_name))
            return;

    // Open File in Write mode
    ioc = g_io_channel_new_file(file_name, "w", &gerr);
    if(!ioc)
    {
        message =
            g_strdup_printf
            ("There was an error trying to open %s in write mode.\n\n%s\n",
             file_name, gerr->message);
        gui_error(message);
        g_error_free(gerr);
    }
    else
    {
        // Get the data_text from the console
        gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER
                                       (sa->applet_gui->seti_console_text),
                                       &start_iter);
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &end_iter);
        data_text =
            gtk_text_buffer_get_text(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &start_iter, &end_iter, FALSE);

        // Save the data
        g_io_channel_write_chars(ioc, data_text, strlen(data_text), &nb_written,
                                 &gerr);
        if(gerr != NULL)
        {
            message =
                g_strdup_printf("There was an error trying to save %s.\n\n%s\n",
                                file_name, gerr->message);
            gui_error(message);
            g_error_free(gerr);
        }
        g_io_channel_unref(ioc);
    }

    gtk_widget_destroy(sa->applet_gui->seti_console_file_sel);
}

static      gboolean on_SetiAppletConsole_WindowDelete_event(GtkWidget * widget,
                                                             GdkEvent * event,
                                                             gpointer user_data)
{
    on_SetiAppletConsole_ButtonClose_clicked(NULL, user_data);
}

 /*
  * on_SetiAppletConsole_ButtonClose_clicked
  * Hide the Console Dialog Box
  *
  * Param: the button
  * Param: sa the setiapplet structure for the applet in use
  */
static void on_SetiAppletConsole_ButtonClose_clicked(GtkButton * button,
                                                     gpointer user_data)
{
    setiapplet *sa = (setiapplet *) user_data;
    if(sa->applet_gui->seti_console)
    {
        //gtk_widget_hide (sa->applet_gui->seti_console);
        gtk_widget_destroy(sa->applet_gui->seti_console);
        sa->applet_gui->seti_console = NULL;
        sa->applet_gui->seti_console_text = NULL;
        sa->applet_gui->seti_console_file_sel = NULL;
    }
}


/*
 * on_SetiAppletConsole_ButtonClear_clicked
 * Clear the Console text buffer area
 *
 * Param: the button
 * Param: sa the setiapplet structure for the applet in use
 */
static void on_SetiAppletConsole_ButtonClear_clicked(GtkButton * button,
                                                     gpointer user_data)
{
    setiapplet *sa = (setiapplet *) user_data;

    console_clear(sa);
}


/*
 * on_SetiAppletConsole_ButtonCopy_clicked
 * Copy the Console text buffer area to clipboard
 *
 * Param: the button
 * Param: sa the setiapplet structure for the applet in use
 */
static void on_SetiAppletConsole_ButtonCopy_clicked(GtkButton * button,
                                                    gpointer user_data)
{
    GtkTextIter start_iter;
    GtkTextIter end_iter;
    setiapplet *sa = (setiapplet *) user_data;

    if(sa->applet_gui->seti_console_text)
    {
        // Select the whole text, copy and unselect
        // May be it is not the easiest way to do this copy
        gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER
                                       (sa->applet_gui->seti_console_text),
                                       &start_iter);
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &end_iter);

        gtk_text_buffer_place_cursor(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &end_iter);
        gtk_text_buffer_move_mark(GTK_TEXT_BUFFER
                                  (sa->applet_gui->seti_console_text),
                                  gtk_text_buffer_get_mark(GTK_TEXT_BUFFER
                                                           (sa->applet_gui->
                                                            seti_console_text),
                                                           "selection_bound"),
                                  &start_iter);

        gtk_text_buffer_copy_clipboard(GTK_TEXT_BUFFER
                                       (sa->applet_gui->seti_console_text),
                                       gtk_clipboard_get
                                       (GDK_SELECTION_CLIPBOARD));

        gtk_text_buffer_move_mark(GTK_TEXT_BUFFER
                                  (sa->applet_gui->seti_console_text),
                                  gtk_text_buffer_get_mark(GTK_TEXT_BUFFER
                                                           (sa->applet_gui->
                                                            seti_console_text),
                                                           "selection_bound"),
                                  &end_iter);
    }
}


/*
 * on_SetiAppletConsole_ButtonSave_clicked
 * Save the Console text buffer area to disk
 *
 * Param: the button
 * Param: sa the setiapplet structure for the applet in use
 */
static void on_SetiAppletConsole_ButtonSave_clicked(GtkButton * button,
                                                    gpointer user_data)
{
    setiapplet *sa = (setiapplet *) user_data;

    sa->applet_gui->seti_console_file_sel =
        gtk_file_selection_new(_("Save Console Log"));

    gtk_widget_set_sensitive(GTK_FILE_SELECTION
                             (sa->applet_gui->seti_console_file_sel)->file_list,
                             TRUE);

    g_signal_connect(G_OBJECT
                     (GTK_FILE_SELECTION
                      (sa->applet_gui->seti_console_file_sel)->ok_button),
                     "clicked", G_CALLBACK(console_save), sa);

    g_signal_connect_swapped(G_OBJECT
                             (GTK_FILE_SELECTION
                              (sa->applet_gui->seti_console_file_sel)->
                              cancel_button), "clicked",
                             G_CALLBACK(gtk_widget_destroy),
                             G_OBJECT(sa->applet_gui->seti_console_file_sel));

    gtk_window_set_modal(GTK_WINDOW(sa->applet_gui->seti_console_file_sel),
                         TRUE);
    gtk_widget_show(sa->applet_gui->seti_console_file_sel);
}


/*
 * create_SetiAppletConsole
 * Create the console
 *
 * Param: sa the setiapplet structure for the applet in use
 */
void
create_SetiAppletConsole(setiapplet * sa)
{
    GtkWidget  *SetiAppletConsole_TextLog;
    GtkWidget  *SetiAppletConsole_ButtonClear;
    GtkWidget  *SetiAppletConsole_ButtonCopy;
    GtkWidget  *SetiAppletConsole_ButtonSave;
    GtkWidget  *SetiAppletConsole_ButtonClose;
    GladeXML   *gui = NULL;

    // Load the druid glade file
    gui = glade_xml_new(GLADE_FILE, "SetiAppletConsole", NULL);
    if(!gui)
        gui_warning("Error loading glade file");

    //gui_warning("Some message");

    // Save widget to console structure
    sa->applet_gui->seti_console =
        glade_xml_get_widget(gui, "SetiAppletConsole");
    SetiAppletConsole_TextLog =
        glade_xml_get_widget(gui, "SetiAppletConsole_TextLog");
    sa->applet_gui->seti_console_text =
        gtk_text_view_get_buffer
                   (GTK_TEXT_VIEW(SetiAppletConsole_TextLog));

    // Save some usefuls widgets
    SetiAppletConsole_ButtonClear =
        glade_xml_get_widget(gui, "SetiAppletConsole_ButtonClear");
    SetiAppletConsole_ButtonCopy =
        glade_xml_get_widget(gui, "SetiAppletConsole_ButtonCopy");
    SetiAppletConsole_ButtonSave =
        glade_xml_get_widget(gui, "SetiAppletConsole_ButtonSave");
    SetiAppletConsole_ButtonClose =
        glade_xml_get_widget(gui, "SetiAppletConsole_ButtonClose");

    // Callback connections
    g_signal_connect(G_OBJECT(SetiAppletConsole_ButtonClear),
                     "clicked",
                     G_CALLBACK(on_SetiAppletConsole_ButtonClear_clicked), sa);
    g_signal_connect(G_OBJECT(SetiAppletConsole_ButtonCopy), "clicked",
                     G_CALLBACK(on_SetiAppletConsole_ButtonCopy_clicked), sa);
    g_signal_connect(G_OBJECT(SetiAppletConsole_ButtonSave), "clicked",
                     G_CALLBACK(on_SetiAppletConsole_ButtonSave_clicked), sa);
    g_signal_connect(G_OBJECT(SetiAppletConsole_ButtonClose), "clicked",
                     G_CALLBACK(on_SetiAppletConsole_ButtonClose_clicked), sa);
    gtk_signal_connect(GTK_OBJECT(sa->applet_gui->seti_console), "delete_event",
                       GTK_SIGNAL_FUNC(on_SetiAppletConsole_WindowDelete_event),
                       sa);
    if(sa->seti_console_buffer)
    {
        GtkTextIter iter;
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER
                                     (sa->applet_gui->seti_console_text),
                                     &iter);
        gtk_text_buffer_insert(GTK_TEXT_BUFFER
                               (sa->applet_gui->seti_console_text), &iter,
                               sa->seti_console_buffer,
                               strlen(sa->seti_console_buffer));
    }
}
