/*
 * seti@home client integrated into panel
 * (C) 2000,2001,2002 Richard Kinder
 *
 * Author: Richard Kinder <r_kinder@yahoo.com>
 *
 * Date: 8/1/00
 *
 * Changelog:
 * $Log: seti_applet.c,v $
 * Revision 1.15.2.1.2.20  2003/03/06 06:03:55  r_kinder
 * Misc GUI fixes.
 *
 * Revision 1.15.2.1.2.19  2003/03/06 00:53:29  r_kinder
 * Implement current CPU time feature.
 *
 * Revision 1.15.2.1.2.18  2003/03/05 05:57:54  r_kinder
 * GUI fixes, bug fixes.
 *
 * Revision 1.15.2.1.2.17  2003/03/05 02:51:05  r_kinder
 * Border applet option implemented (feature request 690359)
 *
 * Revision 1.15.2.1.2.16  2003/03/04 02:50:10  r_kinder
 * Fixups for RH 8.0 rpm build, creation of a new skin (nestor).
 *
 * Revision 1.15.2.1.2.15  2003/01/30 22:32:14  r_kinder
 * Fixes for vertically aligned panels - shrink it to be only a radar.
 *
 * Revision 1.15.2.1.2.14  2003/01/28 08:11:22  r_kinder
 * Minor bug fixes, change the default skin look so it looks good on small
 * panels (when fixed size with only the radar is selected).
 *
 * Revision 1.15.2.1.2.13  2003/01/23 23:37:28  r_kinder
 * Do a 'make indent' on the files. Remove legacy code that is no longer
 * pertinant.
 *
 * Revision 1.15.2.1.2.12  2003/01/14 00:30:45  r_kinder
 * Various fixes.
 *
 * Revision 1.15.2.1.2.11  2002/11/29 07:08:32  r_kinder
 * Port to GNOME 2.x intermediate checkin.
 *
 * Revision 1.15.2.1.2.10  2002/11/05 21:31:30  jmardantz
 * Gnome 2 port
 *
 * Revision 1.15.2.1.2.9  2002/04/21 23:15:30  r_kinder
 * Added more default values for initial properties.
 *
 * Revision 1.15.2.1.2.8  2002/04/09 11:26:12  r_kinder
 * Fixes for crash on removal from panel, misc warning fixes.
 *
 * Revision 1.15.2.1.2.7  2002/04/07 23:44:36  r_kinder
 * Functionally almost complete version - still need to work on the graphics, but
 * the splitting of UI from the main code is almost complete (still debating
 * whether to put the properties interface in a new module). Works.
 *
 * Revision 1.15.2.1.2.6  2002/03/28 22:43:31  r_kinder
 * Intermediate checkin - will compile, build, run, but the graphics will be a
 * bit screwy.
 *
 * Revision 1.15.2.1.2.5  2002/03/25 11:10:55  r_kinder
 * Properties now save and restore correctly. Intermediate checkin.
 *
 * Revision 1.15.2.1.2.4  2002/03/07 11:39:33  r_kinder
 * Intermediate commit in case my laptop blows up.
 *
 * Now works quite well with the panel (no start/stop yet). Looking into how
 * to use the gconf stuff to store data.
 *
 * Revision 1.15.2.1.2.3  2002/02/28 21:01:48  r_kinder
 * Major changes to clean up cruft.
 *
 * This version actually works in gnome-panel-2!!!
 *
 * Revision 1.15.2.1.2.2  2002/02/25 22:58:54  r_kinder
 * First version that builds (NOTE: WILL NOT RUN!!!)
 *
 * Revision 1.15.2.1.2.1  2002/02/24 10:24:21  r_kinder
 * Update to CVS in case my laptop blows up.
 *
 * Revision 1.15.2.1  2002/02/11 12:10:27  r_kinder
 * Re-merge fixes into stable branch.
 *
 * Revision 1.16  2001/08/07 00:29:48  r_kinder
 * Fixed bug 448605 - no green radar when entering user details for the
 * first time.
 *
 * Revision 1.15  2001/06/28 03:00:48  r_kinder
 * Fix bug where changing from panel of one size to another (same
 * orientation) does not force the applet to change sizes. General code
 * cleanup.
 *
 * Revision 1.14  2001/06/13 03:02:56  r_kinder
 * Changes to catch error condition on user submitting details to seti@home
 * server.
 *
 * Revision 1.13  2001/06/12 07:27:20  r_kinder
 * Fix all of the known bugs from 0.3.4-2, add the new user detection code
 * and gui to set up seti@home accounts. Clean up some of the code.
 *
 */

#include "config.h"
#include "seti_applet.h"
#include "seti_control.h"
#include "default-skin.h"
#include "nestor-skin.h"
#include "seti_druid.h"
#include "seti_console.h"
#include "seti_properties.h"

#include <sys/stat.h>
#include <fcntl.h>

#include <panel-applet.h>
#include <panel-applet-gconf.h>

#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-window-icon.h>

//Defines

//Filenames for state and user info
#define USER_INFO_SAH              "user_info.sah"
#define USER_INFO_TXT              "user_info.txt"
#define STATE_SAH                  "state.sah"
#define STATE_TXT                  "state.txt"

#define FILE_SAH                   0
#define FILE_TXT                   1
#define FILE_SIZE                  2000

#define RADAR_DX                   44.0
#define RADAR_DY                   44.0

//1300ms between pixmap changes
#define ATTENTION_INTERVAL         1300
#define NO_DBL_CLICK_INTERVAL      1000

#define X_TO_Y_RATIO 3.00

//For the detecting of a new user clicking start for the first time.
#define HANDLE_STATE_STR_1         "1 to set up a new"
#define HANDLE_STATE_STR_NV        "New software version"
#define HANDLE_STATE_STR_NO_SERVER "Server host unknown"
#define HANDLE_STATE_STR_EOP       "Scanning data file"

//For the GConf schemas keys
#define GCONF_KEY_RUN_DIR "seti_run_dir"
#define GCONF_KEY_VERSION "seti_key_version"
#define GCONF_KEY_DISPLAY_PERCENT_INFO "display_percent_info"
#define GCONF_KEY_DISPLAY_SPIKE_INFO "display_spike_info"
#define GCONF_KEY_DISPLAY_GAUSSIAN_INFO "display_gaussian_info"
#define GCONF_KEY_DISPLAY_NUM_UNITS_INFO "display_num_units_info"
#define GCONF_KEY_DISPLAY_CPU_TIME_INFO "display_cpu_time_info"
#define GCONF_KEY_DISPLAY_CPU_CURRENT_TIME_INFO "display_cpu_current_time_info"
#define GCONF_KEY_AUTO_LAUNCH "auto_launch"
#define GCONF_KEY_NICE_VALUE "nice_value"
#define GCONF_KEY_EXTRA_PARAMS "extra_params"
#define GCONF_KEY_GUI_CYCLE_INTERVAL "gui_cycle_interval"
#define GCONF_KEY_FILE_UPDATE_INTERVAL "file_update_interval"
#define GCONF_KEY_SEPARATE_EXE_DIR "seti_separate_exe_dir"
#define GCONF_KEY_XSETI_LAUNCH "xseti_launch"
#define GCONF_KEY_EXE_DIR "seti_exe_dir"
#define GCONF_KEY_AUTO_SIZE "GCONF_KEY_auto_size"
#define GCONF_KEY_DRAW_BORDER "draw_border"
#define GCONF_KEY_LOCK_XY "lock_xy"
#define GCONF_KEY_ONLY_RADAR "only_radar"
#define GCONF_KEY_X_VALUE "x_value"
#define GCONF_KEY_Y_VALUE "y_value"

//Prototypes of statics
static void setup_gaus_canvas_item(setiapplet *);
static void setup_spike_canvas_item(setiapplet *);
static void setup_percent_canvas_item(setiapplet *);
static void setup_units_canvas_item(setiapplet *);
static void setup_cputime_canvas_item(setiapplet *);
static void applet_change_orient(PanelApplet *,
                                 PanelAppletOrient,
                                 gpointer);
static void applet_change_pixel_size(PanelApplet *,
                                     int,
                                     gpointer);
static void applet_change_background(PanelApplet *,
                                     PanelAppletBackgroundType,
                                     GdkColor *,
                                     const gchar *,
                                     setiapplet *);
static void attention_callback(setiapplet *);
static void display_callback(setiapplet *);
static void state_file_callback(setiapplet *);
static void user_file_callback(setiapplet *);
static void parse_state_file(setiapplet *,
                             char *);
static void parse_user_file(setiapplet *,
                            char *);
static void update_display(setiapplet *,
                           gint);
static void another_function(setiapplet *sa)
{
    gtk_timeout_remove(sa->start_stop_seti_id);
    sa->start_stop_seti_id = 0;
    start_stop_seti_cb(PANEL_APPLET(sa->applet_gui->applet), sa);
}

static void
start_stop_seti_timeout(setiapplet * sa)
{
    another_function(sa);
}

static gboolean
radar_event(GnomeCanvasItem * item,
            GdkEvent * event,
            gpointer data)
{
    setiapplet *sa = (setiapplet *) data;
    switch (event->type)
    {
        case GDK_2BUTTON_PRESS:
            //if(event->button.button == 1)
            {
                if(sa->launch_xseti)
                {
                    g_message("Starting xsetiathome");
                    start_xseti(sa);
                }
                if(sa->start_stop_seti_id)
                {
                    gtk_timeout_remove(sa->start_stop_seti_id);
                    sa->start_stop_seti_id = 0;
                }
                return TRUE;
            }
            break;
        case GDK_BUTTON_PRESS:
            if(event->button.button == 1)
            {
                //start_stop_seti_cb(PANEL_APPLET(sa->applet_gui->applet), sa);
                if(!sa->start_stop_seti_id)
                {
                    sa->start_stop_seti_id = gtk_timeout_add(NO_DBL_CLICK_INTERVAL, (GtkFunction) start_stop_seti_timeout, sa);
                }
                return TRUE;
            }
            break;
        default:
            return FALSE;
    }
    return FALSE;
}

/*
 * Load GConf preferences
 * This function load preferences from GConf seti_applet schema
 */
void
panel_applet_preferences_load(setiapplet * sa,
                              PanelApplet * applet)
{
    GError     *error = NULL;

    panel_applet_gconf_get_string(applet, GCONF_KEY_RUN_DIR, &error);
    if(error)
    {
        g_print("%s \n", error->message);
        g_error_free(error);
        error = NULL;
        g_message("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        g_print("%s \n", error->message);
        return;
    }
    if(panel_applet_gconf_get_int(applet, GCONF_KEY_VERSION, NULL) == 0)
    {
        g_message("applet not run before");
        return;
    }

    if(strlen(panel_applet_gconf_get_string(applet, GCONF_KEY_RUN_DIR, NULL)) == 0)
        sa->seti_dir = NULL;
    else
        sa->seti_dir = panel_applet_gconf_get_string(applet, GCONF_KEY_RUN_DIR, NULL);

    sa->value_mask[PERCENT] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_PERCENT_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->value_mask[SPIKE] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_SPIKE_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->value_mask[GAUSSIAN] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_GAUSSIAN_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->value_mask[UNITS] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_NUM_UNITS_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->value_mask[CPUTIME] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_CPU_TIME_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->value_mask[CPUTIMECURRENT] =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DISPLAY_CPU_CURRENT_TIME_INFO,
                                    NULL) ? VALUE_DISPLAY : VALUE_HIDE;

    sa->launch_on_start =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_AUTO_LAUNCH, NULL);
    sa->nice_value = panel_applet_gconf_get_int(applet, GCONF_KEY_NICE_VALUE, NULL);

    if(sa->extra_params != NULL)
        g_free(sa->extra_params);

    if(strlen(panel_applet_gconf_get_string(applet, GCONF_KEY_EXTRA_PARAMS, NULL)) == 0)
        sa->extra_params = NULL;
    else
        sa->extra_params =
            panel_applet_gconf_get_string(applet, GCONF_KEY_EXTRA_PARAMS, NULL);

    sa->update_interval =
        panel_applet_gconf_get_int(applet, GCONF_KEY_GUI_CYCLE_INTERVAL, NULL);
    sa->file_update_interval =
        panel_applet_gconf_get_int(applet, GCONF_KEY_FILE_UPDATE_INTERVAL, NULL);

    sa->separate_exe_dir =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_SEPARATE_EXE_DIR, NULL);
    sa->launch_xseti = 
        panel_applet_gconf_get_bool(applet, GCONF_KEY_XSETI_LAUNCH, NULL);
    if(sa->separate_exe_dir)
    {
        if(sa->seti_exe_dir != NULL)
            g_free(sa->seti_exe_dir);

        if(strlen(panel_applet_gconf_get_string(applet, GCONF_KEY_EXE_DIR, NULL)) == 0)
            sa->seti_exe_dir = NULL;
        else
            sa->seti_exe_dir =
                panel_applet_gconf_get_string(applet, GCONF_KEY_EXE_DIR, NULL);
    }

    sa->applet_gui->custom_size->auto_size =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_AUTO_SIZE, NULL);

    sa->applet_gui->custom_size->draw_border =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_DRAW_BORDER, NULL);

    sa->applet_gui->custom_size->lock_xy =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_LOCK_XY, NULL);

    sa->applet_gui->custom_size->only_radar =
        panel_applet_gconf_get_bool(applet, GCONF_KEY_ONLY_RADAR, NULL);

    sa->applet_gui->custom_size->x =
        panel_applet_gconf_get_int(applet, GCONF_KEY_X_VALUE, NULL);

    sa->applet_gui->custom_size->y =
        panel_applet_gconf_get_int(applet, GCONF_KEY_Y_VALUE, NULL);
}


/*
 * Save GConf preferences
 * This function save preferences to a GConf seti_applet schema
 */
void
panel_applet_preferences_save(setiapplet * sa,
                              PanelApplet * applet)
{
    panel_applet_gconf_set_string(applet, GCONF_KEY_RUN_DIR,
                                  sa->seti_dir == NULL ? "" : sa->seti_dir,
                                  NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DISPLAY_SPIKE_INFO,
                                sa->
                                value_mask[SPIKE] & VALUE_DISPLAY ? TRUE :
                                FALSE, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DISPLAY_GAUSSIAN_INFO,
                                sa->
                                value_mask[GAUSSIAN] & VALUE_DISPLAY ? TRUE :
                                FALSE, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DISPLAY_NUM_UNITS_INFO,
                                sa->
                                value_mask[UNITS] & VALUE_DISPLAY ? TRUE :
                                FALSE, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DISPLAY_CPU_TIME_INFO,
                                sa->
                                value_mask[CPUTIME] & VALUE_DISPLAY ? TRUE :
                                FALSE, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DISPLAY_CPU_CURRENT_TIME_INFO,
                                sa->
                                value_mask[CPUTIMECURRENT] & VALUE_DISPLAY ?
                                TRUE : FALSE, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_AUTO_LAUNCH, sa->launch_on_start, NULL);

    panel_applet_gconf_set_int(applet, GCONF_KEY_NICE_VALUE, sa->nice_value, NULL);
    panel_applet_gconf_set_int(applet, GCONF_KEY_VERSION, 212, NULL);

    panel_applet_gconf_set_string(applet, GCONF_KEY_EXTRA_PARAMS,
                                  sa->extra_params ==
                                  NULL ? "" : sa->extra_params, NULL);

    panel_applet_gconf_set_int(applet, GCONF_KEY_GUI_CYCLE_INTERVAL, sa->update_interval,
                               NULL);

    panel_applet_gconf_set_int(applet, GCONF_KEY_FILE_UPDATE_INTERVAL,
                               sa->file_update_interval, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_SEPARATE_EXE_DIR, sa->separate_exe_dir,
                                NULL);
    panel_applet_gconf_set_bool(applet, GCONF_KEY_XSETI_LAUNCH, sa->launch_xseti,
                                NULL);

    panel_applet_gconf_set_string(applet, GCONF_KEY_EXE_DIR,
                                  sa->seti_exe_dir ==
                                  NULL ? "" : sa->seti_exe_dir, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_AUTO_SIZE,
                                sa->applet_gui->custom_size->auto_size, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_DRAW_BORDER,
                                sa->applet_gui->custom_size->draw_border, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_LOCK_XY,
                                sa->applet_gui->custom_size->lock_xy, NULL);

    panel_applet_gconf_set_bool(applet, GCONF_KEY_ONLY_RADAR,
                                sa->applet_gui->custom_size->only_radar, NULL);

    panel_applet_gconf_set_int(applet, GCONF_KEY_X_VALUE, sa->applet_gui->custom_size->x,
                               NULL);

    panel_applet_gconf_set_int(applet, GCONF_KEY_Y_VALUE, sa->applet_gui->custom_size->y,
                               NULL);
}


/*
 * check_handle_output
 * This function basically handles all of the input/output which results
 * from a user having not run the seti@home client previously.
 * It is implemented as a very simplistic state machine which looks for
 * specific strings in the client output and prompts the user for input.
 *
 * Param: message the string received from the server
 * Param: sa the setiapplet structure for the applet in use
 */
void
check_handle_output(gchar * message,
                    setiapplet * sa)
{
    // Lets do things as simply as possible...
    switch (sa->new_user_state)
    {
        case HANDLE_STATE_1:
            // A case to ignore and try again is when the string 'Server host unknown'
            // is returned. In this case, display a user error and stop seti@home
            if(strstr(message, HANDLE_STATE_STR_NO_SERVER) != NULL)
            {
                gui_error(_
                          ("Could not contact the seti@home server, try again later."));
                start_stop_seti_cb(PANEL_APPLET(sa->applet_gui->applet), sa);
            }
            else if(strstr(message, HANDLE_STATE_STR_1) != NULL)
            {
                // The initial '1 to ...; 2 to ...' string
                gui_new_user_druid(sa);
            }
            // We are finished with the every pipe write parsing once the 'Scanning
            // data file' message is received. After this point, parse no more!
            else if(strstr(message, HANDLE_STATE_STR_EOP) != NULL)
            {
                sa->registered = TRUE;
                check_change_pixmap(sa);
            }
            break;
        default:
            sa->new_user_state = HANDLE_STATE_1;
    }
}


/*
 * gui_warning
 * Create a popup dialog box indicating a warning
 *
 * Param: message the message to display in the dialog
 */
void
gui_warning(gchar * message)
{
    GtkWidget  *warning = NULL;

    warning = gtk_message_dialog_new(NULL,
                                     0,
                                     GTK_MESSAGE_INFO, GTK_BUTTONS_OK, message);
    gtk_dialog_run(GTK_DIALOG(warning));
    gtk_widget_destroy(warning);
}


/*
 * gui_error
 * Create a popup dialog box indicating an error
 *
 * Param: message the message to display in the dialog
 */
void
gui_error(gchar * message)
{
    GtkWidget  *error = NULL;

    error = gtk_message_dialog_new(NULL,
                                   0,
                                   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, message);
    gtk_dialog_run(GTK_DIALOG(error));
    gtk_widget_destroy(error);
}


static void
help_cb(BonoboUIComponent * uic,
        setiapplet * sa,
        const gchar * verbname)
{
#ifdef FIXME
    static GnomeHelpMenuEntry help_entry = { NULL, "index.html" };
    help_entry.name = gnome_app_id;
    gnome_help_display(NULL, &help_entry);
#endif
}


static void
console_cb(BonoboUIComponent * uic,
           gpointer user_data,
           const gchar * verbname)
{
    setiapplet *sa = (setiapplet *) user_data;

    // Create the console only if not already created
    if(sa->applet_gui->seti_console == NULL)
    {
        create_SetiAppletConsole(sa);

        gtk_widget_show(sa->applet_gui->seti_console);
    }
    else
    {
        gtk_widget_show(sa->applet_gui->seti_console);
    }
}


static void
refresh_tool_tip(setiapplet * sa)
{
    gchar      *skin_tooltip =
        sa->applet_gui->current_skin->tooltip(sa->applet_gui->current_skin->private, sa->current_unit_info,
                                              sa->user_info);

    if(!skin_tooltip)
    {
        skin_tooltip = "No information currently available";
    }
    gtk_tooltips_set_tip(sa->applet_gui->tip_text,
                         GTK_WIDGET(sa->applet_gui->the_canvas), skin_tooltip,
                         "SETI state");
    gtk_tooltips_enable(sa->applet_gui->tip_text);
    g_free(skin_tooltip);
}

static void
applet_change_pixel_size(PanelApplet * widget,
                         int size,
                         gpointer data)
{
    setiapplet *sa = (setiapplet *) data;
    sa->applet_gui->custom_size->panel_size = size;
    check_change_applet(widget, data);
}

static void
applet_change_background(PanelApplet               *applet,
                         PanelAppletBackgroundType  type,
                         GdkColor                  *color,
                         const gchar               *pixmap,
                         setiapplet                *sa)
{
    if(type == PANEL_NO_BACKGROUND)
    {
        GtkRcStyle *rc_style = gtk_rc_style_new();
        gtk_widget_modify_style(sa->applet_gui->the_canvas,rc_style);
        g_object_unref(rc_style);
    }
    else if(type == PANEL_COLOR_BACKGROUND)
    {
        gtk_widget_modify_bg(sa->applet_gui->the_canvas, GTK_STATE_NORMAL, color);
    }
    else
    {
        //??? What do we do here?
        g_message("Unsupported background change...");
    }
}


void
check_change_applet(PanelApplet * widget,
                    gpointer data)
{
    setiapplet *sa = (setiapplet *) data;
    GnomeCanvasGroup *group;
    //gdouble image_size = RADAR_DX;
    static gint prev_size = -1;
    static gint prev_auto_size = -1;
    static gint prev_only_radar = -1;
    static gint prev_x = -1;
    static gint prev_y = -1;
    static PanelAppletOrient prev_orient;
    short       new_dx, new_dy;

    if(prev_size == -1)
    {
        prev_size = sa->applet_gui->custom_size->panel_size;
        prev_orient = sa->applet_gui->custom_size->orientation;
    }

    if(prev_auto_size == -1)
        prev_auto_size = (gint) sa->applet_gui->custom_size->auto_size;

    if(prev_only_radar == -1)
        prev_only_radar = (gint) sa->applet_gui->custom_size->only_radar;

    if(prev_x == -1)
        prev_x = (gint) sa->applet_gui->custom_size->x;

    if(prev_x == -1)
        prev_y = (gint) sa->applet_gui->custom_size->y;

    // Bug out if nothing has changed.
    /*
    if((prev_size == sa->applet_gui->custom_size->panel_size) &&
       (prev_orient == sa->applet_gui->custom_size->orientation) &&
       (prev_auto_size == (gint) sa->applet_gui->custom_size->auto_size) &&
       (prev_only_radar == (gint) sa->applet_gui->custom_size->only_radar) &&
       (prev_x == (gint) sa->applet_gui->custom_size->x) &&
       (prev_y == (gint) sa->applet_gui->custom_size->y))
    {
        return;
    }
    */

    gtk_widget_hide(sa->applet_gui->the_canvas);
    gtk_widget_hide(sa->applet_gui->frame);
    if(!sa->applet_gui->custom_size->draw_border)
    {
        gtk_frame_set_shadow_type(GTK_FRAME(sa->applet_gui->frame), GTK_SHADOW_NONE);
    }
    else
    {
        gtk_frame_set_shadow_type(GTK_FRAME(sa->applet_gui->frame), GTK_SHADOW_OUT);

    }
    group = gnome_canvas_root(GNOME_CANVAS(sa->applet_gui->the_canvas));
    gtk_tooltips_disable(sa->applet_gui->tip_text);

/*
  if (sa->applet_gui->custom_size->panel_size < 48.0)
    {
      image_size = 14.0;
    }
*/

    // The way this works is:
    // 1) Work out whether we need a new version of the radar dish
    // 2) If so, remove existing, get new one from the skin
    // 3) Work out whether we need a new version of the title
    // 4) If so, remove existing, get new one from the skin

    //Work out the applet size first
    if(sa->applet_gui->custom_size->auto_size)
    {
        if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
           || sa->applet_gui->custom_size->orientation ==
           PANEL_APPLET_ORIENT_RIGHT)
        {
            // Look at the panel size - this determines the x/y values.
            new_dx = sa->applet_gui->custom_size->panel_size - 4;
            //new_dy = new_dx * X_TO_Y_RATIO;
            new_dy = new_dx;
        }
        else
        {
            new_dy = sa->applet_gui->custom_size->panel_size - 4;
            new_dx = new_dy * X_TO_Y_RATIO;
        }
    }
    else
    {
        new_dx = sa->applet_gui->custom_size->x-4;
        new_dy = sa->applet_gui->custom_size->y-4;
    }
    gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(sa->applet_gui->the_canvas),
                                     new_dy / 128.0);
    gtk_widget_set_usize(sa->applet_gui->the_canvas, new_dx, new_dy);
    if(sa->applet_gui->custom_size->auto_size)
    {
        gnome_canvas_set_scroll_region(GNOME_CANVAS
                                   (sa->applet_gui->the_canvas), 0, 0,
                                   128, 128);
    }

    gtk_widget_queue_resize(sa->applet_gui->the_canvas);
    gtk_widget_queue_resize(sa->applet_gui->the_canvas->parent);
    gtk_widget_queue_resize(sa->applet_gui->the_canvas->parent->parent);
    gtk_widget_queue_resize(sa->applet_gui->frame);
    gtk_widget_queue_resize(sa->applet_gui->frame->parent);
    gtk_widget_queue_resize(sa->applet_gui->frame->parent->parent);
    gtk_widget_show(sa->applet_gui->the_canvas);
    gtk_widget_show(sa->applet_gui->frame);

    // Get rid of the title
    if(sa->applet_gui->title_text_item != NULL)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->title_text_item));
        sa->applet_gui->title_text_item = 0;
    }

    // Grab a new one from the skin.
    sa->applet_gui->title_text_item =
        sa->applet_gui->current_skin->title(sa->applet_gui->current_skin->private, group, sa->applet_gui->custom_size);

    // Get rid of red/green/grey radars
    if(sa->applet_gui->radar_red_item != NULL)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->radar_red_item));
        sa->applet_gui->radar_red_item = 0;
    }
    if(sa->applet_gui->radar_green_item != NULL)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->radar_green_item));
        sa->applet_gui->radar_green_item = 0;
    }
    if(sa->applet_gui->radar_grey_item != NULL)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->radar_grey_item));
        sa->applet_gui->radar_grey_item = 0;
    }

    sa->applet_gui->radar_red_item =
        sa->applet_gui->current_skin->radar_red(sa->applet_gui->current_skin->private, group,
                                                sa->applet_gui->custom_size);
    sa->applet_gui->radar_green_item =
        sa->applet_gui->current_skin->radar_green(sa->applet_gui->current_skin->private, group,
                                                  sa->applet_gui->custom_size);
    sa->applet_gui->radar_grey_item =
        sa->applet_gui->current_skin->radar_grey(sa->applet_gui->current_skin->private, group,
                                                 sa->applet_gui->custom_size);
    g_signal_connect(G_OBJECT(sa->applet_gui->radar_red_item),
                         "event", G_CALLBACK(radar_event), sa);
    g_signal_connect(G_OBJECT(sa->applet_gui->radar_green_item),
                         "event", G_CALLBACK(radar_event), sa);

    // Update the local statics so we know whats happening the next time round.
    prev_size = sa->applet_gui->custom_size->panel_size;
    prev_orient = sa->applet_gui->custom_size->orientation;
    prev_auto_size = (gint) sa->applet_gui->custom_size->auto_size;
    prev_only_radar = (gint) sa->applet_gui->custom_size->only_radar;
    prev_x = (gint) sa->applet_gui->custom_size->x;
    prev_y = (gint) sa->applet_gui->custom_size->y;

    re_read_state_file(sa);
    re_read_user_file(sa);
    update_display(sa, 1);
    start_callback(sa);
}


static void
applet_change_orient(PanelApplet * widget,
                     PanelAppletOrient or,
                     gpointer data)
{
    setiapplet *sa = (setiapplet *) data;
    sa->applet_gui->custom_size->orientation = or;
    sa->applet_gui->custom_size->panel_size =
        panel_applet_get_size(PANEL_APPLET(sa->applet_gui->applet));
    if(or == PANEL_APPLET_ORIENT_LEFT || or == PANEL_APPLET_ORIENT_RIGHT)
    {
        sa->applet_gui->custom_size->only_radar = TRUE;
    }
    else
    {
        sa->applet_gui->custom_size->only_radar = panel_applet_gconf_get_bool(sa->applet_gui->applet, GCONF_KEY_ONLY_RADAR, NULL);
    }
    check_change_applet(widget, data);
}


/*
 * Display the about box
 */
static void
about_cb(BonoboUIComponent * uic,
         gpointer user_data,
         const gchar * verbname)
{
    GtkWidget  *about;
    GdkPixbuf  *pixbuf = NULL;
    gchar      *file;

    static const gchar *authors[] = {
        "Richard Kinder <r_kinder@yahoo.com>",
        "Jean-Michel Ardantz <jmardantz@ifrance.com>",
        "Matt Herbert <mherbert@bit-net.com>",
        NULL
    };

    const gchar *documenters[] = {
        NULL
    };

    // Note to translators: put here your name (and address) so it will shop up in the "about" box
    const gchar *translator_credits = _("translator_credits");

    file =
        gnome_program_locate_file(NULL, GNOME_FILE_DOMAIN_PIXMAP,
                                  "seti_applet/skins/nestor/seti_on.png",
                                  TRUE, NULL);
    pixbuf = gdk_pixbuf_new_from_file(file, NULL);
    g_free(file);

    about = gnome_about_new(_("seti@home applet"), VERSION,
                            "(C) 2000 - 2003",
                            _
                            ("An applet that shows the status of the seti@home client within the panel."),
                            authors, documenters, strcmp(translator_credits,
                                                         "translator_credits")
                            != 0 ? translator_credits : NULL, pixbuf);
    if(pixbuf)
        gdk_pixbuf_unref(pixbuf);

    gtk_window_set_wmclass(GTK_WINDOW(about), "seti@home", "seti@home");
    gnome_window_icon_set_from_file(GTK_WINDOW(about),
                                    GNOME_ICONDIR
                                    "/seti_applet/skins/nestor/seti_on_20.png");

    g_signal_connect(G_OBJECT(about), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &about);

    gtk_widget_show(about);
    return;
}


gchar      *
re_read_file(gchar * full_path)
{
    int         fd;
    int         stat_rv;
    gchar      *read_buf = g_malloc0(FILE_SIZE);
    struct stat *stat_buf = g_malloc0(sizeof(struct stat));

    /* Get file stats, check it exists, has size > 0 */
    stat_rv = stat(full_path, stat_buf);
    if(stat_rv == -1)
    {
        g_free(read_buf);
        read_buf = 0;
    }
    else
    {
        if((stat_buf->st_size) == 0)
        {
            g_free(read_buf);
            read_buf = 0;
        }
        else
        {

            fd = open(full_path, O_RDONLY);
            if(fd != -1)
            {
                int         num_bytes_read;
                num_bytes_read = read(fd, read_buf, FILE_SIZE);
                close(fd);
            }
            else
            {
                g_free(read_buf);
                read_buf = 0;
            }
        }
    }
    g_free(stat_buf);
    return read_buf;
}


void
re_read_state_file(setiapplet * sa)
{
    gchar      *file_contents;
    gchar      *full_path;
    /*Try the new seti state.sah file name */
    full_path = g_strdup_printf("%s/%s", sa->seti_dir, STATE_SAH);
    file_contents = re_read_file(full_path);
    /*Try the old seti state.txt file name */
    if(!file_contents)
    {
        g_free(full_path);
        full_path = g_strdup_printf("%s/%s", sa->seti_dir, STATE_TXT);
        file_contents = re_read_file(full_path);
        sa->file_version = FILE_TXT;
    }
    parse_state_file(sa, file_contents);
    g_free(full_path);
    /*file_contents may be a null pointer - re_read_file returns null if
     * the open file returned an error. Only free the memory if we have
     * a clean open file*/
    if(file_contents)
    {
        g_free(file_contents);
    }
}


void
re_read_user_file(setiapplet * sa)
{
    gchar      *file_contents;
    gchar      *full_path;
    full_path = g_strdup_printf("%s/%s", sa->seti_dir, USER_INFO_SAH);
    file_contents = re_read_file(full_path);
    if(!file_contents)
    {
        g_free(full_path);
        full_path = g_strdup_printf("%s/%s", sa->seti_dir, USER_INFO_TXT);
        file_contents = re_read_file(full_path);
    }
    parse_user_file(sa, file_contents);
    g_free(full_path);
    /*As for re_read_state_file - dont try to free a null pointer */
    if(file_contents)
    {
        g_free(file_contents);
    }
}

static void
change_pixmap(setiapplet * sa)
{
    static int  timeout_image = 0;

    if((timeout_image % 2) == 1)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_show(sa->applet_gui->radar_red_item);
        gnome_canvas_item_raise_to_top(sa->applet_gui->radar_red_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_green_item);
    }
    else
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_show(sa->applet_gui->radar_green_item);
        gnome_canvas_item_raise_to_top(sa->applet_gui->radar_green_item);
    }
    timeout_image++;
}

static void
attention_timeout(setiapplet * sa)
{
    change_pixmap(sa);
    attention_callback(sa);
}

static void
attention_timeout_end(setiapplet * sa)
{
    int         current_status = seti_status(sa);
    if(sa->attention_timeout_id != 0)
    {
        gtk_timeout_remove(sa->attention_timeout_id);
    }
    sa->attention_timeout_id = 0;
    check_change_pixmap(sa);
    update_applet_menu(sa);
    if(current_status == SETI_RUNNING)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_show(sa->applet_gui->radar_green_item);
        gnome_canvas_item_raise_to_top(sa->applet_gui->radar_green_item);
    }
    else if(current_status == SETI_STOPPED)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_show(sa->applet_gui->radar_red_item);
        gnome_canvas_item_raise_to_top(sa->applet_gui->radar_red_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_green_item);
    }
    else if(current_status == SETI_NONEXISTANT)
    {
        gnome_canvas_item_show(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_raise_to_top(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_green_item);
    }
}

static void
attention_callback(setiapplet * sa)
{
    if(sa->attention_timeout_id)
    {
        gtk_timeout_remove(sa->attention_timeout_id);
    }
    if(seti_status(sa) == SETI_RUNNING)
    {
        sa->attention_timeout_id =
            gtk_timeout_add(ATTENTION_INTERVAL, (GtkFunction) attention_timeout,
                            sa);
    }
    else
    {
        check_change_pixmap(sa);
    }
}

static void
get_gaus_values(setiapplet * sa,
                char *buf)
{
    gdouble     d_temp;
    gchar      *c_temp;
    gchar      *loc;
    gint        i;

    for(i = 0; i < 64; i++)
    {
        c_temp = g_strdup_printf("bg_pot %d=", i);
        loc = strstr(buf, c_temp);
        if(loc == NULL)
        {
            int         j;
            for(j = i; j < 64; j++)
            {
                sa->current_unit_info->gaus_values[j] = 0.0;
            }
            g_free(c_temp);
            return;
        }
        loc += strlen(c_temp);
        d_temp = g_strtod(loc, NULL);
        g_free(c_temp);
        sa->current_unit_info->gaus_values[i] = d_temp;
        if(d_temp > sa->current_unit_info->max_gaus)
        {
            sa->current_unit_info->max_gaus = d_temp;
        }
    }
    if(sa->current_unit_info->max_gaus == 0.0)
    {
        sa->current_unit_info->max_gaus = 1.0;
    }
}


/**
 * Show percentage
 */
static void
setup_show_percentage(setiapplet * sa,
                      GtkWidget * the_canvas)
{
    sa->applet_gui->percentage_item =
        sa->applet_gui->current_skin->
        progress(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
                 sa->current_unit_info->percent_complete,
                 sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->percentage_item;
}


/**
 * Show total cputime
 */
static void
setup_show_cputime(setiapplet * sa,
                   GtkWidget * the_canvas)
{
    sa->applet_gui->cpu_time_item =
        sa->applet_gui->current_skin->
        cputimeoverall(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
                       sa->user_info->cumulative_cpu_time,
                       sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->cpu_time_item;
}

/**
 * Show current cputime
 */
static void
setup_show_cputimecurrent(setiapplet * sa,
                   GtkWidget * the_canvas)
{
    sa->applet_gui->cpu_time_current_item =
        sa->applet_gui->current_skin->
        cputimecurrent(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
                       sa->current_unit_info->current_cpu_time,
                       sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->cpu_time_current_item;
}

/**
 * Show units
 */
static void
setup_show_units(setiapplet * sa,
                 GtkWidget * the_canvas)
{
    sa->applet_gui->num_units_item =
        sa->applet_gui->current_skin->
        units(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
              sa->user_info->units_processed, sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->num_units_item;
}


/**
 * Show spike
 */
static void
setup_show_spike(setiapplet * sa,
                 GtkWidget * the_canvas)
{
    sa->applet_gui->spike_item =
        sa->applet_gui->current_skin->
        spike(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
              sa->current_unit_info->best_spike, sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->spike_item;
}


/**
 * Show gaussian
 */
static void
setup_show_gaus(setiapplet * sa,
                GtkWidget * the_canvas)
{
    sa->applet_gui->gaus_graph_item =
        sa->applet_gui->current_skin->
        gaussian(sa->applet_gui->current_skin->private, gnome_canvas_root(GNOME_CANVAS(the_canvas)),
                 sa->current_unit_info->gaus_values,
                 sa->current_unit_info->best_gaussian_power,
                 sa->current_unit_info->best_gaussian_score,
                 sa->current_unit_info->max_gaus, 64,
                 sa->applet_gui->custom_size);
    sa->applet_gui->current_canvas_item = sa->applet_gui->gaus_graph_item;
}


static void
parse_state_file(setiapplet * sa,
                 char *read_buf)
{
    gdouble     d_temp;
    gint        attention_tmout;
    gchar      *loc;

    if(!read_buf)
    {
        /*Start the flash callback */
        attention_timeout(sa);
        sa->value_mask[PERCENT] &= ~VALUE_VALID;
        sa->value_mask[PERCENT] |= VALUE_INVALID;
        sa->value_mask[SPIKE] &= ~VALUE_VALID;
        sa->value_mask[SPIKE] |= VALUE_INVALID;
        sa->value_mask[GAUSSIAN] &= ~VALUE_VALID;
        sa->value_mask[GAUSSIAN] |= VALUE_INVALID;
        sa->value_mask[CPUTIMECURRENT] &= ~VALUE_VALID;
        sa->value_mask[CPUTIMECURRENT] |= VALUE_INVALID;
    }
    else
    {
        if(sa->attention_timeout_id)
        {
            attention_timeout_end(sa);
        }
        loc = strstr(read_buf, "cpu=");
        if(loc == NULL)
        {
            sa->value_mask[CPUTIMECURRENT] &= ~VALUE_VALID;
            sa->value_mask[CPUTIMECURRENT] |= VALUE_INVALID;
            return;
        }
        else
        {
            sa->value_mask[CPUTIMECURRENT] &= ~VALUE_INVALID;
            sa->value_mask[CPUTIMECURRENT] |= VALUE_VALID;
            loc += 4;
            d_temp = g_strtod(loc, NULL);
            sa->current_unit_info->current_cpu_time = d_temp;
        }
        loc = strstr(read_buf, "prog=");
        if(loc == NULL)
        {
            sa->value_mask[PERCENT] &= ~VALUE_VALID;
            sa->value_mask[PERCENT] |= VALUE_INVALID;
            return;
        }
        else
        {
            sa->value_mask[PERCENT] &= ~VALUE_INVALID;
            sa->value_mask[PERCENT] |= VALUE_VALID;
            loc += 5;
            d_temp = g_strtod(loc, NULL);
            sa->current_unit_info->percent_complete = d_temp;
        }
        loc = strstr(loc, "bs_power=");
        if(loc == NULL)
        {
            sa->value_mask[SPIKE] &= ~VALUE_VALID;
            sa->value_mask[SPIKE] |= VALUE_INVALID;
            return;
        }
        else
        {
            sa->value_mask[SPIKE] &= ~VALUE_INVALID;
            sa->value_mask[SPIKE] |= VALUE_VALID;
            loc += 9;
            sa->current_unit_info->best_spike = g_strtod(loc, NULL);
        }
        loc = strstr(loc, "bg_score=");
        if(loc == NULL)
        {
            sa->value_mask[GAUSSIAN] &= ~VALUE_VALID;
            sa->value_mask[GAUSSIAN] |= VALUE_INVALID;
            return;
        }
        else
        {
            loc += 9;
            sa->current_unit_info->best_gaussian_score = g_strtod(loc, NULL);
            loc = strstr(loc, "bg_power=");
            if(loc == NULL)
            {
                sa->value_mask[GAUSSIAN] &= ~VALUE_VALID;
                sa->value_mask[GAUSSIAN] |= VALUE_INVALID;
                return;
            }
            else
            {
                sa->value_mask[GAUSSIAN] &= ~VALUE_VALID;
                sa->value_mask[GAUSSIAN] |= VALUE_INVALID;
                loc += 9;
                sa->current_unit_info->best_gaussian_power =
                    g_strtod(loc, NULL);
            }
        }
        get_gaus_values(sa, loc);
    }
}

static void
parse_user_file(setiapplet * sa,
                char *read_buf)
{
    gdouble     d_temp;
    gchar      *loc;

    if(!read_buf)
    {
        sa->value_mask[UNITS] &= ~VALUE_VALID;
        sa->value_mask[UNITS] |= VALUE_INVALID;
        sa->value_mask[CPUTIME] &= ~VALUE_VALID;
        sa->value_mask[CPUTIME] |= VALUE_INVALID;
    }
    else
    {
        loc = strstr(read_buf, "nresults=");
        if(loc == NULL)
        {
            sa->value_mask[UNITS] &= ~VALUE_VALID;
            sa->value_mask[UNITS] |= VALUE_INVALID;
            //gchar *end_of_loc = strstr(loc,"\n");
        }
        else
        {
            sa->value_mask[UNITS] &= ~VALUE_INVALID;
            sa->value_mask[UNITS] |= VALUE_VALID;
            loc += 9;
            d_temp = g_strtod(loc, NULL);
            sa->user_info->units_processed = d_temp;
        }

        loc = strstr(loc, "total_cpu=");
        if(loc == NULL)
        {
            sa->value_mask[CPUTIME] &= ~VALUE_VALID;
            sa->value_mask[CPUTIME] |= VALUE_INVALID;
        }
        else
        {
            sa->value_mask[CPUTIME] &= ~VALUE_INVALID;
            sa->value_mask[CPUTIME] |= VALUE_VALID;
            loc += 10;
            d_temp = g_strtod(loc, NULL);
            sa->user_info->cumulative_cpu_time = d_temp;
        }
    }
}


static void
properties_cb(BonoboUIComponent * uic,
              setiapplet * sa,
              const gchar * verbname)
{
    create_SetiAppletProperties(sa);
}


static void
update_display(setiapplet * sa,
               gint direction)
{
    gint        internal_count = 0;
    check_change_pixmap(sa);
    update_applet_menu(sa);
    sa->current_count += direction;

    if(sa->current_count < 0)
    {
        sa->current_count = 5;
    }

    if(sa->current_count >= 6)
    {
        sa->current_count = sa->current_count % 6;
    }
    while((sa->value_mask[sa->current_count % 6] & VALUE_HIDE)
          && (internal_count < 6))
    {
        internal_count++;
        sa->current_count++;
    }
    if(internal_count < 7)
    {
        /*Remove the previous canvas item */
        switch (sa->current_count % 6)
        {
            case GAUSSIAN:
                setup_gaus_canvas_item(sa);
                break;
            case PERCENT:
                setup_percent_canvas_item(sa);
                break;
            case SPIKE:
                setup_spike_canvas_item(sa);
                break;
            case CPUTIME:
                setup_cputime_canvas_item(sa);
                break;
            case UNITS:
                setup_units_canvas_item(sa);
                break;
            case CPUTIMECURRENT:
                setup_cputime_current_canvas_item(sa);
                break;
        }
        if(sa->applet_gui->current_canvas_item)
        {
            gnome_canvas_item_lower_to_bottom(sa->applet_gui->current_canvas_item);
        }
    }
    else
    {
        /*Show an error image thingo */
        /*setup_default_canvas_item(); */
    }
}


/**
 * Setup the display of CPU time canvas
 */
static void
setup_cputime_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_cputime(sa, sa->applet_gui->the_canvas);
}

/**
 * Setup the display of current CPU time canvas
 */
static void
setup_cputime_current_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_cputimecurrent(sa, sa->applet_gui->the_canvas);
}


/**
 * Setup the display of unit time canvas
 */
static void
setup_units_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_units(sa, sa->applet_gui->the_canvas);
}


/**
 * Setup the display of percent time canvas
 */
static void
setup_percent_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_percentage(sa, sa->applet_gui->the_canvas);
}


/**
 * Setup the display of spike canvas
 */
static void
setup_spike_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_spike(sa, sa->applet_gui->the_canvas);
}


/**
 * Setup the display of gaussian canvas
 */
static void
setup_gaus_canvas_item(setiapplet * sa)
{
    if(sa->applet_gui->current_canvas_item)
    {
        gtk_object_destroy(GTK_OBJECT(sa->applet_gui->current_canvas_item));
    }
    sa->applet_gui->current_canvas_item = 0;
    if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        if(sa->applet_gui->custom_size->panel_size <= 24.0)
        {
            return;
        }
    }
    setup_show_gaus(sa, sa->applet_gui->the_canvas);
}


static void
display_timeout_up(setiapplet * sa)
{
    update_display(sa, -1);
    display_callback(sa);
}

static void
display_timeout(setiapplet * sa)
{
    update_display(sa, 1);
    display_callback(sa);
}

static void
update_state_file_timeout(setiapplet * sa)
{
    re_read_state_file(sa);
    refresh_tool_tip(sa);
    state_file_callback(sa);
}

static void
update_user_file_timeout(setiapplet * sa)
{
    re_read_user_file(sa);
    user_file_callback(sa);
}

static void
state_file_callback(setiapplet * sa)
{
    gint        delay;

    if(sa->file_update_interval < 1)
    {
        delay = 120;            /* (CJO) changed because 2.4 updates every 60 sec. */
    }
    else
    {
        delay = sa->file_update_interval;
    }
    delay *= 1000;
    if(sa->update_state_file_timeout_id)
    {
        gtk_timeout_remove(sa->update_state_file_timeout_id);
    }
    sa->update_state_file_timeout_id =
        gtk_timeout_add(delay, (GtkFunction) update_state_file_timeout, sa);
}

static void
user_file_callback(setiapplet * sa)
{
    gint        delay;

    /*
     * (CJO) for now, use the file update interval, just like the
     *       state file callback does.
     */
    if(sa->file_update_interval < 1)
    {
        delay = 120;            /* (CJO) changed because 2.4 updates every 60 sec. */
    }
    else
    {
        delay = sa->file_update_interval;
    }
    if(sa->update_user_file_timeout_id)
    {
        gtk_timeout_remove(sa->update_user_file_timeout_id);
    }
    /*
     * (CJO) - changed timeout to every two minutes , because one hour 
     *         just seems way longer than necessary.
     */
    sa->update_user_file_timeout_id =
        gtk_timeout_add(1000 * 60 * 2, (GtkFunction) update_user_file_timeout,
                        sa);
}

static void
display_callback(setiapplet * sa)
{
    gint        delay;

    if(sa->update_interval < 1)
    {
        /* (CJO) for now, try changing message every 5 sec. */
        delay = 5;
    }
    else
    {
        delay = sa->update_interval;
    }
    if(sa->display_timeout_id)
    {
        gtk_timeout_remove(sa->display_timeout_id);
    }
    delay *= 1000;
    sa->display_timeout_id =
        gtk_timeout_add(delay, (GtkFunction) display_timeout, sa);
}


void
start_callback(setiapplet * sa)
{
    state_file_callback(sa);
    user_file_callback(sa);
    display_callback(sa);
    refresh_tool_tip(sa);
    update_applet_menu(sa);
}


/**
 * check_change_pixmap, check if the process is running, change the image
 * appropriately.
 */
void
check_change_pixmap(setiapplet * sa)
{
    int         current_status = seti_status(sa);
    if(current_status == SETI_RUNNING)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_show(sa->applet_gui->radar_green_item);
    }
    else if(current_status == SETI_STOPPED)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_show(sa->applet_gui->radar_red_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_green_item);
    }
    else if((sa->attention_timeout_id != 0) && current_status == SETI_STOPPED)
    {
        gnome_canvas_item_hide(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_show(sa->applet_gui->radar_green_item);
    }
    else if(current_status == SETI_NONEXISTANT)
    {
        gnome_canvas_item_show(sa->applet_gui->radar_grey_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_red_item);
        gnome_canvas_item_hide(sa->applet_gui->radar_green_item);
    }
    sa->previous_status = current_status;
}

static void
click_event(GnomeCanvasItem * item,
            GdkEvent * event,
            gpointer data)
{
    setiapplet *sa = (setiapplet *) data;
    switch (event->button.button)
    {
        case 4:
            display_timeout_up(sa);
            break;
        case 5:
            display_timeout(sa);
            break;
    }
}

static void
applet_destroy(GtkWidget * applet,
               setiapplet * sa)
{
    if(sa->attention_timeout_id)
    {
        gtk_timeout_remove(sa->attention_timeout_id);
    }
    if(sa->display_timeout_id)
    {
        gtk_timeout_remove(sa->display_timeout_id);
    }
    if(sa->update_user_file_timeout_id)
    {
        gtk_timeout_remove(sa->update_user_file_timeout_id);
    }
    if(sa->update_state_file_timeout_id)
    {
        gtk_timeout_remove(sa->update_state_file_timeout_id);
    }

/*
  if(sa->applet_gui->current_canvas_item)
  {
    gtk_object_destroy (GTK_OBJECT (sa->applet_gui->current_canvas_item));
  }
  if(sa->applet_gui->frame)
  {
    gtk_widget_destroy(sa->applet_gui->frame);
  }
  if(sa->applet_gui->the_canvas)
  {
    gtk_object_destroy(GTK_OBJECT(sa->applet_gui->the_canvas));
  }
*/
    if(sa->applet_gui->seti_console)
    {
        gtk_widget_destroy(sa->applet_gui->seti_console);
    }
    if(sa->applet_gui->seti_console_text)
    {
        gtk_widget_destroy(GTK_WIDGET(sa->applet_gui->seti_console_text));
    }
    if(sa->seti_dir)
    {
        g_free(sa->seti_dir);
    }
    if(sa->seti_console_buffer)
    {
        g_free(sa->seti_console_buffer);
    }
    if(sa->druid_gui)
    {
        g_free(sa->druid_gui);
    }
    if(sa->properties_gui)
    {
        g_free(sa->properties_gui);
    }
    g_free(sa->applet_gui->custom_size);
    g_free(sa->applet_gui);
    g_free(sa->user_info);
    g_free(sa->current_unit_info);
    g_free(sa);
}


static void
create_applet_gui(setiapplet * sa)
{
    GnomeCanvasGroup *group;
    GtkWidget  *scrolledwindow1;
    short       new_dx, new_dy;

    // Let the skin initialise itself first
    sa->applet_gui->current_skin->setup();

    sa->applet_gui->frame = gtk_frame_new(NULL);

    /*
    if(sa->applet_gui->custom_size->only_radar)
    {
        gtk_frame_set_shadow_type(GTK_FRAME(sa->applet_gui->frame), GTK_SHADOW_NONE);
    }
    else
    {
        gtk_frame_set_shadow_type(GTK_FRAME(sa->applet_gui->frame), GTK_SHADOW_OUT);
    }
    */
    if(sa->applet_gui->custom_size->draw_border)
    {
        gtk_frame_set_shadow_type(GTK_FRAME(sa->applet_gui->frame), GTK_SHADOW_OUT);
    }

    sa->applet_gui->bg_type = panel_applet_get_background(PANEL_APPLET(sa->applet_gui->applet), &sa->applet_gui->bg_colour, &sa->applet_gui->bg_pixmap);

    sa->applet_gui->tip_text = gtk_tooltips_new();

    gtk_widget_push_colormap(gdk_rgb_get_colormap());
    gtk_widget_push_visual(gdk_rgb_get_visual());

    sa->applet_gui->the_canvas = gnome_canvas_new_aa();

    gtk_widget_pop_visual();
    gtk_widget_pop_colormap();

    // Create a Gnome canvas
    group = gnome_canvas_root(GNOME_CANVAS(sa->applet_gui->the_canvas));

    sa->applet_gui->title_text_item =
        sa->applet_gui->current_skin->title(sa->applet_gui->current_skin->private, group, sa->applet_gui->custom_size);
    gnome_canvas_item_lower_to_bottom(sa->applet_gui->title_text_item);
    sa->applet_gui->radar_red_item =
        sa->applet_gui->current_skin->radar_red(sa->applet_gui->current_skin->private, group,
                                                sa->applet_gui->custom_size);
    sa->applet_gui->radar_green_item =
        sa->applet_gui->current_skin->radar_green(sa->applet_gui->current_skin->private, group,
                                                  sa->applet_gui->custom_size);
    sa->applet_gui->radar_grey_item =
        sa->applet_gui->current_skin->radar_grey(sa->applet_gui->current_skin->private, group,
                                                 sa->applet_gui->custom_size);

    if(sa->applet_gui->radar_red_item)
    {
        //gtk_widget_set_events(sa->applet_gui->radar_red_item, gtk_widget_get_events(sa->applet_gui->radar_red_item) | GDK_BUTTON_PRESS_MASK);
        g_signal_connect(G_OBJECT(sa->applet_gui->radar_red_item),
                         "event", G_CALLBACK(radar_event), sa);
    }

    if(sa->applet_gui->radar_green_item)
    {
        //gtk_widget_set_events(sa->applet_gui->radar_green_item, gtk_widget_get_events(sa->applet_gui->radar_green_item) | GDK_BUTTON_PRESS_MASK);
        g_signal_connect(G_OBJECT(sa->applet_gui->radar_green_item),
                         "event", G_CALLBACK(radar_event), sa);
    }

    if(seti_status(sa) == SETI_STOPPED)
    {
        // Lower the green radar dish to allow the red one to show
        gnome_canvas_item_lower(sa->applet_gui->radar_green_item, 1);
    }
    //Green radar will show by default, so we need do nothing if seti is started
    gtk_container_add(GTK_CONTAINER(sa->applet_gui->applet),
                      sa->applet_gui->frame);
    gtk_container_add(GTK_CONTAINER(sa->applet_gui->frame),
                      sa->applet_gui->the_canvas);
    gtk_widget_show_all(sa->applet_gui->applet);

    // Work out the applet size first
    if(sa->applet_gui->custom_size->auto_size)
    {
        if(sa->applet_gui->custom_size->orientation == PANEL_APPLET_ORIENT_LEFT
           || sa->applet_gui->custom_size->orientation ==
           PANEL_APPLET_ORIENT_RIGHT)
        {
            // Look at the panel size - this determines the x/y values.
            new_dx = sa->applet_gui->custom_size->panel_size;
            new_dy = new_dx * X_TO_Y_RATIO;
        }
        else
        {
            new_dy = sa->applet_gui->custom_size->panel_size;
            new_dx = new_dy * X_TO_Y_RATIO;
        }
    }
    else
    {
        new_dx = sa->applet_gui->custom_size->x;
        new_dy = sa->applet_gui->custom_size->y;
    }
    gnome_canvas_set_scroll_region(GNOME_CANVAS
                                   (sa->applet_gui->the_canvas), 0, 0,
                                   128, 128 * 3);
    gnome_canvas_set_center_scroll_region(GNOME_CANVAS
                                          (sa->applet_gui->the_canvas), FALSE);
    gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(sa->applet_gui->the_canvas),
                                     new_dy / 128.0);
    gtk_widget_set_usize(sa->applet_gui->the_canvas, new_dx, new_dy);
    gtk_widget_queue_resize(sa->applet_gui->the_canvas);
    gtk_widget_queue_resize(sa->applet_gui->the_canvas->parent);
}


/**
 * Hide the Start or Stop menu
 */
void
update_applet_menu(setiapplet * sa)
{
    if(seti_status(sa) == SETI_RUNNING)
    {
        bonobo_ui_component_set_prop(sa->applet_gui->menu_component,
                                     "/commands/SetiStart", "hidden", "1",
                                     NULL);
        bonobo_ui_component_set_prop(sa->applet_gui->menu_component,
                                     "/commands/SetiStop", "hidden", "0", NULL);
    }
    else
    {
        bonobo_ui_component_set_prop(sa->applet_gui->menu_component,
                                     "/commands/SetiStop", "hidden", "1", NULL);
        bonobo_ui_component_set_prop(sa->applet_gui->menu_component,
                                     "/commands/SetiStart", "hidden", "0",
                                     NULL);
    }
}


/* Bonobo Verbs for our popup menu */
static const BonoboUIVerb seti_menu_verbs[] = {
    BONOBO_UI_UNSAFE_VERB("SetiProperties", properties_cb),
    BONOBO_UI_UNSAFE_VERB("SetiConsole", console_cb),
    BONOBO_UI_UNSAFE_VERB("SetiStart", start_stop_seti_cb),
    BONOBO_UI_UNSAFE_VERB("SetiStop", start_stop_seti_cb),
    BONOBO_UI_UNSAFE_VERB("SetiAbout", about_cb),
    BONOBO_UI_UNSAFE_VERB("SetiHelp", about_cb),
    BONOBO_UI_VERB_END
};


/**
 * Creation of the applet
 */
create_applet(PanelApplet * applet)
{
    setiapplet *sa;

    sa = g_new0(setiapplet, 1);
    sa->user_info = g_new0(setiapplet_user_info, 1);
    sa->current_unit_info = g_new0(setiapplet_current_unit, 1);
    sa->properties_gui = g_new0(setiapplet_gui_properties, 1);
    sa->applet_gui = g_new0(setiapplet_gui, 1);
    sa->applet_gui->custom_size = g_new0(seti_applet_size_t, 1);
    sa->applet_gui->current_skin = new_nestor_skin();
    sa->druid_gui = g_new0(setiapplet_gui_druid, 1);

    //Fill in some defaults for the applet
    sa->nice_value = 19;
    sa->update_interval = 10;
    sa->file_update_interval = 60;
    sa->value_mask[PERCENT] = VALUE_DISPLAY;
    sa->value_mask[SPIKE] = VALUE_DISPLAY;
    sa->value_mask[GAUSSIAN] = VALUE_HIDE;
    sa->value_mask[UNITS] = VALUE_DISPLAY;
    sa->value_mask[CPUTIME] = VALUE_DISPLAY;
    sa->value_mask[CPUTIMECURRENT] = VALUE_DISPLAY;
    sa->applet_gui->custom_size->auto_size = 1;
    sa->applet_gui->custom_size->x = 48;
    sa->applet_gui->custom_size->y = 48;
    sa->applet_gui->custom_size->draw_border = TRUE;
    sa->applet_gui->applet = GTK_WIDGET(applet);
    sa->applet_gui->menu_component = panel_applet_get_popup_component(applet);

    gnome_window_icon_set_default_from_file(GNOME_ICONDIR
                                            "/seti_applet-logo-green.png");

    // Connect to the GConf structure  
    panel_applet_add_preferences(applet, "/schemas/apps/seti_applet/prefs",
                                 NULL);

    // For the destroying of the applet.
    g_signal_connect(G_OBJECT(sa->applet_gui->applet), "destroy",
                     G_CALLBACK(applet_destroy), sa);

    // signals to ensure the applet is updated appropriately when moved
    g_signal_connect(G_OBJECT(sa->applet_gui->applet), "change_size",
                     G_CALLBACK(applet_change_pixel_size), sa);

    g_signal_connect(G_OBJECT(sa->applet_gui->applet), "change_orient",
                     G_CALLBACK(applet_change_orient), sa);

    g_signal_connect(G_OBJECT(sa->applet_gui->applet), "change_background",
                     G_CALLBACK(applet_change_background), sa);

    // in the rare case that the communication with the panel failed, error out
    if(!sa->applet_gui->applet)
    {
        g_error("NULL applet passed in\n");
        return FALSE;
    }

    // Loading the preferences from gconf
    panel_applet_preferences_load(sa, applet);

    sa->applet_gui->custom_size->panel_size =
        panel_applet_get_size(PANEL_APPLET(sa->applet_gui->applet));
    sa->applet_gui->custom_size->orientation = panel_applet_get_orient(applet);

    create_applet_gui(sa);

    // Get some initial values or error message from the state and user files
    re_read_state_file(sa);
    re_read_user_file(sa);

    // Refresh the tool tip
    refresh_tool_tip(sa);
    update_display(sa, 1);

    // Make sure we have a useful sized applet...
    applet_change_orient(PANEL_APPLET(sa->applet_gui->applet),
                         panel_applet_get_orient(PANEL_APPLET
                                                 (sa->applet_gui->applet)), sa);

    // Load in the menus for the right click
    panel_applet_setup_menu_from_file(PANEL_APPLET(sa->applet_gui->applet),
                                      NULL,
                                      "GNOME_SetiApplet.xml",
                                      NULL, seti_menu_verbs, sa);

    // Start setiathome if not already started
    if(sa->launch_on_start == TRUE)
    {
        if(seti_status(sa) == SETI_STOPPED)
        {
            start_stop_seti_cb(PANEL_APPLET(sa->applet_gui->applet), sa);
        }
    }

    // Start callback
    start_callback(sa);

    return 0;
}


/**
 * Functions called by factory
 */
static      gboolean
seti_applet_fill(PanelApplet * applet)
{
    create_applet(applet);
    return TRUE;
}

static      gboolean
seti_applet_factory(PanelApplet * applet,
                    const gchar * iid,
                    gpointer data)
{
    gboolean    rv = FALSE;
    g_message("Initialising seti@home applet");
    if(!strcmp(iid, "OAFIID:GNOME_SetiApplet"))
    {
        rv = seti_applet_fill(applet);
    }
    return rv;
}


/**
 * Connect to Factory to for creating applet
 */
PANEL_APPLET_BONOBO_FACTORY("OAFIID:GNOME_SetiApplet_Factory",
                            PANEL_TYPE_APPLET,
                            PACKAGE, VERSION, seti_applet_factory, NULL)
