/***************************************************************************

    Incredible Technologies/Strata system
    (8-bit blitter variant)

    driver by Aaron Giles

    Games supported:
        * Strata Bowling [2 sets]
        * Super Strike Bowling
        * Wheel of Fortune [2 sets]
        * Golden Tee Golf [2 sets]
        * Golden Tee Golf II [3 sets]
        * Slick Shot [3 sets]
        * Dyno-Bop
        * Arlington Horse Racing
        * Neck & Neck
        * Peggle [2 sets]
        * Poker Dice
        * Hot Shots Tennis [2 sets]
        * Rim Rockin' Basketball [4 sets]
        * Ninja Clowns

    Known issues:
        * Rim Rockin' Basketball should use an HD6309, but that core is
          broken, so we're using the 6809 for now
        * Ninja Clowns main ROM dump claims it's bad

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1 (6809 games)
    ========================================================================
    Note that many games have the regions 0000-0FFF and 1000-1FFF swapped.
    Also, Golden Tee Golf II (V2.2) has most of the addresses in the
    0000-0FFF range swizzled randomly.
    ========================================================================
    0100          W   xxxxxxxx    Unknown
    0120          W   xxxxxxxx    Sound data
    0140        R     xxxxxxxx    Input port
    0140          W   xxxxxxxx    GROM blitter bank select
    0160        R     xxxxxxxx    Input port
    0160          W   xx------    Display page select
                  W   x-------       (upper page select)
                  W   -x------       (unknown)
    0180        R     xxxxxxxx    Input port/DIP switches
    0180          W   xxxxxxxx    Video RAM upper data latch
    01A0          W   --------    NMI acknowledge (on some boards)
    01C0-01C1   R/W   xxxxxxxx    Blitter chip address high
    01C2-01C3   R/W   xxxxxxxx    Blitter chip address low
    01C4-01C5   R/W   ---xxxxx    Blitter chip flags
                R/W   ---x----       (render transparent)
                R/W   ----x---       (expand RLE compressed data)
                R/W   -----x--       (flip vertically)
                R/W   ------x-       (flip horizontally)
                R/W   -------x       (shift data left 4 bits)
    01C6-01C7   R     x-------    Blitter chip status (1=busy)
    01C6-01C7     W   --------    Blitter chip trigger
    01C8-01C9   R/W   xxxxxxxx    Blitter chip source width
    01CA-01CB   R/W   xxxxxxxx    Blitter chip source height
    01CC-01CD   R/W   xxxxxxxx    Blitter chip source data mask
    01CE-01CF   R/W   xxxxxxxx    Blitter chip output port
                  W   --x-----       (main program ROM bank select)
    01D0-01D1   R/W   xxxxxxxx    Blitter chip X starting coordinate
    01D2-01D3   R/W   xxxxxxxx    Blitter chip maximum scanline count
    01D4-01D5   R/W   xxxxxxxx    Blitter chip X ending coordinate
    01D6-01D7   R/W   xxxxxxxx    Blitter chip initial scanline skip
    01D8-01DF   R     xxxxxxxx    Blitter chip input ports
    01E0          W   xxxxxxxx    Palette address latch
    01E2-01E3     W   --xxxxxx    Palette data latches
    1000-11FF   R/W   xxxxxxxx    TMS34061 register access
    1200-13FF   R/W   xxxxxxxx    TMS34061 XY addressing access
    1400-15FF   R/W   xxxxxxxx    TMS34061 register access
    1600-17FF   R/W   xxxxxxxx    TMS34061 direct access
    1800-19FF   R/W   xxxxxxxx    TMS34061 shift register read
    1A00-1BFF   R/W   xxxxxxxx    TMS34061 shift register write
    2000-3FFF   R/W   xxxxxxxx    RAM (battery-backed)
    4000-7FFF   R     xxxxxxxx    Banked program ROM
    8000-FFFF   R     xxxxxxxx    Fixed program ROM
    ========================================================================
    Interrupts:
        NMI generated by VBLANK
        IRQ generated by TMS34061 scanline interrupt
        FIRQ generated by blitter when finished
    ========================================================================

    ========================================================================
    CPU #1 (68000 games)
    ========================================================================
    000000-003FFF   R/W   xxxxxxxx xxxxxxxx   RAM (battery-backed)
    004000-03FFFF   R     xxxxxxxx xxxxxxxx   Main program ROM
    100080          W     xxxxxxxx --------   Sound data
    100100          R     xxxxxxxx --------   Input port
    100100            W   xxxxxxxx --------   GROM blitter bank select
    100180          R     xxxxxxxx --------   Input port
    100100            W   xx------ --------   Display page select
                      W   x------- --------       (upper page select)
                      W   -x------ --------       (unknown)
    100240            W   xxxxxxxx --------   Video RAM upper data latch
    100280          R     xxxxxxxx --------   Input port/DIP switches
    100280            W   -------- --------   Unknown
    100300-10031F   R/W   xxxxxxxx xxxxxxxx   Blitter chip (as above)
    100380            W   xxxxxxxx --------   Palette address latch
    1003a0            W   --xxxxxx --------   Palette data latches
    110000-1101FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
    110200-1103FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 XY addressing access
    110400-1105FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
    110600-1107FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 direct access
    110800-1109FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register read
    110A00-110BFF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register write
    ========================================================================
    Interrupts:
        INT 3 generated by VBLANK
        INT 2 generated by blitter when finished
        INT 1 generated by ???
    ========================================================================

    ========================================================================
    CPU #2
    ========================================================================
    0000          W   xxxxxxxx    Unknown
    1000        R     xxxxxxxx    Sound command input
    2000-2003   R/W   xxxxxxxx    Yamaha chip I/O
    3000-37FF   R/W   xxxxxxxx    RAM
    4000        R/W   xxxxxxxx    OKI 6295 I/O
    5000-5003   R/W   xxxxxxxx    6521 PIA chip (on early YM3812 boards)
    5000-500F   R/W   xxxxxxxx    6522 VIA chip (on later YM3812 boards)
    8000-FFFF   R     xxxxxxxx    Program ROM
    ========================================================================
    Interrupts:
        NMI not connected
        IRQ generated by write to sound command input
        FIRQ generated by Yamaha chip, or by 6522 VIA (if present)
    ========================================================================

****************************************************************************

    PCB layouts

****************************************************************************

    Summary:

        There are 7 known variants of PCBs for these games. All the PCBs
        have the following features in common:

            68B09 @ 2 MHz for the main CPU
            8k RAM for the main CPU

            68B09 @ 2 MHz for the sound CPU
            2k RAM for the sound CPU
            YM2203C or YM3812 for music
            OKI M6295 for speech

            TMS34061 for the video controller
            ITV4400 custom blitter for rendering
            6-bit RAMDAC for palette
            From 2-8 64k x 4-bit VRAM chips for frame buffers

        An overview of each style PCB is given below:

        Wheel of Fortune-style
            * Single board
            * YM2203C for music
            * 2 VRAM chips for a single 8-bit 256x256 video page

        Strata Bowling-style
            * Single board
            * YM2203C for music
            * 3 VRAM chips for an 8-bit 256x256 video background
              plus a 4-bit 256x256 video foreground layer

        Slick Shot-style
            * Single board
            * YM2203C for music
            * 4 VRAM chips for two 8-bit 256x256 video pages
            * additional Z80 and program ROM for sensor reading

        Hot Shots Tennis-style
            * Single board
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Rim Rockin' Basketball-style
            * Separate main and sound boards
            * HD63C09 @ 3MHz replaces 68B09 for main CPU
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Ninja Clowns-style
            * Separate main and sound boards
            * 68000 @ 12MHz replaces 68B09 for main CPU
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Golden Tee Golf II-style
            * Separate main and sound boards
            * YM3812 for music
            * I haven't see one yet, so I can't provide additional details

****************************************************************************

    ----------------
    Wheel of Fortune
    ----------------

        +------------------------------------+-+
        |    Bt476KP35                       | |
        |      12MHz  DIPSW4                 | |
        |(MT42C4064Z-10)                     | |
        |(MT42C4064Z-10)                     | |
        | MT42C4064Z-10                      | |
        | MT42C4064Z-10                      | |
        |                                    | |
        |       TMS34061                     +-
        |                                      |
        |          ITV4400   ITVS              |
        |                          YM2203C     |
        |                              OKI6295 |
        | (GROM4)  Battery                     |
        |  GROM3    CDM6264LE12      SROM0     |
        |  GROM2    WOF-PGM   HY6116           |
        |  GROM1              WOFSND      8MHz |
        |  GROM0                               |
        |        MC68B09         MC68B09       |
        +--------------------------------------+

        GROM0-3 = AM27C256
        WOF-PGM = ???
        WOFSND = AM27C256
        SROM0 = AM27C010
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM (2 populated on Wheel of Fortune)
        CDM6464 = 8k x 8 RAM
        HY6116 = 2k x 8 RAM
        Bt476 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ------------------------------
    Strata Bowling/Golden Tee Golf
    ------------------------------

        +---------------------------------+-+
        |                       IMSG176   | |
        | MT42C4064Z-10                   | |
        | MT42C4064Z-10                   | |
        | MT42C4064Z-10    12MHz          | |
        |                                 | |
        |       TMS34061         DIPSW4   | |
        |                                 | |
        |    ITV4400                      +-
        |                                   |
        |        Battery        ITVS        |
        | (GROM5)                           |
        | (GROM4)                           |
        | (GROM3)   P5164                   |
        |  GROM2   SBPROG   MS6516   SROM0  |
        |  GROM1            SBSNDS          |
        |  GROM0                      8MHz  |
        |        MC68B09          MC68B09   |
        +-----------------------------------+

        GROM0-3 = AM27C010
        SBPROG = 27C256
        SBSNDS = 27C256
        SROM0 = AM27C010
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164 = 8k x 8 RAM
        MS6516 = 2k x 8 RAM
        IMSG176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ---------------------
    Slick Shot/Poker Dice
    ---------------------
    (Likely Dyno Bop as well)

        +------------------------------------+-+
        |   Z80PGM   4Mhz      itvs5         | |
        |            Z80       rev.1         | |
        |                                    | |
        | MT42C4064Z-10   MS176-50           | |
        | MT42C4064Z-10                      | |
        | MT42C4064Z-10   12MHz              | |
        | MT42C4064Z-10           DIPSW4     | |
        |                                    +-
        |      TMS34061                        |
        |                                      |
        |         ITV4400    ITVS              |
        |                          YM2203C     |
        |                              OKI6295 |
        | (GROM4)  Battery                     |
        | (GROM3)   P5164S           SROM0     |
        | (GROM2)   POOLPGM   P5116            |
        |  GROM1             POOLSND      8MHz |
        |  GROM0                               |
        |        MC68B09         MC68B09       |
        +--------------------------------------+

        (lower half is almost identical to Wheel of Fortune)
        (top contains extra Z80 and logic)
        (Z80 and ROM not populated on Poker Dice)

        GROM0-1 = AM27C010
        POOLPGM = ???
        POOLSND = AM27C256
        SROM0 = 27C512
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164S = 8k x 8 RAM
        P5116 = 2k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    -----------------------------------
    Hot Shots Tennis/Peggle/Neck & Neck
    -----------------------------------
    (Likely Arlington Horse Racing as well)

        +------------------------------------------------+-+
        | MT42C4064Z-10    8MHz     MS176-50             | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10   TMS34061    DIPSW4             | |
        | MT42C4064Z-10           12MHz                  +-
        | MT42C4064Z-10                                    |
        |                 ITV4400                          |
        |  (GROM5)                        YM3812  OKI6295  |
        |  (GROM4) Battery  ITVS-3        MS6516           |
        |  (GROM3)   P5164              PEG-SND    SROM0   |
        |   GROM2    PEGGLE.BIN                    6821    |
        |   GROM1                                          |
        |   GROM0                                          |
        |         EF68B09     8MHz      EF68B09            |
        +--------------------------------------------------+

        GROM0-5 = AM27C010
        POOLPGM = ???
        POOLSND = AM27C256
        SROM0 = 27C512
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164S = 8k x 8 RAM
        P5116 = 2k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ----------------------
    Rim Rockin' Basketball
    ----------------------

        Lower board

        +--------------- --------------------------------+
        |               |                                |
        +---------------+   MT42C4064Z-10  MT42C4064Z-10 |
        |                   MT42C4064Z-10  MT42C4064Z-10 |
        |      IMSG176      MT42C4064Z-10  MT42C4064Z-10 |
        |                   MT42C4064Z-10  MT42C4064Z-10 |
        |  DIPSW4 8MHz ITBP-1                            |
        |                                                |
        |    MS6264-10                                   |
        |     RRB.BIM          TM34061       ITV4400     |
        |         ITVS-2                                 |
        |                      GROM02        GROM00      |
        |  Battery                                       |
        |  HD63C09   12MHz     GROM03        GROM01      |
        +------------------------------------------------+

        GROM0-3 =
        RRB.BIM =
        ITVS-2 = ???
        MT42C4064Z-10 = 64k x 4 VRAM
        MS6264 = 8k x 8 RAM
        IMSG176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


        Sound board

        +---------------------------------+
        |         YM3812  OKI6295  SROM0  |
        | ITVS-4 MS6516-10                |
        |         NCSND                   |
        |                                 |
        |  EF68B09                        |
        |                                 |
        |   6522                          |
        | P/N 1038 REV2                   |
        +---------------------------------+

        SROM0 = M27C2001
        NCSND = AM27C256
        MS6516-10 = 2k x 8 RAM
        6522 = VIA


    ------------
    Ninja Clowns
    ------------

        Lower board

        +-----------------------------------------------------+-+-
        |  MS6264   PROM1  (PROM3)   DIPSW4                   | |
        |  MS6264   PROM0  (PROM2)                            | |
        |                                                     | |
        |  Battery        MC68000P12                          | |
        |                                                     | |
        |           12MHz   ITVS-5  ITMP-1                    | |
        |                                                     | |
        |  (GROM7)                                            +-
        |  (GROM6)          8MHz                                |
        |   GROM5                                      MS176-50 |
        |   GROM4                                               |
        |   GROM3   TMS34061   MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM2              MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM1   ITV4400    MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM0              MT42C4064Z-10  MT42C4064Z-10     |
        | P/N 1029 REV3A                                        |
        +-------------------------------------------------------+

        GROM0-5 = M27C2001
        PROM0-1 = AM27C010
        ITVS-5 = PAL16L8
        ITMP-1 = PAL20R4
        MT42C4064Z = 64k x 4 VRAM
        MS6264 = 8k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


        Sound board - same as Rim Rockin' Basketball

***************************************************************************/


#include "driver.h"
#include "deprecat.h"
#include "cpu/m6809/m6809.h"
#include "machine/6821pia.h"
#include "machine/6522via.h"
#include "machine/ticket.h"
#include "video/tms34061.h"
#include "video/tlc34076.h"
#include "itech8.h"
#include "sound/2203intf.h"
#include "sound/3812intf.h"
#include "sound/okim6295.h"


#define FULL_LOGGING	0

#define CLOCK_8MHz		(8000000)
#define CLOCK_12MHz		(12000000)



/*************************************
 *
 *  Static data
 *
 *************************************/

static UINT8 blitter_int;
static UINT8 tms34061_int;
static UINT8 periodic_int;

static UINT8 sound_data;

static UINT8 pia_porta_data;
static UINT8 pia_portb_data;

static UINT8 *main_ram;
static size_t main_ram_size;

static const rectangle *visarea;



/*************************************
 *
 *  6821 PIA interface
 *
 *************************************/

static WRITE8_HANDLER( pia_porta_out );
static WRITE8_HANDLER( pia_portb_out );

static const pia6821_interface pia_interface =
{
	0, ticket_dispenser_r, 0, 0, 0, 0,		/* PIA inputs: A, B, CA1, CB1, CA2, CB2 */
	pia_porta_out, pia_portb_out, 0, 0,		/* PIA outputs: A, B, CA2, CB2 */
	0, 0									/* PIA IRQs: A, B */
};



/*************************************
 *
 *  6522 VIA interface
 *
 *************************************/

static void via_irq(int state);

static const struct via6522_interface via_interface =
{
	/*inputs : A/B         */ 0, 0,
	/*inputs : CA/B1,CA/B2 */ 0, 0, 0, 0,
	/*outputs: A/B         */ 0, pia_portb_out,
	/*outputs: CA/B1,CA/B2 */ 0, 0, 0, 0,
	/*irq                  */ via_irq
};



/*************************************
 *
 *  Interrupt handling
 *
 *************************************/

void itech8_update_interrupts(running_machine *machine, int periodic, int tms34061, int blitter)
{
	/* update the states */
	if (periodic != -1) periodic_int = periodic;
	if (tms34061 != -1) tms34061_int = tms34061;
	if (blitter != -1) blitter_int = blitter;

	/* handle the 6809 case */
	if (machine->drv->cpu[0].type == CPU_M6809)
	{
		/* just modify lines that have changed */
		if (periodic != -1) cpunum_set_input_line(machine, 0, INPUT_LINE_NMI, periodic ? ASSERT_LINE : CLEAR_LINE);
		if (tms34061 != -1) cpunum_set_input_line(machine, 0, M6809_IRQ_LINE, tms34061 ? ASSERT_LINE : CLEAR_LINE);
		if (blitter != -1) cpunum_set_input_line(machine, 0, M6809_FIRQ_LINE, blitter ? ASSERT_LINE : CLEAR_LINE);
	}

	/* handle the 68000 case */
	else
	{
		int level = 0;

		/* determine which level is active */
		if (blitter_int) level = 2;
		if (periodic_int) level = 3;

		/* update it */
		if (level)
			cpunum_set_input_line(machine, 0, level, ASSERT_LINE);
		else
			cpunum_set_input_line(machine, 0, 7, CLEAR_LINE);
	}
}



/*************************************
 *
 *  Interrupt generation
 *
 *************************************/

static INTERRUPT_GEN( generate_nmi )
{
	/* signal the NMI */
	itech8_update_interrupts(machine, 1, -1, -1);
	itech8_update_interrupts(machine, 0, -1, -1);

	if (FULL_LOGGING) logerror("------------ VBLANK (%d) --------------\n", video_screen_get_vpos(0));
}


static WRITE8_HANDLER( itech8_nmi_ack_w )
{
/* doesn't seem to hold for every game (e.g., hstennis) */
/*  cpunum_set_input_line(Machine, 0, INPUT_LINE_NMI, CLEAR_LINE);*/
}


static void generate_sound_irq(int state)
{
	cpunum_set_input_line(Machine, 1, M6809_FIRQ_LINE, state ? ASSERT_LINE : CLEAR_LINE);
}



/*************************************
 *
 *  Machine initialization
 *
 *************************************/

static TIMER_CALLBACK( behind_the_beam_update );


static MACHINE_START( itech8 )
{
	pia_config(0, &pia_interface);
}

static MACHINE_START( sstrike )
{
	MACHINE_START_CALL(itech8);

	/* we need to update behind the beam as well */
	timer_set(video_screen_get_time_until_pos(0, 0, 0), NULL, 32, behind_the_beam_update);
}

static MACHINE_RESET( itech8 )
{
	/* make sure bank 0 is selected */
	if (machine->drv->cpu[0].type == CPU_M6809)
		memory_set_bankptr(1, &memory_region(REGION_CPU1)[0x4000]);

	/* reset the PIA (if used) */
	pia_reset();

	/* reset the VIA chip (if used) */
	via_config(0, &via_interface);
	via_set_clock(0, CLOCK_8MHz/4);
	via_reset();

	/* reset the ticket dispenser */
	ticket_dispenser_init(200, TICKET_MOTOR_ACTIVE_HIGH, TICKET_STATUS_ACTIVE_LOW);

	/* reset the palette chip */
	tlc34076_reset(6);

	/* set the visible area */
	if (visarea)
	{
		video_screen_set_visarea(0, visarea->min_x, visarea->max_x, visarea->min_y, visarea->max_y);
		visarea = NULL;
	}
}



/*************************************
 *
 *  Bank switching
 *
 *************************************/

static TIMER_CALLBACK( behind_the_beam_update )
{
	int scanline = param >> 8;
	int interval = param & 0xff;

	/* force a partial update to the current scanline */
	video_screen_update_partial(0, scanline);

	/* advance by the interval, and wrap to 0 */
	scanline += interval;
	if (scanline >= 256) scanline = 0;

	/* set a new timer */
	timer_set(video_screen_get_time_until_pos(0, scanline, 0), NULL, (scanline << 8) + interval, behind_the_beam_update);
}



/*************************************
 *
 *  Bank switching
 *
 *************************************/

static WRITE8_HANDLER( blitter_w )
{
	/* bit 0x20 on address 7 controls CPU banking */
	if (offset / 2 == 7)
		memory_set_bankptr(1, &memory_region(REGION_CPU1)[0x4000 + 0xc000 * ((data >> 5) & 1)]);

	/* the rest is handled by the video hardware */
	itech8_blitter_w(offset, data);
}


static WRITE8_HANDLER( rimrockn_bank_w )
{
	/* banking is controlled here instead of by the blitter output */
	memory_set_bankptr(1, &memory_region(REGION_CPU1)[0x4000 + 0xc000 * (data & 3)]);
}



/*************************************
 *
 *  Input handling
 *
 *************************************/

/* used by most games */
static READ8_HANDLER( special_port0_r )
{
	UINT8 result = readinputport(0);
	result = (result & 0xfe) | (pia_portb_data & 0x01);
	return result;
}


/* used by Rim Rockin' Basketball */
static READ8_HANDLER( special_port1_r )
{
	UINT8 result = readinputport(1);
	result = (result & 0x7f) | ((pia_portb_data & 0x01) << 7);
	return result;
}



/*************************************
 *
 *  6821 PIA handling
 *
 *************************************/

static WRITE8_HANDLER( pia_porta_out )
{
	logerror("PIA port A write = %02x\n", data);
	pia_porta_data = data;
}


static WRITE8_HANDLER( pia_portb_out )
{
	logerror("PIA port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 4 controls the ticket dispenser */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	pia_portb_data = data;
	ticket_dispenser_w(0, (data & 0x10) << 3);
	coin_counter_w(0, (data & 0x20) >> 5);
}


static WRITE8_HANDLER( ym2203_portb_out )
{
	logerror("YM2203 port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	/* bit 7 controls the ticket dispenser */
	pia_portb_data = data;
	ticket_dispenser_w(0, data & 0x80);
	coin_counter_w(0, (data & 0x20) >> 5);
}



/*************************************
 *
 *  Sound communication
 *
 *************************************/

static TIMER_CALLBACK( delayed_sound_data_w )
{
	sound_data = param;
	cpunum_set_input_line(machine, 1, M6809_IRQ_LINE, ASSERT_LINE);
}


static WRITE8_HANDLER( sound_data_w )
{
	timer_call_after_resynch(NULL, data, delayed_sound_data_w);
}


static WRITE8_HANDLER( gtg2_sound_data_w )
{
	/* on the later GTG2 board, they swizzle the data lines */
	data = ((data & 0x80) >> 7) |
	       ((data & 0x5d) << 1) |
	       ((data & 0x20) >> 3) |
	       ((data & 0x02) << 5);
	timer_call_after_resynch(NULL, data, delayed_sound_data_w);
}


static READ8_HANDLER( sound_data_r )
{
	cpunum_set_input_line(Machine, 1, M6809_IRQ_LINE, CLEAR_LINE);
	return sound_data;
}



/*************************************
 *
 *  Sound 6522 VIA handling
 *
 *************************************/

static void via_irq(int state)
{
	if (state)
		cpunum_set_input_line(Machine, 1, M6809_FIRQ_LINE, ASSERT_LINE);
	else
		cpunum_set_input_line(Machine, 1, M6809_FIRQ_LINE, CLEAR_LINE);
}



/*************************************
 *
 *  16-bit memory shunts
 *
 *************************************/

static READ16_HANDLER( blitter16_r )
{
	return (itech8_blitter_r(offset * 2 + 0) << 8) + itech8_blitter_r(offset * 2 + 1);
}


static READ16_HANDLER( tms34061_16_r )
{
	/* since multiple XY accesses can move the pointer multiple times, we have to */
	/* be careful to only perform one read per access here; fortunately, the low */
	/* bit doesn't matter in XY addressing mode */
	if ((offset & 0x700) == 0x100)
	{
		int result = itech8_tms34061_r(offset * 2);
		return (result << 8) | result;
	}
	else
		return (itech8_tms34061_r(offset * 2 + 0) << 8) + itech8_tms34061_r(offset * 2 + 1);
}


static WRITE16_HANDLER( sound_data16_w )
{
	if (ACCESSING_MSB)
		sound_data_w(0, data >> 8);
}


static WRITE16_HANDLER( grom_bank16_w )
{
	if (ACCESSING_MSB)
		*itech8_grom_bank = data >> 8;
}


static WRITE16_HANDLER( display_page16_w )
{
	if (ACCESSING_MSB)
		itech8_page_w(0, ~data >> 8);
}


static WRITE16_HANDLER( tms34061_latch16_w )
{
	if (ACCESSING_MSB)
		tms34061_latch_w(0, data >> 8);
}


static WRITE16_HANDLER( blitter16_w )
{
	if (ACCESSING_MSB)
		itech8_blitter_w(offset * 2 + 0, data >> 8);
	if (ACCESSING_LSB)
		itech8_blitter_w(offset * 2 + 1, data);
}


static WRITE16_HANDLER( palette16_w )
{
	if (ACCESSING_MSB)
		itech8_palette_w(offset / 8, data >> 8);
}


static WRITE16_HANDLER( tms34061_16_w )
{
	if (ACCESSING_MSB)
		itech8_tms34061_w(offset * 2 + 0, data >> 8);
	else if (ACCESSING_LSB)
		itech8_tms34061_w(offset * 2 + 1, data);
}



/*************************************
 *
 *  NVRAM read/write
 *
 *************************************/

static NVRAM_HANDLER( itech8 )
{
	int i;

	if (read_or_write)
		mame_fwrite(file, main_ram, main_ram_size);
	else if (file)
		mame_fread(file, main_ram, main_ram_size);
	else
		for (i = 0; i < main_ram_size; i++)
			main_ram[i] = mame_rand(Machine);
}



/*************************************
 *
 *  Main CPU memory handlers
 *
 *************************************/

/*------ common layout with TMS34061 at 0000 ------*/
static ADDRESS_MAP_START( tmslo_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x0fff) AM_READWRITE(itech8_tms34061_r, itech8_tms34061_w)
	AM_RANGE(0x1100, 0x1100) AM_WRITENOP
	AM_RANGE(0x1120, 0x1120) AM_WRITE(sound_data_w)
	AM_RANGE(0x1140, 0x1140) AM_READWRITE(special_port0_r, MWA8_RAM) AM_BASE(&itech8_grom_bank)
	AM_RANGE(0x1160, 0x1160) AM_READWRITE(input_port_1_r, itech8_page_w)
	AM_RANGE(0x1180, 0x1180) AM_READWRITE(input_port_2_r, tms34061_latch_w)
	AM_RANGE(0x11a0, 0x11a0) AM_WRITE(itech8_nmi_ack_w)
	AM_RANGE(0x11c0, 0x11df) AM_READWRITE(itech8_blitter_r, blitter_w)
	AM_RANGE(0x11e0, 0x11ff) AM_WRITE(itech8_palette_w)
	AM_RANGE(0x2000, 0x3fff) AM_RAM AM_BASE(&main_ram) AM_SIZE(&main_ram_size)
	AM_RANGE(0x4000, 0xffff) AM_ROMBANK(1)
ADDRESS_MAP_END


/*------ common layout with TMS34061 at 1000 ------*/
static ADDRESS_MAP_START( tmshi_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x1000, 0x1fff) AM_READWRITE(itech8_tms34061_r, itech8_tms34061_w)
	AM_RANGE(0x0100, 0x0100) AM_WRITENOP
	AM_RANGE(0x0120, 0x0120) AM_WRITE(sound_data_w)
	AM_RANGE(0x0140, 0x0140) AM_READWRITE(special_port0_r, MWA8_RAM) AM_BASE(&itech8_grom_bank)
	AM_RANGE(0x0160, 0x0160) AM_READWRITE(input_port_1_r, itech8_page_w)
	AM_RANGE(0x0180, 0x0180) AM_READWRITE(input_port_2_r, tms34061_latch_w)
	AM_RANGE(0x01a0, 0x01a0) AM_WRITE(itech8_nmi_ack_w)
	AM_RANGE(0x01c0, 0x01df) AM_READWRITE(itech8_blitter_r, blitter_w)
	AM_RANGE(0x01e0, 0x01ff) AM_WRITE(itech8_palette_w)
	AM_RANGE(0x2000, 0x3fff) AM_RAM AM_BASE(&main_ram) AM_SIZE(&main_ram_size)
	AM_RANGE(0x4000, 0xffff) AM_ROMBANK(1)
ADDRESS_MAP_END


/*------ Golden Tee Golf II 1992 layout ------*/
static ADDRESS_MAP_START( gtg2_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0100, 0x0100) AM_READWRITE(input_port_0_r, itech8_nmi_ack_w)
	AM_RANGE(0x0120, 0x0120) AM_READWRITE(input_port_1_r, itech8_page_w)
	AM_RANGE(0x0140, 0x015f) AM_WRITE(itech8_palette_w)
	AM_RANGE(0x0140, 0x0140) AM_READ(input_port_2_r)
	AM_RANGE(0x0160, 0x0160) AM_WRITE(MWA8_RAM) AM_BASE(&itech8_grom_bank)
	AM_RANGE(0x0180, 0x019f) AM_READWRITE(itech8_blitter_r, blitter_w)
	AM_RANGE(0x01c0, 0x01c0) AM_WRITE(gtg2_sound_data_w)
	AM_RANGE(0x01e0, 0x01e0) AM_WRITE(tms34061_latch_w)
	AM_RANGE(0x1000, 0x1fff) AM_READWRITE(itech8_tms34061_r, itech8_tms34061_w)
	AM_RANGE(0x2000, 0x3fff) AM_RAM AM_BASE(&main_ram) AM_SIZE(&main_ram_size)
	AM_RANGE(0x4000, 0xffff) AM_ROMBANK(1)
ADDRESS_MAP_END


/*------ Ninja Clowns layout ------*/
static ADDRESS_MAP_START( ninclown_map, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x00007f) AM_RAM AM_REGION(REGION_CPU1, 0)
	AM_RANGE(0x000080, 0x003fff) AM_RAM AM_BASE((void *)&main_ram) AM_SIZE(&main_ram_size)
	AM_RANGE(0x004000, 0x07ffff) AM_ROM
	AM_RANGE(0x100080, 0x100081) AM_WRITE(sound_data16_w)
	AM_RANGE(0x100100, 0x100101) AM_READWRITE(input_port_0_word_r, grom_bank16_w) AM_BASE((void *)&itech8_grom_bank)
	AM_RANGE(0x100180, 0x100181) AM_READWRITE(input_port_1_word_r, display_page16_w)
	AM_RANGE(0x100240, 0x100241) AM_WRITE(tms34061_latch16_w)
	AM_RANGE(0x100280, 0x100281) AM_READWRITE(input_port_2_word_r, MWA16_NOP)
	AM_RANGE(0x100300, 0x10031f) AM_READWRITE(blitter16_r, blitter16_w)
	AM_RANGE(0x100380, 0x1003ff) AM_WRITE(palette16_w)
	AM_RANGE(0x110000, 0x110fff) AM_READWRITE(tms34061_16_r, tms34061_16_w)
ADDRESS_MAP_END



/*************************************
 *
 *  Sound CPU memory handlers
 *
 *************************************/

/*------ YM2203-based sound ------*/
static ADDRESS_MAP_START( sound2203_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x0000) AM_WRITENOP
	AM_RANGE(0x1000, 0x1000) AM_READ(sound_data_r)
	AM_RANGE(0x2000, 0x2000) AM_MIRROR(0x0002) AM_READWRITE(YM2203_status_port_0_r, YM2203_control_port_0_w)
	AM_RANGE(0x2001, 0x2001) AM_MIRROR(0x0002) AM_WRITE(YM2203_write_port_0_w)
	AM_RANGE(0x3000, 0x37ff) AM_RAM
	AM_RANGE(0x4000, 0x4000) AM_READWRITE(OKIM6295_status_0_r, OKIM6295_data_0_w)
	AM_RANGE(0x8000, 0xffff) AM_ROM
ADDRESS_MAP_END


/*------ YM3812-based sound ------*/
static ADDRESS_MAP_START( sound3812_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x0000) AM_WRITENOP
	AM_RANGE(0x1000, 0x1000) AM_READ(sound_data_r)
	AM_RANGE(0x2000, 0x2000) AM_READWRITE(YM3812_status_port_0_r, YM3812_control_port_0_w)
	AM_RANGE(0x2001, 0x2001) AM_WRITE(YM3812_write_port_0_w)
	AM_RANGE(0x3000, 0x37ff) AM_RAM
	AM_RANGE(0x4000, 0x4000) AM_READWRITE(OKIM6295_status_0_r, OKIM6295_data_0_w)
	AM_RANGE(0x5000, 0x5003) AM_READWRITE(pia_0_r, pia_0_w)
	AM_RANGE(0x8000, 0xffff) AM_ROM
ADDRESS_MAP_END


/*------ external YM3812-based sound board ------*/
static ADDRESS_MAP_START( sound3812_external_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x0000) AM_WRITENOP
	AM_RANGE(0x1000, 0x1000) AM_READ(sound_data_r)
	AM_RANGE(0x2000, 0x2000) AM_READWRITE(YM3812_status_port_0_r, YM3812_control_port_0_w)
	AM_RANGE(0x2001, 0x2001) AM_WRITE(YM3812_write_port_0_w)
	AM_RANGE(0x3000, 0x37ff) AM_RAM
	AM_RANGE(0x4000, 0x4000) AM_READWRITE(OKIM6295_status_0_r, OKIM6295_data_0_w)
	AM_RANGE(0x5000, 0x500f) AM_READWRITE(via_0_r, via_0_w)
	AM_RANGE(0x8000, 0xffff) AM_ROM
ADDRESS_MAP_END



/*************************************
 *
 *  Other CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( slikz80_mem_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7ff) AM_ROM
ADDRESS_MAP_END


static ADDRESS_MAP_START( slikz80_io_map, ADDRESS_SPACE_IO, 8 )
	ADDRESS_MAP_FLAGS( AMEF_ABITS(8) )
	AM_RANGE(0x00, 0x00) AM_READWRITE(slikz80_port_r, slikz80_port_w)
ADDRESS_MAP_END



/*************************************
 *
 *  Port definitions
 *
 *************************************/

#define UNUSED_ANALOG	\
	PORT_START	\
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )


static INPUT_PORTS_START( wfortune )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x07, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Blue Player") PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Yellow Player") PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Red Player") PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */

	PORT_START		/* analog D */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(75) PORT_KEYDELTA(10) PORT_PLAYER(1)

	UNUSED_ANALOG	/* analog E */

	PORT_START		/* analog F */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(75) PORT_KEYDELTA(10) PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( stratab )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Right Hook") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Left Hook") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Right Hook") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Left Hook") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START	/* analog D */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START	/* analog E */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START	/* analog F */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( gtg )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	/* it is still unknown how the second player inputs are muxed in */
	/* currently we map both sets of controls to the same inputs */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Swing") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Swing") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( gtgt )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START	/* analog D */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( gtg2t )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Face Right") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Face Left") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START	/* analog D */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START	/* analog E */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START	/* analog F */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( slikshot )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SPECIAL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Yellow")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Red")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Green")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */

	PORT_START_TAG("FAKEX") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START_TAG("FAKEY") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( dynobop )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* ball gate */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* ball detect */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */

	PORT_START_TAG("FAKEX") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START_TAG("FAKEY") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( sstrike )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SPECIAL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Left Hook")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Right Hook")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */

	PORT_START_TAG("FAKEX") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START_TAG("FAKEY") /* fake */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( pokrdice )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Lower Right") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x10, DEF_STR( On ))
	PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Upper Right") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Middle") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Lower Left") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Raise")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Upper Left") PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Play")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( hstennis )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ))	/* see code at fbb5 */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x10, DEF_STR( On ))
	PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Soft") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Hard") PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Soft") PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Hard") PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( arlingtn )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ))	/* see code at e23c */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x08, DEF_STR( On ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Place")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Win")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Show")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) PORT_NAME("Start Race")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Collect")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( peggle )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( pegglet )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x3e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */

	PORT_START		/* analog D */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( neckneck )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ))	/* see code at e23c */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x08, DEF_STR( On ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Horse 3")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Horse 2")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Horse 1")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Horse 4")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Horse 6")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Horse 5")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( rimrockn )
	PORT_START	/* 40 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* special 161 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Pass") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Shoot") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* special 162 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Pass") PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Shoot") PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	/* special 163 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P3 Shoot") PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P3 Pass") PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START3 )

	PORT_START	/* special 164 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P4 Shoot") PORT_PLAYER(4)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P4 Pass") PORT_PLAYER(4)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(4)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(4)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(4)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(4)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START4 )

	PORT_START	/* special 165 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE4 )
	PORT_DIPNAME( 0x18, 0x00, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x18, "1 player" )
	PORT_DIPSETTING(    0x10, "2 players" )
	PORT_DIPSETTING(    0x08, "3 players" )
	PORT_DIPSETTING(    0x00, "4 players" )
	PORT_DIPNAME( 0x20, 0x00, "Coin Slots" )
	PORT_DIPSETTING(    0x04, "Common" )
	PORT_DIPSETTING(    0x00, "Individual" )
	PORT_DIPNAME( 0x40, 0x00, "Video Sync" )
	PORT_DIPSETTING(    0x02, "Positive" )
	PORT_DIPSETTING(    0x00, "Negative" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


static INPUT_PORTS_START( ninclown )
	PORT_START	/* 40 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x0200, IP_ACTIVE_LOW )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 60 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("P1 Throw") PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Kick") PORT_PLAYER(1)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Punch") PORT_PLAYER(1)

	PORT_START	/* 80 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("P2 Throw") PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Kick") PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Punch") PORT_PLAYER(2)

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


static INPUT_PORTS_START( gtg2 )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Face Right") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Face Left") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	/* analog C */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START	/* analog D */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START	/* analog E */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START	/* analog F */
    PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( gpgolf )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 60 */

	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Swing") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END



/*************************************
 *
 *  Sound definitions
 *
 *************************************/

static const struct YM2203interface ym2203_interface =
{
	0,
	0,
	0,
	ym2203_portb_out,
	generate_sound_irq
};


static const struct YM3812interface ym3812_interface =
{
	generate_sound_irq
};



/*************************************
 *
 *  Machine driver
 *
 *************************************/

/************* core pieces ******************/

static MACHINE_DRIVER_START( itech8_core_lo )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M6809, CLOCK_8MHz/4)
	MDRV_CPU_PROGRAM_MAP(tmslo_map,0)
	MDRV_CPU_VBLANK_INT(generate_nmi,1)

	MDRV_SCREEN_REFRESH_RATE(60)

	MDRV_MACHINE_START(itech8)
	MDRV_MACHINE_RESET(itech8)
	MDRV_NVRAM_HANDLER(itech8)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_UPDATE_BEFORE_VBLANK)
	MDRV_VIDEO_START(itech8)

	MDRV_SCREEN_FORMAT(BITMAP_FORMAT_RGB32)
	MDRV_SCREEN_SIZE(512, 263)

	/* sound hardware */
	MDRV_SPEAKER_STANDARD_MONO("mono")

	MDRV_SOUND_ADD_TAG("oki", OKIM6295, CLOCK_8MHz/8) // was /128??
	MDRV_SOUND_CONFIG(okim6295_interface_region_1_pin7high) // was /128, not /132, so unsure so pin 7 not verified
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.75)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( itech8_core_hi )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PROGRAM_MAP(tmshi_map,0)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( itech8_sound_ym2203 )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("sound", M6809, CLOCK_8MHz/4)
	MDRV_CPU_PROGRAM_MAP(sound2203_map,0)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("ym", YM2203, CLOCK_8MHz/2)
	MDRV_SOUND_CONFIG(ym2203_interface)
	MDRV_SOUND_ROUTE(0, "mono", 0.07)
	MDRV_SOUND_ROUTE(1, "mono", 0.07)
	MDRV_SOUND_ROUTE(2, "mono", 0.07)
	MDRV_SOUND_ROUTE(3, "mono", 0.75)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( itech8_sound_ym3812 )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("sound", M6809, CLOCK_8MHz/4)
	MDRV_CPU_PROGRAM_MAP(sound3812_map,0)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("ym", YM3812, CLOCK_8MHz/2)
	MDRV_SOUND_CONFIG(ym3812_interface)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.75)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( itech8_sound_ym3812_external )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("sound", M6809, CLOCK_8MHz/4)
	MDRV_CPU_PROGRAM_MAP(sound3812_external_map,0)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("ym", YM3812, CLOCK_8MHz/2)
	MDRV_SOUND_CONFIG(ym3812_interface)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.75)
MACHINE_DRIVER_END


/************* full drivers ******************/

static MACHINE_DRIVER_START( wfortune )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2layer)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( stratab_hi )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2layer)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( stratab_lo )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2layer)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slikshot_hi )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	MDRV_CPU_ADD(Z80, CLOCK_8MHz/2)
	MDRV_CPU_PROGRAM_MAP(slikz80_mem_map,0)
	MDRV_CPU_IO_MAP(slikz80_io_map,0)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(slikshot)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slikshot_lo )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	MDRV_CPU_ADD(Z80, CLOCK_8MHz/2)
	MDRV_CPU_PROGRAM_MAP(slikz80_mem_map,0)
	MDRV_CPU_IO_MAP(slikz80_io_map,0)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(slikshot)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slikshot_lo_noz80 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_IMPORT_FROM(itech8_sound_ym2203)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2page)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( sstrike )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(slikshot_lo)
	MDRV_MACHINE_START(sstrike)

MACHINE_DRIVER_END


static MACHINE_DRIVER_START( hstennis_hi )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym3812)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 399, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2page_large)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( hstennis_lo )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_IMPORT_FROM(itech8_sound_ym3812)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 399, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2page_large)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( rimrockn )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym3812_external)

	MDRV_CPU_REPLACE("main", M6809/*HD6309*/, CLOCK_12MHz/4)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(24, 375, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2page_large)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( ninclown )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_hi)
	MDRV_IMPORT_FROM(itech8_sound_ym3812_external)

	MDRV_CPU_REPLACE("main", M68000, CLOCK_12MHz)
	MDRV_CPU_PROGRAM_MAP(ninclown_map,0)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(64, 423, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2page_large)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( gtg2 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(itech8_core_lo)
	MDRV_IMPORT_FROM(itech8_sound_ym3812_external)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PROGRAM_MAP(gtg2_map,0)

	/* video hardware */
	MDRV_SCREEN_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_VIDEO_UPDATE(itech8_2layer)
MACHINE_DRIVER_END



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( wfortune )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "wofpgm", 0x04000, 0x4000, CRC(bd984654) SHA1(8e16d2feb26e9a6f86c4a36bf0f03db80ded03f6) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "wofsnd", 0x08000, 0x8000, CRC(0a6aa5dc) SHA1(42eef40a4300d6d16d9e2af678432a02be05f104) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "wofgrom0", 0x00000, 0x10000, CRC(9a157b2c) SHA1(c349b41ba00cf6e2fec32872627c8cfdd8b5c1b9) )
	ROM_LOAD( "wofgrom1", 0x10000, 0x10000, CRC(5064739b) SHA1(424e3f94333f8ca21ac39b64b684cf6b487164d3) )
	ROM_LOAD( "wofgrom2", 0x20000, 0x10000, CRC(3d393b2b) SHA1(2c94d2dab7369c099c470cf96391b033f39add78) )
	ROM_LOAD( "wofgrom3", 0x30000, 0x10000, CRC(117a2ce9) SHA1(8d601c1cf9f783a42617f13c6862a5835553ac4f) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "wofsbom0", 0x00000, 0x20000, CRC(5c28c3fe) SHA1(eba64ede749fb26f9926f644d66860b54b4c76e7) )
ROM_END


ROM_START( wfortuna )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "wofpgmr1.bin", 0x04000, 0x4000, CRC(c3d3eb21) SHA1(21137663afd19fba875e188640f0347fc8c5dcf0) )
	ROM_CONTINUE(             0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "wofsnd", 0x08000, 0x8000, CRC(0a6aa5dc) SHA1(42eef40a4300d6d16d9e2af678432a02be05f104) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "wofgrom0", 0x00000, 0x10000, CRC(9a157b2c) SHA1(c349b41ba00cf6e2fec32872627c8cfdd8b5c1b9) )
	ROM_LOAD( "wofgrom1", 0x10000, 0x10000, CRC(5064739b) SHA1(424e3f94333f8ca21ac39b64b684cf6b487164d3) )
	ROM_LOAD( "wofgrom2", 0x20000, 0x10000, CRC(3d393b2b) SHA1(2c94d2dab7369c099c470cf96391b033f39add78) )
	ROM_LOAD( "wofgrom3", 0x30000, 0x10000, CRC(117a2ce9) SHA1(8d601c1cf9f783a42617f13c6862a5835553ac4f) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "wofsbom0", 0x00000, 0x20000, CRC(5c28c3fe) SHA1(eba64ede749fb26f9926f644d66860b54b4c76e7) )
ROM_END


ROM_START( stratab )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "sbprogv3.bin", 0x08000, 0x8000, CRC(a5ae728f) SHA1(85098eef1614d5148e8082df4c936883662292ee) )
	ROM_COPY( REGION_CPU1,    0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sbsnds.bin", 0x08000, 0x8000, CRC(b36c8f0a) SHA1(c4c3edf3352d95561f76705087338c1946137447) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a915b0bd) SHA1(0955c7ebb48e97ccffc18c5deec6eccce1d68de8) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(340c661f) SHA1(150f9158fa6d956d08051c67c17723b1d8c66867) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(5df9f1cf) SHA1(cad87f63ac0e902dffeeaa42538fc73f792d87d9) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( stratab1 )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "sbprgv1.bin",  0x08000, 0x8000, CRC(46d51604) SHA1(de7b6306fdcee4907b07667baf874bd195822e6a) )
	ROM_COPY( REGION_CPU1,    0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sbsnds.bin", 0x08000, 0x8000, CRC(b36c8f0a) SHA1(c4c3edf3352d95561f76705087338c1946137447) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a915b0bd) SHA1(0955c7ebb48e97ccffc18c5deec6eccce1d68de8) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(340c661f) SHA1(150f9158fa6d956d08051c67c17723b1d8c66867) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(5df9f1cf) SHA1(cad87f63ac0e902dffeeaa42538fc73f792d87d9) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( gtg )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, CRC(61984272) SHA1(be735f8576fb2cccc0e9e6ea6f2fd54b6c0b3bb3) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(358d2440) SHA1(7b09350c89f9d2c86dc187d8812bbf26b576a38f) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(5b393314) SHA1(9e314a75ea52373369904915ec786f09eee725a9) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(1cccbfdf) SHA1(546059fea2e7cd5627a666d80b1fc3ed8fcc0762) )
ROM_END


ROM_START( gtgt )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "pgm-u5.512", 0x04000, 0x4000, CRC(ec70b510) SHA1(318984d77eb1df6258b855781ae1c9a09aa74f15) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "snd-u27.256", 0x08000, 0x8000, CRC(471da557) SHA1(32bfe450a42d9eb6c14edcfa2b4e33f65a11126e) )

	ROM_REGION( 0xb0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "grom5.512", 0xa0000, 0x10000, CRC(44b47015) SHA1(5dde4c932a697b51fe02eab8d948889b3fe7baff) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.010", 0x00000, 0x20000, CRC(d041e0c9) SHA1(1d90f37071d92c714ff69ab2b0337c7c66147995) )
ROM_END


ROM_START( gtg2t )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5",     0x10000, 0x4000, CRC(c7b3a9f3) SHA1(5edaca6fd6ee58bd1676dc9b2c86da4dd2f51687) )
	ROM_CONTINUE(       0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(dd2a5905) SHA1(dc93f13de3953852a6757361eb9683a57d3ed326) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom00.bin", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) )

	ROM_REGION( 0x0200, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "tibpal16l8.u11", 0x0000, 0x0104, CRC(9bf5a75f) SHA1(79786f7ce656f30a33a92887a290b767a7cbbf31) )
ROM_END


ROM_START( gtg2j )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, CRC(9c95ceaa) SHA1(d9fd2b2419c026822a07d2ba51d6ab40b7cd0d49) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(dd2a5905) SHA1(dc93f13de3953852a6757361eb9683a57d3ed326) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(1cccbfdf) SHA1(546059fea2e7cd5627a666d80b1fc3ed8fcc0762) )

	ROM_REGION( 0x0200, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "tibpal16l8.u11", 0x0000, 0x0104, CRC(9bf5a75f) SHA1(79786f7ce656f30a33a92887a290b767a7cbbf31) )
ROM_END


ROM_START( slikshot )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "pgm20.u5",  0x04000, 0x4000, CRC(370a00eb) SHA1(b2878f161f4931d9fc3979a84b29660941e2608f) )
	ROM_CONTINUE(          0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u53.bin", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(e60c2804) SHA1(e62d11b6c4439a70a2f32df72c8c64e2f110351e) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(d764d542) SHA1(43fc0c9b627484a670d87da91e212741b137e995) )

	ROM_REGION( 0x10000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( sliksh17 )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, CRC(09d70554) SHA1(a009cd3b22261c60f1028694baef51f61713154f) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u53.bin", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(e60c2804) SHA1(e62d11b6c4439a70a2f32df72c8c64e2f110351e) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(d764d542) SHA1(43fc0c9b627484a670d87da91e212741b137e995) )

	ROM_REGION( 0x10000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( sliksh16 )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "poolpgm-16.u5", 0x04000, 0x4000, CRC(c0f17012) SHA1(5d466e058daf91b4f52e634498df9d2a03627aaa) )
	ROM_CONTINUE(              0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1,     0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u53.bin", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "pool-grom.0", 0x00000, 0x10000, CRC(e6d0edc6) SHA1(5287a31bbdde1e4291d8e9e6b99d3aa12bfb6e18) )
	ROM_LOAD( "pool-grom.1", 0x10000, 0x10000, CRC(5a071aa2) SHA1(9c5506e37625d213429b1231d457d7ce8a7a81ff) )
	ROM_LOAD( "pool-grom.2", 0x20000, 0x10000, CRC(c0bdf4e0) SHA1(3b7c635375c5e5fddcbc1bd1b186c960081ec37e) )
	ROM_LOAD( "pool-grom.3", 0x30000, 0x10000, CRC(cb0bd9a3) SHA1(107ff127f9adad84a5f92077851423249fce8e30) )

	ROM_REGION( 0x10000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( dynobop )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "dynobop.u5", 0x04000, 0x4000, CRC(98452c40) SHA1(9b9316fc258792e0d825f16e0fadf8e0c35a864e) )
	ROM_CONTINUE(           0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1,  0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "dynobop.u27", 0x08000, 0x8000, CRC(a37d862b) SHA1(922eeae184df2c5c28040da27699dd55744f8dca) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "dynobop.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(            0x00000, 0x0800 )
	ROM_CONTINUE(            0x00000, 0x0800 )
	ROM_CONTINUE(            0x00000, 0x0800 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "dynobop.gr0", 0x00000, 0x20000, CRC(3525a7a3) SHA1(fe0b08203c135d55507506936dc34e1503e4906b) )
	ROM_LOAD( "dynobop.gr1", 0x20000, 0x20000, CRC(1544a232) SHA1(60bba76537c82887db8e38c6a87c528afdd385d0) )

	ROM_REGION( 0x10000, REGION_SOUND1, 0 )
	ROM_LOAD( "dynobop.sr0", 0x00000, 0x10000, CRC(b355bf1d) SHA1(c9745638feb33ec181b8d9a894421f9854286cd9) )
ROM_END


ROM_START( sstrike )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "sstrku5.bin", 0x08000, 0x8000, CRC(af00cddf) SHA1(b866e8dfce1449f7462a79efa385ea6b55cdc6e7) )
	ROM_COPY( REGION_CPU1,    0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sstrku27.bin", 0x08000, 0x8000, CRC(efab7252) SHA1(eb3b2002531e551e3d67958ea3cc56a69fa660e2) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "spstku53.bin", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "sstgrom0.bin", 0x00000, 0x20000, CRC(9cfb9849) SHA1(5aa860c0c6e3916ebdb8898ee44f633bf3347ca8) )
	ROM_LOAD( "sstgrom1.bin", 0x20000, 0x20000, CRC(d9ea14e1) SHA1(4cddf3237c203b0a3f7ae770f85f1be35e9e1b78) )
	ROM_LOAD( "sstgrom2.bin", 0x40000, 0x20000, CRC(dcd97bf7) SHA1(95361222ac58bf74539f2a7e80574bcd848c615e) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "sstsrom0.bin", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( pokrdice )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "pd-v17.u5", 0x04000, 0x4000, CRC(5e24be82) SHA1(97e50cc023ff651fb09cc5e85a1bef1bc234ccb9) )
	ROM_CONTINUE(          0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "pd-snd.bin", 0x08000, 0x8000, CRC(4925401c) SHA1(e35983bec4a0dd4cb1d942fd909790b1adeb415d) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "pd-grom0.bin", 0x00000, 0x20000, CRC(7c2573e7) SHA1(d6a2a16277ad854c66927d88c5617d05eefe1057) )
	ROM_LOAD( "pd-grom1.bin", 0x20000, 0x20000, CRC(e7c06aeb) SHA1(4be54b078d886359bf6ed376019cc1f6f04f52d6) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "pd-srom.0", 0x00000, 0x10000, CRC(f85dbd6f) SHA1(fce53019432e4a84e52f0ae1996b2e0d94b32bc0) )
ROM_END


ROM_START( hstennis )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "tenbim.v11", 0x04000, 0x4000, CRC(faffab5c) SHA1(4de525f6adb16205c47788b78aecdebd57008295) )
	ROM_CONTINUE(           0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1,  0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "tensnd.v1", 0x08000, 0x8000, CRC(f034a694) SHA1(3540e2edff2ce47504260ec856bab9b638d9260d) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(1e69ebae) SHA1(a32e2c2f4e4a527ae6b57adeccd2c4d2045ab5fe) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(4e6a22d5) SHA1(3c2d51dd874f61c0a557ea2c1968afa02d9bfc42) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(c0b643a9) SHA1(d240f703a55c39ce4a969612fbb9cd76e4b849ac) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(54afb456) SHA1(0a7f10b1a490825a85489ac3f678eb1e11a0ac65) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(ee09d645) SHA1(8233cfbd17d9de548ae3ca5e30928061712e0781) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(d9ce58c3) SHA1(92574e60497d86b8608fba6278ccfc9036cb7f22) )

	ROM_REGION( 0x0200, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "pal16l8-itvs.u11", 0x0000, 0x0104, CRC(fee03727) SHA1(e784ff18505cdccc1020dbe5cb0e7cc9efc068a4) )
ROM_END


ROM_START( hstenn10 )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "tenbim.v10", 0x04000, 0x4000, CRC(d108a6e0) SHA1(1041e1d95b10245fc50f6484e710803db2706f9a) )
	ROM_CONTINUE(           0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "tensnd.v1", 0x08000, 0x8000, CRC(f034a694) SHA1(3540e2edff2ce47504260ec856bab9b638d9260d) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(1e69ebae) SHA1(a32e2c2f4e4a527ae6b57adeccd2c4d2045ab5fe) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(4e6a22d5) SHA1(3c2d51dd874f61c0a557ea2c1968afa02d9bfc42) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(c0b643a9) SHA1(d240f703a55c39ce4a969612fbb9cd76e4b849ac) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(54afb456) SHA1(0a7f10b1a490825a85489ac3f678eb1e11a0ac65) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(ee09d645) SHA1(8233cfbd17d9de548ae3ca5e30928061712e0781) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(d9ce58c3) SHA1(92574e60497d86b8608fba6278ccfc9036cb7f22) )

	ROM_REGION( 0x0200, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "pal16l8-itvs.u11", 0x0000, 0x0104, CRC(fee03727) SHA1(e784ff18505cdccc1020dbe5cb0e7cc9efc068a4) )
ROM_END


ROM_START( arlingtn )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "ahrd121.bin", 0x10000, 0x4000, CRC(00aae02e) SHA1(3bcfbd256c34ae222dde24ba9544f19da70b698e) )
	ROM_CONTINUE(            0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "ahrsnd11.bin", 0x08000, 0x8000, CRC(dec57dca) SHA1(21a8ead10b0434629f41f6b067c49b6622569a6c) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(5ef57fe5) SHA1(e877979e034a61968b432037501e25a302a17a9a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(6aca95c0) SHA1(da7a899bf0812a7af178e48b5a626ce56a836579) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x10000, CRC(6d6fde1b) SHA1(aaabc45d4b566be42e8d28d767e4771a96d9caae) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x40000, CRC(56087f81) SHA1(1d4a1f396ee9d8ed51d0417ea94b0b379312d72f) )
ROM_END


ROM_START( peggle )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "j-stick.u5", 0x04000, 0x4000, CRC(140d5a9c) SHA1(841e5f45c6f306d9bd286e7d3e3c75b169c932e1) )
	ROM_CONTINUE(           0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, CRC(b99beb70) SHA1(8d82c3b081a1afb236afa658abb3aa605c6c2264) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(5c02348d) SHA1(c85352728d94b3a5ca78c2493e98bdb4b3206bed) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(85a7a3a2) SHA1(7fb7be67c75867fcccf4272f1ce42f2af94c16e6) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(bfe11f18) SHA1(5e8d36c6a86a8eb883f50620fbfcea3d8398faeb) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, CRC(001846ea) SHA1(6a6b77f6d51c4a68105c935804b896aad9981719) )
ROM_END


ROM_START( pegglet )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "trakball.u5", 0x04000, 0x4000, CRC(d2694868) SHA1(9945a308550c9d89a647d80257e3ab14f793ac6f) )
	ROM_CONTINUE(            0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, CRC(b99beb70) SHA1(8d82c3b081a1afb236afa658abb3aa605c6c2264) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(5c02348d) SHA1(c85352728d94b3a5ca78c2493e98bdb4b3206bed) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(85a7a3a2) SHA1(7fb7be67c75867fcccf4272f1ce42f2af94c16e6) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(bfe11f18) SHA1(5e8d36c6a86a8eb883f50620fbfcea3d8398faeb) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, CRC(001846ea) SHA1(6a6b77f6d51c4a68105c935804b896aad9981719) )
ROM_END


ROM_START( neckneck )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "nn_prg12.u5", 0x04000, 0x4000, CRC(8e51734a) SHA1(c184af73670235a9245bfdeec2b58acfe93170e3) )
	ROM_CONTINUE(            0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "nn_snd10.u27", 0x08000, 0x8000, CRC(74771b2f) SHA1(0a963d2962699bb1b4d08bd486979151d0a228da) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "nn_grom0.bin", 0x00000, 0x20000, CRC(064d1464) SHA1(4ce5bcadad93586c9af2b0c499d7a7140b080cdb) )
	ROM_LOAD( "nn_grom1.bin", 0x20000, 0x20000, CRC(622d9a0b) SHA1(f084dcf194cfc658f97f36972f2b2ff30e834fc5) )
	ROM_LOAD( "nn_grom2.bin", 0x40000, 0x20000, CRC(e7eb4020) SHA1(29ceb535d1dfddf8c12cf78919ea10f6b956b8b3) )
	ROM_LOAD( "nn_grom3.bin", 0x60000, 0x20000, CRC(765c8593) SHA1(a39d757a01d61b637b264bdb4449e090a1e90219) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "nn_srom0.bin", 0x00000, 0x40000, CRC(33687201) SHA1(7c84d453fb7e59b0a27cd68a049d019ff65f6f19) )
ROM_END


ROM_START( rimrockn )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "u5-2_2",    0x04000, 0x4000, CRC(97777683) SHA1(0998dde26daaa2d2b78e83647e03ba01b0ef31f2) )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00",       0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "grom01",       0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "grom02-2.st2", 0x80000, 0x40000, CRC(47904233) SHA1(6a4d10e8f7b75582f706a74b37d59788613ffc61) )
	ROM_LOAD( "grom03-2.st2", 0xc0000, 0x40000, CRC(f005f118) SHA1(aa39f12d07827e21eceb286557e37973e519b433) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )

	ROM_REGION( 0x0600, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "pal16l8.u14", 0x0000, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4.u45", 0x0200, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8.u29", 0x0400, 0x0104, NO_DUMP ) /* PAL is read protected */
ROM_END


ROM_START( rimrck20 )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "rrb.bin",   0x04000, 0x4000, CRC(7e9d5545) SHA1(2aa028b3f5d05bec4ee289e7d39eaad30b3d4d5f) )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00",       0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "grom01",       0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "grom02-2.st2", 0x80000, 0x40000, CRC(47904233) SHA1(6a4d10e8f7b75582f706a74b37d59788613ffc61) )
	ROM_LOAD( "grom03-2.st2", 0xc0000, 0x40000, CRC(f005f118) SHA1(aa39f12d07827e21eceb286557e37973e519b433) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrck16 )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "rrbbv16.u5",0x04000, 0x4000, CRC(999cd502) SHA1(8ad0d641a9f853eff27be1d4de04ab86b9275d57) )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrck12 )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "rrbbv12.u5",0x04000, 0x4000, CRC(661761a6) SHA1(7224b1eac2fd0969d70657448ab241a433143df4) )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "rrbsndv1.u27", 0x08000, 0x8000, CRC(8eda5f53) SHA1(f256544a8c87125587719460ed0fef14efef9015) )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( ninclown )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "prog1", 0x00000, 0x20000, CRC(fabfdcd2) SHA1(7a9852838cf7772d8f8f956b03823c4222520a5a) )
	ROM_LOAD16_BYTE( "prog0", 0x00001, 0x20000, CRC(eca63db5) SHA1(b86d75ee81e155a21de18c2e0fa898f15d61560d) )
	ROM_COPY(    REGION_CPU1, 0x00000, 0x40000, 0x40000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "nc-snd", 0x08000, 0x8000, CRC(f9d5b4e1) SHA1(e5c3774db349b60baf11baecf55ac432871e612c) )

	ROM_REGION( 0x180000, REGION_GFX1, 0 )
	ROM_LOAD( "nc-grom0", 0x000000, 0x40000, CRC(532f7bff) SHA1(cc7a64ad1581f37ff4bcad78c11aad355a6e9aa8) )
	ROM_LOAD( "nc-grom1", 0x040000, 0x40000, CRC(45640d4a) SHA1(1b59ae2a74fef535c646586422f17b39c9ae247b) )
	ROM_LOAD( "nc-grom2", 0x080000, 0x40000, CRC(c8281d06) SHA1(f75d4e858f9bc19e5234f184b75818e448638c15) )
	ROM_LOAD( "nc-grom3", 0x0c0000, 0x40000, CRC(2a6d33ac) SHA1(0773a53ece0b09e7c33f0e5beb4518059f76b7af) )
	ROM_LOAD( "nc-grom4", 0x100000, 0x40000, CRC(910876ba) SHA1(d65d0a5043afe2239596ba76174a27fea60ffb35) )
	ROM_LOAD( "nc-grom5", 0x140000, 0x40000, CRC(2533279b) SHA1(c21b218b11d20f2cdac2321ffcb1334ec9ec9ffc) )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x40000, CRC(f6b501e1) SHA1(0f1eb6f40f891d11822f1255c0a84975526b2b1b) )
ROM_END


ROM_START( gpgolf )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "gpgjv1_1.bin",   0x04000, 0x4000, CRC(631e77e0) SHA1(847ba1e00d31441620a2a1f45a9aa58df84bde8b) )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sndv1.u27", 0x08000, 0x8000, CRC(55734876) SHA1(eb5ef816acbc6e35642749e38a2908b7ba359b9d) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00.bin", 0x00000, 0x40000, CRC(c3a7b54b) SHA1(414d693bc5337d578d2630817dd647cf7e5cbcf7) )
	ROM_LOAD( "grom01.bin", 0x40000, 0x40000, BAD_DUMP CRC(2c834cf9) SHA1(49eb070ca4439d4fcd2d8a2db62d3c99b824bf99) ) /* Self test checksum reports BAD */
	ROM_LOAD( "grom02.bin", 0x80000, 0x40000, CRC(aebe6c45) SHA1(15e64fcb36cb1064988ee5cd45699d501a6e7f01) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom00.bin", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) )
ROM_END


ROM_START( gtg2 )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.2",   0x10000, 0x4000, CRC(4a61580f) SHA1(7c64648d47418fbcc0f9b5bd91f88856209bc0f5) )
	ROM_CONTINUE(       0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sndv1.u27", 0x08000, 0x8000, CRC(55734876) SHA1(eb5ef816acbc6e35642749e38a2908b7ba359b9d) )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom00.bin", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) )

	ROM_REGION( 0x0200, REGION_PLDS, ROMREGION_DISPOSE )
	ROM_LOAD( "tibpal16l8.u29", 0x0000, 0x0104, NO_DUMP ) /* PAL is read protected */
ROM_END



/*************************************
 *
 *  Driver-specific init
 *
 *************************************/

static DRIVER_INIT( slikshot )
{
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0180, 0x0180, 0, 0, slikshot_z80_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x01cf, 0x01cf, 0, 0, slikshot_z80_control_r);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x01cf, 0x01cf, 0, 0, slikshot_z80_control_w);
}


static DRIVER_INIT( sstrike )
{
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x1180, 0x1180, 0, 0, slikshot_z80_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x11cf, 0x11cf, 0, 0, slikshot_z80_control_r);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x11cf, 0x11cf, 0, 0, slikshot_z80_control_w);
}


static DRIVER_INIT( hstennis )
{
	static const rectangle visible = { 0, 375, 0, 239 };
	visarea = &visible;
}


static DRIVER_INIT( arligntn )
{
	static const rectangle visible = { 16, 389, 0, 239 };
	visarea = &visible;
}


static DRIVER_INIT( peggle )
{
	static const rectangle visible = { 18, 367, 0, 239 };
	visarea = &visible;
}


static DRIVER_INIT( neckneck )
{
	static const rectangle visible = { 8, 375, 0, 239 };
	visarea = &visible;
}


static DRIVER_INIT( rimrockn )
{
	/* additional input ports */
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0160, 0x0160, 0, 0, special_port1_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0161, 0x0161, 0, 0, input_port_3_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0162, 0x0162, 0, 0, input_port_4_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0163, 0x0163, 0, 0, input_port_5_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0164, 0x0164, 0, 0, input_port_6_r);
	memory_install_read8_handler (0, ADDRESS_SPACE_PROGRAM, 0x0165, 0x0165, 0, 0, input_port_7_r);

	/* different banking mechanism (disable the old one) */
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x01a0, 0x01a0, 0, 0, rimrockn_bank_w);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x01c0, 0x01df, 0, 0, itech8_blitter_w);
}



/*************************************
 *
 *  Game drivers
 *
 *************************************/

/* Wheel of Fortune-style PCB */
GAME( 1989, wfortune, 0,        wfortune,          wfortune, 0,        ROT0,   "GameTek", "Wheel Of Fortune", 0 )
GAME( 1989, wfortuna, wfortune, wfortune,          wfortune, 0,        ROT0,   "GameTek", "Wheel Of Fortune (alternate)", 0 )

/* Strata Bowling-style PCB */
GAME( 1990, stratab,  0,        stratab_hi,        stratab,  0,        ROT270, "Strata/Incredible Technologies", "Strata Bowling (V3)", 0 )
GAME( 1990, stratab1, stratab,  stratab_hi,        stratab,  0,        ROT270, "Strata/Incredible Technologies", "Strata Bowling (V1)", 0 )
GAME( 1990, gtg,      0,        stratab_hi,        gtg,      0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Joystick, v3.1)", 0 )
GAME( 1989, gtgt,     gtg,      stratab_hi,        gtgt,     0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Trackball, v1.0)", 0 )
GAME( 1989, gtg2t,    gtg2,     stratab_hi,        gtg2t,    0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V1.1)", 0 )
GAME( 1991, gtg2j,    gtg2,     stratab_lo,        gtg,      0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Joystick, V1.0)", 0 )

/* Slick Shot-style PCB */
GAME( 1990, slikshot, 0,        slikshot_hi,       slikshot, slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V2.2)", 0 )
GAME( 1990, sliksh17, slikshot, slikshot_hi,       slikshot, slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V1.7)", 0 )
GAME( 1990, sliksh16, slikshot, slikshot_hi,       slikshot, slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V1.6)", 0 )
GAME( 1990, dynobop,  0,        slikshot_hi,       dynobop,  slikshot, ROT90,  "Grand Products/Incredible Technologies", "Dyno Bop", 0 )
GAME( 1990, sstrike,  0,        sstrike,           sstrike,  sstrike,  ROT270, "Strata/Incredible Technologies", "Super Strike Bowling", 0 )
GAME( 1991, pokrdice, 0,        slikshot_lo_noz80, pokrdice, 0,        ROT90,  "Strata/Incredible Technologies", "Poker Dice", 0 )

/* Hot Shots Tennis-style PCB */
GAME( 1990, hstennis, 0,        hstennis_hi,       hstennis, hstennis, ROT90,  "Strata/Incredible Technologies", "Hot Shots Tennis (V1.1)", 0 )
GAME( 1990, hstenn10, hstennis, hstennis_hi,       hstennis, hstennis, ROT90,  "Strata/Incredible Technologies", "Hot Shots Tennis (V1.0)", 0 )
GAME( 1991, arlingtn, 0,        hstennis_hi,       arlingtn, arligntn, ROT0,   "Strata/Incredible Technologies", "Arlington Horse Racing (v1.21-D)", 0 )
GAME( 1991, peggle,   0,        hstennis_lo,       peggle,   peggle,   ROT90,  "Strata/Incredible Technologies", "Peggle (Joystick, v1.0)", 0 )
GAME( 1991, pegglet,  peggle,   hstennis_lo,       pegglet,  peggle,   ROT90,  "Strata/Incredible Technologies", "Peggle (Trackball, v1.0)", 0 )
GAME( 1992, neckneck, 0,        hstennis_lo,       neckneck, neckneck, ROT0,   "Bundra Games/Incredible Technologies", "Neck-n-Neck (v1.2)", 0 )

/* Rim Rockin' Basketball-style PCB */
GAME( 1991, rimrockn, 0,        rimrockn,          rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.2)", 0 )
GAME( 1991, rimrck20, rimrockn, rimrockn,          rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.0)", 0 )
GAME( 1991, rimrck16, rimrockn, rimrockn,          rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.6)", 0 )
GAME( 1991, rimrck12, rimrockn, rimrockn,          rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.2)", 0 )

/* Ninja Clowns-style PCB */
GAME( 1991, ninclown, 0,        ninclown,          ninclown, 0,        ROT0,   "Strata/Incredible Technologies", "Ninja Clowns (08/27/91)", 0 )

/* Golden Tee Golf II-style PCB */
GAME( 1992, gpgolf,   0,        gtg2,              gpgolf,   0,        ROT0,   "Strata/Incredible Technologies", "Golden Par Golf (Joystick, V1.1)", 0 )
GAME( 1992, gtg2,     0,        gtg2,              gtg2,     0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V2.2)", 0 )
