/*
 * robfrontend.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file robfrontend.cpp
  * Implements frontends.
  */

#include "robfrontend.h"
#include "robmanage.h"
#include "robtopmgr.h"
#include "robvars.h"
#include "robstrings.h"
#include "robtrans.h"

// plugins
#ifndef __NO_SIF__
#  include "robstreamif.h"
#endif
#include "robsound.h"
#include "robrating.h"
#include "robprofile.h"
#ifdef __WXWIN__
#  include "robvis.h"
#endif

#include <rtmath.h>
#include <rtcollect.h>
#include <rtiterator.h>

using namespace lrt;

namespace rt {

/////////////// Frontend

void Frontend::initPlugins()
{
#ifndef __NO_SIF__
	plugins.append(new StreamInterface(this));
#endif
	plugins.append(new SoundPlugin(this));
	plugins.append(new RatingPlugin(this));
	plugins.append(new ProfilePlugin(this));
#ifdef __WXWIN__
	plugins.append(new VisPlugin(this));
#endif
}

FrontendPlugin* Frontend::findPlugin(const String& name)
{
	FrontendPlugin* ret = 0;
	for(List<FrontendPlugin*>::Node* n = plugins.first(); n; n = n->getNext())
		if(n->accessElement()->getCodeName() == name)
			ret = n->accessElement();
	return ret;
}

Frontend::~Frontend() 
{
	deleteAll(plugins.begin());
	delete manager;
}

void Frontend::reset()
{
	// reset all of the plugins
	for(Iterator<FrontendPlugin*> iter = plugins.begin(); iter.hasElement(); ++iter)
	{
		iter.get()->reset(); 
	}

	// we don't have to do anything ourselves, 
	// but derived classes may have to. 
}

bool Frontend::interpreteParams(const Array<String>& params)
{
	files.clear(); // needed when doing another tournament (doAllAgain)

	Array<bool> used(params.length() + 1);
	for(int u = 0; u < params.length(); u++)
		used[u] = false;
	used[params.length()] = true;
	
	// do plugins before
	for(Iterator<FrontendPlugin*> iter = plugins.begin(); iter.hasElement(); ++iter)
	{
		FrontendPlugin* plug = iter.get();
		if(plug->getInterpretePos() != fpInterPosBefore)
			continue;
		if(!plug->interpreteParams(params, used))
			return false;
	}

	// do myself
	if(!interpreteParamsImpl(params, used))
		return false;

	// do plugins after
	for(Iterator<FrontendPlugin*> iter2 = plugins.begin(); iter2.hasElement(); ++iter2)
	{
		FrontendPlugin* plug = iter2.get();
		if(plug->getInterpretePos() != fpInterPosAfter)
			continue;
		if(!plug->interpreteParams(params, used))
			return false;
	}
	
	return true;
}

void Frontend::fillOptions(SimOptions& options)
{
	options.numRepeats = numRepeats;
	if(!options.glob.getGlobals(optionFile))
		handleSystemError(18, String::format(_("Could not read option file %s"), optionFile.cStr()));
	if(noRandom) Math::seed(1);

	if(files.length() < 1) {
		handleSystemError(17, _("No robots specified."));
		return;
	}
	options.files = files;

	// do plugins
	for(Iterator<FrontendPlugin*> iter = plugins.begin(); iter.hasElement(); ++iter)
	{
		FrontendPlugin* plug = iter.get();
		plug->fillOptions(options);
	}
}

SimManager* Frontend::getManager()
{
	// re-creating the manager in another tournament
	if(manager) {
		delete manager;
		manager = 0;
	}

	switch(fightType) {
		case ftSingle:
			manager = new ChartsManager(false);
			break;
		case ftCharts:
			manager = new ChartsManager(true);
			break;
#ifndef __NO_TOP_MGR__
		case ftTop:
			manager = new TopManager(false, fightTypeParam, competitionFolder);
			break;
		case ftTopWithOld:
			manager = new TopManager(true, fightTypeParam, competitionFolder);
			break;
#endif
		case ftAllInOne:
			manager = new AllInOneManager();
			break;
		default:
			System::exit(49, _("Unknown fight type!"));
	}
	return manager;
}

Frontend::Frontend() : doAllAgain(false), manager(0), fightType(ftCharts), fightTypeParam(0),
	noRandom(false), numRepeats(10), optionFile("robocom.rco"), files(0)
{
	competitionFolder = File::getCurrentFolder();
	initPlugins();
}



} // namespace

