/*
 * makehtml.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */
// implements most of the makehtml program
// (the HtmlCreator classes in robhtml.cpp are heavily used too) 

#include <rtstring.h>
#include <rtcollect.h>
#include <rtsystem.h>
#include <rtfile.h>

#include "robvars.h"
#include "robmanage.h"
#include "robhtml.h"
#include "robtrans.h"

// Have Resources => Have Fun!
#include "resources.rh"

// always include this file as the last one
#include <rtmain.h>

namespace rt {

class MkHtmlArgs{
public:
	MkHtmlArgs() : links(true), downloadLinks(true),
		compFolder(lrt::File::getCurrentFolder()), outFolder(lrt::File::getCurrentFolder()),
		files(0), compName(_("New Competition")), recalc(false), printHelp(false) {}

	bool links, downloadLinks;
	lrt::File compFolder, outFolder;
	lrt::Vector<lrt::String> files;
	lrt::String compName;
	bool recalc;
	bool printHelp;
};

MkHtmlArgs parseArgs(const lrt::Array<lrt::String>& args)
{
	MkHtmlArgs margs;

	for( int i = 0; i < args.length(); i++ )
	{
		if( args[i][0] == '-' )
		{
			lrt::String arg = args[i].substring(1);

			// competition "name"
			if(!arg.compareIgnoreCase("name"))
			{
				i++;
				if(i >= args.length())
					lrt::System::exit(1, _("Parameter missing for argument -") + arg);
				margs.compName = args[i];
			}
			// "c"ompetition "f"older
			else if(!arg.compareIgnoreCase("cf"))
			{
				i++;
				if(i >= args.length())
					lrt::System::exit(1, _("Parameter missing for argument -") + arg);
				// ensure that a path was given
				lrt::String temp = args[i];
				if(!temp.endsWith("/") && !temp.endsWith(lrt::File::separator))
					temp += lrt::File::separatorChar;

				margs.compFolder = temp;
			}
			// "o"utput "f"older
			else if(!arg.compareIgnoreCase("of"))
			{
				i++;
				if(i >= args.length())
					lrt::System::exit(1, _("Parameter missing for argument -") + arg);
				// ensure that a path was given
				lrt::String temp = args[i];
				if(!temp.endsWith("/") && !temp.endsWith(lrt::File::separator))
					temp += lrt::File::separatorChar;

				margs.outFolder = temp;
			}
			// "batch" mode (non-interactive)
			else if(!arg.compareIgnoreCase("batch"))
			{
				lrt::System::setInteractive(false);
			}
			// create "no links"
			else if(!arg.compareIgnoreCase("nolinks"))
			{
				margs.links = false;
			}
			// create "no download" links
			else if(!arg.compareIgnoreCase("nodownload"))
			{
				margs.downloadLinks = false;
			}
			// recalculate the cahts
			else if(!arg.compareIgnoreCase("recalc"))
			{
				margs.recalc = true;
			}
			// "h"elp or "help"
			else if((arg == "h") || (arg == "H") || !arg.compareIgnoreCase("help"))
			{
				margs.printHelp = true;
			}

#ifdef __SYMBIAN32__
			// current folder
			else if(!arg.compareIgnoreCase("cd"))
			{
				i++; // look at next arg
				if(i >= args.length())
					System::exit(1, _("Parameter missing for argument -") + arg);
				lrt::String newCurrentFolder = args[i];
				// check that a path was given
				char c = newCurrentFolder[newCurrentFolder.length() - 1];
				// no path => append slash
				if(!(c == '/') && !(c == lrt::File::separatorChar))
					newCurrentFolder += lrt::File::separatorChar;

				// get competition folders
				lrt::String compFolder = margs.compFolder.getLocalName();
				lrt::String outFolder = margs.outFolder.getLocalName();

				// set current folder
				lrt::File::setCurrentFolder(File(newCurrentFolder));

				// set competition folders
				margs.compFolder = lrt::File(compFolder);
				margs.outFolder = lrt::File(outFolder);
			}
#endif
			else
			{
				lrt::System::println(_("Warning: unknown parameter ") + args[i]);
			}
		}
		else // file or file pattern
		{
			int starPos = args[i].indexOf('*'); // has got a star?
			if(starPos < 0) // no, so just add it
				margs.files += lrt::File(margs.compFolder, args[i]).getLocalName(margs.compFolder);
			else { // yes, so add all files matching the pattern
				lrt::Array<lrt::File> f(lrt::StarFilenameFilter::getFiles(
					lrt::File(margs.compFolder, args[i]).getName()));
				for(int fi = 0; fi < f.length(); fi++)
					margs.files += f[fi].getLocalName(margs.compFolder);
			}
		}
	}

	return margs;
}

void printHelp()
{
	lrt::System::println(_("Creates charts and detailed results for competitions\n"
		 				   "that were run using the top mode (-t) of RoboTour. "));
	lrt::System::println();

	lrt::System::println(_("Usage: makehtml [options] [prog1 [prog2 [...]]]"));
	lrt::System::println(lrt::String(_("Version ")) + getVersion());
	lrt::System::println();
	lrt::System::println(_("Options:"));
#ifdef __SYMBIAN32__
	lrt::System::println(_("  -cd F      Set current folder to F before running robotour."));
#endif
	lrt::System::println(_("  -name N     Sets the competition name to N (default: \"New Competition\")"));
	lrt::System::println(_("  -cf   F     Sets the competition folder to F (default: current)"));
	lrt::System::println(_("  -of   F     Sets the output folder to F (default: current)"));
	lrt::System::println(_("  -nolinks    Does not create links between the pages. "));
	lrt::System::println(_("  -nodownload Does not create download links for the rob files."));
	lrt::System::println(_("  -recalc     Recalculates the charts (useful after kickbot). "));
 	lrt::System::println(_("  -h, -help   Print this help text.")); 
	lrt::System::println(_("  progn       Any robot file name, or file pattern.\n"
						   "              These file names should be entered relative to the\n"
						   "              competition folder (see -cf). If no files are given,\n"
						   "              only the charts page, but no detailed pages is created."));
}

} // namespace

/** Since rtmain.h includes system includes, we may not import any namespace in
  * this file. Only in this file, really. */

int rtRun(const lrt::Array<lrt::String>& args)
{
	lrt::System::setAppName("MakeHTML");
	initAllResources();

	rt::MkHtmlArgs margs = rt::parseArgs(args);

	if(margs.printHelp)
	{
		rt::printHelp();
		return 0;
	}

	rt::HtmlCreator html(margs.compName, margs.compFolder, margs.outFolder, 
		margs.links, margs.downloadLinks);
	if(!html.createChartsHtml(margs.recalc)) 
	{
		lrt::System::println(_("warning: couldn't create charts html"));
		if(!margs.files.length()) // probably, no args at all
			rt::printHelp();
	}

	for(int f = 0; f < margs.files.length(); f++)
	{
		if(!html.createDetailedHtml(margs.files[f]))
			lrt::System::println(_("warning: couldn't create detailed html for robot ") + 
				margs.files[f]);
	}

	return 0;
}


// WXWIN SUPPORT /////////////////////////////////
#ifdef __WXWIN__
#	include "robuntrans.h"
#	include "robwxutil.h"
#	include <wx/wx.h>

const lrt::Array<lrt::String>* theArgs;

class MakehtmlWxApp : public wxApp {
public:
	virtual bool OnInit() {
		bool locOk = myLocale.Init();

#ifdef EXTRA_DATA_DIR
		myLocale.AddCatalogLookupPathPrefix(wxString(wxT(EXTRA_DATA_DIR)) + wxFILE_SEP_PATH + wxT("locale")); 
#endif
		// look near the executable file also
		myLocale.AddCatalogLookupPathPrefix(Wx::Conv(lrt::File(lrt::File::getExecutableFile().getParent(), "locale").getName()));

		
		locOk &= myLocale.AddCatalog(wxT("robotour"));
		/*if(!locOk)
			lrt::System::println("warning: Program translation for language "
			+ Wx::Conv(myLocale.GetName()) + " could not be loaded. \nThe program will be in English.");
		*/
		// in some configurations, we bring out own version of the wxstd catalog
		myLocale.AddCatalog(wxT("robotour-wxstd")); 

		return true; 
	}

	virtual int OnRun() {
		return rtRun(*theArgs); 
	}

private:
	wxLocale myLocale;
};

IMPLEMENT_APP_NO_MAIN(MakehtmlWxApp)

int rtMain(const lrt::Array<lrt::String>& args)
{
	theArgs = &args; 
	return Wx::Startup();
}


#else // !WXWIN

int rtMain(const lrt::Array<lrt::String>& args)
{
	return rtRun(args); 
}


#endif // WXWIN

