/*
 * rtcollect.special.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/* Template specializations for Array<T> and Vector<T>. */

#if !defined(__LRT_COLLECT_SPECIALIZE__) && defined(__LRT_SPECIALIZE_COLLECT__)
#define __LRT_COLLECT_SPECIALIZE__
//#include "rtsystem.h"
//#include "rtstring.h"

namespace lrt {
template<> class Array<char>;

// typedef int (*voidPtrCompareFunction)(const void*&, const void*&);

template<> class Array<void*>
{
public:
   explicit Array(int length);
   Array(const Array<void*>&);
   Array(void** const ptr, int off, int len);
   Array<void*>& operator= (const Array<void*>&);
   virtual ~Array();

   inline int length() const;
   inline void* const operator[](int index) const;
   inline void*& operator[](int index);

   int indexOf(void* elem, int pos = 0) const;
   inline void** const getData() const;
   inline void exchange(int p1, int p2);
//   inline void sort();
   void reverse();
//   void sort(int (*compareFun)(const void*&, const void*&), int l = 0, int r = Math::MAX_INT);

protected:
   int len;
   void**  data;

   inline int check(int);
   inline int checkConst(int) const;
   static void copy(const Array<void*> *source, int srcPos, Array<void*> *dest, int destPos, int length);
};

template<> class Vector<void*> : public Array<void*>
{
public:
	explicit Vector(int length = 0);
	Vector(const Vector<void*>&);
	Vector<void*>& operator= (const Vector<void*>&);
	int check(int);

	inline void* const operator[](int index) const;
	inline void*& operator[](int index);

	Vector<void*>& operator +=(void*);
	Vector<void*>& operator += (const Array<void*> &);

	void insert(void* elem, int before);
	void remove(int pos);
	void remove(int start, int length);
	void clear();
/* // delete(void*) is undefined 
	inline void deleteAt(int index) { lrt::deleteAt(*this, index); }
	inline void deleteAll() { lrt::deleteAll(*this); }
*/
protected:
	int capacity;

	inline int up2pow(int val);
	static void copy(const Array<void*> *source, int srcPos, Vector<void*> *dest, int destPos, int length);
};


template<class T> class Array<T*> : protected Array<void*>
{
public:
   inline explicit Array(int length) : Array<void*>(length) {}
   inline Array(const Array<T*>& arr) : Array<void*>((const Array<void*>&)arr) {}
   inline Array(const T** ptr, int off, int len) : Array<void*>((void**)ptr, off, len) {}
   inline Array<T*>& operator= (const Array<T*>& arr) { return ((Array<T*>&)Array<void*>::operator=((const Array<void*>&)arr)); }

   inline int length() const { return Array<void*>::length(); }
   inline T* const operator[](int index) const { return ((T* const)Array<void*>::operator[](index)); }
   inline T*& operator[](int index) { return ((T*&)Array<void*>::operator[](index)); }

   int indexOf(const T& elem, int pos = 0) const;
   inline T** const getData() const { return (T**) Array<void*>::getData(); }
   inline void exchange(int p1, int p2) { Array<void*>::exchange(p1, p2); }
//   inline void sort() { Array<void*>::sort(p1, p2); }
   inline void reverse() { Array<void*>::reverse(p1, p2); }
//   inline void sort(int (*compareFun)(const T&, const T&), int l = 0, int r = Math::MAX_INT) { Array<void*>::sort((voidPtrCompareFunction)compareFun, l, r); }

protected:
   inline int check(int pos) { return Array<void*>::check(pos); }
   inline int checkConst(int pos) const { return Array<void*>::checkConst(pos); }
   inline static void copy(const Array<T*> *source, int srcPos, Array<T*> *dest, int destPos, int length)
   { Array<void*>::copy((const Array<void*>*) source, srcPos, (const Array<void*>*) dest, destPos, length); }
};

template<class T> class Vector<T*> : protected Vector<void*>
{
public:
   inline explicit Vector(int length = 0) : Vector<void*>(length) {}
   inline Vector(const Vector<T*>& arr) : Vector<void*>((const Vector<void*>&)arr) {}
   inline Vector<T*>& operator= (const Vector<T*>& arr) { return ((Vector<T*>&)Vector<void*>::operator=((const Vector<void*>&)arr)); }
   inline int check(int pos) { return Vector<void*>::check(pos); }

   inline int length() const { return Vector<void*>::length(); }
   inline T* const operator[](int index) const { return ((T* const)Vector<void*>::operator[](index)); }
   inline T*& operator[](int index) { return ((T*&)Vector<void*>::operator[](index)); }

   inline Vector<T*>& operator +=(T* elem) { return (Vector<T*>&) Vector<void*>::operator+=((void*) elem); }
   inline Vector<T*>& operator +=(const Array<T*>& arr) { return (Vector<T*>&) Vector<void*>::operator+=((const Array<void*>&) arr); }

   inline const T** getData() const { return (T**) Vector<void*>::getData(); }
   inline void exchange(int p1, int p2) { Vector<void*>::exchange(p1, p2); }
//   inline void sort() { Vector<void*>::sort(p1, p2); }
   inline void reverse() { Vector<void*>::reverse(p1, p2); }
//   inline void sort(int (*compareFun)(const T&, const T&), int l = 0, int r = Math::MAX_INT) { Vector<void*>::sort((voidPtrCompareFunction)compareFun, l, r); }

   inline void insert(T* elem, int before) { Vector<void*>::insert((void*) elem, before); }
   inline void remove(int pos) { Vector<void*>::remove(pos); }
   inline void remove(int start, int length) { Vector<void*>::remove(start, length); }
   inline void clear() { Vector<void*>::clear(); }

   inline void deleteAt(int index) { lrt::deleteAt(*this, index); }
   inline void deleteAll() { lrt::deleteAll(*this); }

protected:
   inline int checkConst(int pos) const { return Vector<void*>::checkConst(pos); }
   inline static void copy(const Array<T*> *source, int srcPos, Vector<T*> *dest, int destPos, int length)
	{ Vector<void*>::copy((const Array<void*>*) source, srcPos, (const Vector<void*>*) dest, destPos, length); }
};


//////////////////////////// optimization for strings
template<> class Array<char>
{
public:
   explicit Array(int length);
   Array(const Array<char>&);
   Array(const char* ptr, int off, int len);
   Array<char>& operator= (const Array<char>&);
   virtual ~Array();

   inline int length() const;
   inline char operator[](int index) const;
   inline char& operator[](int index);

   int indexOf(char elem, int pos = 0) const;
   inline const char* getData() const;
   inline void exchange(int p1, int p2);
   inline void sort();
   void reverse();
   void sort(int (*compareFun)(char, char), int l = 0, int r = Math::MAX_INT);

protected:
   int len;
   char* data;

   inline int check(int);
   inline int checkConst(int) const;
   static void copy(const Array<char> *source, int srcPos, Array<char> *dest, int destPos, int length);
};

template<> class Vector<char> : public Array<char>
{
public:
	explicit Vector(int length = 0);
	Vector(const Vector<char>&);
	Vector<char>& operator= (const Vector<char>&);
	int check(int);

	inline char operator[](int index) const;
	inline char& operator[](int index);

	Vector<char>& operator +=(char);
	Vector<char>& operator += (const Array<char> &);

	void insert(char elem, int before);
	void remove(int pos);
	void remove(int start, int length);
	void clear();
protected:
	int capacity;

	inline int up2pow(int val);
	static void copy(const Array<char> *source, int srcPos, Vector<char> *dest, int destPos, int length);
};




//////////// inlines
//////////// Array<void*>
inline void* const Array<void*>::operator[](int i) const
{
   return data[checkConst(i)];
}

inline void*& Array<void*>::operator[](int i)
{
   return data[check(i)];
}

inline int Array<void*>::length() const
{
	return len;
}

inline void** const Array<void*>::getData() const
{
	return data;
}

inline void Array<void*>::exchange(int p1, int p2)
{
	checkConst(p1);
	checkConst(p2);
	void* temp(data[p1]);
	data[p1] = data[p2];
	data[p2] = temp;
}

inline int Array<void*>::check(int i)
{
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   // there is a chicken-and-egg-problem with lrt::String so we can't provide fancy errors here
	   System::exit(-1, "Array index out of bounds!");
   }
#endif
   return i;
}

inline int Array<void*>::checkConst(int i) const
{ 
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   // there is a chicken-and-egg-problem with lrt::String so we can't provide fancy errors here
	   System::exit(-1, "Array Index out of bounds!");
   }
#endif
   return i; 
}

//////////////// Vector<void*>

inline void* const Vector<void*>::operator[](int i) const 
{ 
   return data[checkConst(i)]; 
}

inline void*& Vector<void*>::operator[](int i)
{ 
	void*& ret = data[check(i)];
	if(len <= i)
	  len = i+1;
	return ret;
}

inline int Vector<void*>::up2pow(int val)
{
	int ret = 16;
	while(ret < val)
		ret <<= 1;
	return ret;
}


//////////// Array<char>
inline char Array<char>::operator[](int i) const
{
   return data[checkConst(i)];
}

inline char& Array<char>::operator[](int i)
{
   return data[check(i)];
}

inline int Array<char>::length() const
{
	return len;
}

inline const char* Array<char>::getData() const
{
	return data;
}

inline void Array<char>::exchange(int p1, int p2)
{
	checkConst(p1);
	checkConst(p2);
	char temp = data[p1];
	data[p1] = data[p2];
	data[p2] = temp;
}

inline int Array<char>::check(int i)
{
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   // there is a chicken-and-egg-problem with lrt::String so we can't provide fancy errors here
	   System::exit(-1, "Array index out of bounds!");
   }
#endif
   return i; 
}

inline int Array<char>::checkConst(int i) const
{ 
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   // there is a chicken-and-egg-problem with lrt::String so we can't provide fancy errors here
	   System::exit(-1, "Array Index out of bounds!");
   }
#endif
   return i; 
}

//////////////// Vector<char>

inline char Vector<char>::operator[](int i) const
{
   return data[checkConst(i)];
}

inline char& Vector<char>::operator[](int i)
{
	char& ret = data[check(i)];
	if(len <= i)
	  len = i+1;
	return ret;
}

inline int Vector<char>::up2pow(int val)
{
	int ret = 16;
	while(ret < val)
		ret <<= 1;
	return ret;
}

/////////////// end inlines

} // namespace

#endif
