/*
 * rtcollect.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * The array-style collection templates <tt>Array</tt> and <tt>Vector</tt>
  * are declared and implemented in <tt>rtcollect.h</tt>.
  */

#ifndef __LRT_COLLECT__
#define __LRT_COLLECT__

#include "rtsystem.h"
#include "rtdefs.h"
#include "rtmath.h"

namespace lrt {

/** The class template Array<T> implements a bounds-checked, constant-length array of values of type T.
  * Methods for access to elements are provided.
  */
template<class T> class Array {
public:
   /** Creates an array of given length. */
   explicit Array(int length);
   
   /** Copies the contents of the given array into this one. */
   Array(const Array<T>&);
   
   /** Creates an array by extracting data from the given C pointer. 
     * You are responsible yourself for checking the validity of the pointer,
     * <tt>off</tt> and <tt>len</tt>! */
   Array(const T* ptr, int off, int len);
   
   /** Destroys the array and all contained data. */
   virtual ~Array();
   
   /** Returns the array's length. For Array<T> objects, the value will be constant during their lifetime. */
   inline int length() const;
   
   /** Returns a reference to the index'th element of the array. 
     * If <tt>index</tt> is negative or larger than the array's size, the program will be closed.
     */
   inline const T& operator[](int index) const;

   /** Returns a reference to the index'th element of the array.
     * This method is used for changing the element.
     * If <tt>index</tt> is negative or larger than the array's size, the program will be closed.
     */
   inline T& operator[](int index);

   /** Copies the contents of the given array over to this one.
     */
   Array<T>& operator= (const Array<T>&);

   /** Returns a C-style pointer to the array's data. 
     * The data cannot be modified through this pointer, but changes made to the array using the [] operators
	 * will be visible by it. This implies that should only use the pointer for a short time,
	 * e.g. to call some system C functions with it.
     */
   inline const T* getData() const;

   /** Finds the first occurence of <tt>elem</tt>, starting from <tt>pos</tt>.
     * Note that an <tt>operator==(T&,T&)</tt> must be defined.
	 * If <tt>elem</tt> is not found, returns <tt>-1</tt>. 
     */
   int indexOf(const T& elem, int pos = 0) const;

   /** Exchanges the elements at the given positions. */
   inline void exchange(int p1, int p2);

   /** Sorts the array using a quicksort algorithm and the default 
     * comparison function stdCompare. */
   inline void sort();

   /** Reverses the order of all the elements in the array. */
   void reverse();
   
   /** Sorts the array using a quicksort algorithm and a custom comparison function. */
   void sort(int (*compareFun)(const T&, const T&), int l = 0, int r = Math::MAX_INT);

protected:
   /** The array's length. */
   int len;
   /** The array's data. */
   T*  data;

   /** Checks if a given array index is valid. If not, exits the application. 
     * This method should be used to check accesses which may modify the array.
     */
   inline int check(int);

   /** Checks if a given array index is valid. If not, exits the application. 
     * This method should be used to check accesses which may not modify the array.
     */
   inline int checkConst(int) const;

   /** Copies some data between arrays very fastly.
     */
   static void copy(const Array<T> *source, int srcPos, Array<T> *dest, int destPos, int length);
}; 

/** The class template Vector<T> implements a growable array of components of type T.
  * If there is enough memory available, setting an element with positive index will never
  * fail using this class, but resize the array instead so that it becomes large enough to hold
  * the given number of elements.
  * Elements can also easily be appended to the array (this method should be reasonably fast).
  * Some methods are provided to insert or remove elements from the middle of the array, but these are slow.
  */
template<class T> class Vector : public Array<T> {
    
public:
	/** Creates a growable array of given initial length. */
	explicit Vector(int length = 0);
	
	/** Creates a new growable array by copying the elements of the given array over. */
	Vector(const Vector<T>&);

	/** Returns a reference to the index'th element of the array. 
	 * If <tt>index</tt> is negative or larger than the array's size, the program will be closed.
	 */
	inline const T& operator[](int index) const;

    /** Returns a reference to the index'th element of this array through which the element can be changed.
	  * If the array is not yet large enough to store <tt>index</tt> elements, it is first resized appropriately.
	  * <p>
	  * <b>Note:</b> Be careful when exporting references to <tt>Vector</tt>s! If you add more elements
	  * to the vector after exporting the reference, the vector's data may become relocated and the
	  * reference may become invalid. 
	  */
	inline T& operator[](int index);

	/** Copies the data from another array over to this one.
	  */
	Vector<T>& operator= (const Vector<T>&);

	/** Checks an array index for validity. If the index is too large, the array is resized appropriately.
	  */
	int check(int);

	/** Appends an element to this array.
	  * Most times, this method should be quite fast.
	  */
	Vector<T>& operator +=(const T &);

	/** Appends all the elements from another array to this one.
	  * Most times, this method should be quite fast.
	  */
	Vector<T>& operator += (const Array<T> &);

	/** Inserts an element before the given index.
	  * Since this method moves around all elements after <tt>before</tt>, it may be quite slow.
	  */
	void insert(T elem, int before);

	/** Removes the element at index <tt>pos</tt>.
	  * Since this method moves around all elements after <tt>pos</tt>, it may be quite slow.
	  */
	void remove(int pos);

	/** Removes all elements between indexes <tt>start</tt> and (not including) <tt>start + length</tt>.
	  */
	void remove(int start, int length);

	/** Removes all elements. */
	void clear();

protected:
	/** The internal allocated memory size. */
	int capacity;

	/** Returns the smallest power of two (but at least 16) which is equal or larger than <tt>val</tt>.
	  * This method is used to grow the array exponentially.
	  */
	inline int up2pow(int val);

    /** Copies some data over to a vector. If the vector is not large enough to hold the data, it is first resized.
	  */
	static void copy(const Array<T> *source, int srcPos, Vector<T> *dest, int destPos, int length);
};

// MSVC++ Bug Workaround
#ifndef __LRT_TPTR_DEFINED
#define __LRT_TPTR_DEFINED
LRT_DEFINE_PTR(T)
#endif // TPTR_DEFINED

/** Removes the element at given position from the vector, and deletes it.
  * @param vect A vector.
  * @param index The element's position. */
template<class LRT_NPTR(T)> void deleteAt(Vector<LRT_PTR(T)>& vect, int index);

/** Removes all elements from the vector and deletes them. 
  * @param vect A vector. */
template<class LRT_NPTR(T)> void deleteAll(Vector<LRT_PTR(T)>& vect);


} // namespace

#ifndef __MSVC__ // can't cope with the template specializations
#define __LRT_SPECIALIZE_COLLECT__
#include "rtcollect.special.h"
#endif // MSVC

#include "rtcollect.templ.cpp"

#endif

