//******************************************************************************
//
// File:    DrawingItem.java
// Package: edu.rit.draw.item
// Unit:    Class edu.rit.draw.item.DrawingItem
//
// This Java source file is copyright (C) 2006 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.draw.item;

import edu.rit.draw.Drawing;

import java.awt.Graphics2D;

import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

/**
 * Class DrawingItem is the abstract base class for each item on a {@linkplain
 * edu.rit.draw.Drawing}. Subclasses provide different kinds of drawing items,
 * such as text, lines, and shapes.
 * <P>
 * Any drawing item may be rotated by calling the <TT>rotationCenter()</TT> and
 * <TT>rotationAngle()</TT> methods to specify the center point for the rotation
 * and the rotation angle, respectively. The default rotation center point is
 * (0,0) and the default rotation angle is 0 -- that is, no rotation. A positive
 * angle rotates the positive X axis towards the positive Y axis. Since the
 * positive X axis points right and the positive Y axis points down, this is a
 * clockwise rotation. A negative rotation angle gives a counterclockwise
 * rotation. The rotation angle is specified in radians.
 * <P>
 * Each drawing item is contained within a rectangular <B>bounding box</B>.
 * Class DrawingItem's methods return the location and dimensions of the
 * bounding box. The returned bounding box is the bounding box <I>before</I> the
 * drawing item's rotation is applied.
 * <P>
 * The <TT>transform()</TT> method takes a point, applies the drawing item's
 * rotation to the point, and returns the transformed point. You can use this
 * method, for example, to find out where points on the drawing item's bounding
 * box would end up after the rotation.
 *
 * @author  Alan Kaminsky
 * @version 08-Sep-2006
 */
public abstract class DrawingItem
	implements Externalizable
	{

// Exported constants.

	/**
	 * The normal rotation center point (0,0).
	 */
	public static final Point NORMAL_ROTATION_CENTER = new Point();

	/**
	 * The normal rotation angle (0).
	 */
	public static final double NORMAL_ROTATION_ANGLE = 0.0;

// Hidden data members.

	private static final long serialVersionUID = -4091523056693860792L;

	// Default attributes.
	static Point theDefaultRotationCenter = NORMAL_ROTATION_CENTER;
	static double theDefaultRotationAngle = NORMAL_ROTATION_ANGLE;

	// Attributes.
	Point myRotationCenter = theDefaultRotationCenter;
	double myRotationAngle = theDefaultRotationAngle;

// Exported constructors.

	/**
	 * Construct a new drawing item.
	 */
	public DrawingItem()
		{
		}

	/**
	 * Construct a new drawing item that is the same as the given drawing item.
	 *
	 * @param  theItem  Drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theItem</TT> is null.
	 */
	public DrawingItem
		(DrawingItem theItem)
		{
		this.myRotationCenter = theItem.myRotationCenter;
		this.myRotationAngle = theItem.myRotationAngle;
		}

// Exported operations.

	/**
	 * Returns the default rotation center point.
	 *
	 * @return  Default rotation center point.
	 */
	public static Point defaultRotationCenter()
		{
		return theDefaultRotationCenter;
		}

	/**
	 * Set the default rotation center point. Before calling this method the
	 * first time, the default rotation center point is (0,0).
	 *
	 * @param  theCenter  Default rotation center point.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theCenter</TT> is null.
	 */
	public static void defaultRotationCenter
		(Point theCenter)
		{
		if (theCenter == null) throw new NullPointerException();
		theDefaultRotationCenter = theCenter;
		}

	/**
	 * Returns the default rotation angle.
	 *
	 * @return  Default rotation angle.
	 */
	public static double defaultRotationAngle()
		{
		return theDefaultRotationAngle;
		}

	/**
	 * Set the default rotation angle. Before calling this method the first
	 * time, the default rotation angle is 0.
	 *
	 * @param  theAngle  Default rotation angle.
	 */
	public static void defaultRotationAngle
		(double theAngle)
		{
		theDefaultRotationAngle = theAngle;
		}

	/**
	 * Returns this drawing item's rotation center point.
	 *
	 * @return  Rotation center point.
	 */
	public Point rotationCenter()
		{
		return myRotationCenter;
		}

	/**
	 * Set this drawing item's rotation center point.
	 *
	 * @param  theCenter  Rotation center point.
	 *
	 * @return  This drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theCenter</TT> is null.
	 */
	public DrawingItem rotationCenter
		(Point theCenter)
		{
		doRotationCenter (theCenter);
		return this;
		}

	/**
	 * Returns this drawing item's rotation angle.
	 *
	 * @return  Rotation angle.
	 */
	public double rotationAngle()
		{
		return myRotationAngle;
		}

	/**
	 * Set this drawing item's rotation angle.
	 *
	 * @param  theAngle  Rotation angle.
	 *
	 * @return  This drawing item.
	 */
	public DrawingItem rotationAngle
		(double theAngle)
		{
		doRotationAngle (theAngle);
		return this;
		}

	/**
	 * Returns the size of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Size.
	 */
	public Size size()
		{
		return new Size (width(), height());
		}

	/**
	 * Returns the width of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Width.
	 */
	public abstract double width();

	/**
	 * Returns the height of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Height.
	 */
	public abstract double height();

	/**
	 * Returns the northwest corner point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Northwest corner point.
	 */
	public abstract Point nw();

	/**
	 * Returns the north middle point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  North middle point.
	 */
	public Point n()
		{
		return nw().add (0.5 * width(), 0.0);
		}

	/**
	 * Returns the northeast corner point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Northeast corner point.
	 */
	public Point ne()
		{
		return nw().add (width(), 0.0);
		}

	/**
	 * Returns the west middle point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  West middle point.
	 */
	public Point w()
		{
		return nw().add (0.0, 0.5 * height());
		}

	/**
	 * Returns the center point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Center point.
	 */
	public Point c()
		{
		return nw().add (0.5 * width(), 0.5 * height());
		}

	/**
	 * Returns the east middle point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  East middle point.
	 */
	public Point e()
		{
		return nw().add (width(), 0.5 * height());
		}

	/**
	 * Returns the southwest corner point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Southwest corner point.
	 */
	public Point sw()
		{
		return nw().add (0.0, height());
		}

	/**
	 * Returns the south middle point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  South middle point.
	 */
	public Point s()
		{
		return nw().add (0.5 * width(), height());
		}

	/**
	 * Returns the southeast corner point of this drawing item's bounding box.
	 * <P>
	 * The return value refers to the bounding box <I>before</I> rotating this
	 * drawing item.
	 *
	 * @return  Southeast corner point.
	 */
	public Point se()
		{
		return nw().add (width(), height());
		}

	/**
	 * Add this drawing item to the end of the default drawing's sequence of
	 * drawing items.
	 *
	 * @return  This drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if there is no default drawing.
	 *
	 * @see  Drawing#defaultDrawing()
	 */
	public DrawingItem add()
		{
		doAdd (Drawing.defaultDrawing());
		return this;
		}

	/**
	 * Add this drawing item to the end of the given drawing's sequence of
	 * drawing items.
	 *
	 * @param  theDrawing  Drawing.
	 *
	 * @return  This drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theDrawing</TT> is null.
	 */
	public DrawingItem add
		(Drawing theDrawing)
		{
		doAdd (theDrawing);
		return this;
		}

	/**
	 * Add this drawing item to the beginning of the default drawing's sequence
	 * of drawing items.
	 *
	 * @return  This drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if there is no default drawing.
	 *
	 * @see  Drawing#defaultDrawing()
	 */
	public DrawingItem addFirst()
		{
		doAddFirst (Drawing.defaultDrawing());
		return this;
		}

	/**
	 * Add this drawing item to the beginning of the given drawing's sequence of
	 * drawing items.
	 *
	 * @param  theDrawing  Drawing.
	 *
	 * @return  This drawing item.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theDrawing</TT> is null.
	 */
	public DrawingItem addFirst
		(Drawing theDrawing)
		{
		doAddFirst (theDrawing);
		return this;
		}

	/**
	 * Write this drawing item to the given object output stream.
	 *
	 * @param  out  Object output stream.
	 *
	 * @exception  IOException
	 *     Thrown if an I/O error occurred.
	 */
	public void writeExternal
		(ObjectOutput out)
		throws IOException
		{
		myRotationCenter.writeExternal (out);
		out.writeDouble (myRotationAngle);
		}

	/**
	 * Read this drawing item from the given object input stream.
	 *
	 * @param  in  Object input stream.
	 *
	 * @exception  IOException
	 *     Thrown if an I/O error occurred.
	 * @exception  ClassNotFoundException
	 *     Thrown if any class needed to deserialize this drawing item cannot be
	 *     found.
	 */
	public void readExternal
		(ObjectInput in)
		throws IOException, ClassNotFoundException
		{
		myRotationCenter = new Point();
		myRotationCenter.readExternal (in);
		myRotationAngle = in.readDouble();
		}

	/**
	 * Draw this drawing item in the given graphics context. This method is
	 * allowed to change the graphics context's paint, stroke, and transform,
	 * and it doesn't have to change them back.
	 *
	 * @param  g2d  2-D graphics context.
	 */
	public void draw
		(Graphics2D g2d)
		{
		AffineTransform t = g2d.getTransform();
		concatenateTransform (t);
		g2d.setTransform (t);

		// Subclass draw() does the rest.
		}

	/**
	 * Returns the rectangular region this drawing item occupies, <I>without</I>
	 * applying this drawing item's rotation.
	 *
	 * @return  Unrotated bounding box.
	 */
	public Rectangle2D boundingBox()
		{
		Point nw = nw();
		return new Rectangle2D.Double (nw.x, nw.y, width(), height());
		}

	/**
	 * Returns the rectangular region this drawing item occupies, <I>after</I>
	 * applying this drawing item's rotation.
	 *
	 * @return  Rotated bounding box.
	 */
	public Rectangle2D transformedBoundingBox()
		{
		Point nw = transform (nw());
		Point ne = transform (ne());
		Point sw = transform (sw());
		Point se = transform (se());
		double x1 = Math.min (Math.min (nw.x, ne.x), Math.min (sw.x, se.x));
		double x2 = Math.max (Math.max (nw.x, ne.x), Math.max (sw.x, se.x));
		double y1 = Math.min (Math.min (nw.y, ne.y), Math.min (sw.y, se.y));
		double y2 = Math.max (Math.max (nw.y, ne.y), Math.max (sw.y, se.y));
		return new Rectangle2D.Double (x1, y1, x2-x1, y2-y1);
		}

	/**
	 * Transform the given point by applying this drawing item's rotation.
	 *
	 * @param  thePoint  Point before applying transformation.
	 *
	 * @return  Point after applying transformation.
	 */
	public Point transform
		(Point thePoint)
		{
		AffineTransform t = new AffineTransform();
		concatenateTransform (t);
		double[] pt = new double[] {thePoint.x, thePoint.y};
		t.transform (pt, 0, pt, 0, 1);
		return new Point (pt[0], pt[1]);
		}

// Hidden operations.

	/**
	 * Set this drawing item's rotation center point.
	 *
	 * @param  theCenter  Rotation center point.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theCenter</TT> is null.
	 */
	void doRotationCenter
		(Point theCenter)
		{
		if (theCenter == null) throw new NullPointerException();
		myRotationCenter = theCenter;
		}

	/**
	 * Set this drawing item's rotation angle.
	 *
	 * @param  theAngle  Rotation angle.
	 */
	void doRotationAngle
		(double theAngle)
		{
		myRotationAngle = theAngle;
		}

	/**
	 * Add this drawing item to the end of the given drawing's sequence of
	 * drawing items.
	 *
	 * @param  theDrawing  Drawing.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theDrawing</TT> is null.
	 */
	void doAdd
		(Drawing theDrawing)
		{
		theDrawing.add (this);
		}

	/**
	 * Add this drawing item to the beginning of the given drawing's sequence of
	 * drawing items.
	 *
	 * @param  theDrawing  Drawing.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>theDrawing</TT> is null.
	 */
	void doAddFirst
		(Drawing theDrawing)
		{
		theDrawing.addFirst (this);
		}

	/**
	 * Concatenate this drawing item's rotation to the given affine transform.
	 *
	 * @param  theTransform  Affine transform.
	 */
	void concatenateTransform
		(AffineTransform theTransform)
		{
		theTransform.rotate
			(/*theta*/ myRotationAngle,
			 /*x    */ myRotationCenter.x,
			 /*y    */ myRotationCenter.y);
		}

	}
