// File:        XmlMDataStd_ConstraintDriver.cxx
// Created:     Fri Aug 24 20:46:58 2001
// Author:      Alexnder GRIGORIEV
// Copyright:   Open Cascade 2001
// History:

#include <XmlMDataStd_ConstraintDriver.ixx>

#include <TDataStd_Constraint.hxx>
#include <TDataStd_Real.hxx>
#include <TNaming_NamedShape.hxx>

#include <XmlObjMgt.hxx>

static TDataStd_ConstraintEnum    ConstraintTypeEnum
                                                (const XmlObjMgt_DOMString&);
static const XmlObjMgt_DOMString& ConstraintTypeString
                                                (const TDataStd_ConstraintEnum);

IMPLEMENT_DOMSTRING (TypeString,             "contype")
IMPLEMENT_DOMSTRING (ValueString,            "valueref")
IMPLEMENT_DOMSTRING (GeometriesString,       "geometries")
IMPLEMENT_DOMSTRING (PlaneString,            "plane")
IMPLEMENT_DOMSTRING (StatusString,           "flags")
     // planar constraints
IMPLEMENT_DOMSTRING (ConRadiusString,        "radius")
IMPLEMENT_DOMSTRING (ConDiameterString,      "diameter")
IMPLEMENT_DOMSTRING (ConMinRadiusString,     "minorradius")
IMPLEMENT_DOMSTRING (ConMajRadiusString,     "majorradius")
IMPLEMENT_DOMSTRING (ConTangentString,       "tangent")
IMPLEMENT_DOMSTRING (ConParallelString,      "parallel")
IMPLEMENT_DOMSTRING (ConPerpendicularString, "perpendicular")
IMPLEMENT_DOMSTRING (ConConcentricString,    "concentric")
IMPLEMENT_DOMSTRING (ConCoincidentString,    "coincident")
IMPLEMENT_DOMSTRING (ConDistanceString,      "distance")
IMPLEMENT_DOMSTRING (ConAngleString,         "angle")
IMPLEMENT_DOMSTRING (ConEqualRadiusString,   "equalradius")
IMPLEMENT_DOMSTRING (ConSymmetryString,      "symmetry")
IMPLEMENT_DOMSTRING (ConMidPointString,      "midpoint")
IMPLEMENT_DOMSTRING (ConEqualDistanceString, "equaldist")
IMPLEMENT_DOMSTRING (ConFixString,           "fix")
IMPLEMENT_DOMSTRING (ConRigidString,         "rigid")
     // placement constraints
IMPLEMENT_DOMSTRING (ConFromString,          "from")
IMPLEMENT_DOMSTRING (ConAxisString,          "axis")
IMPLEMENT_DOMSTRING (ConMateString,          "mate")
IMPLEMENT_DOMSTRING (ConAlignFacesString,    "alignfaces")
IMPLEMENT_DOMSTRING (ConAlignAxesString,     "alignaxes")
IMPLEMENT_DOMSTRING (ConAxesAngleString,     "axesangle")
IMPLEMENT_DOMSTRING (ConFacesAngleString,    "facesangle")
IMPLEMENT_DOMSTRING (ConRoundString,         "round")
IMPLEMENT_DOMSTRING (ConOffsetString,        "offset")

//=======================================================================
//function : XmlMDataStd_ConstraintDriver
//purpose  : Constructor
//=======================================================================
XmlMDataStd_ConstraintDriver::XmlMDataStd_ConstraintDriver
                        (const Handle(CDM_MessageDriver)& theMsgDriver)
      : XmlMDF_ADriver (theMsgDriver, NULL)
{}

//=======================================================================
//function : NewEmpty
//purpose  : 
//=======================================================================
Handle(TDF_Attribute) XmlMDataStd_ConstraintDriver::NewEmpty() const
{
  return (new TDataStd_Constraint());
}

//=======================================================================
//function : Paste
//purpose  : persistent -> transient (retrieve)
//=======================================================================
Standard_Boolean XmlMDataStd_ConstraintDriver::Paste
                        (const XmlObjMgt_Persistent&  theSource,
                         const Handle(TDF_Attribute)& theTarget,
                         XmlObjMgt_RRelocationTable&  theRelocTable) const
{
  Handle(TDataStd_Constraint) aC = 
    Handle(TDataStd_Constraint)::DownCast(theTarget);
  const XmlObjMgt_Element& anElem = theSource;

  Standard_Integer aNb;
  TCollection_ExtendedString aMsgString;

  // value
  Handle(TDataStd_Real) aTValue;
  XmlObjMgt_DOMString aDOMStr = anElem.getAttribute(::ValueString());
  if (aDOMStr != NULL)
  {
    if (!aDOMStr.GetInteger(aNb))
    {
      aMsgString = TCollection_ExtendedString
        ("XmlMDataStd_ConstraintDriver: "
         "Cannot retrieve reference on Integer attribute from \"")
          + aDOMStr + "\"";
      WriteMessage (aMsgString);
      return Standard_False;
    }
    if (aNb > 0)
    {
      if (theRelocTable.IsBound(aNb))
        aTValue = Handle(TDataStd_Real)::DownCast(theRelocTable.Find(aNb));
      else
      {
        aTValue = new TDataStd_Real;
        theRelocTable.Bind(aNb, aTValue);
      }
      aC->SetValue(aTValue);
    }
  }

  // geometries
  aDOMStr = anElem.getAttribute(::GeometriesString());
  if (aDOMStr != NULL)
  {
    Standard_CString aGs = Standard_CString(aDOMStr.GetString());

    // first geometry
    if (!XmlObjMgt::GetInteger(aGs, aNb))
    {
      aMsgString = TCollection_ExtendedString
        ("XmlMDataStd_ConstraintDriver: Cannot retrieve reference on first Geometry from \"")
          + aDOMStr + "\"";
      WriteMessage (aMsgString);
      return Standard_False;
    }
    Standard_Integer i = 1;
    while (aNb > 0)
    {
      Handle(TNaming_NamedShape) aG;
      if (theRelocTable.IsBound(aNb))
        aG = Handle(TNaming_NamedShape)::DownCast(theRelocTable.Find(aNb));
      else
      {
        aG = new TNaming_NamedShape;
        theRelocTable.Bind(aNb, aG);
      }
      aC->SetGeometry (i, aG);

      // next geometry
      if (!XmlObjMgt::GetInteger(aGs, aNb)) aNb = 0;
      i++;
    }
  }

  // plane
  aDOMStr = anElem.getAttribute(::PlaneString());
  if (aDOMStr != NULL)
  {
    if (!aDOMStr.GetInteger(aNb))
    {
      aMsgString = TCollection_ExtendedString
        ("XmlMDataStd_ConstraintDriver: Cannot retrieve reference on Plane from \"")
          + aDOMStr + "\"";
      WriteMessage (aMsgString);
      return Standard_False;
    }
    Handle(TNaming_NamedShape) aTPlane;
    if (aNb > 0)
    {
      if (theRelocTable.IsBound(aNb))
        aTPlane = Handle(TNaming_NamedShape)::DownCast(theRelocTable.Find(aNb));
      else
      {
        aTPlane = new TNaming_NamedShape;
        theRelocTable.Bind(aNb, aTPlane);
      }
      aC->SetPlane(aTPlane);
    }
  }

  // constraint type
  XmlObjMgt_DOMString aType = anElem.getAttribute(::TypeString());
  aC->SetType(ConstraintTypeEnum(aType));

  // flags
  XmlObjMgt_DOMString aString = anElem.getAttribute(::StatusString());
  const char * aPtr = aString.GetString();
  aC->Verified((*aPtr) == '+');
  aPtr ++;
  aC->Inverted((*aPtr) == '+');
  aPtr ++;
  aC->Reversed((*aPtr) == '+');

  return Standard_True;
}

//=======================================================================
//function : Paste
//purpose  : transient -> persistent (store)
//=======================================================================
void XmlMDataStd_ConstraintDriver::Paste
                        (const Handle(TDF_Attribute)& theSource,
                         XmlObjMgt_Persistent&        theTarget,
                         XmlObjMgt_SRelocationTable&  theRelocTable) const
{
  Handle(TDataStd_Constraint) aC =
    Handle(TDataStd_Constraint)::DownCast(theSource);
  XmlObjMgt_Element& anElem = theTarget;

  Standard_Integer aNb;

  // value
  Handle(TDataStd_Real) aValue = aC->GetValue();
  if (!aValue.IsNull())
  {
    aNb = theRelocTable.FindIndex(aValue);
    if (aNb == 0)
    {
      aNb = theRelocTable.Add(aValue);
    }
    anElem.setAttribute(::ValueString(), aNb);
  }

  // geometries
  Standard_Integer NbGeom = aC->NbGeometries();
  if (NbGeom >= 1)
  {
    TCollection_AsciiString aGsStr;
    for (Standard_Integer i = 1; i <= NbGeom; i++)
    {
      Handle(TNaming_NamedShape) aG = aC->GetGeometry(i);
      aNb = 0;
      if (!aG.IsNull())
      {
        aNb = theRelocTable.FindIndex(aG);
        if (aNb == 0)
        {
          aNb = theRelocTable.Add(aG);
        }
        aGsStr += TCollection_AsciiString(aNb) + " ";
      }
      else aGsStr += "0 ";
    }
    anElem.setAttribute(::GeometriesString(), aGsStr.ToCString());
  }

  // plane
  Handle(TNaming_NamedShape) aTPlane = aC->GetPlane();
  if (!aTPlane.IsNull())
  {
    aNb = theRelocTable.FindIndex(aTPlane);
    if (aNb == 0)
    {
      aNb = theRelocTable.Add(aTPlane);
    }
    anElem.setAttribute(::PlaneString(), aNb);
  }
  
  // constraint type
  anElem.setAttribute(::TypeString(), ConstraintTypeString(aC->GetType()));

  // flags
  TCollection_AsciiString aStatusStr;

  if (aC->Verified()) aStatusStr += "+";
  else aStatusStr += "-";

  if (aC->Inverted()) aStatusStr += "+";
  else aStatusStr += "-";

  if (aC->Reversed()) aStatusStr += "+";
  else aStatusStr += "-";

  anElem.setAttribute(::StatusString(), aStatusStr.ToCString());
}

//=======================================================================
//function : ConstraintTypeEnum
//purpose  : 
//=======================================================================
static TDataStd_ConstraintEnum ConstraintTypeEnum (const XmlObjMgt_DOMString& theString) 
{
  TDataStd_ConstraintEnum aResult = TDataStd_RADIUS;
  // planar constraints
  if (!theString.equals (::ConRadiusString()))
  {
    if (theString.equals (::ConDiameterString()))
      aResult = TDataStd_DIAMETER;
    else if (theString.equals (::ConMinRadiusString()))
      aResult = TDataStd_MINOR_RADIUS;
    else if (theString.equals (::ConMajRadiusString()))
      aResult = TDataStd_MAJOR_RADIUS;
    else if (theString.equals (::ConTangentString()))
      aResult = TDataStd_TANGENT;
    else if (theString.equals (::ConParallelString()))
      aResult = TDataStd_PARALLEL;
    else if (theString.equals (::ConPerpendicularString()))
      aResult = TDataStd_PERPENDICULAR;
    else if (theString.equals (::ConConcentricString()))
      aResult = TDataStd_CONCENTRIC;
    else if (theString.equals (::ConCoincidentString()))
      aResult = TDataStd_COINCIDENT;
    else if (theString.equals (::ConDistanceString()))
      aResult = TDataStd_DISTANCE;
    else if (theString.equals (::ConAngleString()))
      aResult = TDataStd_ANGLE;
    else if (theString.equals (::ConEqualRadiusString()))
      aResult = TDataStd_EQUAL_RADIUS;
    else if (theString.equals (::ConSymmetryString()))
      aResult = TDataStd_SYMMETRY;
    else if (theString.equals (::ConMidPointString()))
      aResult = TDataStd_MIDPOINT;
    else if (theString.equals (::ConEqualDistanceString()))
      aResult = TDataStd_EQUAL_DISTANCE;
    else if (theString.equals (::ConFixString()))
      aResult = TDataStd_FIX;
    else if (theString.equals (::ConRigidString()))
      aResult = TDataStd_RIGID;
    // placement constraints
    else if (theString.equals (::ConFromString()))
      aResult = TDataStd_FROM;
    else if (theString.equals (::ConAxisString()))
      aResult = TDataStd_AXIS;
    else if (theString.equals (::ConMateString()))
      aResult = TDataStd_MATE;
    else if (theString.equals (::ConAlignFacesString()))
      aResult = TDataStd_ALIGN_FACES;
    else if (theString.equals (::ConAlignAxesString()))
      aResult = TDataStd_ALIGN_AXES;
    else if (theString.equals (::ConAxesAngleString()))
      aResult = TDataStd_AXES_ANGLE;
    else if (theString.equals (::ConFacesAngleString()))
      aResult = TDataStd_FACES_ANGLE;
    else if (theString.equals (::ConRoundString()))
      aResult = TDataStd_ROUND;
    else if (theString.equals (::ConOffsetString()))
      aResult = TDataStd_OFFSET;
    else
      Standard_DomainError::Raise
        ("TDataStd_ConstraintEnum; string value without enum term equivalence");
  }
  return aResult;
}

//=======================================================================
//function : ConstraintTypeString
//purpose  : 
//=======================================================================
static const XmlObjMgt_DOMString& ConstraintTypeString (const TDataStd_ConstraintEnum theE) 
{
  switch (theE)
  {
    // planar constraints
  case TDataStd_RADIUS         : return ::ConRadiusString();
  case TDataStd_DIAMETER       : return ::ConDiameterString();
  case TDataStd_MINOR_RADIUS   : return ::ConMinRadiusString();
  case TDataStd_MAJOR_RADIUS   : return ::ConMajRadiusString();
  case TDataStd_TANGENT        : return ::ConTangentString();
  case TDataStd_PARALLEL       : return ::ConParallelString();
  case TDataStd_PERPENDICULAR  : return ::ConPerpendicularString();
  case TDataStd_CONCENTRIC     : return ::ConConcentricString();
  case TDataStd_COINCIDENT     : return ::ConCoincidentString();
  case TDataStd_DISTANCE       : return ::ConDistanceString();
  case TDataStd_ANGLE          : return ::ConAngleString();
  case TDataStd_EQUAL_RADIUS   : return ::ConEqualRadiusString();
  case TDataStd_SYMMETRY       : return ::ConSymmetryString();
  case TDataStd_MIDPOINT       : return ::ConMidPointString();   
  case TDataStd_EQUAL_DISTANCE : return ::ConEqualDistanceString();
  case TDataStd_FIX            : return ::ConFixString();
  case TDataStd_RIGID          : return ::ConRigidString();
    // placement constraints
  case TDataStd_FROM           : return ::ConFromString(); 
  case TDataStd_AXIS           : return ::ConAxisString(); 
  case TDataStd_MATE           : return ::ConMateString();
  case TDataStd_ALIGN_FACES    : return ::ConAlignFacesString();
  case TDataStd_ALIGN_AXES     : return ::ConAlignAxesString();
  case TDataStd_AXES_ANGLE     : return ::ConAxesAngleString();
  case TDataStd_FACES_ANGLE    : return ::ConFacesAngleString();
  case TDataStd_ROUND          : return ::ConRoundString();
  case TDataStd_OFFSET         : return ::ConOffsetString();
    
  default:
    Standard_DomainError::Raise("TDataStd_ConstraintEnum; enum term unknown");
  }
  static XmlObjMgt_DOMString aNullString;
  return aNullString; // To avoid compilation error message.
}
