#include <iostream>
using namespace std;

#include <QString>
#include <QSqlError>
#include "dbcheck.h"
#include "datadirect.h"          // for DataDirectProcessor::FixProgramIDs
#include "videodisplayprofile.h" // for "1214"

#include "dbutil.h"
#include "mythcontext.h"
#include "schemawizard.h"
#include "mythdb.h"
#include "mythverbose.h"


#define MINIMUM_DBMS_VERSION 5,0,15

/* If currentDatabaseVersion gets updated, the following files need updated:
   mythtv/bindings/python/MythTV/MythStatic.py
*/
/// This is the DB schema version expected by the running MythTV instance.
const QString currentDatabaseVersion = "1254";

static bool UpdateDBVersionNumber(const QString &newnumber);
static bool performActualUpdate(
    const char **updates, const char *version, QString &dbver);
static bool InitializeDatabase(void);
static bool doUpgradeTVDatabaseSchema(void);
bool convert_diseqc_db(void);

/** \defgroup db_schema MythTV Database Schema

\section db_intro Introduction

\verbatim
The schema contains the following tables:
\endverbatim

\htmlonly
<table>
<tr><td>callsignnetworkmap         <td>pk(id)
<tr><td>capturecard                <td>pk(cardid)
<tr><td>cardinput                  <td>pk(cardinputid)
<tr><td>channel                    <td>pk(chanid) k(channum,sourceid)
<tr><td>channelgroup               <td>pk(id)
<tr><td>channelgroupnames          <td>pk(grpid)
<tr><td>codecparams                <td>pk(profile,name)
<tr><td>conflictresolutionany      <td>k(prefertitle) k(disliketitle)
<tr><td>conflictresolutionoverride <td>k(chanid,starttime) k(endtime)
<tr><td>conflictresolutionsingle   <td>k(preferchanid,preferstarttime) k(preferendtime)
<tr><td>credits                    <td>uk(chanid,starttime,person,role) k(person,role)
<tr><td>dtv_multiplex              <td>pk(mplexid)
<tr><td>dtv_privatetypes
<tr><td>dvb_signal_quality         <td>pk(id) k(sampletime,cardid)
<tr><td>housekeeping               <td>pk(tag)
<tr><td>jobqueue                   <td>pk(id) uk(chanid,starttime,type,inserttime)
<tr><td>jumppoints                 <td>pk(destination,hostname)
<tr><td>keybindings                <td>pk(context,action,hostname)
<tr><td>keyword                    <td>uk(phrase,searchtype)
<tr><td>mythlog                    <td>pk(logid)
<tr><td>networkiconmap             <td>pk(id) uk(network)
<tr><td>oldprogram                 <td>pk(oldtitle)
<tr><td>oldrecorded                <td>k(endtime) k(title) k(seriesid) k(programid)
                                       pk(station,starttime,title)
<tr><td>people                     <td>pk(person) uk(name)
<tr><td>pidcache                   <td>
<tr><td>profilegroups              <td>pk(id) uk(name,hostname)
<tr><td>program                    <td>k(endtime) k(title) k(title_pronounce) k(seriesid)
                                       k(programid) k(chanid,starttime,endtime)
<tr><td>programgenres              <td>pk(chanid,starttime,relevance)
<tr><td>programrating              <td>uk(chanid,starttime,system,rating)
                                       k(starttime,system)
<tr><td>recgrouppassword           <td>pk(recgroup) uk(recgroup)
<tr><td>record                     <td>pk(recordid) k(chanid,starttime) k(title)
                                       k(seriesid) k(programid)
<tr><td>recorded                   <td>k(endtime) k(seriesid) k(programid)
<tr><td>recordedmarkup             <td>pk(chanid,starttime,mark,type)
<tr><td>recordingprofiles          <td>pk(id)
<tr><td>recordoverride             <td>
<tr><td>settings                   <td>k(value,hostname)
<tr><td>videosource                <td>pk(sourceid) uk(name)
<tr><td>displayprofilegroups       <td>pk(name, host), uk(profileid)
<tr><td>displayprofiles            <td>pk(profileid),
</table>
\endhtmlonly

Where pk means PRIMARY KEY, uk means "UNIQUE KEY", and k means "KEY".
BUSQ refers to the Big Ugly SQL Query in scheduler.cpp

\section capturecard_table Capure Card Table (capturecard)
This table describes the attributes of a capture card used by MythTV.

The primary key 'cardid' is automatically generated by the database.

The 'videodevice' field normally points to a device used to capture
the video, but for DVB where multiple devices are used it is simply
a number indicating which set of devices do use. The 'audiodevice'
is only used by framegrabber cards, encoding cards like the Haupauge
PVR line encode the audio as part of the video stream, while DTV
cards like the various ATSC and DVB cards recieve the audio as part
of the video stream. The same applies to the 'vbidevice', PVR and
DTV cards have captions encoded in the digital stream, while
framegrabber cards use this device to provide caption and other
data to MythTV.

The 'audioratelimit' field is used to set the audio capture rate
when using the 'audiodevice' field. This is mostly useful for
V4L cards that have troublesome on-board audio, such as some
cards supported by the bttv driver.


The 'cardtype' is an important field for all cards as it guides the
interpretation of the rest of the fields, it has several possible
values: "V4L", "MJPEG", "DVB", "MPEG", "FIREWIRE", and "GO7007".
"V4L" indicates a V4L compatible device, this could be a
true V4L device or something like a Firewire camera with the
"firewire->v4l" driver.
"MJPEG" describes a Matrox G200 or DC10 card which captures
individual frames a JPEG images.
"DVB" can describe any card supported by the Linux DVB
drivers, including the pcHDTV and other ATSC cards.
The "MPEG" type describes a device using the ivtv drivers such as
the Haupauge PVR[150,250,350,500]s +Freestyle cards, the Yuan MPEG
series, the Avermedia M179, and the Adaptec VideOh! AVC-2410.
The "FIREWIRE" card type does not describe Firewire cameras of
capture cards but instead describes the Firewire port required by
law in the USA on cable boxes for subscribers who recieve HDTV
channels and request the Firewire port feature. This port lets
you communicate with your cable box without needing to build an
IR transmitter; in some cases, it also allows you to digitally
capture video.
The "GO7008" card type describes a USB MPEG-4 encoder such
as the Plextor ConvertX.

The 'hostname' field is another important field for all cards
as it specifies which backend the capture card is connected to.

The 'defaultinput' field is a another important field for all
cards except "FIREWIRE", "FREEBOX", "HDHOMERUN", and "IMPORT" cards.
It specifies which
input of the card to use. This does not have to mean a specific
physical input, but may also indicate a different use for the
same phyicial input.

The 'signal_timeout' and 'channel_timeout' indicate in
milliseconds how long it should take to get a signal and
channel lock respectively.

The 'dvb_swfilter', 'dvb_sat_type', 'dvb_wait_for_seqstart',
'skipbtaudio', 'dvb_on_demand', and 'dvb_diseqc_type' columns
are all "DVB" specific configuration parameters.

Both 'dvb_recordts' and 'dvb_hw_decoder' are unused, and
will be dropped in future versions of MythTV.

The 'firewire_port', 'firewire_node', 'firewire_speed',
'firewire_model', 'firewire_connection' are all "FIREWIRE" specific
configuration parameters. The first three define the connection,
and function much like 'videodevice' does for other capture "cards".
The 'firewire_model' describes the cable box model, for example
"DCT-6000" decribes a box that communicates using the same protocol
as the Motorola DCT-6000. The 'firewire_connection' field descibes
the overall communication protocol, i.e. are we using "Broadcast"
or "Point-to-Point" communication.

\section dtv_multiplex_table Digital Television Multiplex Table (dtv_multiplex)
This table contains the information needed to tune to a particular
frequency on a particular input.

The primary key, 'mplexid', is automatically generated by the database.
The most important field is the 'frequency' field which contains the
transmission frequency in Hertz. The 'sourceid' which tells us which
card and on what input of that card this multiplex applies to according
to the \ref videosource_table. Along with a channel number 'sourceid'
also uniquely identifies a channel in the \ref channel_table. The
'modulation' field tells us whether the modulation mode is "qpsk",
"qam_16", "qam_32", "qam_64", "qam_128", "qam_256", "8vsb", "16vsb",
or "auto". Finally, the 'sistandard' field is the last field which
applies to both DVB and ATSC, it contains on of four strings, "dvb",
"atsc", "mpeg" or "opencable".

When describing a DVB Multiplex, the networkid, and transportid are
used along with the serviceid in the \ref channel_table to tune to
a particular MPEG stream in DVBChannel.

When descrining an ATSC Multiplex, the \ref channel_table contains
the major and minor channels used to tune to a particular stream.

Other fields are used by different DVBx implementations to tune to a
a particular type of service and Myth just passes them along to the
drivers to take care of.



\section dtv_privatetypes_table Digital Television Private Types Table (dtv_privatetypes)
This table contains free form data pertaining to DVB services, it is
like a settings table for DVB.



\section videosource_table Video Source Table (videosource)
This table tells us about a particular input on a video capture card.

The primary key, 'sourceid', is automatically generated by the database.
The 'name' field gives the input name as passed to the video capture card
driver. The 'xmltvgrabber' field tells us where we get our listings from.
The 'userid' and 'password' fields are optionally used to log in to a
listings service, with the 'lineupid' being used to select a particular
lineup available from the listings service. Finally, the 'freqtable'
tells us which channel number to frequency mapping to use with
analog channels, us, us-cable, pal, etc.



\section channel_table Channel Table (channel)
The primary key, 'chanid' is generated by combining the source id with
a unique number from 0 to 99.

The 'channum' field contains the channel number to be input to jump
to a channel.

The 'sourceid' field contains the \ref videosource_table key, which
specifies which listings service to use and which input on which video
capture card to find this channel.

The 'callsign' field contains the government mandated licence number
of the channel's broadcaster if this is an over-the-air channel.

The 'name' field provides the alphanumberic name of this channel for
use in the %MythTV user interface.

The 'icon' field provides the path to an optional channel icon for this
channel.

The 'freqid' field provides the channel number to tune to, as
differentiated from the 'channum' used in the %MythTV user interface.
When the 'mplexid' is set it is used instead. And the 'finetune' field
is used to adjust the frequency provided by the combination of 'freqid'
and 'tvformat', this can be used when there is carrier drift in a cable
system.

The 'xmltvid' field is used to identify this channel to the listings
provider.

The 'tmoffset' field is used to apply an offset (in minutes) from the listings
provided by the provider to a new time in the MythTV program guide database.
This is very handy when the listings provider has listings which are offset
by a few hours on individual channels with the rest of them being correct.

The 'recpriority' field is used tell the scheduler from which of two
otherwise equivalent programs on two different channels should be
prefered, a higher number means this channel is more preferred.

The 'tvformat' is one of "ntsc", "ntsc-jp", "pal", "atsc", etc and is used
for various purposes, such as translating a channel number to a frequency,
setting the recording frame size, etc.

'commfree' tells us whether this is a commercial free channel, such as
those on the BBC and CBC networks. On commercial free channels we
do not need to run the commercial detector.

'commmethod' tells us which commercial flagger to use on the channel, the
default is to use the global setting

'visible' tells us whether we should show this channel in the channel
guide.

'outputfilters' contains a list of filters that should be applied
when playing recordings from this channel.

The 'contrast' 'brightness', 'colour', and 'hue' fields are used by frame
grabbing recorders to adjust the picture before encoding each frame.
And, 'videofilters' contains a list of filters that should be applied
when recording from channel using a frame grabbing recorder.

For digital television, the 'mplexid' field contains the
\ref dtv_multiplex_table key for determining additional digital tuning
parameters. The 'serviceid' field contains the MPEG program number when
describing a DVB service. The 'useonairguide' field is used by DVB to
determine whether to use the on-air-guide data from this channel to
update the listings. Using the on-air-guide is currently experimental
and must be selected when you compile %MythTV. Finally, the 'atscsrcid'
field currently contains both the major and minor atsc channels, encoded
in the form (atsc_major_chan * 256 | atsc_minor_chan).

\section program_table Program Entry Table (program)
'category_type' holds one of these exact four strings:
"movie", "series", "sports" or "tvshow".

'airdate' is a string representing the year of release for movies and
may have no meaning for other types of shows.

'stars' is a floating point number from 0.0 to 1.0. On a four star scale,
1.0 would be four stars, 0.75 would be three stars and so on.

'originalairdate' if provided is the date when a show was, or will be,
first televised.

'previouslyshown' is a field created by MythTV to try to determine if
a showing is more than 14 days after its original air date or if the
show was marked as a repeat and did not have a date for the first
airing. If this is "0" it usually means that this is a brand new show
or a rebroadcast within the first two weeks.

'programid' is the Tribune Media Service database record identifier
for each program description. In general, these start with a two
letter prefix, MV, EP, SP or SH that equate to the 'category_type'.
For most, the last four digits are "0000" except EP where the last
four digits are the episode number in the series. Note that these
are generated by TMS and not the show's producers but they are
usually in the same order as the original air dates for the episodes.
Detailed information can be found in the Data Direct documentation
at http://labs.zap2it.com/ .

\section oldrecorded_table Old Recorded Table (oldrecorded)

oldrecorded imposes the restriction of one entry per
title, starttime and callsign.  The scheduler relies on this when
restoring any previous status for programs that aren't currently
recording and for catching reactivation requests.

The duplicate field is used to indicate if this record should be used
to check for duplicates in the BUSQ

\section oldfind_table Old Find Table (oldfind)

If a matching entry exists in
the oldfind table, the program is considered a duplicate regarless of
the duplicate setting in recorded and oldrecorded.  oldfind is an
imperfect optimization to avoid greatly increasing the time needed for
the big scheduler query.

But the real reason for oldfind is that a search
rule may not be able to match by title. For instance, a FindWeekly
"Bill Murray (People Search)" needs to know that "Meatballs" is a
dup for this week if "Stripes" has already been recorded. Oldfind
tracks if the rule (recordid) has recorded in the current time
interval (findid). When a recording has a findid, ForgetHistory()
and DeleteHistory() update oldfind while updating oldrecorded.

\section recorded_table Recorded Table (recorded

The duplicate field is used to indicate if this record should be used
to check for duplicates in the BUSQ
 */

/** \fn UpdateDBVersionNumber(const QString&)
 *  \brief Updates the schema version stored in the database.
 *
 *   Updates "DBSchemaVer" property in the settings table.
 *  \param newnumber New schema version.
 */
static bool UpdateDBVersionNumber(const QString &newnumber)
{
    // delete old schema version
    MSqlQuery query(MSqlQuery::InitCon());

    QString thequery = "DELETE FROM settings WHERE value='DBSchemaVer';";
    query.prepare(thequery);

    if (!query.exec())
    {
        QString msg =
            QString("DB Error (Deleting old DB version number): \n"
                    "Query was: %1 \nError was: %2 \nnew version: %3")
            .arg(thequery)
            .arg(MythDB::DBErrorMessage(query.lastError()))
            .arg(newnumber);
        VERBOSE(VB_IMPORTANT, msg);
        return false;
    }

    // set new schema version
    thequery = QString("INSERT INTO settings (value, data, hostname) "
                       "VALUES ('DBSchemaVer', %1, NULL);").arg(newnumber);
    query.prepare(thequery);

    if (!query.exec())
    {
        QString msg =
            QString("DB Error (Setting new DB version number): \n"
                    "Query was: %1 \nError was: %2 \nnew version: %3")
            .arg(thequery)
            .arg(MythDB::DBErrorMessage(query.lastError()))
            .arg(newnumber);
        VERBOSE(VB_IMPORTANT, msg);
        return false;
    }

    return true;
}

/** \fn performActualUpdate(const char **, const char*, QString&)
 *  \brief Runs a number of SQL commands, and updates the schema version.
 *
 *  \param updates  array of SQL commands to issue, terminated by a NULL string.
 *  \param version  version we are updating db to.
 *  \param dbver    the database version at the end of the function is returned
 *                  in this parameter, if things go well this will be 'version'.
 *  \return true on success, false on failure
 */
static bool performActualUpdate(
    const char **updates, const char *version, QString &dbver)
{
    MSqlQuery query(MSqlQuery::InitCon());

    VERBOSE(VB_IMPORTANT, QString("Upgrading to MythTV schema version ") +
            version);

    int counter = 0;
    const char *thequery = updates[counter];

    while (thequery != NULL)
    {
        if (strlen(thequery) && !query.exec(thequery))
        {
            QString msg =
                QString("DB Error (Performing database upgrade): \n"
                        "Query was: %1 \nError was: %2 \nnew version: %3")
                .arg(thequery)
                .arg(MythDB::DBErrorMessage(query.lastError()))
                .arg(version);
            VERBOSE(VB_IMPORTANT, msg);
            return false;
        }

        counter++;
        thequery = updates[counter];
    }

    if (!UpdateDBVersionNumber(version))
        return false;

    dbver = version;
    return true;
}

/**
 *  \brief Called from outside dbcheck.cpp to update the schema.
 *
 *   If the "DBSchemaVer" property equals the currentDatabase version this
 *   returns true immediately. If not, we try to do a database backup,
 *   prompt the user for permission to do the upgrade,
 *   lock the schemalock table, call doUpgradeTVDatabaseSchema()
 *   to do the actual update, and then we unlock the schemalock table.
 *
 *   If the program running this function is killed while
 *   this is running then the schema may be corrupted.
 *
 *  \return false on failure, error, or if the user selected "Exit."
 */
bool UpgradeTVDatabaseSchema(const bool upgradeAllowed,
                             const bool upgradeIfNoUI)
{
    SchemaUpgradeWizard  * DBup;


    GetMythDB()->SetSuppressDBMessages(true);
    gContext->ActivateSettingsCache(false);

    if (!gContext->GetNumSetting("MythFillFixProgramIDsHasRunOnce", 0))
        DataDirectProcessor::FixProgramIDs();

    DBup = SchemaUpgradeWizard::Get("DBSchemaVer", "MythTV",
                                    currentDatabaseVersion);

    // There may be a race condition where another program (e.g. mythbackend)
    // is upgrading, so wait up to 5 seconds for a more accurate version:
    DBup->CompareAndWait(5);

    if (DBup->versionsBehind == 0)  // same schema
    {
        gContext->ActivateSettingsCache(true);
        GetMythDB()->SetSuppressDBMessages(false);
        return true;
    }

    if (!upgradeAllowed)
        VERBOSE(VB_IMPORTANT, "Not allowed to upgrade the database.");

    // Pop up messages, questions, warnings, et c.
    switch (DBup->PromptForUpgrade("TV", upgradeAllowed,
                                   upgradeIfNoUI, MINIMUM_DBMS_VERSION))
    {
        case MYTH_SCHEMA_USE_EXISTING:
            gContext->ActivateSettingsCache(true);
            GetMythDB()->SetSuppressDBMessages(false);
            return true;
        case MYTH_SCHEMA_ERROR:
        case MYTH_SCHEMA_EXIT:
            GetMythDB()->SetSuppressDBMessages(false);
            return false;
        case MYTH_SCHEMA_UPGRADE:
            break;
    }

    MSqlQuery query(MSqlQuery::InitCon());
    if (!query.exec(QString("ALTER DATABASE %1 DEFAULT"
                            " CHARACTER SET utf8 COLLATE utf8_general_ci;")
                    .arg(gContext->GetDatabaseParams().dbName)))
        MythDB::DBError("UpgradeTVDatabaseSchema -- alter charset", query);


    VERBOSE(VB_IMPORTANT, "Newest MythTV Schema Version : "+
            currentDatabaseVersion);

    if (!DBUtil::lockSchema(query))
    {
        GetMythDB()->SetSuppressDBMessages(false);
        return false;
    }

    bool ret = doUpgradeTVDatabaseSchema();

    if (ret)
        VERBOSE(VB_IMPORTANT, "Database Schema upgrade complete, unlocking.");
    else
        VERBOSE(VB_IMPORTANT, "Database Schema upgrade FAILED, unlocking.");

    DBUtil::unlockSchema(query);
    gContext->ActivateSettingsCache(true);

    GetMythDB()->SetSuppressDBMessages(false);
    return ret;
}

/** \fn doUpgradeTVDatabaseSchema(void)
 *  \brief This is called by UpgradeTVDatabaseSchema() to actually upgrade
 *         the schema to what MythTV expects.
 *
 *   This function Initializes the database if the "DBSchemaVer" property
 *   does not exist in the database. Otherwise it upgrades the schema
 *   from the one in that property to the currentDatabaseVersion, by
 *   running the SQL commands needed for each upgrade between the two
 *   versions. This will only upgrade the schema, so bad things may
 *   happen if you upgrade and then downgrade MythTV across DB Schema
 *   versions.
 *
 */
static bool doUpgradeTVDatabaseSchema(void)
{
    QString dbver = gContext->GetSetting("DBSchemaVer");

    if (dbver == currentDatabaseVersion)
    {
        return true;
    }

    if (DBUtil::IsNewDatabase())
    {
        if (!InitializeDatabase())
            return false;
        dbver = gContext->GetSetting("DBSchemaVer");
    }

    if (dbver.isEmpty() || dbver.toInt() <  1027)
    {
        VERBOSE(VB_IMPORTANT, "Unrecognized database schema version. "
                              "Unable to upgrade database.");
        return false;
    }

    if (dbver == "1027")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS dvb_signal_quality ("
"    id INT UNSIGNED AUTO_INCREMENT NOT NULL PRIMARY KEY,"
"    sampletime TIMESTAMP NOT NULL,"
"    cardid INT UNSIGNED NOT NULL,"
"    fe_snr INT UNSIGNED NOT NULL,"
"    fe_ss  INT UNSIGNED NOT NULL,"
"    fe_ber INT UNSIGNED NOT NULL,"
"    fe_unc INT UNSIGNED NOT NULL,"
"    myth_cont INT UNSIGNED NOT NULL,"
"    myth_over INT UNSIGNED NOT NULL,"
"    myth_pkts INT UNSIGNED NOT NULL,"
"    FOREIGN KEY(cardid) REFERENCES capturecard(id),"
"    INDEX (sampletime,cardid)"
");",
"ALTER TABLE capturecard ADD skipbtaudio TINYINT(1) DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1028", dbver))
            return false;
    }

    if (dbver == "1028") {
        const char *updates[] = {
"ALTER TABLE channel ADD COLUMN commfree TINYINT NOT NULL default '0';",
"ALTER TABLE record ADD COLUMN recgroup VARCHAR(32) default 'Default';",
"ALTER TABLE record ADD COLUMN dupmethod INT NOT NULL DEFAULT 6;",
"ALTER TABLE record ADD COLUMN dupin INT NOT NULL DEFAULT 15;",
"UPDATE record SET dupmethod = 1 WHERE recorddups = 2;",
"UPDATE record SET dupin = 2 WHERE recorddups = 1;",
"ALTER TABLE record DROP COLUMN recorddups;",
"ALTER TABLE recorded ADD COLUMN recgroup VARCHAR(32) default 'Default';",
"ALTER TABLE recorded ADD COLUMN recordid INT DEFAULT NULL;",
"CREATE TABLE recgrouppassword ("
"  recgroup VARCHAR(32) NOT NULL PRIMARY KEY, "
"  password VARCHAR(10) NOT NULL, "
"  UNIQUE(recgroup)"
");",
NULL
};
        if (!performActualUpdate(updates, "1029", dbver))
            return false;
    }

    if (dbver == "1029")
    {
        const char *updates[] = {
"ALTER TABLE record CHANGE preroll startoffset INT DEFAULT 0 NOT NULL;",
"ALTER TABLE record CHANGE postroll endoffset INT DEFAULT 0 NOT NULL;",
NULL
};
        if (!performActualUpdate(updates, "1030", dbver))
            return false;
    }

    if (dbver == "1030")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD COLUMN visible TINYINT(1) NOT NULL default '1';",
"UPDATE channel SET visible = 1;",
NULL
};
        if (!performActualUpdate(updates, "1031", dbver))
            return false;
    }

    if (dbver == "1031") {
        const char *updates[] = {
"ALTER TABLE capturecard ADD dvb_on_demand TINYINT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1032", dbver))
            return false;
    }

    if (dbver == "1032")
    {
        const char *updates[] = {
"UPDATE record SET dupmethod = 6 WHERE dupmethod = 22;",
NULL
};
        if (!performActualUpdate(updates, "1033", dbver))
            return false;
    }

    if (dbver == "1033")
    {
        const char *updates[] = {
"ALTER TABLE program ADD title_pronounce VARCHAR(128) NULL;",
"ALTER TABLE program ADD INDEX (title_pronounce);",
NULL
};
        if (!performActualUpdate(updates, "1034", dbver))
            return false;
    }

    if (dbver == "1034")
    {
        const char *updates[] = {
"CREATE TABLE mythlog ("
"  logid int(10) unsigned PRIMARY KEY NOT NULL auto_increment,"
"  module char(32) NOT NULL,"
"  priority int(11) NOT NULL,"
"  acknowledged bool default 0,"
"  logdate datetime,"
"  host varchar(128),"
"  message varchar(255) NOT NULL,"
"  details text"
");",
"CREATE TABLE housekeeping ("
"  tag varchar(64) PRIMARY KEY NOT NULL,"
"  lastrun datetime"
");",
NULL
};
        if (!performActualUpdate(updates, "1035", dbver))
            return false;
    }

    if (dbver == "1035")
    {
        const char *updates[] = {
"ALTER TABLE dvb_sat CHANGE pos pos FLOAT;",
"ALTER TABLE dvb_sat ADD diseqc_pos SMALLINT DEFAULT 0 AFTER diseqc_port;",
NULL
};
        if (!performActualUpdate(updates,"1036", dbver))
            return false;
    }

    if (dbver == "1036")
    {
        const char *updates[] = {
"UPDATE channel SET callsign=chanid WHERE callsign IS NULL OR callsign='';",
"ALTER TABLE record ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE recordoverride ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1037", dbver))
            return false;
    }

    if (dbver == "1037")
    {
        const char *updates[] = {
"ALTER TABLE videosource ADD lineupid VARCHAR(64) NULL;",
"ALTER TABLE videosource ADD password VARCHAR(64) NULL;",
"ALTER TABLE program ADD ( "
"    stereo bool, "
"    subtitled bool, "
"    hdtv bool, "
"    closecaptioned bool, "
"    partnumber int, "
"    parttotal int, "
"    seriesid char(12), "
"    originalairdate date, "
"    showtype varchar(30), "
"    colorcode varchar(20), "
"    syndicatedepisodenumber varchar(20), "
"    programid char(12) "
");",
"DELETE FROM credits;",
"ALTER TABLE credits DROP INDEX chanid;",
"ALTER TABLE credits ADD UNIQUE chanid (chanid, starttime, person, role);",
"DELETE FROM people;",
"ALTER TABLE people DROP INDEX name;",
"ALTER TABLE people ADD UNIQUE name (name(41));",
"CREATE TABLE programgenres ( "
"    chanid int unsigned NOT NULL, "
"    starttime timestamp NOT NULL, "
"    relevance char(1) NOT NULL, "
"    genre char(30), "
"    PRIMARY KEY (chanid, starttime, relevance) "
");",
NULL
};
        if (!performActualUpdate(updates, "1038", dbver))
            return false;
    }

    if (dbver == "1038")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS programgenres ( "
"    chanid int unsigned NOT NULL, "
"    starttime timestamp NOT NULL, "
"    relevance char(1) NOT NULL, "
"    genre char(30), "
"    PRIMARY KEY (chanid, starttime, relevance) "
");",
NULL
};
        if (!performActualUpdate(updates, "1039", dbver))
            return false;
    }

    if (dbver == "1039")
    {
        const char *updates[] = {
"ALTER TABLE channel CHANGE name name VARCHAR(64);",
NULL
};
        if (!performActualUpdate(updates, "1040", dbver))
            return false;
    }

    if (dbver == "1040")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD outputfilters VARCHAR(255) NULL",
NULL
};
        if (!performActualUpdate(updates, "1041", dbver))
            return false;
    }

    if (dbver == "1041")
    {
        const char *updates[] = {
"ALTER TABLE record ADD seriesid varchar(12) NULL;",
"ALTER TABLE record ADD programid varchar(12) NULL;",
"ALTER TABLE recorded ADD seriesid varchar(12) NULL;",
"ALTER TABLE recorded ADD programid varchar(12) NULL;",
"ALTER TABLE oldrecorded ADD seriesid varchar(12) NULL;",
"ALTER TABLE oldrecorded ADD programid varchar(12) NULL;",
"ALTER TABLE program ADD INDEX (seriesid);",
"ALTER TABLE program ADD INDEX (programid);",
"ALTER TABLE record ADD INDEX (seriesid);",
"ALTER TABLE record ADD INDEX (programid);",
"ALTER TABLE recorded ADD INDEX (seriesid);",
"ALTER TABLE recorded ADD INDEX (programid);",
"ALTER TABLE oldrecorded ADD INDEX (seriesid);",
"ALTER TABLE oldrecorded ADD INDEX (programid);",
NULL
};
        if (!performActualUpdate(updates, "1042", dbver))
            return false;
    }

    if (dbver == "1042")
    {
        const char *updates[] = {
"INSERT INTO settings SET value=\"DataDirectMessage\";",
NULL
};
        if (!performActualUpdate(updates, "1043", dbver))
            return false;
    }

    if (dbver == "1043")
    {
        const char *updates[] = {
"ALTER TABLE program CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE category_type category_type VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE title_pronounce title_pronounce VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE showtype showtype VARCHAR(30) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE colorcode colorcode VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE syndicatedepisodenumber syndicatedepisodenumber VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE channel CHANGE channum channum VARCHAR(5) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE callsign callsign VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE name name VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE icon icon VARCHAR(255) NOT NULL DEFAULT 'none';",
"ALTER TABLE channel CHANGE videofilters videofilters VARCHAR(255) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE xmltvid xmltvid VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE outputfilters outputfilters VARCHAR(255) NOT NULL DEFAULT '';",

"ALTER TABLE record CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE profile profile VARCHAR(128) NOT NULL DEFAULT 'Default';",
"ALTER TABLE record CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE recgroup recgroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE record CHANGE station station VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE recorded CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE hostname hostname varchar(64) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE recgroup recgroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE recorded CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE oldrecorded CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1044", dbver))
            return false;
    }

    if (dbver == "1044")
    {
        const char *updates[] = {
"UPDATE channel SET icon = 'none' WHERE icon = '';",
"UPDATE record SET profile = 'Default' WHERE profile = '';",
"UPDATE record SET recgroup = 'Default' WHERE recgroup = '';",
"UPDATE recorded SET recgroup = 'Default', starttime = starttime WHERE recgroup = '';",
NULL
};
        if (!performActualUpdate(updates, "1045", dbver))
            return false;
    }

    if (dbver == "1045")
    {
        const char *updates[] = {
"UPDATE recorded SET recgroup = 'Default', starttime = starttime WHERE recgroup = '';",
NULL
};
        if (!performActualUpdate(updates, "1046", dbver))
            return false;
    }

    if (dbver == "1046")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN search INT UNSIGNED NOT NULL DEFAULT 0;",
"UPDATE record SET search = 0 WHERE search IS NULL;",
NULL
};
        if (!performActualUpdate(updates, "1047", dbver))
            return false;
    }

    if (dbver == "1047")
    {
        const char *updates[] = {
"CREATE TABLE networkiconmap ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    network VARCHAR(20) NOT NULL UNIQUE,"
"    url VARCHAR(255) NOT NULL,"
"    PRIMARY KEY(id)"
");",
"CREATE TABLE callsignnetworkmap ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    callsign VARCHAR(20) NOT NULL UNIQUE,"
"    network VARCHAR(20) NOT NULL,"
"    PRIMARY KEY(id)"
");",
NULL
};
        if (!performActualUpdate(updates, "1048", dbver))
            return false;
    }

    if (dbver == "1048")
    {
        const char *updates[] = {
"ALTER TABLE cardinput CHANGE preference preference INT NOT NULL DEFAULT 0;",
"UPDATE cardinput SET preference = 0 WHERE preference IS NULL;",
NULL
};
        if (!performActualUpdate(updates, "1049", dbver))
            return false;
    }

    if (dbver == "1049")
    {
        const char *updates[] = {
"ALTER TABLE keyword ADD COLUMN searchtype INT UNSIGNED NOT NULL DEFAULT 3;",
"ALTER TABLE keyword DROP INDEX phrase;",
"ALTER TABLE keyword DROP PRIMARY KEY;",
"ALTER TABLE keyword ADD UNIQUE(phrase,searchtype);",
NULL
};
        if (!performActualUpdate(updates, "1050", dbver))
            return false;
    }


    if (dbver == "1050")
    {
        const char *updates[] = {
"ALTER TABLE recorded CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recorded CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE recorded ADD COLUMN lastmodified TIMESTAMP NOT NULL;",
"ALTER TABLE recorded ADD COLUMN filesize BIGINT(20) DEFAULT 0 NOT NULL;",
"ALTER TABLE credits CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE oldprogram CHANGE airdate airdate DATETIME NOT NULL;",
"ALTER TABLE oldrecorded CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE oldrecorded CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE program CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE program CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE programgenres CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE programrating CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordedmarkup CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordoverride CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordoverride CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE transcoding CHANGE starttime starttime DATETIME NOT NULL;",
NULL
};
        if (!performActualUpdate(updates, "1051", dbver))
            return false;
    }

    if (dbver == "1051")
    {
        const char *updates[] = {
"update record set dupmethod = 6 where dupmethod = 0;",
"update record set dupin = 15 where dupin = 0;",
NULL
};
        if (!performActualUpdate(updates, "1052", dbver))
            return false;
    }

    if (dbver == "1052")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN stars FLOAT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN previouslyshown TINYINT(1) DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN originalairdate DATE;",
"INSERT INTO settings VALUES ('HaveRepeats', '0', NULL);",
NULL
};
        if (!performActualUpdate(updates, "1053", dbver))
            return false;
    }

    if (dbver == "1053")
    {
        const char *updates[] = {
"ALTER TABLE channel CHANGE freqid freqid VARCHAR(10);",
NULL
};
        if (!performActualUpdate(updates, "1054", dbver))
            return false;
    }

    if (dbver == "1054")
    {
        const char *updates[] = {
"ALTER TABLE program ADD INDEX id_start_end (chanid,starttime,endtime);",
"ALTER TABLE channel ADD INDEX channel_src (channum,sourceid);",
NULL
};
        if (!performActualUpdate(updates, "1055", dbver))
            return false;
    }

    if (dbver == "1055")
    {
        const char *updates[] = {
"UPDATE record SET dupmethod=6, dupin=4 WHERE dupmethod=8;",
NULL
};
        if (!performActualUpdate(updates, "1056", dbver))
            return false;
    }

    if (dbver == "1056")
    {
        const char *updates[] = {
"CREATE TABLE jobqueue ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    chanid INTEGER(10) NOT NULL,"
"    starttime DATETIME NOT NULL,"
"    inserttime DATETIME NOT NULL,"
"    type INTEGER NOT NULL,"
"    cmds INTEGER NOT NULL DEFAULT 0,"
"    flags INTEGER NOT NULL DEFAULT 0,"
"    status INTEGER NOT NULL DEFAULT 0,"
"    statustime TIMESTAMP NOT NULL,"
"    hostname varchar(64) NOT NULL DEFAULT '',"
"    args BLOB NOT NULL DEFAULT '',"
"    comment VARCHAR(128) NOT NULL DEFAULT '',"
"    PRIMARY KEY(id),"
"    UNIQUE(chanid, starttime, type, inserttime)"
");",
"ALTER TABLE record ADD COLUMN autotranscode TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autocommflag TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob1 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob2 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob3 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob4 TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1057", dbver))
            return false;

        if (gContext->GetNumSetting("AutoCommercialFlag", 1))
        {
            MSqlQuery query(MSqlQuery::InitCon());
            query.prepare("UPDATE record SET autocommflag = 1;");
            if (!query.exec())
                MythDB::DBError("dbcheck -- set autocommflag", query);
        }
    }
    if (dbver == "1057")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS transcoding;",
NULL
};
        if (!performActualUpdate(updates, "1058", dbver))
            return false;
    }

    if (dbver == "1058")
    {
        const char *updates[] = {
"UPDATE program SET category_type='series' WHERE showtype LIKE '%series%';",
NULL
};
        if (!performActualUpdate(updates, "1059", dbver))
            return false;
    }

    if (dbver == "1059")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN preserve TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1060", dbver))
            return false;
    }

    if (dbver == "1060")
    {
        const char *updates[] = {

"ALTER TABLE record ADD COLUMN record.findday TINYINT NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN record.findtime TIME NOT NULL DEFAULT '00:00:00';",
"ALTER TABLE record ADD COLUMN record.findid INT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN recorded.findid INT NOT NULL DEFAULT 0;",
"ALTER TABLE oldrecorded ADD COLUMN oldrecorded.findid INT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1061", dbver))
            return false;
    }

    if (dbver == "1061")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN inactive TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1062", dbver))
            return false;
    }

    if (dbver == "1062")
    {
        const char *updates[] = {
"ALTER TABLE cardinput ADD COLUMN freetoaironly TINYINT(1) DEFAULT 1;",
"ALTER TABLE channel ADD COLUMN useonairguide TINYINT(1) DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN dvb_diseqc_type SMALLINT(6);",
"ALTER TABLE cardinput ADD COLUMN diseqc_port SMALLINT(6);",
"ALTER TABLE cardinput ADD COLUMN diseqc_pos FLOAT;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_switch INT(11) DEFAULT 11700000;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_hi INT(11) DEFAULT 10600000;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_lo INT(11) DEFAULT 9750000;",
"ALTER TABLE channel ADD COLUMN mplexid SMALLINT(6);",
"ALTER TABLE channel ADD COLUMN serviceid SMALLINT(6);",
"ALTER TABLE channel ADD COLUMN atscsrcid INT(11) DEFAULT NULL;",
"CREATE TABLE dtv_multiplex ("
"  mplexid smallint(6) NOT NULL auto_increment, "
"  sourceid smallint(6) default NULL,"
"  transportid int(11) default NULL,"
"  networkid int(11) default NULL,"
"  frequency int(11) default NULL,"
"  inversion char(1) default 'a',"
"  symbolrate int(11) default NULL,"
"  fec varchar(10) default 'auto',"
"  polarity char(1) default NULL,"
"  modulation varchar(10) default 'auto',"
"  bandwidth char(1) default 'a',"
"  lp_code_rate varchar(10) default 'auto',"
"  transmission_mode char(1) default 'a',"
"  guard_interval varchar(10) default 'auto',"
"  visible smallint(1) NOT NULL default '0',"
"  constellation varchar(10) default 'auto',"
"  hierarchy varchar(10) default 'auto',"
"  hp_code_rate varchar(10) default 'auto',"
"  sistandard varchar(10) default 'dvb',"
"  serviceversion smallint(6) default 33,"
"  updatetimestamp timestamp(14) NOT NULL,"
"  PRIMARY KEY  (mplexid)"
") TYPE=MyISAM;",
// These should be included in an update after the 0.17 release.
// "DROP TABLE IF EXISTS dvb_channel;",
// "DROP TABLE IF EXISTS dvb_pids;",
// "DROP TABLE IF EXISTS dvb_sat;",
"CREATE TABLE dtv_privatetypes ("
"  sitype varchar(4) NOT NULL, "
"  networkid int(11) NOT NULL, "
"  private_type varchar(20) NOT NULL, "
"  private_value varchar(100) NOT NULL "
");",
//# UK DVB-T
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',9018,'channel_numbers','131');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',9018,'guide_fixup','2');",
//# Bell ExpressVu Canada
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',256,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',257,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',256,'tv_types','1,150,134,133');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',257,'tv_types','1,150,134,133');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'sdt_mapping','1');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'tv_types','1,145,154');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'guide_fixup','1');",

//# NSAB / Sirius
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',94,'tv_types','1,128');",
//# WUNC Guide
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('atsc',1793,'guide_fixup','3');",
NULL
};
        if (!performActualUpdate(updates, "1063", dbver))
            return false;
    }

    if (dbver == "1063")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS recordmatch (recordid int unsigned, "
"chanid int unsigned, starttime datetime, INDEX (recordid));",
NULL
};
        if (!performActualUpdate(updates, "1064", dbver))
            return false;
    }

    if (dbver == "1064")
    {
        const char *updates[] = {
"ALTER TABLE program CHANGE stereo stereo TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE subtitled subtitled TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE hdtv hdtv TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE closecaptioned closecaptioned TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE partnumber partnumber INT( 11 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE parttotal parttotal INT( 11 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE program CHANGE programid programid VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE oldrecorded CHANGE programid programid VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE recorded CHANGE programid programid VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE record CHANGE programid programid VARCHAR( 20 ) NOT NULL;",
NULL
};
        if (!performActualUpdate(updates, "1065", dbver))
            return false;
    }

    if (dbver == "1065")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = 'FireWire Input', cardtype = 'FIREWIRE', is_default = 1;",
"ALTER TABLE capturecard ADD COLUMN firewire_port INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN firewire_node INT UNSIGNED NOT NULL DEFAULT 2;",
"ALTER TABLE capturecard ADD COLUMN firewire_speed INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN firewire_model varchar(32) NULL;",
NULL
};
        if (!performActualUpdate(updates, "1066", dbver))
            return false;
    }

    if (dbver == "1066")
    {
        const char *updates[] = {
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES('dvb', '40999', 'guide_fixup', '4');",
NULL
};
        if (!performActualUpdate(updates, "1067", dbver))
            return false;
    }

    if (dbver == "1067")
    {
        const char *updates[] = {
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',70,'force_guide_present','yes');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',70,'guide_ranges','80,80,96,96');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4112,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4115,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4116,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12802,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12803,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12829,'channel_numbers','131');",
NULL
};
        if (!performActualUpdate(updates, "1068", dbver))
            return false;
    }

    if (dbver == "1068")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN deletepending TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1069", dbver))
            return false;
    }

    if (dbver == "1069")
    {
        const char *updates[] = {
"UPDATE jumppoints SET description = 'Weather forecasts' "
    "WHERE description = 'Weather forcasts';",
NULL
};
        if (!performActualUpdate(updates, "1070", dbver))
            return false;
    }

    if (dbver == "1070")
    {
        const char *updates[] = {
"UPDATE recorded SET bookmark = NULL WHERE bookmark = 'NULL';",
NULL
};
        if (!performActualUpdate(updates, "1071", dbver))
            return false;
    }

    if (dbver == "1071")
    {
        const char *updates[] = {
"ALTER TABLE program ADD COLUMN manualid INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE program DROP PRIMARY KEY;",
"ALTER TABLE program ADD PRIMARY KEY (chanid, starttime, manualid);",
"ALTER TABLE recordmatch ADD COLUMN manualid INT UNSIGNED;",
NULL
};
        if (!performActualUpdate(updates, "1072", dbver))
            return false;
    }

    if (dbver == "1072")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD INDEX (title);",
NULL
};
        if (!performActualUpdate(updates, "1073", dbver))
            return false;
    }

    if (dbver == "1073")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD COLUMN firewire_connection INT UNSIGNED NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1074", dbver))
            return false;
    }

    if (dbver == "1074")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = \"USB Mpeg-4 Encoder (Plextor ConvertX, etc)\", cardtype = 'GO7007', is_default = 1;",
"INSERT INTO recordingprofiles SET name = \"Default\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"Live TV\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"Low Quality\", profilegroup = 8;",
NULL
};
        if (!performActualUpdate(updates, "1075", dbver))
            return false;
    }

    if (dbver == "1075")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS recordedprogram ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  category_type varchar(64) NOT NULL default '',"
"  airdate year(4) NOT NULL default '0000',"
"  stars float unsigned NOT NULL default '0',"
"  previouslyshown tinyint(4) NOT NULL default '0',"
"  title_pronounce varchar(128) NOT NULL default '',"
"  stereo tinyint(1) NOT NULL default '0',"
"  subtitled tinyint(1) NOT NULL default '0',"
"  hdtv tinyint(1) NOT NULL default '0',"
"  closecaptioned tinyint(1) NOT NULL default '0',"
"  partnumber int(11) NOT NULL default '0',"
"  parttotal int(11) NOT NULL default '0',"
"  seriesid varchar(12) NOT NULL default '',"
"  originalairdate date default NULL,"
"  showtype varchar(30) NOT NULL default '',"
"  colorcode varchar(20) NOT NULL default '',"
"  syndicatedepisodenumber varchar(20) NOT NULL default '',"
"  programid varchar(20) NOT NULL default '',"
"  manualid int(10) unsigned NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime,manualid),"
"  KEY endtime (endtime),"
"  KEY title (title),"
"  KEY title_pronounce (title_pronounce),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY id_start_end (chanid,starttime,endtime)"
");",
"CREATE TABLE IF NOT EXISTS recordedcredits ("
"  person mediumint(8) unsigned NOT NULL default '0',"
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  role set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,person,role),"
"  KEY person (person,role)"
");",
"CREATE TABLE IF NOT EXISTS recordedrating ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  system char(8) NOT NULL default '',"
"  rating char(8) NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,system,rating),"
"  KEY starttime (starttime,system)"
");",
NULL
};

        if (!performActualUpdate(updates, "1076", dbver))
            return false;
    }

    if (dbver == "1076")
    {
        const char *updates[] = {
"ALTER TABLE channel MODIFY COLUMN serviceid mediumint unsigned;",
NULL
};
        if (!performActualUpdate(updates, "1077", dbver))
            return false;
    }

    if (dbver == "1077")
    {
        const char *updates[] = {
"INSERT INTO dtv_privatetypes "
"(sitype,networkid,private_type,private_value) VALUES "
"('dvb',40999,'parse_subtitle_list',"
"'1070,1049,1041,1039,1038,1030,1016,1131,1068,1069');",
NULL
};
        if (!performActualUpdate(updates, "1078", dbver))
            return false;
    }

    if (dbver == "1078")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD COLUMN dvb_hw_decoder INT DEFAULT '0';",
NULL
};
        if (!performActualUpdate(updates, "1079", dbver))
            return false;
    }

    if (dbver == "1079")
    {
        const char *updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN recordid INT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1080", dbver))
            return false;
    }

    if (dbver == "1080")
    {
        const char *updates[] = {
"ALTER table recorded ADD INDEX (recordid);",
"ALTER table oldrecorded ADD INDEX (recordid);",
NULL
};
        if (!performActualUpdate(updates, "1081", dbver))
            return false;
    }

    if (dbver == "1081")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS oldfind ("
"recordid INT NOT NULL DEFAULT 0,"
"findid INT NOT NULL DEFAULT 0,"
"PRIMARY KEY (recordid, findid)"
");",
"ALTER TABLE record ADD COLUMN parentid INT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1082", dbver))
            return false;
    }

    if (dbver == "1082")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS videobookmarks ("
"  filename varchar(255) NOT NULL,"
"  bookmark varchar(128) default NULL,"
"  PRIMARY KEY (filename)"
");",
NULL
};
        if (!performActualUpdate(updates, "1083", dbver))
            return false;
    }

    if (dbver == "1083")
    {
        const char *updates[] = {
"DELETE FROM settings where value = 'UseCategoriesAsRecGroups';",
NULL
};
        if (!performActualUpdate(updates, "1084", dbver))
            return false;
    }

    if (dbver == "1084")
    {
        const char *updates[] = {
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 6;",
"INSERT INTO recordingprofiles SET name = 'Medium Quality', profilegroup = 6;",
"INSERT INTO recordingprofiles SET name = 'Low Quality', profilegroup = 6;",
"REPLACE INTO settings SET value = 'DefaultTranscoder', data = '0';",
"ALTER TABLE record ADD COLUMN transcoder INT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN transcoder INT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1085", dbver))
            return false;

        MSqlQuery recordids(MSqlQuery::InitCon());
        recordids.prepare(
            "SELECT r.recordid"
            "  FROM record r, recordingprofiles p, codecparams c"
            "  WHERE c.name = 'autotranscode'"
            "    AND c.value = 1"
            "    AND c.profile = p.id"
            "    AND r.profile = p.name"
            ";");
        if (recordids.exec() && recordids.isActive() && recordids.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (recordids.next())
            {
                update.prepare(QString(
                    "UPDATE record SET autotranscode = 1 WHERE recordid = %1;")
                    .arg(recordids.value(0).toInt()));
                if (!update.exec())
                    MythDB::DBError("dbcheck -- set autotranscode", update);
            }
        }
    }

    if (dbver == "1085")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS pidcache;",
"CREATE TABLE IF NOT EXISTS pidcache ("
"  chanid smallint(6) NOT NULL default '0',"
"  pid int(11) NOT NULL default '-1',"
"  tableid int(11) NOT NULL default '-1',"
"  INDEX(chanid)"
");",
NULL
};
        if (!performActualUpdate(updates, "1086", dbver))
            return false;
    }

    if (dbver == "1086")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS xvmc_buffer_settings ( "
"  id int(11) NOT NULL auto_increment, "
"  description varchar(255) NOT NULL default '',"
"  osd_num int(11) NOT NULL default '0',"
"  osd_res_num int(11) NOT NULL default '0',"
"  min_surf int(11) NOT NULL default '0',"
"  max_surf int(11) NOT NULL default '0',"
"  decode_num int(11) NOT NULL default '0',"
"  agressive int(11) NOT NULL default '1',"
"  PRIMARY KEY  (id) );",
"INSERT INTO xvmc_buffer_settings VALUES (1,'Default / nVidia',2,2,8,16,8,1);",
"INSERT INTO xvmc_buffer_settings VALUES (2,'VLD (More decode buffers)',2,2,8,16,16,1);",
NULL
};

        if (!performActualUpdate(updates, "1087", dbver))
            return false;
    }

    if (dbver == "1087")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN tsdefault FLOAT NOT NULL DEFAULT 1.0;",
"ALTER TABLE recorded ADD COLUMN timestretch FLOAT NOT NULL DEFAULT 1.0;",
"UPDATE record SET tsdefault = 1.0;", // we've had the default not take before
"UPDATE recorded SET timestretch = 1.0;",
NULL
};
        if (!performActualUpdate(updates, "1088", dbver))
            return false;
    }

    if (dbver == "1088")
    {
        const char *updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
"UPDATE oldrecorded SET station=chanid;",
"ALTER TABLE oldrecorded ADD rectype INT(10) UNSIGNED NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET rectype=1;",
"ALTER TABLE oldrecorded ADD duplicate TINYINT(1) NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET duplicate=1;",
"ALTER TABLE oldrecorded ADD recstatus INT NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET recstatus=-3;",
"ALTER TABLE oldrecorded DROP PRIMARY KEY;",
"ALTER TABLE oldrecorded ADD PRIMARY KEY (station,starttime,title);",
NULL
};

        if (!performActualUpdate(updates, "1089", dbver))
            return false;
    }

    if (dbver == "1089")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = 'DBOX2 Input', cardtype = 'DBOX2', is_default = 1;",
"ALTER TABLE capturecard ADD COLUMN dbox2_port INT UNSIGNED NOT NULL DEFAULT 31338;",
"ALTER TABLE capturecard ADD COLUMN dbox2_httpport INT UNSIGNED NOT NULL DEFAULT 80;",
"ALTER TABLE capturecard ADD COLUMN dbox2_host varchar(32) NULL;",
NULL
};

        if (!performActualUpdate(updates, "1090", dbver))
            return false;
    }

    if (dbver == "1090")
    {
        const char *updates[] = {
"DELETE FROM dtv_privatetypes WHERE sitype='dvb' AND networkid=40999 AND private_type='parse_subtitle_list';",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',40999,'parse_subtitle_list','1070,1308,1041,1306,1307,1030,1016,1131,1068,1069');",
NULL
};

        if (!performActualUpdate(updates, "1091", dbver))
            return false;
    }

    if (dbver == "1091")
    {
        const char *updates[] = {
"ALTER TABLE capturecard CHANGE dvb_recordts dvb_recordts INT DEFAULT '1';",
"UPDATE capturecard SET dvb_recordts=1;",
NULL
};

        if (!performActualUpdate(updates, "1092", dbver))
            return false;
    }

    if (dbver == "1092")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN recpriority INT NOT NULL DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1093", dbver))
            return false;
    }

    if (dbver == "1093")
    {
        MSqlQuery recordids(MSqlQuery::InitCon());
        recordids.prepare("SELECT recordid,recpriority FROM record;");
        if (!recordids.exec())
            return false;

        if (recordids.isActive() && recordids.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (recordids.next())
            {
                update.prepare(QString(
                    "UPDATE recorded SET recpriority = %1 WHERE recordid = %2;")
                    .arg(recordids.value(1).toInt())
                    .arg(recordids.value(0).toInt()));
                if (!update.exec())
                    return false;
            }
        }

        if (!UpdateDBVersionNumber("1094"))
            return false;

        dbver = "1094";
    }

    if (dbver == "1094")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN basename varchar(128) NOT NULL DEFAULT '';",
"UPDATE recorded SET basename = CONCAT(chanid, '_', DATE_FORMAT(starttime, '%Y%m%d%H%i00'), '_', DATE_FORMAT(endtime, '%Y%m%d%H%i00'), '.nuv');",
NULL
};

        if (!performActualUpdate(updates, "1095", dbver))
            return false;
    }

    if (dbver == "1095")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN progstart DATETIME NOT NULL;",
"UPDATE recorded SET progstart = starttime;",
"ALTER TABLE recorded ADD COLUMN progend DATETIME NOT NULL;",
"UPDATE recorded SET progend = endtime;",
NULL
};

        if (!performActualUpdate(updates, "1096", dbver))
            return false;
    }

    if (dbver == "1096")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD COLUMN signal_timeout int NOT NULL default '1000';",
"ALTER TABLE capturecard ADD COLUMN channel_timeout int NOT NULL default '3000';",
"DROP TABLE IF EXISTS dvb_signal_quality;",
"DROP TABLE IF EXISTS dvb_transport;",
"DROP TABLE IF EXISTS dvb_channel;",
"DROP TABLE IF EXISTS dvb_pids;",
"DROP TABLE IF EXISTS dvb_sat;",
NULL
};

        if (!performActualUpdate(updates, "1097", dbver))
            return false;
    }

    if (dbver == "1097")
    {
        const char *updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN reactivate SMALLINT NOT NULL DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1098", dbver))
            return false;
    }

    if (dbver == "1098")
    {
        const char *updates[] = {
"UPDATE record SET findid=TO_DAYS(startdate) WHERE type=6;",
NULL
};

        if (!performActualUpdate(updates, "1099", dbver))
            return false;
    }

    if (dbver == "1099")
    {
        const char *updates[] = {
"ALTER TABLE videosource ADD COLUMN useeit SMALLINT NOT NULL DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1100", dbver))
            return false;
    }

    if (dbver == "1100")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS tvchain ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  chainid varchar(128) NOT NULL default '',"
"  chainpos int(10) NOT NULL default '0',"
"  discontinuity tinyint(1) NOT NULL default '0',"
"  watching int(10) NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime)"
");",
NULL
};

        if (!performActualUpdate(updates, "1101", dbver))
            return false;
    }

    if (dbver == "1101")
    {
        const char *updates[] = {
"ALTER TABLE tvchain ADD COLUMN hostprefix VARCHAR(128) NOT NULL DEFAULT '';",
NULL
};

        if (!performActualUpdate(updates, "1102", dbver))
            return false;
    }

    if (dbver == "1102")
    {
        const char *updates[] = {
"ALTER TABLE tvchain ADD COLUMN cardtype VARCHAR(32) NOT NULL DEFAULT 'V4L';",
NULL
};
        if (!performActualUpdate(updates, "1103", dbver))
            return false;
    }

    if (dbver == "1103")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS inuseprograms ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  playid varchar(128) NOT NULL default '',"
"  lastupdatetime datetime NOT NULL default '0000-00-00 00:00:00',"
"  INDEX (chanid,starttime)"
");",
NULL
};

        if (!performActualUpdate(updates, "1104", dbver))
            return false;
    }

    if (dbver == "1104")
    {
        const char *updates[] = {
"ALTER TABLE tvchain ADD COLUMN input VARCHAR(32) NOT NULL DEFAULT '';",
"ALTER TABLE tvchain ADD COLUMN channame VARCHAR(32) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1105", dbver))
            return false;
    }

    if (dbver == "1105")
    {
        const char *updates[] = {
"UPDATE settings SET data = '1' WHERE value = 'AutoExpireMethod' "
      " AND data = '0';",
NULL
};
        if (!performActualUpdate(updates, "1106", dbver))
            return false;
    }

    if (dbver == "1106")
    {
        const char *updates[] = {
"ALTER TABLE inuseprograms ADD COLUMN hostname varchar(64) NOT NULL DEFAULT '';",
"ALTER TABLE inuseprograms CHANGE playid recusage VARCHAR(128) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1107", dbver))
            return false;
    }

    if (dbver == "1107")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN playgroup VARCHAR(32) NOT NULL DEFAULT 'Default'; ",
"UPDATE record SET playgroup = 'Default'; ",
"ALTER TABLE recorded ADD COLUMN playgroup VARCHAR(32) NOT NULL DEFAULT 'Default'; ",
"UPDATE recorded SET playgroup = 'Default'; ",
"CREATE TABLE playgroup ("
"  name VARCHAR(32) NOT NULL PRIMARY KEY, "
"  titlematch VARCHAR(255) NOT NULL DEFAULT '', "
"  skipahead INT NOT NULL DEFAULT 0, "
"  skipback INT NOT NULL DEFAULT 0, "
"  timestretch INT NOT NULL DEFAULT 0 "
");",
"INSERT INTO playgroup (name,skipahead,skipback,timestretch) "
"            VALUES ('Default',30,5,100);",
NULL
};

        if (!performActualUpdate(updates, "1108", dbver))
            return false;
    }

    if (dbver == "1108")
    {
        const char *updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN generic TINYINT(1) DEFAULT 0;",
"ALTER TABLE program ADD COLUMN generic TINYINT(1) DEFAULT 0;",
"UPDATE program SET generic = 1 WHERE "
"  ((programid = '' AND subtitle = '' AND description = '') OR "
"   (programid <> '' AND category_type = 'series' AND "
"    program.programid LIKE '%0000'));",
NULL
};

        if (!performActualUpdate(updates, "1109", dbver))
            return false;
    }

    if (dbver == "1109")
    {
        const char *updates[] = {
"ALTER TABLE recordedprogram ADD COLUMN generic TINYINT(1) DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1110", dbver))
            return false;
    }

    if (dbver == "1110")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN profile VARCHAR(32) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1111", dbver))
            return false;
    }

    if (dbver == "1111")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS conflictresolutionany;",
"DROP TABLE IF EXISTS conflictresolutionoverride;",
"DROP TABLE IF EXISTS conflictresolutionsingle;",
"DROP TABLE IF EXISTS recordoverride;",
NULL
};
        if (!performActualUpdate(updates, "1112", dbver))
            return false;
    }

    if (dbver == "1112")
    {
        const char *updates[] = {
"DELETE from settings WHERE value LIKE 'DailyWakeup%';",
NULL
};

        if (!performActualUpdate(updates, "1113", dbver))
            return false;
    }

    if (dbver == "1113")
    {
        const char *updates[] = {
"INSERT INTO settings VALUES ('NextSuggestedMythfilldatabaseRun',NULL,NULL);",
"INSERT INTO settings VALUES ('HonorGrabberNextSuggestedMythfilldatabaseRunTime','1',NULL);",
NULL
};

        if (!performActualUpdate(updates, "1114", dbver))
            return false;
    }

    if (dbver == "1114")
    {
        const char *updates[] = {
"ALTER TABLE tvchain ADD COLUMN endtime DATETIME NOT NULL default '0000-00-00 00:00:00';",
NULL
};
        if (!performActualUpdate(updates, "1115", dbver))
            return false;
    }

    if (dbver == "1115")
    {
        const char *updates[] = {
"DELETE FROM keybindings WHERE context = 'Music' AND action LIKE 'JUMP%';",
"DELETE FROM keybindings WHERE context = 'Game' AND action LIKE 'JUMP%';",
NULL
};

        if (!performActualUpdate(updates, "1116", dbver))
            return false;
    }

    if (dbver == "1116")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD COLUMN displayname VARCHAR(32) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1117", dbver))
            return false;
    }

    if (dbver == "1117")
    {
        const char *updates[] = {
"ALTER TABLE capturecard DROP COLUMN displayname;",
"ALTER TABLE cardinput ADD COLUMN displayname VARCHAR(64) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1118", dbver))
            return false;
    }

    if (dbver == "1118")
    {
        const char *updates[] = {
"UPDATE settings SET value = 'MythFillGrabberSuggestsTime' "
    "WHERE value = 'HonorGrabberNextSuggestedMythfilldatabaseRunTime';",
"UPDATE settings SET value = 'MythFillSuggestedRunTime', "
    "    data = '1970-01-01T00:00:00' "
    "WHERE value = 'NextSuggestedMythfilldatabaseRun';",
NULL
};
        if (!performActualUpdate(updates, "1119", dbver))
            return false;
    }

    if (dbver == "1119")
    {
        const char *updates[] = {
// "UPDATE playgroup, settings set playgroup.skipahead = settings.data"
// " WHERE settings.value = 'FastForwardAmount' AND playgroup.name = 'Default';",
// "UPDATE playgroup, settings set playgroup.skipback = settings.data"
// " WHERE settings.value = 'RewindAmount' AND playgroup.name = 'Default';",
"ALTER TABLE playgroup ADD COLUMN jump INT NOT NULL DEFAULT 0;",
// "UPDATE playgroup, settings set playgroup.jump = settings.data"
// " WHERE settings.value = 'JumpAmount' AND playgroup.name = 'Default';",
NULL
};
        if (!performActualUpdate(updates, "1120", dbver))
            return false;
    }

    if (dbver == "1120")
    {
        const char *updates[] = {
"UPDATE keybindings SET action = 'ADJUSTSTRETCH' "
"       WHERE action = 'TOGGLESTRETCH';",
NULL
};
        if (!performActualUpdate(updates, "1121", dbver))
            return false;
    }

    if (dbver == "1121")
    {
        const char *updates[] = {
"ALTER TABLE channel CHANGE channum channum VARCHAR(10) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1122", dbver))
            return false;
    }

    if (dbver == "1122")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD duplicate TINYINT(1) NOT NULL DEFAULT 0;",
"UPDATE recorded SET duplicate=1;",
NULL
};

        if (!performActualUpdate(updates, "1123", dbver))
            return false;
    }

    if (dbver == "1123")
    {
        const char *updates[] = {
"ALTER TABLE cardinput ADD COLUMN radioservices TINYINT(1) DEFAULT 1;",
NULL
};
        if (!performActualUpdate(updates, "1124", dbver))
            return false;
    }

    if (dbver == "1124")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD parentid int(10) NOT NULL DEFAULT 0;",
"ALTER TABLE cardinput ADD childcardid int(10) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1125", dbver))
            return false;
    }

    if (dbver == "1125")
    {
       const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN transcoded TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1126", dbver))
            return false;
    }

    if (dbver == "1126")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS xvmc_buffer_settings;",
"ALTER TABLE capturecard DROP COLUMN dvb_dmx_buf_size;",
"ALTER TABLE capturecard DROP COLUMN dvb_pkt_buf_size;",
NULL
};
        if (!performActualUpdate(updates, "1127", dbver))
            return false;
    }

    if(dbver == "1127")
    {
       const char *updates[] = {
"ALTER TABLE cardinput ADD COLUMN dishnet_eit TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1128", dbver))
            return false;
    }

    if (dbver == "1128")
    {

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT chanid, starttime, cutlist FROM recorded"
                      " WHERE LENGTH(cutlist) > 1;");
        if (query.exec() && query.isActive() && query.size() > 0)
        {
            MSqlQuery insert(MSqlQuery::InitCon());
            while (query.next())
            {
                const QString cutlist = query.value(2).toString();
                const QStringList strlist =
                    cutlist.split("\n", QString::SkipEmptyParts);
                QStringList::const_iterator i;
                for (i = strlist.begin(); i != strlist.end(); ++i)
                {
                    long long start = 0, end = 0;
                    QByteArray tmp = (*i).toAscii();
                    if (sscanf(tmp.constData(),
                               "%lld - %lld", &start, &end) == 2)
                    {
                        insert.prepare(
                               "INSERT INTO recordedmarkup (chanid, starttime,"
                               " type, mark) "
                               "VALUES (:CHANID, :STARTTIME, :TYPE, :MARK);");
                        insert.bindValue(":CHANID", query.value(0).toString());
                        insert.bindValue(":STARTTIME",
                                         query.value(1).toDateTime());

                        insert.bindValue(":TYPE", 1);
                        insert.bindValue(":MARK", start);
                        if (!insert.exec())
                            MythDB::DBError("dbcheck -- recordedmarkup, "
                                            "cut start", insert);

                        insert.bindValue(":TYPE", 0);
                        insert.bindValue(":MARK", end);
                        if (!insert.exec())
                            MythDB::DBError("dbcheck -- recordedmarkup, "
                                            "cut end", insert);
                    }
                }
            }
        }

        const char *updates[] = {
"UPDATE recorded SET bookmark='0' WHERE bookmark IS NULL",
"INSERT INTO recordedmarkup (chanid, starttime, type, mark) SELECT"
" chanid, starttime, '2', bookmark FROM recorded WHERE bookmark <> 0;",
"ALTER TABLE recorded CHANGE cutlist cutlist TINYINT(1) NOT NULL DEFAULT 0",
"UPDATE recorded SET bookmark='1' WHERE bookmark > 0",
"ALTER TABLE recorded CHANGE bookmark bookmark TINYINT(1) NOT NULL DEFAULT 0",
NULL
};
        if (!performActualUpdate(updates, "1129", dbver))
            return false;
    }

    if (dbver == "1129")
    {
        const char *updates[] = {
"INSERT INTO dtv_privatetypes (sitype, networkid, private_type, private_value) "
"       VALUES                ('dvb',  4096,     'guide_fixup', '5');",
NULL
};
        if (!performActualUpdate(updates, "1130", dbver))
            return false;
    }

    if (dbver == "1130")
    {
        const char *updates[] = {
"ALTER TABLE cardinput ADD COLUMN recpriority INT NOT NULL DEFAULT 0;",
"UPDATE cardinput SET recpriority = preference;",
NULL
};
        if (!performActualUpdate(updates, "1131", dbver))
            return false;
    }

    if (dbver == "1131")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD COLUMN dvb_tuning_delay INT UNSIGNED NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1132", dbver))
            return false;
    }

    if (dbver == "1132")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD COLUMN tmoffset INT NOT NULL default '0';",
NULL
};

        if (!performActualUpdate(updates, "1133", dbver))
            return false;
    }

    if (dbver == "1133")
    {
        const char *updates[] = {
"UPDATE capturecard SET defaultinput='DVBInput' WHERE defaultinput='Television' AND cardtype = 'DVB';",
NULL
};

        if (!performActualUpdate(updates, "1134", dbver))
            return false;
    }

    if (dbver == "1134")
    {
        const char *updates[] = {
"ALTER TABLE jumppoints CHANGE keylist keylist VARCHAR(128);",
"ALTER TABLE keybindings CHANGE keylist keylist VARCHAR(128);",
NULL
};

       if (!performActualUpdate(updates, "1135", dbver))
            return false;
    }

    if (dbver == "1135")
    {
        const char *updates[] = {
"ALTER TABLE program ADD listingsource INT NOT NULL default '0';",
NULL
};

       if (!performActualUpdate(updates, "1136", dbver))
            return false;
    }

    if (dbver == "1136")
    {
        const char *updates[] = {
"ALTER TABLE recordedprogram ADD listingsource INT NOT NULL default '0';",
NULL
};

       if (!performActualUpdate(updates, "1137", dbver))
            return false;
    }

    if (dbver == "1137")
    {
        const char *updates[] = {
"UPDATE dtv_multiplex SET networkid='0' WHERE sistandard='atsc';",
NULL
};

       if (!performActualUpdate(updates, "1138", dbver))
            return false;
    }

    if (dbver == "1138")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD contrast   INT NOT NULL default '0';",
"ALTER TABLE capturecard ADD brightness INT NOT NULL default '0';",
"ALTER TABLE capturecard ADD colour     INT NOT NULL default '0';",
"ALTER TABLE capturecard ADD hue        INT NOT NULL default '0';",
NULL
};

       if (!performActualUpdate(updates, "1139", dbver))
            return false;
    }

    if (dbver == "1139")
    {
        const char *updates[] = {
"UPDATE dtv_multiplex set modulation = '8psk' WHERE modulation = 'qpsk_8';",
NULL
};

       if (!performActualUpdate(updates, "1140", dbver))
            return false;
    }

    if (dbver == "1140")
    {
        const char *updates[] = {
"ALTER TABLE oldrecorded ADD INDEX (recstatus,programid,seriesid);",
"ALTER TABLE oldrecorded ADD INDEX (recstatus,title,subtitle);",
NULL
};

       if (!performActualUpdate(updates, "1141", dbver))
            return false;
    }

    if (dbver == "1141")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS customexample ("
"  rulename VARCHAR(64) NOT NULL PRIMARY KEY, "
"  fromclause text NOT NULL, "
"  whereclause text NOT NULL "
");",
NULL
};
        if (!performActualUpdate(updates, "1142", dbver))
            return false;
    }

    if (dbver == "1142")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN prefinput int(10) NOT NULL DEFAULT 0;",
NULL
};

       if (!performActualUpdate(updates, "1143", dbver))
            return false;
    }

    if (dbver == "1143")
    {
        const char *updates[] = {
"UPDATE settings SET data=0 WHERE value='PlaybackHue' AND (data >= 45 AND data <= 55);",
NULL
};

       if (!performActualUpdate(updates, "1144", dbver))
            return false;
    }

    if (dbver == "1144")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS eit_cache ("
"  chanid  INT(10) NOT NULL, "
"  eventid SMALLINT UNSIGNED NOT NULL, "
"  tableid TINYINT UNSIGNED NOT NULL, "
"  version TINYINT UNSIGNED NOT NULL, "
"  endtime INT UNSIGNED NOT NULL, "
"  PRIMARY KEY (chanid, eventid) "
");",
NULL
};

       if (!performActualUpdate(updates, "1145", dbver))
            return false;
    }

    if (dbver == "1145")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = 'Freebox Input', cardtype = 'Freebox', is_default = 1;",
NULL
};

        if (!performActualUpdate(updates, "1146", dbver))
            return false;
    }

    if (dbver == "1146")
    {
        const char *updates[] = {
"ALTER TABLE program ADD first TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE program ADD last  TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1147", dbver))
            return false;
    }

    if (dbver == "1147")
    {
        const char *updates[] = {
"ALTER TABLE recordedprogram ADD first TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE recordedprogram ADD last  TINYINT(1) NOT NULL DEFAULT 0;",
NULL
};

        if (!performActualUpdate(updates, "1148", dbver))
            return false;
    }

    if (dbver == "1148")
    {
        const char *updates[] = {
"INSERT INTO recordingprofiles SET name = 'Default',      profilegroup = 10;",
"INSERT INTO recordingprofiles SET name = 'Live TV',      profilegroup = 10;",
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 10;",
"INSERT INTO recordingprofiles SET name = 'Low Quality',  profilegroup = 10;",

"INSERT INTO profilegroups SET name = 'HDHomeRun Recorders', "
"  cardtype = 'HDHOMERUN', is_default = 1;",
"INSERT INTO recordingprofiles SET name = 'Default',      profilegroup = 11;",
"INSERT INTO recordingprofiles SET name = 'Live TV',      profilegroup = 11;",
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 11;",
"INSERT INTO recordingprofiles SET name = 'Low Quality',  profilegroup = 11;",

"INSERT INTO profilegroups SET name = 'CRC IP Recorders', "
"  cardtype = 'CRC_IP', is_default = 1;",
"INSERT INTO recordingprofiles SET name = 'Default',      profilegroup = 12;",
"INSERT INTO recordingprofiles SET name = 'Live TV',      profilegroup = 12;",
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 12;",
"INSERT INTO recordingprofiles SET name = 'Low Quality',  profilegroup = 12;",
NULL
};

        if (!performActualUpdate(updates, "1149", dbver))
            return false;
    }

    if (dbver == "1149")
    {
        const char *updates[] = {
"UPDATE settings SET data='channum' WHERE value='ChannelOrdering' AND data!='callsign';",
NULL
};

        if (!performActualUpdate(updates, "1150", dbver))
            return false;
    }

    if (dbver == "1150")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD atsc_major_chan INT UNSIGNED NOT NULL default '0';",
"ALTER TABLE channel ADD atsc_minor_chan INT UNSIGNED NOT NULL default '0';",
/* the updates were split in two for mysql 3.23 compatibility. */
"UPDATE channel SET atsc_major_chan = atscsrcid;",
"UPDATE channel SET atsc_major_chan = atsc_major_chan / 256;",
"UPDATE channel SET atsc_minor_chan = atscsrcid;",
"UPDATE channel SET atsc_minor_chan = atsc_minor_chan % 256;",
NULL
};

        if (!performActualUpdate(updates, "1151", dbver))
            return false;
    }

    if (dbver == "1151")
    {
        const char *updates[] = {
"ALTER TABLE programgenres ADD INDEX (genre);",
NULL
};

        if (!performActualUpdate(updates, "1152", dbver))
            return false;
    }

    if (dbver == "1152")
    {
        const char *updates[] = {
"ALTER TABLE recordedmarkup DROP PRIMARY KEY, ADD PRIMARY KEY (chanid,starttime,type,mark);",
"CREATE TABLE IF NOT EXISTS recordedseek ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  mark bigint(20) NOT NULL default '0',"
"  offset varchar(32) default NULL,"
"  type int(11) NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime,type,mark));",
"INSERT IGNORE INTO recordedseek (chanid, starttime, mark, offset, type) SELECT"
" chanid, starttime, mark, offset, type FROM recordedmarkup WHERE type in (6, 7, 9);",
"DELETE FROM recordedmarkup WHERE type in (6, 7, 9);",
NULL
};

       if (!performActualUpdate(updates, "1153", dbver))
            return false;
    }

    if (dbver == "1153")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS diseqc_config "
" ( cardinputid INT(10) UNSIGNED NOT NULL, "
"   diseqcid    INT(10) UNSIGNED NOT NULL, "
"   value       VARCHAR(16) NOT NULL default '', "
"   KEY id (cardinputid) );",
"CREATE TABLE IF NOT EXISTS diseqc_tree "
" ( diseqcid        INT(10) UNSIGNED NOT NULL auto_increment, "
"   parentid        INT(10) UNSIGNED default NULL, "
"   ordinal         TINYINT(3) UNSIGNED NOT NULL, "
"   type            VARCHAR(16) NOT NULL default '', "
"   subtype         VARCHAR(16) NOT NULL default '', "
"   description     VARCHAR(32) NOT NULL default '', "
"   switch_ports    TINYINT(3) UNSIGNED NOT NULL default 0, "
"   rotor_hi_speed  FLOAT NOT NULL default 0.0, "
"   rotor_lo_speed  FLOAT NOT NULL default 0.0, "
"   rotor_positions VARCHAR(255) NOT NULL default '', "
"   lnb_lof_switch  INT(10) NOT NULL default 0, "
"   lnb_lof_hi      INT(10) NOT NULL default 0, "
"   lnb_lof_lo      INT(10) NOT NULL default 0, "
"   PRIMARY KEY (diseqcid), KEY parentid (parentid) );",
"ALTER TABLE capturecard ADD diseqcid INT(10) UNSIGNED default NULL;",
NULL
};

        if (!performActualUpdate(updates, "1154", dbver))
            return false;

        convert_diseqc_db();
    }

    if (dbver == "1154")
    {
        const char *updates[] = {
"ALTER TABLE cardinput CHANGE startchan startchan VARCHAR(10);",
"ALTER TABLE cardinput CHANGE tunechan  tunechan  VARCHAR(10);",
NULL
};

        if (!performActualUpdate(updates, "1155", dbver))
            return false;
    }

    if (dbver == "1155")
    {
        const char *updates[] = {
"ALTER TABLE `diseqc_tree` ADD `cmd_repeat` INT NOT NULL DEFAULT '1';",
NULL
};

        if (!performActualUpdate(updates, "1156", dbver))
            return false;
    }

    if (dbver == "1156")
    {
        const char *updates[] = {
"UPDATE codecparams SET value='0' WHERE name='mpeg4optionidct' OR name='mpeg4optionime';",
NULL
};

        if (!performActualUpdate(updates, "1157", dbver))
            return false;
    }

    if (dbver == "1157")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN next_record DATETIME NOT NULL;",
"ALTER TABLE record ADD COLUMN last_record DATETIME NOT NULL;",
"ALTER TABLE record ADD COLUMN last_delete DATETIME NOT NULL;",
NULL
};

        if (!performActualUpdate(updates, "1158", dbver))
            return false;
    }

    if (dbver == "1158")
    {
        const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN watched TINYINT NOT NULL DEFAULT '0';",
NULL
};

        if (!performActualUpdate(updates, "1159", dbver))
            return false;
    }

    if (dbver == "1159")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT DISTINCT chanid, starttime FROM recordedmarkup "
                      "WHERE type = 1;");
        if (query.exec() && query.isActive() && query.size() > 0)
        {
            MSqlQuery fixup(MSqlQuery::InitCon());
            while (query.next())
            {
                fixup.prepare(
                       "UPDATE recorded SET cutlist = 1 "
                       "WHERE chanid = :CHANID AND starttime =  :STARTTIME;");
                fixup.bindValue(":CHANID", query.value(0).toString());
                fixup.bindValue(":STARTTIME", query.value(1).toDateTime());

                if (!fixup.exec())
                    MythDB::DBError("dbcheck -- cutlist", fixup);
            }
        }

        const char *updates[] = { NULL };
        if (!performActualUpdate(updates, "1160", dbver))
            return false;
    }

    if (dbver == "1160")
    {
        const char *updates[] = {
"ALTER TABLE diseqc_tree ADD COLUMN lnb_pol_inv TINYINT NOT NULL DEFAULT '0';",
NULL
};

        if (!performActualUpdate(updates, "1161", dbver))
            return false;
    }

    if (dbver == "1161")
    {
        const char *updates[] = {
"UPDATE settings SET data='Auto' WHERE value='EITTimeOffset';",
NULL
};

        if (!performActualUpdate(updates, "1162", dbver))
            return false;
    }

    if (dbver == "1162")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        QMap<int,int> fixupMap;

        // For new mask values, see playbackbox.h
        fixupMap[0] = 0x01; // Titles
        fixupMap[1] = 0x03; // Titles & Categories
        fixupMap[2] = 0x07; // Titles & Categories & Recording Groups
        fixupMap[3] = 0x05; // Titles & Recording Groups
        fixupMap[4] = 0x02; // Categories
        fixupMap[5] = 0x06; // Categories & Recording Groups
        fixupMap[6] = 0x04; // Recording Groups

        query.prepare("SELECT data, hostname FROM settings "
                          "WHERE value = 'DisplayGroupDefaultView'");
        if (query.exec() && query.isActive() && query.size() > 0)
        {
            MSqlQuery fixup(MSqlQuery::InitCon());

            fixup.prepare("INSERT settings ( value, data, hostname ) "
                              "VALUES ( :VALUE, :DATA, :HOSTNAME )");
            fixup.bindValue(":VALUE", "DisplayGroupDefaultViewMask");

            while (query.next())
            {
                fixup.bindValue(":DATA", fixupMap[query.value(0).toInt()]);
                fixup.bindValue(":HOSTNAME", query.value(1).toString());
                if (!fixup.exec())
                    MythDB::DBError("dbcheck -- DisplayGroupDefaultViewMask",
                                    fixup);
            }
        }

        const char *updates[] = { NULL };
        if (!performActualUpdate(updates, "1163", dbver))
            return false;
    }

    if (dbver == "1163")
    {
        const char *updates[] = {
"ALTER TABLE capturecard DROP COLUMN dvb_recordts;",
"ALTER TABLE capturecard DROP COLUMN dvb_hw_decoder;",
NULL
};

        if (!performActualUpdate(updates, "1164", dbver))
            return false;
    }

    if (dbver == "1164")
    {
        const char *updates[] = {
"CREATE INDEX program_manualid ON program (manualid);",
NULL
};

        if (!performActualUpdate(updates, "1165", dbver))
            return false;
    }

    if (dbver == "1165")
    {
        const char *updates[] = {
"ALTER TABLE recordedrating CHANGE rating rating CHAR(16);",
"ALTER TABLE programrating CHANGE rating rating CHAR(16);",
NULL
};

        if (!performActualUpdate(updates, "1166", dbver))
            return false;
    }

    if (dbver == "1166")
    {
        const char *updates[] = {
"DELETE FROM keybindings where action = 'VIEWVIDEOSOURCE';",
NULL
};

        if (!performActualUpdate(updates, "1167", dbver))
            return false;
    }

    if (dbver == "1167")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS dvdbookmark ("
"   serialid varchar(16) NOT NULL default '',"
"   name varchar(32) default NULL,"
"   title smallint(6) NOT NULL default 0,"
"   audionum tinyint(4) NOT NULL default -1,"
"   subtitlenum tinyint(4) NOT NULL default -1,"
"   framenum bigint(20) NOT NULL default 0,"
"   timestamp timestamp NOT NULL, "
"   PRIMARY KEY (serialid));",
NULL
};

        if (!performActualUpdate(updates, "1168", dbver))
            return false;
    }

    if (dbver == "1168")
    {
        const char *updates[] = {
"UPDATE dtv_multiplex SET sistandard='mpeg', networkid=NULL, transportid=NULL "
"WHERE sistandard = 'dvb' AND networkid IS NULL OR networkid < 1;",
NULL
};

        if (!performActualUpdate(updates, "1169", dbver))
            return false;
    }

    if (dbver == "1169")
    {
        QString thequery =
            "SELECT mplexid "
            "FROM channel "
            "WHERE mplexid != 0     AND "
            "      mplexid != 32767 AND "
            "      atsc_minor_chan > 0  "
            "GROUP BY mplexid";

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare(thequery);
        if (!query.exec())
        {
            QString msg =
                QString("DB Error (Performing database upgrade): \n"
                        "Query was: %1 \nError was: %2 \nnew version: %3")
                .arg(thequery)
                .arg(MythDB::DBErrorMessage(query.lastError()))
                .arg("1170");
            VERBOSE(VB_IMPORTANT, msg);
            return false;
        }

        QString in = "(";

        while (query.next())
            in += query.value(0).toString() + ",";

        thequery = "";
        if (in.length() > 2)
        {
            in.truncate(in.length() - 1);
            thequery =
                "UPDATE dtv_multiplex "
                "SET sistandard='atsc' "
                "WHERE mplexid IN " + in + ")";
        }

        const char *updates[] = {
thequery.toAscii().constData(),
NULL
};

        if (!performActualUpdate(updates, "1170", dbver))
            return false;
    }

    if (dbver == "1170")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS storagegroup ( "
"    id           INT(11) NOT NULL auto_increment, "
"    groupname    VARCHAR(32) NOT NULL, "
"    hostname     VARCHAR(64) NOT NULL DEFAULT '', "
"    dirname      VARCHAR(235) NOT NULL DEFAULT '', "
"    PRIMARY KEY (id), "
"    UNIQUE KEY grouphostdir (groupname, hostname, dirname)"
"    );",
"INSERT storagegroup (groupname, hostname, dirname) "
"    SELECT DISTINCT 'Default', hostname, data "
"    FROM settings "
"    WHERE value = 'RecordFilePrefix' "
"        AND hostname IS NOT NULL "
"        AND hostname <> '';",
"ALTER TABLE recorded ADD storagegroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE record   ADD storagegroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE inuseprograms ADD rechost VARCHAR(64) NOT NULL;",
"ALTER TABLE inuseprograms ADD recdir VARCHAR(255) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1171", dbver))
            return false;
    }

    if (dbver == "1171")
    {
        // Add Firewire and DBox2 default recording profiles..
        const char *updates[] = {
"INSERT INTO recordingprofiles SET name = \"Default\",      profilegroup = 7;",
"INSERT INTO recordingprofiles SET name = \"Live TV\",      profilegroup = 7;",
"INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = 7;",
"INSERT INTO recordingprofiles SET name = \"Low Quality\",  profilegroup = 7;",
"INSERT INTO recordingprofiles SET name = \"Default\",      profilegroup = 9;",
"INSERT INTO recordingprofiles SET name = \"Live TV\",      profilegroup = 9;",
"INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = 9;",
"INSERT INTO recordingprofiles SET name = \"Low Quality\",  profilegroup = 9;",
NULL
};
        if (!performActualUpdate(updates, "1172", dbver))
            return false;
    }

    if (dbver == "1172")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD COLUMN last_record DATETIME NOT NULL;",
NULL
};
        if (!performActualUpdate(updates, "1173", dbver))
            return false;
    }

    if (dbver == "1173")
    {
        const char *updates[] = {
"UPDATE settings SET data = CONCAT(data, \" --settime $time\") "
"WHERE value = \"MythShutdownNvramCmd\";",
NULL
};
        if (!performActualUpdate(updates, "1174", dbver))
            return false;
    }

    if (dbver == "1174")
    {
        const char *updates[] = {
"UPDATE record SET dupin = 31 WHERE dupin = 4;",
NULL
};
        if (!performActualUpdate(updates, "1175", dbver))
            return false;
    }

    if (dbver == "1175")
    {
        QString thequery =
            "SELECT cardid "
            "FROM capturecard "
            "WHERE cardtype='FIREWIRE'";

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare(thequery);
        if (!query.exec())
        {
            QString msg =
                QString("DB Error (Performing database upgrade): \n"
                        "Query was: %1 \nError was: %2 \nnew version: %3")
                .arg(thequery)
                .arg(MythDB::DBErrorMessage(query.lastError()))
                .arg("1176");
            VERBOSE(VB_IMPORTANT, msg);
            return false;
        }

        QString in = "(";

        while (query.next())
            in += query.value(0).toString() + ",";

        thequery = "";
        if (in.length() > 2)
        {
            in.truncate(in.length() - 1);
            thequery =
                "DELETE FROM cardinput "
                "WHERE cardid IN " + in + ")";
        }

        const char *updates[] = {
"DELETE FROM capturecard WHERE cardtype = 'FIREWIRE';",
thequery.toAscii().constData(),
NULL
};
        if (!performActualUpdate(updates, "1176", dbver))
            return false;
    }

    if (dbver == "1176")
    {
        const char *updates[] = {
"ALTER TABLE cardinput ADD COLUMN quicktune TINYINT NOT NULL default '0';",
NULL
};
        if (!performActualUpdate(updates, "1177", dbver))
            return false;
    }

    if (dbver == "1177")
    {
        const char *updates[] = {
"ALTER TABLE channel           ADD INDEX (sourceid, xmltvid, chanid);",
"ALTER TABLE channel           ADD INDEX (visible);",
"ALTER TABLE inuseprograms     ADD INDEX (recusage, lastupdatetime);",
"ALTER TABLE mythlog           ADD INDEX (module);",
"ALTER TABLE profilegroups     ADD INDEX (cardtype);",
"ALTER TABLE program           DROP INDEX programid;",
"ALTER TABLE program           ADD INDEX (previouslyshown);",
"ALTER TABLE program           ADD INDEX (programid,starttime);",
"ALTER TABLE program           ADD INDEX (starttime);",
"ALTER TABLE record            ADD INDEX (maxepisodes);",
"ALTER TABLE record            ADD INDEX (search);",
"ALTER TABLE record            ADD INDEX (type);",
"ALTER TABLE recorded          ADD INDEX (deletepending, lastmodified);",
"ALTER TABLE recorded          ADD INDEX (recgroup, endtime);",
"ALTER TABLE recordingprofiles ADD INDEX (profilegroup);",
"ALTER TABLE storagegroup      ADD INDEX (hostname);",
NULL
};
        if (!performActualUpdate(updates, "1178", dbver))
            return false;
    }

    if (dbver == "1178")
    {
        const char *updates[] = {
"ALTER TABLE program         CHANGE seriesid  seriesid  VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE program         CHANGE programid programid VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE record          CHANGE seriesid  seriesid  VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE record          CHANGE programid programid VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE recorded        CHANGE seriesid  seriesid  VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE recorded        CHANGE programid programid VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE recordedprogram CHANGE seriesid  seriesid  VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE recordedprogram CHANGE programid programid VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded     CHANGE seriesid  seriesid  VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded     CHANGE programid programid VARCHAR(40) NOT NULL DEFAULT '';",
"ALTER TABLE channel ADD COLUMN default_authority VARCHAR(32) NOT NULL DEFAULT '';",
NULL
};
        if (!performActualUpdate(updates, "1179", dbver))
            return false;
    }

    if (dbver == "1179")
    {
        const char *updates[] = {
"ALTER TABLE eit_cache ADD COLUMN status TINYINT NOT NULL default '0';",
"ALTER TABLE eit_cache DROP PRIMARY KEY, ADD PRIMARY KEY (chanid,eventid,status);",
NULL
};
        if (!performActualUpdate(updates, "1180", dbver))
            return false;
    }

    if (dbver == "1180")
    {
        const char *updates[] = {
"ALTER TABLE recordedmarkup MODIFY mark MEDIUMINT UNSIGNED NOT NULL DEFAULT 0, "
                           "MODIFY type TINYINT NOT NULL DEFAULT 0;",
"ALTER TABLE recordedseek MODIFY mark MEDIUMINT UNSIGNED NOT NULL DEFAULT 0,"
                         "MODIFY offset BIGINT UNSIGNED NOT NULL,"
                         "MODIFY type TINYINT NOT NULL DEFAULT 0;",
NULL
};
        if (!performActualUpdate(updates, "1181", dbver))
            return false;
    }

    if (dbver == "1181")
    {
        VERBOSE(VB_IMPORTANT, "Upgrading to MythTV schema version 1182");

        MSqlQuery airdates(MSqlQuery::InitCon());
        airdates.prepare("SELECT chanid, starttime FROM recordedprogram "
                         "WHERE originalairdate = '0000-00-00';");
        if (!airdates.exec())
            return false;

        if (airdates.isActive() && airdates.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (airdates.next())
            {
                update.prepare("UPDATE recorded "
                               "SET originalairdate = '0000-00-00' "
                               "WHERE chanid = :CHANID "
                               "AND starttime = :STARTTIME;");

                update.bindValue(":CHANID", airdates.value(0).toString());
                update.bindValue(":STARTTIME", airdates.value(1).toDateTime());
                if (!update.exec())
                    MythDB::DBError("dbcheck -- DisplayGroupDefaultViewMask",
                                    update);
            }
        }

        if (!UpdateDBVersionNumber("1182"))
            return false;

        dbver = "1182";
    }

    if (dbver == "1182")
    {
        const char *updates[] = {
"ALTER TABLE jobqueue ADD schedruntime datetime NOT NULL default '2007-01-01 00:00:00';",
NULL
};
        if (!performActualUpdate(updates, "1183", dbver))
            return false;
    }

    if (dbver == "1183")
    {
        const char *updates[] = {
"ALTER TABLE channel ADD COLUMN commmethod INT NOT NULL default '-1';",
"UPDATE channel SET commmethod = -2 WHERE commfree = 1;",
NULL
};
        if (!performActualUpdate(updates, "1184", dbver))
            return false;
    }

    if (dbver == "1184")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS powerpriority ("
"  priorityname VARCHAR(64) NOT NULL PRIMARY KEY, "
"  recpriority int(10) NOT NULL default '0',"
"  selectclause text NOT NULL "
");",
NULL
};
        if (!performActualUpdate(updates, "1185", dbver))
            return false;
    }

    if (dbver == "1185")
    {
        VERBOSE(VB_IMPORTANT, "Upgrading to MythTV schema version 1186");

        MSqlQuery ppuq(MSqlQuery::InitCon());

        int oncepriority = gContext->GetNumSetting("OnceRecPriority", 0);
        int ccpriority   = gContext->GetNumSetting("CCRecPriority", 0);

        if (oncepriority)
        {
            ppuq.prepare("INSERT INTO powerpriority "
                         "(priorityname, recpriority, selectclause) "
                         "VALUES('Priority When Shown Once', :VALUE, "
                         "'program.first > 0 AND program.last > 0');");
            ppuq.bindValue(":VALUE", oncepriority);
            if (!ppuq.exec())
                MythDB::DBError("dbcheck -- OnceRecPriority", ppuq);
        }

        if (ccpriority)
        {
            ppuq.prepare("INSERT INTO powerpriority "
                         "(priorityname, recpriority, selectclause) "
                         "VALUES('Close Captioned Priority', :VALUE, "
                         "'program.closecaptioned > 0');");
            ppuq.bindValue(":VALUE", ccpriority);
            if (!ppuq.exec())
                MythDB::DBError("dbcheck -- CCRecPriority", ppuq);
        }

        if (!UpdateDBVersionNumber("1186"))
            return false;

        dbver = "1186";
    }

    if (dbver == "1186")
    {
        const char *updates[] = {
"ALTER TABLE eit_cache MODIFY eventid INT UNSIGNED NOT NULL default 0;",
NULL
};
        if (!performActualUpdate(updates, "1187", dbver))
            return false;
    }

    if (dbver == "1187")
    {
        const char *updates[] = {
"UPDATE keybindings SET keylist=REPLACE(keylist, '\\\\\\\\', '\\\\');",
"UPDATE keybindings SET keylist=REPLACE(keylist, '\\\\\\\"', '\"');",
"UPDATE jumppoints SET keylist=REPLACE(keylist, '\\\\\\\\', '\\\\');",
"UPDATE jumppoints SET keylist=REPLACE(keylist, '\\\\\\\"', '\"');",
NULL
};
        if (!performActualUpdate(updates, "1188", dbver))
            return false;
    }

    if (dbver == "1188")
    {
        const char *updates[] = {
"ALTER TABLE customexample ADD COLUMN search TINYINT NOT NULL default '0';",
"REPLACE INTO customexample VALUES('New Flix', '', "
"'program.category_type = ''movie'' AND program.airdate >= \n"
"     YEAR(DATE_SUB(NOW(), INTERVAL 1 YEAR)) \n"
"AND program.stars > 0.5 ', 1);",
NULL
};
        if (!performActualUpdate(updates, "1189", dbver))
            return false;
    }

    if (dbver == "1189")
    {
        const char *updates[] = {
"CREATE TEMPORARY TABLE tmpcodecparams AS "
    "SELECT * FROM codecparams WHERE name = 'mpeg4scalebitrate';",
"UPDATE tmpcodecparams SET name = 'scalebitrate';",
"INSERT codecparams SELECT * FROM tmpcodecparams;",
"DROP TABLE tmpcodecparams;",
NULL
};
        if (!performActualUpdate(updates, "1190", dbver))
            return false;
    }

    if (dbver == "1190")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS recordedfile;",
"CREATE TABLE recordedfile ("
"    chanid                  INT(10) UNSIGNED  NOT NULL DEFAULT 0,"
"    starttime               DATETIME          NOT NULL DEFAULT '0000-00-00 00:00:00',"
"    basename                VARCHAR(128)      NOT NULL DEFAULT '',"
"    filesize                BIGINT(20)        NOT NULL DEFAULT 0,"
"    width                   SMALLINT UNSIGNED NOT NULL DEFAULT 0,"
"    height                  SMALLINT UNSIGNED NOT NULL DEFAULT 0,"
"    fps                     FLOAT(6,3)        NOT NULL DEFAULT 0,"
"    aspect                  FLOAT(8,6)        NOT NULL DEFAULT 0,"
"    audio_sample_rate       SMALLINT UNSIGNED NOT NULL DEFAULT 0,"
"    audio_bits_per_sample   SMALLINT UNSIGNED NOT NULL DEFAULT 0,"
"    audio_channels          TINYINT UNSIGNED  NOT NULL DEFAULT 0,"
"    audio_type              VARCHAR(255)      NOT NULL DEFAULT '',"
"    video_type              VARCHAR(255)      NOT NULL DEFAULT '',"
"    comment                 VARCHAR(255)      NOT NULL DEFAULT '',"
"    PRIMARY KEY (chanid, starttime, basename),"
"    INDEX       (basename)"
");",
NULL
};
        if (!performActualUpdate(updates, "1191", dbver))
            return false;
    }

    if (dbver == "1191")
    {
        const char *updates[] = {
"DROP TABLE IF EXISTS displayprofilegroups;",
"DROP TABLE IF EXISTS displayprofiles;",
"CREATE TABLE IF NOT EXISTS displayprofilegroups ("
"  name varchar(128) NOT NULL,"
"  hostname  varchar(64) NOT NULL,"
"  profilegroupid int(10) unsigned NOT NULL auto_increment,"
"  PRIMARY KEY nameid (name,hostname),"
"  UNIQUE KEY (profilegroupid)"
");",
"CREATE TABLE IF NOT EXISTS displayprofiles ("
"  profilegroupid int(10) unsigned NOT NULL,"
"  profileid int(10) unsigned NOT NULL auto_increment,"
"  value varchar(128) NOT NULL,"
"  data  varchar(255) NOT NULL default '',"
"  FOREIGN KEY (profilegroupid) "
"    REFERENCES displayprofilegroups(profilegroupid),"
"  KEY (profileid,value),"
"  INDEX (profileid)"
");",
NULL
};
       if (!performActualUpdate(updates, "1192", dbver))
            return false;
    }

    if (dbver == "1192")
    {
        const char *updates[] = {
"ALTER TABLE record ADD COLUMN avg_delay INT NOT NULL default 100;",
NULL
};
        if (!performActualUpdate(updates, "1193", dbver))
            return false;
    }

    if (dbver == "1193")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS inputgroup ("
"  cardinputid int(10) unsigned NOT NULL, "
"  inputgroupid int(10) unsigned NOT NULL, "
"  inputgroupname varchar(32) NOT NULL "
");",
NULL
};
        if (!performActualUpdate(updates, "1194", dbver))
            return false;
    }

    if (dbver == "1194")
    {
        const char *updates[] = {
"UPDATE settings SET data='G.A.N.T' WHERE value='Theme' AND data='G.A.N.T.';",
NULL
};
        if (!performActualUpdate(updates, "1195", dbver))
            return false;
    }

    if (dbver == "1195")
    {
        const char *updates[] = {
"UPDATE settings SET data=REPLACE(data, '--no-delete', '')"
" WHERE value='MythFillDatabaseArgs';",
NULL
};
        if (!performActualUpdate(updates, "1196", dbver))
            return false;
    }

    if (dbver == "1196")
    {
        const char *updates[] = {
"UPDATE videosource "
"SET xmltvgrabber='schedulesdirect1', userid='', password=NULL "
"WHERE xmltvgrabber='datadirect' OR xmltvgrabber='technovera';",
NULL
};
        if (!performActualUpdate(updates, "1197", dbver))
            return false;
    }

    if (dbver == "1197")
    {
        QString thequery =
            "SELECT cardid "
            "FROM capturecard "
            "WHERE cardtype='HDTV'";

        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare(thequery);
        if (!query.exec())
        {
            QString msg =
                QString("DB Error (Performing database upgrade): \n"
                        "Query was: %1 \nError was: %2 \nnew version: %3")
                .arg(thequery)
                .arg(MythDB::DBErrorMessage(query.lastError()))
                .arg("1198");
            VERBOSE(VB_IMPORTANT, msg);
            return false;
        }

        QString in = "(";

        while (query.next())
            in += query.value(0).toString() + ",";

        thequery = "";
        if (in.length() > 2)
        {
            in.truncate(in.length() - 1);
            thequery =
                "DELETE FROM cardinput "
                "WHERE cardid IN " + in + ")";
        }

        const char *updates[] = {
"DELETE FROM capturecard WHERE cardtype = 'HDTV';",
thequery.toAscii().constData(),
NULL
};
        if (!performActualUpdate(updates, "1198", dbver))
            return false;
    }

    if (dbver == "1198")
    {
        const char *updates[] = {
"DELETE FROM keybindings "
"WHERE context = 'TV PLAYBACK' AND action = 'TOGGLEASPECT';",
NULL
};
        if (!performActualUpdate(updates, "1199", dbver))
            return false;
    }

    if (dbver == "1199")
    {
        const char *updates[] = {
"UPDATE codecparams SET value = 1 "
"WHERE name='mpeg4maxquality' AND value = 0;",
NULL
};
        if (!performActualUpdate(updates, "1200", dbver))
            return false;
    }

    if (dbver == "1200")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT id "
                      "FROM recordingprofiles "
                      "WHERE audiocodec='MPEG-2 Hardware Encoder'");
        QString in = "";
        if (query.exec())
        {
            while (query.next())
                in += "'" + query.value(0).toString() + "',";
        }
        in = (in.isEmpty()) ? "'5','6','7','8'" : in.left(in.length()-1);
        QString qtmp = QString(
            "UPDATE codecparams "
            "SET value=48000 "
            "WHERE name='samplerate' AND profile IN (%1);").arg(in);
        QByteArray tmp = qtmp.toAscii();
        const char *updates[] = {
            tmp.constData(),
            NULL
        };

        if (!performActualUpdate(updates, "1201", dbver))
            return false;
    }

    if (dbver == "1201")
    {
        const char *updates[] = {
"ALTER TABLE recordmatch ADD oldrecduplicate TINYINT(1); ",
"ALTER TABLE recordmatch ADD recduplicate TINYINT(1); ",
"ALTER TABLE recordmatch ADD findduplicate TINYINT(1); ",
"ALTER TABLE recordmatch ADD oldrecstatus INT(11); ",
NULL
};
        if (!performActualUpdate(updates, "1202", dbver))
            return false;
    }

    if (dbver == "1202")
    {
        const char *updates[] = {
"CREATE TABLE upnpmedia ("
"intid int(10) unsigned NOT NULL default '0',"
"class varchar(64) NOT NULL default '',"
"itemtype varchar(128) NOT NULL default '',"
"parentid int(10) unsigned NOT NULL default '0',"
"itemproperties varchar(255) NOT NULL default '',"
"filepath varchar(512) NOT NULL default '',"
"title varchar(255) NOT NULL default '',"
"filename varchar(512) NOT NULL default '',"
"PRIMARY KEY  (intid),"
"KEY class (class),"
"KEY filepath (filepath),"
"KEY parentid (parentid)"
");",
NULL
};
       if (!performActualUpdate(updates, "1203", dbver))
           return false;
    }

    if (dbver == "1203")
    {
        const char *updates[] = {
"UPDATE keybindings SET keylist = '[,{,F10,Volume Down' "
"WHERE action = 'VOLUMEDOWN' AND keylist = '[,{,F10';",
"UPDATE keybindings SET keylist = '],},F11,Volume Up' "
"WHERE action = 'VOLUMEUP' AND keylist = '],},F11';",
"UPDATE keybindings SET keylist = '|,\\\\,F9,Volume Mute' "
"WHERE action = 'MUTE' AND keylist = '|,\\\\,F9';",
NULL
};
       if (!performActualUpdate(updates, "1204", dbver))
           return false;
    }

    if (dbver == "1204")
    {
        const char *updates[] = {
"ALTER TABLE upnpmedia ADD coverart varchar(512) NOT NULL default ''; ",
NULL
};
    if (!performActualUpdate(updates, "1205", dbver))
            return false;
    }

    // Until we merge to the trunk this must be done manually...
    if (dbver == "1205")
    {
        const char *updates[] = {
"DELETE FROM keybindings "
"WHERE context = 'TV PLAYBACK' AND "
"      (action = 'TOGGLEINPUTS' OR action = 'SWITCHCARDS');",
NULL
};
        if (!performActualUpdate(updates, "1206", dbver))
            return false;
    }

    if (dbver == "1206")
    {
        const char *updates[] = {
"ALTER TABLE capturecard DROP parentid;",
"ALTER TABLE cardinput DROP childcardid;",
NULL
};
        if (!performActualUpdate(updates, "1207", dbver))
            return false;
    }

    if (dbver == "1207")
    {
        const char *updates[] = {
"ALTER TABLE recorded MODIFY basename VARCHAR(255) NOT NULL;",
NULL
};
        if (!performActualUpdate(updates, "1208", dbver))
            return false;
    }

    if (dbver == "1208")
    {
        const char *updates[] = {
"ALTER TABLE capturecard ADD dvb_eitscan tinyint(1) NOT NULL default '1'; ",
NULL
};
        if (!performActualUpdate(updates, "1209", dbver))
            return false;
    }

    if (dbver == "1209")
    {
        const char *updates[] = {
"ALTER TABLE program ADD audioprop     tinyint(3) unsigned NOT NULL; ",
"ALTER TABLE program ADD subtitletypes tinyint(3) unsigned NOT NULL; ",
"ALTER TABLE program ADD videoprop     tinyint(3) unsigned NOT NULL; ",
"ALTER TABLE recordedprogram ADD audioprop     tinyint(3) unsigned NOT NULL; ",
"ALTER TABLE recordedprogram ADD subtitletypes tinyint(3) unsigned NOT NULL; ",
"ALTER TABLE recordedprogram ADD videoprop     tinyint(3) unsigned NOT NULL; ",
"UPDATE program SET audioprop     = stereo;",
"UPDATE program SET subtitletypes = closecaptioned | (subtitled << 1);",
"UPDATE program SET videoprop     = hdtv;",
"UPDATE recordedprogram SET audioprop     = stereo;",
"UPDATE recordedprogram SET subtitletypes = closecaptioned | (subtitled << 1);",
"UPDATE recordedprogram SET videoprop     = hdtv;",
NULL
};
        if (!performActualUpdate(updates, "1210", dbver))
            return false;
    }

    if (dbver == "1210")
    {
        const char *updates[] = {
"ALTER TABLE program CHANGE COLUMN audioprop audioprop "
"    SET('STEREO', 'MONO', 'SURROUND', 'DOLBY', 'HARDHEAR', 'VISUALIMPAIR') NOT NULL; ",
"ALTER TABLE program CHANGE COLUMN videoprop videoprop "
"    SET('HDTV', 'WIDESCREEN', 'AVC') NOT NULL; ",
"ALTER TABLE program CHANGE COLUMN subtitletypes subtitletypes "
"    SET('HARDHEAR', 'NORMAL', 'ONSCREEN', 'SIGNED') NOT NULL; ",

"ALTER TABLE recordedprogram CHANGE COLUMN audioprop audioprop "
"    SET('STEREO', 'MONO', 'SURROUND', 'DOLBY', 'HARDHEAR', 'VISUALIMPAIR') NOT NULL; ",
"ALTER TABLE recordedprogram CHANGE COLUMN videoprop videoprop "
"    SET('HDTV', 'WIDESCREEN', 'AVC') NOT NULL; ",
"ALTER TABLE recordedprogram CHANGE COLUMN subtitletypes subtitletypes "
"    SET('HARDHEAR', 'NORMAL', 'ONSCREEN', 'SIGNED') NOT NULL; ",
NULL
};
        if (!performActualUpdate(updates, "1211", dbver))
            return false;
    }

    if (dbver == "1211")
    {
        const char *updates[] = {
"CREATE TEMPORARY TABLE tmpdisplayprofiles AS "
"    SELECT * FROM displayprofiles WHERE value = 'pref_osdfade';",
"UPDATE tmpdisplayprofiles SET value = 'pref_max_cpus', data = '1';",
"DELETE FROM tmpdisplayprofiles WHERE profileid IN "
"    (SELECT profileid FROM displayprofiles WHERE value = 'pref_max_cpus');",
"INSERT displayprofiles SELECT * FROM tmpdisplayprofiles;",
"DROP TABLE tmpdisplayprofiles;",
NULL
};

        if (!performActualUpdate(updates, "1212", dbver))
        return false;
    }

    if (dbver == "1212")
    {
        const char *updates[] = {
"DELETE FROM keybindings WHERE action LIKE 'MENU%' AND context='TV Playback';",
"DELETE FROM keybindings WHERE action='TEXTEXIT' AND context='TV Playback';",
"DELETE FROM keybindings WHERE action='SIGNALMON' AND context='TV Playback' "
"   AND keylist='F7';",
"UPDATE keybindings SET context='TV Playback' WHERE context='ITV Menu';",
NULL
};

        if (!performActualUpdate(updates, "1213", dbver))
        return false;
    }

    if (dbver == "1213")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT hostname "
                      "FROM displayprofilegroups "
                      "GROUP BY hostname");
        bool ok = query.exec();
        while (ok && query.next())
        {
            QString host = query.value(0).toString();
            QStringList profiles = VideoDisplayProfile::GetProfiles(host);
            if (profiles.contains("High Quality") &&
                profiles.contains("Normal") &&
                profiles.contains("Slim"))
            {
                continue;
            }

            VideoDisplayProfile::CreateNewProfiles(host);
            profiles = VideoDisplayProfile::GetProfiles(host);
            QString profile = VideoDisplayProfile::GetDefaultProfileName(host);

            if (profiles.contains("Normal") &&
                (profile=="CPU++" || profile=="CPU+" || profile=="CPU--"))
            {
                VideoDisplayProfile::SetDefaultProfileName("Normal", host);
            }
        }

        const char *updates[] = { NULL };
        if (!performActualUpdate(updates, "1214", dbver))
            return false;
    }

    if (dbver == "1214")
    {
        const char *updates[] = {
"UPDATE capturecard SET audioratelimit = NULL;",
NULL
};
        if (!performActualUpdate(updates, "1215", dbver))
            return false;
    }

    if (dbver == "1215")
    {
        // Check for corruption before performing the DB charset conversion
        // (Some system configurations have been converting the UTF8 data Myth
        // writes to latin1)
        MSqlQuery query(MSqlQuery::InitCon());
        int tableIndex      = 0;
        QString tables[][3] = {
            { "people",    // table name
              "name",      // columns to convert, colon-separated
              "name(41)"}, // columns for test index, comma-separated
            { "oldprogram",
              "oldtitle",
              "oldtitle"},
            { "oldrecorded",
              "title:subtitle:description",
              "station, starttime, title"},
            { "recorded",
              "title:subtitle:description",
              ""},
            { "",
              "",
              ""} // This blank entry must exist, do not remove.
        };

        while (!tables[tableIndex][0].isEmpty())
        {
            bool    ok          = true;
            QString table       = tables[tableIndex][0];
            QStringList columns = tables[tableIndex][1].split(':');
            QString index_cols  = tables[tableIndex][2];
            QString thequery    = QString("CREATE TEMPORARY TABLE temp_%1 "
                                          "SELECT * FROM %2;")
                                          .arg(table).arg(table);
            QString modify;
            ok &= query.exec(thequery);
            if (ok)
            {
                thequery = QString("ALTER TABLE temp_%1 ").arg(table);
                QStringList::const_iterator column;
                for (column = columns.constBegin();
                     column != columns.constEnd(); ++column)
                {
                    if ("description" == *column)
                        modify = " MODIFY %1 BLOB NOT NULL,";
                    else
                        modify = " MODIFY %1 VARBINARY(255)"
                                 "    NOT NULL default '',";
                    thequery.append(modify.arg(*column));
                }
                thequery.chop(1);
                thequery.append(";");
                ok &= query.exec(thequery);
            }
            if (ok)
            {
                thequery = QString("ALTER TABLE temp_%1 ").arg(table);
                QStringList::const_iterator column;
                for (column = columns.constBegin();
                     column != columns.constEnd(); ++column)
                {
                    if ("description" == *column)
                        modify = " MODIFY %1 TEXT"
                                 "           CHARACTER SET utf8 "
                                 "           COLLATE utf8_bin ";
                    else
                        modify = " MODIFY %1 CHAR(255)"
                                 "           CHARACTER SET utf8 "
                                 "           COLLATE utf8_bin "
                                 "    NOT NULL default '',";
                    thequery.append(modify.arg(*column));
                }
                thequery.chop(1);
                thequery.append(";");
                ok &= query.exec(thequery);
            }
            if (!ok)
            {
                MythDB::DBError(QString("Unable to perform test for database "
                                "corruption before character set conversion."),
                                thequery);
                return false;
            }
            // If the conversion to utf8 resulted in warnings, the data in the
            // database is not in utf8 format/is corrupt
            thequery = "SHOW COUNT(*) WARNINGS;";
            if (query.exec(thequery) && query.isActive() &&
                query.size() > 0 && query.next())
            {
                int warnings = query.value(0).toInt();
                if (warnings)
                {
                    QString msg = QString("Database corruption detected. "
                                          "Unable to proceed with database "
                                          "upgrade. (Table: %1, Warnings: %2)")
                                          .arg(table).arg(warnings);
                    VERBOSE(VB_IMPORTANT, msg);
                    VERBOSE(VB_IMPORTANT, "Your database must be fixed before "
                            "you can upgrade beyond 0.21-fixes. Please see "
                            "http://www.mythtv.org/wiki/"
                            "Fixing_Corrupt_Database_Encoding for information "
                            "on fixing your database.");
                    return false;
                }
            }
            else
            {
                MythDB::DBError(QString("Error getting database warnings for "
                                "database corruption test."),
                                thequery);
                return false;
            }
            // Test creating an index to see if we had partial corruption that
            // results in duplicates after condensing to UTF-8
            if (!index_cols.isEmpty())
            {
                thequery = QString("ALTER TABLE temp_%1 "
                                   "  ADD UNIQUE INDEX verify_unique ( %2 );")
                                   .arg(table).arg(index_cols);
                ok &= query.exec(thequery);
                if (!ok)
                {
                    MythDB::DBError(QString("Index creation failed."),
                                    thequery);
                    VERBOSE(VB_IMPORTANT, "DB charset pre-conversion test "
                            "failed! Your database seems to be partially "
                            "corrupted. Please move the backup to a safe "
                            "place. Your database must be fixed before you "
                            "can upgrade beyond 0.21-fixes. Please see "
                            "http://www.mythtv.org/wiki/Fixing_"
                            "Corrupt_Database_Encoding for information.");
                    return false;
                }
            }
            thequery = QString("DROP TEMPORARY TABLE temp_%1;").arg(table);
            if (!query.exec(thequery))
                MythDB::DBError(QString("Error dropping temporary table %1.")
                                .arg(table), thequery);

            tableIndex++;
        }

        // Perform the actual upgrade
        QString qtmp = QString(
            "ALTER DATABASE %1 DEFAULT CHARACTER SET latin1;")
            .arg(gContext->GetDatabaseParams().dbName);
        QByteArray tmp = qtmp.toAscii();
        const char *updates[] = {
tmp.constData(),
"ALTER TABLE callsignnetworkmap"
"  MODIFY callsign varbinary(20) NOT NULL default '',"
"  MODIFY network varbinary(20) NOT NULL default '';",
"ALTER TABLE capturecard"
"  MODIFY videodevice varbinary(128) default NULL,"
"  MODIFY audiodevice varbinary(128) default NULL,"
"  MODIFY vbidevice varbinary(128) default NULL,"
"  MODIFY cardtype varbinary(32) default 'V4L',"
"  MODIFY defaultinput varbinary(32) default 'Television',"
"  MODIFY hostname varbinary(64) default NULL,"
"  MODIFY firewire_model varbinary(32) default NULL,"
"  MODIFY dbox2_host varbinary(32) default NULL;",
"ALTER TABLE cardinput"
"  MODIFY inputname varbinary(32) NOT NULL default '',"
"  MODIFY externalcommand varbinary(128) default NULL,"
"  MODIFY shareable binary(1) default 'N',"
"  MODIFY tunechan varbinary(10) default NULL,"
"  MODIFY startchan varbinary(10) default NULL,"
"  MODIFY displayname varbinary(64) NOT NULL default '';",
"ALTER TABLE channel"
"  MODIFY channum varbinary(10) NOT NULL default '',"
"  MODIFY freqid varbinary(10) default NULL,"
"  MODIFY callsign varbinary(20) NOT NULL default '',"
"  MODIFY name varbinary(64) NOT NULL default '',"
"  MODIFY icon varbinary(255) NOT NULL default 'none',"
"  MODIFY videofilters varbinary(255) NOT NULL default '',"
"  MODIFY xmltvid varbinary(64) NOT NULL default '',"
"  MODIFY tvformat varbinary(10) NOT NULL default 'Default',"
"  MODIFY outputfilters varbinary(255) NOT NULL default '',"
"  MODIFY default_authority varbinary(32) NOT NULL default '';",
"ALTER TABLE codecparams"
"  MODIFY name varbinary(128) NOT NULL default '',"
"  MODIFY value varbinary(128) default NULL;",
"ALTER TABLE customexample"
"  MODIFY rulename varbinary(64) NOT NULL,"
"  MODIFY fromclause blob NOT NULL,"
"  MODIFY whereclause blob NOT NULL;",
"ALTER TABLE diseqc_config"
"  MODIFY value varbinary(16) NOT NULL default '';",
"ALTER TABLE diseqc_tree"
"  MODIFY type varbinary(16) NOT NULL default '',"
"  MODIFY subtype varbinary(16) NOT NULL default '',"
"  MODIFY description varbinary(32) NOT NULL default '',"
"  MODIFY rotor_positions varbinary(255) NOT NULL default '';",
"ALTER TABLE displayprofilegroups"
"  MODIFY name varbinary(128) NOT NULL,"
"  MODIFY hostname varbinary(64) NOT NULL;",
"ALTER TABLE displayprofiles"
"  MODIFY value varbinary(128) NOT NULL,"
"  MODIFY data varbinary(255) NOT NULL default '';",
"ALTER TABLE dtv_multiplex"
"  MODIFY inversion binary(1) default 'a',"
"  MODIFY fec varbinary(10) default 'auto',"
"  MODIFY polarity binary(1) default NULL,"
"  MODIFY modulation varbinary(10) default 'auto',"
"  MODIFY bandwidth binary(1) default 'a',"
"  MODIFY lp_code_rate varbinary(10) default 'auto',"
"  MODIFY transmission_mode binary(1) default 'a',"
"  MODIFY guard_interval varbinary(10) default 'auto',"
"  MODIFY constellation varbinary(10) default 'auto',"
"  MODIFY hierarchy varbinary(10) default 'auto',"
"  MODIFY hp_code_rate varbinary(10) default 'auto',"
"  MODIFY sistandard varbinary(10) default 'dvb';",
"ALTER TABLE dtv_privatetypes"
"  MODIFY sitype varbinary(4) NOT NULL default '',"
"  MODIFY private_type varbinary(20) NOT NULL default '',"
"  MODIFY private_value varbinary(100) NOT NULL default '';",
"ALTER TABLE dvdbookmark"
"  MODIFY serialid varbinary(16) NOT NULL default '',"
"  MODIFY name varbinary(32) default NULL;",
"ALTER TABLE housekeeping"
"  MODIFY tag varbinary(64) NOT NULL default '';",
"ALTER TABLE inputgroup"
"  MODIFY inputgroupname varbinary(32) NOT NULL;",
"ALTER TABLE inuseprograms"
"  MODIFY recusage varbinary(128) NOT NULL default '',"
"  MODIFY hostname varbinary(64) NOT NULL default '',"
"  MODIFY rechost varbinary(64) NOT NULL,"
"  MODIFY recdir varbinary(255) NOT NULL default '';",
"ALTER TABLE jobqueue"
"  MODIFY hostname varbinary(64) NOT NULL default '',"
"  MODIFY comment varbinary(128) NOT NULL default '';",
"ALTER TABLE jumppoints"
"  MODIFY destination varbinary(128) NOT NULL default '',"
"  MODIFY description varbinary(255) default NULL,"
"  MODIFY keylist varbinary(128) default NULL,"
"  MODIFY hostname varbinary(64) NOT NULL default '';",
"ALTER TABLE keybindings"
"  MODIFY context varbinary(32) NOT NULL default '',"
"  MODIFY action varbinary(32) NOT NULL default '',"
"  MODIFY description varbinary(255) default NULL,"
"  MODIFY keylist varbinary(128) default NULL,"
"  MODIFY hostname varbinary(64) NOT NULL default '';",
"ALTER TABLE keyword"
"  MODIFY phrase varbinary(128) NOT NULL default '';",
"ALTER TABLE mythlog"
"  MODIFY module varbinary(32) NOT NULL default '',"
"  MODIFY host varbinary(128) default NULL,"
"  MODIFY message varbinary(255) NOT NULL default '',"
"  MODIFY details blob;",
"ALTER TABLE networkiconmap"
"  MODIFY network varbinary(20) NOT NULL default '',"
"  MODIFY url varbinary(255) NOT NULL default '';",
"ALTER TABLE oldprogram"
"  MODIFY oldtitle varbinary(128) NOT NULL default '';",
"ALTER TABLE oldrecorded"
"  MODIFY title varbinary(128) NOT NULL default '',"
"  MODIFY subtitle varbinary(128) NOT NULL default '',"
"  MODIFY description blob NOT NULL,"
"  MODIFY category varbinary(64) NOT NULL default '',"
"  MODIFY seriesid varbinary(40) NOT NULL default '',"
"  MODIFY programid varbinary(40) NOT NULL default '',"
"  MODIFY station varbinary(20) NOT NULL default '';",
"ALTER TABLE people"
"  MODIFY name varbinary(128) NOT NULL default '';",
"ALTER TABLE playgroup"
"  MODIFY name varbinary(32) NOT NULL default '',"
"  MODIFY titlematch varbinary(255) NOT NULL default '';",
"ALTER TABLE powerpriority"
"  MODIFY priorityname varbinary(64) NOT NULL,"
"  MODIFY selectclause blob NOT NULL;",
"ALTER TABLE profilegroups"
"  MODIFY name varbinary(128) default NULL,"
"  MODIFY cardtype varbinary(32) NOT NULL default 'V4L',"
"  MODIFY hostname varbinary(64) default NULL;",
"ALTER TABLE program"
"  MODIFY title varbinary(128) NOT NULL default '',"
"  MODIFY subtitle varbinary(128) NOT NULL default '',"
"  MODIFY description blob NOT NULL,"
"  MODIFY category varbinary(64) NOT NULL default '',"
"  MODIFY category_type varbinary(64) NOT NULL default '',"
"  MODIFY title_pronounce varbinary(128) NOT NULL default '',"
"  MODIFY seriesid varbinary(40) NOT NULL default '',"
"  MODIFY showtype varbinary(30) NOT NULL default '',"
"  MODIFY colorcode varbinary(20) NOT NULL default '',"
"  MODIFY syndicatedepisodenumber varbinary(20) NOT NULL default '',"
"  MODIFY programid varbinary(40) NOT NULL default '';",
"ALTER TABLE programgenres"
"  MODIFY relevance binary(1) NOT NULL default '',"
"  MODIFY genre varbinary(30) default NULL;",
"ALTER TABLE programrating"
"  MODIFY system varbinary(8) NOT NULL default '',"
"  MODIFY rating varbinary(16) default NULL;",
"ALTER TABLE recgrouppassword"
"  MODIFY recgroup varbinary(32) NOT NULL default '',"
"  MODIFY password varbinary(10) NOT NULL default '';",
"ALTER TABLE record"
"  MODIFY title varbinary(128) NOT NULL default '',"
"  MODIFY subtitle varbinary(128) NOT NULL default '',"
"  MODIFY description blob NOT NULL,"
"  MODIFY category varbinary(64) NOT NULL default '',"
"  MODIFY profile varbinary(128) NOT NULL default 'Default',"
"  MODIFY recgroup varbinary(32) NOT NULL default 'Default',"
"  MODIFY station varbinary(20) NOT NULL default '',"
"  MODIFY seriesid varbinary(40) NOT NULL default '',"
"  MODIFY programid varbinary(40) NOT NULL default '',"
"  MODIFY playgroup varbinary(32) NOT NULL default 'Default',"
"  MODIFY storagegroup varbinary(32) NOT NULL default 'Default';",
"ALTER TABLE recorded"
"  MODIFY title varbinary(128) NOT NULL default '',"
"  MODIFY subtitle varbinary(128) NOT NULL default '',"
"  MODIFY description blob NOT NULL,"
"  MODIFY category varbinary(64) NOT NULL default '',"
"  MODIFY hostname varbinary(64) NOT NULL default '',"
"  MODIFY recgroup varbinary(32) NOT NULL default 'Default',"
"  MODIFY seriesid varbinary(40) NOT NULL default '',"
"  MODIFY programid varbinary(40) NOT NULL default '',"
"  MODIFY basename varbinary(255) NOT NULL,"
"  MODIFY playgroup varbinary(32) NOT NULL default 'Default',"
"  MODIFY profile varbinary(32) NOT NULL default '',"
"  MODIFY storagegroup varbinary(32) NOT NULL default 'Default';",
"ALTER TABLE recordedfile"
"  MODIFY basename varbinary(128) NOT NULL default '',"
"  MODIFY audio_type varbinary(255) NOT NULL default '',"
"  MODIFY video_type varbinary(255) NOT NULL default '',"
"  MODIFY comment varbinary(255) NOT NULL default '';",
"ALTER TABLE recordedmarkup"
"  MODIFY offset varbinary(32) default NULL;",
"ALTER TABLE recordedprogram"
"  MODIFY title varbinary(128) NOT NULL default '',"
"  MODIFY subtitle varbinary(128) NOT NULL default '',"
"  MODIFY description blob NOT NULL,"
"  MODIFY category varbinary(64) NOT NULL default '',"
"  MODIFY category_type varbinary(64) NOT NULL default '',"
"  MODIFY title_pronounce varbinary(128) NOT NULL default '',"
"  MODIFY seriesid varbinary(40) NOT NULL default '',"
"  MODIFY showtype varbinary(30) NOT NULL default '',"
"  MODIFY colorcode varbinary(20) NOT NULL default '',"
"  MODIFY syndicatedepisodenumber varbinary(20) NOT NULL default '',"
"  MODIFY programid varbinary(40) NOT NULL default '';",
"ALTER TABLE recordedrating"
"  MODIFY system varbinary(8) NOT NULL default '',"
"  MODIFY rating varbinary(16) default NULL;",
"ALTER TABLE recordingprofiles"
"  MODIFY name varbinary(128) default NULL,"
"  MODIFY videocodec varbinary(128) default NULL,"
"  MODIFY audiocodec varbinary(128) default NULL;",
"ALTER TABLE settings"
"  MODIFY value varbinary(128) NOT NULL default '',"
"  MODIFY data blob,"
"  MODIFY hostname varbinary(64) default NULL;",
"ALTER TABLE storagegroup"
"  MODIFY groupname varbinary(32) NOT NULL,"
"  MODIFY hostname varbinary(64) NOT NULL default '',"
"  MODIFY dirname varbinary(235) NOT NULL default '';",
"ALTER TABLE tvchain"
"  MODIFY chainid varbinary(128) NOT NULL default '',"
"  MODIFY hostprefix varbinary(128) NOT NULL default '',"
"  MODIFY cardtype varbinary(32) NOT NULL default 'V4L',"
"  MODIFY input varbinary(32) NOT NULL default '',"
"  MODIFY channame varbinary(32) NOT NULL default '';",
"ALTER TABLE upnpmedia"
"  MODIFY class varbinary(64) NOT NULL default '',"
"  MODIFY itemtype varbinary(128) NOT NULL default '',"
"  MODIFY itemproperties varbinary(255) NOT NULL default '',"
"  MODIFY filepath varbinary(512) NOT NULL default '',"
"  MODIFY title varbinary(255) NOT NULL default '',"
"  MODIFY filename varbinary(512) NOT NULL default '',"
"  MODIFY coverart varbinary(512) NOT NULL default '';",
"ALTER TABLE videosource"
"  MODIFY name varbinary(128) NOT NULL default '',"
"  MODIFY xmltvgrabber varbinary(128) default NULL,"
"  MODIFY userid varbinary(128) NOT NULL default '',"
"  MODIFY freqtable varbinary(16) NOT NULL default 'default',"
"  MODIFY lineupid varbinary(64) default NULL,"
"  MODIFY password varbinary(64) default NULL;",
NULL
};

        if (!performActualUpdate(updates, "1216", dbver))
        {
            VERBOSE(VB_IMPORTANT, "DB charset conversions update failed! "
                    "Your database seems to be partially corrupted. Please "
                    "move the backup to a safe place. Your database must be "
                    "fixed before you can upgrade beyond 0.21-fixes. Please "
                    "see http://www.mythtv.org/wiki/Fixing_Corrupt_"
                    "Database_Encoding for information.");
            return false;
        }
    }

    if (dbver == "1216")
    {
        QString qtmp = QString(
            "ALTER DATABASE %1 DEFAULT CHARACTER SET "
            "utf8 COLLATE utf8_general_ci;")
            .arg(gContext->GetDatabaseParams().dbName);
        QByteArray tmp = qtmp.toAscii();
        const char *updates[] = {
tmp.constData(),
"ALTER TABLE callsignnetworkmap"
"  DEFAULT CHARACTER SET default,"
"  MODIFY callsign varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY network varchar(20) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE capturecard"
"  DEFAULT CHARACTER SET default,"
"  MODIFY videodevice varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY audiodevice varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY vbidevice varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY cardtype varchar(32) CHARACTER SET utf8 default 'V4L',"
"  MODIFY defaultinput varchar(32) CHARACTER SET utf8 default 'Television',"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 default NULL,"
"  MODIFY firewire_model varchar(32) CHARACTER SET utf8 default NULL,"
"  MODIFY dbox2_host varchar(32) CHARACTER SET utf8 default NULL;",
"ALTER TABLE cardinput"
"  DEFAULT CHARACTER SET default,"
"  MODIFY inputname varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY externalcommand varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY shareable char(1) CHARACTER SET utf8 default 'N',"
"  MODIFY tunechan varchar(10) CHARACTER SET utf8 default NULL,"
"  MODIFY startchan varchar(10) CHARACTER SET utf8 default NULL,"
"  MODIFY displayname varchar(64) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE channel"
"  DEFAULT CHARACTER SET default,"
"  MODIFY channum varchar(10) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY freqid varchar(10) CHARACTER SET utf8 default NULL,"
"  MODIFY callsign varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY name varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY icon varchar(255) CHARACTER SET utf8 NOT NULL default 'none',"
"  MODIFY videofilters varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY xmltvid varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY tvformat varchar(10) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY outputfilters varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY default_authority varchar(32) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE codecparams"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY value varchar(128) CHARACTER SET utf8 default NULL;",
"ALTER TABLE credits"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE customexample"
"  DEFAULT CHARACTER SET default,"
"  MODIFY rulename varchar(64) CHARACTER SET utf8 NOT NULL,"
"  MODIFY fromclause text CHARACTER SET utf8 NOT NULL,"
"  MODIFY whereclause text CHARACTER SET utf8 NOT NULL;",
"ALTER TABLE diseqc_config"
"  DEFAULT CHARACTER SET default,"
"  MODIFY value varchar(16) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE diseqc_tree"
"  DEFAULT CHARACTER SET default,"
"  MODIFY type varchar(16) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtype varchar(16) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY rotor_positions varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE displayprofilegroups"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(128) CHARACTER SET utf8 NOT NULL,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL;",
"ALTER TABLE displayprofiles"
"  DEFAULT CHARACTER SET default,"
"  MODIFY value varchar(128) CHARACTER SET utf8 NOT NULL,"
"  MODIFY data varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE dtv_multiplex"
"  DEFAULT CHARACTER SET default,"
"  MODIFY inversion char(1) CHARACTER SET utf8 default 'a',"
"  MODIFY fec varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY polarity char(1) CHARACTER SET utf8 default NULL,"
"  MODIFY modulation varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY bandwidth char(1) CHARACTER SET utf8 default 'a',"
"  MODIFY lp_code_rate varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY transmission_mode char(1) CHARACTER SET utf8 default 'a',"
"  MODIFY guard_interval varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY constellation varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY hierarchy varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY hp_code_rate varchar(10) CHARACTER SET utf8 default 'auto',"
"  MODIFY sistandard varchar(10) CHARACTER SET utf8 default 'dvb';",
"ALTER TABLE dtv_privatetypes"
"  DEFAULT CHARACTER SET default,"
"  MODIFY sitype varchar(4) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY private_type varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY private_value varchar(100) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE dvdbookmark"
"  DEFAULT CHARACTER SET default,"
"  MODIFY serialid varchar(16) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY name varchar(32) CHARACTER SET utf8 default NULL;",
"ALTER TABLE eit_cache"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE favorites"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE housekeeping"
"  DEFAULT CHARACTER SET default,"
"  MODIFY tag varchar(64) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE inputgroup"
"  DEFAULT CHARACTER SET default,"
"  MODIFY inputgroupname varchar(32) CHARACTER SET utf8 NOT NULL;",
"ALTER TABLE inuseprograms"
"  DEFAULT CHARACTER SET default,"
"  MODIFY recusage varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY rechost varchar(64) CHARACTER SET utf8 NOT NULL,"
"  MODIFY recdir varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE jobqueue"
"  DEFAULT CHARACTER SET default,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY comment varchar(128) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE jumppoints"
"  DEFAULT CHARACTER SET default,"
"  MODIFY destination varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description varchar(255) CHARACTER SET utf8 default NULL,"
"  MODIFY keylist varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE keybindings"
"  DEFAULT CHARACTER SET default,"
"  MODIFY context varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY action varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description varchar(255) CHARACTER SET utf8 default NULL,"
"  MODIFY keylist varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE keyword"
"  DEFAULT CHARACTER SET default,"
"  MODIFY phrase varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '';",
"ALTER TABLE mythlog"
"  DEFAULT CHARACTER SET default,"
"  MODIFY module varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY host varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY message varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY details text CHARACTER SET utf8;",
"ALTER TABLE networkiconmap"
"  DEFAULT CHARACTER SET default,"
"  MODIFY network varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY url varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE oldfind"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE oldprogram"
"  DEFAULT CHARACTER SET default,"
"  MODIFY oldtitle varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '';",
"ALTER TABLE oldrecorded"
"  DEFAULT CHARACTER SET default,"
"  MODIFY title varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtitle varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description text CHARACTER SET utf8 NOT NULL,"
"  MODIFY category varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY seriesid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY programid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY station varchar(20) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE people"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name char(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '';",
"ALTER TABLE pidcache"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE playgroup"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY titlematch varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE powerpriority"
"  DEFAULT CHARACTER SET default,"
"  MODIFY priorityname varchar(64) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL,"
"  MODIFY selectclause text CHARACTER SET utf8 NOT NULL;",
"ALTER TABLE profilegroups"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY cardtype varchar(32) CHARACTER SET utf8 NOT NULL default 'V4L',"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 default NULL;",
"ALTER TABLE program"
"  DEFAULT CHARACTER SET default,"
"  MODIFY title varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtitle varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description text CHARACTER SET utf8 NOT NULL,"
"  MODIFY category varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY category_type varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY title_pronounce varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY seriesid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY showtype varchar(30) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY colorcode varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY syndicatedepisodenumber varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY programid varchar(40) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE programgenres"
"  DEFAULT CHARACTER SET default,"
"  MODIFY relevance char(1) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY genre char(30) CHARACTER SET utf8 default NULL;",
"ALTER TABLE programrating"
"  DEFAULT CHARACTER SET default,"
"  MODIFY system char(8) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY rating char(16) CHARACTER SET utf8 default NULL;",
"ALTER TABLE recgrouppassword"
"  DEFAULT CHARACTER SET default,"
"  MODIFY recgroup varchar(32) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '',"
"  MODIFY password varchar(10) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE record"
"  DEFAULT CHARACTER SET default,"
"  MODIFY title varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtitle varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description text CHARACTER SET utf8 NOT NULL,"
"  MODIFY category varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY profile varchar(128) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY recgroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY station varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY seriesid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY programid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY playgroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY storagegroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default';",
"ALTER TABLE recorded"
"  DEFAULT CHARACTER SET default,"
"  MODIFY title varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtitle varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description text CHARACTER SET utf8 NOT NULL,"
"  MODIFY category varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY recgroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY seriesid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY programid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY basename varchar(255) CHARACTER SET utf8 NOT NULL,"
"  MODIFY playgroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default',"
"  MODIFY profile varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY storagegroup varchar(32) CHARACTER SET utf8 NOT NULL default 'Default';",
"ALTER TABLE recordedcredits"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE recordedfile"
"  DEFAULT CHARACTER SET default,"
"  MODIFY basename varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY audio_type varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY video_type varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY comment varchar(255) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE recordedmarkup"
"  DEFAULT CHARACTER SET default,"
"  MODIFY offset varchar(32) CHARACTER SET utf8 default NULL;",
"ALTER TABLE recordedprogram"
"  DEFAULT CHARACTER SET default,"
"  MODIFY title varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY subtitle varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY description text CHARACTER SET utf8 NOT NULL,"
"  MODIFY category varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY category_type varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY title_pronounce varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY seriesid varchar(40) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY showtype varchar(30) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY colorcode varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY syndicatedepisodenumber varchar(20) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY programid varchar(40) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE recordedrating"
"  DEFAULT CHARACTER SET default,"
"  MODIFY system char(8) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY rating char(16) CHARACTER SET utf8 default NULL;",
"ALTER TABLE recordedseek"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE recordingprofiles"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY videocodec varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY audiocodec varchar(128) CHARACTER SET utf8 default NULL;",
"ALTER TABLE recordmatch"
"  DEFAULT CHARACTER SET default;",
"ALTER TABLE settings"
"  DEFAULT CHARACTER SET default,"
"  MODIFY value varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY data text CHARACTER SET utf8,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 default NULL;",
"ALTER TABLE storagegroup"
"  DEFAULT CHARACTER SET default,"
"  MODIFY groupname varchar(32) CHARACTER SET utf8 NOT NULL,"
"  MODIFY hostname varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY dirname varchar(235) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '';",
"ALTER TABLE tvchain"
"  DEFAULT CHARACTER SET default,"
"  MODIFY chainid varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY hostprefix varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY cardtype varchar(32) CHARACTER SET utf8 NOT NULL default 'V4L',"
"  MODIFY input varchar(32) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY channame varchar(32) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE upnpmedia"
"  DEFAULT CHARACTER SET default,"
"  MODIFY class varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY itemtype varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY itemproperties varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY filepath varchar(512) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY title varchar(255) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY filename varchar(512) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY coverart varchar(512) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE videosource"
"  DEFAULT CHARACTER SET default,"
"  MODIFY name varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY xmltvgrabber varchar(128) CHARACTER SET utf8 default NULL,"
"  MODIFY userid varchar(128) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY freqtable varchar(16) CHARACTER SET utf8 NOT NULL default 'default',"
"  MODIFY lineupid varchar(64) CHARACTER SET utf8 default NULL,"
"  MODIFY password varchar(64) CHARACTER SET utf8 default NULL;",
NULL
};

        if (!performActualUpdate(updates, "1217", dbver))
        {
            VERBOSE(VB_IMPORTANT, "DB charset conversions update failed! "
                    "Your database seems to be partially corrupted. Please "
                    "move the backup to a safe place. Your database must be "
                    "fixed before you can upgrade beyond 0.21-fixes. Please "
                    "see http://www.mythtv.org/wiki/Fixing_Corrupt_"
                    "Database_Encoding for information.");
            return false;
        }
    }

    if (dbver == "1217")
    {
        const char *updates[] = {
            "DROP TABLE IF EXISTS videobookmarks;",
            NULL
        };

        if (!performActualUpdate(updates, "1218", dbver))
            return false;
    }

    if (dbver == "1218")
    {
        const char *updates[] = {
"ALTER TABLE cardinput"
"  DROP COLUMN preference,"
"  DROP COLUMN diseqc_port,"
"  DROP COLUMN diseqc_pos,"
"  DROP COLUMN lnb_lof_switch,"
"  DROP COLUMN lnb_lof_hi,"
"  DROP COLUMN lnb_lof_lo;",
"ALTER TABLE capturecard"
"  DROP COLUMN firewire_port,"
"  DROP COLUMN firewire_node;",
"ALTER TABLE channel"
"  DROP COLUMN atscsrcid,"
"  DROP COLUMN commfree;",
"ALTER TABLE recordedmarkup DROP COLUMN offset;",
NULL
};

        if (!performActualUpdate(updates, "1219", dbver))
            return false;
    }

    if (dbver == "1219")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS channelscan ("
"  scanid       int(3)           UNSIGNED NOT NULL AUTO_INCREMENT,"
"  cardid       int(3)           UNSIGNED NOT NULL,"
"  sourceid     int(3)           UNSIGNED NOT NULL,"
"  processed    tinyint(1)       UNSIGNED NOT NULL,"
"  scandate     datetime                  NOT NULL,"
"  PRIMARY KEY (scanid)"
");",
"CREATE TABLE IF NOT EXISTS channelscan_dtv_multiplex ("
"  transportid  int(6)           UNSIGNED NOT NULL AUTO_INCREMENT,"
"  scanid       int(3)           UNSIGNED NOT NULL,"
"  mplexid      smallint(6)      UNSIGNED NOT NULL,"
"  frequency    bigint(12)       UNSIGNED NOT NULL,"
"  inversion    char(1)                   NOT NULL default 'a',"
"  symbolrate   bigint(12)       UNSIGNED NOT NULL default '0',"
"  fec          varchar(4)                NOT NULL default 'auto',"
"  polarity     char(1)                   NOT NULL default ' ',"
"  hp_code_rate varchar(4)                NOT NULL default 'auto',"
"  lp_code_rate varchar(4)                NOT NULL default 'auto',"
"  modulation   varchar(4)                NOT NULL default 'auto',"
"  transmission_mode char(1)              NOT NULL default 'a',"
"  guard_interval varchar(4)              NOT NULL default 'auto',"
"  hierarchy    varchar(4)                NOT NULL default 'auto',"
"  bandwidth    char(1)                   NOT NULL default 'a',"
"  sistandard   varchar(10)               NOT NULL,"
"  tuner_type   smallint(2)      UNSIGNED NOT NULL,"
"  PRIMARY KEY (transportid)"
");",
"CREATE TABLE IF NOT EXISTS channelscan_channel ("
"  transportid  int(6)           UNSIGNED NOT NULL,"
"  scanid       int(3)           UNSIGNED NOT NULL,"
"  mplex_id     smallint(6)               NOT NULL,"
"  source_id    int(3)           UNSIGNED NOT NULL,"
"  channel_id   int(3)           UNSIGNED NOT NULL default '0',"
"  callsign     varchar(20)               NOT NULL default '',"
"  service_name varchar(64)               NOT NULL default '',"
"  chan_num     varchar(10)               NOT NULL default '',"
"  service_id   mediumint(8)     UNSIGNED NOT NULL default '0',"
"  atsc_major_channel int(4)     UNSIGNED NOT NULL default '0',"
"  atsc_minor_channel int(4)     UNSIGNED NOT NULL default '0',"
"  use_on_air_guide tinyint(1)            NOT NULL default '0',"
"  hidden       tinyint(1)                NOT NULL default '0',"
"  hidden_in_guide tinyint(1)             NOT NULL default '0',"
"  freqid       varchar(10)               NOT NULL default '',"
"  icon         varchar(255)              NOT NULL default '',"
"  tvformat     varchar(10)               NOT NULL default 'Default',"
"  xmltvid      varchar(64)               NOT NULL default '',"
"  pat_tsid     int(5)           UNSIGNED NOT NULL default '0',"
"  vct_tsid     int(5)           UNSIGNED NOT NULL default '0',"
"  vct_chan_tsid int(5)          UNSIGNED NOT NULL default '0',"
"  sdt_tsid     int(5)           UNSIGNED NOT NULL default '0',"
"  orig_netid   int(5)           UNSIGNED NOT NULL default '0',"
"  netid        int(5)           UNSIGNED NOT NULL default '0',"
"  si_standard  varchar(10)               NOT NULL,"
"  in_channels_conf tinyint(1)   UNSIGNED NOT NULL default '0',"
"  in_pat       tinyint(1)       UNSIGNED NOT NULL default '0',"
"  in_pmt       tinyint(1)       UNSIGNED NOT NULL default '0',"
"  in_vct       tinyint(1)       UNSIGNED NOT NULL default '0',"
"  in_nit       tinyint(1)       UNSIGNED NOT NULL default '0',"
"  in_sdt       tinyint(1)       UNSIGNED NOT NULL default '0',"
"  is_encrypted tinyint(1)       UNSIGNED NOT NULL default '0',"
"  is_data_service tinyint(1)    UNSIGNED NOT NULL default '0',"
"  is_audio_service tinyint(1)   UNSIGNED NOT NULL default '0',"
"  is_opencable tinyint(1)       UNSIGNED NOT NULL default '0',"
"  could_be_opencable tinyint(1) UNSIGNED NOT NULL default '0',"
"  decryption_status smallint(2) UNSIGNED NOT NULL default '0'"
");",
NULL
};
        if (!performActualUpdate(updates, "1220", dbver))
            return false;
    }

    if (dbver == "1220")
    {
        const char *updates[] = {
"UPDATE settings SET data='7' "
"  WHERE value='CommercialSkipMethod' AND data='255'",
"UPDATE settings SET data='261' "
"  WHERE value='CommercialSkipMethod' AND data='511'",
"UPDATE channel SET commmethod='7' WHERE commmethod='255'",
"UPDATE channel SET commmethod='261' WHERE commmethod='511'",
NULL
};
        if (!performActualUpdate(updates, "1221", dbver))
            return false;
    }

    if (dbver == "1221")
    {
        bool ok = true;
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare(
            "SELECT cardid, videodevice "
            "FROM capturecard "
            "WHERE cardtype = 'DVB'");

        if (!query.exec())
        {
            MythDB::DBError(
                "Could not perform select for update to '1222'", query);
            ok = false;
        }
        else
        {
            MSqlQuery query2(MSqlQuery::InitCon());
            while (query.next())
            {
                bool conv_ok;
                uint dvbnum = query.value(1).toString().toUInt(&conv_ok);
                if (!conv_ok)
                    continue;

                QString videodevice =
                    QString("/dev/dvb/adapter%1/frontend0").arg(dvbnum);

                query2.prepare(
                    "UPDATE capturecard "
                    "SET videodevice = :VIDEODEV "
                    "WHERE cardid = :CARDID");
                query2.bindValue(":VIDEODEV", videodevice);
                query2.bindValue(":CARDID",   query.value(0).toUInt());
                if (!query2.exec())
                {
                    MythDB::DBError(
                        "Could not perform update for '1222'", query2);
                    ok = false;
                }
            }
        }

        if (!ok)
            return false;

        const char *updates[] = { NULL };
        if (!performActualUpdate(updates, "1222", dbver))
            return false;
    }

    if (dbver == "1222")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = 'HD-PVR Recorders', "
"  cardtype = 'HDPVR', is_default = 1;",
"INSERT INTO recordingprofiles SET name = 'Default',      profilegroup = 13;",
"INSERT INTO recordingprofiles SET name = 'Live TV',      profilegroup = 13;",
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 13;",
"INSERT INTO recordingprofiles SET name = 'Low Quality',  profilegroup = 13;",
NULL
};
        if (!performActualUpdate(updates, "1223", dbver))
            return false;
    }

    if (dbver == "1223")
    {
        const char *updates[] = {
"ALTER TABLE recordedmarkup ADD COLUMN data INT(11) UNSIGNED;",
NULL
};
        if (!performActualUpdate(updates, "1224", dbver))
            return false;
    }

    if (dbver == "1224")
    {
        // Fix data corruption caused by a data conversion issue in the
        // original implementation of the 1215-1216 database upgrade
        const char *updates[] = {
"CREATE TEMPORARY TABLE bad_people AS SELECT * FROM people;",
"CREATE TEMPORARY TABLE bad_credits AS SELECT * FROM credits;",
"CREATE TEMPORARY TABLE bad_recordedcredits AS SELECT * FROM recordedcredits;",
"ALTER TABLE bad_people ADD INDEX (person);",
"TRUNCATE TABLE people;",
"TRUNCATE TABLE credits;",
"TRUNCATE TABLE recordedcredits;",
"UPDATE bad_people SET NAME = TRIM(TRAILING 0x00 FROM name);",
"INSERT people ( name ) SELECT DISTINCT name FROM bad_people;",
"INSERT credits "
"    SELECT p.person, bc.chanid, bc.starttime, bc.role "
"        FROM people p, bad_people bp, bad_credits bc "
"        WHERE p.name = bp.name "
"            AND bp.person = bc.person;",
"INSERT recordedcredits "
"    SELECT p.person, br.chanid, br.starttime, br.role "
"        FROM people p, bad_people bp, bad_recordedcredits br "
"        WHERE p.name = bp.name "
"            AND bp.person = br.person;",
"DROP TEMPORARY TABLE bad_people;",
"DROP TEMPORARY TABLE bad_credits;",
"DROP TEMPORARY TABLE bad_recordedcredits;",
"UPDATE programgenres SET genre = TRIM(TRAILING 0x00 FROM genre);",
"REPLACE INTO programrating (chanid, starttime, system, rating) "
"  SELECT chanid, starttime, "
"         TRIM(TRAILING 0x00 FROM system), TRIM(TRAILING 0x00 FROM rating) "
"    FROM programrating;",
"DELETE FROM programrating WHERE system LIKE '%\\0' OR rating LIKE '%\\0';",
"REPLACE INTO recordedrating (chanid, starttime, system, rating) "
"  SELECT chanid, starttime, "
"         TRIM(TRAILING 0x00 FROM system), TRIM(TRAILING 0x00 FROM rating) "
"    FROM recordedrating;",
"DELETE FROM recordedrating WHERE system LIKE '%\\0' OR rating LIKE '%\\0';",
NULL
};
        if (!performActualUpdate(updates, "1225", dbver))
            return false;
    }

    if (dbver == "1225")
    {
        const char *updates[] = {
"ALTER TABLE people         CHANGE name   name   VARCHAR(128) "
"  CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '';",
"ALTER TABLE programgenres  CHANGE genre  genre  VARCHAR(30);",
"ALTER TABLE programrating  CHANGE rating rating VARCHAR(16);",
"ALTER TABLE programrating  CHANGE system system VARCHAR(8);",
"ALTER TABLE recordedrating CHANGE rating rating VARCHAR(16);",
"ALTER TABLE recordedrating CHANGE system system VARCHAR(8);",
NULL
};

        if (!performActualUpdate(updates, "1226", dbver))
            return false;
    }

    if (dbver == "1226")
    {
        const char *updates[] = {
"ALTER TABLE diseqc_tree ADD COLUMN address "
"   TINYINT(3) UNSIGNED NOT NULL default 0;",
"UPDATE diseqc_tree SET address = 16 WHERE type = 'switch';",
NULL
};
        if (!performActualUpdate(updates, "1227", dbver))
            return false;
    }

    if (dbver == "1227")
    {
        const char *updates[] = {
"DELETE FROM keybindings WHERE action='CREATEPIPVIEW' AND keylist='1';",
"DELETE FROM keybindings WHERE action='CREATEPBPVIEW' AND keylist='Ctrl+1';",
NULL
};
        if (!performActualUpdate(updates, "1228", dbver))
            return false;
    }

    if (dbver == "1228")
    {
        const char *updates[] = {
"ALTER TABLE dtv_multiplex ADD modulation_system TINYINT(1) DEFAULT 1"
    "    AFTER hp_code_rate;",
"ALTER TABLE dtv_multiplex ADD roll_off VARCHAR(4) NULL DEFAULT NULL"
    "    AFTER modulation_system;",
NULL
};
        if (!performActualUpdate(updates, "1229", dbver))
            return false;
    }

    if (dbver == "1229")
    {
        const char *updates[] = {
"CREATE TABLE IF NOT EXISTS tvosdmenu ("
"osdcategory VARCHAR(32) NOT NULL PRIMARY KEY,"
"livetv tinyint(4) NOT NULL default '0',"
"recorded tinyint(4) NOT NULL default '0',"
"video tinyint(4) NOT NULL default '0',"
"dvd tinyint(4) NOT NULL default '0',"
"description varchar(32) NOT NULL);",
NULL
};
        if (!performActualUpdate(updates, "1230", dbver))
            return false;
    }
    if (dbver == "1230")
    {
        const char *updates[] = {
// Content identfiers are authority (32) + data (29) + instance (3) = 64
"ALTER TABLE program"
"  MODIFY programid varchar(64) CHARACTER SET utf8 NOT NULL default '',"
"  MODIFY seriesid varchar(64) CHARACTER SET utf8 NOT NULL default '';",
// tidy up content identifiers without authorities
"UPDATE program SET programid = '' WHERE programid like '/%';",
"UPDATE program SET seriesid = '' WHERE seriesid like '/%';",
// Add network level default authorities to the multiplex tables
"ALTER TABLE dtv_multiplex ADD COLUMN"
"    default_authority varchar(32) CHARACTER SET utf8 NOT NULL default '';",
"ALTER TABLE channelscan_dtv_multiplex ADD COLUMN"
"    default_authority varchar(32) CHARACTER SET utf8 NOT NULL default '';",
NULL
};
        if (!performActualUpdate(updates, "1231", dbver))
            return false;
    }
    if (dbver == "1231")
    {
        const char *updates[] = {
"ALTER TABLE recordedprogram CHANGE COLUMN videoprop videoprop "
"    SET('HDTV', 'WIDESCREEN', 'AVC', '720', '1080') NOT NULL; ",
NULL
};
        if (!performActualUpdate(updates, "1232", dbver))
            return false;
    }

    if (dbver == "1232")
    {
        const char *updates[] = {
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN fec "
        "VARCHAR(10) NOT NULL DEFAULT 'auto';",
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN hp_code_rate "
    "VARCHAR(10) NOT NULL DEFAULT 'auto';",
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN lp_code_rate "
    "VARCHAR(10) NOT NULL DEFAULT 'auto';",
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN modulation "
    "VARCHAR(10) NOT NULL DEFAULT 'auto';",
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN guard_interval "
    "VARCHAR(10) NOT NULL DEFAULT 'auto';",
"ALTER TABLE channelscan_dtv_multiplex MODIFY COLUMN hierarchy "
    "VARCHAR(10) NOT NULL DEFAULT 'auto';",
NULL
};
        if (!performActualUpdate(updates, "1233", dbver))
            return false;
    }

    if (dbver == "1233")
    {
       const char *updates[] = {
"CREATE TABLE IF NOT EXISTS channelgroup ("
"  id int(10) unsigned NOT NULL auto_increment,"
"  chanid int(11) unsigned NOT NULL default '0',"
"  grpid int(11) NOT NULL default '1', PRIMARY KEY (id));",
"CREATE TABLE IF NOT EXISTS channelgroupnames ("
"  grpid int(10) unsigned NOT NULL auto_increment,"
"  name varchar(64) NOT NULL default '0',"
"  PRIMARY KEY (grpid));",
"INSERT INTO channelgroupnames (grpid, name) VALUES (1, 'Favorites');",
NULL
        };
        if (!performActualUpdate(updates, "1234", dbver))
            return false;

        // copy existing favorites to the new channel group
        MSqlQuery favorites(MSqlQuery::InitCon());
        favorites.prepare("SELECT DISTINCT chanid FROM favorites");
        if (favorites.exec() && favorites.isActive() && favorites.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (favorites.next())
            {
                update.prepare("INSERT INTO channelgroup (chanid, grpid) "
                               "VALUES (:CHANID, :GRPID);");
                update.bindValue(":CHANID", favorites.value(0).toInt());
                update.bindValue(":GRPID", 1);
                if (!update.exec())
                    MythDB::DBError("dbcheck -- channelgroup", update);
            }
        }

        // finally drop the favorites table
        favorites.prepare("DROP TABLE IF EXISTS favorites;");
        if (!favorites.exec())
            MythDB::DBError("dbcheck -- dropping favorites", favorites);
    }

    if (dbver == "1234")
    {
       const char *updates[] = {
"ALTER TABLE channelscan_channel ADD COLUMN"
"    default_authority varchar(32) CHARACTER SET utf8 NOT NULL default '';",
NULL
};
        if (!performActualUpdate(updates, "1235", dbver))
            return false;
    }

    if (dbver == "1235")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT cardid, videodevice, dbox2_port "
                      "FROM capturecard "
                      "WHERE cardtype = 'HDHOMERUN' "
                      "ORDER BY cardid");
        bool ok = query.exec();

        while (ok && query.next())
        {
            uint    cardid   = query.value(0).toUInt();
            QString device   = query.value(1).toString();
            uint    tunernum = query.value(2).toUInt();

            if (device.contains("-"))
                continue; // already in newstyle..

            device = device + "-" + QString::number(tunernum);

            MSqlQuery update(MSqlQuery::InitCon());
            update.prepare(
                "UPDATE capturecard "
                "SET videodevice = :DEVICE "
                "WHERE cardid = :CARDID");
            update.bindValue(":DEVICE", device);
            update.bindValue(":CARDID", cardid);

            if (!update.exec())
            {
                MythDB::DBError(
                    "Could not perform an update for '1236'", update);
                ok = false;
            }

        }

        if (!ok)
            return false;

        const char * updates[] = { NULL };

        if (!performActualUpdate(updates, "1236", dbver))
            return false;
    }

    if (dbver == "1236")
    {
       const char *updates[] = {
"UPDATE settings SET data= data | 0x8000 WHERE value='DisplayGroupDefaultViewMask';",
NULL
};
        if (!performActualUpdate(updates, "1237", dbver))
            return false;
    }

    if (dbver == "1237")
    {
        const char *updates[] = {
"ALTER TABLE videosource ADD COLUMN configpath VARCHAR(4096) DEFAULT NULL;", NULL
};

        if (!performActualUpdate(updates, "1238", dbver))
            return false;

        bool ok = true;
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SELECT name, sourceid FROM videosource");

        if (!query.exec())
        {
            MythDB::DBError(
                "Could not perform select for update to '1238'", query);
            ok = false;
        }
        else
        {
            MSqlQuery query2(MSqlQuery::InitCon());
            while (query.next())
            {
                QString videosource = query.value(0).toString();
                QString configpath =
                                gContext->GetSetting(QString("XMLTVConfig.%1")
                                .arg(videosource));

                if (!configpath.isEmpty())
                {
                    query2.prepare(
                        "UPDATE videosource "
                        "SET configpath = :PATH "
                        "WHERE sourceid = :SOURCEID ");
                    query2.bindValue(":PATH", configpath);
                    query2.bindValue(":SOURCEID", query.value(1).toString());
                    if (!query2.exec())
                    {
                         MythDB::DBError(
                            "Could not perform update for '1238'", query2);
                         ok = false;
                    }
                }
            }
        }
        if (!ok)
            return false;

        MSqlQuery query3(MSqlQuery::InitCon());
        query3.prepare("DELETE FROM settings WHERE value LIKE 'XMLTVConfig.%'");
        if (!query3.exec())
        {
            MythDB::DBError(
                "Could not perform update for '1238'", query3);
            ok = false;
        }

        if (!ok)
            return false;
    }


    if (dbver == "1238")
    {
       const char *updates[] = {
"ALTER TABLE dtv_multiplex CHANGE COLUMN modulation_system mod_sys VARCHAR(10) DEFAULT NULL;",
"ALTER TABLE dtv_multiplex CHANGE COLUMN roll_off rolloff VARCHAR(4) NULL DEFAULT NULL;",
"ALTER TABLE channelscan_dtv_multiplex ADD mod_sys VARCHAR(10) DEFAULT NULL"
    "    AFTER hp_code_rate;",
"ALTER TABLE channelscan_dtv_multiplex ADD rolloff VARCHAR(4) NULL DEFAULT NULL"
    "    AFTER mod_sys;",
NULL
};
        if (!performActualUpdate(updates, "1239", dbver))
            return false;
    }

    if (dbver == "1239")
    {
       const char *updates[] = {
"ALTER TABLE recordedfile ADD hostname VARCHAR(64) NOT NULL",
"ALTER TABLE recordedfile ADD storagegroup VARCHAR(32) NOT NULL",
NULL
};
        if (!performActualUpdate(updates, "1240", dbver))
            return false;
    }

    if (dbver == "1240")
    {
       const char *updates[] = {
"ALTER TABLE recordedfile DROP PRIMARY KEY;",
"ALTER TABLE recordedfile ADD COLUMN id int(11) NOT NULL AUTO_INCREMENT, ADD PRIMARY KEY (id);",
"ALTER TABLE recordedfile ADD UNIQUE (chanid, starttime, basename);",
NULL
};
        if (!performActualUpdate(updates, "1241", dbver))
            return false;
    }

    if (dbver == "1241")
    {
       const char *updates[] = {
"INSERT INTO settings ( value, data, hostname ) VALUES ( 'StorageScheduler', 'Combination', NULL );",
NULL
};
        if (!performActualUpdate(updates, "1242", dbver))
            return false;
    }

    if (dbver == "1242")
    {
       const char *updates[] = {
"DELETE FROM keybindings WHERE action = 'PAGETOP' and context = 'TV Frontend';",
"DELETE FROM keybindings WHERE action = 'PAGEBOTTOM' and context = 'TV Frontend';",
"DELETE FROM keybindings WHERE action = 'PAGEMIDDLE' and context = 'TV Frontend';",
NULL
};
        if (!performActualUpdate(updates, "1243", dbver))
            return false;
    }

    if (dbver == "1243")
    {
       const char *updates[] = {
"DELETE FROM keybindings WHERE action = 'DELETE' "
    "AND context IN ('qt', 'TV Frontend', 'TV Playback');",
"DELETE FROM keybindings WHERE action = 'DELETE' and context = 'qt';",
"DELETE FROM keybindings WHERE action = 'EDIT' and context = 'qt';",
"DELETE FROM keybindings WHERE action = 'TOGGLEEDIT' "
    "AND context IN ('TV Playback', 'TV Editing');",
NULL
};
        if (!performActualUpdate(updates, "1244", dbver))
            return false;
    }

    if (dbver == "1244")
    {
       const char *updates[] = {
"ALTER TABLE cardinput DROP COLUMN freetoaironly;",
"ALTER TABLE cardinput DROP COLUMN radioservices;",
NULL
};
        if (!performActualUpdate(updates, "1245", dbver))
            return false;
    }

    if (dbver == "1245")
    {
       const char *updates[] = {
"DELETE FROM capturecard WHERE cardtype = 'DBOX2';",
"DELETE FROM profilegroups WHERE cardtype = 'DBOX2';",
"ALTER TABLE capturecard DROP COLUMN dbox2_port;",
"ALTER TABLE capturecard DROP COLUMN dbox2_httpport;",
"ALTER TABLE capturecard DROP COLUMN dbox2_host;",
NULL
};
       if (!performActualUpdate(updates, "1246", dbver))
            return false;
    }

    if (dbver == "1246")
    {
       const char *updates[] = {
"ALTER TABLE recorded ADD COLUMN bookmarkupdate timestamp default 0 NOT NULL",
"UPDATE recorded SET bookmarkupdate = lastmodified+1 WHERE bookmark = 1",
"UPDATE recorded SET bookmarkupdate = lastmodified WHERE bookmark = 0",
NULL
};
       if (!performActualUpdate(updates, "1247", dbver))
            return false;
    }

    if (dbver == "1247")
    {
        const char *updates[] = {
"INSERT INTO profilegroups SET name = \"Import Recorder\", cardtype = 'IMPORT', is_default = 1;",
"INSERT INTO recordingprofiles SET name = \"Default\", profilegroup = 14;",
"INSERT INTO recordingprofiles SET name = \"Live TV\", profilegroup = 14;",
"INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = 14;",
"INSERT INTO recordingprofiles SET name = \"Low Quality\", profilegroup = 14;",
NULL
};
        if (!performActualUpdate(updates, "1248", dbver))
            return false;
    }

    if (dbver == "1248")
    {
       const char *updates[] = {
"DELETE FROM keybindings WHERE action = 'CUSTOMEDIT' "
   "AND context = 'TV Frontend' AND keylist = 'E';",
NULL
};
        if (!performActualUpdate(updates, "1249", dbver))
            return false;
    }

    if (dbver == "1249")
    {
        MSqlQuery select(MSqlQuery::InitCon());
        select.prepare("SELECT hostname, data FROM settings "
                       " WHERE value = 'StickyKeys'");

        if (!select.exec())
        {
            MythDB::DBError("Unable to retrieve StickyKeys values.", select);
        }
        else
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (select.next())
            {
                QString hostname = select.value(0).toString();
                QString sticky_keys = select.value(1).toString();

                if ("1" == sticky_keys)
                {
                    // Only remap the keys if they're currently set to defaults
                    update.prepare("UPDATE keybindings "
                                   "   SET keylist  = :KEYS "
                                   " WHERE context  = 'TV Playback' AND "
                                   "       action   = :ACTION AND "
                                   "       hostname = :HOSTNAME AND "
                                   "       keylist  = :DEFAULT_KEYS");

                    QString keylist = "";
                    QString action = "SEEKFFWD";
                    QString default_keys = "Right";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);

                    keylist = "";
                    action = "SEEKRWND";
                    default_keys = "Left";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);

                    keylist = ">,.,Right";
                    action = "FFWDSTICKY";
                    default_keys = ">,.";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);

                    keylist = ",,<,Left";
                    action = "RWNDSTICKY";
                    default_keys = ",,<";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);
                }
            }
        }

        if (!UpdateDBVersionNumber("1250"))
            return false;

        dbver = "1250";
    }

    if (dbver == "1250")
    {
       const char *updates[] = {
"UPDATE recorded SET bookmark = 1 WHERE bookmark != 0;",
NULL
};
        if (!performActualUpdate(updates, "1251", dbver))
            return false;
    }

    if (dbver == "1251")
    {
        MSqlQuery query(MSqlQuery::InitCon());
        query.prepare("SHOW INDEX FROM recgrouppassword");

        if (!query.exec())
        {
            MythDB::DBError("Unable to retrieve current indices on "
                            "recgrouppassword.", query);
        }
        else
        {
            while (query.next())
            {
                QString index_name = query.value(2).toString();

                if ("recgroup" == index_name)
                {
                    MSqlQuery update(MSqlQuery::InitCon());
                    update.prepare("ALTER TABLE recgrouppassword "
                                   " DROP INDEX recgroup");

                    if (!update.exec())
                         MythDB::DBError("Unable to drop duplicate index on "
                                         "recgrouppassword. Ignoring.",
                                         update);
                    break;
                }
            }
        }

        if (!UpdateDBVersionNumber("1252"))
            return false;

        dbver = "1252";
    }

    if (dbver == "1252")
    {
        MSqlQuery select(MSqlQuery::InitCon());
        select.prepare("SELECT hostname, data FROM settings "
                       " WHERE value = 'StickyKeys'");

        if (!select.exec())
        {
            MythDB::DBError("Unable to retrieve StickyKeys values.", select);
        }
        else
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (select.next())
            {
                QString hostname = select.value(0).toString();
                QString sticky_keys = select.value(1).toString();

                if ("1" == sticky_keys)
                {
                    // Only remap the keys if they're currently set to defaults
                    update.prepare("UPDATE keybindings "
                                   "   SET keylist  = :KEYS "
                                   " WHERE context  = 'TV Playback' AND "
                                   "       action   = :ACTION AND "
                                   "       hostname = :HOSTNAME AND "
                                   "       keylist  = :DEFAULT_KEYS");

                    QString keylist = ">,.";
                    QString action = "FFWDSTICKY";
                    QString default_keys = ">,.,Right";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);

                    keylist = ",,<";
                    action = "RWNDSTICKY";
                    default_keys = ",,<,Left";

                    update.bindValue(":KEYS", keylist);
                    update.bindValue(":ACTION", action);
                    update.bindValue(":HOSTNAME", hostname);
                    update.bindValue(":DEFAULT_KEYS", default_keys);
                    if (!update.exec())
                         MythDB::DBError("Unable to update keybindings",
                                         update);
                }
            }
        }

        if (!UpdateDBVersionNumber("1253"))
            return false;

        dbver = "1253";
    }

    if (dbver == "1253")
    {
        if (gContext->GetNumSetting("have-nit-fix") == 1)
        {
            // User has previously applied patch from ticket #7486.
            VERBOSE (VB_IMPORTANT, "Sneaky schema change detected");
            if (!UpdateDBVersionNumber("1254"))
                return false;
        }
        else
        {
            const char *updates[] = {
                "ALTER TABLE videosource ADD dvb_nit_id INT(6) DEFAULT -1;",
                NULL
            };
            if (!performActualUpdate(updates, "1254", dbver))
                return false;
        }
    }

    return true;
}

/**
 * command to get the the initial database layout from an empty database:
 *
 * mysqldump --skip-comments --skip-opt --compact --skip-quote-names \
 *     --ignore-table=schemalock mythconverg | \
 *   sed '/^SET.*;$/d;s/^.*[^;]$/"&"/;s/^);$/");",/'
 *
 * don't forget to add auto_increment annotations
 *
 * command to get the initial data:
 *
 * mysqldump --skip-comments --skip-opt --compact --skip-quote-names -t \
 *     mythconverg | sed 's/^.*$/"&",/'
 *
 * don't forget to delete host specific data
 *
 */
bool InitializeDatabase(void)
{
    MSqlQuery query(MSqlQuery::InitCon());
    query.prepare("SHOW TABLES;");

    // check for > 1 table here since the schemalock table should exist
    if (query.exec() && query.isActive() && query.size() > 1)
    {
        QString msg = QString(
            "Told to create a NEW database schema, but the database\n"
            "already has %1 tables.\n"
            "If you are sure this is a good mythtv database, verify\n"
            "that the settings table has the DBSchemaVer variable.\n")
            .arg(query.size() - 1);
        VERBOSE(VB_IMPORTANT, msg);
        return false;
    }

    VERBOSE(VB_IMPORTANT, "Inserting MythTV initial database information.");

    QString qtmp = QString("ALTER DATABASE %1 DEFAULT CHARACTER SET utf8;")
        .arg(gContext->GetDatabaseParams().dbName);
    QByteArray tmp = qtmp.toAscii();

    const char *updates[] = {
tmp.constData(),
"CREATE TABLE callsignnetworkmap ("
"  id int(11) NOT NULL AUTO_INCREMENT,"
"  callsign varchar(20) NOT NULL default '',"
"  network varchar(20) NOT NULL default '',"
"  PRIMARY KEY  (id),"
"  UNIQUE KEY callsign (callsign)"
");",
"CREATE TABLE capturecard ("
"  cardid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  videodevice varchar(128) default NULL,"
"  audiodevice varchar(128) default NULL,"
"  vbidevice varchar(128) default NULL,"
"  cardtype varchar(32) default 'V4L',"
"  defaultinput varchar(32) default 'Television',"
"  audioratelimit int(11) default NULL,"
"  hostname varchar(64) default NULL,"
"  dvb_swfilter int(11) default '0',"
"  dvb_sat_type int(11) NOT NULL default '0',"
"  dvb_wait_for_seqstart int(11) NOT NULL default '1',"
"  skipbtaudio tinyint(1) default '0',"
"  dvb_on_demand tinyint(4) NOT NULL default '0',"
"  dvb_diseqc_type smallint(6) default NULL,"
"  firewire_speed int(10) unsigned NOT NULL default '0',"
"  firewire_model varchar(32) default NULL,"
"  firewire_connection int(10) unsigned NOT NULL default '0',"
"  dbox2_port int(10) unsigned NOT NULL default '31338',"
"  dbox2_httpport int(10) unsigned NOT NULL default '80',"
"  dbox2_host varchar(32) default NULL,"
"  signal_timeout int(11) NOT NULL default '1000',"
"  channel_timeout int(11) NOT NULL default '3000',"
"  dvb_tuning_delay int(10) unsigned NOT NULL default '0',"
"  contrast int(11) NOT NULL default '0',"
"  brightness int(11) NOT NULL default '0',"
"  colour int(11) NOT NULL default '0',"
"  hue int(11) NOT NULL default '0',"
"  diseqcid int(10) unsigned default NULL,"
"  dvb_eitscan tinyint(1) NOT NULL default '1',"
"  PRIMARY KEY  (cardid)"
");",
"CREATE TABLE cardinput ("
"  cardinputid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  cardid int(10) unsigned NOT NULL default '0',"
"  sourceid int(10) unsigned NOT NULL default '0',"
"  inputname varchar(32) NOT NULL default '',"
"  externalcommand varchar(128) default NULL,"
"  shareable char(1) default 'N',"
"  tunechan varchar(10) default NULL,"
"  startchan varchar(10) default NULL,"
"  freetoaironly tinyint(1) default '1',"
"  displayname varchar(64) NOT NULL default '',"
"  radioservices tinyint(1) default '1',"
"  dishnet_eit tinyint(1) NOT NULL default '0',"
"  recpriority int(11) NOT NULL default '0',"
"  quicktune tinyint(4) NOT NULL default '0',"
"  PRIMARY KEY  (cardinputid)"
");",
"CREATE TABLE channel ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  channum varchar(10) NOT NULL default '',"
"  freqid varchar(10) default NULL,"
"  sourceid int(10) unsigned default NULL,"
"  callsign varchar(20) NOT NULL default '',"
"  `name` varchar(64) NOT NULL default '',"
"  icon varchar(255) NOT NULL default 'none',"
"  finetune int(11) default NULL,"
"  videofilters varchar(255) NOT NULL default '',"
"  xmltvid varchar(64) NOT NULL default '',"
"  recpriority int(10) NOT NULL default '0',"
"  contrast int(11) default '32768',"
"  brightness int(11) default '32768',"
"  colour int(11) default '32768',"
"  hue int(11) default '32768',"
"  tvformat varchar(10) NOT NULL default 'Default',"
"  visible tinyint(1) NOT NULL default '1',"
"  outputfilters varchar(255) NOT NULL default '',"
"  useonairguide tinyint(1) default '0',"
"  mplexid smallint(6) default NULL,"
"  serviceid mediumint(8) unsigned default NULL,"
"  tmoffset int(11) NOT NULL default '0',"
"  atsc_major_chan int(10) unsigned NOT NULL default '0',"
"  atsc_minor_chan int(10) unsigned NOT NULL default '0',"
"  last_record datetime NOT NULL,"
"  default_authority varchar(32) NOT NULL default '',"
"  commmethod int(11) NOT NULL default '-1',"
"  PRIMARY KEY  (chanid),"
"  KEY channel_src (channum,sourceid),"
"  KEY sourceid (sourceid,xmltvid,chanid),"
"  KEY visible (visible)"
");",
"CREATE TABLE channelgroup ("
"  id int(10) unsigned NOT NULL auto_increment,"
"  chanid int(11) unsigned NOT NULL default '0',"
"  grpid int(11) NOT NULL default '1', PRIMARY KEY (id)"
");",
"CREATE TABLE channelgroupnames ("
"  grpid int(10) unsigned NOT NULL auto_increment,"
"  name varchar(64) NOT NULL default '0',"
"  PRIMARY KEY (grpid)"
");",
"CREATE TABLE channelscan ("
"  scanid int(3) unsigned NOT NULL AUTO_INCREMENT,"
"  cardid int(3) unsigned NOT NULL,"
"  sourceid int(3) unsigned NOT NULL,"
"  processed tinyint(1) unsigned NOT NULL,"
"  scandate datetime NOT NULL,"
"  PRIMARY KEY  (scanid)"
");",
"CREATE TABLE channelscan_channel ("
"  transportid int(6) unsigned NOT NULL,"
"  scanid int(3) unsigned NOT NULL,"
"  mplex_id smallint(6) NOT NULL,"
"  source_id int(3) unsigned NOT NULL,"
"  channel_id int(3) unsigned NOT NULL default '0',"
"  callsign varchar(20) NOT NULL default '',"
"  service_name varchar(64) NOT NULL default '',"
"  chan_num varchar(10) NOT NULL default '',"
"  service_id mediumint(8) unsigned NOT NULL default '0',"
"  atsc_major_channel int(4) unsigned NOT NULL default '0',"
"  atsc_minor_channel int(4) unsigned NOT NULL default '0',"
"  use_on_air_guide tinyint(1) NOT NULL default '0',"
"  hidden tinyint(1) NOT NULL default '0',"
"  hidden_in_guide tinyint(1) NOT NULL default '0',"
"  freqid varchar(10) NOT NULL default '',"
"  icon varchar(255) NOT NULL default '',"
"  tvformat varchar(10) NOT NULL default 'Default',"
"  xmltvid varchar(64) NOT NULL default '',"
"  pat_tsid int(5) unsigned NOT NULL default '0',"
"  vct_tsid int(5) unsigned NOT NULL default '0',"
"  vct_chan_tsid int(5) unsigned NOT NULL default '0',"
"  sdt_tsid int(5) unsigned NOT NULL default '0',"
"  orig_netid int(5) unsigned NOT NULL default '0',"
"  netid int(5) unsigned NOT NULL default '0',"
"  si_standard varchar(10) NOT NULL,"
"  in_channels_conf tinyint(1) unsigned NOT NULL default '0',"
"  in_pat tinyint(1) unsigned NOT NULL default '0',"
"  in_pmt tinyint(1) unsigned NOT NULL default '0',"
"  in_vct tinyint(1) unsigned NOT NULL default '0',"
"  in_nit tinyint(1) unsigned NOT NULL default '0',"
"  in_sdt tinyint(1) unsigned NOT NULL default '0',"
"  is_encrypted tinyint(1) unsigned NOT NULL default '0',"
"  is_data_service tinyint(1) unsigned NOT NULL default '0',"
"  is_audio_service tinyint(1) unsigned NOT NULL default '0',"
"  is_opencable tinyint(1) unsigned NOT NULL default '0',"
"  could_be_opencable tinyint(1) unsigned NOT NULL default '0',"
"  decryption_status smallint(2) unsigned NOT NULL default '0'"
");",
"CREATE TABLE channelscan_dtv_multiplex ("
"  transportid int(6) unsigned NOT NULL AUTO_INCREMENT,"
"  scanid int(3) unsigned NOT NULL,"
"  mplexid smallint(6) unsigned NOT NULL,"
"  frequency bigint(12) unsigned NOT NULL,"
"  inversion char(1) NOT NULL default 'a',"
"  symbolrate bigint(12) unsigned NOT NULL default '0',"
"  fec varchar(4) NOT NULL default 'auto',"
"  polarity char(1) NOT NULL default '',"
"  hp_code_rate varchar(4) NOT NULL default 'auto',"
"  lp_code_rate varchar(4) NOT NULL default 'auto',"
"  modulation varchar(4) NOT NULL default 'auto',"
"  transmission_mode char(1) NOT NULL default 'a',"
"  guard_interval varchar(4) NOT NULL default 'auto',"
"  hierarchy varchar(4) NOT NULL default 'auto',"
"  bandwidth char(1) NOT NULL default 'a',"
"  sistandard varchar(10) NOT NULL,"
"  tuner_type smallint(2) unsigned NOT NULL,"
"  PRIMARY KEY  (transportid)"
");",
"CREATE TABLE codecparams ("
"  `profile` int(10) unsigned NOT NULL default '0',"
"  `name` varchar(128) NOT NULL default '',"
"  `value` varchar(128) default NULL,"
"  PRIMARY KEY  (`profile`,`name`)"
");",
"CREATE TABLE credits ("
"  person mediumint(8) unsigned NOT NULL default '0',"
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  role set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,person,role),"
"  KEY person (person,role)"
");",
"CREATE TABLE customexample ("
"  rulename varchar(64) NOT NULL,"
"  fromclause text NOT NULL,"
"  whereclause text NOT NULL,"
"  search tinyint(4) NOT NULL default '0',"
"  PRIMARY KEY  (rulename)"
");",
"CREATE TABLE diseqc_config ("
"  cardinputid int(10) unsigned NOT NULL,"
"  diseqcid int(10) unsigned NOT NULL,"
"  `value` varchar(16) NOT NULL default '',"
"  KEY id (cardinputid)"
");",
"CREATE TABLE diseqc_tree ("
"  diseqcid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  parentid int(10) unsigned default NULL,"
"  ordinal tinyint(3) unsigned NOT NULL,"
"  `type` varchar(16) NOT NULL default '',"
"  subtype varchar(16) NOT NULL default '',"
"  description varchar(32) NOT NULL default '',"
"  switch_ports tinyint(3) unsigned NOT NULL default '0',"
"  rotor_hi_speed float NOT NULL default '0',"
"  rotor_lo_speed float NOT NULL default '0',"
"  rotor_positions varchar(255) NOT NULL default '',"
"  lnb_lof_switch int(10) NOT NULL default '0',"
"  lnb_lof_hi int(10) NOT NULL default '0',"
"  lnb_lof_lo int(10) NOT NULL default '0',"
"  cmd_repeat int(11) NOT NULL default '1',"
"  lnb_pol_inv tinyint(4) NOT NULL default '0',"
"  PRIMARY KEY  (diseqcid),"
"  KEY parentid (parentid)"
");",
"CREATE TABLE displayprofilegroups ("
"  `name` varchar(128) NOT NULL,"
"  hostname varchar(64) NOT NULL,"
"  profilegroupid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  PRIMARY KEY  (`name`,hostname),"
"  UNIQUE KEY profilegroupid (profilegroupid)"
");",
"CREATE TABLE displayprofiles ("
"  profilegroupid int(10) unsigned NOT NULL,"
"  profileid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  `value` varchar(128) NOT NULL,"
"  `data` varchar(255) NOT NULL default '',"
"  KEY profilegroupid (profilegroupid),"
"  KEY profileid (profileid,`value`),"
"  KEY profileid_2 (profileid)"
");",
"CREATE TABLE dtv_multiplex ("
"  mplexid smallint(6) NOT NULL AUTO_INCREMENT,"
"  sourceid smallint(6) default NULL,"
"  transportid int(11) default NULL,"
"  networkid int(11) default NULL,"
"  frequency int(11) default NULL,"
"  inversion char(1) default 'a',"
"  symbolrate int(11) default NULL,"
"  fec varchar(10) default 'auto',"
"  polarity char(1) default NULL,"
"  modulation varchar(10) default 'auto',"
"  bandwidth char(1) default 'a',"
"  lp_code_rate varchar(10) default 'auto',"
"  transmission_mode char(1) default 'a',"
"  guard_interval varchar(10) default 'auto',"
"  visible smallint(1) NOT NULL default '0',"
"  constellation varchar(10) default 'auto',"
"  hierarchy varchar(10) default 'auto',"
"  hp_code_rate varchar(10) default 'auto',"
"  sistandard varchar(10) default 'dvb',"
"  serviceversion smallint(6) default '33',"
"  updatetimestamp timestamp NOT NULL default CURRENT_TIMESTAMP,"
"  PRIMARY KEY  (mplexid)"
");",
"CREATE TABLE dtv_privatetypes ("
"  sitype varchar(4) NOT NULL default '',"
"  networkid int(11) NOT NULL default '0',"
"  private_type varchar(20) NOT NULL default '',"
"  private_value varchar(100) NOT NULL default ''"
");",
"CREATE TABLE dvdbookmark ("
"  serialid varchar(16) NOT NULL default '',"
"  `name` varchar(32) default NULL,"
"  title smallint(6) NOT NULL default '0',"
"  audionum tinyint(4) NOT NULL default '-1',"
"  subtitlenum tinyint(4) NOT NULL default '-1',"
"  framenum bigint(20) NOT NULL default '0',"
"  `timestamp` timestamp NOT NULL default CURRENT_TIMESTAMP,"
"  PRIMARY KEY  (serialid)"
");",
"CREATE TABLE eit_cache ("
"  chanid int(10) NOT NULL,"
"  eventid int(10) unsigned NOT NULL default '0',"
"  tableid tinyint(3) unsigned NOT NULL,"
"  version tinyint(3) unsigned NOT NULL,"
"  endtime int(10) unsigned NOT NULL,"
"  `status` tinyint(4) NOT NULL default '0',"
"  PRIMARY KEY  (chanid,eventid,`status`)"
");",
"CREATE TABLE housekeeping ("
"  tag varchar(64) NOT NULL default '',"
"  lastrun datetime default NULL,"
"  PRIMARY KEY  (tag)"
");",
"CREATE TABLE inputgroup ("
"  cardinputid int(10) unsigned NOT NULL,"
"  inputgroupid int(10) unsigned NOT NULL,"
"  inputgroupname varchar(32) NOT NULL"
");",
"CREATE TABLE inuseprograms ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  recusage varchar(128) NOT NULL default '',"
"  lastupdatetime datetime NOT NULL default '0000-00-00 00:00:00',"
"  hostname varchar(64) NOT NULL default '',"
"  rechost varchar(64) NOT NULL,"
"  recdir varchar(255) NOT NULL default '',"
"  KEY chanid (chanid,starttime),"
"  KEY recusage (recusage,lastupdatetime)"
");",
"CREATE TABLE jobqueue ("
"  id int(11) NOT NULL AUTO_INCREMENT,"
"  chanid int(10) NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  inserttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  `type` int(11) NOT NULL default '0',"
"  cmds int(11) NOT NULL default '0',"
"  flags int(11) NOT NULL default '0',"
"  `status` int(11) NOT NULL default '0',"
"  statustime timestamp NOT NULL default CURRENT_TIMESTAMP,"
"  hostname varchar(64) NOT NULL default '',"
"  args blob NOT NULL,"
"  `comment` varchar(128) NOT NULL default '',"
"  schedruntime datetime NOT NULL default '2007-01-01 00:00:00',"
"  PRIMARY KEY  (id),"
"  UNIQUE KEY chanid (chanid,starttime,`type`,inserttime)"
");",
"CREATE TABLE jumppoints ("
"  destination varchar(128) NOT NULL default '',"
"  description varchar(255) default NULL,"
"  keylist varchar(128) default NULL,"
"  hostname varchar(64) NOT NULL default '',"
"  PRIMARY KEY  (destination,hostname)"
");",
"CREATE TABLE keybindings ("
"  `context` varchar(32) NOT NULL default '',"
"  `action` varchar(32) NOT NULL default '',"
"  description varchar(255) default NULL,"
"  keylist varchar(128) default NULL,"
"  hostname varchar(64) NOT NULL default '',"
"  PRIMARY KEY  (`context`,`action`,hostname)"
");",
"CREATE TABLE keyword ("
"  phrase varchar(128) collate utf8_bin NOT NULL default '',"
"  searchtype int(10) unsigned NOT NULL default '3',"
"  UNIQUE KEY phrase (phrase,searchtype)"
");",
"CREATE TABLE mythlog ("
"  logid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  module varchar(32) NOT NULL default '',"
"  priority int(11) NOT NULL default '0',"
"  acknowledged tinyint(1) default '0',"
"  logdate datetime default NULL,"
"  host varchar(128) default NULL,"
"  message varchar(255) NOT NULL default '',"
"  details text,"
"  PRIMARY KEY  (logid),"
"  KEY module (module)"
");",
"CREATE TABLE networkiconmap ("
"  id int(11) NOT NULL AUTO_INCREMENT,"
"  network varchar(20) NOT NULL default '',"
"  url varchar(255) NOT NULL default '',"
"  PRIMARY KEY  (id),"
"  UNIQUE KEY network (network)"
");",
"CREATE TABLE oldfind ("
"  recordid int(11) NOT NULL default '0',"
"  findid int(11) NOT NULL default '0',"
"  PRIMARY KEY  (recordid,findid)"
");",
"CREATE TABLE oldprogram ("
"  oldtitle varchar(128) collate utf8_bin NOT NULL default '',"
"  airdate datetime NOT NULL default '0000-00-00 00:00:00',"
"  PRIMARY KEY  (oldtitle)"
");",
"CREATE TABLE oldrecorded ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  seriesid varchar(40) NOT NULL default '',"
"  programid varchar(40) NOT NULL default '',"
"  findid int(11) NOT NULL default '0',"
"  recordid int(11) NOT NULL default '0',"
"  station varchar(20) NOT NULL default '',"
"  rectype int(10) unsigned NOT NULL default '0',"
"  `duplicate` tinyint(1) NOT NULL default '0',"
"  recstatus int(11) NOT NULL default '0',"
"  reactivate smallint(6) NOT NULL default '0',"
"  generic tinyint(1) default '0',"
"  PRIMARY KEY  (station,starttime,title),"
"  KEY endtime (endtime),"
"  KEY title (title),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY recordid (recordid),"
"  KEY recstatus (recstatus,programid,seriesid),"
"  KEY recstatus_2 (recstatus,title,subtitle)"
");",
"CREATE TABLE people ("
"  person mediumint(8) unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) CHARACTER SET utf8 COLLATE utf8_bin NOT NULL default '',"
"  PRIMARY KEY  (person),"
"  UNIQUE KEY `name` (`name`(41))"
");",
"CREATE TABLE pidcache ("
"  chanid smallint(6) NOT NULL default '0',"
"  pid int(11) NOT NULL default '-1',"
"  tableid int(11) NOT NULL default '-1',"
"  KEY chanid (chanid)"
");",
"CREATE TABLE playgroup ("
"  `name` varchar(32) NOT NULL default '',"
"  titlematch varchar(255) NOT NULL default '',"
"  skipahead int(11) NOT NULL default '0',"
"  skipback int(11) NOT NULL default '0',"
"  timestretch int(11) NOT NULL default '0',"
"  jump int(11) NOT NULL default '0',"
"  PRIMARY KEY  (`name`)"
");",
"CREATE TABLE powerpriority ("
"  priorityname varchar(64) collate utf8_bin NOT NULL,"
"  recpriority int(10) NOT NULL default '0',"
"  selectclause text NOT NULL,"
"  PRIMARY KEY  (priorityname)"
");",
"CREATE TABLE profilegroups ("
"  id int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) default NULL,"
"  cardtype varchar(32) NOT NULL default 'V4L',"
"  is_default int(1) default '0',"
"  hostname varchar(64) default NULL,"
"  PRIMARY KEY  (id),"
"  UNIQUE KEY `name` (`name`,hostname),"
"  KEY cardtype (cardtype)"
");",
"CREATE TABLE program ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  category_type varchar(64) NOT NULL default '',"
"  airdate year(4) NOT NULL default '0000',"
"  stars float NOT NULL default '0',"
"  previouslyshown tinyint(4) NOT NULL default '0',"
"  title_pronounce varchar(128) NOT NULL default '',"
"  stereo tinyint(1) NOT NULL default '0',"
"  subtitled tinyint(1) NOT NULL default '0',"
"  hdtv tinyint(1) NOT NULL default '0',"
"  closecaptioned tinyint(1) NOT NULL default '0',"
"  partnumber int(11) NOT NULL default '0',"
"  parttotal int(11) NOT NULL default '0',"
"  seriesid varchar(40) NOT NULL default '',"
"  originalairdate date default NULL,"
"  showtype varchar(30) NOT NULL default '',"
"  colorcode varchar(20) NOT NULL default '',"
"  syndicatedepisodenumber varchar(20) NOT NULL default '',"
"  programid varchar(40) NOT NULL default '',"
"  manualid int(10) unsigned NOT NULL default '0',"
"  generic tinyint(1) default '0',"
"  listingsource int(11) NOT NULL default '0',"
"  `first` tinyint(1) NOT NULL default '0',"
"  `last` tinyint(1) NOT NULL default '0',"
"  audioprop set('STEREO','MONO','SURROUND','DOLBY','HARDHEAR','VISUALIMPAIR') NOT NULL,"
"  subtitletypes set('HARDHEAR','NORMAL','ONSCREEN','SIGNED') NOT NULL,"
"  videoprop set('HDTV','WIDESCREEN','AVC') NOT NULL,"
"  PRIMARY KEY  (chanid,starttime,manualid),"
"  KEY endtime (endtime),"
"  KEY title (title),"
"  KEY title_pronounce (title_pronounce),"
"  KEY seriesid (seriesid),"
"  KEY id_start_end (chanid,starttime,endtime),"
"  KEY program_manualid (manualid),"
"  KEY previouslyshown (previouslyshown),"
"  KEY programid (programid,starttime),"
"  KEY starttime (starttime)"
");",
"CREATE TABLE programgenres ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  relevance char(1) NOT NULL default '',"
"  genre varchar(30) default NULL,"
"  PRIMARY KEY  (chanid,starttime,relevance),"
"  KEY genre (genre)"
");",
"CREATE TABLE programrating ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  system varchar(8) default NULL,"
"  rating varchar(16) default NULL,"
"  UNIQUE KEY chanid (chanid,starttime,system,rating),"
"  KEY starttime (starttime,system)"
");",
"CREATE TABLE recgrouppassword ("
"  recgroup varchar(32) collate utf8_bin NOT NULL default '',"
"  `password` varchar(10) NOT NULL default '',"
"  PRIMARY KEY  (recgroup),"
"  UNIQUE KEY recgroup (recgroup)"
");",
"CREATE TABLE record ("
"  recordid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  `type` int(10) unsigned NOT NULL default '0',"
"  chanid int(10) unsigned default NULL,"
"  starttime time NOT NULL default '00:00:00',"
"  startdate date NOT NULL default '0000-00-00',"
"  endtime time NOT NULL default '00:00:00',"
"  enddate date NOT NULL default '0000-00-00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  `profile` varchar(128) NOT NULL default 'Default',"
"  recpriority int(10) NOT NULL default '0',"
"  autoexpire int(11) NOT NULL default '0',"
"  maxepisodes int(11) NOT NULL default '0',"
"  maxnewest int(11) NOT NULL default '0',"
"  startoffset int(11) NOT NULL default '0',"
"  endoffset int(11) NOT NULL default '0',"
"  recgroup varchar(32) NOT NULL default 'Default',"
"  dupmethod int(11) NOT NULL default '6',"
"  dupin int(11) NOT NULL default '15',"
"  station varchar(20) NOT NULL default '',"
"  seriesid varchar(40) NOT NULL default '',"
"  programid varchar(40) NOT NULL default '',"
"  search int(10) unsigned NOT NULL default '0',"
"  autotranscode tinyint(1) NOT NULL default '0',"
"  autocommflag tinyint(1) NOT NULL default '0',"
"  autouserjob1 tinyint(1) NOT NULL default '0',"
"  autouserjob2 tinyint(1) NOT NULL default '0',"
"  autouserjob3 tinyint(1) NOT NULL default '0',"
"  autouserjob4 tinyint(1) NOT NULL default '0',"
"  findday tinyint(4) NOT NULL default '0',"
"  findtime time NOT NULL default '00:00:00',"
"  findid int(11) NOT NULL default '0',"
"  inactive tinyint(1) NOT NULL default '0',"
"  parentid int(11) NOT NULL default '0',"
"  transcoder int(11) NOT NULL default '0',"
"  tsdefault float NOT NULL default '1',"
"  playgroup varchar(32) NOT NULL default 'Default',"
"  prefinput int(10) NOT NULL default '0',"
"  next_record datetime NOT NULL,"
"  last_record datetime NOT NULL,"
"  last_delete datetime NOT NULL,"
"  storagegroup varchar(32) NOT NULL default 'Default',"
"  avg_delay int(11) NOT NULL default '100',"
"  PRIMARY KEY  (recordid),"
"  KEY chanid (chanid,starttime),"
"  KEY title (title),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY maxepisodes (maxepisodes),"
"  KEY search (search),"
"  KEY `type` (`type`)"
");",
"CREATE TABLE recorded ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  hostname varchar(64) NOT NULL default '',"
"  bookmark tinyint(1) NOT NULL default '0',"
"  editing int(10) unsigned NOT NULL default '0',"
"  cutlist tinyint(1) NOT NULL default '0',"
"  autoexpire int(11) NOT NULL default '0',"
"  commflagged int(10) unsigned NOT NULL default '0',"
"  recgroup varchar(32) NOT NULL default 'Default',"
"  recordid int(11) default NULL,"
"  seriesid varchar(40) NOT NULL default '',"
"  programid varchar(40) NOT NULL default '',"
"  lastmodified timestamp NOT NULL default CURRENT_TIMESTAMP,"
"  filesize bigint(20) NOT NULL default '0',"
"  stars float NOT NULL default '0',"
"  previouslyshown tinyint(1) default '0',"
"  originalairdate date default NULL,"
"  preserve tinyint(1) NOT NULL default '0',"
"  findid int(11) NOT NULL default '0',"
"  deletepending tinyint(1) NOT NULL default '0',"
"  transcoder int(11) NOT NULL default '0',"
"  timestretch float NOT NULL default '1',"
"  recpriority int(11) NOT NULL default '0',"
"  basename varchar(255) NOT NULL,"
"  progstart datetime NOT NULL default '0000-00-00 00:00:00',"
"  progend datetime NOT NULL default '0000-00-00 00:00:00',"
"  playgroup varchar(32) NOT NULL default 'Default',"
"  `profile` varchar(32) NOT NULL default '',"
"  `duplicate` tinyint(1) NOT NULL default '0',"
"  transcoded tinyint(1) NOT NULL default '0',"
"  watched tinyint(4) NOT NULL default '0',"
"  storagegroup varchar(32) NOT NULL default 'Default',"
"  PRIMARY KEY  (chanid,starttime),"
"  KEY endtime (endtime),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY title (title),"
"  KEY recordid (recordid),"
"  KEY deletepending (deletepending,lastmodified),"
"  KEY recgroup (recgroup,endtime)"
");",
"CREATE TABLE recordedcredits ("
"  person mediumint(8) unsigned NOT NULL default '0',"
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  role set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,person,role),"
"  KEY person (person,role)"
");",
"CREATE TABLE recordedfile ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  basename varchar(128) NOT NULL default '',"
"  filesize bigint(20) NOT NULL default '0',"
"  width smallint(5) unsigned NOT NULL default '0',"
"  height smallint(5) unsigned NOT NULL default '0',"
"  fps float(6,3) NOT NULL default '0.000',"
"  aspect float(8,6) NOT NULL default '0.000000',"
"  audio_sample_rate smallint(5) unsigned NOT NULL default '0',"
"  audio_bits_per_sample smallint(5) unsigned NOT NULL default '0',"
"  audio_channels tinyint(3) unsigned NOT NULL default '0',"
"  audio_type varchar(255) NOT NULL default '',"
"  video_type varchar(255) NOT NULL default '',"
"  `comment` varchar(255) NOT NULL default '',"
"  PRIMARY KEY  (chanid,starttime),"
"  KEY basename (basename)"
");",
"CREATE TABLE recordedmarkup ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  mark mediumint(8) unsigned NOT NULL default '0',"
"  `type` tinyint(4) NOT NULL default '0',"
"  `data` int(11) unsigned default NULL,"
"  PRIMARY KEY  (chanid,starttime,`type`,mark)"
");",
"CREATE TABLE recordedprogram ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  category_type varchar(64) NOT NULL default '',"
"  airdate year(4) NOT NULL default '0000',"
"  stars float unsigned NOT NULL default '0',"
"  previouslyshown tinyint(4) NOT NULL default '0',"
"  title_pronounce varchar(128) NOT NULL default '',"
"  stereo tinyint(1) NOT NULL default '0',"
"  subtitled tinyint(1) NOT NULL default '0',"
"  hdtv tinyint(1) NOT NULL default '0',"
"  closecaptioned tinyint(1) NOT NULL default '0',"
"  partnumber int(11) NOT NULL default '0',"
"  parttotal int(11) NOT NULL default '0',"
"  seriesid varchar(40) NOT NULL default '',"
"  originalairdate date default NULL,"
"  showtype varchar(30) NOT NULL default '',"
"  colorcode varchar(20) NOT NULL default '',"
"  syndicatedepisodenumber varchar(20) NOT NULL default '',"
"  programid varchar(40) NOT NULL default '',"
"  manualid int(10) unsigned NOT NULL default '0',"
"  generic tinyint(1) default '0',"
"  listingsource int(11) NOT NULL default '0',"
"  `first` tinyint(1) NOT NULL default '0',"
"  `last` tinyint(1) NOT NULL default '0',"
"  audioprop set('STEREO','MONO','SURROUND','DOLBY','HARDHEAR','VISUALIMPAIR') NOT NULL,"
"  subtitletypes set('HARDHEAR','NORMAL','ONSCREEN','SIGNED') NOT NULL,"
"  videoprop set('HDTV','WIDESCREEN','AVC') NOT NULL,"
"  PRIMARY KEY  (chanid,starttime,manualid),"
"  KEY endtime (endtime),"
"  KEY title (title),"
"  KEY title_pronounce (title_pronounce),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY id_start_end (chanid,starttime,endtime)"
");",
"CREATE TABLE recordedrating ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  system varchar(8) default NULL,"
"  rating varchar(16) default NULL,"
"  UNIQUE KEY chanid (chanid,starttime,system,rating),"
"  KEY starttime (starttime,system)"
");",
"CREATE TABLE recordedseek ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  mark mediumint(8) unsigned NOT NULL default '0',"
"  `offset` bigint(20) unsigned NOT NULL,"
"  `type` tinyint(4) NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime,`type`,mark)"
");",
"CREATE TABLE recordingprofiles ("
"  id int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) default NULL,"
"  videocodec varchar(128) default NULL,"
"  audiocodec varchar(128) default NULL,"
"  profilegroup int(10) unsigned NOT NULL default '0',"
"  PRIMARY KEY  (id),"
"  KEY profilegroup (profilegroup)"
");",
"CREATE TABLE recordmatch ("
"  recordid int(10) unsigned default NULL,"
"  chanid int(10) unsigned default NULL,"
"  starttime datetime default NULL,"
"  manualid int(10) unsigned default NULL,"
"  oldrecduplicate tinyint(1) default NULL,"
"  recduplicate tinyint(1) default NULL,"
"  findduplicate tinyint(1) default NULL,"
"  oldrecstatus int(11) default NULL,"
"  KEY recordid (recordid)"
");",
"CREATE TABLE settings ("
"  `value` varchar(128) NOT NULL default '',"
"  `data` text,"
"  hostname varchar(64) default NULL,"
"  KEY `value` (`value`,hostname)"
");",
"CREATE TABLE storagegroup ("
"  id int(11) NOT NULL AUTO_INCREMENT,"
"  groupname varchar(32) NOT NULL,"
"  hostname varchar(64) NOT NULL default '',"
"  dirname varchar(235) collate utf8_bin NOT NULL default '',"
"  PRIMARY KEY  (id),"
"  UNIQUE KEY grouphostdir (groupname,hostname,dirname),"
"  KEY hostname (hostname)"
");",
"CREATE TABLE tvchain ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  chainid varchar(128) NOT NULL default '',"
"  chainpos int(10) NOT NULL default '0',"
"  discontinuity tinyint(1) NOT NULL default '0',"
"  watching int(10) NOT NULL default '0',"
"  hostprefix varchar(128) NOT NULL default '',"
"  cardtype varchar(32) NOT NULL default 'V4L',"
"  input varchar(32) NOT NULL default '',"
"  channame varchar(32) NOT NULL default '',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  PRIMARY KEY  (chanid,starttime)"
");",
"CREATE TABLE upnpmedia ("
"  intid int(10) unsigned NOT NULL default '0',"
"  class varchar(64) NOT NULL default '',"
"  itemtype varchar(128) NOT NULL default '',"
"  parentid int(10) unsigned NOT NULL default '0',"
"  itemproperties varchar(255) NOT NULL default '',"
"  filepath varchar(512) NOT NULL default '',"
"  title varchar(255) NOT NULL default '',"
"  filename varchar(512) NOT NULL default '',"
"  coverart varchar(512) NOT NULL default '',"
"  PRIMARY KEY  (intid),"
"  KEY class (class),"
"  KEY filepath (filepath(333)),"
"  KEY parentid (parentid)"
");",
"CREATE TABLE videosource ("
"  sourceid int(10) unsigned NOT NULL AUTO_INCREMENT,"
"  `name` varchar(128) NOT NULL default '',"
"  xmltvgrabber varchar(128) default NULL,"
"  userid varchar(128) NOT NULL default '',"
"  freqtable varchar(16) NOT NULL default 'default',"
"  lineupid varchar(64) default NULL,"
"  `password` varchar(64) default NULL,"
"  useeit smallint(6) NOT NULL default '0',"
"  PRIMARY KEY  (sourceid),"
"  UNIQUE KEY `name` (`name`)"
");",

"INSERT INTO customexample VALUES ('New Flix','','program.category_type = ''movie'' AND program.airdate >= \n     YEAR(DATE_SUB(NOW(), INTERVAL 1 YEAR)) \nAND program.stars > 0.5 ',1);",
"INSERT INTO dtv_privatetypes VALUES ('dvb',9018,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',9018,'guide_fixup','2');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',256,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',257,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',256,'tv_types','1,150,134,133');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',257,'tv_types','1,150,134,133');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4100,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4101,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4102,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4103,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4104,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4105,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4106,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4107,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4097,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4098,'sdt_mapping','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4100,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4101,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4102,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4103,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4104,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4105,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4106,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4107,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4097,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4098,'tv_types','1,145,154');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4100,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4101,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4102,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4103,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4104,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4105,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4106,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4107,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4096,'guide_fixup','5');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4097,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4098,'guide_fixup','1');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',94,'tv_types','1,128');",
"INSERT INTO dtv_privatetypes VALUES ('atsc',1793,'guide_fixup','3');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',40999,'guide_fixup','4');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',70,'force_guide_present','yes');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',70,'guide_ranges','80,80,96,96');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4112,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4115,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4116,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',12802,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',12803,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',12829,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',40999,'parse_subtitle_list','1070,1308,1041,1306,1307,1030,1016,1131,1068,1069');",
"INSERT INTO dtv_privatetypes VALUES ('dvb',4096,'guide_fixup','5');",
"INSERT INTO playgroup VALUES ('Default','',30,5,100,0);",
"INSERT INTO profilegroups VALUES (1,'Software Encoders (v4l based)','V4L',1,NULL);",
"INSERT INTO profilegroups VALUES (2,'IVTV MPEG-2 Encoders','MPEG',1,NULL);",
"INSERT INTO profilegroups VALUES (3,'Hardware MJPEG Encoders (Matrox G200-TV, Miro DC10, etc)','MJPEG',1,NULL);",
"INSERT INTO profilegroups VALUES (4,'Hardware HDTV','HDTV',1,NULL);",
"INSERT INTO profilegroups VALUES (5,'Hardware DVB Encoders','DVB',1,NULL);",
"INSERT INTO profilegroups VALUES (6,'Transcoders','TRANSCODE',1,NULL);",
"INSERT INTO profilegroups VALUES (7,'FireWire Input','FIREWIRE',1,NULL);",
"INSERT INTO profilegroups VALUES (8,'USB Mpeg-4 Encoder (Plextor ConvertX, etc)','GO7007',1,NULL);",
"INSERT INTO profilegroups VALUES (9,'DBOX2 Input','DBOX2',1,NULL);",
"INSERT INTO profilegroups VALUES (10,'Freebox Input','Freebox',1,NULL);",
"INSERT INTO profilegroups VALUES (11,'HDHomeRun Recorders','HDHOMERUN',1,NULL);",
"INSERT INTO profilegroups VALUES (12,'CRC IP Recorders','CRC_IP',1,NULL);",
"INSERT INTO profilegroups VALUES (13,'HD-PVR Recorders','HDPVR',1,NULL);",
"INSERT INTO recordingprofiles VALUES (1,'Default',NULL,NULL,1);",
"INSERT INTO recordingprofiles VALUES (2,'Live TV',NULL,NULL,1);",
"INSERT INTO recordingprofiles VALUES (3,'High Quality',NULL,NULL,1);",
"INSERT INTO recordingprofiles VALUES (4,'Low Quality',NULL,NULL,1);",
"INSERT INTO recordingprofiles VALUES (5,'Default',NULL,NULL,2);",
"INSERT INTO recordingprofiles VALUES (6,'Live TV',NULL,NULL,2);",
"INSERT INTO recordingprofiles VALUES (7,'High Quality',NULL,NULL,2);",
"INSERT INTO recordingprofiles VALUES (8,'Low Quality',NULL,NULL,2);",
"INSERT INTO recordingprofiles VALUES (9,'Default',NULL,NULL,3);",
"INSERT INTO recordingprofiles VALUES (10,'Live TV',NULL,NULL,3);",
"INSERT INTO recordingprofiles VALUES (11,'High Quality',NULL,NULL,3);",
"INSERT INTO recordingprofiles VALUES (12,'Low Quality',NULL,NULL,3);",
"INSERT INTO recordingprofiles VALUES (13,'Default',NULL,NULL,4);",
"INSERT INTO recordingprofiles VALUES (14,'Live TV',NULL,NULL,4);",
"INSERT INTO recordingprofiles VALUES (15,'High Quality',NULL,NULL,4);",
"INSERT INTO recordingprofiles VALUES (16,'Low Quality',NULL,NULL,4);",
"INSERT INTO recordingprofiles VALUES (17,'Default',NULL,NULL,5);",
"INSERT INTO recordingprofiles VALUES (18,'Live TV',NULL,NULL,5);",
"INSERT INTO recordingprofiles VALUES (19,'High Quality',NULL,NULL,5);",
"INSERT INTO recordingprofiles VALUES (20,'Low Quality',NULL,NULL,5);",
"INSERT INTO recordingprofiles VALUES (21,'RTjpeg/MPEG4',NULL,NULL,6);",
"INSERT INTO recordingprofiles VALUES (22,'MPEG2',NULL,NULL,6);",
"INSERT INTO recordingprofiles VALUES (23,'Default',NULL,NULL,8);",
"INSERT INTO recordingprofiles VALUES (24,'Live TV',NULL,NULL,8);",
"INSERT INTO recordingprofiles VALUES (25,'High Quality',NULL,NULL,8);",
"INSERT INTO recordingprofiles VALUES (26,'Low Quality',NULL,NULL,8);",
"INSERT INTO recordingprofiles VALUES (27,'High Quality',NULL,NULL,6);",
"INSERT INTO recordingprofiles VALUES (28,'Medium Quality',NULL,NULL,6);",
"INSERT INTO recordingprofiles VALUES (29,'Low Quality',NULL,NULL,6);",
"INSERT INTO recordingprofiles VALUES (30,'Default',NULL,NULL,10);",
"INSERT INTO recordingprofiles VALUES (31,'Live TV',NULL,NULL,10);",
"INSERT INTO recordingprofiles VALUES (32,'High Quality',NULL,NULL,10);",
"INSERT INTO recordingprofiles VALUES (33,'Low Quality',NULL,NULL,10);",
"INSERT INTO recordingprofiles VALUES (34,'Default',NULL,NULL,11);",
"INSERT INTO recordingprofiles VALUES (35,'Live TV',NULL,NULL,11);",
"INSERT INTO recordingprofiles VALUES (36,'High Quality',NULL,NULL,11);",
"INSERT INTO recordingprofiles VALUES (37,'Low Quality',NULL,NULL,11);",
"INSERT INTO recordingprofiles VALUES (38,'Default',NULL,NULL,12);",
"INSERT INTO recordingprofiles VALUES (39,'Live TV',NULL,NULL,12);",
"INSERT INTO recordingprofiles VALUES (40,'High Quality',NULL,NULL,12);",
"INSERT INTO recordingprofiles VALUES (41,'Low Quality',NULL,NULL,12);",
"INSERT INTO recordingprofiles VALUES (42,'Default',NULL,NULL,7);",
"INSERT INTO recordingprofiles VALUES (43,'Live TV',NULL,NULL,7);",
"INSERT INTO recordingprofiles VALUES (44,'High Quality',NULL,NULL,7);",
"INSERT INTO recordingprofiles VALUES (45,'Low Quality',NULL,NULL,7);",
"INSERT INTO recordingprofiles VALUES (46,'Default',NULL,NULL,9);",
"INSERT INTO recordingprofiles VALUES (47,'Live TV',NULL,NULL,9);",
"INSERT INTO recordingprofiles VALUES (48,'High Quality',NULL,NULL,9);",
"INSERT INTO recordingprofiles VALUES (49,'Low Quality',NULL,NULL,9);",
"INSERT INTO recordingprofiles VALUES (50,'Default',NULL,NULL,13);",
"INSERT INTO recordingprofiles VALUES (51,'Live TV',NULL,NULL,13);",
"INSERT INTO recordingprofiles VALUES (52,'High Quality',NULL,NULL,13);",
"INSERT INTO recordingprofiles VALUES (53,'Low Quality',NULL,NULL,13);",
"INSERT INTO settings VALUES ('mythfilldatabaseLastRunStart',NULL,NULL);",
"INSERT INTO settings VALUES ('mythfilldatabaseLastRunEnd',NULL,NULL);",
"INSERT INTO settings VALUES ('mythfilldatabaseLastRunStatus',NULL,NULL);",
"INSERT INTO settings VALUES ('DataDirectMessage',NULL,NULL);",
"INSERT INTO settings VALUES ('HaveRepeats','0',NULL);",
"INSERT INTO settings VALUES ('DBSchemaVer','1226',NULL);",
"INSERT INTO settings VALUES ('DefaultTranscoder','0',NULL);",
NULL
};

    QString dbver = "";
    if (!performActualUpdate(updates, "1226", dbver))
        return false;
    return true;
}

/* vim: set expandtab tabstop=4 shiftwidth=4: */
