#include "plugin.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include "plugin-setup.h"
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>

extern int DEBUG;

static void sig_child(int signo)
{
    wait(0L);
}

/*
 * Copyright (c) 2002 - 2003
 * NetGroup, Politecnico di Torino (Italy)
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright 
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright 
 * notice, this list of conditions and the following disclaimer in the 
 * documentation and/or other materials provided with the distribution. 
 * 3. Neither the name of the Politecnico di Torino nor the names of its 
 * contributors may be used to endorse or promote products derived from 
 * this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */


/* 
 * the above copyright applies to pthread_suspend only 
 */

#ifdef GTK_ENABLED
gboolean mediacallback(void *data)
{
    nsPluginInstance *instance;

    if (DEBUG > 1)
	printf("in mediacallback\n");
    instance = (nsPluginInstance *) data;
    if (instance->mediaCompleteCallback != NULL)
	NPN_GetURL(instance->mInstance,
		   instance->mediaCompleteCallback, "_self");
    return FALSE;
}
#endif

void pthread_suspend(int msec)
{

    struct timespec abstime;
    struct timeval now;

    pthread_cond_t cond;
    pthread_mutex_t mutex;
    pthread_mutexattr_t attr;

    pthread_mutexattr_init(&attr);
    pthread_mutex_init(&mutex, &attr);
    pthread_mutex_lock(&mutex);

    pthread_cond_init(&cond, NULL);

    gettimeofday(&now, NULL);

    abstime.tv_sec = now.tv_sec + msec / 1000;
    abstime.tv_nsec = now.tv_usec * 1000 + (msec % 1000) * 1000 * 1000;

    pthread_cond_timedwait(&cond, &mutex, &abstime);

    pthread_mutex_destroy(&mutex);
    pthread_cond_destroy(&cond);

}

//this function must be called with control_mutex locked
void launchPlayerThread(nsPluginInstance * instance)
{
    void *thread_return;

    if (DEBUG)
	printf("In launchPlayerThread, state = %d\n", instance->state);

    if (instance->threadlaunched == 1)
	pthread_join(instance->player_thread, &thread_return);

    if (instance->js_state == JS_STATE_UNDEFINED) {

	pthread_create(&(instance->player_thread),
		       &(instance->thread_attr), playPlaylist,
		       (void *) (instance->td));
	instance->js_state = JS_STATE_INITIALIZING;
	instance->threadlaunched = 1;
	instance->threadsignaled = 0;
    } else {
	printf
	    ("****WARNING: launching duplicate player thread, js_state = %d\n",
	     instance->js_state);
    }
}

void signalPlayerThread(nsPluginInstance * instance)
{
    //signal the player thread
    if (DEBUG) {
	printf("Signalling Player thread, state = %d, js_state = %d\n",
	       instance->state, instance->js_state);
    }

    pthread_mutex_lock(&(instance->control_mutex));
    while (instance->js_state == JS_STATE_INITIALIZING) {
	if (DEBUG)
	    printf("Waiting for player thread to start....%i\n",
		   instance->js_state);
	pthread_mutex_unlock(&(instance->control_mutex));
	pthread_suspend(10);
	pthread_mutex_lock(&(instance->control_mutex));
    }
    if (instance->js_state == JS_STATE_BUFFERING) {
	pthread_mutex_lock(&(instance->playlist_cond_mutex));
	pthread_cond_signal(&(instance->playlist_complete_cond));
	pthread_mutex_unlock(&(instance->playlist_cond_mutex));
	instance->threadsignaled = 1;
	pthread_mutex_unlock(&(instance->control_mutex));
    } else {
	if (DEBUG)
	    printf("****Player thread did not start correctly****\n");
	pthread_mutex_unlock(&(instance->control_mutex));
    }
}

FILE *mypopen(char **argv, pid_t * pid, int *control)
{
    int filedesr[2], filedesw[2];
    pid_t child;
    long flags;
    sigset_t newmask;

    pipe(filedesr);
    pipe(filedesw);
    child = fork();

    if (!child) {
	if (DEBUG) {
	    char **tmp;
	    tmp = argv;
	    printf("Starting: ");
	    while (*tmp)
		printf("%s ", *(tmp++));
	    printf("\n");
	}

	dup2(filedesw[0], 0);
	dup2(filedesr[1], 1);
	dup2(filedesr[1], 2);
	close(filedesw[1]);
	close(filedesr[0]);
	setsid();
	setpgid(0, 0);

	if (execvp(argv[0], argv) < 0)
	    perror("execv");
	_exit(0);
    } else {
	// setup signal handler for child
	signal(SIGCHLD, sig_child);
	sigemptyset(&newmask);
	sigaddset(&newmask, SIGCHLD);
	pthread_sigmask(SIG_UNBLOCK, &newmask, 0L);

	*pid = child;
	*control = filedesw[1];
	close(filedesw[0]);
	close(filedesr[1]);

	// make the operations on the control pipe non-blocking
	flags = fcntl(*control, F_GETFL, 0);
	flags |= O_NONBLOCK;
	flags |= O_NDELAY;
	fcntl(*control, F_SETFL, flags);

	return fdopen(filedesr[0], "r");
    }
}


void SetupPlayer(nsPluginInstance * instance, XEvent * event)
{
    int i;
    char xval[32], yval[32];
    char *filename;
    char *baseurl;
    char buffer[1024];

    if (instance->threadsetup == 1)
	return;
    if (instance->toolkitok != 0)
	return;
    instance->td->w = instance->widget;
    instance->td->instance = instance;

#ifdef X_ENABLED
    DrawUI(instance->widget, instance, "Loading Media...", 0, -1);
#endif
#ifdef GTK_ENABLED
    if (instance->status != NULL)
	gtk_label_set_text(instance->status, "Loading Media...");
#endif

    if (instance->td->list == NULL) {
	instance->td->list = instance->list;
    }

    if (instance->href) {
	snprintf(instance->td->list->url, 1024, "%s", instance->href);
    } else if (instance->fname) {
	snprintf(instance->td->list->url, 1024, "%s", instance->fname);
    } else {
	snprintf(instance->td->list->url, 1024, "%s", instance->url);
    }

    if (instance->mode == NP_FULL) {
	snprintf(xval, 32, "%i", instance->window_width);
	snprintf(yval, 32, "%i", instance->window_height);
    } else {
	snprintf(xval, 32, "%i", instance->embed_width);
	snprintf(yval, 32, "%i", instance->embed_height);
    }

    baseurl = NULL;
    if (instance->baseurl == NULL) {
	baseurl = getURLBase(instance->td->list->url);

	if (baseurl != NULL) {
	    if (instance->baseurl == NULL) {
		instance->baseurl = baseurl;
	    } else {
		if (strcmp(instance->baseurl, baseurl) != 0) {
		    NPN_MemFree(instance->baseurl);
		    instance->baseurl = baseurl;
		} else {
		    NPN_MemFree(baseurl);
		}
	    }
	}
    }

    if (instance->hostname == NULL)
	instance->hostname = getURLHostname(instance->td->list->url);

    if (instance->keep_download == 1) {
	instance->td->list->remove = 0;
	filename = getURLFilename(instance->td->list->url);
	snprintf(instance->td->list->fname, 1024, "%s/%s",
		 instance->download_dir, filename);
	if (filename)
	    NPN_MemFree(filename);
    } else {
	if (instance->nomediacache == 0) {
	    if (strlen(instance->td->list->fname) == 0) {
		snprintf(instance->td->list->fname, 1024, "%s",
			 tempnam("/tmp", "mplayerplug-inXXXXXX"));
	    }
	} else {
	    // fix up the URL
	    pthread_mutex_lock(&(instance->playlist_mutex));
	    fullyQualifyURL(instance, instance->td->list->url, buffer);
	    if (DEBUG)
		printf("url %s\nbuffer %s\n", instance->td->list->url,
		       buffer);
	    snprintf(instance->td->list->url, 1024, "%s", buffer);
	    pthread_mutex_unlock(&(instance->playlist_mutex));
	}
    }

    // set up the vars
    i = 0;
    while (i < 50) {
	instance->td->argv[i++] = NULL;
    }

    i = 0;
    snprintf(buffer, 1024, "mplayer");
    instance->td->argv[i++] = strdup(buffer);
    if (instance->novop == 1) {
	snprintf(buffer, 1024, "-vop");
	instance->td->argv[i++] = strdup(buffer);
	snprintf(buffer, 1024, "null");
	instance->td->argv[i++] = strdup(buffer);
    } else {
	if (instance->vop != NULL) {
	    snprintf(buffer, 1024, "-vop");
	    instance->td->argv[i++] = strdup(buffer);
	    snprintf(buffer, 1024, "%s", instance->vop);
	    instance->td->argv[i++] = strdup(buffer);
	    if (strncmp(instance->vop, "scale=", 6) == 0) {
		snprintf(buffer, 1024, "-fs");
		instance->td->argv[i++] = strdup(buffer);
	    }
	}
    }


    if ((instance->mode == NP_EMBED) && (instance->noembed == 0)) {
	if (instance->window != 0) {
	    snprintf(buffer, 1024, "-wid");
	    instance->td->argv[i++] = strdup(buffer);
#ifdef X_ENABLED
//          instance->player_window =
//              XCreateSimpleWindow(instance->display, instance->window, 0,
//                                  0, instance->embed_width,
//                                  instance->embed_height, 0, 0, 0);
	    snprintf(buffer, 1024, "0x%x", (int) instance->window);
	    instance->td->argv[i++] = strdup(buffer);
#endif
#ifdef GTK_ENABLED
	    instance->drawing_area = gtk_drawing_area_new();
	    gtk_fixed_put(GTK_FIXED(instance->fixed_container),
			  instance->drawing_area, 0, 0);
	    gtk_widget_show(instance->drawing_area);
	    gtk_widget_set_events(instance->drawing_area,
				  GDK_BUTTON_PRESS_MASK);
	    instance->player_window =
		GDK_WINDOW_XWINDOW(instance->drawing_area->window);
	    snprintf(buffer, 1024, "0x%x", (int) instance->player_window);
	    instance->td->argv[i++] = strdup(buffer);
#endif
	} else {
	    instance->player_window = 0;
	}
    }

    if ((instance->mode == NP_FULL) && (instance->noembed == 0)) {
	if (instance->window != 0) {
	    snprintf(buffer, 1024, "-wid");
	    instance->td->argv[i++] = strdup(buffer);
#ifdef X_ENABLED
	    snprintf(buffer, 1024, "0x%x", (int) instance->window);
	    instance->td->argv[i++] = strdup(buffer);
#endif
#ifdef GTK_ENABLED
	    instance->drawing_area = gtk_drawing_area_new();
	    gtk_fixed_put(GTK_FIXED(instance->fixed_container),
			  instance->drawing_area, 10, 100);
	    gtk_widget_set_usize(GTK_WIDGET(instance->status),
				 instance->window_width - 20, 19);
	    gtk_widget_show(instance->drawing_area);
	    gtk_widget_set_events(instance->drawing_area,
				  GDK_BUTTON_PRESS_MASK);
	    instance->player_window =
		GDK_WINDOW_XWINDOW(instance->drawing_area->window);
	    snprintf(buffer, 1024, "0x%x", (int) instance->player_window);
	    instance->td->argv[i++] = strdup(buffer);
#endif
	} else {
	    instance->player_window = 0;
	}
    }

    if ((instance->embed_width == 0)
	|| (instance->noembed == 1)) {
	// do nothing
    } else {
	if (instance->mode == NP_EMBED) {
	    if (instance->maintain_aspect == 1) {
		snprintf(buffer, 1024, "-xy");
		instance->td->argv[i++] = strdup(buffer);
		snprintf(buffer, 1024, "%s", xval);
		instance->td->argv[i++] = strdup(buffer);
	    } else {
		snprintf(buffer, 1024, "-x");
		instance->td->argv[i++] = strdup(buffer);
		snprintf(buffer, 1024, "%s", xval);
		instance->td->argv[i++] = strdup(buffer);
		snprintf(buffer, 1024, "-y");
		instance->td->argv[i++] = strdup(buffer);
		snprintf(buffer, 1024, "%s", yval);
		instance->td->argv[i++] = strdup(buffer);
	    }
	}
    }

    if (instance->rtsp_use_tcp == 1) {
	snprintf(buffer, 1024, "-rtsp-stream-over-tcp");
	instance->td->argv[i++] = strdup(buffer);
    }

    if (instance->cachesize > 0) {
	snprintf(buffer, 1024, "-cache");
	instance->td->argv[i++] = strdup(buffer);
	snprintf(buffer, 1024, "%i", instance->cachesize);
	instance->td->argv[i++] = strdup(buffer);
    } else {
	snprintf(buffer, 1024, "-nocache");
	instance->td->argv[i++] = strdup(buffer);
    }

//    if (instance->loop == 1) {
//      snprintf(instance->td->argv[i++], 1024, "-loop");
//      snprintf(instance->td->argv[i++], 1024, "0");
//    }

    if (instance->vo) {
	snprintf(buffer, 1024, "-vo");
	instance->td->argv[i++] = strdup(buffer);
	snprintf(buffer, 1024, "%s", instance->vo);
	instance->td->argv[i++] = strdup(buffer);
	if ((strcmp(buffer, "x11") == 0)
	    || (strstr(buffer, ",x11") != NULL)) {
	    snprintf(buffer, 1024, "-zoom");	/* -vo x11 needs -zoom for rescaling */
	    instance->td->argv[i++] = strdup(buffer);
	}
    }

    if (instance->ao) {
	snprintf(buffer, 1024, "-ao");
	instance->td->argv[i++] = strdup(buffer);
	snprintf(buffer, 1024, "%s", instance->ao);
	instance->td->argv[i++] = strdup(buffer);
    }

    if (instance->output_display) {
	snprintf(buffer, 1024, "-display");
	instance->td->argv[i++] = strdup(buffer);
	snprintf(buffer, 1024, "%s", instance->output_display);
	instance->td->argv[i++] = strdup(buffer);
    }

    snprintf(buffer, 1024, "-osdlevel");
    instance->td->argv[i++] = strdup(buffer);
    snprintf(buffer, 1024, "%i", instance->osdlevel);
    instance->td->argv[i++] = strdup(buffer);

    /* some extra flags for mplayer */
    snprintf(buffer, 1024, "-nojoystick");	/* annoying... */
    instance->td->argv[i++] = strdup(buffer);
//    snprintf(buffer, 1024, "-nofs");  /* no full screen */
//    instance->td->argv[i++] = strdup(buffer);
    snprintf(buffer, 1024, "-slave");	/* slave mode */
    instance->td->argv[i++] = strdup(buffer);
    instance->td->argv[i++] = NULL;

    // ok we have the command line setup
    if (DEBUG)
	printf("ready to setup threads\n");

    pthread_attr_setdetachstate(&(instance->thread_attr),
				PTHREAD_CREATE_JOINABLE);

    if (DEBUG)
	printf("creating thread - NP_EMBED\n");
#ifdef X_ENABLED
    DrawUI(instance->widget, instance, "Getting playlist...", 0, -1);
#endif
#ifdef GTK_ENABLED
    if (instance->status != NULL)
	gtk_label_set_text(instance->status, "Getting playlist...");
#endif

    if (instance->state < STATE_GETTING_PLAYLIST)
	instance->state = STATE_GETTING_PLAYLIST;

    if (0) {
	printf("SetupPlayer: printing instance->list\n");
	printList(instance->list);
	printf("SetupPlayer: printing instance->td->list\n");
	printList(instance->td->list);
    }

    if (DEBUG)
	printf("creating player thread\n");


    pthread_mutex_lock(&(instance->control_mutex));
    launchPlayerThread(instance);
    instance->threadsetup = 1;
    pthread_mutex_unlock(&(instance->control_mutex));

    // recommended slight pause
    usleep(1);

    if (DEBUG)
	printf("MAIN THREAD DONE\n");
}

void refresh_frame(char *buffer, _ThreadData * td, Node * node)
{
    int seconds;
    static int oldseconds = -1;
    char *start;
    char *endptr;
    area *runner;
    area *this_area;
#ifdef SMILDEBUG
    static FILE *error = NULL;
    char errortmp[64];
    if (!error)
	error = fopen("/tmp/mplayerplugin-smil.log", "a");
#endif

    /* just to be sure */
    if (node == NULL || node->area == NULL)
	return;

    start = buffer;
    while ((start = strstr(start, "A:")) != NULL && strlen(start) > 7) {
	start += 2;
	seconds = (int) strtol(start, &endptr, 0);
	if (seconds == oldseconds || start == endptr)
	    continue;
#ifdef SMILDEBUG
	strlcpy(errortmp, start, 63);
	errortmp[64] = 0;
	fprintf(error, "->seconds: %d, oldseconds: %d, buffer: '%s'\n",
		seconds, oldseconds, errortmp);
	fflush(error);
#endif
	runner = node->area;
	this_area = node->area;
	while (runner) {
	    if (runner->begin < seconds
		&& runner->begin > this_area->begin)
		this_area = runner;
	    if (runner->begin == seconds) {
		NPN_GetURL(td->instance->mInstance,
			   runner->url, runner->target);
		break;
	    }
	    runner = runner->next;
	}
	/*
	 *  If it was a seek we have to follow.
	 */
	if ((oldseconds - seconds > 1 || seconds - oldseconds > 1)
	    && !runner)
	    NPN_GetURL(td->instance->mInstance, this_area->url,
		       this_area->target);
	oldseconds = seconds;
    }
}


//return true if we should try to play this again immediately, false if not 

int playNode(ThreadData * local_td, Node * local_list, char *local_url,
	     int local_mmsstream, int *usefps, int *nomouseinput)
{

    char buffer[1024];

    char message[1024];
    int notfound;
    char *vo;
    char vm[10];
    char *cf;
    double cfpercent;
    int retval = FALSE;
    char *eos;
    char *msg;

    long cfbytes;
    int i;
    int zerocfbytes_count = 0;
    char url_copy[1024];


#define TRYAGAIN_FALSE 0
#define TRYAGAIN_TRUE 1
#define TRYAGAIN_FALLBACK 2

    int tryagain = TRYAGAIN_FALLBACK;


/*
  the meaning of the above is as follows:
   TRYAGAIN_TRUE:  we should definitely try playing this url again immediately
   TRYAGAIN_FALSE: we should definitely NOT do that.
   TRYAGAIN_FALLBACK if we should try playing this url again immediately
                    only if we have a fallback url.

   these are for internal use in this function only. The function
   should still return either 0 or 1 (and never 2). 

*/

    pthread_cleanup_push((void (*)(void *))
			 pthread_mutex_unlock,
			 (void *) &(local_td->instance->control_mutex));
    pthread_mutex_lock(&(local_td->instance->control_mutex));
    sendCommand(local_td->instance, "get_time_length");
    pthread_mutex_unlock(&(local_td->instance->control_mutex));
    pthread_cleanup_pop(0);

    while (1) {
	pthread_testcancel();
	if (feof(local_td->instance->player)) {
	    pthread_testcancel();
	    break;
	}
	pthread_testcancel();

	assert(local_td != NULL);
	assert(local_td->instance != NULL);
	assert(local_td->instance->control > 0);
	assert(local_td->instance->player != NULL);

	pthread_testcancel();
#ifdef GTK_ENABLED
	g_idle_add(gtkgui_save_enable, local_td->instance);
#endif
	if (fgets(buffer, 1024, local_td->instance->player) == NULL) {
	    continue;
	}

	pthread_testcancel();

	if (DEBUG) {
	    printf("READ: %s \n", buffer);
	}

	refresh_frame(buffer, local_td, local_list);
	pthread_testcancel();
	if (strstr(buffer, "Cache fill:") != NULL) {
	    strlcpy(message, strstr(buffer, "Cache fill"), 1024);
	    notfound = 1;
	    while (notfound) {
		eos = strrchr(message, '\r');
		if (eos == NULL)
		    eos = message;
		if (strstr(eos, "Cache fill:") == NULL) {
		    // if we don't find "Cache fill:" then replace the \r with \0, making message shorter
		    eos[0] = '\0';
		    // break out of the loop if too short
		    if (strlen(message) < 5)
			break;
		} else {
		    // if we found the last one in the message string, move from \r to C in the string
		    eos++;
		    notfound = 0;
		}
	    }

	    // Only update the display if there is something worth displaying
	    if (strstr(eos, "Cache fill:") != NULL) {
		cf = strstr(eos, "Cache fill:");
		i = sscanf(cf, "Cache fill: %lf %% (%ld bytes)",
			   &cfpercent, &cfbytes);
		//the following is a workaround for an mplayer bug
		// we can try to fallback to mmst when we get
		// Cache Fill: 0% (0 bytes) several times.
		if (i == 2 && cfbytes == 0) {
		    zerocfbytes_count++;
		}
		if (DEBUG && zerocfbytes_count > 0) {
		    printf("Cache Fill: 0%% (0 bytes), count = %d\n",
			   zerocfbytes_count);
		}
		if (zerocfbytes_count >= 1 &&
		    ((strncmp(local_url, "mms://", 6) == 0) ||
		     local_mmsstream
		     && strncmp(local_url, "http://", 7) == 0)) {

		    if (DEBUG) {
			printf("Exiting. Will try again with mmst://\n");
		    }
		    tryagain = TRYAGAIN_FALLBACK;
		    break;
		}

		strlcpy(message, "Buffering ", 1024);
		strlcat(message, local_url, 1000);

#ifdef X_ENABLED
		DrawUI(local_td->instance->widget,
		       local_td->instance, eos, -1, (int) cfpercent);
#endif
#ifdef GTK_ENABLED
		snprintf(local_td->instance->lastmessage, 1024, "%s",
			 message);
		g_idle_add(gtkgui_message, local_td->instance);
		local_td->instance->percent = (cfpercent / 100.0);
		g_idle_add(gtkgui_progress, local_td->instance);

#endif
	    }
	}

	if (strstr(buffer, "Starting") != NULL) {
	    pthread_testcancel();
	    strlcpy(message, "Playing ", 1024);
	    strlcat(message, local_url, 1000);
#ifdef X_ENABLED
	    DrawUI(local_td->instance->widget,
		   local_td->instance, message, 0, -1);
#endif
#ifdef GTK_ENABLED
	    snprintf(local_td->instance->lastmessage, 1024, "%s", message);
	    g_idle_add(gtkgui_message, local_td->instance);
#endif
	}

	if (strstr(buffer, "VO:") != NULL) {
	    if (local_td->instance->mode == NP_EMBED
		&& local_td->instance->noembed == 0)
		local_td->instance->noredraw = 1;
	    vo = strstr(buffer, "VO:");
	    sscanf(vo, "VO: [%9[^]]] %ix%i => %ix%i", vm,
		   &(local_list->actual_x),
		   &(local_list->actual_y),
		   &(local_list->play_x), &(local_list->play_y));
	    if (local_td->instance->mode == NP_EMBED
		&& local_td->instance->noembed == 0) {
		if (local_td->instance->panel_height == 0)
		    local_td->instance->panel_height =
			local_td->instance->embed_height -
			local_list->play_y;

		if (local_td->instance->player_window != 0) {

		    local_td->instance->movie_height = local_list->play_y;
		    local_td->instance->movie_width = local_list->play_x;
#ifdef GTK_ENABLED
		    g_idle_add(gtkgui_resize, local_td->instance);
		    g_idle_add(gtkgui_draw, local_td->instance);
#endif
		}
		if (DEBUG)
		    printf
			("----player thread: panel height in thread = %i\n",
			 local_td->instance->panel_height);

	    } else {
		local_td->instance->panel_height = 16;
		local_td->instance->movie_height = local_list->play_y;
		local_td->instance->movie_width = local_list->play_x;
		if (local_td->instance->player_window != 0) {

#ifdef GTK_ENABLED
		    g_idle_add(gtkgui_resize, local_td->instance);
#endif
		}
		local_td->instance->embed_height =
		    local_td->instance->window_height;
		local_td->instance->embed_width =
		    local_td->instance->window_width;
		local_td->instance->panel_height =
		    local_td->instance->window_height;
#ifdef GTK_ENABLED
		g_idle_add(gtkgui_draw, local_td->instance);
#endif
	    }

	}

	if (strstr(buffer, "Video: no video") != NULL) {
	    if (local_td->instance->panel_height == 0)
		local_td->instance->panel_height = 32;
	    local_td->instance->noredraw = 0;
	    local_td->instance->movie_height = local_list->play_y - 32;
	    local_td->instance->movie_width = local_list->play_x;
#ifdef GTK_ENABLED
	    g_idle_add(gtkgui_resize, local_td->instance);
	    g_idle_add(gtkgui_draw, local_td->instance);
#endif
	}
	// mplayer answer back messages
	if (strstr(buffer, "ANS_LENGTH") != 0) {
	    msg = strstr(buffer, "ANS_LENGTH");
	    sscanf(msg, "ANS_LENGTH=%f",
		   &(local_td->instance->mediaLength));
	    if (DEBUG > 1)
		printf("Media Length = %f\n",
		       local_td->instance->mediaLength);
	}

	if (strstr(buffer, "ANS_PERCENT_POSITION") != 0) {
	    msg = strstr(buffer, "ANS_PERCENT_POSITION");
	    sscanf(msg, "ANS_PERCENT_POSITION=%i",
		   &(local_td->instance->mediaPercent));
	    if (local_td->instance->mediaPercent == 0) {
		if (local_td->instance->mediaLength != 0)
		    local_td->instance->mediaPercent =
			(int) ((local_td->instance->mediaTime * 100) /
			       local_td->instance->mediaLength);
	    }
	    if (DEBUG > 1)
		printf("Percent Played = %i\n",
		       local_td->instance->mediaPercent);
#ifdef GTK_ENABLED
	    g_idle_add(gtkgui_drawMediaProgress, local_td->instance);
#endif
	}
	// this is only needed for the retry code
	if (strstr(buffer, "\rA: ") != 0 || strstr(buffer, "\rV: ") != 0) {
	    //audio or video is playing
	    msg = strstr(buffer, "\rA: ");
	    if (msg != NULL) {
		sscanf(msg, "\rA: %f", &(local_td->instance->mediaTime));
		if (DEBUG > 1)
		    printf("mediaTime = %f\n",
			   local_td->instance->mediaTime);
	    } else {
		msg = strstr(buffer, "\rV: ");
		if (msg != NULL) {
		    sscanf(msg, "\rV: %f",
			   &(local_td->instance->mediaTime));
		    if (DEBUG > 1)
			printf("mediaTime = %f\n",
			       local_td->instance->mediaTime);
		}
	    }
	    pthread_cleanup_push((void (*)(void *))
				 pthread_mutex_unlock,
				 (void *) &(local_td->instance->
					    control_mutex));
	    pthread_mutex_lock(&(local_td->instance->control_mutex));

	    local_td->instance->js_state = JS_STATE_PLAYING;
	    pthread_mutex_unlock(&(local_td->instance->control_mutex));
	    pthread_cleanup_pop(0);
	    if (local_td->instance->paused == 0) {
		pthread_cleanup_push((void (*)(void *))
				     pthread_mutex_unlock,
				     (void *) &(local_td->instance->
						control_mutex));
		pthread_mutex_lock(&(local_td->instance->control_mutex));
		sendCommand(local_td->instance, "get_percent_pos");
		pthread_mutex_unlock(&(local_td->instance->control_mutex));
		pthread_cleanup_pop(0);
	    }
	    tryagain = TRYAGAIN_FALSE;
	}

	if (strstr(buffer, "Connect") != NULL) {
	    snprintf(message, 1024, "%s", buffer);
#ifdef X_ENABLED
	    DrawUI(local_td->instance->widget,
		   local_td->instance, message, 0, -1);
#endif
#ifdef GTK_ENABLED
	    snprintf(local_td->instance->lastmessage, 1024, "%s", message);
	    g_idle_add(gtkgui_message, local_td->instance);
#endif
	}

	if (strstr(buffer, "Error while decoding") != NULL) {
	    if (!isMms(local_url)) {
		if (DEBUG)
		    printf("Resetting stream, 1 sec rewind\n");
		pthread_suspend(1000);
		pthread_cleanup_push((void (*)(void *))
				     pthread_mutex_unlock,
				     (void *) &(local_td->instance->
						control_mutex));
		pthread_mutex_lock(&(local_td->instance->control_mutex));
		sendCommand(local_td->instance, "seek -1 0\n");
		pthread_mutex_unlock(&(local_td->instance->control_mutex));
		pthread_cleanup_pop(0);

	    }
	}

	if (strstr(buffer, "Quit") != NULL) {
	    if (DEBUG)
		printf("----player thread: breaking read loop - Quit\n");
	    local_td->instance->js_state = JS_STATE_UNDEFINED;
	    tryagain = TRYAGAIN_FALSE;
	    break;
	}
	// detect Quicktime file with old codec
	if (strstr(buffer, "MOV: missing header (moov/cmov) chunk!") !=
	    NULL) {
	    if (DEBUG) {
		printf
		    ("----player thread: waiting to download entire movie\n");
	    }
	    tryagain = TRYAGAIN_TRUE;
	    while (1) {
		pthread_testcancel();
		usleep(100);
		pthread_testcancel();
		pthread_mutex_lock(&(local_td->instance->playlist_mutex));
		local_td->instance->state = STATE_DOWNLOADING;
		if (local_list->retrieved == 1) {
		    local_td->instance->state = STATE_PLAYING;
		    pthread_mutex_unlock(&
					 (local_td->instance->
					  playlist_mutex));
		    printf("----player thread: Movie downloaded\n");
		    break;
		}
		pthread_mutex_unlock(&
				     (local_td->instance->playlist_mutex));
	    }
	    break;
	}

	if (strstr(buffer, "FPS not specified") != NULL) {
	    tryagain = TRYAGAIN_TRUE;
	    *usefps = 1;
	    break;
	}

	if (strstr(buffer, "nomouseinput") != NULL) {
	    *nomouseinput = 1;
	    tryagain = TRYAGAIN_TRUE;
	    break;
	}

	if (strstr(buffer, "Exiting") != NULL) {
	    if (DEBUG)
		printf
		    ("----player thread: breaking read loop - Exiting\n");
	    if (local_td->instance->mediaPercent != 0) {
		local_td->instance->mediaPercent = 100;
#ifdef GTK_ENABLED
		g_idle_add(gtkgui_drawMediaProgress, local_td->instance);
#endif
	    }
	    tryagain = TRYAGAIN_FALSE;
	    break;
	}
	// break out if mplayer crashes
	if (strstr(buffer, "interrupted") != NULL) {
	    if (DEBUG)
		printf
		    ("----player thread: breaking read loop - interrupted\n");
	    break;
	}

	if (strstr(buffer, "explicit kill") != NULL) {
	    if (DEBUG)
		printf("----player thread: breaking read loop - killed\n");
	    break;
	}

	if (strstr(buffer, "everything done") != NULL) {
	    if (DEBUG)
		printf
		    ("----player thread: breaking read loop - codec issue\n");
	    break;
	}


	assert(local_td->instance->player != NULL);

	//may not be completely safe
	pthread_mutex_lock(&(local_td->instance->playlist_mutex));
	if (local_td->instance->cancelled == 1) {
	    pthread_mutex_unlock(&(local_td->instance->playlist_mutex));
	    if (DEBUG)
		printf
		    ("----player thread: breaking read loop - cancelled\n");
	    tryagain = TRYAGAIN_FALSE;
	    break;
	}
	pthread_mutex_unlock(&(local_td->instance->playlist_mutex));
    }


    if (DEBUG) {
	printf("---player thread: tryagain = %d\n", tryagain);
    }
    //we always return either true or false
    if (tryagain == TRYAGAIN_TRUE) {
	retval = TRUE;
    }

    if (tryagain == TRYAGAIN_FALSE) {
	retval = FALSE;
    }

    if (tryagain == TRYAGAIN_FALLBACK) {
	//fallback to msst if we can
	strlcpy(url_copy, local_url, 1023);
	url_copy[1023] = '\0';

	if (strncmp(local_url, "mms://", 6) == 0) {
	    snprintf(local_url, 1023, "mmst://%s", url_copy + 6);
	    retval = TRUE;
	} else if (local_mmsstream &&
		   strncmp(local_url, "http://", 7) == 0) {

	    snprintf(local_url, 1023, "mmst://%s", url_copy + 7);
	    retval = TRUE;
	} else {
	    retval = FALSE;
	}
    }

    return retval;
}

void *playPlaylist(void *td)
{

    FILE *playlist;
    char cmd[2048];
    char message[1024];
    char mmsplaylist[1024];

    int argc = 0, base_argc = 0;
    ThreadData *local_td;
    Node *local_list;
    char local_url[1024];
    int local_mmsstream;
    int i;
    Node *node;

    int tryagain;
    int usefps;
    int nomouseinput;
    int listempty;
    
    local_td = (ThreadData *) td;
    local_list = local_td->list;

    if (local_td == NULL)
	pthread_exit(0);

    if (DEBUG)
	printf("----player thread: in playPlayList\n");

    pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock,
			 (void *) &(local_td->instance->
				    playlist_cond_mutex));
    pthread_mutex_lock(&(local_td->instance->playlist_cond_mutex));

    pthread_mutex_lock(&(local_td->instance->control_mutex));
    local_td->instance->js_state = JS_STATE_BUFFERING;
    pthread_mutex_unlock(&(local_td->instance->control_mutex));

    if (DEBUG)
	printf
	    ("----player thread: about to go to sleep, js_state = %d, state = %d\n",
	     local_td->instance->js_state, local_td->instance->state);


    if (local_td->instance->state < STATE_STARTED_PLAYER) {
	pthread_cond_wait(&(local_td->instance->playlist_complete_cond),
			  &(local_td->instance->playlist_cond_mutex));

	assert(local_td != NULL);
	assert(local_td->instance != NULL);

	local_td->instance->state = STATE_STARTED_PLAYER;
	if (DEBUG)
	    printf
		("---player thread: got wakeup signal, js_state = %d, state = %d\n",
		 local_td->instance->js_state, local_td->instance->state);

    }

    pthread_mutex_unlock(&(local_td->instance->playlist_cond_mutex));
    pthread_cleanup_pop(0);

    if (DEBUG)
	printf("----player thread: playPlayList - waiting complete\n");

    pthread_testcancel();
#ifdef X_ENABLED
    DrawUI(local_td->w, local_td->instance, "Loading Media...", 0, -1);
#endif
#ifdef GTK_ENABLED
    snprintf(local_td->instance->lastmessage, 1024, "Loading Media...");
    g_idle_add(gtkgui_message, local_td->instance);
#endif

    memset(cmd, '\0', 1);
    i = 0;
    if (DEBUG)
	printf("----player thread: building command string\n");

    pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock,
			 (void *) &(local_td->instance->playlist_mutex));
    pthread_mutex_lock(&(local_td->instance->playlist_mutex));

    while (local_td->argv[argc] != NULL) {
	if (DEBUG)
	    printf("PLAY %i:%s\n", i, local_td->argv[argc]);
	argc++;
    }

    base_argc = argc;

    pthread_mutex_unlock(&(local_td->instance->playlist_mutex));
    pthread_cleanup_pop(0);

    //handle qtNext
    //shouldn't this be somewhere else?
    i = 0;
    while (local_td->instance->qtNext[i] != NULL) {
	node = newNode();
	if (DEBUG)
	    printf("----player thread: adding %s\n",
		   local_td->instance->qtNext[i]);
	strlcpy(node->url, local_td->instance->qtNext[i], 1024);
	i++;
	addToEnd(local_list, node);
    }

    // preview playlist
    local_list = local_td->list;
    if (DEBUG) {
	while (local_list != NULL) {
	    printf
		("----player thread: local_list =  %p\nPL URL: %s\nplay = %i, cancelled= %i\nnext= %p\n",
		 local_list,
		 local_list->url, local_list->play,
		 local_list->cancelled, local_list->next);
	    local_list = local_list->next;
	}
    }

    while (1) {
	if (DEBUG)
	    printf("----player thread: entering loop\n");

	pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock,
			     (void *) &(local_td->instance->
					playlist_mutex));

	pthread_mutex_lock(&(local_td->instance->playlist_mutex));

	//find a node to play
	if (DEBUG)
	    printf("----player thread: looking for node to play\n");

	listempty = 0;
	local_list = local_td->list;
	while (local_list != NULL) {
	    if (local_list->play && !(local_list->played)) {
		break;
	    }
	    local_list = local_list->next;
	}

	if (local_list == NULL) {
	    //nothing to play: will exit
	    if (local_td->instance->loop) {
		local_list = local_td->list;
		while (local_list != NULL) {
		    if (local_list->play)
			local_list->played = 0;
		    local_list = local_list->next;
		}

		// rescan again
		local_list = local_td->list;
		while (local_list != NULL) {
		    if (local_list->play && !(local_list->played)) {
			break;
		    }
		    local_list = local_list->next;
		}
	    }
	}

	local_td->instance->currentnode = local_list;
	if (local_list == NULL) {
	    if (DEBUG)
		printf("----player thread: nothing on the list to play\n");
	    listempty = 1;
	    local_mmsstream = 0;
	} else {
	    //we have a node to play
	    if (DEBUG) 
	        printf("----player thread: chose url %s\n", local_list->url);
	    listempty = 0;
	    snprintf(local_url, 1024, "%s", local_list->url);
	    local_url[1023] = '\0';
	    local_mmsstream = local_list->mmsstream;
        }
	
	pthread_mutex_unlock(&(local_td->instance->playlist_mutex));
	pthread_cleanup_pop(0);

	if (listempty == 1)
	    break;
	
	pthread_testcancel();

	usefps = 0;
	nomouseinput = 0;

	do {

	    pthread_testcancel();

	    tryagain = 0;
	    argc = base_argc;
	    while (argc < 50) {
		if (local_td->argv[argc] != NULL)
		    free(local_td->argv[argc]);
		local_td->argv[argc++] = NULL;
	    }
	    argc = base_argc;

	    if (DEBUG)
		printf("----player thread: playing url: %s\n", local_url);
	    strlcpy(message, "Playing ", 1024);
	    strlcat(message, local_list->url, 1000);
#ifdef X_ENABLED
	    DrawUI(local_td->w, local_td->instance, message, 0, -1);
#endif
#ifdef GTK_ENABLED
	    snprintf(local_td->instance->lastmessage, 1024, "%s", message);
	    g_idle_add(gtkgui_message, local_td->instance);
#endif
	    pthread_cleanup_push((void (*)(void *))
				 pthread_mutex_unlock,
				 (void *) &(local_td->instance->
					    playlist_mutex));
	    pthread_mutex_lock(&(local_td->instance->playlist_mutex));



	    // for rtsp streams we need to specify FPS
	    if (usefps == 1) {
		if (strncmp(local_list->url, "rtsp", 4) == 0) {
		    local_td->argv[argc++] = strdup("-fps");
		    local_td->argv[argc++] = strdup("30");
		}
		usefps = 0;
	    }

	    if (nomouseinput == 0) {
		local_td->argv[argc++] = strdup("-nomouseinput");
	    } else {
		nomouseinput = 1;
	    }

	    if (local_list->playlist == 1)
		local_td->argv[argc++] = strdup("-playlist");

	    if (local_td->instance != NULL)
		local_td->instance->mmsstream = local_list->mmsstream;
	    if (local_list->mmsstream) {
#ifdef GTK_ENABLED
		g_idle_add(gtkgui_updatebuttons, local_td->instance);
#endif
		local_td->argv[argc++] = strdup(local_url);
		if (local_td->instance->keep_download) {
		    snprintf(mmsplaylist, sizeof(mmsplaylist),
			     "%s/playlist",
			     local_td->instance->download_dir);
		    playlist = fopen(mmsplaylist, "a");
		    if (playlist != NULL) {
			fprintf(playlist, "%s\n", local_url);
			fclose(playlist);
		    }
		}
	    } else {
		if (strlen(local_list->fname) == 0) {
		    local_td->argv[argc++] = strdup(local_url);
		} else {
		    local_td->argv[argc++] = strdup(local_list->fname);
		}
	    }

	    if (DEBUG) {
		printf("----player thread: URL: %s\n", local_url);
	    }

	    pthread_mutex_unlock(&(local_td->instance->playlist_mutex));
	    pthread_cleanup_pop(0);

	    pthread_testcancel();

	    //start up mplayer
	    pthread_cleanup_push((void (*)(void *))
				 pthread_mutex_unlock,
				 (void *) &(local_td->instance->
					    control_mutex));
	    pthread_mutex_lock(&(local_td->instance->control_mutex));

	    local_td->instance->player =
		mypopen(local_td->argv, &local_td->instance->pid,
			&local_td->instance->control);
	    if (local_td->instance->player != NULL) {
		local_td->instance->js_state = JS_STATE_PLAYING;
	    }
	    pthread_mutex_unlock(&(local_td->instance->control_mutex));
	    pthread_cleanup_pop(0);

	    local_td->instance->state = STATE_PLAYING;
	    if (local_td->instance->player != NULL) {

		assert(local_td->instance->control > 0);

		// reset media counters
		local_td->instance->mediaLength = 0.0;
		local_td->instance->mediaPercent = 0;

		tryagain =
		    playNode(local_td, local_list, local_url,
			     local_mmsstream, &usefps, &nomouseinput);

		if (DEBUG)
		    printf
			("----player thread: playNode returned = %d\n",
			 tryagain);

		pthread_testcancel();
		assert(local_list != NULL);

		local_td->instance->mediaPercent = 0;
#ifdef GTK_ENABLED
		g_idle_add(gtkgui_drawMediaProgress, local_td->instance);
#endif

		// this cancels the download of this media
		// if there are more than 1, saves bandwidth
		//also revents us from playing it again
		pthread_mutex_lock(&(local_td->instance->playlist_mutex));
		if (!tryagain) {
		    local_list->played = 1;
		}
		pthread_mutex_unlock(&
				     (local_td->instance->playlist_mutex));

		assert(local_td->instance != NULL);

		//close the pipes to mplayer
		pthread_mutex_lock(&(local_td->instance->control_mutex));
		local_td->instance->js_state = JS_STATE_TRANSITIONING;
		pthread_mutex_unlock(&(local_td->instance->control_mutex));


		assert(local_td->instance->control > 0);
		close(local_td->instance->control);
		local_td->instance->control = -1;

		assert(local_td->instance->player != NULL);
		fclose(local_td->instance->player);
		local_td->instance->player = NULL;

		if (DEBUG)
		    printf("----player thread: close done\n");

		local_td->instance->state = STATE_PLAYLIST_NEXT;

	    }
	} while (tryagain);

	local_td->instance->currentnode = NULL;

	//if we get here, either the control pipes are closed
	//  or they were never opened in the first place

	if (DEBUG) {
	    printf("----player thread: transitioning to next item\n");
	}

	pthread_mutex_lock(&(local_td->instance->control_mutex));
	local_td->instance->js_state = JS_STATE_TRANSITIONING;
	pthread_mutex_unlock(&(local_td->instance->control_mutex));

	pthread_testcancel();
	assert(local_td != NULL);
	assert(local_td->instance != NULL);
	pthread_testcancel();


	// if we have media type that changes from video to audio 
	// in X mode this will take care of that situation
	local_td->instance->noredraw = 0;

	/*
	   assert(local_td->instance->state != STATE_PLAY_CANCELLED);
	 */
    }				// end main while loop

    if (DEBUG) {
	printf("----player thread: nothing to play\n");
    }
#ifdef GTK_ENABLED
    g_idle_add(gtkgui_stop, local_td->instance);
    // call media complete when we are done with the playlist
    if (local_td->instance->mediaCompleteCallback != NULL) {
	g_idle_add(mediacallback, local_td->instance);
    }
#endif

    if (DEBUG) {
	printf("----player thread: callbacks complete\n");
    }

    pthread_mutex_lock(&(local_td->instance->control_mutex));
    local_td->instance->js_state = JS_STATE_UNDEFINED;
    pthread_mutex_unlock(&(local_td->instance->control_mutex));

    //once we get here, the thread is only allowed to access stack
    // variables and then exit (since another player thread may have
    // been started at this time).    

    if (DEBUG) {
	printf("----player thread: normal exit\n");
    }
    pthread_exit(0);

    return NULL;
}
