/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package test.javax.management.monitor;

import javax.management.Attribute;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.management.monitor.MonitorNotification;

import junit.framework.TestCase;

/**
 * Class GaugeMonitorTest, tests the Gauge monitoring service
 *
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @version $Revision: 1.10 $
 */
public class GaugeMonitorTest extends TestCase
{
	private MBeanServer server;

  /**
   * Dummy test interface
   */
  public interface TestClassMBean
  {
    public Double getDouble();

    public void setDouble(Double number);
  }

  public static class TestClass implements TestClassMBean
  {
    private Double number;

    public TestClass(Double number)
    {
      this.number = number;
    }

    public Double getDouble()
    {
      return number;
    }

    public void setDouble(Double number)
    {
      this.number = number;
    }
  }

  public static class MonitorListener implements NotificationListener
  {
  	MonitorNotification lastNotification;

    public void handleNotification(Notification notification, Object handback)
    {
      if (notification instanceof MonitorNotification)
      {
        lastNotification = (MonitorNotification)notification;
      }
    }

    public MonitorNotification getLastNotification()
    {
      return lastNotification;
    }
  }

  /**
   * Constructor requested by the JUnit framework
   */
  public GaugeMonitorTest()
  {
    super("GaugeMonitor Test");
  }

  /**
   * Constructor requested by the JUnit framework
   */
  public GaugeMonitorTest(String name)
  {
		super(name);
  }

  protected void setUp() throws Exception
  {
    super.setUp();
    server = MBeanServerFactory.createMBeanServer("GaugeMonitor");
  }

  public void testBasics() throws Exception
  {
    ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
    try
    {
      server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
      // test defaults
      assertEquals(new Integer(0), server.getAttribute(monitorName, "HighThreshold"));
      assertEquals(new Integer(0), server.getAttribute(monitorName, "LowThreshold"));
      assertEquals(Boolean.FALSE, server.getAttribute(monitorName, "DifferenceMode"));
      assertEquals(Boolean.FALSE, server.getAttribute(monitorName, "NotifyHigh"));
      assertEquals(Boolean.FALSE, server.getAttribute(monitorName, "NotifyLow"));
      // set tests
      boolean exception = false;
      try
      {
        server.invoke(monitorName, "setThresholds", new Object[] {new Double(-1), null}, new String[] {"java.lang.Double", "java.lang.Double"});
      }
      catch (Exception e)
      {
        exception = true;
      }
      assertTrue(exception);
      exception = false;
      try
      {
        server.invoke(monitorName, "setThresholds", new Object[] {null, new Double(-1)}, new String[] {"java.lang.Number", "java.lang.Number"});
      }
      catch (Exception e)
      {
        exception = true;
      }
      assertTrue(exception);
      exception = false;
      try
      {
        server.invoke(monitorName, "setThresholds", new Object[] {new Double(7), new Float(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      }
      catch (Exception e)
      {
        exception = true;
      }
      assertTrue(exception);
      exception = false;
      try
      {
        server.invoke(monitorName, "setThresholds", new Object[] {new Double(3), new Double(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      }
      catch (Exception e)
      {
        exception = true;
      }
      assertTrue(exception);
      try
      {
        server.invoke(monitorName, "setThresholds", new Object[] {new Integer(3), new Integer(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      }
      catch (Exception e)
      {
        exception = true;
      }
      assertTrue(exception);
      exception = false;
      server.invoke(monitorName, "setThresholds", new Object[] {new Double(10), new Double(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      assertEquals(new Double(10), server.getAttribute(monitorName, "HighThreshold"));
      assertEquals(new Double(5), server.getAttribute(monitorName, "LowThreshold"));
      server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
      assertEquals(Boolean.TRUE, server.getAttribute(monitorName, "DifferenceMode"));
      server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
      assertEquals(Boolean.TRUE, server.getAttribute(monitorName, "NotifyHigh"));
      server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.TRUE));
      assertEquals(Boolean.TRUE, server.getAttribute(monitorName, "NotifyLow"));
    }
    finally
    {
      server.unregisterMBean(monitorName);
    }
  }

  public void testLegacyMonitor() throws Exception
  {
    ObjectName name1 = new ObjectName("GaugeMonitor:name=monitorTarget1");
    ObjectName name2 = new ObjectName("GaugeMonitor:name=monitorTarget2");
    ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
    try {
      TestClass a = new TestClass(new Double(7));
      server.registerMBean(a, name1);
      TestClass b = new TestClass(new Double(5));
      server.registerMBean(b, name2);
      server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
      server.invoke(monitorName, "addObservedObject", new Object []{name1},new String[]{"javax.management.ObjectName"});
      server.invoke(monitorName, "addObservedObject", new Object []{name2},new String[]{"javax.management.ObjectName"});
      server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Double"));
      server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
      server.invoke(monitorName, "setThresholds", new Object[] {new Double(10), new Double(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
      MonitorListener listener = new MonitorListener();
      server.addNotificationListener(monitorName, listener, null, null);
      server.invoke(monitorName, "start", new Object[] {}, new String[] {});

		// Wait some more than 1 second, we discovered that on windows it is not enough
		Thread.sleep(1200);

      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      long timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      assertTrue(timestamp>0);
      // b should trigger one notification
      assertNotNull(listener.getLastNotification());

		Thread.sleep(1000);

      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      server.setAttribute(name1, new Attribute("Double", new Double(11d)));
      server.setAttribute(name2, new Attribute("Double", new Double(11d)));

		Thread.sleep(1000);

      assertEquals(new Double(4), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(6), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      long sequence = listener.getLastNotification().getSequenceNumber();

		Thread.sleep(1000);

      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      // test hysteresis
      server.setAttribute(name1, new Attribute("Double", new Double(15d)));
      server.setAttribute(name2, new Attribute("Double", new Double(17d)));

		Thread.sleep(1000);

      assertEquals(new Double(4), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(6), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      server.setAttribute(name1, new Attribute("Double", new Double(4d)));
      server.setAttribute(name2, new Attribute("Double", new Double(4d)));

		Thread.sleep(1000);

      assertEquals(new Double(-11), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(-13), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
      sequence = listener.getLastNotification().getSequenceNumber();
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();

		Thread.sleep(1000);

      // check new threshold
      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(0), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
      server.setAttribute(name1, new Attribute("Double", new Double(16d)));
      server.setAttribute(name2, new Attribute("Double", new Double(18d)));

		Thread.sleep(1000);

      // check new threshold
      assertEquals(new Double(12), server.invoke(monitorName, "getDerivedGauge", new Object[]{name1},new String [] {"javax.management.ObjectName"}));
      assertEquals(new Double(14), server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String [] {"javax.management.ObjectName"}));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue() > timestamp);
      timestamp = ((Long)server.invoke(monitorName, "getDerivedGaugeTimeStamp", new Object[]{name2},new String [] {"javax.management.ObjectName"})).longValue();
    } finally {
      server.unregisterMBean(name1);
      server.unregisterMBean(name2);
      server.unregisterMBean(monitorName);
    }
  }

  public void testMonitor() throws Exception
  {
    ObjectName name = new ObjectName("GaugeMonitor:name=monitorTarget");
    ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
    try {
      TestClass a = new TestClass(new Double(7));
      server.registerMBean(a, name);
      server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
      server.setAttribute(monitorName, new Attribute("ObservedObject", name));
      server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Double"));
      server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
      server.invoke(monitorName, "setThresholds", new Object[] {new Double(10), new Double(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
      MonitorListener listener = new MonitorListener();
      server.addNotificationListener(monitorName, listener, null, null);
      server.invoke(monitorName, "start", new Object[] {}, new String[] {});

		// Wait some more than 1 second, we discovered that on windows it is not enough
		Thread.sleep(1200);

      assertEquals(new Double(0), server.getAttribute(monitorName, "DerivedGauge"));
      long timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      assertTrue(timestamp>0);
      assertTrue(listener.getLastNotification() == null);

		Thread.sleep(1000);

      assertEquals(new Double(0), server.getAttribute(monitorName, "DerivedGauge"));
      assertTrue(listener.getLastNotification() == null);
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      server.setAttribute(name, new Attribute("Double", new Double(11d)));

		Thread.sleep(1000);

      assertEquals(new Double(4), server.getAttribute(monitorName, "DerivedGauge"));
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      long sequence = listener.getLastNotification().getSequenceNumber();

		Thread.sleep(1000);

      assertEquals(new Double(0), server.getAttribute(monitorName, "DerivedGauge"));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      // test hysteresis
      server.setAttribute(name, new Attribute("Double", new Double(15d)));

		Thread.sleep(1000);

      assertEquals(new Double(4), server.getAttribute(monitorName, "DerivedGauge"));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      server.setAttribute(name, new Attribute("Double", new Double(4d)));

		Thread.sleep(1000);

      assertEquals(new Double(-11), server.getAttribute(monitorName, "DerivedGauge"));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
      sequence = listener.getLastNotification().getSequenceNumber();
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();

		Thread.sleep(1000);

      // check new threshold
      assertEquals(new Double(0), server.getAttribute(monitorName, "DerivedGauge"));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
      server.setAttribute(name, new Attribute("Double", new Double(16d)));

		Thread.sleep(1000);

      // check new threshold
      assertEquals(new Double(12), server.getAttribute(monitorName, "DerivedGauge"));
      // repeat last notification. i.e. no new notifications
      assertEquals(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, listener.getLastNotification().getType());
      assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
      assertTrue(((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue() > timestamp);
      timestamp = ((Long)server.getAttribute(monitorName
      	, "DerivedGaugeTimeStamp")).longValue();
    } finally {
      server.unregisterMBean(name);
      server.unregisterMBean(monitorName);
    }
  }

	public void testStartStopSequence() throws Exception
  {
    ObjectName name = new ObjectName("GaugeMonitor:name=monitorTarget");
    ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
    try
    {
      TestClass a = new TestClass(new Double(7));
      server.registerMBean(a, name);
      server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
      server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
			server.setAttribute(monitorName, new Attribute("ObservedObject", name));
      server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Double"));
      server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
      server.invoke(monitorName, "setThresholds", new Object[] {new Double(10), new Double(5)}, new String[] {"java.lang.Number", "java.lang.Number"});
      server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.TRUE));
      server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
      MonitorListener listener = new MonitorListener();
      server.addNotificationListener(monitorName, listener, null, null);
      server.invoke(monitorName, "start", new Object[] {}, new String[] {});
			server.invoke(monitorName, "stop", new Object[] {}, new String[] {});
			server.invoke(monitorName, "start", new Object[] {}, new String[] {});
		}
		finally
    {
      server.unregisterMBean(name);
      server.unregisterMBean(monitorName);
    }
	}

    public void testAddRemove() throws Exception{
        ObjectName name1 = new ObjectName("GaugeMonitor:name=monitorTarget1");
        ObjectName name2 = new ObjectName("GaugeMonitor:name=monitorTarget2");
        ObjectName name3 = new ObjectName("GaugeMonitor:name=monitorTarget3");
        ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
        try {
            TestClass a = new TestClass(new Double(0));
            server.registerMBean(a, name1);
            TestClass b = new TestClass(new Double(1.5));
            server.registerMBean(b, name2);
            TestClass c = new TestClass(new Double(2.3));
            server.registerMBean(c, name3);
            server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
            //server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.invoke(monitorName, "addObservedObject", new Object[] {name1}, new String [] {"javax.management.ObjectName"});
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});

            // should have one observed object
            ObjectName [] names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0], name1);

            // add another, make sure we have two
            server.invoke(monitorName, "addObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name2);

            // remove one, make sure we're back to one
            server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0],name1);

            // now add back a different one
            server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name3);

            // try to remove one that doesn't exist. Should NOT get IllegalArgumentException since JMX 1.2
            boolean errorFound = false;
            try{
                server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);

            // try to add one that already exists. Should NOT get IllegalArgumentException since JMX 1.2
            errorFound = false;
            try{
                server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);
        }finally {
            server.unregisterMBean(name1);
            server.unregisterMBean(name2);
            server.unregisterMBean(name3);
            server.unregisterMBean(monitorName);
        }
    }



      public void testBug710028() throws Exception
      {
        // Test case for bug #710028
        ObjectName name = new ObjectName("GaugeMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
        try {
          // Initial value < lowThreshold
          TestClass a = new TestClass(new Double(4));
          server.registerMBean(a, name);
          server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
          server.setAttribute(monitorName,
                              new Attribute("DifferenceMode", Boolean.TRUE));
          server.setAttribute(monitorName, new Attribute("ObservedObject", name));
          server.setAttribute(monitorName,
                              new Attribute("ObservedAttribute", "Double"));
          server.setAttribute(monitorName,
                              new Attribute("GranularityPeriod", new Long(1000L)));
          server.invoke(monitorName, "setThresholds", new Object[] {new Double(10),
                        new Double(5)}
                        , new String[] {"java.lang.Number", "java.lang.Number"});
          server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
          server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.FALSE));
          server.setAttribute(monitorName,
                              new Attribute("DifferenceMode", Boolean.TRUE));
          MonitorListener listener = new MonitorListener();
          server.addNotificationListener(monitorName, listener, null, null);
          server.invoke(monitorName, "start", new Object[] {}
                        , new String[] {});

          // set over high threshold
          server.setAttribute(name, new Attribute("Double", new Double(10)));

          assertNull(listener.getLastNotification());
          Thread.sleep(1200);

          assertNotNull(listener.getLastNotification());
          long sequence = listener.getLastNotification().getSequenceNumber();

          // set below low threshold
          server.setAttribute(name, new Attribute("Double", new Double(4)));

          Thread.sleep(1200);

          assertEquals(sequence, listener.getLastNotification().getSequenceNumber());

          // set over high threshold
          server.setAttribute(name, new Attribute("Double", new Double(10)));
          Thread.sleep(1200);

          // we expect a new notification
          assertTrue(sequence != listener.getLastNotification().getSequenceNumber());

        }
        finally {
          server.unregisterMBean(name);
          server.unregisterMBean(monitorName);
        }
      }

  public void testBug742554() throws Exception
  {
     // Test case for bug #742554
     ObjectName name = new ObjectName("GaugeMonitor:name=monitorTarget");
     ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
     try {
       // Initial value < lowThreshold
       TestClass a = new TestClass(new Double(4));
       server.registerMBean(a, name);
       server.createMBean("javax.management.monitor.GaugeMonitor", monitorName, null);
       server.setAttribute(monitorName,
                           new Attribute("DifferenceMode", Boolean.TRUE));
       server.setAttribute(monitorName, new Attribute("ObservedObject", name));
       server.setAttribute(monitorName,
                           new Attribute("ObservedAttribute", "Double"));
       server.setAttribute(monitorName,
                           new Attribute("GranularityPeriod", new Long(1000L)));
       server.invoke(monitorName, "setThresholds", new Object[] {new Double(5),
                     new Double(0)}
                     , new String[] {"java.lang.Number", "java.lang.Number"});
       server.setAttribute(monitorName, new Attribute("NotifyHigh", Boolean.TRUE));
       server.setAttribute(monitorName, new Attribute("NotifyLow", Boolean.TRUE));
       server.setAttribute(monitorName,
                           new Attribute("DifferenceMode", Boolean.TRUE));
       MonitorListener listener = new MonitorListener();
       server.addNotificationListener(monitorName, listener, null, null);
       server.invoke(monitorName, "start", new Object[] {}
                     , new String[] {});
       server.setAttribute(name, new Attribute("Double", new Double(8)));

       assertNull(listener.getLastNotification());
       Thread.sleep(1500);

       assertNotNull(listener.getLastNotification());
       if (MonitorNotification.THRESHOLD_ERROR.equals(listener.getLastNotification().getType())) fail();
     }
     finally {
       server.unregisterMBean(name);
       server.unregisterMBean(monitorName);
     }
  }

  public void testBug822849() throws Exception
  {
	 // Test case for bug #822849
	 ObjectName name = new ObjectName("GaugeMonitor:name=monitorTarget");
	 ObjectName monitorName = new ObjectName("GaugeMonitor:name=monitor1");
	 try {

	   TestClass a = new TestClass(new Double(4));
	   server.registerMBean(a, name);
	   server.createMBean("javax.management.monitor.GaugeMonitor", monitorName);
	   server.setAttribute(monitorName,
						   new Attribute("DifferenceMode", Boolean.FALSE));
	   server.invoke(monitorName, "addObservedObject",
	   		new Object[]{ name }, new String[] { ObjectName.class.getName() });
	   server.setAttribute(monitorName,
						   new Attribute("ObservedAttribute", "Double"));
	   server.setAttribute(monitorName,
						   new Attribute("GranularityPeriod", new Long(250L)));
	   /*
		server.invoke(monitorName, "setThresholds", new Object[] {new Double(3),
					  new Double(0)}
					  , new String[] {"java.lang.Number", "java.lang.Number"});
	  */
	   server.invoke(monitorName, "start", new Object[] {}
		  		     , new String[] {});
	   Thread.sleep(500L);

//	   Log.getLogger(Monitor.class.getName()).setPriority(Logger.ERROR);
	   Object value = server.getAttribute(monitorName,"DerivedGauge");

	   assertEquals(new Double(4),value);
	   server.invoke(monitorName, "stop", new Object[] {}
					 , new String[] {});
	 }
	 finally {
	   server.unregisterMBean(name);
	   server.unregisterMBean(monitorName);
	 }
  }
}

