/* MatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.LinkedList;
import java.util.Queue;
import org.grinvin.GraphBundleView;
import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.computers.*;

/**
 * This class represents an invariant computers which computes
 * the matching number for a given graph. Different algorithms(and
 * thus different invariant computers) are used for bipartite and
 * non-bipartite graphs.
 */
public class MatchingNumber extends AbstractInvariantComputer {

    /**
     * This method returns the id of the invariant.
     * @return The id of the invariant.
     */
    public String getInvariantId() {
        return "org.grinvin.invariants.MatchingNumber";
    }

    /**
     * Computes the matching number of the given graph.
     * @return The matching number as a InvariantValue-object.
     */
    public InvariantValue compute(GraphBundleView bundle) {
        int[][] adjlist = bundle.adjacencyList();
        if(adjlist.length == 0)
            return new IntegerValue(0, this);
        int[] partition = new int[adjlist.length];
        if(isBipartite(adjlist, partition))
            return new IntegerValue(new BipartiteMatchingNumber().compute(adjlist), this);
        else
            return new IntegerValue(new NonBipartiteMatchingNumber().compute(adjlist),this);
    }

    /**
     * This method test if the given graph is bipartite using BFS.
     * @return true if given graph is bipartite
     */
    private boolean isBipartite(int[][] adjlist, int[] partition) {
        //int[] partition = new int[adjlist.length];
        for(int i = 0; i < partition.length; i++)
            partition[i] = -1;
        for(int v = 0; v < adjlist.length; v++) {
            if(partition[v] == -1) {
                partition[v] = 0;
                Queue<Integer> q = new LinkedList<Integer>();
                q.offer(new Integer(v));
                while(!q.isEmpty()) {
                    int vertex = q.poll().intValue();
                    for(int i = 0; i < adjlist[vertex].length; i++) {
                        int neighbour = adjlist[vertex][i];
                        if(partition[vertex] == partition[neighbour])
                            return false;
                        else {
                            if(partition[neighbour] == -1) {
                                partition[neighbour] = partition[vertex]==0 ? 1 : 0;
                                q.offer(new Integer(neighbour));
                            }
                        }
                    }
                }
            }
        }
        return true;
    }    
}
