/* VertexConnectivity.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.Arrays;
import org.grinvin.GraphBundleView;
import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * Invariant computer which returns the vertex-connectivity.
 */
public class VertexConnectivity extends AbstractInvariantComputer {
    
    public IntegerValue compute (GraphBundleView bundle) {
        //first calculate minimum degree (i.e. upperbound for vertex-connectivity)
        int[][] adjlist = bundle.adjacencyList ();
        if(adjlist.length<2)
            return new IntegerValue(0, this);
        int mindeg = Integer.MAX_VALUE;
        for (int[] neighbours : adjlist)
            if (neighbours.length < mindeg)
                mindeg = neighbours.length;  
        if(mindeg==adjlist.length-1)
            //complete graph
            return new IntegerValue (mindeg, this);
        
        boolean[] cut = new boolean[adjlist.length];
        Arrays.fill(cut, false);
        int minimumSize = mindeg;

        if(!isConnected(adjlist, cut))
            //disconnected graph
            return new IntegerValue(0, this);

        
        for(int i=0; i<adjlist.length; i++){
            cut[i]=true;
            minimumSize = findMinimumCutSize(adjlist, cut, minimumSize, 1, i);
            cut[i]=false;
        }
        
        return new IntegerValue(minimumSize, this);
    }
    
    private int findMinimumCutSize(int[][] adjlist, boolean[] cut, int minimumSize, int currentSize, int lastAddedVertex){
        if(currentSize>=minimumSize)
            return minimumSize;
        
        //System.out.println("Current size     : " + currentSize);
        //System.out.println("Last added vertex: " + lastAddedVertex);
        
        if(!isConnected(adjlist, cut)){
            //System.out.println("Not Connected: " + Arrays.toString(cut));
            //printDistancematrix(adjlist, cut);
            return currentSize;
        }
        
        for(int i = lastAddedVertex + 1; i<adjlist.length; i++){
            cut[i]=true;
            minimumSize = findMinimumCutSize(adjlist, cut, minimumSize, currentSize + 1, i);
            cut[i]=false;
        }
        
        return minimumSize;
    }
    
    private boolean isConnected(int[][] adjlist, boolean[] cut){
        int n = adjlist.length;
        int[][] dist = new int[n][n];
        
        // compute the adjacency matrix
        for(int i = 0; i<n; i++)
            if(!cut[i])
                for(int vertex : adjlist[i])
                    if(!cut[vertex]){
                        dist[i][vertex] = 1;
                    }
        
        // compute the distance matrix
        int d = 1;
        int count;
        do {
            int d1 = d++;
            count = 0;
            for (int i = 0; i < n; i++)
                for (int j = i + 1; j < n; j++)
                    if (dist[i][j] == 0)
                        for (int k = 0; k < n; k++)
                            if (dist[i][k] == 1 && dist[j][k] == d1) {
                                dist[i][j] = d;
                                dist[j][i] = d;
                                count++;
                                break;
                            }
        } while (count > 0);
        
        //check if any vertices that aren't in the possible cut are at distance infinity of each other
        for(int i = 0; i<n; i++)
            if(!cut[i])
                for(int j = i+1; j<n; j++)
                    if(!cut[j] && dist[i][j]==0)
                        return false;

        return true;
    }
    
   private void printDistancematrix(int[][] adjlist, boolean[] cut){
        int n = adjlist.length;
        int[][] dist = new int[n][n];
        
        // compute the adjacency matrix
        for(int i = 0; i<n; i++)
            if(!cut[i])
                for(int vertex : adjlist[i])
                    if(!cut[vertex]){
                        dist[i][vertex] = 1;
                    }

        // compute the distance matrix
        int d = 1;
        int count;
        do {
            int d1 = d++;
            count = 0;
            for (int i = 0; i < n; i++)
                for (int j = i + 1; j < n; j++)
                    if (dist[i][j] == 0)
                        for (int k = 0; k < n; k++)
                            if (dist[i][k] == 1 && dist[j][k] == d1) {
                                dist[i][j] = d;
                                dist[j][i] = d;
                                count++;
                                break;
                            }
        } while (count > 0);
        
        //print distance matrix
        for(int i = 0; i<n; i++)
            System.out.println(Arrays.toString(dist[i]));
    }
    
    public String getInvariantId () {
        return "org.grinvin.invariants.VertexConnectivity";
    }
    
}
