/* MinimumLocalIndependence.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.other;

import java.util.Arrays;

import org.grinvin.GraphBundleView;
import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * Invariant computer which returns the maximum local independence.
 */
public class MinimumLocalIndependence extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] adjacencyList = bundle.adjacencyList();
        int n = adjacencyList.length;
        boolean [] allowed = new boolean [n];
        int minimumLocalIndependence = Integer.MAX_VALUE;
        for(int[] neighborhood : adjacencyList){
            Arrays.fill(allowed, false);
            for(int vertex : neighborhood)
                allowed[vertex]=true;
            int localIndependence = findIndependenceNumber(0, allowed, neighborhood.length, adjacencyList, 0);
            if(minimumLocalIndependence>localIndependence)
                minimumLocalIndependence = localIndependence;
        }
        return new IntegerValue(minimumLocalIndependence, this);
    }
    
    /** */
    private int findIndependenceNumber(int size, boolean[] allowed, int numberAllowed,
            int[][] adjacencyList, int maxsize) {
        if (numberAllowed == 0) {
            if (size > maxsize)
                return size;
        } else if (numberAllowed + size < maxsize)
            return maxsize; // it will never be  possible to find a better maxsize
        else {
            int v = 0;
            while (numberAllowed != 0) {
                while (!allowed[v])
                    v++;
                allowed[v] = false;
                numberAllowed --;
                
                // remove neighbours of v from allowed
                boolean[] newAllowed = allowed.clone();
                int newNumberAllowed = numberAllowed;
                for (int nb : adjacencyList[v])
                    if (newAllowed[nb]) {
                    newAllowed[nb] = false;
                    newNumberAllowed --;
                    }
                maxsize = findIndependenceNumber(size+1,newAllowed, newNumberAllowed,
                        adjacencyList, maxsize);
            }
        }
        return maxsize;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.MinimumLocalIndependence";
    }
    
}
