/* JtIndex.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.chemical;

import org.grinvin.Edge;
import org.grinvin.GraphBundleView;
import org.grinvin.GraphView;
import org.grinvin.invariants.RealValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * computes the Jt index of the graph (related to the Balaban index).
 */
public class JtIndex extends AbstractInvariantComputer {
    
    public RealValue compute(GraphBundleView bundle) {
        GraphView graph = bundle.getGraph();
        int[][] distanceMatrix = bundle.distanceMatrix();
        int[][] adjlist = bundle.adjacencyList();
        int numberOfVertices = graph.getNumberOfVertices();
        int numberOfEdges = graph.getNumberOfEdges();
        int cyclomaticNumber = numberOfEdges - numberOfVertices + 1;
        
        //calculate ti = vertex distance degree of vertex i / degree of vertex i
        //and check for connectedness
        boolean connected = true;
        int[] t = new int[numberOfVertices];
        for(int i = 0; i < numberOfVertices; i++){
            for(int j = 0; j < numberOfVertices; j++){
                connected = connected && !(distanceMatrix[i][j]==0 && j != i);
                t[i] += distanceMatrix[i][j];
            }
            t[i] = t[i]/adjlist[i].length; 
        }
        
        if(!connected)
            return RealValue.undefinedValue(this);

        //calculate Balaban index
        double JtIndex = 0;
        for (Edge e: graph.edges())
            JtIndex += 1/Math.sqrt(t[e.getFirstEndpoint().getIndex()] * t[e.getSecondEndpoint().getIndex()]);
        JtIndex *= numberOfEdges/(cyclomaticNumber + 1);
        return new RealValue(JtIndex, this);
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.JtIndex";
    }
}
