/* BipartiteFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs.standard;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.factories.graphs.AbstractGraphFactory;

/**
 * Factory that creates complete bipartite grahs.
 */
public class BipartiteFactory extends AbstractGraphFactory {
    
    // implements AbstractGraphFactory
    @Override
    protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int order1 = ((Integer)values[0]).intValue();
        int order2 = ((Integer)values[1]).intValue();
        if (order1 < 2 || order2 < 2)
            throw new FactoryParameterException("Both parameters should be at least 2");
    }
    
    //
    @Override
    protected void createGraph(Graph graph, Embedding embedding) {
        embedding.setDimension(2);
        int order1 = ((Integer)values[0]).intValue();
        int order2 = ((Integer)values[1]).intValue();
        
        Vertex[] result1 = new Vertex[order1];
        for (int i = 0; i < order1; i++) {
            double[] coords = new double[2];
            coords[0] = -1.0;
            coords[1] = 2.0*i/(order1-1) - 1.0;
            result1[i] = graph.addNewVertex(null);
            embedding.setCoordinates(result1[i], coords);
        }
        
        Vertex[] result2 = new Vertex[order2];
        for (int i = 0; i < order2; i++) {
            double[] coords = new double[2];
            coords[0] = 1.0;
            coords[1] = 2.0*i/(order2-1) - 1.0;
            result2[i] = graph.addNewVertex(null);
            embedding.setCoordinates(result2[i], coords);
        }
        
        for (int i=0; i < order1; i++)
            for (int j=0; j < order2; j++)
                graph.addNewEdge(result1[i], result2[j], null);
    }
    
}
