/* EmbeddingLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.graphs;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.GraphView;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;

import org.jdom.Element;

/**
 * Loads an <tt>embedding.xml</tt>-section from an input stream.
 */
public class EmbeddingLoader {
    
    //
    private EmbeddingLoader() { }
    
    /**
     * Construct the embedding for the current graph from the given JDOM element.
     */
    private static void toEmbedding(Embedding embedding, Element element) throws IOFormatException {
        int dimension = Integer.parseInt(element.getAttributeValue("dimension"));
        embedding.setDimension(dimension);
        GraphView graph = embedding.getPeer();
        for (Object obj: element.getChildren("coordinates")) {
            Element c = (Element)obj;
            int id = Integer.parseInt(c.getAttributeValue("id"));
            double[] coordinates = new double[dimension];
            List dlist = c.getChildren("double");
            for (int i=0; i < dimension; i++) {
                Element d = (Element)dlist.get(i);
                coordinates[i] = Double.parseDouble(d.getAttributeValue("value"));
            }
            embedding.setCoordinates(graph.getVertex(id), coordinates);
        }
    }
    
    /**
     * Load the embedding from the input stream.
     */
    public static void load(Embedding embedding, InputStream input) throws IOException {
        toEmbedding(embedding, LoaderSaverHelper.loadElement(input));
    }

    public static void loadNonValidating(Embedding embedding, InputStream input) throws IOException {
        toEmbedding(embedding, LoaderSaverHelper.loadElementNonValidating(input, "embedding"));
    }

    
    public static void load(Embedding embedding, Element element) throws IOException {
        toEmbedding(embedding, LoaderSaverHelper.checkElement(element, "embedding"));
    }
    
}
