package com.limegroup.gnutella.gui;

import com.apple.mrj.*;
import com.apple.mrj.jdirect.*;
import com.apple.mrj.macos.carbon.*;
import com.apple.mrj.macos.frameworks.*;
import javax.swing.*;
import com.limegroup.gnutella.browser.ExternalControl;

/**
 * This class handles the Mac OS X-specific code for receiving a GURL
 * Apple event after a user clicks a magnet: URL in a browser. Copied
 * most of the source from the PreferencesItem.java LimeWire source and
 * from the AESendandReceive sample source.
 */
final class GURLHandler implements ApplicationServices, Carbon, 
	AEEventHandlerInterface {
	
	/**
	 * The application creator code, must match Info.plist!
	 */
	private static final String appCreator = "LIME";

	/**
	 * Code for the Apple event <tt>MRJOSType</tt> for registering the
	 * GURL event.
	 */
	private static final int kGURLEventClass = new MRJOSType("GURL").toInt();
	
	/**
	 * Code for the GURL event <tt>MRJOSType</tt>
	 */
	 private static final int kGURLEventID = new MRJOSType("GURL").toInt();

	/**
	 * Code for the direct parameter object.
	 */
	public static final int keyDirectObject = new MRJOSType("----").toInt();
    
	/**
	 * Code that denotes the AppleEvent text data type.
	 */
	public static final int typeChar = new MRJOSType("TEXT").toInt();

	/**
	 * Maximum length of the URL string.
	 */
	protected static final int kMaxTextSize = 255;

    /**
     * Linker for tying everything together.
     */
	private static final Object linker = new Linker(GURLHandler.class);

    /**
	 * Singleton, only one GURL handler needed per process.
	 */
	private static GURLHandler handler;

	private GURLHandler() {
	}

	public static GURLHandler getInstance() {
		if (handler == null)
			handler = new GURLHandler();
		return handler;
	}

    /**
     * Temporarily acquires the native Carbon lock to install the event handler
     * for the GURL event.
     */
	public void register() {
		try {
			CarbonLock.acquire();
			int err = 
				AEInstallEventHandler(kGURLEventClass, 
				                      kGURLEventID,
				                      new AEEventHandlerClosureUPP(this).getProc(), 
				                      0, 
				                      false);
			if(err != 0) {
				java.lang.System.err.println("Registering GURL handler: "+err);
			}
		} finally {
			CarbonLock.release();
		}
	}
	
	public short AEEventHandler(int event, int reply, int refcon) {
		int[] actualType = new int[1];
		int[] actualSize = new int[1];
		byte[] theText = new byte[kMaxTextSize];
		short err;
		PointerStruct eventPtr = new PointerStruct(event) {
			public int getSize() {
				return AppleEventStruct.sizeOfAppleEvent;
			}
		};
		AppleEventStruct aeStruct = new AppleEventStruct(eventPtr, 0);
		err = AEGetParamPtr(aeStruct.getByteArray(), keyDirectObject,
							typeChar, actualType, theText, kMaxTextSize,
							actualSize);
		final String url = new String(theText, 0, actualSize[0]);
		if ( ExternalControl.isInitialized() ) {
			Runnable GURLRunnable = new Runnable() {
				public void run() {
					java.lang.System.err.println("Got GURL event:"+url);
					ExternalControl.handleMagnetRequest(url);
				} 
			};
			SwingUtilities.invokeLater(GURLRunnable);
		} else {
		    ExternalControl.enqueueMagnetRequest(url);
		}
		return 0;
	}

	/**
	 * Native method that installs the event handler.
	 */
	private static native short AEInstallEventHandler(int eventClass, int id,
		int handler, int refcon, boolean sysHandler);

	/**
	 * Native method to extract parameters from an AppleEvent.
	 */
	private static native short AEGetParamPtr(byte [] event, int AEKeyWord,
		int desiredType, int [] actualType, byte [] dataPtr,
		int maxSize, int [] actualSize);
}

