package com.limegroup.gnutella.update;

import java.net.*;
import java.io.*;

/**
 * This class handles accessing the servlet, sending it data about the client
 * configuration, and obtaining information for the update.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class ServletAccessor {

	/**
	 * Variable for the object that contains information retrieved from
	 * the servlet.  
	 */
	private volatile ClientRemoteUpdateInfo _remoteInfo = 
		new ClientRemoteUpdateInfo();
    
    /**
	 * Constant number of milliseconds to wait before timing out the
	 * connection to the servlet.
	 */
	private final int CONNECT_TIMEOUT = 2000;
    
	/**
	 * The url for connecting to the servlet.
	 */
	private final String SERVLET_URL = 
		"http://version.limewire.com:8080/update/servlet/UpdateHandler";


	/**
	 * Variable for whether or not the servlet has timed out.
	 */
	private volatile boolean _timedOut = false;

	/**
	 * Package-access constructor
	 */
	ServletAccessor() {}

	/**
	 * Contacts the application servlet and sends it the information 
	 * contained in the <tt>ClientLocalUpdateInfo</tt> object.  This method 
	 * also builds a <tt>ClientRemoteUpdateInfo</tt> object from the information 
	 * obtained from the servlet.
	 *
	 * @return a <tt>ClientRemoteUpdateInfo</tt> object that encapsulates the 
	 *         appropriate update action to take
	 * @param localInfo is an object encapsulating information about the
	 *                  local machine to send to the remote server
	 */
	synchronized ClientRemoteUpdateInfo getRemoteUpdateInfo(ClientLocalUpdateInfo localInfo) {
		Thread getRemoteInfoThread = 
		    new Thread(new RemoteInfoHandler(localInfo), "RemoteInfoHandler");
		getRemoteInfoThread.start();
		try {
			this.wait(CONNECT_TIMEOUT);
		} catch(InterruptedException ie) {
		}
		
		_timedOut = true;		
		return _remoteInfo;
	}

	/** 
	 * This class handles sending the client update data to the servlet
	 * and retrieving the response.  If it gets the response in time,
	 * it will set the ClientRemoteUpdateInfo object for this class to be the
	 * object it has constructed from the servlet data.
	 */
	private class RemoteInfoHandler implements Runnable {
		
		/**
		 * Variable for the object containing information for the local
		 * configuration.
		 */
		private ClientLocalUpdateInfo _localInfo;

		/** 
		 * The constructor simply sets the <tt>ClientLocalUpdateInfo</tt> 
		 * object for use in contacting the server.
		 */
		private RemoteInfoHandler(ClientLocalUpdateInfo localInfo) {
			_localInfo = localInfo;
		}

		/**
		 * Implements the Runnable interface.
		 *
		 * Performs server access, input and output.
		 */
		public void run() {
			String postRequestString = _localInfo.getPostRequestString();
			URL url;

			// initialized these to null so that we can close them later
			URLConnection urlConn = null;
			DataOutputStream out = null;
			BufferedReader in = null;
			ClientRemoteUpdateInfo remoteInfo = new ClientRemoteUpdateInfo();
			
			try {			
				// create the URL object for the servlet url
				url = new URL(SERVLET_URL);

				// open the connection to the servlet
				urlConn = url.openConnection();
            
				// set the connection so that we can do input and output
				urlConn.setDoOutput(true);				
				urlConn.setDoInput(true);
				
				// set no caching
				urlConn.setUseCaches(false);
				
				// specify the content type
				urlConn.setRequestProperty("Content-Type", 
										   "application/x-www-form-urlencoded");
				
				// specify the content length
				urlConn.setRequestProperty("Content-Length", "" 
										   + postRequestString.length());			
				// get the outpost stream
				out = new DataOutputStream(urlConn.getOutputStream());
				
				// write the POST Data
				out.writeBytes(postRequestString);
				out.flush();
				out.close();
				
				// get the input stream from the servlet connection
				InputStream is = urlConn.getInputStream();
				
				// get the input stream reader from the servlet
				InputStreamReader isr = new InputStreamReader(is);
				
				// create the buffered reader to read from the stream
				in = new BufferedReader(isr);           
				
				StringBuffer lineBuffer = new StringBuffer();
				String line;
				while((line = in.readLine()) != null) {
					lineBuffer.append(line + "\r\n");
				}
				
				// send the data string from the servlet to the ClientRemoteUpdateInfo
				// object for processing.
				remoteInfo.addRemoteInfo(lineBuffer.toString());				
			} catch(MalformedURLException murle) {
				// the url was unusable for some reason
			} catch(IOException ioe) {
				// io error making the connection or in parsing the data from
                // the server.
			}

			finally {
				// close the streams.				
				try {
					if(out != null) {
						out.flush();
						out.close();
					}
				} catch(IOException ioe) {
				}
				try {
					if(in != null) in.close();						
				} catch(IOException ioe) {
				}
				
				// close the urlConnection
				if(urlConn instanceof HttpURLConnection) 
					((HttpURLConnection)urlConn).disconnect();

				// obtain the lock on this ServletAccessor object, notifying
				// any waiting theads as necessary and performing general 
				// clean up.
				synchronized(ServletAccessor.this) {
					// if it has not timed out, notify the waiting thread
					// and reset the _remoteInfo object
					if(!_timedOut) {
						_remoteInfo = remoteInfo;
						ServletAccessor.this.notify();
					}
				}
			}
		}
	}
}
