package com.limegroup.gnutella.gui.search;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.search.*;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.settings.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import com.sun.java.util.collections.*;

/**
 * This class handles the dislay of search results.
 */
final class SearchResultDisplayer implements ThemeObserver {
	
    //The minimum connection speed and search label are not currently 
    //displayed, though they are still added to the window and 
    //updated.
    private JLabel searchLabel = new JLabel("        ");

	/**
	 * <tt>JPanel</tt> containting the primary components of the search result
	 * display.
	 */
	private final JPanel MAIN_PANEL = new BoxPanel(BoxPanel.Y_AXIS);

	/**
	 * The main tabbed pane for displaying different search results.
	 */
	private final JTabbedPane TABBED_PANE = new JTabbedPane();

    /** 
     * <tt>Timer</tt> for changing the searching icon when 
	 * SEARCH_ANIMATION_TIME (from <tt>SettingsManager</tt>) has expired.
     */
    private final Timer ANIMATION_TIMER;

	/** 
	 * Constant for the <tt>CancelSearchIconProxy</tt> that displays the 
	 * kill search icon in a search tab.
	 */
    private final CancelSearchIconProxy KILL_ICON =
         new CancelSearchIconProxy();

    /** The contents of tabbedPane. 
     *  INVARIANT: entries.size()==# of tabs in tabbedPane 
     *  LOCKING: +obtain entries' monitor before adjusting number of 
     *            outstanding searches, i.e., the number of tabs
     *           +obtain a ResultPanel's monitor before adding or removing 
     *            results + to prevent deadlock, never obtain ResultPanel's
     *            lock if holding entries'.
     */
    private static final com.sun.java.util.collections.List /* of ResultPanel */ entries =
		new LinkedList();

    /** Results is a panel that displays either a JTabbedPane when lots of
     *  results exist OR a blank ResultPanel when nothing is showing.
     *  Use switcher to switch between the two.  The first entry is the
     *  blank results panel; the second is the tabbed panel. */
    private final JPanel results = new JPanel();
    private final CardLayout switcher = new CardLayout();

    /** A popup menu to show the user various options */
    private JPopupMenu menu;

    private ResultPanel dummyResultPanel;

	/** Flag that a search has been stopped with a random GUID */
	private static final GUID  stoppedGUID = new GUID(GUID.makeGuid());
    

	/**
	 * Constructs the search display elements.
	 */
	SearchResultDisplayer() {
        MAIN_PANEL.setMinimumSize(new Dimension(0,0));
        ANIMATION_TIMER = 
            new Timer(QueryHandler.MAX_QUERY_TIME, new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    try {
                        GUIMediator.instance().setSearching(false);
                    } catch(Throwable e) {
                        GUIMediator.showInternalError(e,
                            "SearchResultDisplayer");
                    }
                }
            });
        ANIMATION_TIMER.setRepeats(false);

        // make the results panel take up as much space as possible
        // for when the window is resized. 
        results.setPreferredSize(new Dimension(10000, 10000));
        results.setLayout(switcher);
        JPanel dummy=new JPanel();
        dummy.setLayout(new BorderLayout());
        JLabel resultLabel=new JLabel
               (GUIMediator.getStringResource("SEARCH_RESULT_TABLE_LABEL"));
        //This padding puts the top of the empty table at the same
        //height as normal result tables.  Yup, it's a bit of a hack.
        resultLabel.setBorder(BorderFactory.createEmptyBorder(10,0,0,0));
		dummyResultPanel=new ResultPanel();
        dummy.add(dummyResultPanel,BorderLayout.CENTER);
        results.add("dummy",dummy);
        results.add("tabbedPane",TABBED_PANE);
        switcher.first(results);    
        // --------------------------------------------------------------------*

        MAIN_PANEL.add(results);

        TABBED_PANE.addMouseListener(new TabbedPaneMouseListener());
	}

    /** 
     * @modifies tabbed pane, entries
     * @effects adds an entry for a search for stext with GUID guid
     *  to the tabbed pane.  This is used both for normal searching 
     *  and browsing.  Returns the ResultPanel added.
     */
    ResultPanel addResultTab(GUID guid, String stext, String richQuery,
                             boolean isBrowseHostTab) {
		MediaType type = SearchMediator.getSelectedMediaType();
		final String label=stext;
		final ResultPanel panel=new ResultPanel(guid, stext, type,
												richQuery, isBrowseHostTab);

        searchLabel.setText(GUIMediator.getStringResource
                            ("SEARCH_SEARCHING_LABEL"));
        //searchField.setText("");
		SearchMediator.setSearchString("");
        synchronized (entries) {
            entries.add(panel);
            TABBED_PANE.addTab(label, KILL_ICON, panel);        
            TABBED_PANE.setSelectedIndex(entries.size()-1);
					
            //Remove an old search if necessary
            if (entries.size() 
                > SearchSettings.PARALLEL_SEARCH.getValue()) {
                killSearchAtIndex(0);
            }
        }
        GUIMediator.instance().setSearching(true);
        ANIMATION_TIMER.restart();
        switcher.last(results);  //show tabbed results

        // If there are lots of tabs, this ensures everything
        // is properly visible. 
        MAIN_PANEL.revalidate();
			
        return panel;
    }


    /**
     * If i rp is no longer the i'th panel of this, returns silently. Otherwise
     * adds line to rp under the given group.  Updates the count on the tab in
     * this and restarts the spinning lime.
     *     @requires this is called from Swing thread, group is null or similar
     *      to line and already in rp
     *     @modifies this 
     */
    void addQueryResult(byte[] replyGUID, TableLine line, TableLine group,
                        ResultPanel rp) {

        //Actually add the line.   Must obtain rp's monitor first.
        synchronized (rp) {
            if(!rp.matches(new GUID(replyGUID)))//GUID of rp!=replyGuid
                throw new IllegalArgumentException("guids don't match");
            rp.addLine(line, group);
        }

        int resultPanelIndex = -1;
        synchronized (entries) {
            // Search for the ResultPanel to verify it exists.
            resultPanelIndex = entries.indexOf(rp);

            // If we couldn't find it, silently exit.
            if( resultPanelIndex == -1 ) return;
            
            //Update index on tab.  Don't forget to add 1 since line hasn't
            //actually been added!
            TABBED_PANE.setTitleAt(resultPanelIndex,
                rp.getQuery() + " ("+rp.numResults() + ")");
        }
        
        // make sure the lime keeps spinning
        GUIMediator.instance().setSearching(true);
    }

	/**
	 * Shows the popup menu that displays various options to the user.
	 *
	 * @param comp the component invoking the command to show the menu
	 * @param x the x position on the screen to diplay the menu 
	 * @param y the y position on the screen to diplay the menu 
	 */
	void showMenu(Component comp, int x, int y) {
        ResultPanel rp;
        synchronized(entries){
            int i=TABBED_PANE.getSelectedIndex();
            if(i==-1)
                return;
            try{
                rp = (ResultPanel)entries.get(i);
            }catch(IndexOutOfBoundsException e){
                return;
            }
        }
        menu = rp.getPopup();
        menu.show(comp, x, y);
    }

	/**
	 * Returns the currently selected <tt>ResultPanel</tt> instance.
	 *
	 * @return the currently selected <tt>ResultPanel</tt> instance,
	 *  or <tt>null</tt> if there is no currently selected panel
	 */
	ResultPanel getSelectedResultPanel() {
		return (ResultPanel)TABBED_PANE.getSelectedComponent();		
	}

    /**
     * Returns the <tt>ResultPanel</tt> for the specified GUID.
     *
     * @param rguid the guid to search for
     * @return the ResultPanel that matches the specified GUID, or null
     *  if none match.
     */
     ResultPanel getResultPanelForGUID(GUID rguid) {
        synchronized (entries) {
            for (int i=0; i<entries.size(); i++) {
                ResultPanel rp = (ResultPanel)entries.get(i);
                if (rp.matches(rguid)) //order matters: rp may be a dummy guid.
                    return rp;
            }
        }
		return null;
	}

	/**
	 * Returns the <tt>ResultPanel</tt> at the specified index.
	 * 
	 * @param index the index of the desired <tt>ResultPanel</tt>
	 * @return the <tt>ResultPanel</tt> at the specified index
	 */
	ResultPanel getPanelAtIndex(int index) {
		synchronized(entries) {
			return (ResultPanel)entries.get(index);
		}
	}

	/**
	 * Returns the index in the list of search panels that corresponds
	 * to the specified guid, or -1 if the specified guid does not
	 * exist.
	 *
	 * @param rguid the guid to search for
	 * @return the index of the specified guid, or -1 if it does not
	 *  exist.
	 */
	int getIndexForGUID(GUID rguid) {
        synchronized (entries) {
            for (int i=0; i<entries.size(); i++) {
                ResultPanel rp = (ResultPanel)entries.get(i);
                if (rp.matches(rguid)) //order matters: rp may be a dummy guid.
                    return i;
            }
        }
		return -1;
	}

    /**
     * @modifies tabbed pane, entries
     * @effects removes the currently selected result window (if any)
     *  from this
     */
    void killSearch() {
        synchronized (entries) {
            int i=TABBED_PANE.getSelectedIndex();
            if (i==-1)  //nothing selected?!
                return;
            killSearchAtIndex(i);
        }   
    }


    /**
     * @modifies tabbed pane, entries
     * @effects removes the window at i from this
     */
    void killSearchAtIndex(int i) {
        synchronized (entries) {
            ResultPanel killed = (ResultPanel) entries.remove(i);
            GUIMediator.removeThemeObserver(killed);
            RouterService.stopQuery(new GUID(killed.getGUID()));
            TABBED_PANE.removeTabAt(i);
            if (entries.size()==0) {
                switcher.first(results); //show dummy table
				GUIMediator.instance().setSearching(false);
            }
			else 
			    this.checkToStopLime();
        }   
    }


    void browseHostFailed(GUID guid) {
        int i = getIndexForGUID(guid);
        if (i > -1) {
            ResultPanel rp = getPanelAtIndex(i);
            GUIMediator.showError("ERROR_BROWSE_HOST_FAILED_BEGIN_KEY",
                                  rp.getQuery(), 
                                  "ERROR_BROWSE_HOST_FAILED_END_KEY",
                                  QuestionsHandler.BROWSE_HOST_FAILED);
            killSearchAtIndex(i);
        }
    }


    /**
     * @modifies result tab
     * @effects resets the GUID of the result tab so that all future results are
     *  thrown out.  If all searches are stopped, then the Lime stops spinning.
     */
    void stopSearch() {
        synchronized (entries) {
            int i=TABBED_PANE.getSelectedIndex();
            if (i==-1)  //nothing selected?!
                return;
			ResultPanel panel = (ResultPanel) entries.get(i);
            RouterService.stopQuery(new GUID(panel.getGUID()));
			panel.setGUID(stoppedGUID);

			this.checkToStopLime();
        }   
    }

    /**
     * @modifies spinning lime state
     * @effects If all searches are stopped, then the Lime stops spinning.
     */
	private void checkToStopLime() {

		ResultPanel panel;

		// Decide if we definitely can stop the lime
		boolean stopLime = true;
		for (int i=0; i<entries.size(); i++) {
			panel = (ResultPanel)entries.get(i);
			stopLime &= panel.matches(stoppedGUID);
		}
		if ( stopLime ) {
			GUIMediator.instance().setSearching(false);
		}
	}

    /**
     * called by ResultPanel when the views are changed. Used to set the
     * tab to indicate the correct number of TableLines in the current
     * view.
     */
    void setTabDisplayCount(ResultPanel rp){
        Object panel;
        int i=0;
        boolean found = false;
        synchronized(entries){
            for(;i<entries.size();i++){//safe its synchronized
                panel = entries.get(i);
                if (panel == rp){
                    found = true;
                    break;
                }
            }
            if(found)//find the number of lines in model
                TABBED_PANE.setTitleAt(i,rp.getQuery()+" ("+(rp.numResults())+")");
        }
    }

	/**
	 * Returns the <tt>Icon</tt> object for the selected tab in the 
	 * <tt>JTabbedPane</tt>.
	 * @return the <tt>Icon</tt> for the selected tab
	 */
	Icon getSelectedIcon() {
		return TABBED_PANE.getIconAt(TABBED_PANE.getSelectedIndex());
	}

	/**
	 * Accessor for the <tt>ResultPanel</tt> instance that shows no active
	 * searches.
	 *
	 * @return the <tt>ResultPanel</tt> instance that shows no active
	 * searches
	 */
    ResultPanel getDummyResultPanel(){
		return dummyResultPanel;
    }

	/**
	 * Restarts the <tt>Timer</tt> instance that controls the search status 
	 * indicator (the spinning lime).
	 *
	 * @return  the <tt>Timer</tt> instance that controls the search status 
	 * indicator (the spinning lime)
	 */
	void restartTimer() {
		ANIMATION_TIMER.restart();
	}

	/**
	 * Returns the <tt>JComponent</tt> instance containing all of the search
	 * result ui components.
	 *
	 * @return the <tt>JComponent</tt> instance containing all of the search
	 *  result ui components
	 */
	JComponent getComponent() {
		return MAIN_PANEL;
	}

	// inherit doc comment
	public void updateTheme() {
        KILL_ICON.updateTheme();
		dummyResultPanel.updateTheme();
		int numResultPanels = TABBED_PANE.getTabCount();
		for(int i=0; i<numResultPanels;i++) {
			ResultPanel curPanel = (ResultPanel)TABBED_PANE.getComponentAt(i);
			curPanel.updateTheme();
		}
	}
}













