package com.limegroup.gnutella.gui.search;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.search.*;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.downloader.*;
import com.limegroup.gnutella.settings.*;
import com.sun.java.util.collections.List;
import com.sun.java.util.collections.ArrayList;
import com.sun.java.util.collections.Iterator;
import com.sun.java.util.collections.HashMap;
import com.sun.java.util.collections.Set;
import java.util.Locale;
import java.io.*;
import com.limegroup.gnutella.xml.gui.*;
import com.limegroup.gnutella.xml.*;

import javax.swing.*;
import java.awt.*;

/**
 * This class acts 
as a mediator between the various search components --
 * the hub that all traffic passes through.  This allows the decoupling of
 * the various search packages and simplfies the responsibilities of the
 * underlying classes.
 */
public final class SearchMediator implements ThemeObserver {

    static final String DOWNLOAD_STRING =
        GUIMediator.getStringResource("SEARCH_DOWNLOAD_BUTTON_LABEL");
    static final String KILL_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_KILL_STRING");
    static final String STOP_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_STOP_STRING");
    static final String LAUNCH_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_LAUNCH_STRING");
    static final String BROWSE_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_BROWSE_STRING");
    static final String CHAT_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_CHAT_STRING");
    static final String REPEAT_SEARCH_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_REPEAT_SEARCH_STRING");
    static final String BROWSE_HOST_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_BROWSE_STRING");
    static final String BITZI_LOOKUP_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_BITZI_LOOKUP_STRING");
    static final String BLOCK_STRING =
        GUIMediator.getStringResource("SEARCH_PUBLIC_BLOCK_STRING");

    /** 
     * Define how many rows are allowed based on platform
     * Note:  Mac goes out of memory quickly. 
     */
    private static int MAX_RESULTS_TO_DISPLAY =
        CommonUtils.isMacClassic() ? 1000 : 3000;

    /**
     * The index of the download button in the button row.
     */
    static final int DOWNLOAD_BUTTON_INDEX =
        SearchButtons.DOWNLOAD_BUTTON_INDEX;

    /**
     * The index of the chat button in the button row.
     */
    static final int CHAT_BUTTON_INDEX =
        SearchButtons.CHAT_BUTTON_INDEX;

    /**
     * This object is used to synchronize the code that causes a repeat
     * search to be initiated, and the code that adds lines to the
     * resultPanel.
     * <p>
     * If the code in these two blocks are interleaved, the code that
     * initiates a repeatSearch - could clear the grouper, and not the
     * model yet, before another line is added to the grouper.
     * <p> 
     * Therefore the code in ResultPanel.repeatSearch() and the code in
     * SearchView.handleQueryReplyInternal, have to be atomic with respect
     * to each other. They are both synchronized on this object.
     */
    public static final Object REPEAT_SEARCH_LOCK = new Object();

    /**
     * Variable for the component that handles all search input from the user.
     */
    private static final SearchInputManager INPUT_MANAGER =
        new SearchInputManager();

    /**
     * This instance handles the display of all search results.
     */
    private static final SearchResultDisplayer RESULT_DISPLAYER =
        new SearchResultDisplayer();

    /**
     * Constant for the class that receives search results and pipelines
     * them.
     */
    //private static final SearchResultPipeliner RESULT_PIPELINER =
    //    new SearchResultPipeliner();

    /**
     * Constructs the UI components of the search result display area of the 
     * search tab.
     */
    public SearchMediator() {
        // Set the splash screen text...
        final String splashScreenString =
            GUIMediator.getStringResource("SPLASH_STATUS_SEARCH_WINDOW");
        GUIMediator.setSplashScreenString(splashScreenString);
        GUIMediator.addThemeObserver(this);
    }

    /**
     * Shows the popup menu that displays various options to the user.
     *
     * @param comp the component invoking the command to show the menu
     * @param x the x position on the screen to diplay the menu 
     * @param y the y position on the screen to diplay the menu 
     */
    static void showMenu(Component comp, int x, int y) {
        RESULT_DISPLAYER.showMenu(comp, x, y);
    }

    /**
     * Returns the Icon object for the selected tab in the JTabbedPane.
     * @return the Icon for the selected tab.
     */
    static Icon getSelectedIcon() {
        return RESULT_DISPLAYER.getSelectedIcon();
    }

    /** 
     * Repeats the given search.  type and richQuery may be null.
     * even schemaURI may be null.
     */
    static byte[] repeatSearch(MediaType type, String stext,
                               String richQuery) {
        if (stext.equals(""))
            return null;

        if (validateQueryString(stext) == false)
            return null;

        // 1. Update panel with new GUID
        byte [] guidBytes = RouterService.newQueryGUID();
        final GUID newGuid = new GUID(guidBytes);

        SearchMediator.setSearchString("");

        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if(rp == null) {
            throw new NullPointerException("result panel is null");
        }
        RouterService.stopQuery(new GUID(rp.getGUID()));
        rp.setGUID(newGuid);
        GUIMediator.instance().setSearching(true);
        RESULT_DISPLAYER.restartTimer();

        // 2. Kick off backend search.
        if (richQuery == null)
            // Normal search
            RouterService.query(guidBytes, stext, type);
        else 
            // XML search
            RouterService.query(guidBytes, stext, richQuery, type);

        return guidBytes;
    }

    /**
     * Adds the passed search to the GUI, by creating a new Result Tab 
     * for that
     * @param queryText The text of the query to be added
     * @param guid GUID of the search
     * @param isBrowseHost If the tab is going to support browse host
     */
    private static void addSearchInGui(String queryText, GUID guid, 
                                       boolean isBrowseHost) {
        if (isBrowseHost)
            addBrowseHostTab(guid, queryText);
        else
            addResultTab(guid, queryText);
    }

    /**
     * Browses the first selected host. Fails silently if couldn't browse.
     */
    static void doBrowseHost() {
        // Get the appropriate tableline....
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) return;
        if (!rp.areRowsSelected()) return;
        TableLine line = rp.getFirstSelectedTableLine();
        if (line == null) return;

        // See if it is firewalled
        RemoteFileDesc rfd = line.toRemoteFileDesc();
        byte[] serventIDBytes = rfd.getClientGUID();
        final Set proxies = rfd.getPushProxies();
        final GUID serventID = new GUID(serventIDBytes);

        // Get the host's address....
        final String host = line.getBrowseHostEnabledHost();
        final int port = line.getBrowseHostEnabledPort();
        if (host == null || port == -1)
            return;
        
        Thread thread = new Thread("BrowseHoster") {
                public void run() {
                    doBrowseHost2(host, port, serventID, proxies);
                }
            };
        
        thread.setDaemon(true);
        thread.start();
     }

    /**
     * Allows for browsing of a host from outside of the search package.
     */
    public static void doBrowseHost(final RemoteFileDesc rfd) {
        Thread thread = new Thread("BrowseHoster") {
            public void run() {
                doBrowseHost2(rfd.getHost(), rfd.getPort(),
                              new GUID(rfd.getClientGUID()),
                              rfd.getPushProxies());
            }
        };
        
        thread.setDaemon(true);
        thread.start();
    }

    /**
     * Allows for browsing of a host from outside of the search package
     * without an rfd.
     */
    public static void doBrowseHost(final String host, final int port,
                                    final GUID guid) {
        Thread thread = new Thread("BrowseHoster") {
            public void run() {
                if (guid == null)
                    doBrowseHost2(host, port, null, null);
                else
                    doBrowseHost2(host, port, new GUID(guid.bytes()), null);
            }
        };
        thread.setDaemon(true);
        thread.start();
    }

    /**
     * Re-browses the host.  Fails silently if browse failed...
     * TODO: WILL NOT WORK FOR RE-BROWSES THAT REQUIRES A PUSH!!!
     */
    static void reBrowseHost(final String host, final int port,
                             ResultPanel in) {
        // Update the GUID
        final GUID guid = new GUID(GUID.makeGuid());
        in.setGUID(guid);
        Thread thread = new Thread("BrowseHoster") {
                public void run() {
                    RouterService.doBrowseHost(host, port, guid, 
                                         new GUID(GUID.makeGuid()), null);
                }
            };
        thread.setDaemon(true);
        thread.start();
    }

    /**
     * Browses the passed host at the passed port.
     * Fails silently if couldn't browse.
     * @param host The host to browse
     * @param port The port at which to browse
     */
    static private void doBrowseHost2(String host, int port,
                                      GUID serventID, 
                                      Set proxies) {
        // Update the GUI
        GUID guid = new GUID(GUID.makeGuid());
        addSearchInGui(host + ":" + port, guid, true);
        // Do the actual browse host

        RouterService.doBrowseHost(host, port, guid, serventID, proxies);
    }

    /**
     * Call this when a Browse Host fails.
     * @param guid The guid associated with this Browse. 
     */
    public static void browseHostFailed(GUID guid) {
        RESULT_DISPLAYER.browseHostFailed(guid);
    }

    /**
     * Triggers a search for the given text.  For testing purposes returns
     * the 16-byte GUID of the search or null if the search didn't happen
     * because it was greedy, etc.
     */
    public static byte[] triggerSearch(String stext) {
        //searchField.setText(stext);
        return triggerSearch();
    }

    private static boolean isGreedy(String search) {
        // Eli filter.
        if (search.equals("*.*"))
            return true;

        // Turns "*.mp3", "*MP3*" to "mp3".
        // This is a quick-and-dirty implementation.
        // TODO: catch a.mp3 as well.
        String canonical = search.toLowerCase(Locale.US).
            replace('*', ' ').replace('.', ' ').trim();
        return (canonical.equals("mp3")  || canonical.equals("mpg")
            ||  canonical.equals("asf")  || canonical.equals("jpg") 
            ||  canonical.equals("mpeg") || canonical.equals("gif")
        // Shouldn't we detect these too ?
        //  ||  canonical.equals("avi")  || canonical.equals("txt")
        //  ||  canonical.equals("htm")  || canonical.equals("html")
        //  ||  canonical.equals("xml")  || canonical.equals("url")
        //  ||  canonical.equals("zip")  || canonical.equals("exe")
            );
    }

    /**
     * Triggers a search given the text in the search field.  For testing
     * purposes returns the 16-byte GUID of the search or null if the search
     * didn't happen because it was greedy, etc.  
     */
    static byte[] triggerSearch() {
        String stext = INPUT_MANAGER.getSearchString();
        if (stext.equals(""))
            return null;

        if (validateQueryString(stext) == false) return null;

        // 1. Update GUI.  (Change label, add new table.)
        byte[] guid=RouterService.newQueryGUID();
        INPUT_MANAGER.storeSearch(); // add to autocomplete
        addResultTab(new GUID(guid), stext);

        // 2. Start backend.
        MediaType type = INPUT_MANAGER.getSelectedMediaType();
        RouterService.query(guid, stext, type);

        return guid;
    }

    /**
     * Triggers an XML search, obtaining the XML data from the specified
     * <tt>InputPanel</tt> instance.
     *
     * @param inputPanel the <tt>InputPanel</tt> instance containing the
     *  search criteria
     * @return the guid for the query, used in testing
     */
    static byte[] triggerSearchXML(InputPanel inputPanel) {
        // In this case we would like to construct the search
        // string from the first rich field populated by the user.
        String regString = inputPanel.getStandardQuery();
        if (regString == null || regString.equals(""))
            return null; // user never entered anything anywhere

        // TODO3: Validate fields of search
        String xmlString = inputPanel.getInput();
        
        //System.out.println("Sumeet: " + xmlString);

        // Other info
        byte[] guidByte=RouterService.newQueryGUID();

        // Try to get a type, we don't want to get irrelevant files for
        // audio, video meta-searches...
        MediaType type = null;
        String schemaURI = inputPanel.getSchemaURI();
        if ((LimeXMLSchema.getDisplayString(schemaURI)).equals
                                     (XMLStringUtils.AUDIO_SCHEMA_TAG))
            type = MediaType.getAudioMediaType();
        else if ((LimeXMLSchema.getDisplayString(schemaURI)).equals
                                     (XMLStringUtils.VIDEO_SCHEMA_TAG))
            type = MediaType.getVideoMediaType();

        if (xmlString == null) {
            // I am going to ignore the 2 letter limit imposed on searches
            GUID guid = new GUID(guidByte);
            INPUT_MANAGER.storeSearch();
            addResultTab(guid,regString);
            RouterService.query(guidByte, regString, type);
        } else {
            // Calls the new overloaded version of RouterService.query(...)
            GUID guid = new GUID(guidByte);
            inputPanel.storeInput();
            addResultTab(guid,regString,xmlString,false);
            RouterService.query(guidByte, regString, xmlString, type);
        }
        return guidByte;
    }

    private static boolean validateQueryString(String stext) {
        if (stext.length() <= 2 && !(stext.length() == 2 && 
             ((Character.isDigit(stext.charAt(0)) && 
               Character.isLetter(stext.charAt(1)))   ||
              (Character.isLetter(stext.charAt(0)) && 
               Character.isDigit(stext.charAt(1)))))) {
            GUIMediator.showError("ERROR_THREE_CHARACTER_SEARCH");
            return false;
        } else if (isGreedy(stext)) {
            String msgKey = "ERROR_NETWORK_FLOODING";
            int response = GUIMediator.showYesNoMessage(msgKey);
            if (response != GUIMediator.YES_OPTION) return false;
        }

        return true;
    }

    /**
     * This method has the signature of the old method. The two methods
     * will be called from different places. The idea is that the new
     * method that also takes a LimeXMLSchema as a parameter, should
     * behave a exactly like before if the schema is null.
     * <p>
     * Otherwise the behaviour is different.
     */
    private static ResultPanel addResultTab(GUID guid, String stext) {
        return addResultTab(guid, stext, null, false);
    }

    private static ResultPanel addBrowseHostTab(GUID guid, String stext) {
        return addResultTab(guid, stext, null, true);
    }

    /** 
     * @modifies tabbed pane, entries
     * @effects adds an entry for a search for stext with GUID guid
     *  to the tabbed pane.  This is used both for normal searching 
     *  and browsing.  Returns the ResultPanel added.
     */
    private static ResultPanel addResultTab(GUID guid, String stext,
                                            String richQuery, 
                                            boolean isBrowseHostTab) {
        return RESULT_DISPLAYER.addResultTab(guid, stext, richQuery,
                                             isBrowseHostTab);
    }

    
    /**
     * If i rp is no longer the i'th panel of this, returns silently.
     * Otherwise adds line to rp under the given group.  Updates the count
     * on the tab in this and restarts the spinning lime.
     * @requires this is called from Swing thread, group is null or
     *  similar to line and already in rp
     * @modifies this
     */
    public static void handleQueryResult(RemoteFileDesc rfd,  HostData data) {
        synchronized(SearchMediator.REPEAT_SEARCH_LOCK) {
            byte[] replyGUID = data.getMessageGUID();
            ResultPanel rp =
                SearchMediator.getResultPanelForGUID(new GUID(replyGUID));
                
            if (rp == null) return;

            // Don't load more than a defined number of results.
            if (rp.numResults() >= MAX_RESULTS_TO_DISPLAY)
                return; // false;

            TableLine line = new TableLine(rfd,data.isMeasuredSpeed());

            TableLineGrouper grouper = rp.getGrouper();
            // OK we created a Line out of a response.
            // Do the grouping.  This is expensive!  May return null.
            TableLine group = grouper.match(line);
            if (group == null)
                grouper.add(line);

            RESULT_DISPLAYER.addQueryResult(replyGUID,line, group, rp);
        }
    }

    /**
     * Downloads the selected files in the currently displayed
     * <tt>ResultPanel</tt> if there is one.
     */
    static void doDownload() {
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) return;
        if (rp.isEmpty()) {
            SearchMediator.doRequeryDownload();
        } else {
            SearchMediator.downloadAll(rp);
        }
    }

    /**
     * Gets the appropriate result panel, closes it, but spawns a
     * RequeryDownloader to service the 'potential' download.
     */
    private static void doRequeryDownload() {
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) return;
        try {
            RouterService.download(rp.getQuery(),
                                   rp.getRichQuery(),
                                   rp.getGUID(),
                                   rp.getMediaType());
        } catch (AlreadyDownloadingException e) {
            // Give up.
            GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
                                  "\"" + e.getFilename() + "\".",
                                  QuestionsHandler.ALREADY_DOWNLOADING);
        }
    }

    /**
     * Opens a chat session with the first chat-enabled host in the first
     * selected <tt>TableLine</tt> (either the host for the
     * <tt>TableLine</tt> itself, or for one of its children if it is a
     * parent).  If niether the parent nor any of its children are
     * chat-enabled, this method fails silently.
     */
    static void doChat() {
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) return;
        if (!rp.areRowsSelected()) return;
        TableLine line = rp.getFirstSelectedTableLine();
        if (line == null) return;
        line.doChat();
    }

    /**
     * Fires the external web browser to the Bitzi info page
     * on the selected file's SHA1 value
     */
    static void doBitziLookup() {
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) return;
        if (!rp.areRowsSelected()) return;
        TableLine line = rp.getFirstSelectedTableLine();
        if (line == null) return;
        line.doBitziLookup();
    }

    /**
     * blocks the host that sent the selected result.
     */
    static void blockHost() {
        ResultPanel rp = RESULT_DISPLAYER.getSelectedResultPanel();
        if (rp == null) 
            return;
        if (!rp.areRowsSelected())
            return;
        TableLine line = rp.getFirstSelectedTableLine();
        if (line == null)
            return;
        String host = line.getHostname();
        if (host != null) {
            String[] bannedIps = FilterSettings.BLACK_LISTED_IP_ADDRESSES.getValue();
            // Ignore if this host is already banned.
            for (int i = 0; i < bannedIps.length; i++)
                if (host.equalsIgnoreCase(bannedIps[i]))
                    return;
            String[] newBannedIps = new String[bannedIps.length + 1];
            System.arraycopy(bannedIps, 0, newBannedIps, 0,
                             bannedIps.length);
            newBannedIps[bannedIps.length] = host;
           FilterSettings.BLACK_LISTED_IP_ADDRESSES.setValue(newBannedIps);
            RouterService.adjustSpamFilters();
        }
    }

    /**
     * .
     */
    private static void downloadAll(ResultPanel rp) {

        List /* of RemoteFileDesc[] */ downloaders=new ArrayList();
        synchronized (rp) {
            // 1. Put files in buckets.  Note that children are stuck
            // together if their parent is selected...
            Iterator /* of TableLine */ selected = rp.getSelectedRows();
            while (selected.hasNext()) {
                TableLine line = (TableLine)selected.next();
                // The next method can return null
                if (line == null) continue;

                // If non-leaf, add children instead of this. 
                if (line.children != null) {
                    RemoteFileDesc[] childrenRFDs=
                        new RemoteFileDesc[line.children.size()];
                    for (int i = 0; i < childrenRFDs.length; i++) {
                        TableLine child = (TableLine)line.children.get(i);
                        RemoteFileDesc rfd = child.toRemoteFileDesc();
                        childrenRFDs[i] = rfd;
                    }
                    downloaders.add(childrenRFDs);
                }
                // If leaf, add if wasn't already added because of parent.  Note
                // that we only need to look at the last element of downloaders.
                else {
                    RemoteFileDesc rfd = line.toRemoteFileDesc();
                    boolean alreadyAdded = false;
                    if (!downloaders.isEmpty()) {
                        // Peek at last element of downloaders.
                        RemoteFileDesc[] last =
                        (RemoteFileDesc[])downloaders.get(downloaders.size()-1);
                        for (int i = 0; i < last.length; i++) {
                            if (last[i].equals(rfd)) {
                                alreadyAdded = true;
                                break;
                            }
                        }
                    }
                    if (!alreadyAdded)
                        downloaders.add(new RemoteFileDesc[] { rfd });
                }
            }
        }

        // 2. Download--one at a time or all at once.
        for (Iterator iter = downloaders.iterator(); iter.hasNext(); ) 
            downloadWithOverwritePrompt((RemoteFileDesc[])iter.next());
    }

    /**
     * Downloads the given files, prompting the user if the file already exists.
     */
    static void downloadWithOverwritePrompt(RemoteFileDesc[] rfds) {
        
        if (rfds.length < 1)
            return;
        if (containsExe(rfds)) {
            if (!userWantsExeDownload())
                return;
        }

        // Before proceeding...check if there is an rfd withpure metadata
        // ie no file
        int actLine = 0;
        boolean pureFound = false;
        for (; actLine < rfds.length; actLine++) {
            if (rfds[actLine].getIndex() ==
               LimeXMLProperties.DEFAULT_NONFILE_INDEX) {
                // we have our line
                pureFound = true;
                break;
            }
        }
        
        if (pureFound) {
            LimeXMLDocument doc = rfds[actLine].getXMLDoc();
            String action = doc.getAction();
            if (action != null && !action.equals("")) { // valid action
                try{
                    GUIMediator.openURL(action);
                } catch (IOException e) {
                    return; // goodbye
                }
                return; // goodbye
            }
        }
        // No pure metadata lines found...continue as usual...
        
        try {
            // Try without overwrite.
            try {
                RouterService.download(rfds, false);
            } catch (FileExistsException e) {
                String fname = e.getFileName();
                // If there's a problem, ask the user...
                String msgKey = "MESSAGE_OVERWRITE_EXISTING_FILE";
                String msg = "(" + fname + ")?";
                int response = GUIMediator.showYesNoMessage(msgKey, msg);
                if (response != GUIMediator.YES_OPTION) return;

                //...user wants to overwrite.  Do it again.
                try {
                    RouterService.download(rfds, true);
                } catch (FileExistsException e2) {
                    Assert.that(false,
                                "download(rfd, true) threw unexpected exception");
                }
            }
        } catch (AlreadyDownloadingException e) {
            // Give up.
            GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
                                  "\"" + e.getFilename() + "\".",
                                  QuestionsHandler.ALREADY_DOWNLOADING);
        } catch (java.io.FileNotFoundException fnfe) {
            GUIMediator.showError("ERROR_ACCESSING_SAVE_DIRECTORY");
        }
    }

    /**
     * Returns true if any of the entries of rfd contains a .exe file.
     */
    private static boolean containsExe(RemoteFileDesc[] rfd) {
        for (int i = 0; i < rfd.length; i++) {
            if (rfd[i].getFileName().toLowerCase(Locale.US).endsWith("exe"))
                return true;
        }
        return false;
    }

    /**
     * Prompts the user if they want to download an .exe file.
     * Returns true s/he said yes.
     */
    private static boolean userWantsExeDownload() {        
        String middleMsg = GUIMediator.getStringResource("SEARCH_VIRUS_MSG_TWO");        
        int response = GUIMediator.showYesNoMessage("SEARCH_VIRUS_MSG_ONE",
                                                    middleMsg,
                                                    "SEARCH_VIRUS_MSG_THREE",
                                            QuestionsHandler.PROMPT_FOR_EXE);
        return response == GUIMediator.YES_OPTION;
    }

    ////////////////////////// Other Controls ///////////////////////////

    /**
     * called by ResultPanel when the views are changed. Used to set the
     * tab to indicate the correct number of TableLines in the current
     * view.
     */
    static void setTabDisplayCount(ResultPanel rp) {
        RESULT_DISPLAYER.setTabDisplayCount(rp);
    }

    /**
     * Package access, called by a resultPanle, when its column config is
     * changed.  Basically we have to inform the dummyResultPanel that its
     * columns need to be updated too.
     */
    static void columnsChanged() {
        SearchMediator.getDummyResultPanel().dummyColumnsChanged();
    }

    /**
     * Package access, called by ResultPanel, when its columns are resized
     * for some reason (user changed or columns were added or removed).
     * This method is called so the dummy panels column widths also get set
     * to the corresponding values.
     */
    static void columnSizeChanged(HashMap map) {
        SearchMediator.getDummyResultPanel().
            dummyCommitColumnSizeChanged(map);
    }

    /**
     * .
     */
    static ResultPanel getDummyResultPanel() {
        return RESULT_DISPLAYER.getDummyResultPanel();
    }

    /**
     * @modifies tabbed pane, entries
     * @effects removes the currently selected result window (if any)
     *  from this
     */
    static void killSearch() {
        RESULT_DISPLAYER.killSearch();
    }

    /**
     * @modifies result tab
     * @effects resets the GUID of the result tab so that all future
     *  results are thrown out.  If all searches are stopped, then the
     *  Lime stops spinning.
     */
    static void stopSearch() {
        RESULT_DISPLAYER.stopSearch();
    }
    
    /**
     * Returns the <tt>ResultPanel</tt> for the specified GUID.
     * 
     * @param rguid the guid to search for
     * @return the <tt>ResultPanel</tt> that matches the GUID, or null
     *  if none match.
     */
    static ResultPanel getResultPanelForGUID(GUID rguid) {
        return RESULT_DISPLAYER.getResultPanelForGUID(rguid);
    }

    /**
     * Returns the currently selected <tt>MediaType</tt> for the search 
     * tab.
     *
     * @return the currently selected <tt>MediaType</tt> for the search 
     *  tab
     */
    static MediaType getSelectedMediaType() {
        return INPUT_MANAGER.getSelectedMediaType();
    }

    /**
     * Sets the currently displayed string in the search field.
     *
     * @param searchString the search string to display
     */
    static void setSearchString(String searchString) {
        // TODO::shouldn't this take special action in the case where there
        // are multiple search fields??
        INPUT_MANAGER.setSearchString(searchString);
    }

    /**
     * Notifies the search window of the current connection status,
     * making any necessary changes as a result, such as the 
     * enabled/disabled status of the search button.
     *
     * @param connected the connected status of the client
     */
    public static void setConnected(final boolean connected) {
        INPUT_MANAGER.setConnected(connected);
    }

    /**
     * Returns the search input panel component.
     *
     * @return the search input panel component
     */
    public static JComponent getSearchBoxPanel() {
        return INPUT_MANAGER.getSearchBoxPanel();
    }

    /**
     * Returns the currently selected <tt>InputPanel</tt> instance.
     * 
     * @return the currently selected <tt>InputPanel</tt> instance
     */
    public static InputPanel getCurrentInputPanel() {
        return INPUT_MANAGER.getCurrentInputPanel();
    }

    /**
     * Returns the <tt>JComponent</tt> instance containing all of the
     * search result UI components.
     *
     * @return the <tt>JComponent</tt> instance containing all of the
     *  search result UI components
     */
    public static JComponent getComponent() {
        return RESULT_DISPLAYER.getComponent();
    }

    // Inherit doc comment
    public void updateTheme() {
        INPUT_MANAGER.updateTheme();
        RESULT_DISPLAYER.updateTheme();
    }
}

