package com.limegroup.gnutella.gui.mp3;

import java.io.*;
import com.sun.java.util.collections.*;

import com.limegroup.gnutella.util.StringUtils;
import com.limegroup.gnutella.mp3.MP3Info;

/** This class encapsulates the abstraction of a mp3 playlist (.m3u).  This
 *  class IS thread-safe.
 */
public class PlayList {

    /**
     * Used when reading/writing playlist to backing store.
     */
    private File _file;

    /**
     * Contains the File Objects of the song in the playlist.
     */
    private Vector _songs;

    /**
     * Contains the File Objects of the songs ADDED to the playlist.
     */
    private Vector _newSongs;

    /**
     * The index of the currently 'playing' file.
     */
    private int _currPlaying = -1;

    /**
     * Get the index of the currently 'playing' file.
     */
    public int getCurrSongIndex() {
        debug("PlayList.getCurrSongIndex(): current index = " +
              _currPlaying);
        return _currPlaying;
    }

    /**
     * Set the index of the currently 'playing' file.
     */
    public int setCurrSongIndex(int index) {
        _currPlaying = index;
        return _currPlaying;
    }

    /**
     * Flag notifying whether a delete has taken place.
     */
    private boolean _songDeleted = false;

    /**
     * Instance variable used when the user wants to skip backwards as opposed
     * to forwards in the playlist.  Only active for 1 call to getNextSong().
     */
    private boolean _backwardsMode = false;

    /**
     * If you want the next call to getNextSong() to return the previous song
     * that was 'played', call this method.  This only lasts one call to
     * getNextSong().  Moreover, the 'history' for shuffled mode is only 1
     * song.
     */
    public void setBackwardsMode() {
        _backwardsMode = true;
    }

    /**
     * int just kept around for easy access of last song 'played'.
     */
    private int _lastIndexPlayed = 0;

    /**
     * Indicates whether the songs should be 'shuffled'.
     */
    private boolean _shuffleSongs = false;

    /**
     * Returns whether the songs should be 'shuffled'.
     */
    public boolean isShuffled() {
        return _shuffleSongs;
    }

    /**
     * Toggles 'shuffled' mode for playing songs.
     */
    public void toggleShuffle() {
        _shuffleSongs = !_shuffleSongs;
        debug("PlayList.toggleShuffle(): shuffle? " +
              _shuffleSongs);
    }

    /**
     * Used for shuffling.
     */
    Random rand = new Random();

    /**
     * Creates a PlayList accessible from the given filename.
     * If the File 'filename' exists, the playlist is loaded from that file.
     * @exception Exception Thrown if input filename is an invalid m3u file.
     */
    public PlayList(String filename) throws Exception {
        debug("PlayList(): entered.");

        _file = new File(filename);
        if (_file.isDirectory())
            throw new Exception();

        _songs = new Vector();
        if (_file.exists())
            loadM3UFile(); // load the playlist entries....

        _newSongs = new Vector(); // create this for adding new guys...

        debug("PlayList(): songs = " + _songs);
        debug("PlayList(): returning.  size is now " +
              getNumSongs());
    }

    private static final String M3U_HEADER = "#EXTM3U";
    private static final String SONG_DELIM = "#EXTINF";
    private static final String SEC_DELIM  = ":";

    /**
     * @exception Exception Thrown if load failed.<p>
     *
     * Format of playlist (.m3u) files is:<br>
     * ----------------------<br>
     * #EXTM3U<br>
     * #EXTINF:numSeconds<br>
     * /path/of/file/1<br>
     * #EXTINF:numSeconds<br>
     * /path/of/file/2<br>
     * ----------------------<br>
     */
    private void loadM3UFile() throws Exception {
        try {
            BufferedReader m3uFile = new BufferedReader(new FileReader(_file));
            String currLine = null;
            currLine = m3uFile.readLine();
            if (!currLine.startsWith(M3U_HEADER))
                throw new Exception();
            for (currLine = m3uFile.readLine(); currLine != null;
                 currLine = m3uFile.readLine()) {
                if (currLine.startsWith(SONG_DELIM)) {
                    currLine = m3uFile.readLine();
                    File toAdd = new File(currLine);
                    if (toAdd.exists() && !toAdd.isDirectory())
                        _songs.add(toAdd);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw e; // rethrow
        }
    }

    /**
     * Call this when you want to save the contents of the playlist.
     * @exception Exception Throw when save failed.
     */
    public synchronized void save() throws Exception {
        try {
            // if all songs are new, just get rid of the old file.  this may
            // happen if a delete was done....
            if (_songs.size() == 0)
                if (_file.exists())
                    _file.delete();

            boolean fileExists = _file.exists();
            FileWriter fw = new FileWriter(_file.getCanonicalPath(), true);
            PrintWriter m3uFile = new PrintWriter(fw);

            if (!fileExists) {
                m3uFile.write(M3U_HEADER);
                m3uFile.println();
            }

            for (int i = 0; i < _newSongs.size(); i++) {
                File currFile = (File)_newSongs.get(i);
                // first line of song description...
                m3uFile.write(SONG_DELIM);
                m3uFile.write(SEC_DELIM);
                // try to write out seconds info....
                try {
                    MP3Info currMP3 = new MP3Info(currFile.getCanonicalPath());
                    m3uFile.write("" + currMP3.getLengthInSeconds() + ",");
                } catch (Exception ignored) {
                    // didn't work, just write a placeholder
                    m3uFile.write("-1,");
                }
                m3uFile.write(currFile.getName());
                m3uFile.println();
                // canonical path follows...
                m3uFile.write(currFile.getCanonicalPath());
                m3uFile.println();
                // move from _newSongs to _songs...
                _songs.add(currFile);
            }
            m3uFile.flush();
            m3uFile.close();

            _songDeleted = false;
            _newSongs.clear(); // we are done!
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception();
        }
    }

    /**
     * Get the total number of songs in current playlist, including those
     * that were recently added.
     */
    public int getNumSongs() {
        return _songs.size() + _newSongs.size();
    }

    /**
     * Adds a song to the playlist.
     *
     * @return the index of the song added.  -1 is returned if the song
     *  was not successfuly added.
     */
    public int addSong(String filename) {
        return addSong(new File(filename));
    }

    /**
     * Deletes a song from the playlist.  The list is capsized, meaning
     * all indices following the deleted should be decremented by one.
     * This is the natural JTable deletion semantics.
     */
    public synchronized void deleteSong(int index) {
        if (!_songDeleted) {
            // call all songs newSongs, since we need to save them all over
            Vector temp = _newSongs;
            _newSongs = new Vector();
            // add old songs...
            for (int i = 0; i < _songs.size(); i++)
                _newSongs.add(_songs.get(i));
            // add new songs...
            for (int i = 0; i < temp.size(); i++)
                _newSongs.add(temp.get(i));

            // cleanup
            _songs.clear();
            temp = null;

            _songDeleted = true;
        }
        // now actually get rid of the appropriate song...
        if (index >= 0 && index < _newSongs.size())
            _newSongs.remove(index);

        // adjust currPlaying index appropriately
        if (_currPlaying == index)
            _currPlaying = -1;
        else if (_currPlaying > index)
            _currPlaying--;

        // adjust lastIndexPlayed appropriately
        if (_lastIndexPlayed == index)
            _lastIndexPlayed = 0;
    }

    /**
     * Adds a song to the playlist.
     *
     * @return the index of the song added; -1 is returned if the song
     *  was not successfuly added
     */
    public int addSong(File newEntry) {
        int retInt = -1;
        if (newEntry.exists() && !newEntry.isDirectory()) {
            retInt = getNumSongs();
            _newSongs.add(newEntry);
        }
        return retInt;
    }

    /**
     * Internal.
     */
    private int getNextSongIndex() {
        debug("PlayList.getNextSongIndex(): BEFORE _currPlaying = " +
              _currPlaying);
        if (isShuffled()) {
            if (_backwardsMode) {
                _currPlaying = _lastIndexPlayed;
                _backwardsMode = false;
            } else {
                int tempInt = _currPlaying;
                // record last index played...
                if (_currPlaying > -1)
                    _lastIndexPlayed = _currPlaying;
                do {
                    _currPlaying = rand.nextInt(getNumSongs());
                } while (tempInt == _currPlaying && getNumSongs() > 1);
            }
        } else {
            if (_backwardsMode) {
                _currPlaying -= 1;
                if (_currPlaying < 0)
                    _currPlaying = getNumSongs() - 1;
                _backwardsMode = false;
            } else if (++_currPlaying == getNumSongs()) {
                _currPlaying = 0;
            }
        }
        debug("PlayList.getNextSongIndex(): AFTER _currPlaying = " +
              _currPlaying);
        return _currPlaying;
    }

    /**
     * For your convenience, this method can be called repeatedly to get the
     * next song to play.  Assumes the repeat button is pressed. If a user
     * picks a certain song, use getSong(int).
     */
    public File getNextSong() {
        File retFile = null;
        synchronized (_songs) {
            if (getNumSongs() > 0) {
                int nextIndex = getNextSongIndex();
                if (nextIndex < _songs.size())
                    retFile = (File) _songs.get(nextIndex);
                else
                    retFile = (File) _newSongs.get(nextIndex - _songs.size());
            }
        }
        return retFile;
    }

    /**
     * Get a reference to the File at the indicated index in the playlist.
     */
    public File getSong(int index) {
        File retFile = null;
        if (index < getNumSongs()) {
            if (index < _songs.size())
                retFile = (File)_songs.get(index);
            else
                retFile = (File)_newSongs.get(index - _songs.size());
        }
        return retFile;
    }

    /**
     * Get the current index of the referenced song File; returns -1 if
     * that song is not in this Playlist.
     */
    public int getIndexOfSong(File in) {
        int retInt = -1;
        if (_songs.contains(in))
            retInt = _songs.indexOf(in);
        else if (_newSongs.contains(in))
            retInt = _songs.size() + _newSongs.indexOf(in);
        return retInt;
    }

    /**
     * The last order used when sorting songs by name.
     */
    private boolean _sortAscending = false;

    /**
     * Sort the playlist by name, toggling between ascending and descending
     * orders.
     */
    public synchronized void sortByName() {
        // call delete song not to delete a song, but to get into state where
        // EVERY song is a new song (implying a rewrite to disk is needed)
        deleteSong(-1);
        _currPlaying = -1;
        Collections.sort(_newSongs,
            PlayListSorter.getInstance(_sortAscending = !_sortAscending));
    }


    //Unit test

    private final boolean debugOn = false;

    private void debug(String out) {
        if (debugOn)
            System.out.println(out);
    }

    public static void main(String argv[]) throws Exception {
        PlayList pl = new PlayList(argv[0]);
        System.out.println("Number of songs: " + pl.getNumSongs());
        System.out.println("----------------------------");
        for (int i = 0; i < pl.getNumSongs(); i++)
            System.out.println(""+pl.getSong(i));
        System.out.println("----------------------------");
        pl.toggleShuffle();
        for (int i = 0; i < pl.getNumSongs(); i++)
            System.out.println(""+pl.getNextSong());
        System.out.println("----------------------------");
        if (pl.getNumSongs() > 0)
            pl.addSong(pl.getSong(0));
        else {
            pl.addSong(new File("./limewire.props"));
            pl.addSong(new File("./gnutella.net"));
        }
        System.out.println("" + pl.getSong(pl.getNumSongs() - 1));
        pl.save();
    }

}

/**
 * Implementation utility for the PlayList class (package protected).  Use the
 * getInstance(boolean ascending) factory to get an instance of this class.
 */
class PlayListSorter implements Comparator {

    /**
     * Return one of the two possible shared instances of this comparator.
     * Public instance factory.
     */
    public static PlayListSorter getInstance(boolean ascending) {
        if (ascending) {
            if (ascendingInstance == null)
                ascendingInstance = new PlayListSorter(true);
            return ascendingInstance;
        } else {
            if (descendingInstance == null)
                descendingInstance = new PlayListSorter(false);
            return descendingInstance;
        }
    }

    // Implements Comparator overrides
    public int compare(Object o1, Object o2) {
        try {
            final String s1 = ((File)o1).getName();
            final String s2 = ((File)o2).getName();
            // Fast lexicographic comparator, sorting first without case
            // sensitivity then with case differences for sort stability.
            // Does not allocate any new object or string.
            int order = StringUtils.compareIgnoreCase(s1, s2);
            if (order == 0)
                order = s1.compareTo(s2);
            return ascending ? order : (-order);
        } catch (Exception ignored) {}
        return 0; // default is equal
    }

    /**
     * Private constructor, use getInstance(boolean) to get one.
     */
    private PlayListSorter(boolean ascending) {
        this.ascending = ascending;
    }

    // Only two instances of this class
    private static PlayListSorter ascendingInstance = null;
    private static PlayListSorter descendingInstance = null;

    // Only attribute of this instance
    private boolean ascending;
}


