package com.limegroup.gnutella.gui.library;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.gui.tables.*;
import com.limegroup.gnutella.FileDesc;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.downloader.IncompleteFileManager;
import com.limegroup.gnutella.xml.LimeXMLSchemaRepository;
import com.limegroup.gnutella.xml.LimeXMLDocument;
import com.limegroup.gnutella.xml.MetaFileManager;
import com.limegroup.gnutella.xml.SchemaNotFoundException;
import com.limegroup.gnutella.xml.LimeXMLUtils;
import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.util.NameValue;


import com.sun.java.util.collections.List;
import com.sun.java.util.collections.Iterator;
import com.sun.java.util.collections.LinkedList;

import java.io.File;
import java.io.IOException;
import java.awt.Color;

/**
 * This class acts as a single line containing all
 * the necessary Library info.
 * @author Sam Berlin
 */

//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|

public final class LibraryTableDataLine extends AbstractDataLine
	implements ThemeObserver {

    /**
     * 0 final constant to preserve memory & allocations
     */
    private static final Integer ZERO_INTEGER = new Integer(0);

    /**
     * 0 / 0 final constant UploadCountHolder to preserve memory & allocations
     */
    private static final UploadCountHolder ZERO_UPLOAD_COUNT_HOLDER
                                            = new UploadCountHolder(0, 0);

    /**
     * Whether or not tooltips will display XML info.
     */
    private static boolean _allowXML;

    /**
     * The schemas available
     */
    private static String[] _schemas;

    /**
     * The meta file manager
     */
    private static MetaFileManager _mfm;

	/**
	 * Constant for the column with the name of the file.
	 */
	static final int NAME_IDX = 0;
	private static final LimeTableColumn NAME_COLUMN =
	    new LimeTableColumn(NAME_IDX, "LIBRARY_TABLE_NAME",
	                150, true, ColoredCell.class);

	/**
	 * Constant for the column storing the size of the file.
	 */
	static final int SIZE_IDX = 1;
	private static final LimeTableColumn SIZE_COLUMN =
	    new LimeTableColumn(SIZE_IDX, "LIBRARY_TABLE_SIZE",
	                30, true, ColoredCell.class);

	/**
	 * Constant for the column storing the file type (extension or more
	 * more general type) of the file.
	 */
	static final int TYPE_IDX = 2;
	private static final LimeTableColumn TYPE_COLUMN =
	    new LimeTableColumn(TYPE_IDX, "LIBRARY_TABLE_TYPE",
	                10, true, ColoredCell.class);

	/**
	 * Constant for the column storing the file's path
	 */
	static final int PATH_IDX = 3;
	private static final LimeTableColumn PATH_COLUMN =
	    new LimeTableColumn(PATH_IDX, "LIBRARY_TABLE_PATH",
	                -1, true, ColoredCell.class);

	/**
	 * Constant for the column storing the number of upload count info
	 *
	 */
	static final int UPLOADS_IDX = 4;
	private static final LimeTableColumn UPLOADS_COLUMN =
	    new LimeTableColumn(UPLOADS_IDX, "LIBRARY_TABLE_UPLOAD_COUNT",
                    10, true, UploadCountHolder.class);

	/**
	 * Constant for the column storing the numbe of hits
	 * of the file.
	 */
	static final int HITS_IDX = 5;
	private static final LimeTableColumn HITS_COLUMN =
	    new LimeTableColumn(HITS_IDX, "LIBRARY_TABLE_HITCOUNT",
	                10, true, Integer.class);

	/**
	 * Constant for the column storing the number of alt locations
	 *
	 */
	static final int ALT_LOC_IDX = 6;
	private static final LimeTableColumn ALT_LOC_COLUMN =
	    new LimeTableColumn(ALT_LOC_IDX, "LIBRARY_TABLE_NUMALTLOC",
	                10, true, Integer.class);

	/**
	 * Number of columns
	 */
	static final int NUMBER_OF_COLUMNS = 7;

	/** Variable for the file */
	private File _file;

	/** Variable for the name */
	private String _name;

	/** Variable for the type */
	private String _type;

	/** Variable for the size */
	private int _size;

	/** Variable to hold the file descriptor */
	private FileDesc _fileDesc;

	/** Variable for the path */
	private String _path;

	/**
	 * The colors for cells.
	 */
	private Color _cellColor;
	private Color _othercellColor;

	public LibraryTableDataLine() {
		super();
		updateTheme();
		GUIMediator.addThemeObserver(this);
	}
	
	/**
	 * This must be removed from the theme observer list in
	 * order to be garbage-collected.
	 */
	public void cleanup() {
	    GUIMediator.removeThemeObserver(this);
	}

	// inherit doc comment
	public void updateTheme() {
		_cellColor = ThemeFileHandler.WINDOW8_COLOR.getValue();
		_othercellColor = ThemeFileHandler.NOT_SHARING_LABEL_COLOR.getValue();
	}

	public FileDesc getFileDesc() { return _fileDesc; }

	public int getColumnCount() { return NUMBER_OF_COLUMNS; }

	/**
	 * Initialize the object.
	 * It will fail if not given a FileDesc or a File
	 * (File is retained for compatability with the Incomplete folder)
	 */
    public void initialize(Object o) {
        File file;
        if ( o instanceof FileDesc ) {
            file = ((FileDesc)o).getFile();
            _fileDesc = (FileDesc)o;
        }
        else {
            file = (File)o;
            _fileDesc = null;
        }

        super.initialize(file);

        try {
    	    _file = file;
    		String fullPath = _file.getCanonicalPath();
    		_name = _file.getName();
    		_type = "";
    		int index = _name.lastIndexOf(".");
    		int index2 = fullPath.lastIndexOf(File.separator);
            _path = fullPath.substring(0,index2);
    		if(index != -1 && index != 0) {
    			_type = _name.substring(index+1);
    			_name = _name.substring(0, index);
    		}
    		_size = (int)file.length();
    	} catch(IOException ioe) {
    	    /* ignore ? this could be bad. */
        }
    }

    public void changeFile(File newFile) {
        FileDesc tmpFD = _fileDesc;
        initialize(newFile);
        _fileDesc = tmpFD;
    }

	/**
	 * Returns the object stored in the specified cell in the table.
	 *
	 * @param idx  The column of the cell to access
	 *
	 * @return  The <code>Object</code> stored at the specified "cell" in
	 *          the list
	 */
	public Object getValueAt(int idx) {
	    switch (idx) {
	    case NAME_IDX:
	        String nm = _name;
	        // note: this fits better in the data line because
	        // sorting and whatnot will work correctly.
	        if ( LibraryMediator.instance().incompleteDirectoryIsSelected() ) {
	            try {
                //Ideally we'd eliminate the dependency on IFM, but this seems
                //better than adding yet another method to RouterService.
                    nm = IncompleteFileManager.getCompletedName(_file);
                } catch (IllegalArgumentException e) {
                    //Not an incomplete file?  Just return untranslated value.
                }
            }
	        return new ColoredCellImpl(nm,
	                    _fileDesc == null ? _othercellColor : _cellColor);
	    case SIZE_IDX:
	        return new ColoredCellImpl(new SizeHolder(_size),
	                    _fileDesc == null ? _othercellColor : _cellColor);
	    case TYPE_IDX:
	        return new ColoredCellImpl(_type,
	                    _fileDesc == null ? _othercellColor : _cellColor);
	    case HITS_IDX:
	        if ( _fileDesc == null ) return null;
	        int hits = _fileDesc.getHitCount();
	        // don't allocate if we don't have to
	        return hits == 0 ? ZERO_INTEGER : new Integer(hits);
	        //note: we use Integer here because its compareTo is
	        //      smarter than String's, and it has a toString anyway.
	    case ALT_LOC_IDX:
	        if ( _fileDesc == null ) return null;
	        int locs = _fileDesc.getNumberOfAlternateLocations() - 1;
	        return locs <= 0 ? ZERO_INTEGER : new Integer(locs);
	    case UPLOADS_IDX:
	        if ( _fileDesc == null ) return null;
	        int a = _fileDesc.getAttemptedUploads();
	        int c = _fileDesc.getCompletedUploads();
	        return a == 0 && c == 0 ? ZERO_UPLOAD_COUNT_HOLDER :
	                                  new UploadCountHolder(a, c);
	    case PATH_IDX:
	        return new ColoredCellImpl(_path,
	                    _fileDesc == null ? _othercellColor : _cellColor);
	    }
	    return null;
	}

	public LimeTableColumn getColumn(int idx) {
	    switch(idx) {
	        case NAME_IDX:          return NAME_COLUMN;
	        case SIZE_IDX:          return SIZE_COLUMN;
	        case TYPE_IDX:          return TYPE_COLUMN;
	        case PATH_IDX:          return PATH_COLUMN;
	        case HITS_IDX:          return HITS_COLUMN;
	        case ALT_LOC_IDX:       return ALT_LOC_COLUMN;
	        case UPLOADS_IDX:       return UPLOADS_COLUMN;
	    }
	    return null;
	}

	public boolean isDynamic(int idx) {
	    switch(idx) {
	        case HITS_IDX:
	        case ALT_LOC_IDX:
	        case UPLOADS_IDX:
	            return true;
	    }
	    return false;
	}

	/**
	 * Initialize things we only need to do once
	 */
	static void setXMLEnabled(boolean en) {
	    _allowXML = en;
	    if ( _allowXML ) {
	        _schemas =
	            LimeXMLSchemaRepository.instance().getAvailableSchemaURIs();
    	    FileManager fm = RouterService.getFileManager();
    	    if ( fm instanceof MetaFileManager ) _mfm = (MetaFileManager)fm;
	    } else {
	        _schemas = null;
	        _mfm = null;
	    }

	}

	public String[] getToolTipArray() {

	    // if XML isn't finished loading, no schemas exist,
	    // we don't have a meta file manager, or we don't
	    // have a FileDesc, get out of here.
	    if ( !_allowXML
	         || _schemas == null || _schemas.length == 0
	         || _mfm == null || _fileDesc == null
	        ) return null;

        // Dynamically add the information.
        List data = new LinkedList();
        List nameValues;
        LimeXMLDocument doc;
        boolean found = false;
        
        List docs = _fileDesc.getLimeXMLDocuments();
        for(Iterator i = docs.iterator(); i.hasNext(); ) {
            doc = (LimeXMLDocument)i.next();
            //data.add( LimeXMLSchema.getDisplayString(schemas[i]) );
            try {
                nameValues = doc.getOrderedNameValueList();
                // For each name/value pair...
                for( Iterator j = nameValues.iterator(); j.hasNext(); ) {
                    found = true;
                    NameValue nv = (NameValue)j.next();
                    // Add the name & value to the tooltip list
                    data.add( LimeXMLUtils.processColName(nv.getName())
                        + ": " + nv.getValue() );
                }
            } catch ( SchemaNotFoundException e ) {
                //data.remove( LimeXMLSchema.getDisplayString(schemas[i]) );
            }
        }

        if ( found ) {
            // if it had meta-data, display the filename in the tooltip also.
            data.add(0, _name);
            return (String[])data.toArray(new String[data.size()]);
	    } else {
	        return null;
	        //return new String[] { "No meta-data exists.", "Click 'annotate' to add some." };
	    }
	}
}
