package com.limegroup.gnutella.gui.chat;

import com.sun.java.util.collections.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.chat.*;

/**
 * Manages all chat session and provides an interface to access each
 * chat session.
 */
public final class ChatUIManager {

    /**
     * Constant for the single <tt>ChatManager</tt> instance for 
     * singleton.
     */
	private static final ChatUIManager INSTANCE = new ChatUIManager();
	
	/** 
	 * <tt>Map</tt> of <tt>Chatter</tt> instances.
	 */
    private Map _chats = Collections.synchronizedMap(new HashMap());

	/** 
	 * Private constructor to ensure that this class cannot be 
	 * constructed from any other class. 
	 */
	private ChatUIManager() {}
	
	/** 
	 * Returns the single instance of this class, following singleton.
	 * 
	 * @return the single <tt>ChatManager</tt> instance
	 */
	public static ChatUIManager instance() {
		return INSTANCE;
	}
	
	/**
	 * Accepts a new chat session with a new user.
	 *
	 * @chatter the new <tt>Chatter</tt> instance to chat with
	 */
	public void acceptChat(Chatter chatter) {
		ChatFrame cframe = new ChatFrame(chatter);
		cframe.setVisible(true);
		_chats.put(chatter, cframe);
	}

    /**
     * Removes the specified chat session from the list of active 
     * sessions.
     *
     * @param the <tt>Chatter</tt> instance to remove
     */
	public void removeChat(Chatter chatter) {
		ChatFrame cframe = null;
		try {
			cframe = (ChatFrame)(_chats.remove(chatter));
		} catch (ClassCastException e) {

		}
		if (cframe != null) {
		    cframe.dispose();
			cframe.setVisible(false);
		}
			
	}
	
	/**
	 * Receives a message for the session associated with the specified
	 * <tt>Chatter</tt> instance.
	 *
	 * @param chatter the <tt>Chatter</tt> instance with which the new
	 *  message is associated
	 */
	public void recieveMessage(Chatter chatter) {
		ChatFrame cframe;
		try {
			cframe = (ChatFrame)_chats.get(chatter);
			cframe.addResponse( chatter.getMessage() );
		} catch (NullPointerException ee) {
			// this should never happen, but it is not specified
			// that someone should not call recieveMessage
			// before accept, so this is just in case.
		}
	}

	/** 
	 * Lets the user know that a host is no longer available. 
	 *
	 * @param the <tt>Chatter</tt> instance for the host that is no longer
	 *  available
	 */
	public void chatUnavailable(Chatter chatter) {
		ChatFrame cframe;
		try {
			cframe = (ChatFrame)_chats.get(chatter);
			cframe.chatUnavailable();
		} catch (ClassCastException e) {
			
		} catch (NullPointerException ee) {

		}
	}

	/** 
	 * Display an error message in the chat gui for the specified chat
	 * session.
	 *
	 * @param chatter the <tt>Chatter</tt> instance associated with the error
	 * @param str the error message to display
	 */
	public void chatErrorMessage(Chatter chatter, String str) {
		ChatFrame cframe;
		try {
			cframe = (ChatFrame)_chats.get(chatter);
			cframe.displayErrorMessage(str);
		} catch (ClassCastException e) {
			
		} catch (NullPointerException ee) {

		}
    }

}
