 
package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.ErrorService;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.plaf.basic.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.plaf.*;
import java.net.*;
import javax.swing.plaf.metal.*;
import java.io.*;

/**
 * Handles the visual representation of <tt>JButton</tt> instances for the
 * application.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public class LimeButtonUI extends BasicButtonUI {

    private static final LimeButtonUI limeButtonUI = new LimeButtonUI();

	private static ImageIcon _default0; 
	private static ImageIcon _default1; 
	private static ImageIcon _default2; 
	private static ImageIcon _default3; 
	private static ImageIcon _pressed0; 
	private static ImageIcon _pressed1; 
	private static ImageIcon _pressed2; 
	private static ImageIcon _pressed3;
	
	public LimeButtonUI() {
	    themeChanged();
	}
	

	public static void themeChanged() {
		_default0    = getImageResource("button1.gif");
		_default1    = getImageResource("button2.gif");
		_default2    = getImageResource("button3.gif");
		_default3    = getImageResource("button4.gif");
		_pressed0    = getImageResource("button1_press.gif");
		_pressed1    = getImageResource("button2_press.gif");
		_pressed2    = getImageResource("button3_press.gif");
		_pressed3    = getImageResource("button4_press.gif");
	}
	
	/**
	 * Creates the <tt>ComponentUI</tt> instance for this class.
	 *
	 * @param c the component to create the ui for
	 * @return the specialized <tt>ComponentUI</tt> instance for
	 *         handling the ui for this typr of component
	 */
    public static ComponentUI createUI(JComponent c) {
        return limeButtonUI;
    }

	/**
	 * Installs the user interface elements for all future
	 * paint calls on this button.
	 *
	 * @param c the <tt>JComponent</tt> to install ui elements for,
	 *          in this case a button
	 */
  	public void installUI(JComponent c) {
		super.installUI(c);
		AbstractButton b = (AbstractButton)c;
		if(b instanceof JButton  && 
		   !(b instanceof MediaButton) &&
		   !(b instanceof MetalComboBoxButton) &&
		   !(b instanceof BasicArrowButton) &&
		   !(b instanceof MetalScrollButton) &&
		   !(b.getText() == null) &&
		   !(b.getText().equals("")) &&
		   (shouldInstallUI(b))) {
			
			setButtonValues(b);
			setIcons(b);
		}
  	}
  	
  	/**
  	 * Removes the UI stuff that was added to this component.
  	 */
  	public void uninstallUI(JComponent c) {
  	    super.uninstallUI(c);
  	    
  	    removeButtonValues((AbstractButton)c);
  	    removeIcons((AbstractButton)c);
  	}   

    /**
     * This sets the button values to the appropriate value for our
     * LimeLookAndFeel.
     * It caches the previously value as a client property so that 
     * it can be undone later.
     * This is arguably better than creating distinct UI's per each button.
     * See http://java.sun.com/products/jfc/tsc/articles/architecture/ for
     * details on Stateless vs Stateful UI delegates.
     */
	private void setButtonValues(AbstractButton b) {
	    b.putClientProperty("setBorderPainted", b.isBorderPainted() ? 
	        Boolean.TRUE : Boolean.FALSE );
		b.setBorderPainted(false);
	    b.putClientProperty("setContentAreaFilled", b.isContentAreaFilled() ?
	        Boolean.TRUE : Boolean.FALSE );
		b.setContentAreaFilled(false);
	    b.putClientProperty("setHorizontalTextPosition",
	        new Integer(b.getHorizontalTextPosition()));
		b.setHorizontalTextPosition(SwingConstants.CENTER);
	    b.putClientProperty("setFocusPainted", b.isFocusPainted() ?
	        Boolean.TRUE : Boolean.FALSE );
		b.setFocusPainted(false);
	}
	
	/**
	 * This returns the values to what they were prior to installing the UI.
	 */
	private void removeButtonValues(AbstractButton b) {
	    b.setBorderPainted( getBoolean("setBorderPainted", b) );
	    b.setContentAreaFilled( getBoolean("setContentAreaFilled", b) );
	    b.setHorizontalTextPosition(getInteger("setHorzontalTextPosition", b));
	    b.setFocusPainted( getBoolean("setFocusPainted", b) );
    }
    
    private boolean getBoolean(String s, AbstractButton b) {
        Boolean bool = (Boolean)b.getClientProperty(s);
        return bool == null ? false : bool.booleanValue();
    }
    
    private int getInteger(String s, AbstractButton b) {
        Integer integr = (Integer)b.getClientProperty(s);
        return integr == null ? 0 : integr.intValue();
    }

	private boolean shouldInstallUI(AbstractButton button) {
		Icon icon = button.getIcon();
		if(icon == null) return true;
		if(icon == _default0) return false;		   
		if(icon == _default1) return false;
		if(icon == _default2) return false;
		if(icon == _default3) return false;		

		Icon pressedIcon = button.getPressedIcon();
		if(pressedIcon == null) return true;
		if(pressedIcon == _pressed0) return false;		   
		if(pressedIcon == _pressed1) return false;
		if(pressedIcon == _pressed2) return false;
		if(pressedIcon == _pressed3) return false;		
		return true;
	}


	/**
	 * Sets the standard icon as well as the pressed icon for the
	 * <tt>button</tt> argument.  This caches the previous icons
	 * so that when the UI is uninstalled, the values can be returned.
	 *
	 * @param button the button to use for setting the icons
	 */
	private void setIcons(AbstractButton button) {
		FontMetrics fm = button.getFontMetrics(button.getFont());
		int stringWidth = fm.stringWidth(button.getText());
		Icon buttonIcon;
		Icon pressedButtonIcon;
		if(stringWidth < 30) {
			buttonIcon = _default0;
			pressedButtonIcon = _pressed0;
		}
		else if(stringWidth < 53) {
			buttonIcon = _default1;
			pressedButtonIcon = _pressed1;
		}
		else if(stringWidth < 72) {
			buttonIcon = _default2;
			pressedButtonIcon = _pressed2;
		}
		else {
			buttonIcon = _default3;
			pressedButtonIcon = _pressed3;
		}		
	    button.putClientProperty("setIcon", button.getIcon());
		button.setIcon(buttonIcon);
	    button.putClientProperty("setDisabledIcon", button.getDisabledIcon());
        button.setDisabledIcon(buttonIcon);        
	    button.putClientProperty("setPressedIcon", button.getPressedIcon());
		button.setPressedIcon(pressedButtonIcon);
	    button.putClientProperty("setDisabledSelectedIcon",
	        button.getDisabledSelectedIcon());
        button.setDisabledSelectedIcon(pressedButtonIcon);
	    button.putClientProperty("setPreferredSize", button.getPreferredSize());       
		button.setPreferredSize(new Dimension(buttonIcon.getIconWidth(),
											  buttonIcon.getIconHeight()));	
	}
	
	/**
	 * Returns the icons to what they were prior to installation.
	 */
	private void removeIcons(AbstractButton b) {
	    b.setIcon((Icon)b.getClientProperty("setIcon"));
	    b.setDisabledIcon((Icon)b.getClientProperty("setDisabledIcon"));
	    b.setPressedIcon((Icon)b.getClientProperty("setPressedIcon"));
	    b.setDisabledSelectedIcon(
	        (Icon)b.getClientProperty("setDisabledSelectedIcon"));
	    b.setPreferredSize((Dimension)b.getClientProperty("setPreferredSize"));	    
    }
    
	/**
	 * Accesses the button icons.
	 * 
	 * @return the Icon object specified in the param string
	 * @param imageName the name of the icon to return
	 */
  	private static ImageIcon getImageResource(String imageName) {
		File themeDir = ThemeSettings.THEME_DIR.getValue();
		return new ImageIcon(toURL(new File(themeDir, imageName)));
  	}

    private static URL toURL(File f) {
		try {
			return new URL("file", "", slashify(f.getAbsolutePath(), f));
		} catch(MalformedURLException e) {
		    ErrorService.error(e);
			return null;
		}
    }

    private static String slashify(String path, File f) {
		String p = path;
		if (File.separatorChar != '/')
			p = p.replace(File.separatorChar, '/');
		if (!p.startsWith("/"))
			p = "/" + p;
		if (!p.endsWith("/") && f.isDirectory())
			p = p + "/";
		return p;
    }
}

   

