package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.settings.*;
import javax.swing.*;
import javax.swing.filechooser.*;
import java.awt.*;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * This is a utility class that displays a file chooser dialog to the user,
 * automatically selecting the appropriate dialog based on the operating
 * system, the current theme, etc.  For example, if the user is on OS X
 * and is not using the default theme, this displays the standard
 * <tt>MetalLookAndFeel</tt> file chooser, as that is the only one that
 * will appear with themes.
 */
public final class FileChooserHandler {

	/**
	 * Displays a directory chooser to the user and returns the selected
	 * <tt>File</tt>.  This uses the main application frame as the parent
	 * component.
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a directory was not selected correctly
	 */
	public static File getInputDirectory() {
		return getInputDirectory(GUIMediator.getAppFrame());
	}

	/**
	 * Same as <tt>getInputDirectory</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent) {
		return getInputDirectory(parent, 
								 "FILE_CHOOSER_DIRECTORY_TITLE", 
								 CommonUtils.getCurrentDirectory());
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param directory the directory to open the dialog to
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, 
										 File directory) {
		return getInputDirectory(parent, 
								 "FILE_CHOOSER_DIRECTORY_TITLE", 
								 "FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
								 directory);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param directory the directory to open the dialog to
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, 
										 File directory,
										 FileFilter filter) {
		return getInputDirectory(parent, 
								 "FILE_CHOOSER_DIRECTORY_TITLE",  
								 "FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
								 directory,
								 filter);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param directory the directory to open the dialog to
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, String titleKey,
										 File directory) {
		return getInputDirectory(parent, 
								 titleKey, 
								 "FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
								 directory);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param directory the directory to open the dialog to
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, String titleKey,
										 File directory, FileFilter filter) {
		return getInputDirectory(parent, 
								 titleKey, 
								 "FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
								 directory,
								 filter);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, String titleKey,
										 String approveKey, File directory) {
		return getInputDirectory(parent, 
								 titleKey, 
								 approveKey,
								 directory,
								 null);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser as well as other options.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputDirectory(Component parent, String titleKey,
										 String approveKey, File directory,
										 FileFilter filter) {
		return getInput(parent, 
						titleKey, 
						approveKey,
						directory,
						JFileChooser.DIRECTORIES_ONLY,
						JFileChooser.APPROVE_OPTION,
						filter);
	}



	/**
	 * Displays a file chooser to the user and returns the selected
	 * <tt>File</tt>.  This uses the main application frame as the parent
	 * component.
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile() {
		return getInputFile(GUIMediator.getAppFrame());
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent) {
		return getInputFile(parent, 
							"FILE_CHOOSER_DIRECTORY_TITLE", 
							"FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
							CommonUtils.getCurrentDirectory());
	}


	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent, FileFilter filter) {
		return getInputFile(parent, 
							"FILE_CHOOSER_DIRECTORY_TITLE", 
							"FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
							CommonUtils.getCurrentDirectory(),
							filter);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param directory the directory to open the dialog to
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent, String titleKey,
									File directory) {
		return getInputFile(parent, 
							titleKey, 
							"FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
							directory);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param directory the directory to open the dialog to
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent, String titleKey,
									File directory, FileFilter filter) {
		return getInputFile(parent, 
							titleKey, 
							"FILE_CHOOSER_DIRECTORY_BUTTON_LABEL",
							directory,
							filter);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent, String titleKey,
									String approveKey, File directory) {
		return getInput(parent, 
						titleKey, 
						approveKey,
						directory,
						JFileChooser.FILES_ONLY,
						JFileChooser.APPROVE_OPTION);
	}

	/**
	 * Same as <tt>getInputFile</tt> that takes no arguments,
	 * except this allows the caller to specify the parent component of
	 * the chooser.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 *
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInputFile(Component parent, String titleKey,
									String approveKey, File directory,
									FileFilter filter) {
		return getInput(parent, 
						titleKey, 
						approveKey,
						directory,
						JFileChooser.FILES_ONLY,
						JFileChooser.APPROVE_OPTION,
						filter);
	}


	/**
	 * The implementation that the other methods delegate to.  This
	 * provides the caller with all available options for customizing
	 * the <tt>JFileChooser</tt> instance.  If a <tt>FileDialog</tt>
	 * is displayed instead of a <tt>JFileChooser</tt> (on OS X, for
	 * example), most or all of these options have no effect.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 * @param mode the "mode" to open the <tt>JFileChooser</tt> in from 
	 *  the <tt>JFileChooser</tt> class, such as 
	 *  <tt>JFileChooser.DIRECTORIES_ONLY</tt>
	 * @param option the option to look for in the return code, such as
	 *  <tt>JFileChooser.APPROVE_OPTION</tt>
	 * 
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInput(Component parent, String titleKey,
								String approveKey,
								File directory, int mode,
								int option) {
		return getInput(parent, titleKey, approveKey, directory, mode,
						option, null);
	}

	/**
	 * The implementation that the other methods delegate to.  This
	 * provides the caller with all available options for customizing
	 * the <tt>JFileChooser</tt> instance.  If a <tt>FileDialog</tt>
	 * is displayed instead of a <tt>JFileChooser</tt> (on OS X, for
	 * example), most or all of these options have no effect.
	 *
	 * @param parent the <tt>Component</tt> that should be the dialog's
	 *  parent
	 * @param titleKey the key for the locale-specific string to use for
	 *  the file dialog title
	 * @param approveKey the key for the locale-specific string to use for
	 *  the approve button text
	 * @param directory the directory to open the dialog to
	 * @param mode the "mode" to open the <tt>JFileChooser</tt> in from 
	 *  the <tt>JFileChooser</tt> class, such as 
	 *  <tt>JFileChooser.DIRECTORIES_ONLY</tt>
	 * @param option the option to look for in the return code, such as
	 *  <tt>JFileChooser.APPROVE_OPTION</tt>
	 * @param filter the <tt>FileFilter</tt> instance for customizing 
	 *  the files that are displayed -- if this is null, no filter is used
	 * 
	 * @return the selected <tt>File</tt> instance, or <tt>null</tt> if
	 *  a file was not selected correctly
	 */
	public static File getInput(Component parent, String titleKey,
								String approveKey,
								File directory, int mode,
								int option,
								FileFilter filter) {
		if(!CommonUtils.isMacOSX() || !ThemeSettings.isDefaultTheme()) {
			JFileChooser fileChooser = 
				getDirectoryChooser(titleKey, approveKey, directory, mode, 
                                    filter);
			
			if(fileChooser.showOpenDialog(parent) != option)
				return null;
			
			return fileChooser.getSelectedFile();
		} else {
            Method setProperty = null;
            if (CommonUtils.isJava14OrLater())  {
                // we need to use reflection to get this to compile on 118
                try {
                    setProperty = System.class.getMethod("setProperty", new Class[] {String.class, String.class});
                    setProperty.invoke(System.class, new Object[] {"apple.awt.fileDialogForDirectories", "true"});
                } catch (SecurityException e) {
                    // nothing we can do
                } catch (NoSuchMethodException e) {
                    // nothing we can do
                } catch (IllegalAccessException e) {
                    // nothing we can do
                } catch (InvocationTargetException e) {
                    // nothing we can do
                }
            }
			FileDialog fileDialog = getOSXDirectoryChooser();
			fileDialog.setVisible(true);
			String dirStr = fileDialog.getDirectory();
			String fileStr = fileDialog.getFile();
            if (CommonUtils.isJava14OrLater() && setProperty != null)  {
                try {
                    setProperty.invoke(System.class, new Object[] {"apple.awt.fileDialogForDirectories", "false"});
                } catch (IllegalAccessException e) {
                    // nothing we can do
                } catch (InvocationTargetException e) {
                    // nothing we can do
                }
            }
			if((dirStr==null) || (fileStr==null)) return null;
			return new File(dirStr, fileStr);
		}		
	}

	/**
	 * Returns a new <tt>JFileChooser</tt> instance for selecting directories 
	 * and with internationalized strings for the caption and the selection 
	 * button.  If the user is running OS X, the getOSXDirectoryChooser method
	 * should be called over this.
	 *
	 * @return a new <tt>JFileChooser</tt> instance for selecting directories.
	 * @throws <tt>RuntimeException</tt> if this method is called 
	 *  on an OS X system
	 */  
	private static JFileChooser getDirectoryChooser(String titleKey,
													String approveKey,
													File directory, 
													int mode,
													FileFilter filter) {
	    JFileChooser chooser = new JFileChooser(directory);
		if(filter != null) {
			chooser.setFileFilter(filter);
		}
	    chooser.setFileSelectionMode(mode);
		String title = 
		    GUIMediator.getStringResource(titleKey);
		chooser.setDialogTitle(title);
		
		String approveButtonText = 
		    GUIMediator.getStringResource(approveKey);
		chooser.setApproveButtonText(approveButtonText);
		return chooser;
	}
	
	/**
	 * Return the <tt>FileDialog</tt> instance that is a specialized file 
	 * chooser that meets the OS X look and feel for directory choosers.
	 *
	 * @return a <tt>FileDialog</tt> specialized as a directory chooser for
	 *  OS X
	 * @throws <tt>RuntimeException</tt> if this method is called 
	 *  on a non-OS X system
	 */
	private static FileDialog getOSXDirectoryChooser() {
	    if(!CommonUtils.isMacOSX()) {
	        throw new java.lang.RuntimeException("non-OS X "+
	            "systems should use the getOSXDirectoryChooser method");
	    }

	    // return a FileDialog with the "secret" parameter of 3 to select
	    // only directories
	    return new FileDialog(GUIMediator.getAppFrame(), "", CommonUtils.isJava14OrLater() ? FileDialog.LOAD : 3);
	}
}
