#ifndef JMP_H__
#define JMP_H__

#include <jni.h>
#include <jvmpi.h>
#include <hash.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if ENABLE_NLS
#  include <libintl.h>
#  include <locale.h>
#  define _(x) gettext(x)
#else
#  define _(x) (x)
#endif

/** These are standard type of objects, they are defined in there
 *  respective header files.
 */
typedef struct cls cls;
typedef struct obj obj;
typedef struct method method;
typedef struct methodtime methodtime;
typedef struct jthread jthread;
typedef struct arena arena;
typedef struct timerstack timerstack;
typedef struct mvector mvector;

#ifdef linux
/** A useful function for debug purposes. */
void get_backtrace ();
#endif 

/** Create a new monitor */
JVMPI_RawMonitor jmp_create_monitor (char *);

/** Delete a monitor */
void jmp_delete_monitor (JVMPI_RawMonitor);

/** Lock a monitor */
void jmp_lock_monitor (JVMPI_RawMonitor);

/** Unlock a monitor */
void jmp_unlock_monitor (JVMPI_RawMonitor);

/** Wait for a monitor */
void jmp_monitor_wait (JVMPI_RawMonitor, jlong milis);

/** Notify all waiting for a monitor */
void jmp_monitor_notify_all (JVMPI_RawMonitor);

/** Get the jmp class object from its id. */
cls*        get_class (const jobjectID class_id);

/** Get the jmp object from its id. */
obj*        get_object (const jobjectID obj_id);

/** Get the jmp method object from its id. */
method*     get_method (const jmethodID method_id);

/** Get the jmp thread object from its id. */
jthread*    get_jthread (const JNIEnv* env_id);

/** Get the jmp arena object from its id. */
arena*      get_arena (const jint arena_id);

/** Get all the classes. 
 * @return A hashtable with cls* 
 */
hashtab*    get_classes ();

/** Get all the methods. 
 * @return A hashtable with method*
 */
hashtab*    get_methods ();

/** Get the timerstack for a thread. */
timerstack* get_timerstack (JNIEnv* env_id);

/** Get all the threads */
hashtab*    get_threads ();

/** Ask the jvm about a specific object allocation. */
void get_object_alloc (jobjectID obj_id);

/** request the object info... */
void get_class_load (jobjectID class_id);

/** request the object info... 
 *  Note: this method is not available in the SUN jvm.
 */
void get_arena_load (jint arena_id);

/** Get the super class of the specified class. 
 *  This will also fill in the cls->super pointer..
 * @param c the cls to get the super class of.
 * @return the super class of c 
 */
cls* get_super_class (cls* c);

/** Call get_last_down_link after this to get the object links.
 * @param o the object to get down links for.
 */
void get_instance_info (obj* o);

/** Perform some basic checks after a heap dump. */
void check_objects (size_t cnt);


/** returns true if we are currently tracing objects. */
int tracing_objects ();

/** returns true if we are currently tracing methods. */
int tracing_methods ();

/** return true if we are currently tracing monitors */
int tracing_monitors ();

/** Test if jmp is using its GUI */
int usingUI ();

/** Notify the shutdown handler that the UI is finished. */
void ui_shutdown_complete ();

/** Reset all current counters... */
void reset_counters ();

/** Restore all counters... */
void restore_counters ();

/** Run the java garbage collector. */
void run_GC ();

/** Get the size of the heap. */
jlong current_heap_size ();

/** request that jmp dumps it data. 
 * @return the id of the dump file.
 */
int run_data_dump ();

/** Run the heap dump. */
void run_heap_dump ();

/** Run the object heap dump. 
 *  This is a smaller, faster version of the full heap dump
 *  Will only return a list of objectIDs. Useful for making sure 
 *  we have all the objects
 */
void run_object_dump ();

/** Run the monitor dump. 
 *  The hashtab should be freed when done with. 
 *  This method is not thread safe, do not run it in several threads
 *  at the same time.
 */
hashtab* run_monitor_dump ();

/** Clean up monitor information, that is free all allocated data.
 */
void cleanup_monitor_information ();

/** request that jmp dumps objects alloced in the given method. 
 * @param m the method that has alloced objects.
 */
void run_show_objects_alloced_by_method (method* m);

/** request that jmp dumps objects by a given class
 * @param c the class of the object to dump.
 */
void run_show_objects_alloced_by_class (cls* c);

/** request that jmp show object owners for all instances of a given class.
 * @param c the class of the instaces jmp should find owners for.
 */
void run_find_owners (cls* c);

/** request that jmp show object owners for the given instances.
 * @param o the object jmp should find owners for.
 */
void run_find_instance_owners (obj* o);

/** request that jmp show object owners statistics for a given class.
 * @param c the class of the instaces jmp should find owners for.
 */
void run_owners_statistics (cls* c);

typedef void   (*string_callback) (obj*, void*);
/** perform the given callback on each string. */
void for_each_string (string_callback sc, void* data);

/** Enable object tracing */
void enable_object_events ();

/** Disable object tracing */
void disable_object_events ();

/** Fix the stack so that it is correct (after stack 
 *  underflow or after method_events have been enabled). 
 */
void get_call_trace (jthread* elem);

/** Fix the stack so that it is correct (after stack 
 *  underflow or after method_events have been enabled). 
 */
void get_call_trace_env (JNIEnv* tid);

/** Enable method tracing */
void enable_method_events ();

/** Disable method tracing */
void disable_method_events ();

/** Enable monitor tracing */
void enable_monitor_events ();

/** Disable monitor tracing */
void disable_monitor_events ();

/** Get the state of a thread. */
jint get_thread_state (const jthread* jt);

#endif /* JMP_H__ */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
