/*
 * @(#)StringUtil.java	1.2 02/04/03
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.util;

import java.util.Properties;


/**
 * StringUtil
 *
 * @version	1.2
 *
 */
public class StringUtil {

    /**
     * Expand property variables a string with the
     * corresponding values in a Properties instance. A property
     * variable has the form ${some.property}. The variable
     * ${/} is shorthand for ${file.separator}. So if the string contains
     * ${jmq.varhome}${/}${jmq.instancename}${/}store then it would be
     * expanded to something like /var/opt/SUNWjmq/jmqbroker/store
     * ${NL} is shorthand for ${line.separator}
     *
     * If there are no variables in 'str' then you will get back your
     * original string.
     *
     * @param str   The string to expand variables in
     * @param props The Properties object to extract variables from
     *
     * @returns A string with all variables expanded
     */
    public static String expandVariables(String str, Properties props) {

	if (str == null) return str;

	String vname, value = null;

	int len = str.length();
	StringBuffer sbuf = null;

	int current, vstart, vend;
	current = vstart = vend = 0;

	while (current < len) {
	    // Locate the start of a variable
	    if ((vstart = str.indexOf('$', current)) == -1) {

                if (sbuf != null) {
	            // No more variables. Copy remainder of string and stop
	            sbuf.append(str.substring(current, len));
                }
		break;
	    }

	    if (str.charAt(vstart + 1) == '{') {
		// We have a variable start. Find the end
	        if ((vend = str.indexOf('}', vstart + 1)) == -1) {

                  if (sbuf != null) {
		    //No end. No more vars. Copy remainder of string and stop
		    sbuf.append(str.substring(current, len));
                  }
		  break;
	        }
	    }

            // Looks like we are expanding variables. Allocate buffer
            // if we haven't already.
            if (sbuf == null) {
	        sbuf = new StringBuffer(2 * len);
            }

	    // ${jmq.home}
	    // ^         ^-vend
	    // +- vstart
            if (vend > vstart) {
	        vname = str.substring(vstart + 2, vend);
            } else {
                // Variable is malformed.
                vname = null;
                vend = vstart;
            }

	    // Get variable value
            if (vname == null) {
                value = null;
	    } else if (vname.equals("/")) {
		value = System.getProperty("file.separator");
            } else if (vname.equals("NL")) {
		value = System.getProperty("line.separator");
            } else if (props != null) {
	        value = props.getProperty(vname);
            }

	    // Copy over stuff before variable
	    sbuf.append(str.substring(current, vstart));

	    // Copy variable
	    if (value == null) {
		// No value. Just duplicate variable name and move on
		sbuf.append(str.substring(vstart, vend + 1));
            } else {
		// Good variable. Copy value
		sbuf.append(value);
            }

	    // Advance current pointer past variable and continue
	    current = vend + 1;
	}

        if (sbuf != null) {
	    return sbuf.toString();
        } else {
            // If we never expanded variables we can return the original string
            return str;
        }
    }
}

